/*----------------------------------------------------------------------------
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the this software; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 */
#include <stdarg.h>
#include <lxk_panel.h>
#include <common.h>
#include "lxk_eeprom_info.h"

/*
 * Contains the list of panels that are supported by the current hardware.
 * The actual panel type is detected at run-time.
 * The external clock requirement members may be updated by the probe functions.
 */
static const lxk_panel_ctl_t panels[] = {
	{ mv62xx_disp_ref_panel_probe, mv62xx_disp_ref_panel_status,
		mv62xx_disp_ref_panel_buttons, mv62xx_disp_ref_panel_type,
		mv62xx_disp_ref_panel_subtype, mv62xx_disp_ref_panel_extclkreq},

#ifdef CONFIG_LXK_PANEL_VGA

#ifdef CONFIG_LXK_PANEL_SMPN
	{lxk_panel_lb_vga_probe, lxk_panel_vga_status, lxk_panel_lb_vga_buttons, 
					lxk_panel_lb_vga_type, lxk_panel_lb_vga_subtype,
					lxk_panel_lb_vga_extclkreq},
#endif
	{ lxk_panel_vga_probe, lxk_panel_vga_status, lxk_panel_vga_buttons, 
					lxk_panel_vga_type, lxk_panel_vga_subtype,
					lxk_panel_vga_extclkreq},
	{ lxk_panel_vga_probe_gen2, lxk_panel_vga_status, lxk_panel_vga_buttons_gen2,
					lxk_panel_vga_type_gen2, lxk_panel_vga_subtype_gen2,
					lxk_panel_vga_extclkreq},

#endif

#ifdef CONFIG_LXK_PANEL_2X16V1
    {lxk_panel_2x16v1_probe, lxk_panel_2x16v1_status, lxk_panel_2x16v1_buttons, 
    					lxk_panel_2x16v1_type, NULL, NULL},
#endif
};

static int probed __attribute__ ((section (".data"))) = 0;
static const lxk_panel_ctl_t *panel __attribute__ ((section (".data"))) = NULL;

static const lxk_panel_ctl_t *lxk_get_panel(void)
{
	int i;

	if (probed)
		return panel;
	probed = 1;

	for (i = 0; i < ARRAY_SIZE(panels); i++) {
		if (panels[i].probe() == 0) {
			panel = &panels[i];
			break;
		}
	}
	return panel;
}

void lxk_panel_status(int msg, ...)
{
	va_list args;
	const lxk_panel_ctl_t *panel = lxk_get_panel();

	if (!panel || !panel->status)
		return;

	va_start(args, msg);
	panel->status(msg, args);
	va_end(args);
}

unsigned long long lxk_panel_get_buttons(void)
{
	const lxk_panel_ctl_t *panel = lxk_get_panel();

	if (!panel || !panel->buttons)
		return 0;

	return panel->buttons();
}

int lxk_panel_get_type(void)
{
	const lxk_panel_ctl_t *panel = lxk_get_panel();

	if (!panel || !panel->type)
		return LXK_PANEL_TYPE_INVALID;
	
	return panel->type();
}

char *lxk_panel_get_subtype(void)
{
	const lxk_panel_ctl_t *panel = lxk_get_panel();

	if (!panel || !panel->subtype)
		return NULL;
	
	return panel->subtype();
}

int lxk_panel_init(void)
{
	return lxk_get_panel() != NULL;
}

lxk_panel_ext_clk_requirement_t lxk_panel_clk_required(void)
{
	const lxk_panel_ctl_t *panel = lxk_get_panel();

	if (!panel || !panel->ext_clk_required)
		return LXK_PANEL_EXT_CLK_USED;

	return panel->ext_clk_required();
}
