#include <common.h>
#include <i2c.h>
#include <malloc.h>
#include <lxk_panel.h>
#include "lxk_panel_common.h"
#include "lxk_gen2_eeprom.h"

//#define PANEL_EEPROM_DEBUG

#define I2CADDR_PANEL_EEPROM (0xa0 >> 1)

#ifndef CONFIG_PANEL_I2C_BUS
#error CONFIG_PANEL_I2C_BUS must be defined in board config
#endif

static lxk_panel_i2c_info_t i2c_info = {
	.bus_num = -1,
	.dev_addr = I2CADDR_PANEL_EEPROM,
	.read_retries = 0,
	.write_retries = 0,
	.delay = 1000,
	.protocol = PROTOCOL_EEPROM,
};

static char display_name[MAX_NAME_SIZE + 1] = "";

static int red_lut_found = 0;
static int green_lut_found = 0;
static int blue_lut_found = 0;

static int checksum_nv(u8 *buf, size_t count, u16 expected_checksum) {
	int i;
	u16 checksum = 0;
	int rc = 0;

	for (i = 0; i < count; i++) {
		checksum += buf[i] & 0x0ff;
	}

	if(expected_checksum != checksum) {
		printf("%s:%d Checksum mismatch, expected=0x%04x, actual=0x%04x\n",
				__func__, __LINE__, expected_checksum, checksum);
		rc = -1;
	}

	return rc;
}

static int read_record(void *record, int record_num, lxk_gen2_display_info_eeprom *nv) {
	size_t size;
	int offset;
	int rc;

	size = val16(nv->directory[record_num].size_hi, nv->directory[record_num].size_lo);
	offset = sizeof(*nv) + val16(nv->directory[record_num].offset_hi, nv->directory[record_num].offset_lo);

	rc = lxk_panel_i2c_read(&i2c_info, offset, 1, (unsigned char *)record, size);
	if(rc) {
		printf("%s: Read error, record #%d\n", __func__, record_num);
		return -1;
  	}
  	rc = checksum_nv((u8 *)record, size, val16(nv->directory[record_num].checksum_hi, nv->directory[record_num].checksum_lo));
	if(rc) {
		printf("%s: checksum error, record #%d\n", __func__, record_num);
		return -1;
  	}

	return 0;
}

static int init_gamma_lut(vga_panel_data *vpd, gamma_lut_single *lut, DISPLAY_RECORD_TYPE channel) {
	int i;

	if(!vpd->gamma) {
  		vpd->gamma = (gamma_table_t *)malloc(sizeof(gamma_table_t));
  		if(!vpd->gamma) {
  			return -1;
  		}
  		if(channel != GAMMA_LUT_COMMON) {
  			for(i = 0; i < 256; i++) {
  				vpd->gamma->red[i] = i;
  				vpd->gamma->green[i] = i;
  				vpd->gamma->blue[i] = i;
  			}
  		}
  	}
  	switch(channel) {
  	case GAMMA_LUT_COMMON:
  		if(!red_lut_found)
  			memcpy(vpd->gamma->red, lut->data, sizeof(lut->data));
  		if(!green_lut_found)
  			memcpy(vpd->gamma->green, lut->data, sizeof(lut->data));
  		if(!blue_lut_found)
  			memcpy(vpd->gamma->blue, lut->data, sizeof(lut->data));
  		break;
	case GAMMA_LUT_RED:
		memcpy(vpd->gamma->red, lut->data, sizeof(lut->data));
		red_lut_found = 1;
		break;
	case GAMMA_LUT_GREEN:
		memcpy(vpd->gamma->green, lut->data, sizeof(lut->data));
		green_lut_found = 1;
		break;
	case GAMMA_LUT_BLUE:
		memcpy(vpd->gamma->blue, lut->data, sizeof(lut->data));
		blue_lut_found = 1;
		break;
	default:
		break;
  	}

	return 0;
}

lxk_gen2_display_info_t *lxk_gen2_eeprom_probe(vga_panel_data *vpd) {
	lxk_gen2_display_info_eeprom nv;
	lxk_gen2_display_info_t *dynamic = NULL;
	int rc = 0;
	int errline = -1;
	char *errtext = NULL;
	size_t size;
	int i;
	

	rc = lxk_panel_i2c_probe(&i2c_info);
	if(rc) {
		printf("%s:%d: No panel eeprom option found.\n", __func__, __LINE__);
		return NULL;
  	}
  	else {
		printf("%s:%d: Found panel eeprom option.\n", __func__, __LINE__);
  	}

	/*
	 * Eeprom is present, but not yet validated. Set a non-destructive default value
	 * for backlight duty cycle to protect hardware in case contents are invalid.
	 */
	vpd->backlight_duty_max	= vpd->backlight_duty_init = BACKLIGHT_DUTY_FAILSAFE;

	size = sizeof(nv);
	rc = lxk_panel_i2c_read(&i2c_info, 0, 1, (unsigned char *)&nv, size);
	if(rc) {
		errline = __LINE__;
		errtext = "read error";
		rc = -1;
		goto done;
  	}
#ifdef PANEL_EEPROM_DEBUG
	printf("%s:%d: Panel eeprom directory:\n", __func__, __LINE__);
  	print_buffer(0, (void *)&nv, sizeof(char), size, 8); 
#endif

	if(val16(nv.magic_hi, nv.magic_lo) != LXK_GEN2_MAGIC_NUMBER) {
		errline = __LINE__;
		errtext = "blank or unrecognized";
		rc = -1;
		goto done;
	}

	size = RECORD_DIRECTORY_CHECKSUM_SIZE;
	rc = checksum_nv((u8 *)&nv, size, val16(nv.checksum_hi, nv.checksum_lo));
	if(rc) {
		errline = __LINE__;
		errtext = "checksum";
		rc = -1;
		goto done;
  	}

	for(i = 0; i < 	MAX_RECORDS; i++) {
#ifdef PANEL_EEPROM_DEBUG
		printf("%s:%d: Checking record %d.\n", __func__, __LINE__);
#endif
		switch(nv.directory[i].type) {
		case LXK_GEN2_DISPLAY_TIMING:
		{
			lxk_gen2_display_nv display_record;

			rc = read_record(&display_record, i, &nv);
			if(rc) {
				errline = __LINE__;
				errtext = "record read";
				rc = -1;
				goto done;
			}

  			dynamic = (lxk_gen2_display_info_t *)malloc(sizeof(lxk_gen2_display_info_t));
			if(!dynamic) {
				errline = __LINE__;
				errtext = "malloc";
				rc = -1;
				goto done;
  			}

			printf("%s:%d: found LXK_GEN2_DISPLAY_TIMING record.\n", __func__, __LINE__);

			dynamic->lxk_type = LXK_PANEL_TYPE_DYNAMIC;
			snprintf(display_name, MAX_NAME_SIZE, "%s", display_record.name);
			dynamic->name = display_name;
			memcpy((u8 *)&dynamic->dt, (u8 *)&display_record.dt, sizeof(dynamic->dt));
			break;
		}
		case PANEL_HW:
		{
			lxk_gen2_panel_hw hw_record;

			rc = read_record(&hw_record, i, &nv);
			if(rc) {
				errline = __LINE__;
				errtext = "record read";
				rc = -1;
				goto done;
			}

			printf("%s:%d: found PANEL_HW record.\n", __func__, __LINE__);

			if(hw_record.bl_duty_max > 0)
				vpd->backlight_duty_max = hw_record.bl_duty_max;
			if(hw_record.bl_duty_init > 0)
				vpd->backlight_duty_init = hw_record.bl_duty_init;
			if(hw_record.bit_depth > 0)
				vpd->bit_depth = hw_record.bit_depth;
			if(hw_record.lvds_option != LVDS_UNKNOWN)
				vpd->lvds = hw_record.lvds_option;
			vpd->backlight_select = hw_record.bl_select;
			/* TODO: Implement import/translation of remaining members of lxk_gen2_panel_hw */
			break;			
		}
		
		case GAMMA_LUT_RED:
		case GAMMA_LUT_GREEN:
		case GAMMA_LUT_BLUE:
		case GAMMA_LUT_COMMON:
		{
			gamma_lut_single lut;

			rc = read_record(&lut, i, &nv);
			if(rc) {
				errline = __LINE__;
				errtext = "record read";
				rc = -1;
				goto done;
			}

			printf("%s:%d: found GAMMA_LUT record type %d\n", __func__, __LINE__, nv.directory[i].type);

			rc = init_gamma_lut(vpd, &lut, nv.directory[i].type);
			if(rc) {
				errline = __LINE__;
				errtext = "gamma";
				rc = -1;
				goto done;
			}
			break;
		}
		case LINUX_DISPLAY_TIMING:
		case EDID_INFO:
		case EDID_DETAILED:
		case RAW:
			/*
			 * Not implemented yet, fall through.
			 */
		default:
			rc = 0;
		}
	}
done:
	if(rc) {
		if(errline > 0) {
			printf("%s: Panel eeprom %s error reported at line %d\n", __FUNCTION__, (errtext ? errtext : ""), errline);
		}
		return NULL;
	}
	else {
		return dynamic;
	}
}

