#include <common.h>
#include <i2c.h>

#define OPERATION       0x1
#define OPERATION_ON_SHIFT 7
#define OPERATION_ON  (1 << OPERATION_ON_SHIFT)
#define OPERATION_MARGIN_SHIFT 2
#define OPERATION_MARGIN_MASK 0x3C
#define OPERATION_VLO_NOFAULT ((5 << OPERATION_MARGIN_SHIFT) & OPERATION_MARGIN_MASK)
#define OPERATION_VHI_NOFAULT ((9 << OPERATION_MARGIN_SHIFT) & OPERATION_MARGIN_MASK)

#define ON_OFF_CONFIG   0x2
#define ON_OFF_OP7     (1 << 3)
#define ON_OFF_EN      (1 << 2)
#define ON_OFF_ALWAYS_SET 0x13

#define VOUT_ADJ        0xD4
#define VOUT_MARGIN     0xD5
#define CUSTOM_REG      0xD0

/* Positive margin percentages */
static const float pos_margin[] = {0, 0.9, 1.8, 2.8, 3.7, 4.7, 5.7, 6.7, 7.7, 8.8, 9.9, 10.9, 12.0};
/* Absolute negative margin percentages */
static const float neg_margin[] = {0, -1.1, -2.1, -3.2, -4.2, -5.2, -6.2, -7.1, -8.1, -9.0, -9.9, -10.7, -11.6};
/* Absolute positive/negative adjustment percentages */
float adj[] = {0, 0.75, 1.5, 2.25, 3.0, 3.75, 4.5, 5.25, 6.0, 6.75, 7.5, 8.25, 9.0};


/* _tps53819_set_operation
 *
 * PRIVATE function to set the OPERATION register
 *
 * chip: i2c addr of 53819
 * val: 8 bit value to write to the OPERATION register
 *
 * RETURNS: rc of i2c_write
 */
static int _tps53819_set_operation(uint8_t chip, uint8_t val)
{
    int rc = 0;
    uchar resp[1];

    resp[0] = val;
    rc = i2c_write(chip, OPERATION, 1, resp, sizeof(resp));
    if (rc)
        printf("%s:%d i2c_write failed", __func__, __LINE__);

    return rc;
}

/* _tps53819_set_on_off_config
 *
 * PRIVATE function to set the ON/OFF CONFIG register
 *
 * chip: i2c addr of 53819
 * val: 8 bit value to write to the ON/OFF CONFIG register
 *
 * RETURNS: rc of i2c_write
 */
static int _tps53819_set_on_off_config(uint8_t chip, uint8_t val)
{
    int rc = 0;
    uchar resp[1];

    resp[0] = val;
    rc = i2c_write(chip, ON_OFF_CONFIG, 1, resp, sizeof(resp));
    if (rc)
        printf("%s:%d i2c_write failed", __func__, __LINE__);

    return rc;
}

/* _tps53819_set_vout_margin
 *
 * PRIVATE function to set the vout_margin register
 *
 * This function uses the following indices to set the LOW and HIGH
 * margin values. NOTE that even if both margins are set to a non-zero
 * percentage, the OPERATION register will force action to either be HIGH
 * or LOW. Indices for respective LOW and HIGH margin percentages are as
 * follows:
 *
 * H_ and L_MARGINS are set by the following values:
   VAL      MARGIN (L)  MARGIN (H)
   12       -11.6%      +12%
   11       -10.7%      +10.9%
   10       -9.9%       +9.9%
   09       -9.0%       +8.8%
   08       -8.1%       +7.7%
   07       -7.1%       +6.7%
   06       -6.2%       +5.7%
   05       -5.2%       +4.7%
   04       -4.2%       +3.7%
   03       -3.2%       +2.8%
   02       -2.1%       +1.8%
   01       -1.1%       +0.9%
   00       -0.0%       +0.0%

  Example: To set the margin to +6.7%, first set the OPERATION
  register to output voltage margin HIGH and run
  _tps53819_set_vout_margin(chip, 0, 7);

  RETURNS: rc of i2c_write
*/
static int _tps53819_set_vout_margin(uint8_t chip, uint8_t l_margin, uint8_t h_margin)
{
    int rc;
    uchar mresp[1];

    if (h_margin > 12)
        h_margin = 12;
    if (l_margin > 12)
        l_margin = 12;

    mresp[0] = ((h_margin << 4) & 0xf0) | (l_margin & 0x0f);

    rc = i2c_write(chip, VOUT_MARGIN, 1, mresp, sizeof(mresp));
    if(rc)
        printf("%s:%d i2c_write failed\n", __func__, __LINE__);

    return rc;
}

/* _tps53819_set_vout_adj
 *
 * PRIVATE function to set the VOUT ADJUSTMENT register.
 *
 * chip: The i2c chip address
 * adj_index: 0 - 12 depending on the desired Adjustment %, as shown below:

ADJ_INDEX    ABS ADJ %
        0        0
        1        .75
        2        1.5
        3        2.25
        4        3
        5        3.75
        6        4.5
        7        5.25
        8        6
        9        6.75
       10        7.5
       11        8.25
       12        9

 * positive: 0 for no, 1 for yes. Will determine if % increase or decrease
 *
 * RETURNS: rc of i2c_write
 */
static int _tps53819_set_vout_adj(uint8_t chip, uint8_t adj_index, uint8_t positive)
{
    int rc;
    uchar mresp[1];

    if(adj_index > 12)
        adj_index = 12;

    if(positive)
        adj_index += 16;
    else
        adj_index = 15 - adj_index;

    mresp[0] = (adj_index & 0x1F);

    rc = i2c_write(chip, VOUT_ADJ, 1, mresp, sizeof(mresp));
    if(rc)
        printf("%s:%d i2c_write failed\n", __func__, __LINE__);

    return rc;
}

/* _tps53819_set_vtrim
 *
 * PRIVATE function to set the output voltage adjustment and margin to match
 * a desired trim value. Assumption is that nominal voltage is 1V, and val
 * is in mV increase/decrease from 1V.
 *
 * This function will find the combination of adjustment and margin percentages
 * that yields the closest match to the desired trim (can be slightly greater
 * than or slightly less than val)
 *
 * chip: i2c addr of 53819
 * val: mV increase or decrease from 1V (e.g., -25 or 20)
 *
 * RETURNS: rc of attempts to set margin and adj
 */
static int _tps53819_set_vtrim(uint8_t chip, int32_t val)
{
    int rc;
    int i, j;
    float min_diff = 10.0;
    float combined_adj;
    int mar_index = 0;
    int adj_index = 0;
    float signed_adj[13];
    const float *margin;
    float abs_diff;
    int pos = (val >= 0);

    /* Use the proper *_margin array depending on whether
     * val is positive or negative, and force val to +.
     * We will need to modify the adj array for sign, so
     * make a copy here
     */

    memmove(signed_adj, adj, sizeof(signed_adj));

    if(pos)
        margin = pos_margin;
    else
    {
        margin = neg_margin;
        for(i = 0; i < ARRAY_SIZE(signed_adj); i++)
            signed_adj[i] *= -1;
    }

    /* 1. Iterate through all margin and adjustment combinations.
     * 2. Evaluate each combos proximity to the desired val
     * 3. Save the indices of the closest matching combo.
     */

    for(i = 0; i < ARRAY_SIZE(pos_margin); i++)
    {
        for(j = 0; j < ARRAY_SIZE(signed_adj); j++)
        {
            combined_adj = (margin[i]/100) + (signed_adj[j]/100) + (margin[i]/100)*(signed_adj[j]/100);

            abs_diff = ((combined_adj*1000) - val);
            if (abs_diff < 0)
                abs_diff *= -1;

            if(abs_diff < min_diff)
            {
                mar_index = i;
                adj_index = j;
                min_diff = abs_diff;
            }
        }
    }

    if(pos)
    {
        /* Use OPERATION Reg to enable Vmargin H */
        rc = _tps53819_set_operation(chip, OPERATION_ON | OPERATION_VHI_NOFAULT);
        if(rc)
            printf("%s:%d set_operation VHI_NOFAULT failed\n", __func__, __LINE__);
        rc = _tps53819_set_vout_margin(chip, 0, mar_index);
    }
    else
    {
        /* Use OPERATION Reg to enable Vmargin L */
        rc = _tps53819_set_operation(chip, OPERATION_ON | OPERATION_VLO_NOFAULT);
        if(rc)
            printf("%s:%d set_operation VLO_NOFAULT failed\n", __func__, __LINE__);
        rc = _tps53819_set_vout_margin(chip, mar_index, 0);
    }

    if(rc)
        printf("%s:%d set_vout_margin failed\n", __func__, __LINE__);

    rc = _tps53819_set_vout_adj(chip, adj_index, pos);
    if(rc)
        printf("%s:%d set_vout_adj failed\n", __func__, __LINE__);

    return rc;
}

/* tps53819_set_vtrim
 *
 * PUBLIC function to set the output voltage adjustment and margin to match
 * a desired trim value. Assumption is that nominal voltage is 1V, and val
 * is in mV increase/decrease from 1V.
 *
 * bus: i2c bus on which 53819 is located
 * chip: i2c addr of 53819
 * val: mV increase or decrease from 1V (e.g., -25 or 20)
 *
 * RETURNS: rc of PRIVATE set_vtrim function
 */

int tps53819_set_vtrim(int bus, uint8_t chip, int32_t val)
{
    int rc;

    I2C_SET_BUS(bus);
    rc = _tps53819_set_vtrim(chip, val);
    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return rc;
}

int tps53819_init(int bus, uint8_t chip) {
    I2C_SET_BUS(bus);

    /* Set the OPERATION<7> bit before using it as enable for switcher */
    _tps53819_set_operation(chip, OPERATION_ON);

    /* Set 53819 to respond to EN pin and OPERATION<7> bit */
    _tps53819_set_on_off_config(chip, ON_OFF_OP7 | ON_OFF_EN | ON_OFF_ALWAYS_SET);

    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return 0;
}
