#include <common.h>
#include <i2c.h>

#define OPERATION       0x1
#define OPERATION_ON_SHIFT 7
#define OPERATION_ON  (1 << OPERATION_ON_SHIFT)

#define ON_OFF_CONFIG   0x2
#define ON_OFF_CONFIG_CMD (1 << 3)

#define VOUT_TRIM       0x22

#define VOUT_SCALE_LOOP 0x29

/* Enables the part in the OPERATION register */
static int _tps40420_enable(uint8_t chip, int enb) {
    int rc;
    uchar resp[1];

    /* Enable the regulator */
    rc = i2c_read(chip, OPERATION, 1, resp, sizeof(resp));
    if (!rc) {
        resp[0] = (resp[0] & ~OPERATION_ON) | (!!enb << OPERATION_ON_SHIFT);

        rc = i2c_write(chip, OPERATION, 1, resp, sizeof(resp));
        if (rc)
            printf("%s:%d i2c_write failed", __func__, __LINE__);
    }
    else {
        printf("%s:%d i2c_read failed", __func__, __LINE__);
    }

    return rc;
}

int tps40420_enable(int bus, uint8_t chip, int enb) {
    int rc;

    I2C_SET_BUS(bus);
    rc = _tps40420_enable(chip, enb);
    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return rc;
}

/* Set the VOUT_SCALE_LOOP */
static int _tps40420_set_vsl(uint8_t chip, uint16_t val) {
    int rc;
    uchar loop_resp[2];

    loop_resp[0] = val & 0xff;
    loop_resp[1] = (val >> 8) & 0xff;

    rc = i2c_write(chip, VOUT_SCALE_LOOP, 1, loop_resp, sizeof(loop_resp));
    if (rc) {
        printf("%s:%d i2c_write failed", __func__, __LINE__);
    }

    return rc;
}

#if 0
static int _tps40420_set_config(uint8_t chip) {
    int rc = 0;
    uchar on_off_resp[1];

    rc = i2c_read(chip, ON_OFF_CONFIG, 1, on_off_resp, sizeof(on_off_resp));
    if (!rc) {
        if (!(on_off_resp[0] & ON_OFF_CONFIG_CMD)) {
            on_off_resp[0] |= ON_OFF_CONFIG_CMD;
            rc = i2c_write(chip, ON_OFF_CONFIG, 1, on_off_resp, sizeof(on_off_resp));
            if (rc) {
                printf("%s:%d i2c_write failed", __func__, __LINE__);
            }
        }
    }
    else {
        printf("%s:%d i2c_read failed", __func__, __LINE__);
    }

    return rc;
}
#endif

static int _tps40420_get_vtrim(uint8_t chip, int32_t *val) {
    int rc;
    uchar mresp[2];

    rc = i2c_read(chip, VOUT_TRIM, 1, mresp, sizeof(mresp));
    if ((!rc) && val) {
        *val = (mresp[1] << 8) | mresp[0];

        /* Add the sign extension */
        if (*val & (1 << 15))
            *val |= 0xffff0000;

        *val = (*val * 1000) >> 10;

        /* cap val at +/-10% */
        if (*val > 100)
            *val = 100;
        else if (*val < -100)
            *val = -100;
    }
    else {
        printf("%s:%d i2c_write failed", __func__, __LINE__);
    }

    return rc;
}

int tps40420_get_vtrim(int bus, uint8_t chip, int32_t *val) {
    int rc;

    I2C_SET_BUS(bus);
    rc = _tps40420_get_vtrim(chip, val);
    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return rc;
}

/* val is the percentage of nominal to trim, multiplied by 1000 */
static int _tps40420_set_vtrim(uint8_t chip, int32_t val) {
    int rc;
    int pos = (val >= 0) ? 1 : 0;
    uchar mresp[2];

    /* cap val at +/-10% */
    if (val > 100)
        val = 100;
    else if (val < -100)
        val = -100;

    /* Convert to VOUT_TRIM */
    val = (val << 10) / 1000;
    if (pos)
        ++val;

    mresp[0] = val & 0xff;
    mresp[1] = (val >> 8) & 0xff;

    rc = i2c_write(chip, VOUT_TRIM, 1, mresp, sizeof(mresp));
    if (rc) {
        printf("%s:%d i2c_write failed", __func__, __LINE__);
    }

    return rc;
}

int tps40420_set_vtrim(int bus, uint8_t chip, int32_t val) {
    int rc;

    I2C_SET_BUS(bus);
    rc = _tps40420_set_vtrim(chip, val);
    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return rc;
}

int tps40420_init(int bus, uint8_t chip) {
    int rc = 0;

    uchar on_off_resp[1];

    I2C_SET_BUS(bus);

    rc = i2c_read(chip, OPERATION, 1, on_off_resp, sizeof(on_off_resp));
    if (!rc) {
        /*
         *  Make sure that the regulator is enabled in the OPERATION register
         *  before telling it to look at that register for turning on and off
         */
        _tps40420_set_vsl(chip, 0xb933);
//        _tps40420_set_config(chip);
    }

    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return rc;
}
