/*
 * drivers/mtd/nand/mv61x0_nand.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <common.h>
#include <errno.h>
#include <malloc.h>
#include <linux/types.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#if 0
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/clk.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/slab.h>
#endif

extern int mtd_get_partition_info_ssp(struct mtd_info *mtd, struct mtd_partition **partition_table);
#define module_param(name, type, perm)
#define MODULE_PARM_DESC(_parm, desc)
#define dev_err(dev, fmt, args...)  printk(fmt, ## args)
#define dump_stack()           do { } while (0)
#define print_hex_dump(...)    do { } while (0)
#define msleep(a) udelay(a * 1000)

enum irqreturn {
    IRQ_NONE,
    IRQ_HANDLED,
    IRQ_WAKE_THREAD,
};
typedef enum irqreturn irqreturn_t;

static irqreturn_t mv61x0_nand_irq(int irq, void *devid);

struct completion {
    int complete;
};

int init_completion(struct completion *comp) {
    comp->complete = 0;
    return 0;
}

int complete(struct completion *comp) {
    comp->complete = 1;
    return 0;
}

struct mv61x0_nand_info;
int wait_for_completion_timeout(struct mv61x0_nand_info *info, struct completion *comp, int timeout) {
    while (!comp->complete) {
        if (timeout-- == 0)
            break;
        mv61x0_nand_irq(0, info);
        udelay(1000);
    }

    if (timeout == 0)
        return 0;

    comp->complete = 0;
    return 1;
}

#define HZ (1000)
#define	CHIP_DELAY_TIMEOUT	(2 * HZ/10)

/* registers and bit definitions */
#define NDCR		(0x00) /* Control register */
#define NDTR0CS0	(0x04) /* Timing Parameter 0 for CS0 */
#define NDTR1CS0	(0x0C) /* Timing Parameter 1 for CS0 */
#define NDSR		(0x14) /* Status Register */
#define NDPCR		(0x18) /* Page Count Register */
#define NDBDR0		(0x1C) /* Bad Block Register 0 */
#define NDBDR1		(0x20) /* Bad Block Register 1 */
#define NDREDEL		(0x24) /* mv61 Read Enable Return Delay Register */
#define NDECCCTRL	(0x28) /* mv61 ECC Control Register */
#define NDBZCNT		(0x2C) /* mv61 Timer for ND_RnB0 and ND_RnB1 */
#define NDMUTEX		(0x30) /* mv61 NAND Controller mutex lock */
#define NDCMDMAT1	(0x34) /* mv61 Partition Command Match Register */
#define NDCMDMAT2	(0x38) /* mv61 Partition Command Match Register */
#define NDCMDMAT3	(0x3C) /* mv61 Partition Command Match Register */
#define NDDB		(0x40) /* Data Buffer */
#define NDCB0		(0x48) /* Command Buffer0 */
#define NDCB1		(0x4C) /* Command Buffer1 */
#define NDCB2		(0x50) /* Command Buffer2 */
#define NDCB3		(0x54) /* mv61 Command Buffer3 */
#define NDARBCR		(0x5C) /* mv61 NAND DFI Arbitration Control Register */
/* mv61 NDPT{0,1,2,3}CS{0,1} go here */

#define NDCR_SPARE_EN		(0x1 << 31)
#define NDCR_ECC_EN		(0x1 << 30)
#define NDCR_DMA_EN		(0x1 << 29)
#define NDCR_ND_RUN		(0x1 << 28)
#define NDCR_DWIDTH_C		(0x1 << 27)
#define NDCR_DWIDTH_M		(0x1 << 26)
#define NDCR_PAGE_SZ		(0x1 << 24)
#define NDCR_SEQ_DIS		(0x1 << 23)
#define NDCR_ND_MODE		(0x3 << 21)
#define NDCR_NAND_MODE   	(0x0)
#define NDCR_CLR_PG_CNT		(0x1 << 20)
#define NDCR_CLR_ECC		(0x1 << 19)
#define NDCR_RD_ID_CNT_MASK	(0x7 << 16)
#define NDCR_RD_ID_CNT(x)	(((x) << 16) & NDCR_RD_ID_CNT_MASK)

#define NDCR_RA_START		(0x1 << 15)
#define NDCR_PG_PER_BLK		(0x1 << 14)
#define NDCR_ND_ARB_EN		(0x1 << 12)

#define NDSR_MASK		(0x9fff)
#define NDSR_ERR_CNT_MASK       (0x1f << 16)
#define NDSR_ERR_CNT_VALUE(ndsr) ( ((ndsr) & NDSR_ERR_CNT_MASK) >> 16 )
#define NDSR_RDY		(0x1 << 11)
#define NDSR_CS0_PAGED		(0x1 << 10)
#define NDSR_CS1_PAGED		(0x1 << 9)
#define NDSR_CS0_CMDD		(0x1 << 8)
#define NDSR_CS1_CMDD		(0x1 << 7)
#define NDSR_CS0_BBD		(0x1 << 6)
#define NDSR_CS1_BBD		(0x1 << 5)
#define NDSR_UNCERR		(0x1 << 4)
#define NDSR_CORERR		(0x1 << 3)
#define NDSR_WRDREQ		(0x1 << 2)
#define NDSR_RDDREQ		(0x1 << 1)
#define NDSR_WRCMDREQ		(0x1)

#define NDCB0_AUTO_RS		(0x1 << 25)
#define NDCB0_CSEL		(0x1 << 24)
#define NDCB0_CMD_TYPE_MASK	(0x7 << 21)
#define NDCB0_CMD_TYPE(x)	(((x) << 21) & NDCB0_CMD_TYPE_MASK)
#define NDCB0_NC		(0x1 << 20)
#define NDCB0_DBC		(0x1 << 19)
#define NDCB0_ADDR_CYC_MASK	(0x7 << 16)
#define NDCB0_ADDR_CYC(x)	(((x) << 16) & NDCB0_ADDR_CYC_MASK)
#define NDCB0_CMD2_MASK		(0xff << 8)
#define NDCB0_CMD1_MASK		(0xff)
#define NDCB0_ADDR_CYC_SHIFT	(16)

/* error code and state */
enum {
	ERR_NONE	= 0,
	ERR_CORERR	= -1,
	ERR_UNCERR	= -2,
	ERR_SENDCMD	= -3,
	ERR_BBERR	= -4,
};

enum {
	STATE_READY	= 0,
	STATE_CMD_HANDLE,
	STATE_PIO_READING,
	STATE_PIO_WRITING,
};

struct mv61x0_nand_cmdset {
	uint16_t	read1;
	uint16_t	read2;
	uint16_t	program;
	uint16_t	read_status;
	uint16_t	read_id;
	uint16_t	erase;
	uint16_t	reset;
	uint16_t	lock;
	uint16_t	unlock;
	uint16_t	lock_status;
};

struct mv61x0_nand_info {
	struct nand_chip	nand_chip;
	const struct mv61x0_nand_cmdset *cmdset;

//	struct platform_device	 *pdev;

//	struct clk		*clk;
	void __iomem		*mmio_base;
//	unsigned long		mmio_phys;

	unsigned int 		buf_start;
	unsigned int		buf_count;

	unsigned char		*data_buf;
	size_t			data_buf_size;

	uint32_t		reg_ndcr;

	/* saved page_addr during CMD_SEQIN */
	int			seqin_page_addr;

	/* relate to the command */
	unsigned int		state;

	int			use_ecc;	/* use HW ECC ? */

	size_t			data_size;	/* data size in FIFO */
	int 			retcode;
	struct completion 	cmd_complete;

	/* generated NDCBx register values */
	uint32_t		ndcb0;
	uint32_t		ndcb1;
	uint32_t		ndcb2;

	/* Nand chip config info */
	int		read_id_bytes;

	unsigned int	col_addr_cycles;
	unsigned int	row_addr_cycles;

	unsigned int page_per_block;  /* (PG_PER_BLK) */
	unsigned int flash_width;     /* Width of Flash memory (DWIDTH_M) */
	unsigned int dfc_width;       /* Width of flash controller(DWIDTH_C) */
	unsigned int num_blocks;      /* Number of physical blocks in Flash */
	unsigned int chip_id;

	/* Bytes of oob accessible to us when hwecc is ON. This is not the same
	 * as mtd->oobavail, because we don't make all of the accessible oob
	 * available to users */
	unsigned int accessible_oob;

	struct mtd_info mtd;
};

static const struct mv61x0_nand_cmdset largepage_cmdset = {
	.read1		= 0x3000,
	.read2		= 0x0050,
	.program	= 0x1080,
	.read_status	= 0x0070,
	.read_id	= 0x0090,
	.erase		= 0xD060,
	.reset		= 0x00FF,
	.lock		= 0x002A,
	.unlock		= 0x2423,
	.lock_status	= 0x007A,
};

static int enable_ecc = 1;
module_param(enable_ecc, bool, 0600);
MODULE_PARM_DESC(enable_ecc, "enable BCH ECC.");

enum {
	TRACE_READ =  (1 << 0),
	TRACE_WRITE = (1 << 1),
	TRACE_ERASE = (1 << 2),

	TRACE_CMD =   (1 << 4),
	TRACE_IRQ =   (1 << 5),
	TRACE_STACK = (1 << 6),

	TRACE_REGS =  (1 << 8),
	TRACE_REGS_DATA = (1 << 9),
};
uint trace_mode = 0;
module_param(trace_mode, uint, 0600);
MODULE_PARM_DESC(trace_mode, "Bit mask: 0x1-reads, 0x2-writes, 0x4-erases, "
		 "0x10 all cmdfunc, 0x20, IRQ, 0x40-show stacks, "
		 "0x100 regs (except data), 0x200 DATA regs");

#define DEBUG_REGS 0

#if !DEBUG_REGS
/* macros for registers read/write */
#define nand_writel(info, off, val)	\
	__raw_writel((val), (info)->mmio_base + (off))

#define nand_readl(info, off)		\
	__raw_readl((info)->mmio_base + (off))
#else

static const char * const reg_names[] = {
	"NDCR",
	"NDTR0CS0",
	"NDTR0CS1",
	"NDTR1CS0",
	"NDTR1CS1",
	"NDSR",
	"NDPCR",
	"NDBDR0",
	"NDBDR1",
	"NDREDEL",
	"NDECCCTRL",
	"NDBZCNT",
	"NDMUTEX",
	"NDCMDMAT1",
	"NDCMDMAT2",
	"NDCMDMAT3",
	"NDDB",
	"NDDB1",
	"NDCB0",
	"NDCB1",
	"NDCB2",
	"NDCB3",
	"unk54",
	"NDARBCR",
};

void nand_writel(struct mv61x0_nand_info *info, int off, uint32_t val)
{
	if(TRACE_REGS & trace_mode)
		printk(" W: %10.10s 0x%8.8x\n", reg_names[off/4], val);
	__raw_writel((val), (info)->mmio_base + (off));
}

uint32_t nand_readl(struct mv61x0_nand_info *info, int off)
{
	uint32_t val = __raw_readl((info)->mmio_base + (off));
	if(TRACE_REGS & trace_mode)
		printk("R : %10.10s 0x%8.8x\n", reg_names[off/4], val);
	return val;
}

static void __debug_raw_readsl(const void __iomem *addr, void *datap, int len)
{
	if((TRACE_REGS_DATA & trace_mode) || 
		((TRACE_REGS & trace_mode) && len <= 4)) {
		uint32_t *dest = (uint32_t *)datap;
		uint32_t val;

		for(; len != 0; --len) {
			val = __raw_readl(addr);	
			*dest++ = val;
			printk("R :       0x%2.2x 0x%8.8x\n",
			       (uint32_t)addr & 0xff, val);
		}
	} else {
		__raw_readsl((unsigned int)addr, datap, len);
	}
}
#define __raw_readsl __debug_raw_readsl

#endif

#define NDTR0_tCH(c)	(min((c), 7) << 19)
#define NDTR0_tCS(c)	(min((c), 7) << 16)
#define NDTR0_tWH(c)	(min((c), 7) << 11)
#define NDTR0_tWP(c)	(min((c), 7) << 8)
#define NDTR0_tRH(c)	(min((c), 7) << 3)
#define NDTR0_tRP(c)	(min((c), 7) << 0)

#define NDTR1_tR(c)	(min((c), 65535) << 16)
#define NDTR1_tWHR(c)	(min((c), 15) << 4)
#define NDTR1_tAR(c)	(min((c), 15) << 0)

#define tCH_NDTR0(r)	(((r) >> 19) & 0x7)
#define tCS_NDTR0(r)	(((r) >> 16) & 0x7)
#define tWH_NDTR0(r)	(((r) >> 11) & 0x7)
#define tWP_NDTR0(r)	(((r) >> 8) & 0x7)
#define tRH_NDTR0(r)	(((r) >> 3) & 0x7)
#define tRP_NDTR0(r)	(((r) >> 0) & 0x7)

#define tR_NDTR1(r)	(((r) >> 16) & 0xffff)
#define tWHR_NDTR1(r)	(((r) >> 4) & 0xf)
#define tAR_NDTR1(r)	(((r) >> 0) & 0xf)

/* convert nano-seconds to nand flash controller clock cycles */
#define ns2cycle(ns, clk)	(int)(((ns) * (clk / 1000000) / 1000) - 1)

/* convert nand flash controller clock cycles to nano-seconds */
#define cycle2ns(c, clk)	((((c) + 1) * 1000000 + clk / 500) / (clk / 1000))

#define WAIT_EVENT_TIMEOUT	10

static int wait_for_event(struct mv61x0_nand_info *info, uint32_t event)
{
	int timeout = WAIT_EVENT_TIMEOUT;
	uint32_t ndsr;

	while (timeout--) {
		ndsr = nand_readl(info, NDSR) & NDSR_MASK;
		if (ndsr & event) {
			nand_writel(info, NDSR, ndsr & event);
			return 0;
		}
		udelay(10);
	}

	return -ETIMEDOUT;
}

static int prepare_read_prog_cmd(struct mv61x0_nand_info *info,
			uint16_t cmd, int column, int page_addr)
{
	int bch_en = 1;

	/* calculate data size */
	switch (info->mtd.writesize) {
	case 2048:
		info->data_size = (info->use_ecc) ? 2048+64 - 32 : 2048+64;
		break;
	case 512:
		info->data_size = (info->use_ecc) ? 520 : 528;
		bch_en = 0;
		break;
	default:
		return -EINVAL;
	}

	/* generate values for NDCBx registers */
	info->ndcb0 = cmd | ((cmd & 0xff00) ? NDCB0_DBC : 0);
	info->ndcb1 = 0;
	info->ndcb2 = 0;
	info->ndcb0 |= NDCB0_ADDR_CYC(info->row_addr_cycles + info->col_addr_cycles);

	if (info->col_addr_cycles == 2) {
		/* large block, 2 cycles for column address
		 * row address starts from 3rd cycle
		 */
		info->ndcb1 |= page_addr << 16;
		if (info->row_addr_cycles == 3)
			info->ndcb2 = (page_addr >> 16) & 0xff;
	} else
		/* small block, 1 cycles for column address
		 * row address starts from 2nd cycle
		 */
		info->ndcb1 = page_addr << 8;

	if (cmd == info->cmdset->program)
		info->ndcb0 |= NDCB0_CMD_TYPE(1) | NDCB0_AUTO_RS;

	return 0;	
}

static int prepare_erase_cmd(struct mv61x0_nand_info *info,
			uint16_t cmd, int page_addr)
{
	info->ndcb0 = cmd | ((cmd & 0xff00) ? NDCB0_DBC : 0);
	info->ndcb0 |= NDCB0_CMD_TYPE(2) | NDCB0_AUTO_RS | NDCB0_ADDR_CYC(info->row_addr_cycles);
	info->ndcb1 = page_addr;
	info->ndcb2 = 0;
	info->data_size = 0;
	return 0;
}

static int prepare_read_status_cmd(struct mv61x0_nand_info *info, uint16_t cmd)
{
	info->ndcb0 = cmd | ((cmd & 0xff00) ? NDCB0_DBC : 0) | NDCB0_CMD_TYPE(4);
	info->ndcb1 = 0;
	info->ndcb2 = 0;
	info->data_size = 8;
	return 0;
}

static int prepare_read_id_cmd(struct mv61x0_nand_info *info, uint16_t cmd)
{
	info->ndcb0 = cmd | ((cmd & 0xff00) ? NDCB0_DBC : 0) | NDCB0_CMD_TYPE(3);
	info->ndcb0 |= NDCB0_ADDR_CYC(1);  // Read ID requires 1 addr cycle.
	info->ndcb1 = 0;
	info->ndcb2 = 0;
	info->data_size = 8;
	return 0;
}

static int prepare_reset_cmd(struct mv61x0_nand_info *info, uint16_t cmd)
{
	info->ndcb0 = cmd | ((cmd & 0xff00) ? NDCB0_DBC : 0) | NDCB0_CMD_TYPE(5);
	info->ndcb1 = 0;
	info->ndcb2 = 0;
	info->data_size = 0;
	return 0;
}

static void enable_int(struct mv61x0_nand_info *info, uint32_t int_mask)
{
	uint32_t ndcr;

	ndcr = nand_readl(info, NDCR);
	nand_writel(info, NDCR, ndcr & ~int_mask);
}

static void disable_int(struct mv61x0_nand_info *info, uint32_t int_mask)
{
	uint32_t ndcr;

	ndcr = nand_readl(info, NDCR);
	nand_writel(info, NDCR, ndcr | int_mask);
}

/* NOTE: it is a must to set ND_RUN firstly, then write command buffer
 * otherwise, it does not work
 */
static int write_cmd(struct mv61x0_nand_info *info)
{
	uint32_t ndcr;

	/* clear status bits and run */
	nand_writel(info, NDSR, NDSR_MASK | NDSR_ERR_CNT_MASK);
	ndcr = nand_readl(info, NDCR);
	nand_writel(info, NDCR, ndcr & ~NDCR_ND_RUN);
	ndcr = nand_readl(info, NDSR);  //TODO: remove this debug read 
	nand_writel(info, NDECCCTRL, info->use_ecc ? 1 + (3 << 1) : 0); //FIXME: use macros

	ndcr = info->reg_ndcr;

	ndcr |= info->use_ecc ? NDCR_ECC_EN : 0;
	//ndcr |= info->use_dma ? NDCR_DMA_EN : 0;
	ndcr |= NDCR_ND_RUN;

	nand_writel(info, NDCR, ndcr);

	// Fix problem where the NFC asserts UNCERR and sets NDSR.ND_ERR_CNT to 0x1f
	// right off the bat, which makes no sense.
	ndcr = nand_readl(info, NDSR);  //TODO: remove this debug read 
	nand_writel(info, NDSR, NDSR_ERR_CNT_MASK | NDSR_UNCERR);

	if (wait_for_event(info, NDSR_WRCMDREQ)) {
		printk(KERN_ERR "timed out writing command\n");
		return -ETIMEDOUT;
	}

	nand_writel(info, NDCB0, info->ndcb0);
	nand_writel(info, NDCB0, info->ndcb1);
	nand_writel(info, NDCB0, info->ndcb2);
	return 0;
}

static int handle_data_pio(struct mv61x0_nand_info *info)
{
	int ret, timeout = CHIP_DELAY_TIMEOUT;

	switch (info->state) {
	case STATE_PIO_WRITING:
#if DEBUG_REGS
		if(trace_mode & TRACE_REGS)
			printk(" W: %10.10s Writing %d words\n",
				reg_names[NDDB/4],
				DIV_ROUND_UP(info->data_size, 4));
#endif
		__raw_writesl((unsigned int)(info->mmio_base + NDDB), info->data_buf,
				DIV_ROUND_UP(info->data_size, 4));

		enable_int(info, NDSR_CS0_BBD | NDSR_CS0_CMDD);

		ret = wait_for_completion_timeout(info, &info->cmd_complete, timeout);
		if (!ret) {
			printk(KERN_ERR "program command time out\n");
			return -1;
		}
		break;
	case STATE_PIO_READING:
#if DEBUG_REGS
		if(trace_mode & TRACE_REGS)
			printk("R : %10.10s Reading %d words\n",
				reg_names[NDDB/4],
				DIV_ROUND_UP(info->data_size, 4));
#endif
		__raw_readsl((unsigned int)(info->mmio_base + NDDB), info->data_buf,
				DIV_ROUND_UP(info->data_size, 4));
		break;
	default:
		printk(KERN_ERR "%s: invalid state %d\n", __func__,
				info->state);
		return -EINVAL;
	}

	info->state = STATE_READY;
	return 0;
}

static irqreturn_t mv61x0_nand_irq(int irq, void *devid)
{
	struct mv61x0_nand_info *info = devid;
	unsigned int status;

	if(TRACE_IRQ & trace_mode)
		printk("Enter mv61x0_nand_irq\n");
		
	status = nand_readl(info, NDSR);

	if (status & (NDSR_RDDREQ)) {
		if (status & NDSR_UNCERR)
			info->retcode = ERR_UNCERR;
		else if (status & NDSR_CORERR)
			info->retcode = ERR_CORERR;

		disable_int(info, NDSR_RDDREQ | NDSR_UNCERR | NDSR_CORERR);

		info->state = STATE_PIO_READING;
		complete(&info->cmd_complete);
	} else if (status & NDSR_WRDREQ) {
		disable_int(info, NDSR_WRDREQ);
		info->state = STATE_PIO_WRITING;
		complete(&info->cmd_complete);
	} else if (status & (NDSR_CS0_BBD | NDSR_CS0_CMDD)) {
		if (status & NDSR_CS0_BBD)
			info->retcode = ERR_BBERR;

		disable_int(info, NDSR_CS0_BBD | NDSR_CS0_CMDD);
		info->state = STATE_READY;
		complete(&info->cmd_complete);
	}
	nand_writel(info, NDSR, status);
	if(TRACE_IRQ & trace_mode)
		printk("Exit mv61x0_nand_irq\n");
		
	return IRQ_HANDLED;
}

static int mv61x0_nand_do_cmd(struct mv61x0_nand_info *info, uint32_t event)
{
	uint32_t ndcr;
	uint32_t ndsr;
	int ret, timeout = CHIP_DELAY_TIMEOUT;

	if (write_cmd(info)) {
		info->retcode = ERR_SENDCMD;
		goto fail_stop;
	}

	info->state = STATE_CMD_HANDLE;

	enable_int(info, event);

	ret = wait_for_completion_timeout(info, &info->cmd_complete, timeout);
	if (!ret) {
		printk(KERN_ERR "command execution timed out\n");
		info->retcode = ERR_SENDCMD;
		goto fail_stop;
	}

	if (info->data_size > 0) {
		if (handle_data_pio(info))
			goto fail_stop;
	}

	ndsr = nand_readl(info, NDSR);
	nand_writel(info, NDSR, ndsr);  // clear all the status bits
	if(ndsr & NDSR_UNCERR)
		info->retcode = ERR_UNCERR;
	else if(ndsr & NDSR_CORERR)
		info->retcode = ERR_CORERR;

	return 0;

fail_stop:
	ndcr = nand_readl(info, NDCR);
	nand_writel(info, NDCR, ndcr & ~NDCR_ND_RUN);
	udelay(10);
	info->retcode = ERR_SENDCMD;
	return -ETIMEDOUT;
}

static int mv61x0_nand_dev_ready(struct mtd_info *mtd)
{
	return 0;
}

/* The BCH syndrome for a blank page is not, unfortunately all 0xFF's. So, so a blank
 * page read will cause the hardware to declare "Uncorrectable ECC error". Here is
 * our algorithm for handling this situation without having to re-read the page, while
 * allowing for the possibility that a legitimately blank page may have several bits
 * stuck that are normally corrected by BCH:
 * 1. Any time we write a page, always add a pattern containing exactly 32 zero-bits
 *    to the spare area. This is included in the data protected by BCH.
 * 2. After a failed read, this function will count the total number of zero bits
 *    in the whole page. If it is less than 16, caller will consider this a blank pg.
 * 3. Otherwise, treat it like a real read failure.
 */
static inline int count_zero_bits(uint8_t *buf, unsigned int len)
{
	uint32_t *lbuf = (uint32_t *)buf;
	int llen = (int)len / 4;
	int zero_bits = 0;

	// Buf must be 4-byte aligned and sized
	BUG_ON( (len & 0x3) != 0 || ((unsigned int)buf & 0x3) != 0);

	while(llen) {
		zero_bits += 32 - hweight32(*lbuf);
		--llen;
		++lbuf;
	}
	return zero_bits;
}

static inline const char *nand_cmd_to_str(unsigned cmd)
{
	const char *str;

	switch(cmd) {
		case NAND_CMD_READ0:         str = "CMD_READ0";        break;
		case NAND_CMD_READ1:         str = "CMD_READ1";        break;
		case NAND_CMD_RNDOUT:        str = "CMD_RNDOUT";       break;
		case NAND_CMD_PAGEPROG:      str = "CMD_PAGEPROG";     break;
		case NAND_CMD_READOOB:       str = "CMD_READOOB";      break;
		case NAND_CMD_ERASE1:        str = "CMD_ERASE1";       break;
		case NAND_CMD_STATUS:        str = "CMD_STATUS";       break;
		case NAND_CMD_STATUS_MULTI:  str = "CMD_STATUS_MULTI"; break;
		case NAND_CMD_SEQIN:         str = "CMD_SEQIN";        break;
		case NAND_CMD_RNDIN:         str = "CMD_RNDIN";        break;
		case NAND_CMD_READID:        str = "CMD_READID";       break;
		case NAND_CMD_ERASE2:        str = "CMD_ERASE2";       break;
		case NAND_CMD_RESET:         str = "CMD_RESET";        break;

//		case NAND_CMD_LOCK:          str = "CMD_LOCK";         break;
//		case NAND_CMD_UNLOCK1:       str = "CMD_UNLOCK1";      break;
//		case NAND_CMD_UNLOCK2:       str = "CMD_UNLOCK2";      break;

/* Extended commands for large page devices */
		case NAND_CMD_READSTART:     str = "CMD_READSTART";    break;
		case NAND_CMD_RNDOUTSTART:   str = "CMD_RNDOUTSTART";  break;
		case NAND_CMD_CACHEDPROG:    str = "CMD_CACHEDPROG";   break;
		default: str = "INVALID";
	}
	return(str);
}

static void mv61x0_nand_cmdfunc(struct mtd_info *mtd, unsigned command,
				int column, int page_addr)
{
	struct nand_chip *chip = mtd->priv;
	struct mv61x0_nand_info *info = chip->priv;
	int rc;

	if(trace_mode & TRACE_CMD) {
		printk("mv61x0_nand: %s %d %d\n", nand_cmd_to_str(command),
			column, page_addr);
		if(trace_mode & TRACE_STACK) {
			dump_stack();
		}
	}

	info->state = STATE_READY;
	init_completion(&info->cmd_complete);

	switch(command) {
		case NAND_CMD_ERASE1:
			info->retcode = ERR_NONE;
			info->use_ecc = 0;
			info->buf_start = 0;
			info->buf_count = 0;

			if (prepare_erase_cmd(info,
			    info->cmdset->erase, page_addr))
				break;

			mv61x0_nand_do_cmd(info, NDSR_CS0_BBD | NDSR_CS0_CMDD);
			break;
		case NAND_CMD_ERASE2:
			break;
		case NAND_CMD_READID: {
			info->use_ecc = 0;
			info->buf_start = 0;
			info->buf_count = 4;

			if (prepare_read_id_cmd(info,
			    info->cmdset->read_id))
				break;

			mv61x0_nand_do_cmd(info, NDSR_RDDREQ);
			break;
		}
		case NAND_CMD_STATUS: {
			info->use_ecc = 0;
			info->buf_start = 0;
			info->buf_count = 1;

			if (prepare_read_status_cmd(info,
			    info->cmdset->read_status))
				break;

			mv61x0_nand_do_cmd(info, NDSR_RDDREQ);
			break;
		}
		case NAND_CMD_READOOB: {
			/* disable HW ECC to get all the OOB data */
			info->use_ecc = 0;
			info->buf_count = mtd->writesize + mtd->oobsize;
			info->buf_start = mtd->writesize + column;
			memset(info->data_buf, 0xFF, info->buf_count);

			if (prepare_read_prog_cmd(info, info->cmdset->read1, column, page_addr))
				break;

			mv61x0_nand_do_cmd(info, NDSR_RDDREQ | NDSR_UNCERR | NDSR_CORERR);
			break;
		}
		case NAND_CMD_READ0: {
			int raw_count;
			info->retcode = ERR_NONE;
			info->use_ecc = enable_ecc;
			raw_count = mtd->writesize + 
				(info->use_ecc ? info->accessible_oob : mtd->oobsize);
			info->buf_start = column;
			info->buf_count = raw_count - column;
			memset(info->data_buf, 0xFF, raw_count);

			if (prepare_read_prog_cmd(info, info->cmdset->read1,
			    0, page_addr))
				break;

			rc = mv61x0_nand_do_cmd(info, NDSR_RDDREQ | NDSR_UNCERR | NDSR_CORERR);
			if(ERR_UNCERR == info->retcode) {
				/* Do out blank page detection algorithm */
				int zero_bits = count_zero_bits(info->data_buf,
					raw_count);

				if(zero_bits < (mtd->writesize == 2048 ? 16 : 4)) {
					if(zero_bits != 0) // "correct" a few bit flips
						memset(info->data_buf, 0xFF,
							raw_count);
					info->retcode = ERR_NONE;
				}
			}
			break;
		}
		case NAND_CMD_SEQIN:
			info->buf_start = column;
			info->buf_count = mtd->writesize + mtd->oobsize;
			memset(info->data_buf, 0xff, info->buf_count);

			/* save page_addr for next CMD_PAGEPROG */
			info->seqin_page_addr = page_addr;
			break;
		case NAND_CMD_PAGEPROG:
			info->retcode = ERR_NONE;
			info->use_ecc = enable_ecc;

			/* As part of our blank-page detect, we must zero out some
			 * bits in the oob data any time we write. */
			if(info->use_ecc) {
				if(mtd->writesize == 2048)
					memcpy(info->data_buf + mtd->writesize + 24, 
						"programz", 8);
				else
					info->data_buf[mtd->writesize + 7] = '\0';
			}
			if (prepare_read_prog_cmd(info, info->cmdset->program,
						  0, info->seqin_page_addr))
				break;

			mv61x0_nand_do_cmd(info, NDSR_WRDREQ);
			break;
		case NAND_CMD_RESET: {
			int ret;

			if (prepare_reset_cmd(info, info->cmdset->reset))
				break;

			ret = mv61x0_nand_do_cmd(info, NDSR_CS0_CMDD);
			if (ret == 0) {
				int timeout = 2;
				uint32_t ndcr;

				while (timeout--) {
					if (nand_readl(info, NDSR) & NDSR_RDY)
						break;
					msleep(10);
				}

				ndcr = nand_readl(info, NDCR);
				nand_writel(info, NDCR, ndcr & ~NDCR_ND_RUN);
			}
			break;
		}
		default:
			printk(KERN_ERR "mv61x0_nand: unknown command %x.\n", command);
			break;
	}
}

static uint8_t mv61x0_nand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct mv61x0_nand_info *info = chip->priv;
	char retval = 0xFF;

	if (info->buf_start < info->buf_count)
		/* Has just send a new command? */
		retval = info->data_buf[info->buf_start++];

	return retval;
}

static u16 mv61x0_nand_read_word(struct mtd_info *mtd)
{
	struct nand_chip *chip = mtd->priv;
	struct mv61x0_nand_info *info = chip->priv;
	u16 retval = 0xFFFF;

	if (!(info->buf_start & 0x01) && info->buf_start < info->buf_count) {
		retval = *((u16 *)(info->data_buf+info->buf_start));
		info->buf_start += 2;
	}
	return retval;
}

static void mv61x0_nand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct mv61x0_nand_info *info = chip->priv;
	int real_len = min_t(int, len, info->buf_count - info->buf_start);

	memcpy(buf, info->data_buf + info->buf_start, real_len);
	info->buf_start += real_len;
}

static void mv61x0_nand_write_buf(struct mtd_info *mtd,
		const uint8_t *buf, int len)
{
	struct nand_chip *chip = mtd->priv;
	struct mv61x0_nand_info *info = chip->priv;
	int real_len = min_t(int, len, info->buf_count - info->buf_start);

	memcpy(info->data_buf + info->buf_start, buf, real_len);
	info->buf_start += real_len;
}

static int mv61x0_nand_verify_buf(struct mtd_info *mtd,
		const uint8_t *buf, int len)
{
	return 0;
}

static void mv61x0_nand_select_chip(struct mtd_info *mtd, int chip)
{
	return;
}

/* By the time this function is called, there is no actual waiting to do,
 * but we just return status about whether the write or erase failed.
 */
static int mv61x0_nand_waitfunc(struct mtd_info *mtd, struct nand_chip *this)
{
	struct mv61x0_nand_info *info = mtd->priv;

	if (this->state == FL_WRITING || this->state == FL_ERASING) {
		if (info->retcode == ERR_NONE)
			return 0;
		else {
			return 0x01;  // failure
		}
	}

	return 0;
}

static void mv61x0_nand_ecc_hwctl(struct mtd_info *mtd, int mode)
{
	return;
}

static int mv61x0_nand_ecc_calculate(struct mtd_info *mtd,
		const uint8_t *dat, uint8_t *ecc_code)
{
	return 0;
}

/* Our hardware actually does the ECC correction. However, this function
 * is used to report correctable or uncorrectable ECC errors on read
 */
static int mv61x0_nand_ecc_correct(struct mtd_info *mtd,
		uint8_t *dat, uint8_t *read_ecc, uint8_t *calc_ecc)
{
	struct mv61x0_nand_info *info = mtd->priv;

	if (info->retcode == ERR_CORERR)
		return 1;   // Will lead to -EUCLEAN from MTD layer
	else if (info->retcode != ERR_NONE)
		return -1;  // Will lead to -EBADMSG

	return 0;
}

/*
static int __readid(struct mv61x0_nand_info *info, uint32_t *id)
{
	return 0;
}
*/

static int mv61x0_nand_config_flash(struct mv61x0_nand_info *info)
{
	uint32_t ndcr = 0x00000FFF; /* disable all interrupts */

	info->mtd.writesize = 2048;  // FIXME: hard-code for now
	info->read_id_bytes = 4;
	info->col_addr_cycles = (info->mtd.writesize == 2048) ? 2 : 1;
	info->row_addr_cycles = 3; // FIXME base on Read ID!
	info->page_per_block = 64;
	info->flash_width = 8;
	info->dfc_width = 8;   // FIXME: hard-coded
	info->num_blocks = 1000;
	info->cmdset = &largepage_cmdset;
	info->accessible_oob = (info->mtd.writesize == 2048) ? 32 : 8;

        ndcr |= NDCR_ND_ARB_EN;
	ndcr |= (info->col_addr_cycles == 2) ? NDCR_RA_START : 0;
	ndcr |= (info->page_per_block == 64) ? NDCR_PG_PER_BLK : 0;
	ndcr |= (info->mtd.writesize == 2048) ? NDCR_PAGE_SZ : 0;
	ndcr |= (info->flash_width == 16) ? NDCR_DWIDTH_M : 0;
	ndcr |= (info->dfc_width == 16) ? NDCR_DWIDTH_C : 0;

	ndcr |= NDCR_RD_ID_CNT(info->read_id_bytes);
	ndcr |= NDCR_SPARE_EN; /* enable spare by default */
	ndcr |= NDCR_SEQ_DIS;  // Disable sequential row read

	info->reg_ndcr = ndcr;

	//FIXME: mv61x0_nand_set_timing(info, f->timing);
	nand_writel(info, NDTR0CS0, 0xb0030a0a);
	nand_writel(info, NDTR1CS0, 0x104700b1);

	return 0;
}

/*
static int mv61x0_nand_detect_config(struct mv61x0_nand_info *info)
{
	return 0;
}
*/

static int mv61x0_nand_detect_flash(struct mv61x0_nand_info *info)
{
	int ret;

	ret = mv61x0_nand_config_flash(info);
	return ret;
}

/* the maximum possible buffer size for large page with OOB data
 * is: 2048 + 64 = 2112 bytes.
 */
#define MAX_BUF_SIZE	(2048 + 64)

static int mv61x0_nand_init_buff(struct mv61x0_nand_info *info)
{
	info->data_buf = kmalloc(MAX_BUF_SIZE, GFP_KERNEL);
	if (info->data_buf == NULL) {
		return -ENOMEM;
		return 0;
	}
	//info->buf_start = info->buf_count = 0;

	return 0;
}

static struct nand_ecclayout hw_smallpage_ecclayout = {
	.eccbytes = 8,
	.oobavail = 4,
	.oobfree = { {2, 4} }
};

static struct nand_ecclayout hw_largepage_ecclayout = {
	.eccbytes = 40,
	.oobavail = 20,
	.oobfree = { {4, 20} }
};

static uint8_t bbt_pattern[] = {'B', 'b', 't' };
static uint8_t mirror_pattern[] = {'t', 'b', 'B' };

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	20,
	.len = 3,
	.veroffs = 23,
	.maxblocks = 4,
	.pattern = bbt_pattern,
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	20,
	.len = 3,
	.veroffs = 23,
	.maxblocks = 4,
	.pattern = mirror_pattern,
};

static int use_bbt = 1;
module_param(use_bbt, bool, 0600);

static void mv61x0_nand_init_mtd(struct mtd_info *mtd)
{
	struct mv61x0_nand_info *info = (struct mv61x0_nand_info *)mtd->priv;
	struct nand_chip *this = &info->nand_chip;

	this->options = 0;
	if(use_bbt) {
		this->bbt_td = &bbt_main_descr;
		this->bbt_md = &bbt_mirror_descr;

		this->options = NAND_USE_FLASH_BBT;
	//NAND_SKIP_BBTSCAN;  // !NAND_BUSWIDTH_16;
	}

	this->waitfunc		= mv61x0_nand_waitfunc;
	this->select_chip	= mv61x0_nand_select_chip;
	this->dev_ready		= mv61x0_nand_dev_ready;
	this->cmdfunc		= mv61x0_nand_cmdfunc;
	this->read_word		= mv61x0_nand_read_word;
	this->read_byte		= mv61x0_nand_read_byte;
	this->read_buf		= mv61x0_nand_read_buf;
	this->write_buf		= mv61x0_nand_write_buf;
	this->verify_buf	= mv61x0_nand_verify_buf;

	this->ecc.mode		= NAND_ECC_HW;
	this->ecc.hwctl		= mv61x0_nand_ecc_hwctl;
	this->ecc.calculate	= mv61x0_nand_ecc_calculate;
	this->ecc.correct	= mv61x0_nand_ecc_correct;
	this->ecc.size		= 2048;

	if (1 /*f->page_size == 2048*/)
		this->ecc.layout = &hw_largepage_ecclayout;
	else
		this->ecc.layout = &hw_smallpage_ecclayout;

	this->chip_delay = 25;
    this->priv = info;
}

static int wait_on_load = 0;
module_param(wait_on_load, bool, 0644);

//static int mv61x0_nand_probe(struct platform_device *pdev)
int mv61x0_nand_init(struct nand_chip *nand)
{
	struct mv61x0_nand_info *info;
	struct nand_chip *this;
	struct mtd_info *mtd;
//	struct resource *r;
	int ret = 0;//, irq;

	/*** Debug Stuff ***/
	if(wait_on_load) {
		printk("mv61x0_nand: Waiting until:\n  echo 0 > /sys/module/mv61x0_nand/parameters/wait_on_load\n");
		while(wait_on_load)
			msleep(100);
	}

	/*** Basic Setup ***/
	info = kzalloc(sizeof(struct mv61x0_nand_info), GFP_KERNEL);
	if (!info) {
		dev_err(&pdev->dev, "failed to allocate memory\n");
		return -ENOMEM;
	}

	mtd = &info->mtd;
//	info->pdev = pdev;

	this = &info->nand_chip;
	mtd->priv = info;
    mtd->name = "mtd_mv61x0";
#if 0
	mtd->owner = THIS_MODULE;

	/*** Setup Register IO Memory ***/
	r = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (r == NULL) {
		dev_err(&pdev->dev, "no IO memory resource defined\n");
		ret = -ENODEV;
		goto fail_put_clk;
	}

	r = request_mem_region(r->start, resource_size(r), pdev->name);
	if (r == NULL) {
		dev_err(&pdev->dev, "failed to request memory resource\n");
		ret = -EBUSY;
		goto fail_put_clk;
	}

	info->mmio_base = ioremap(r->start, resource_size(r));
	if (info->mmio_base == NULL) {
		dev_err(&pdev->dev, "ioremap() failed\n");
		ret = -ENODEV;
		goto fail_free_res;
	}
	info->mmio_phys = r->start;
#endif
	info->mmio_base = (void *)CONFIG_SYS_NAND_REG_BASE;

	/*** Buffer Init ***/
	ret = mv61x0_nand_init_buff(info);
	if (ret)
        return ret;
//		goto fail_free_io;

	/*** IRQ Setup ***/
	disable_int(info, NDSR_MASK);

#if 0
	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(&pdev->dev, "no IRQ resource defined\n");
		ret = -ENXIO;
		goto fail_free_buf;
	}

	ret = request_irq(irq, mv61x0_nand_irq, IRQF_DISABLED,
			  pdev->name, info);
	if (ret < 0) {
		dev_err(&pdev->dev, "failed to request IRQ\n");
		goto fail_free_buf;
	}
#endif

	ret = mv61x0_nand_detect_flash(info);
	if (ret) {
		dev_err(&pdev->dev, "failed to detect flash\n");
		ret = -ENODEV;
        return ret;
//		goto fail_free_gpios;
	}

	mv61x0_nand_init_mtd(mtd);
    memcpy(nand, &info->nand_chip, sizeof(*nand));

#if 0
	platform_set_drvdata(pdev, mtd);

	if (nand_scan(mtd, 1)) {
		dev_err(&pdev->dev, "failed to scan nand\n");
		ret = -ENXIO;
		goto fail_free_gpios;
	}

	//add_mtd_device(mtd);
#endif

#ifndef CONFIG_MTD_LXKSHIM
	/*** Partitions ***/
	{
        int num_partitions;
        struct mtd_partition *partition_table;

        num_partitions = mtd_get_partition_info_ssp(mtd, &partition_table);
        if (num_partitions > 0)
    		add_mtd_partitions(mtd, partition_table, num_partitions);
	}
#endif

	return 0;

#if 0
fail_free_gpios:
	free_irq(irq, info);
fail_free_buf:
	kfree(info->data_buf);
fail_free_io:
	iounmap(info->mmio_base);
fail_free_res:
	release_mem_region(r->start, resource_size(r));
fail_put_clk:
	//clk_disable(info->clk);
	//clk_put(info->clk);
	return ret;
#endif
}

#if 0
static int mv61x0_nand_remove(struct platform_device *pdev)
{
	struct mtd_info *mtd = platform_get_drvdata(pdev);
	struct mv61x0_nand_info *info = mtd->priv;
	struct resource *r;
	int irq;

	platform_set_drvdata(pdev, NULL);

	mtd_device_unregister(mtd);
	irq = platform_get_irq(pdev, 0);
	if (irq >= 0)
		free_irq(irq, info);

	iounmap(info->mmio_base);
	r = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	release_mem_region(r->start, resource_size(r));

	//clk_disable(info->clk);
	//clk_put(info->clk);

	return 0;
}

static struct platform_driver mv61x0_nand_driver = {
	.driver = {
		.name	= "mv61x0_nand",
	},
	.probe		= mv61x0_nand_probe,
	.remove		= mv61x0_nand_remove,
	.suspend	= NULL,
	.resume		= NULL,
};

static int __init mv61x0_nand_init(void)
{
	return platform_driver_register(&mv61x0_nand_driver);
}
module_init(mv61x0_nand_init);

static void __exit mv61x0_nand_exit(void)
{
	platform_driver_unregister(&mv61x0_nand_driver);
}
module_exit(mv61x0_nand_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("mv61x0 NAND controller driver");

#endif
