/*
 *  Copyright (C) Marvell International Ltd. (kvedere@marvell.com)
 *  Code heavily based on Linux driver
 *      /driver/mmc/host/mvsdh.c
 *  Copyright (C) 2008-2009 Marvell International Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <common.h>
#include <part.h>
#include <malloc.h>
#include <mmc.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <linux/compat.h>

#include "mvsdh.h"

#define CLKRT_OFF                   (~0)
#define GET_REG(host, reg)          readw(host->regbase + reg)
#define SET_REG(host, val, reg)     writew(val, host->regbase + reg)
#define DATA_DIRECTION_READ(data)   (data->flags & MMC_DATA_READ)
#define SET_REG_BIT(host, bit_mask, reg) \
            SET_REG(host, \
                GET_REG(host, reg) | (bit_mask), reg)
#define CLEAR_REG_BIT(host, bit_mask, reg) \
            SET_REG(host, \
                GET_REG(host, reg) & ~(bit_mask), reg)
#define SET_REG_BITS(host, bits_pos, bits_mask, val, reg) \
            SET_REG(host, \
                GET_REG(host, reg) & ~(bits_mask << bits_pos), reg); \
            SET_REG(host, \
                GET_REG(host, reg) | (val << bits_pos), reg)
#define GET_REG_BITS(host, bit_pos, bits_mask, reg) \
            ((GET_REG(host, reg) >> bit_pos) & bits_mask)
#define mmc_resp_type(cmd) ((cmd)->resp_type & (MMC_RSP_PRESENT|MMC_RSP_136|MMC_RSP_CRC|MMC_RSP_BUSY|MMC_RSP_OPCODE))
#define MMC_BUS_WIDTH_1     1
#define MMC_BUS_WIDTH_4     4
#define MMC_BUS_WIDTH_8     8
#define CONFIG_MMC_SDMA     1

#define DMA_INT_BOUND (512 * 1024)

//#define CONFIG_MMC_DEBUG  1

#ifdef CONFIG_MMC_DEBUG
#define mmc_dbg printf
#else
#define mmc_dbg(arg...)
#endif

struct mvsdh_host {
    struct mmc_cmd  *cmd;
    struct mmc_data *data;
    u32 regbase;
    u32 bytes_xfered;
    u32 clkrate;
    u32 clkrt;
    u32 data_len;
    int error;
    char *data_bak;
};

static int mvsdh_cmd_done(struct mvsdh_host *host);
static void mvsdh_dma_data_done(struct mvsdh_host *host);
static void mvsdh_data_done(struct mvsdh_host *host);

#ifdef CONFIG_MMC_DEBUG
static void dump_registers(struct mvsdh_host *host)
{
    unsigned int val;
    int offset;

    for (offset = 0; offset < 0x60; offset += 4) {
        if (offset == 0x20)
            continue;
        val = readl(host->regbase + offset);
        printf("%08x: %08x\n", (unsigned int)host->regbase + offset, val);
    }
    for (offset = 0xE0; offset < 0xF0; offset += 4) {
        val = readl(host->regbase + offset);
        printf("%08x: %08x\n", (unsigned int)host->regbase + offset, val);
    }
    val = readl(host->regbase + 0xFC);
    printf("%08x: %08x\n", (unsigned int)host->regbase + 0xFC, val);
}
#endif

static inline int mvsdh_fls(int x)
{
    int ret;

    asm("clz\t%0, %1" : "=r" (ret) : "r" (x) : "cc");
    ret = 32 - ret;
    return ret;
}

static int mvsdh_wait_reset(struct mvsdh_host *host)
{
    u32 timeout = 1000;
    u16 val;

    do {
        val = GET_REG(host, SD_TO_CTRL_SW_RST);
        if (!(val & (SW_RST_DAT | SW_RST_CMD | SW_RST_ALL)))
            break;
        udelay(1);
    } while (timeout--);

    if (timeout)
        return 0;

    printf("%s: Fatal: Wait RESET timeout.\n",__func__);

    return 1;
}

static void mvsdh_stop_clock(struct mvsdh_host *host)
{
    CLEAR_REG_BIT(host, EXT_CLK_EN, SD_CLOCK_CNTL);
}

static void mvsdh_start_clock(struct mvsdh_host *host)
{
    u32 timeout = 1000;

    SET_REG_BIT(host, INT_CLK_EN, SD_CLOCK_CNTL);
    do {
        if (GET_REG(host, SD_CLOCK_CNTL) & INT_CLK_STABLE)
            break;
        udelay(1);
    } while (timeout--);

    if (!timeout)
        printf("%s: unable to start clock\n",__func__);

    SET_REG_BITS(host, SD_FREQ_SEL_OFFSET, SD_FREQ_SEL_MASK,
        host->clkrt, SD_CLOCK_CNTL);

    /* set as maximum value for data line timeout*/
    SET_REG_BITS(host, DAT_TO_VAL_OFFSET, DAT_TO_MASK,
        (DAT_TO_MASK - 1), SD_TO_CTRL_SW_RST);

    SET_REG_BIT(host, EXT_CLK_EN, SD_CLOCK_CNTL);
}

static void mvsdh_setup_sdma(struct mvsdh_host *host)
{
    char *addr;
    struct mmc_data* data = host->data;

    if (DATA_DIRECTION_READ(data)) {
        addr = data->dest + host->bytes_xfered;
    }
    else {
        addr = (char*) data->src + host->bytes_xfered;
    }

    SET_REG(host, (u32) addr & 0xffff, SD_SYS_ADDR_LOW);
    SET_REG(host, (u32) addr >> 16, SD_SYS_ADDR_HIGH);
}

static void mvsdh_setup_data(struct mvsdh_host *host)
{
#ifdef CONFIG_MMC_SDMA
    mvsdh_setup_sdma(host);
    SET_REG_BITS(host, DMA_SEL_OFFSET, DMA_SEL_MASK, DMA_SEL_SDMA, SD_HOST_CTRL);
#endif
}

static void mvsdh_finish_request(struct mvsdh_host *host)
{
#ifdef CONFIG_MMC_DEBUG
    dump_registers(host);
#endif
    if (host->data && host->error)
        SET_REG_BIT(host, SW_RST_DAT, SD_TO_CTRL_SW_RST);

    mmc_dbg("%s: Finishing CMD%d(%s)\n",__func__, host->cmd->cmdidx,
        (host->error ? "failed" : "done"));

    host->data = NULL;
    host->cmd = NULL;
    host->error = 0;
    host->data_len = 0;
    host->bytes_xfered = 0;
}

static int mvsdh_process_irq(struct mvsdh_host *host,u32 intr_type)
{
    ulong start, curr, diff;
    ushort done = GET_REG(host, SD_NOR_I_STAT) & intr_type;

    reset_timer( );
    start = get_timer(0);

    while (!done) {
        done = GET_REG(host, SD_NOR_I_STAT) & intr_type;
        if (GET_REG(host, SD_NOR_I_STAT) & 0x8000) {
            mmc_dbg("Error! cmd : %d, err : %04x\n", host->cmd->cmdidx, GET_REG(host, SD_ERR_I_STAT));
            host->error = 1;
            mvsdh_finish_request(host);
            if (GET_REG(host, SD_ERR_I_STAT) & 0x1)
                return TIMEOUT;      /* error happened */
            else
                return COMM_ERR;
        }

        curr = get_timer(0);
        diff = (long) curr - (long) start;
        if (diff > (1 * CONFIG_SYS_HZ)) {
            printf("cmd timeout, status : %04x\n", GET_REG(host, SD_NOR_I_STAT));
            printf("xfer mode : %04x\n", GET_REG(host, SD_TRANS_MODE));
            host->error = 1;
            mvsdh_finish_request(host);
            return -ETIMEDOUT;
        }
    }

    if (intr_type == CMD_COMP) {
        mvsdh_cmd_done(host);
        return 0;
    }

#ifdef CONFIG_MMC_SDMA
    if (intr_type == DMA_INT) {
        /*
         * Multiple DMA interrupts can be generated per transation,
         * so clear the status bit after each interrupt.
         */
        SET_REG(host, DMA_INT, SD_NOR_I_STAT);
                mvsdh_dma_data_done(host);
        return 0;
    }
#else
    if ((cmdtype == TX_RDY) || (cmdtype == RX_RDY)) {
        mvsdh_pio_data_done(host);
        return 0;
    }
#endif

    if (intr_type == XFER_COMP) {
        mvsdh_data_done(host);
        return 0;
    }

    return -1;
}

static int mvsdh_start_cmd(struct mvsdh_host *host)
{
    u16 resp = 0;
    u16 xfrmd_val = 0;
    u16 cmd_val = 0;
    u16 val, mask;
    struct mmc_data *data = host->data;
    struct mmc_cmd *cmd = host->cmd;
    int ret;

    BUG_ON(!cmd);

    /*Set Response Type*/
    switch (mmc_resp_type(cmd)) {
        case MMC_RSP_NONE:
            break;

        case MMC_RSP_R1: /* r1, r5, r6, r7 */
            resp = CMD_RESP_48BIT;
            cmd_val |= CMD_CRC_CHK_EN | CMD_IDX_CHK_EN;
            break;

        case MMC_RSP_R2: /* r2 */
            resp = CMD_RESP_136BIT;
            cmd_val |= CMD_CRC_CHK_EN;
            break;

        case MMC_RSP_R3: /* r3, r4*/
            resp = CMD_RESP_48BIT;
            break;

        case MMC_RSP_R1b: /* r1b */
            resp = CMD_RESP_48BITB;
            cmd_val |= CMD_CRC_CHK_EN | CMD_IDX_CHK_EN;
            break;

        default:
            break;
    }

    /*Set Transfer mode regarding to data flag*/
    if (data) {
        cmd_val |= DATA_PRESENT;
        xfrmd_val |= BLK_CNT_EN;
        if (data->blocks > 1)
            xfrmd_val |= MULTI_BLK_SEL;
#ifdef CONFIG_MMC_SDMA
        flush_dcache_all();
        xfrmd_val |= DMA_EN;
#else
        xfrmd_val &= ~DMA_EN;
#endif
        if (DATA_DIRECTION_READ(data))
            xfrmd_val |= TO_HOST_DIR;
        else
            xfrmd_val &= ~TO_HOST_DIR;

        host->data_len = data->blocksize * data->blocks;
    }

    //if (cmd->opcode == 12)
    //  cmd_val |= host, CMD_TYPE_OFFSET, CMD_TYPE_MASK, CMD_TYPE_ABORT, SD_COMMAND);
    SET_REG(host, cmd->cmdarg & 0xffff, SD_ARG_LOW);
    SET_REG(host, cmd->cmdarg >> 16, SD_ARG_HIGH);
    SET_REG(host, xfrmd_val, SD_TRANS_MODE);
    cmd_val |= cmd->cmdidx << CMD_IDX_OFFSET | resp << RESP_TYPE_OFFSET;
    mmc_dbg("%s:Starting CMD%d with ARGUMENT 0x%x\n",__func__, cmd->cmdidx, cmd->cmdarg);

    val = GET_REG(host, SD_PRESENT_STAT_2);
    mask = CMD_LINE_LEVEL_MASK | DATA_LINE_LEVEL_MASK;
    if ((val & mask) != mask)
        mmc_dbg("%s:WARN: CMD/DATA pins are not all high, PRE_STAT=0x%04x\n",__func__,
            GET_REG(host, SD_PRESENT_STAT_2));

    SET_REG(host, cmd_val, SD_COMMAND);

    ret = mvsdh_process_irq(host, CMD_COMP);
    while(!ret && data) {
        /* DMA interrupt will be generated every 512 KB */
        if ((host->data_len - host->bytes_xfered) > DMA_INT_BOUND)
            ret = mvsdh_process_irq(host, DMA_INT);
        else {
            ret = mvsdh_process_irq(host, XFER_COMP);
            break;
        }
    }
#ifdef CONFIG_MMC_SDMA
    flush_dcache_all();
#endif
    return ret;
}



static int mvsdh_cmd_done(struct mvsdh_host *host)
{
    struct mmc_cmd *cmd = host->cmd;
    u32 resp[8];

    BUG_ON(!cmd);

    /* get cmd response */
    resp[0] = GET_REG(host, SD_RESP_0);
    resp[1] = GET_REG(host, SD_RESP_1);
    resp[2] = GET_REG(host, SD_RESP_2);
    resp[3] = GET_REG(host, SD_RESP_3);
    resp[4] = GET_REG(host, SD_RESP_4);
    resp[5] = GET_REG(host, SD_RESP_5);
    resp[6] = GET_REG(host, SD_RESP_6);
    resp[7] = readb(host->regbase + SD_RESP_7);

    if (cmd->resp_type & MMC_RSP_136) {
        cmd->response[0] = resp[5] >> 8 | resp[6] << 8 | resp[7] << 24;
        cmd->response[1] = resp[3] >> 8 | resp[4] << 8 | resp[5] << 24;
        cmd->response[2] = resp[1] >> 8 | resp[2] << 8 | resp[3] << 24;
        cmd->response[3] = resp[0] << 8 | resp[1] << 24;
    }
    else {
        cmd->response[0] = resp[1] << 16 | resp[0];
        cmd->response[1] = resp[3] << 16 | resp[2];
        cmd->response[2] = resp[5] << 16 | resp[4];
        cmd->response[3] = resp[7] << 16 | resp[6];
    }


    mmc_dbg("%s: resp[0]=0x%x resp[1]=0x%x resp[2]=0x%x resp[3]=0x%x\n",__func__,
                cmd->response[0], cmd->response[1], cmd->response[2], cmd->response[3]);
    if (host->error || !host->data) {
        mvsdh_finish_request(host);
    }

    return 1;
}

#ifndef CONFIG_MMC_SDMA
static void mvsdh_pio_data_done(struct mvsdh_host *host)
{
    struct mmc_data *data = host->data;
    u16 blk_size = data->blocksize;
    u16 i = 0;

    if (DATA_DIRECTION_READ(data)) {
        char* dest = data->dest + host->bytes_xfered;
        for (i = 0; i < blk_size; i += sizeof(u32)) {
            *(u32*)(dest + i) = readl(host->regbase + SD_BUF_DPORT_0);
        }
    } else {
        char* src = (char*) data->src + host->bytes_xfered;
        for (i = 0; i < blk_size; i += sizeof(u32)) {
            writel(*(u32*)(src +  i), host->regbase + SD_BUF_DPORT_0);
        }
    }

    if (host->bytes_xfered < host->data_len) {
        host->bytes_xfered = host->bytes_xfered + blk_size;
        mvsdh_setup_data(host);
    }
}
#endif

static void mvsdh_dma_data_done(struct mvsdh_host *host)
{
    struct mmc_data* data = host->data;

    if (host->bytes_xfered < host->data_len) {
        if(host->bytes_xfered == 0) {
            if (DATA_DIRECTION_READ(data)) { 
                host->bytes_xfered = DMA_INT_BOUND - ((int)data->dest % DMA_INT_BOUND);
            }       
            else {  
                host->bytes_xfered = DMA_INT_BOUND - ((int)data->src % DMA_INT_BOUND);
            }
        }
        else {
            host->bytes_xfered = host->bytes_xfered + DMA_INT_BOUND;
        }
        mvsdh_setup_data(host);
    }
}

static void mvsdh_data_done(struct mvsdh_host *host)
{
    mvsdh_finish_request(host);
}

static int mvsdh_request(struct mmc *mmc, struct mmc_cmd *cmd, struct mmc_data *data)
{
    struct mvsdh_host *host = mmc->priv;
    u16 val;
    u32 timeout = 1000;
    int ret = 0;

    host->data = data;
    host->cmd  = cmd;

    if (mvsdh_wait_reset(host)) {
        host->error = 1;
        mvsdh_finish_request(host);
        return -ETIMEDOUT;
    }

    do {
        val = GET_REG(host, SD_PRESENT_STAT_1);
        if (!(val & CMD_INHBT_DAT || val & CMD_INHBT_CMD))
            break;
        udelay(1);
    } while (timeout--);

    if (!timeout) {
        printf("%s: In busy, unable to start the request.\n",__func__);
        host->error = 1;
        mvsdh_finish_request(host);
        return -EBUSY;
    }

    /* Clear Interrupt/Error status */
    SET_REG(host, 0xffff, SD_NOR_I_STAT);
    SET_REG(host, 0xffff, SD_ERR_I_STAT);

    if (data) {
        if (DATA_DIRECTION_READ(data)) {
            /*
             * Work around an ASIC bug where DMAing into memory fails when copying
             * 512KB or less to an address of the form (i * 256K where i is odd)
             * such that the DMA crosses the next 512KB boundary.
             */
            if (((unsigned int)data->dest & 0x40000)
                    && (data->blocks < 0x401)
                    && (((unsigned int)data->dest & 0x80000) != (((unsigned int)data->dest + (data->blocks * data->blocksize)) & 0x80000))) {
                host->data_bak = data->dest;
                data->dest = memalign(0x80000, data->blocks * data->blocksize);
                if (!data->dest) {
                    printf("%s:%d memalign failed\n", __func__, __LINE__);
                    return -ENOMEM;
                }
            }
        }

        mmc_dbg("%s: setup data, blk_sz=%d, blk_cnt=0x%x\n", __func__,
            data->blocksize, data->blocks);
        SET_REG(host, ((u16)HOST_DMA_BDRY_MASK << HOST_DMA_BDRY_OFFSET) | data->blocksize, SD_BLOCK_SIZE);
        SET_REG(host, data->blocks, SD_BLOCK_COUNT);

        mvsdh_setup_data(host);
    }

    ret = mvsdh_start_cmd(host);

    if (host->data_bak) {
        memcpy(host->data_bak, data->dest, data->blocks * data->blocksize);
        free(data->dest);
        data->dest = host->data_bak;
        host->data_bak = NULL;
    }

    return ret;
}

static void mvsdh_set_ios(struct mmc *mmc)
{
    struct mvsdh_host *host = mmc->priv;

    if (mvsdh_wait_reset(host))
        return;

    if (mmc->clock) {
        unsigned long rate = host->clkrate;
        unsigned int clk = rate / mmc->clock;
        unsigned int shift;

        BUG_ON((mmc->clock > mmc->f_max) || (mmc->clock < mmc->f_min));
        if (mmc->clock >= host->clkrate) {
            host->clkrt = 0x00;
        }
        else {
            shift = mvsdh_fls(clk);
            if (rate / clk > mmc->clock)
                shift++;
            host->clkrt = 1 << (shift - 2);
        }

        // FIXME  Right now the fpga is not too happy with a 25MHZ clock so make sure it is
        // divided down to at least 12.5MHZ
        if (host->clkrt <= 0)
            host->clkrt = 1;
        mmc_dbg("%s: set clkrt = %08x\n",__func__,host->clkrt);
        mvsdh_stop_clock(host);
        mvsdh_start_clock(host);

        if (mmc->card_caps & MMC_MODE_HS) {
            //FIXME  fpga does not like having this bit set even when in high speed mode
            //SET_REG_BIT(host, HI_SPEED_EN, SD_HOST_CTRL);
        }
        else {
            CLEAR_REG_BIT(host, HI_SPEED_EN, SD_HOST_CTRL);
        }
    }
    else {
        mvsdh_stop_clock(host);
        if (host->clkrt != CLKRT_OFF) {
            host->clkrt = CLKRT_OFF;
        }
    }

    SET_REG_BITS(host, SDCLK_SEL_OFFSET, SDCLK_SEL_MASK, SDCLK_SEL_INIT_VAL, SD_CLK_BURST_SET);
    SET_REG_BITS(host, SD_BUS_VLT_OFFSET, SD_BUS_VLT_MASK, SD_BUS_VLT_33V, SD_HOST_CTRL);
    SET_REG_BIT(host, SD_BUS_POWER, SD_HOST_CTRL);

    if (IS_SD(mmc))
        CLEAR_REG_BIT(host, MMC_CARD, SD_CE_ATA_2);
    else
        SET_REG_BIT(host, MMC_CARD, SD_CE_ATA_2);

    if (mmc->bus_width == MMC_BUS_WIDTH_8) {
        SET_REG_BIT(host, DATA_WIDTH_8BIT, SD_CE_ATA_2);
        mmc_dbg("%s: set as 8_BIT_MODE.\n",__func__);
    }
    else {
        CLEAR_REG_BIT(host, DATA_WIDTH_8BIT, SD_CE_ATA_2);
        if (mmc->bus_width == MMC_BUS_WIDTH_4) {
            SET_REG_BIT(host, DATA_WIDTH_4BIT, SD_HOST_CTRL);
            mmc_dbg("%s: set as 4_BIT_MODE.\n",__func__);
        }
        else {
            CLEAR_REG_BIT(host, DATA_WIDTH_4BIT, SD_HOST_CTRL);
        }
    }
}

static int sdh_init(struct mmc *mmc)
{
    struct mvsdh_host* host = mmc->priv;
    unsigned int timeout = 1000;

    SET_REG(host, (SW_RST_DAT | SW_RST_CMD | SW_RST_ALL), SD_TO_CTRL_SW_RST);
    while (timeout--) {
        if (!(GET_REG(host, SD_TO_CTRL_SW_RST) & 0x100))
            break;
        udelay(1);
    }

    /* Enable Interrupt status */
    SET_REG(host, 0xffff, SD_NOR_I_STAT_EN);
    SET_REG(host, 0xffff, SD_ERR_I_STAT_EN);

    /* Disable interrupt generation */
    SET_REG(host, 0, SD_NOR_INT_EN);
    SET_REG(host, 0, SD_ERR_INT_EN);

    SET_REG_BIT(host, DIS_PAD_SD_CLK_GATE, SD_FIFO_PARAM);

    return 0;
}

static int mvsdh_controller_init(uint32_t base, uint32_t id) {
    struct mmc *mmc;
    struct mvsdh_host *host;
    uint16_t voltages;

    mmc = malloc(sizeof(*mmc));
    if (!mmc) {
        printf("mmc malloc fail!!\n");
        return -1;
    }
    memset(mmc, 0, sizeof(*mmc));

    host = malloc(sizeof(*host));
    if (!host) {
        printf("host malloc fail!!!\n");
        return -1;
    }
    memset(host, 0, sizeof(*host));

    host->regbase = base;
    host->clkrate = 25000000;
    host->clkrt = CLKRT_OFF;

    sprintf(mmc->name, "mvsdh%u", id);
    mmc->priv = host;
    mmc->send_cmd = mvsdh_request;
    mmc->set_ios = mvsdh_set_ios;
    mmc->init = sdh_init;

    /*
     * Calculate minimum clock rate, rounding up.
     */
    mmc->f_min = (host->clkrate + SD_FREQ_SEL_MASK) / (SD_FREQ_SEL_MASK + 1);
    mmc->f_max = host->clkrate;
    mmc->ocr = 0xffffffff;
    mmc->host_caps = MMC_MODE_4BIT | MMC_MODE_HS | MMC_MODE_HC;
#ifdef CONFIG_MMC3
    mmc->host_caps |= MMC_MODE_8BIT;
#endif

    voltages = GET_REG(host, SD_CAP_1);
    mmc->voltages = 0;
    if (voltages & SD_CAP1_VLT_18_SUPPORT)
        mmc->voltages |= MMC_VDD_165_195;
    if (voltages & SD_CAP1_VLT_30_SUPPORT)
        mmc->voltages |= (MMC_VDD_29_30 | MMC_VDD_30_31);
    if (voltages & SD_CAP1_VLT_33_SUPPORT)
        mmc->voltages |= (MMC_VDD_32_33 | MMC_VDD_33_34);

    mmc_register(mmc);

    return 0;
}

int mvsdh_init(bd_t *bis)
{
    if (mvsdh_controller_init(CONFIG_SYS_MMC_BASE, 0))
        return -1;

    return mvsdh_controller_init(CONFIG_SYS_MMC_BASE + 0x200, 1);
}
