/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * Support for Super Simple Partition.
 */

#include <common.h>
#include <malloc.h>
#include <linux/compiler.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include "part_ssp.h"

#if defined(CONFIG_MTD_PARTITIONS) || defined(CONFIG_MMC)
static int is_pkg_hdr_valid(const packageHeader_t *ph) {
	return (be32_to_cpu(ph->MagicWord) == PKGHDR_TAGWORD);
}

static int is_mbr_valid(const master_boot_record_t *mbr) {
	return (be32_to_cpu(mbr->magic) == MBR_MAGIC);
}
#endif

#ifdef CONFIG_MMC
static int get_pkg_hdr_at(block_dev_desc_t * dev_desc, ulong offset, packageHeader_t *ph) {
	int blk_cnt = sizeof(packageHeader_t) / dev_desc->blksz;
	ulong *buf = (ulong *)ph;
	int using_block_buf = 0;
	int rc = -1;

	if (blk_cnt == 0) {
		/* (MBR < block size), so a block buffer is needed */
		buf = malloc(dev_desc->blksz);
		blk_cnt++;
		using_block_buf = 1;
	}

	if (dev_desc->block_read(dev_desc->dev, offset / dev_desc->blksz,
					blk_cnt, buf) != blk_cnt) {
		goto cleanup;
	}

	if (using_block_buf) {
		memcpy(ph, buf, sizeof(packageHeader_t));
	}

	if (is_pkg_hdr_valid(ph)) {
		rc = 0;
	}

cleanup:
	if (using_block_buf)
		free(buf);

	return rc;
}

static int get_pkg_hdr(block_dev_desc_t *dev_desc, packageHeader_t *ph, int ph_num) {
	/* Look in the known places */
	if (get_pkg_hdr_at(dev_desc, (MBR_OFFSET_0 + sizeof(master_boot_record_t) + (ph_num * sizeof(packageHeader_t))), ph) == 0)
		return 0;
	if (get_pkg_hdr_at(dev_desc, (MBR_OFFSET_1 + sizeof(master_boot_record_t) + (ph_num * sizeof(packageHeader_t))), ph) == 0)
		return 0;
	return -1;
}

static int get_mbr_at(block_dev_desc_t * dev_desc, ulong offset, master_boot_record_t *mbr) {
	int blk_cnt = sizeof(master_boot_record_t) / dev_desc->blksz;
	ulong *buf = (ulong *)mbr;
	int using_block_buf = 0;
	int rc = -1;

	if (blk_cnt == 0) {
		/* (MBR < block size), so a block buffer is needed */
		buf = malloc(dev_desc->blksz);
		blk_cnt++;
		using_block_buf = 1;
	}

	if (dev_desc->block_read(dev_desc->dev, offset / dev_desc->blksz,
					blk_cnt, buf) != blk_cnt) {
		goto cleanup;
	}

	if (using_block_buf) {
		memcpy(mbr, buf, sizeof(master_boot_record_t));
	}

	if (is_mbr_valid(mbr)) {
		rc = 0;
	}

cleanup:
	if (using_block_buf)
		free(buf);

	return rc;
}

static int get_mbr(block_dev_desc_t * dev_desc, master_boot_record_t *mbr) {
	/* Look in the known places */
	if (get_mbr_at(dev_desc, MBR_OFFSET_0, mbr) == 0)
		return 0;
	if (get_mbr_at(dev_desc, MBR_OFFSET_1, mbr) == 0)
		return 0;
	return -1;
}

void print_part_ssp(block_dev_desc_t * dev_desc) {
	master_boot_record_t mbr;
	int i;

	if (get_mbr(dev_desc, &mbr) != 0) {
		printf("** Can't read partition table on %d **\n",
			dev_desc->dev);
		return;
	}

	printf("P  %10s   %10s\n", "Start", "Length");
	for (i = 0; i < MAX_PARTITIONS; i++) {
		if ((mbr.partition_table[i].length != 0) &&
			(mbr.partition_table[i].length != 0xffffffff)) {
			printf("%2d %#010x\t%#010x\n", i,
				be32_to_cpu(mbr.partition_table[i].start),
				be32_to_cpu(mbr.partition_table[i].length));
		}
	}
}

int get_partition_info_ssp(block_dev_desc_t * dev_desc, int part, disk_partition_t * info) {
	master_boot_record_t mbr;
	packageHeader_t ph;

	if (get_mbr(dev_desc, &mbr) != 0) {
		printf("** Can't read partition table on %d **\n",
			dev_desc->dev);
		return -1;
	}

	if (part < 0 || part >= MAX_PARTITIONS) {
		printf("** Invalid partition number passed to %s: %d **\n",
			__func__, part);
		return -1;
	}

	if (get_pkg_hdr(dev_desc, &ph, part)) {
		//printf("** Can't read pkg header on dev %d for partition %d **\n",
		//	dev_desc->dev, part);
		return -1;
	}

	if ((mbr.partition_table[part].length == 0) ||
		(mbr.partition_table[part].length == 0xffffffff)) {
		memset(info, 0, sizeof(*info));
		return 0;
	}

	info->start    = be32_to_cpu(mbr.partition_table[part].start) / dev_desc->blksz;
	info->size     = be32_to_cpu(mbr.partition_table[part].length) / dev_desc->blksz;
	info->data_len = be32_to_cpu((uint32_t)ph.PkgLength);
	info->blksz = dev_desc->blksz;
	info->fcert_len = be16_to_cpu(ph.PkgSignSize);

	sprintf((char *)info->name, "%s", ph.PkgName);
	sprintf((char *)info->type, "U-Boot");

	return 0;
}

int test_part_ssp(block_dev_desc_t * dev_desc)
{
	master_boot_record_t mbr;

	/* Read the master boot record and validate it */
	if (get_mbr(dev_desc, &mbr) != 0)
		return -1;

	return 0;
}
#endif

#ifdef CONFIG_MTD_PARTITIONS
static int mtd_get_pkg_hdr_at(struct mtd_info *mtd, ulong offset, packageHeader_t *ph) {
	size_t retlen = 0;

	if (mtd_read(mtd, offset, sizeof(packageHeader_t), &retlen, (uint8_t *)ph) < 0)
		return -1;

	if (retlen != sizeof(packageHeader_t))
		return -1;

	if (is_pkg_hdr_valid(ph))
		return 0;

	return -1;
}

static int mtd_get_pkg_hdr(struct mtd_info *mtd, packageHeader_t *ph, int ph_num) {
	/* Look in the known places */
	if (mtd_get_pkg_hdr_at(mtd, (MBR_OFFSET_0 + sizeof(master_boot_record_t) + (ph_num * sizeof(packageHeader_t))), ph) == 0)
		return 0;
	if (mtd_get_pkg_hdr_at(mtd, (MBR_OFFSET_1 + sizeof(master_boot_record_t) + (ph_num * sizeof(packageHeader_t))), ph) == 0)
		return 0;
	return -1;
}

static int mtd_get_mbr_at(struct mtd_info *mtd, ulong offset, master_boot_record_t *mbr) {
	size_t retlen = 0;

	if (mtd_read(mtd, offset, sizeof(master_boot_record_t), &retlen, (uint8_t *)mbr) < 0)
		return -1;

	if (retlen != sizeof(master_boot_record_t))
		return -1;

	if (is_mbr_valid(mbr))
		return 0;

	return -1;
}

static int mtd_get_mbr(struct mtd_info *mtd, master_boot_record_t *mbr) {
	/* Look in the known places */
	if (mtd_get_mbr_at(mtd, MBR_OFFSET_0, mbr) == 0)
		return 0;
	if (mtd_get_mbr_at(mtd, MBR_OFFSET_1, mbr) == 0)
		return 0;
	return -1;
}

int mtd_get_partition_info_ssp(struct mtd_info *mtd, struct mtd_partition **partition_table) {
	size_t i, j;
	int num_partitions = 0;
	struct mtd_partition *pt;
	master_boot_record_t mbr;

	if (mtd_get_mbr(mtd, &mbr) != 0)
		return -1;

	for (i=0; i<MAX_PARTITIONS; ++i) {
		if ((mbr.partition_table[i].length != 0) && (mbr.partition_table[i].length != 0xffffffff))
			++num_partitions;
	}

	pt = malloc(sizeof(struct mtd_partition) * num_partitions);
	if (!pt)
		return -1;

	memset(pt, 0, sizeof(struct mtd_partition) * num_partitions);

	for (i=0, j=0; i<MAX_PARTITIONS; ++i) {
		if ((mbr.partition_table[i].length != 0) && (mbr.partition_table[i].length != 0xffffffff)) {
			packageHeader_t ph;

			if (mtd_get_pkg_hdr(mtd, &ph, i) == 0) {
				pt[j].offset   = be32_to_cpu(mbr.partition_table[i].start);
				pt[j].size     = be32_to_cpu(mbr.partition_table[i].length);
				pt[j].data_len = be32_to_cpu((uint32_t)ph.PkgLength);
				pt[j].fcert_len = be16_to_cpu(ph.PkgSignSize);
				pt[j].name   = malloc(PKGNAME_ARRAY_SIZE);
				if (pt[j].name)
					sprintf(pt[j].name, "%s", ph.PkgName);
				++j;
			}
		}
	}

	*partition_table = pt;

	return num_partitions;
}
#endif
