/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * (C) Copyright 2002
 * David Mueller, ELSOFT AG, <d.mueller@elsoft.ch>
 *
 * (C) Copyright 2003
 * Texas Instruments, <www.ti.com>
 * Kshitij Gupta <Kshitij@ti.com>
 *
 * (C) Copyright 2004
 * ARM Ltd.
 * Philippe Robin, <philippe.robin@arm.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <mmc.h>
#include <nand.h>
#include <linux/err.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <asm/pl310.h>
#include <asm/armv7.h>
#include <asm/arch/gpio.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/adc_regmasks.h>
#include <asm/arch/adc_regstructs.h>
#include <asm/arch/apb_config_regmasks.h>
#include <asm/arch/apb_config_regstructs.h>
#include <asm/arch/efuse_regstructs.h>
#include <asm/arch/efuse_regmasks.h>
#include <asm/arch/MC_regmasks.h>
#include <asm/arch/MC_regstructs.h>
#include <asm/arch/TIMEBASE_regstructs.h>
#include <asm/arch/USB2TOP_regstructs.h>
#include <asm/arch/USB2TOP_regmasks.h>
#include <asm/arch/GPIO_regstructs.h>
#include <asm/arch/GPIO_regmasks.h>
#include <asm/arch/VCF_wrapperstructs.h>
#include <asm/arch/VCF_wrappermasks.h>
#include <asm/arch/board_priv.h>
#include <lxk_panel.h>
#include "card.h"
#include "regulator.h"

extern void disable_speculative_mode(void);
extern void disable_pld_instruction(void);
extern void disable_streaming_device_writes(void);
extern void ldm_single_word(void);

static const struct gpio_block_config_t asicGpioConfig[] = {
    { APB_GPIOA_BASE, 0x00000fff },
    { APB_GPIOB_BASE, 0xffffffff },
    { APB_GPIOC_BASE, 0xffffffff },
    { APB_GPIOD_BASE, 0xffffffff },
    { APB_GPIOE_BASE, 0xffffffff },
    { APB_GPIOF_BASE, 0xffffffff },
    { APB_GPIOG_BASE, 0xffffffff },
    { APB_GPIOH_BASE, 0x001fffff }
};

static long dram_sizes[CONFIG_NR_DRAM_BANKS] __attribute__ ((section (".data")));
static char asic_rev[3]  __attribute__ ((section (".data")));

static int sdmmc0_clk_enable = 0;

DECLARE_GLOBAL_DATA_PTR;

void code_verification_failure(void) {
    struct EFUSE_REGS_s *efuse = (struct EFUSE_REGS_s *)APB_EFUSE_BASE;
    struct VCF_WRAPPER_REGS_s *vcf_wrapper = (struct VCF_WRAPPER_REGS_s *)VCF_WRAPPER_BASE;

    /* Populate the efuse status registers */
    efuse->AUTO_Control = 1;
    while (efuse->TOP_Status & EFUSE_TOP_STATUS_BUSY_MASK);

    /* If the VCF is enabled and the life cycle is deployed then put in min fun mode */
    if (((efuse->EfuseStatus63to32_Bank1 & 0xf) == 0x5) && (efuse->EfuseStatus96_Bank2 & 1)) {
        vcf_wrapper->VMFR = VCF_WRAPPER_VMFR_MINFUNC_MASK;
        setenv("MIN_FUNC_MODE", "1");
    }
}

static void USBPhyInit(void *UtmRegAddr) {
    uint32_t temp;
    USB2PHY_USB2_UTM_PLL_REGS_t *PhyRegs;
    USB2PHY_USB2_UTM1_REGS_t *Utm1Regs;

    PhyRegs = (USB2PHY_USB2_UTM_PLL_REGS_t *) USB_USB2PHY_USB2_UTM_PLL_BASE;
    Utm1Regs = (USB2PHY_USB2_UTM1_REGS_t *) UtmRegAddr;
    // This has to be done first to insure everything is going before the clocks are started
    Utm1Regs->RX1 = USB2PHY_USB2_UTM1_RX1_PU_PHY_REPLACE_VAL(Utm1Regs->RX1, 1);
    udelay(100);

    PhyRegs->PLL0 = USB2PHY_USB2_UTM_PLL_PLL0_REFDIV_REPLACE_VAL(PhyRegs->PLL0, 5);
    PhyRegs->PLL0 = USB2PHY_USB2_UTM_PLL_PLL0_FBDIV_REPLACE_VAL(PhyRegs->PLL0, 0x60);
    PhyRegs->PLL1 = USB2PHY_USB2_UTM_PLL_PLL1_PU_REPLACE_VAL(PhyRegs->PLL1, 1);
    udelay(200);  // wait for it to be ready.
    PhyRegs->PLL0 = USB2PHY_USB2_UTM_PLL_PLL0_VCOCAL_START_REPLACE_VAL(PhyRegs->PLL0, 1); // set the vco cal

    // wait for the pll to come ready
    while(!(PhyRegs->PLL0 & USB2PHY_USB2_UTM_PLL_PLL0_PLL_READY_MASK));

    udelay(1000);
    // now do the rcal of the phy
    Utm1Regs->TX = USB2PHY_USB2_UTM1_TX_REG_RCAL_START_REPLACE_VAL(Utm1Regs->TX,1);
    udelay(100);
    Utm1Regs->TX = USB2PHY_USB2_UTM1_TX_REG_RCAL_START_REPLACE_VAL(Utm1Regs->TX,0);
    udelay(400);  // need to wait for the rcal to clean up
    // drop the bus.  This fixes an issue with the chip

    temp = USB2PHY_USB2_UTM1_TEST0_REG_DP_PULLDOWN_REPLACE_VAL(Utm1Regs->TEST0, 1);
    temp = USB2PHY_USB2_UTM1_TEST0_REG_DM_PULLDOWN_REPLACE_VAL(temp, 1);
    Utm1Regs->TEST0 = USB2PHY_USB2_UTM1_TEST0_REG_ARC_DPDM_MODE_REPLACE_VAL(temp, 0);
    udelay(500000);

    temp = USB2PHY_USB2_UTM1_TEST0_REG_DP_PULLDOWN_REPLACE_VAL(Utm1Regs->TEST0, 0);
    temp = USB2PHY_USB2_UTM1_TEST0_REG_DM_PULLDOWN_REPLACE_VAL(temp, 0);
    Utm1Regs->TEST0 = USB2PHY_USB2_UTM1_TEST0_REG_ARC_DPDM_MODE_REPLACE_VAL(temp, 1);
}

/* Set the HiPS PLL to output a 2.5Ghz clock */
static int init_hips_pll(int enable_spread) {
    uint32_t val;
    uint32_t timeout = 1000;
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    /* Put into bypass */
    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_BYPASS_EN_REPLACE_VAL(apb_config->PLL1_CR0, 1);
    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_PU_REPLACE_VAL(apb_config->PLL1_CR0, 0);

    /* Set PLL configuration values */
    val = APB_CONFIG_PLL1_CR0_REFDIV_REPLACE_VAL(apb_config->PLL1_CR0, 3); // 25Mhz input / 3 = 8.33 = 10 +- 20%
    val = APB_CONFIG_PLL1_CR0_VCODIV_SEL_SE_REPLACE_VAL(val, 0);           // VCO post divider = 1
    val = APB_CONFIG_PLL1_CR0_KVCO_REPLACE_VAL(val, 6);                    // KVCO = 2.2-2.5 Ghz
    val = APB_CONFIG_PLL1_CR0_VCO_VRNG_REPLACE_VAL(val, 5);
    val = APB_CONFIG_PLL1_CR0_VCODIV_SEL_DIFF_REPLACE_VAL(val, 6);
    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_FBDIV_REPLACE_VAL(val, 300);

    val = APB_CONFIG_PLL1_CR1_DIFFCLK_EN_REPLACE_VAL(apb_config->PLL1_CR1, 1);
    val = APB_CONFIG_PLL1_CR1_BYPASS_FBDIV_REPLACE_VAL(val, 0);
    val = APB_CONFIG_PLL1_CR1_ICP_REPLACE_VAL(val, 1);
    val = APB_CONFIG_PLL1_CR1_VDDM_REPLACE_VAL(val, 1);
    apb_config->PLL1_CR1 = APB_CONFIG_PLL1_CR1_VDDL_REPLACE_VAL(val, 4);

    /*
     * downspread at a frequency of 32KHz: [2.5GHz / (2 * 32KHz)]
     * spread at a range of 1.0%:          [D / (39062 * 2^-27)]
     */
    val = APB_CONFIG_PLL1_CR2_SSC_FREQ_DIV_REPLACE_VAL(apb_config->PLL1_CR2, 39062);
    apb_config->PLL1_CR2 = APB_CONFIG_PLL1_CR2_SSC_RNGE_REPLACE_VAL(val, 34);

    /* Enable spreading on the DIFF output */
    apb_config->PLL1_CR1 = APB_CONFIG_PLL1_CR1_PI_EN_REPLACE_VAL(apb_config->PLL1_CR1, 1);
    apb_config->PLL1_CR1 = APB_CONFIG_PLL1_CR1_SEL_VCO_DIFF_REPLACE_VAL(apb_config->PLL1_CR1, 0);

    /* Enable SSCG in CR2 */
    val = APB_CONFIG_PLL1_CR2_SSC_CLK_EN_REPLACE_VAL(apb_config->PLL1_CR2, 1);
    apb_config->PLL1_CR2 = APB_CONFIG_PLL1_CR2_SSC_MODE_REPLACE_VAL(val, 1);

    /* Enable frequency offset */
    apb_config->PLL1_CR3 = APB_CONFIG_PLL1_CR3_FREQ_OFFSET_MODE_SELECTION_REPLACE_VAL(apb_config->PLL1_CR3, 1);

    /* Reset */
    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_RESET_REPLACE_VAL(apb_config->PLL1_CR0, 1);
    val = APB_CONFIG_PLL1_CR2_RESET_EXT_REPLACE_VAL(apb_config->PLL1_CR2, 1);
    apb_config->PLL1_CR2 = APB_CONFIG_PLL1_CR2_SSC_RESET_EXT_REPLACE_VAL(val, 1);

    udelay(10);

    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_RESET_REPLACE_VAL(apb_config->PLL1_CR0, 0);
    val = APB_CONFIG_PLL1_CR2_RESET_EXT_REPLACE_VAL(apb_config->PLL1_CR2, 0);
    apb_config->PLL1_CR2 = APB_CONFIG_PLL1_CR2_SSC_RESET_EXT_REPLACE_VAL(val, 0);

    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_PU_REPLACE_VAL(apb_config->PLL1_CR0, 1);

    udelay(50);

    /* Wait for lock */
    while (!(apb_config->PLL1_SR & APB_CONFIG_PLL1_SR_LOCK_MASK)) {
        if (timeout-- == 0)
            break;
        udelay(10);
    }

    /* Pull out of bypass */
    apb_config->PLL1_CR0 = APB_CONFIG_PLL1_CR0_BYPASS_EN_REPLACE_VAL(apb_config->PLL1_CR0, 0);

    /* Enable spread */
    if (enable_spread)
        apb_config->PLL1_CR2 = APB_CONFIG_PLL1_CR2_SSC_EN_REPLACE_VAL(apb_config->PLL1_CR2, 1);

    return 0;
}

static void get_asic_rev(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    switch (APB_CONFIG_AIR_MAJORREV_MASK_SHIFT(apb_config->AIR)) {
        case 0:  asic_rev[0] = 'A';  break;
        case 1:  asic_rev[0] = 'B';  break;
        default:
        case 2:  asic_rev[0] = 'C';  break;
    }

    asic_rev[1] = APB_CONFIG_AIR_MINORREV_MASK_SHIFT(apb_config->AIR) + '0';
    asic_rev[2] = 0;
}

static void enet_phy_init(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;
    uint32_t voltage = 0;
    uint32_t drive_strength = 0;
    const char *card;
    const char *cardrev;

    gpio_set(ENET_RESET, 0);
    mdelay(15);
    gpio_set(ENET_RESET, 1);

    // Set drive strength for ethernet pads
    if (!is_phy_gigabit( )) {
        voltage = 1;
    }
    else {
        voltage = 2;
    }

    card = card_get_type( );
    cardrev = card_get_revision( );

    if (((strcmp(card, "card04") == 0) ||
         (strcmp(card, "card11") == 0) ||
         (strcmp(card, "card09") == 0)) &&
        (cardrev[1] == '0'))
        drive_strength = 1;

    apb_config->PDCR3 = APB_CONFIG_PDCR3_HSIO_VSEL_REPLACE_VAL(apb_config->PDCR3, voltage);
    apb_config->PDCR3 = APB_CONFIG_PDCR3_HSIO_ZN_REPLACE_VAL(apb_config->PDCR3, drive_strength);
    apb_config->PDCR3 = APB_CONFIG_PDCR3_HSIO_ZP_REPLACE_VAL(apb_config->PDCR3, drive_strength);

    /* The 25Mhz is unused except for cards04, 06, and 09 */
    if ((strcmp(card, "card04") != 0) && (strcmp(card, "card06") != 0) && (strcmp(card, "card09") != 0) && (strcmp(card, "corecard") != 0) && (strcmp(card, "card11") != 0)) {
        gpio_direction(ENET_25MHZ, 0);
        gpio_mux(ENET_25MHZ, 0);
    }
}

static void pl310_disable(void) {
    pl310_enable(0);
    v7_outer_cache_flush_all( );
    pl310_unlock_way(0x7f);
}

#ifdef CONFIG_LXK_PANEL
static void lcd_clk_gpio_disable(void) {
    /* drive unused lcd clock output low for emc */
    gpio_direction(VIDEO_CLK, 1);
    gpio_set(VIDEO_CLK, 0);
    gpio_mux(VIDEO_CLK, 0);
}
#endif

static void configure_i2c_pins(void) {
    volatile struct GPIO_REGS_s *gpio_block = (volatile struct GPIO_REGS_s *)APB_GPIOC_BASE;
    uint32_t val;

    /*
     * Disable Raw Input Select and enable Deglitch Timebase for the i2c pins
     * Counter Timebase = System clock * 4
     * Minimum Stable Period = System clock * 12
     * Maximum Stable Period = System clock * 16
     */
    val = GPIO_PINCFG0_RAWALT_REPLACE_VAL(gpio_block->PinCfg0, 0);
    val = GPIO_PINCFG0_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg0 = GPIO_PINCFG0_DEGTB_REPLACE_VAL(val, 3);

    val = GPIO_PINCFG1_RAWALT_REPLACE_VAL(gpio_block->PinCfg1, 0);
    val = GPIO_PINCFG1_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg1 = GPIO_PINCFG1_DEGTB_REPLACE_VAL(val, 3);

    val = GPIO_PINCFG2_RAWALT_REPLACE_VAL(gpio_block->PinCfg2, 0);
    val = GPIO_PINCFG2_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg2 = GPIO_PINCFG2_DEGTB_REPLACE_VAL(val, 3);

    val = GPIO_PINCFG3_RAWALT_REPLACE_VAL(gpio_block->PinCfg3, 0);
    val = GPIO_PINCFG3_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg3 = GPIO_PINCFG3_DEGTB_REPLACE_VAL(val, 3);

    val = GPIO_PINCFG4_RAWALT_REPLACE_VAL(gpio_block->PinCfg4, 0);
    val = GPIO_PINCFG4_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg4 = GPIO_PINCFG4_DEGTB_REPLACE_VAL(val, 3);

    val = GPIO_PINCFG5_RAWALT_REPLACE_VAL(gpio_block->PinCfg5, 0);
    val = GPIO_PINCFG5_FUNCSEL_REPLACE_VAL(val, 1);
    gpio_block->PinCfg5 = GPIO_PINCFG5_DEGTB_REPLACE_VAL(val, 3);
}

int board_early_init_f(void) {
    volatile struct GPIO_REGS_s *gpio_block = (volatile struct GPIO_REGS_s *)APB_GPIOB_BASE;
    volatile unsigned int *cpucr5 = (volatile unsigned int *)0xf8fff034;

    get_asic_rev( );

    /* When booting nand, the bootrom wrongly sets the RAWALT bit on this pin to 0 */
    gpio_block->PinCfg1 |= GPIO_PINCFG1_RAWALT_MASK;

    /* CPU SRAM power down defaults are incorrect for rev B.  Fix'em up here */
    if (asic_rev[0] == 'B') {
        *cpucr5 = (*cpucr5 & 0xf000ffff) | (0x451 << 16);
    }

    /* Enable CPU power capabilities and set extra-low leakage wfi state */
    /* set sram_vddmcvss_ctrl_en bits in CPUCR5 */
    *cpucr5 = (*cpucr5 | 0x00000300);
    /* set pj4b_sram_pdlvmc and sysl2c_sram_pdlvmc bits in CPUCR6 */
    *(cpucr5+1) = (*(cpucr5+1) | 0x00030000);

    /* Set the pin configs to read the SPD */
    configure_i2c_pins( );

    return 0;
}

/* 
 * Miscellaneous board dependent usb adjustments 
 */                                            
static void usb_adjustments(void) {
    struct USB2PHY_USB2_UTM2_REGS_s *Utm2Regs = (struct USB2PHY_USB2_UTM2_REGS_s *)USB_USB2PHY_USB2_UTM2_BASE;
    struct USB2PHY_USB2_UTM3_REGS_s *Utm3Regs = (struct USB2PHY_USB2_UTM3_REGS_s *)USB_USB2PHY_USB2_UTM3_BASE;
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    /* Adjust Bandgap Voltage */
    apb_config->BGCR = APB_CONFIG_BGCR_BG_SEL_REPLACE_VAL(apb_config->BGCR , 0x2);
    /* Adjust Squelch Threshold for UTM2/USBH */
    Utm2Regs->RX0 = USB2PHY_USB2_UTM2_RX0_SQ_THRESH_REPLACE_VAL(Utm2Regs->RX0, 0xE);
    /* Adjust Squelch Threshold for UTM3/USBHX */
    Utm3Regs->RX0 = USB2PHY_USB2_UTM3_RX0_SQ_THRESH_REPLACE_VAL(Utm3Regs->RX0, 0xE);
}

/*
 * Miscellaneous platform dependent initializations
 */
int board_init(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    /* arch number of Obelisk Board */
    gd->bd->bi_arch_number = MACH_TYPE_OBELISK;

    /* adress of boot parameters */
    gd->bd->bi_boot_params = 0x00000100;

    gd->flags = 0;

    pl310_disable( );
    icache_enable( );

    disable_pld_instruction( );

    /* Watchdog resets should assert the system reset line */
    apb_config->FCRR |= APB_CONFIG_FCRR_WATCHDOGRESET_SYSNCHIP_MASK;

    /* The FPGA build uses a completely different address range for the GPIO blocks */
    disable_speculative_mode( );
    disable_streaming_device_writes( );
    ldm_single_word( );
    gpio_block_init(asicGpioConfig, ARRAY_SIZE(asicGpioConfig));
    card_init( );

    por_beep( );

    /*
     *  settings for A1, B0, C0 parts
     * set UPC I/O clock divider
     */
    apb_config->CDCR1 = APB_CONFIG_CDCR1_XIODIV_REPLACE_VAL(apb_config->CDCR1, UPC_IO_CLOCK_DIV);
    /* Set UPC processor clock divider  */
    apb_config->CDCR1 = APB_CONFIG_CDCR1_XCPUDIV_REPLACE_VAL(apb_config->CDCR1, UPC_PROC_CLOCK_DIV);

    enet_phy_init( );

    USBPhyInit((void *) USB_USB2PHY_USB2_UTM2_BASE);
    USBPhyInit((void *) USB_USB2PHY_USB2_UTM3_BASE);

    usb_adjustments();

    /*
     * clk_lvds_ui_init() depends on the USB 240mhz clkgen,
     * so it can't be called in card_init()
     */
    clk_lvds_ui_init();

    return 0;
}

unsigned int board_get_core_freq_hz(void)
{
    return 800 * 1000000;
}

unsigned int board_get_bus_freq_hz(void)
{
    unsigned int bus_freq;
    uint32_t *lcm_block = (uint32_t *)LCM_BASE;

    if (lcm_block[1] > 0x40) {
        bus_freq = lcm_block[1];
    }
    else {
        bus_freq = lcm_block[2];
    }

    return bus_freq * 1000000;
}

/* use_sd - 1 for SD/MMC on SD0, 0 for bootspi */
void mmc_switch_sd0_pins(int use_sd) {
    int mux = (use_sd) ? 2 : 1;
    const char *card = card_get_type( );

    if (strcmp(card, "card09") != 0) {
        gpio_mux(BSPI_CLK, mux);
        gpio_mux(BSPI_CS,  mux);
        gpio_mux(BSPI_TXD, mux);
        gpio_mux(BSPI_RXD, mux);
    }

    gpio_direction(eMMC_ENABLE, 1);
    gpio_set(eMMC_ENABLE, !use_sd);
}

#ifndef CONFIG_DRAM_INIT_ONLY
static int verify_kernel_partition(void) {
    const char *card = card_get_type( );

    return (strcmp(card, "corecard") != 0);
}
#endif

#ifdef CONFIG_MMC
extern int mvsdh_init(bd_t *bis);

int board_mmc_init(bd_t *bis) {
    return(mvsdh_init(bis));
}

/*
 * Construct the boot command for the specified configuration.
 */
static int mmc_get_bootcmd_for_part(int dev_num, int part, disk_partition_t *pinfo, char *cmd, int len, block_dev_desc_t * dev_desc, int boot_recover) {
    if (strcmp(pinfo->name, "Kernel") != 0) /* The kernel partition is on partition 2 for both EXT2 and CRAM */
        return 1;

    /* If the device uses SSP partitioning then it is using CRAM */
    if (dev_desc->part_type == PART_TYPE_SSP) {
        int rc;
        int partition;
        ulong part_size;
        ulong part_sizeB;
        disk_partition_t pinfoB;
        const uint32_t main_addr = 0x100000;
        uint32_t cramfs_addr;
        const char format2[] = "setenv cramfsaddr %#x;"
                               "mmc dev %d;"
                               "mmc read %#x %#lx %#lx;"
                               "sha256verify %#x %#lx %d;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "mmc read %#x %#lx %#lx;"
                               "sha256verify %#x %#lx %d;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "loop.l 0xf8000000 1";
        int verify = verify_kernel_partition( );

        /* Search for recovery partition */
        for (partition = 1; partition < 35; partition++) {
            if (get_partition_info(dev_desc, partition, &pinfoB) == 0) {
                if (strcmp(pinfoB.name, "RecoveryKernel") == 0 && pinfoB.data_len != 0) {
                    break;
                }
            }
        }

        /* If there is no recovery partition then just make it the same as the primary */
        if (partition == 35) {
            memcpy(&pinfoB, pinfo, sizeof(disk_partition_t));
        }

        /* Boot from the recovery partition */
        if (boot_recover) {
            memcpy(pinfo, &pinfoB, sizeof(*pinfo));
        }

        /* Load the kernel CRAM partition right in front of U-Boot allowing space for the malloc area */
        cramfs_addr = ((gd->start_addr_sp - max(pinfo->data_len, pinfoB.data_len)) >> 20) << 20;
        part_size = (pinfo->data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;
        part_sizeB = (pinfoB.data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;

        sprintf(cmd, format2, cramfs_addr,
                dev_num,
                cramfs_addr, pinfo->start, part_size,
                cramfs_addr, pinfo->data_len - pinfo->fcert_len + cramfs_addr, verify,
                main_addr,
                main_addr,
                cramfs_addr, pinfoB.start, part_sizeB,
                cramfs_addr, pinfoB.data_len - pinfoB.fcert_len + cramfs_addr, verify,
                main_addr,
                main_addr);
    }
    else {
        const uint32_t uImage_addr = 0x400000;
        const char format[] = "setenv ACTIVE_MMC_PART %d;"
                              "mmc dev %d;"
                              "ext2load mmc %d:%d %#x /boot/uImage;"
                              "setenv bootargs $bootargs root=/dev/mmcblk0p2 rootwait;"
                              "bootm %#x";

        sprintf(cmd, format, part,
                dev_num,
                dev_num, part, uImage_addr,
                uImage_addr);
    }

    return 0;
}

/*
 * Search for valid partitions on the specified device and, if found,
 * check for a custom boot command.
 */
static int sdmmc_get_bootcmd(int dev_num) {
    block_dev_desc_t *dev_desc = NULL;
    struct mmc *mmc;
    int boot_recover = 0;
    const char *card = card_get_type( );

    if (strcmp(card, "card11") == 0) {
        boot_recover = !!gpio_sample(RECOVERY);
        if (boot_recover)
            setenv_ulong("RECOVERY", 1);
    }

    /* Initialize needed devices */
    mmc = find_mmc_device(dev_num);
    if (mmc) {
        int rc = mmc_init(mmc);
        if (rc)
            return 0;

        if (dev_num == 0)
            sdmmc0_clk_enable = 1;
    }

    dev_desc = get_dev("mmc", dev_num);
    if (dev_desc) {
        int part;

        /*
         * Found a device -- go scan it for partitions
         * Restrict the search to the first few partitions.
         */
        for (part = 1; part < 5; part++) {
            disk_partition_t info;

            if (get_partition_info(dev_desc, part, &info) == 0) {
                if (info.size || info.data_len) {
                    char buf[300];

                    /* Found a valid partition */
                    if (mmc_get_bootcmd_for_part(dev_num, part, &info, buf, sizeof(buf), dev_desc, boot_recover) == 0) {
                        setenv_ulong("MMC_DEVICE_NUM", dev_num);
                        setenv_ulong("MMC_RD_BLK", mmc->read_bl_len);
                        setenv("bootcmd", buf);

                        return 1;
                    }
                }
            }
        }
    }

    return 0;
}

static int mmc_get_bootcmd(int dev_num) {
    const char *card = card_get_type( );

    if (strcmp(card, "card09") != 0) {
        gpio_mux(SDMMC0_DATA0, 0);
        gpio_mux(SDMMC0_DATA1, 0);
        gpio_mux(SDMMC0_DATA2, 0);
        gpio_mux(SDMMC0_DATA3, 0);
    }

    mmc_switch_sd0_pins(1);

    return sdmmc_get_bootcmd(dev_num);
}
#endif

#ifdef CONFIG_CMD_NAND
extern int mv61x0_nand_init(struct nand_chip *nand);
extern int mtd_get_partition_info_ssp(struct mtd_info *mtd, struct mtd_partition **partition_table);
extern int lxkshim_init (struct mtd_info *);

static void init_nand_pins(void) {
    int i;
    uint32_t val;
    struct GPIO_REGS_s *gpio_block = (struct GPIO_REGS_s *)APB_GPIOB_BASE;
    volatile uint32_t *pincfg = &gpio_block->PinCfg0;

    for (i=2; i<16; ++i) {
        val = GPIO_PINCFG0_FUNCSEL_REPLACE_VAL(pincfg[i], 1);
        pincfg[i] = GPIO_PINCFG0_PUEN_REPLACE_VAL(val, 0);
    }
}

static void setup_lxkshim(void)
{
    if (nand_info[1].name)
        printk("Already have a second NAND, skipping lxkshim\n");
    else
        lxkshim_init(&nand_info[1]);
}

int board_nand_init(struct nand_chip *nand) {
    if (!nand->IO_ADDR_R)
        return -ENODEV;
    return mv61x0_nand_init(nand);
}

static int nand_get_bootcmd_for_part(const struct mtd_partition *pinfo, const struct mtd_partition *pinfoB, char *cmd, int len) {
    int verify;
    const uint32_t main_addr = 0x100000;
    uint32_t cramfs_addr;
    const char format2[] = "setenv cramfsaddr %#x;"
                           "nand device 1;"
                           "nand read %#x %#llx %#llx;"
                           "sha256verify %#x %#llx %d;"
                           "cramfsload %#x /main.img;"
                           "source %#x;"
                           "nand read %#x %#llx %#llx;"
                           "sha256verify %#x %#llx %d;"
                           "cramfsload %#x /main.img;"
                           "source %#x;";

    if (strcmp(pinfo->name, "Kernel") != 0) /* /boot is in partition 2 */
        return 1;

    /* If there is no second partition then just make it the same as the primary */
    if (pinfoB == NULL) {
        pinfoB = pinfo;
    }

    verify = verify_kernel_partition( );

    cramfs_addr = ((gd->start_addr_sp - max(pinfo->data_len, pinfoB->data_len)) >> 20) << 20;

    sprintf(cmd, format2, cramfs_addr,
            cramfs_addr, pinfo->offset, pinfo->data_len, cramfs_addr, pinfo->data_len - pinfo->fcert_len + cramfs_addr, verify, main_addr, main_addr,
            cramfs_addr, pinfoB->offset, pinfoB->data_len, cramfs_addr, pinfoB->data_len - pinfoB->fcert_len + cramfs_addr, verify, main_addr, main_addr);

    return 0;
}

static int nand_get_bootcmd(int dev_num) {
    struct mtd_info *mtd;

    init_nand_pins( );
    nand_init( );
    setup_lxkshim( );

    mmc_switch_sd0_pins(1);

    mtd = get_mtd_device_nm("lxkshim");
    if (!IS_ERR(mtd)) {
        int part;
        int num_partitions;
        char buf[300];
        struct mtd_partition *partition_table;
        struct mtd_partition *pinfoB = NULL;

        num_partitions = mtd_get_partition_info_ssp(mtd, &partition_table);
        for (part = 0; part < num_partitions; ++part) {
            if (strcmp(partition_table[part].name, "RecoveryKernel") == 0) {
                pinfoB = &partition_table[part];
                break;
            }
        }
        for (part = 0; part < num_partitions; ++part) {
            if (nand_get_bootcmd_for_part(&partition_table[part], pinfoB, buf, sizeof(buf)) == 0) {
                setenv("bootcmd", buf);
                return 1;
            }
        }
    }
    return 0;
}
#endif

#ifndef CONFIG_DRAM_INIT_ONLY
const char * mmc_get_secondary_boot(int dev_num) {
    struct mmc *mmc;
    const char *sec_boot = "";

    mmc = find_mmc_device(dev_num);
    if (mmc) {
        if (mmc_init(mmc) == 0) {
            sec_boot = "eMMC";

            if (dev_num == 0)
                sdmmc0_clk_enable = 1;
        }
    }

    return sec_boot;
}

const char * nandmmc_get_secondary_boot(int dev_num) {
    const char *sec_boot = mmc_get_secondary_boot(dev_num);

#ifdef CONFIG_CMD_NAND
    if (strlen(sec_boot) <= 0) {
        init_nand_pins( );
        nand_init( );
        setup_lxkshim( );

        if (nand_info[1].name)
            sec_boot = "nand";
    }
#endif

    return sec_boot;
}

/*
 * Set up the bootcmd environment variable as needed for the current
 * code store configuration.  Do this at run-time in order to share
 * a common U-Boot binary between multiple platforms.
 * The algorithm is as follows:
 *    Search each device in dev_list[] for valid EXT partitions.
 *      If found, check if a custom boot command exists for that device.
 *        If so, use it.
 */
static void setup_bootcmd(int ignore_dc) {
    int i;
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;
    struct {
        const char *dev_name;
        int dev_num;
        int lor_pins;
        int cs;
        int dc;
        int (*get_bootcmd)(int dev_num);
        const char * (*get_secondary_boot)(int dev_num);
    } dev_list[] =
    {
#ifdef CONFIG_MMC
        { "SD",   0, 2, 0, 0, sdmmc_get_bootcmd, mmc_get_secondary_boot     },
        { "SD",   0, 4, 0, 0, sdmmc_get_bootcmd, mmc_get_secondary_boot     },
        { "eMMC", 0, 5, 0, 1, mmc_get_bootcmd,   nandmmc_get_secondary_boot },
        { "eMMC", 0, 6, 0, 1, mmc_get_bootcmd,   nandmmc_get_secondary_boot },
        { "eMMC", 1, 5, 1, 0, mmc_get_bootcmd,   mmc_get_secondary_boot     },
        { "eMMC", 0, 2, 0, 0, mmc_get_bootcmd,   nandmmc_get_secondary_boot },
#endif
#ifdef CONFIG_CMD_NAND
        { "eMMC", 0, 1, 0, 1, mmc_get_bootcmd,   nandmmc_get_secondary_boot },
        { "nand", 0, 1, 1, 0, nand_get_bootcmd,  mmc_get_secondary_boot     },
#endif
    };

    /* Search for valid code devices */
    for (i = 0; (i < ARRAY_SIZE(dev_list)); i++) {
        if ((((APB_CONFIG_ASR_CONFIGBOOTSTATUS_MASK_SHIFT(apb_config->ASR) >> 5) & 7) == dev_list[i].lor_pins) &&
            (!dev_list[i].dc || !ignore_dc)) {
            if (dev_list[i].get_bootcmd(dev_list[i].dev_num)) {
                char buf[20];
                const char *sec_boot = dev_list[i].get_secondary_boot(!dev_list[i].cs);

                setenv_ulong("bootCs", dev_list[i].cs);

                sprintf(buf, "%s,%s", (char *)dev_list[i].dev_name, sec_boot);
                setenv("BOOTDEV", buf);

                printf("%s:%d bootcmd: %s\n", dev_list[i].dev_name, dev_list[i].dev_num, getenv("bootcmd"));
                return;
            }
        }
    }

    printf("No boot command found.\n");
}
#endif

int is_min_func_mode(void) {
    struct VCF_WRAPPER_REGS_s *vcf_wrapper = (struct VCF_WRAPPER_REGS_s *)VCF_WRAPPER_BASE;
    
    return !!(vcf_wrapper->VWSR & VCF_WRAPPER_VWSR_MINFUNC_MASK);
}

/* Late initialization -- after most subsystems are up, including the environ */
int board_late_init (void)
{
    const char *ptr;
    char buf[300];
    char *bootargs;
    int pci_enable = 0;
    int i;
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

#ifdef CONFIG_LXK_PANEL
    lxk_panel_init();
    if(lxk_panel_clk_required() == LXK_PANEL_EXT_CLK_UNUSED)
        lcd_clk_gpio_disable();   
    lxk_panel_status(LXK_PANEL_BLANK);
#endif

    /*
     * Do not load code if the boot strap bits indicate booting from USB.
     * The code store could contain corrupt data.
     * The magic numbers are from the ROM spec: Boot_Source_b[2:0].
     */
    if (((APB_CONFIG_ASR_CONFIGBOOTSTATUS_MASK_SHIFT(apb_config->ASR) >> 5) & 3) == 3) {
        struct mmc *mmc;
        const char *lbootdev = "SD";
        struct fcert_v3 *fcert = (struct fcert_v3 *)0x2000000;

        mmc = find_mmc_device(0);
        if (mmc) {
            if ((mmc_init(mmc) == 0) && !IS_SD(mmc)) {
                lbootdev = "eMMC";

                if ((strcmp(card_get_type( ), "card09") != 0) && (strcmp(card_get_type( ), "card11") != 0)) {
                    /* Enable the pins for mmc:0 */
                    gpio_mux(SDMMC0_DATA0, 0);
                    gpio_mux(SDMMC0_DATA1, 0);
                    gpio_mux(SDMMC0_DATA2, 0);
                    gpio_mux(SDMMC0_DATA3, 0);

                    mmc_switch_sd0_pins(1);
                }
            }
        }

        sdmmc0_clk_enable = 1;

        sprintf(buf, "dfu 0 ram 0;"
                     "setenv BOOTDEV %s,;"
                     "setenv CFTRECOVER 1;"
                     "setenv cramfsaddr 0x2000208;"
                     "sha256verify 0x2000208 0x2000000 1;"
                     "cramfsload 0x100000 /ufcert;bootm 0x100000;"
                     "cramfsload 0x100000 /ucmdline;bootm 0x100000;"
                     "cramfsload 0x400000 /uImage;cramfsload 0x1000000 /uInitramfs;bootm 0x400000 0x1000000"
                     , lbootdev);
        setenv("dfu_alt_info", "kernel ram 0x2000000 0x1000000");
        setenv("bootcmd", buf);
        printf("bootcmd: %s\n", buf);
    }
#ifndef CONFIG_DRAM_INIT_ONLY
    else {
        int ignore_dc = 0;

#ifdef CONFIG_LXK_PANEL
        char *keys = getenv("PORKEYS");

        if (keys) {
            unsigned int por_keys = simple_strtoul(keys, NULL, 0);
            int panel_type = lxk_panel_get_type( );

            if ((panel_type == LXK_PANEL_TYPE_ATMEL_2LINE) || (panel_type == LXK_PANEL_TYPE_ATMEL_LED)) {
                if (por_keys == (1 << 5))
                    ignore_dc = 1;
            }
            else if (por_keys == NM_HOME)
                ignore_dc = 1;
        }
#endif
        setup_bootcmd(ignore_dc);
    }
#endif

    if (!sdmmc0_clk_enable) {
        gpio_direction(SDMMC0_CLK, 0);
        gpio_mux(SDMMC0_CLK, 0);
    }

    if (!(APB_CONFIG_ASR_CONFIGBOOTSTATUS_MASK_SHIFT(apb_config->ASR) & 1))
        setenv_ulong("bspi_enabled", 1);

    setenv_ulong("DRAM_SIZE", gd->ram_size);
    setenv_ulong("mem_MiB", gd->ram_size >> 20);

    buf[0] = 0;
    for (i=0; i<CONFIG_NR_DRAM_BANKS; ++i) {
        sprintf(buf, "%s,%ld", buf, dram_sizes[i] >> 20);
    }
    setenv("DRAM_CONFIG", &buf[1]); /* Remove leading comma */

    setenv_ulong("CPUMHZ", (unsigned long) board_get_core_freq_hz() / 1000000);
    setenv_ulong("BUSFREQ", (unsigned long) board_get_bus_freq_hz());
    setenv("ASICREV", asic_rev);
        
    setenv_ulong("APM_SETTING", read_apm_setting());

#ifdef CONFIG_LXK_PANEL
    setenv_ulong("PANELTYPE_ENV", lxk_panel_get_type());

    ptr = lxk_panel_get_subtype();
    if(ptr)
        setenv("PANELSUBTYPE", (char *)ptr);
#endif

    ptr = card_get_type( );
    if (ptr)
        setenv("CARDTYPE", (char *)ptr);

    ptr = card_get_revision( );
    if (ptr)
        setenv("CARDREV", (char *)ptr);

#if defined(CONFIG_CARD_03) \
    || defined(CONFIG_CARD_11)
    if (!(is_min_func_mode( ))) {
		pci_enable = 1;

        if (pcie_is_endpoint()) {
            setenv("pci", "endpoint");
        }
    }
#endif

    bootargs = getenv("bootargs");
    sprintf(buf, "%s BUSFREQ=%lu ASICREV=%s%s", bootargs,
		   	(unsigned long)board_get_bus_freq_hz(),
		   	getenv("ASICREV"),
			pci_enable ? "" : " pci=none");
    setenv("bootargs", buf);

#if defined(CONFIG_CARD_01) \
    || defined(CONFIG_CARD_03) \
    || defined(CONFIG_CARD_05)
    sprintf(buf, "0x%08x", (XENA_BASE + 0x10000));
    setenv("ITCM0", buf);

    sprintf(buf, "0x%08x", XENA_BASE);
    setenv("ITCM1", buf);
#endif

    if (is_min_func_mode( ))
        setenv("MIN_FUNC_MODE", "1");

    /*
     * There are a few quirks that the kernel and initramfs need
     * to know about when on the FPGA build.
     */
    init_hips_pll(asic_rev[0] == 'A');

#if !(defined(CONFIG_CARD_05))
    /*
     * Power down LVDS pads that are used by AFE (only MFP uses Granite
     * imgpipe needed LVDS pads enabled) to save power.
     */
    apb_config->PDCR2 = APB_CONFIG_PDCR2_ILVDS_PD_REPLACE_VAL(apb_config->PDCR2, 0xf);
#endif

#ifdef CONFIG_CRASH_KERNEL_START_MB
    snprintf(buf, sizeof(buf), "%dM@%dM", CONFIG_CRASH_KERNEL_SIZE_MB,
            CONFIG_CRASH_KERNEL_START_MB);
    setenv("crashkernel", buf);
#endif

    return 0;
}

static void ramp_up_pll0(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    /* Setup the core voltage before ramping up the PLL */
    card_config_core_voltage(asic_rev);

    /*
     * If we are already running from DRAM, do not modify the PLL settings
     * as it could cause flakey behavior.
     */
    if ((unsigned int) gd >= 0xC0000000) {
        volatile unsigned int tmp = 10000;

        apb_config->PLL0_CR0 |= APB_CONFIG_PLL0_CR0_BYPASS_EN_MASK;
        apb_config->PLL0_CR0 = APB_CONFIG_PLL0_CR0_VCODIV_SEL_SE_REPLACE_VAL(apb_config->PLL0_CR0, 0);
        apb_config->PLL0_CR0 = APB_CONFIG_PLL0_CR0_FBDIV_REPLACE_VAL(apb_config->PLL0_CR0, 0xc0);

        /* Delay for new value to propagate through the post divider state machine */
        while (tmp--);

        while(!(apb_config->PLL0_SR & APB_CONFIG_PLL0_SR_LOCK_MASK)); // wait for the lock

        apb_config->CDCR1 = APB_CONFIG_CDCR1_BUSDIV_REPLACE_VAL(apb_config->CDCR1, 0x3);

        apb_config->PLL0_CR0 = APB_CONFIG_PLL0_CR0_BYPASS_EN_REPLACE_VAL(apb_config->PLL0_CR0, 0);
    }
}

static uint32_t get_memAddrMap(int i) {
    uint32_t memAddrMap;
    struct MC_REGS_s *mc_reg = (struct MC_REGS_s *)MC_BASE;

    switch (i) {
        default:
        case 0:  memAddrMap = mc_reg->MemAddrMap0; break;
        case 1:  memAddrMap = mc_reg->MemAddrMap1; break;
        case 2:  memAddrMap = mc_reg->MemAddrMap2; break;
    }

    return memAddrMap;
}

void dram_init_banksize(void)
{
    int i;

    for (i=0; i<CONFIG_NR_DRAM_BANKS; ++i) {
        gd->bd->bi_dram[i].start = get_memAddrMap(i) & MC_MEMADDRMAP0_STARTADDR_MASK;
        gd->bd->bi_dram[i].size  = dram_sizes[i];
    }
}

/******************************
 Routine:
 Description:
******************************/
static long dram_size_from_mask(long dram_mask) {
    return (1 << ((dram_mask - 4) + 20));
}

int dram_init(void) {
    int i;
    uint32_t memAddrMap;
    struct ADC_REGS_REGS_s *adc = (struct ADC_REGS_REGS_s *)DEC_ADC_REGS_BASE;

    ramp_up_pll0( );

    /*
     * Some cards need to use the ADC to figure out what the DDR bus width is.
     * The ADC runs off the USB PLL so this has to be running before calling
     * card_dram_init.
     */
    USBPhyInit((void *) USB_USB2PHY_USB2_UTM1_BASE);

    /*
     * Set the rotate bits for the adc
     * This has to take place after the usb is initialized.  This fixes
     * a default problem in this register
     */
    adc->SARADC_Control = ADC_REGS_SARADC_CONTROL_R_ROTATE_SEL_REPLACE_VAL(adc->SARADC_Control,1);

    /* If already running in DRAM (as in USB recovery case), then skip
     * initialization -- it has already been completed and doing it again
     * while running from DRAM can corrupt the code.
     * The DRAM aperture is [0,0xBFFFFFFF].
     */
    if ((unsigned int) gd >= 0xC0000000)
        card_dram_init( );

    gd->ram_size = 0;

    for (i=0; i<CONFIG_NR_DRAM_BANKS; ++i) {
        memAddrMap = get_memAddrMap(i);

        if (memAddrMap & MC_MEMADDRMAP0_VALID_MASK) {
            dram_sizes[i] = dram_size_from_mask(MC_MEMADDRMAP0_AREALEN_MASK_SHIFT(memAddrMap));
            dram_sizes[i] = get_ram_size((long *)(memAddrMap & MC_MEMADDRMAP0_STARTADDR_MASK), dram_sizes[i]);

            gd->ram_size += dram_sizes[i];
        }
        else {
            dram_sizes[i] = 0;
        }
    }

    return 0;
}
