#include <common.h>
#include <ddr_spd.h>
#include <i2c.h>
#include <asm/arch/board_priv.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/MC_regmasks.h>
#include <asm/arch/MC_regstructs.h>

#include "ddr.h"

#ifdef DDR_DEBUG
static void print_ddr3_spd(const ddr3_spd_eeprom_t *spd, unsigned char addr) {
    printf("%s:%d addr:%02x\n", __func__, __LINE__, addr);
    printf("info_size_crc:%02x\n"
           "spd_rev      :%02x\n"
           "mem_type     :%02x\n"
           "module_type  :%02x\n"
           "density_banks:%02x\n",
            spd->info_size_crc,
            spd->spd_rev,
            spd->mem_type,
            spd->module_type,
            spd->density_banks
          );
    printf("addressing   :%02x\n"
           "module_vdd   :%02x\n"
           "organization :%02x\n"
           "bus_width    :%02x\n"
           "ftb_div      :%02x\n",
            spd->addressing,
            spd->module_vdd,
            spd->organization,
            spd->bus_width,
            spd->ftb_div
          );
    printf("mtb_dividend :%02x\n"
           "mtb_divisor  :%02x\n"
           "tCK_min      :%02x\n"
           "res_13       :%02x\n"
           "caslat_lsb   :%02x\n",
            spd->mtb_dividend,
            spd->mtb_divisor,
            spd->tCK_min,
            spd->res_13,
            spd->caslat_lsb
          );
    printf("caslat_msb   :%02x\n"
           "tAA_min      :%02x\n"
           "twr_min      :%02x\n"
           "trcd_min     :%02x\n"
           "trrd_min     :%02x\n",
            spd->caslat_msb,
            spd->tAA_min,
            spd->twr_min,
            spd->trcd_min,
            spd->trrd_min
          );
    printf("trp_min      :%02x\n"
           "tras_trc_ext :%02x\n"
           "tras_min_lsb :%02x\n"
           "trc_min_lsb  :%02x\n"
           "trfc_min_lsb :%02x\n",
            spd->trp_min,
            spd->tras_trc_ext,
            spd->tras_min_lsb,
            spd->trc_min_lsb,
            spd->trfc_min_lsb
          );
    printf("trfc_min_msb :%02x\n"
           "twtr_min     :%02x\n"
           "trtp_min     :%02x\n"
           "tfaw_msb     :%02x\n"
           "tfaw_min     :%02x\n"
           "opt_features :%02x\n",
            spd->trfc_min_msb,
            spd->twtr_min,
            spd->trtp_min,
            spd->tfaw_msb,
            spd->tfaw_min,
            spd->opt_features
          );
}
#endif

static void ddr3_disable_cpu_watchdog(void) {
    volatile unsigned int *ip = (volatile unsigned int *)0xf8fff000;

    /* Disable CPU watchdog */
    *ip = 1;
}

static uint32_t ddr3_get_spd(ddr3_spd_eeprom_t *spd) {
    unsigned char addr;
    int rc;
    uint32_t dimm_found = 0;

    /* Get the serial presence detect information */
    I2C_SET_BUS(CONFIG_SYS_SPD_BUS_NUM);
    for (addr=0x50; addr<0x58; ++addr) {
        rc = i2c_read(addr, 0, 1, (unsigned char *)spd, sizeof(*spd));
        if ((rc == 0) && (spd->mem_type == SPD_MEMTYPE_DDR3) && (spd->module_type == DDR3_SPD_MODULETYPE_SO_DIMM)) {
#ifdef DDR_DEBUG
            print_ddr3_spd(spd, addr);
#endif
            dimm_found = 1;
            break;
        }
    }
    I2C_SET_BUS(CONFIG_I2C_DEFAULT_BUS);

    return dimm_found;
}

static uint32_t ddr3_mtb_to_ps(uint32_t value, uint32_t dividend, uint32_t divisor) {
    return ((value * dividend * 1000) / divisor);
}

static uint32_t ddr3_ps_to_clocks(uint32_t value, uint32_t ps_per_cycle) {
    return ((value + (ps_per_cycle - 1)) / ps_per_cycle);
}

static int ddr3_write_level_byte_lane(int ChipSel, int ByteSelTmp, uint32_t dq_dqs_delay) {
    int regVal;
    int ByteDel = 0;
    int phy_wl_dq_out;
    struct MC_REGS_s *mc_regs = (struct MC_REGS_s *)MC_BASE;

    /* Take encoded ChipSel input and turn into one-hot */
    ChipSel = 1 << ChipSel;

    /* Enable Write-leveling Mode */
    regVal = MC_SDRAMCONTROL2_WRITELEVELMODE_REPLACE_VAL(mc_regs->SDRAMControl2, 1);
    regVal = MC_SDRAMCONTROL2_REFPOSTEDMAX_REPLACE_VAL(regVal, 7);
    mc_regs->SDRAMControl2 = MC_SDRAMCONTROL2_REFPOSTEDEN_REPLACE_VAL(regVal, 1);

    /* Set Write-leveling Mode in CS0 DRAM */
    mc_regs->SDRAMControl1 = MC_SDRAMCONTROL1_RESERVED4_REPLACE_VAL(mc_regs->SDRAMControl1, 1);

    /* Send MRS1 to DRAM to Enable Write-leveling */
    regVal = 0;
    regVal = MC_USERCOMMAND_CHIPSELECT_REPLACE_VAL(regVal, ChipSel);
    mc_regs->UserCommand = MC_USERCOMMAND_USERLMREQ1_REPLACE_VAL(regVal, 1);

    /* Need to satisfy ODT high to MRS (12 cycles). */
    udelay(1);

    /* Set ODT for Write-leveling Mode */
    regVal = (mc_regs->SDRAMControl7 & 0xFFFFFFF) | (5 << 28);    // JPS per John Berry 21-Nov-2011
    regVal = MC_SDRAMCONTROL7_ODT0SWITCHMODE_REPLACE_VAL(regVal, 0);
    regVal = MC_SDRAMCONTROL7_ODT1SWITCHMODE_REPLACE_VAL(regVal, 0);
    regVal = MC_SDRAMCONTROL7_ODT2SWITCHMODE_REPLACE_VAL(regVal, 0);
    mc_regs->SDRAMControl7 = MC_SDRAMCONTROL7_ODT3SWITCHMODE_REPLACE_VAL(regVal, 0);

    /* Need to satisfy ODT high to MRS (12 cycles). */
    udelay(1);

    /* Set CS and Byte for Write-leveling */
    regVal = MC_PHYWRITELEVELSELECT_CSSEL_REPLACE_VAL(mc_regs->PHYWriteLevelSelect, ChipSel);
    mc_regs->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_BYTESEL_REPLACE_VAL(regVal, ByteSelTmp);

    /* Need to satisfy ODT high to MRS (12 cycles). */
    udelay(1);

    do { /* until detect clock is low at DRAM */
        /* Measured that the best DQ/DQS delay is 0x1F add to end of ByteDel + */
        regVal = MC_PHYWLCONTROL_PHY_WL_WCK_DQ_DLY_REPLACE_VAL(mc_regs->PHYWLControl, ByteDel - dq_dqs_delay);
        mc_regs->PHYWLControl = MC_PHYWLCONTROL_PHY_WL_WCK_QS_DLY_REPLACE_VAL(regVal, ByteDel);

        mc_regs->PHYControl15 = MC_PHYCONTROL15_PHY_WL_DQS_PULSE_REPLACE_VAL(mc_regs->PHYControl15, 1);

        phy_wl_dq_out = MC_PHYCONTROL15_PHY_WL_DQ_OUT_MASK_SHIFT(mc_regs->PHYControl15);
        ByteDel = ByteDel + 1;
    } while((ByteDel < 128) &&
            (((ByteSelTmp == 0) && ((phy_wl_dq_out & 0x4) != 0)) ||
             ((ByteSelTmp == 1) && ((phy_wl_dq_out & 0x1) != 0)) ||
             ((ByteSelTmp == 2) && ((phy_wl_dq_out & 0x8) != 0)) ||
             ((ByteSelTmp == 3) && ((phy_wl_dq_out & 0x2) != 0)) ));

    do { /* until detect clock is high at DRAM (rising edge) */
        /* Measured that the best DQ/DQS delay is 0x1F */
        regVal = MC_PHYWLCONTROL_PHY_WL_WCK_DQ_DLY_REPLACE_VAL(mc_regs->PHYWLControl, ByteDel - dq_dqs_delay);
        mc_regs->PHYWLControl = MC_PHYWLCONTROL_PHY_WL_WCK_QS_DLY_REPLACE_VAL(regVal, ByteDel);

        mc_regs->PHYControl15 = MC_PHYCONTROL15_PHY_WL_DQS_PULSE_REPLACE_VAL(mc_regs->PHYControl15, 1);

        phy_wl_dq_out = MC_PHYCONTROL15_PHY_WL_DQ_OUT_MASK_SHIFT(mc_regs->PHYControl15);
        ByteDel = ByteDel + 1;
    } while((ByteDel < 128) &&
            (((ByteSelTmp == 0) && ((phy_wl_dq_out & 0x4) == 0)) ||
             ((ByteSelTmp == 1) && ((phy_wl_dq_out & 0x1) == 0)) ||
             ((ByteSelTmp == 2) && ((phy_wl_dq_out & 0x8) == 0)) ||
             ((ByteSelTmp == 3) && ((phy_wl_dq_out & 0x2) == 0)) ));

    if (ByteDel < (dq_dqs_delay+1)) {
        mc_regs->PHYWLControl = MC_PHYWLCONTROL_PHY_WL_WCK_DQ_DLY_REPLACE_VAL(mc_regs->PHYWLControl, 0);
    }

    /* Disable write-leveling */
    regVal = (mc_regs->SDRAMControl7 & 0xFFFFFFF) | (0 << 28);
    regVal = MC_SDRAMCONTROL7_ODT0SWITCHMODE_REPLACE_VAL(regVal, 2);
    regVal = MC_SDRAMCONTROL7_ODT1SWITCHMODE_REPLACE_VAL(regVal, 2);
    regVal = MC_SDRAMCONTROL7_ODT2SWITCHMODE_REPLACE_VAL(regVal, 2);
    mc_regs->SDRAMControl7 = MC_SDRAMCONTROL7_ODT3SWITCHMODE_REPLACE_VAL(regVal, 2);

    /* Need to satisfy ODT high to MRS (12 cycles). */
    udelay(1);

    /* Disable Write-leveling Mode in CS0 DRAM */
    mc_regs->SDRAMControl1 = MC_SDRAMCONTROL1_RESERVED4_REPLACE_VAL(mc_regs->SDRAMControl1, 0);

    /* Send MRS1 to DRAM to Disable Write-leveling */
    regVal = 0;
    regVal = MC_USERCOMMAND_CHIPSELECT_REPLACE_VAL(regVal, ChipSel);
    mc_regs->UserCommand = MC_USERCOMMAND_USERLMREQ1_REPLACE_VAL(regVal, 1);

    /* Disable Write-leveling Mode */
    regVal = MC_SDRAMCONTROL2_WRITELEVELMODE_REPLACE_VAL(mc_regs->SDRAMControl2, 0);
    regVal = MC_SDRAMCONTROL2_REFPOSTEDMAX_REPLACE_VAL(regVal, 0);
    mc_regs->SDRAMControl2 = MC_SDRAMCONTROL2_REFPOSTEDEN_REPLACE_VAL(regVal, 0);

    return (ByteDel < 128);
}

static void ddr3_write_level(int ddr3_width, uint32_t dimm_found) {
    int ChipSel;
    uint32_t clk_delay = 0xc, dq_dqs_delay = 0xc;
    struct MC_REGS_s *mc_reg = (struct MC_REGS_s *)MC_BASE;

    /*
     * The mv61x0 seems to have delay lines for clock (and may address/control) mis-wired so settings for 0 affect 1 and vice versa.
     * Therefore, change the clock delays for all chip selects first, then do the write leveling.
     */
    for (ChipSel = (dimm_found) ? 0 : 2; ChipSel < 3; ChipSel++) {
       mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_CSSEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 1<<ChipSel);

       // CWC 3-11-11 change clock delay so data delays can be "negative"
       mc_reg->PHYWriteLevelClock0 = MC_PHYWRITELEVELCLOCK0_CK_REPLACE_VAL(mc_reg->PHYWriteLevelClock0, clk_delay);
       mc_reg->PHYWriteLevelClock1 = MC_PHYWRITELEVELCLOCK1_AC_REPLACE_VAL(mc_reg->PHYWriteLevelClock1, clk_delay);
    }

    for (ChipSel = (dimm_found) ? 0 : 2; ChipSel < 3; ChipSel++) {
        if (ddr3_width == DDR3_x32) {
            int ByteSel;

            for (ByteSel = 0; ByteSel < 4; ByteSel++) {
                if (!ddr3_write_level_byte_lane(ChipSel, ByteSel, dq_dqs_delay))
                    printf("Write-leveling failed cs:%d bl:%d\n", ChipSel, ByteSel);
            }
        }
        else {
            if (!ddr3_write_level_byte_lane(ChipSel, 1, dq_dqs_delay))
                printf("Write-leveling failed cs:%d bl:%d\n", ChipSel, 1);
            if (!ddr3_write_level_byte_lane(ChipSel, 3, dq_dqs_delay))
                printf("Write-leveling failed cs:%d bl:%d\n", ChipSel, 3);
        }
    }

    // Un-comment this section if we want to display write leveling values at boot time.
#if 0
    for (ChipSel = 0; ChipSel < 3; ChipSel++) {
        if (ddr3_width == DDR3_x32) {
            int ByteSel;
            for (ByteSel = 0; ByteSel < 4; ByteSel++) {
               mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_CSSEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 1<<ChipSel);
               mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_BYTESEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, ByteSel);
               printf("CS: %d, Byte: %d, PHYWLControl = 0x%08X\r\n", ChipSel, ByteSel,  mc_reg->PHYWLControl);
            }
        }
        else {
            mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_CSSEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 1<<ChipSel);
            mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_BYTESEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 1);
            printf("CS: %d, Byte: 1, PHYWLControl = 0x%08X\r\n", ChipSel,  mc_reg->PHYWLControl);
            mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_CSSEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 1<<ChipSel);
            mc_reg->PHYWriteLevelSelect = MC_PHYWRITELEVELSELECT_BYTESEL_REPLACE_VAL(mc_reg->PHYWriteLevelSelect, 3);
            printf("CS: %d, Byte: 3, PHYWLControl = 0x%08X\r\n", ChipSel,  mc_reg->PHYWLControl);
        }
    }    
#endif

}

int initDDR3_nt5cb128m16dp_be_MHz400(int ddr3_width) {
    int i;
    uint32_t dimm_found = 0;
    ddr3_spd_eeprom_t spd;
    struct MC_REGS_s *mc_reg = (struct MC_REGS_s *)MC_BASE;
    unsigned int ddr_freq = board_get_core_freq_hz( ) / 2;
    uint32_t ps_per_cycle = (ddr_freq == 160000000) ? 6250 : 2500;
    uint32_t val;
    uint32_t tWR  = 15000,
             tRCD = 13125,
             tRRD = max((4 * ps_per_cycle), 10000),
             tRP  = 13125,
             tRC  = 50625,
             tRAS = 37500,
             tRFC = 300000,
             tWTR = max((4 * ps_per_cycle), 7500),
             tRTP = max((4 * ps_per_cycle), 7500),
             tFAW = 50000;

    ddr3_disable_cpu_watchdog( );

    dimm_found = ddr3_get_spd(&spd);

    mc_reg->PHYControl11  = 0x00300000;
    // New values per John Berry - JPS 21-Nov-2011
    mc_reg->SDRAMConfig0  = 0x00266530;
    mc_reg->SDRAMConfig1  = 0x00206530;
    mc_reg->SDRAMConfig2  = 0x00246530;  // Would like 60 ohm, but ASIC is not setting 60 ohm correctly.
//    mc_reg->SDRAMConfig2  = 0x00026420;  // Micron

    if (dimm_found) {
        uint32_t dimm_width, bits_per_chip, num_chips, rank_order = 0, num_rows, num_cols;
        uint64_t tmp, rank_size;

        /* Calculate the bus width */
        dimm_width = 1 << ((spd.bus_width & 7) + 3);

        /* Calculate the number of bits per chip */
        bits_per_chip = 1 << ((spd.organization & 7) + 2);

        num_chips = dimm_width / bits_per_chip;

        /* Calculate the size of each rank */
        rank_size = (uint64_t)1 << ((spd.density_banks & 0xf) + 28);
        rank_size >>= 3; /* Convert bits to bytes */
        rank_size *= num_chips;

        tmp = rank_size >> 20;
        while (!(tmp & (1 << rank_order)))
            ++rank_order;
        rank_order += 4;

        mc_reg->MemAddrMap0 = (0 & MC_MEMADDRMAP0_STARTADDR_MASK) |
                              MC_MEMADDRMAP0_AREALEN_REPLACE_VAL(0, rank_order) |
                              MC_MEMADDRMAP0_VALID_MASK;

        mc_reg->MemAddrMap1 = (rank_size & MC_MEMADDRMAP1_STARTADDR_MASK) |
                              MC_MEMADDRMAP1_AREALEN_REPLACE_VAL(0, rank_order) |
                              MC_MEMADDRMAP1_VALID_MASK;

        mc_reg->MemAddrMap2 = ((rank_size * 2) & MC_MEMADDRMAP2_STARTADDR_MASK) |
                              MC_MEMADDRMAP2_AREALEN_REPLACE_VAL(0, min((rank_order + 1), 0xe)) |
                              MC_MEMADDRMAP2_VALID_MASK;

        /* Set up the correct number of rows and columns */
        num_rows = (((spd.addressing >> 3) & 0x7) + 12) - 10;
        num_cols = ((spd.addressing & 0x7) + 9) - 7;

        val = MC_SDRAMCONFIG0_ROW_REPLACE_VAL(mc_reg->SDRAMConfig0, num_rows);
        mc_reg->SDRAMConfig0 = MC_SDRAMCONFIG0_COL_REPLACE_VAL(val, num_cols);

        val = MC_SDRAMCONFIG1_ROW_REPLACE_VAL(mc_reg->SDRAMConfig1, num_rows);
        mc_reg->SDRAMConfig1 = MC_SDRAMCONFIG1_COL_REPLACE_VAL(val, num_cols);

        /* If the DIMM requires longer times, override the timing parameters */
        tRTP = max(tRTP, ddr3_mtb_to_ps(spd.trtp_min, spd.mtb_dividend, spd.mtb_divisor));
        tWTR = max(tWTR, ddr3_mtb_to_ps(spd.twtr_min, spd.mtb_dividend, spd.mtb_divisor));
        tRC  = max(tRC,  ddr3_mtb_to_ps(((((spd.tras_trc_ext >> 4) & 0xf) << 8) | spd.trc_min_lsb), spd.mtb_dividend, spd.mtb_divisor));
        tRP  = max(tRP,  ddr3_mtb_to_ps(spd.trp_min,  spd.mtb_dividend, spd.mtb_divisor));
        tRRD = max(tRRD, ddr3_mtb_to_ps(spd.trrd_min, spd.mtb_dividend, spd.mtb_divisor));
        tRCD = max(tRCD, ddr3_mtb_to_ps(spd.trcd_min, spd.mtb_dividend, spd.mtb_divisor));
        tWR  = max(tWR,  ddr3_mtb_to_ps(spd.twr_min,  spd.mtb_dividend, spd.mtb_divisor));
        tRFC = max(tRFC, ddr3_mtb_to_ps(((spd.trfc_min_msb << 8) | spd.trfc_min_lsb), spd.mtb_dividend, spd.mtb_divisor));
        tRAS = max(tRAS, ddr3_mtb_to_ps((((spd.tras_trc_ext & 0xf) << 8) | spd.tras_min_lsb), spd.mtb_dividend, spd.mtb_divisor));
        tFAW = max(tFAW, ddr3_mtb_to_ps((((spd.tfaw_msb & 0xf) << 8) | spd.tfaw_min), spd.mtb_dividend, spd.mtb_divisor));
    }
    else {
        mc_reg->MemAddrMap0 = 0x800e0001;
        mc_reg->MemAddrMap1 = 0x400e0001;
        mc_reg->MemAddrMap2 = 0x000e0001;
    }

    /* Convert from ps to clocks */
    tRTP = ddr3_ps_to_clocks(tRTP, ps_per_cycle);
    tWTR = ddr3_ps_to_clocks(tWTR, ps_per_cycle);
    tRC  = ddr3_ps_to_clocks(tRC,  ps_per_cycle);
    tRP  = ddr3_ps_to_clocks(tRP,  ps_per_cycle);
    tRRD = ddr3_ps_to_clocks(tRRD, ps_per_cycle);
    tRCD = ddr3_ps_to_clocks(tRCD, ps_per_cycle);
    tWR  = ddr3_ps_to_clocks(tWR,  ps_per_cycle);
    tRFC = ddr3_ps_to_clocks(tRFC, ps_per_cycle);
    tRAS = ddr3_ps_to_clocks(tRAS, ps_per_cycle);
    tFAW = ddr3_ps_to_clocks(tFAW, ps_per_cycle);

    val = MC_SDRAMTIMING1_TCCD_REPLACE_VAL(0, 4);
    val = MC_SDRAMTIMING1_TRTP_REPLACE_VAL(val, tRTP);
    val = MC_SDRAMTIMING1_TWTR_REPLACE_VAL(val, tWTR);
    val = MC_SDRAMTIMING1_TRC_REPLACE_VAL(val,  tRC);
    mc_reg->SDRAMTiming1 = MC_SDRAMTIMING1_TREFC_REPLACE_VAL(val, 0xc0);

    val = MC_SDRAMTIMING2_TRP_REPLACE_VAL(0, tRP);
    val = MC_SDRAMTIMING2_TRRD_REPLACE_VAL(val, tRRD);
    val = MC_SDRAMTIMING2_TRCD_REPLACE_VAL(val, tRCD);
    val = MC_SDRAMTIMING2_TWR_REPLACE_VAL(val,  tWR);
    val = MC_SDRAMTIMING2_TRFC_REPLACE_VAL(val, tRFC);
    mc_reg->SDRAMTiming2 = MC_SDRAMTIMING2_TMRD_REPLACE_VAL(val, 4);

    mc_reg->SDRAMTiming3  = 0xc200c853;

    if (ddr_freq == 160000000)
        mc_reg->SDRAMTiming4  = 0x421858a3;
    else
        mc_reg->SDRAMTiming4  = 0x44f8b587;

    val = MC_SDRAMTIMING5_TRAS_REPLACE_VAL(0, tRAS);
    val = MC_SDRAMTIMING5_TCCD_CCS_WR_EXT_DLY_REPLACE_VAL(val, 7);
    val = MC_SDRAMTIMING5_TFAW_REPLACE_VAL(val, tFAW);
    mc_reg->SDRAMTiming5 = MC_SDRAMTIMING5_TCCD_CCS_EXT_DLY_REPLACE_VAL(val, 1);

    mc_reg->SDRAMTiming6  = 0x84040200;

    mc_reg->SDRAMControl1 = 0xa7b00000; /* 5 us of idle time before enter auto power saving mode */
    if (ddr3_width == DDR3_x32) {
        mc_reg->SDRAMControl2 = 0x00080000;
        mc_reg->SDRAMControl4 = 0x20c08009;
    }
    else {
        mc_reg->SDRAMControl2 = 0x00040000;
        mc_reg->SDRAMControl4 = 0x20c08008;
    }
    mc_reg->SDRAMControl5 = 0x00060403;
    mc_reg->SDRAMControl6 = 0x00340245;   // JPS 21-Nov-2011 per John Berry
    mc_reg->SDRAMControl7 = 0x000000aa;   // JPS 21-Nov-2011 per John Berry
    mc_reg->MCBControl1   = 0x0905040c;
    mc_reg->MCBControl2   = 0x07040501;
    mc_reg->MCBControl4   = 0x00000e0e;
    mc_reg->PHYControl3   = 0x20004033;
    mc_reg->PHYControl7   = 0x177c288b;
    mc_reg->PHYControl8   = 0x0ff0088a;
    mc_reg->PHYControl9   = 0x00200088;
    mc_reg->PHYControl10  = 0x001135d8;

    for (i=3; i>=0; --i) {
        mc_reg->PHYDLLSelect  = i;
        mc_reg->PHYDLLControl = 0x00002520;
    }

    mc_reg->PHYControl12  = 0x5028a014;
    mc_reg->PHYControl11  = 0x0a2600f4;

    udelay(1000);

#if 0
    mc_reg->PHYControl11 |= MC_PHYCONTROL11_PLLPUPLL_MASK;

    mc_reg->PHYControl14 = 0x10000000;

    while ((mc_reg->PHYControl14 & MC_PHYCONTROL14_PLLPLLLOCK_MASK) !=
        MC_PHYCONTROL14_PLLPLLLOCK_MASK);  // Wait for DSPLL to lock
#endif

    mc_reg->PHYControl14 = MC_PHYCONTROL14_PHYSYNCEN_MASK;
    mc_reg->PHYControl13 = (0x8 << MC_PHYCONTROL13_DLL_RESET_TIMER_SHIFT) | (0x21 << MC_PHYCONTROL13_DLL_DELAY_TEST_SHIFT) | MC_PHYCONTROL13_DLL_AUTO_MANUAL_UP_MASK;
    mc_reg->PHYControl14 = MC_PHYCONTROL14_PHYDLLRST_MASK;
    mc_reg->PHYControl14 = MC_PHYCONTROL14_DLLUPDATEEN_MASK;
    mc_reg->UserCommand  = MC_USERCOMMAND_SDRAMINIT_MASK;

    while ((mc_reg->DRAMStatus & MC_DRAMSTATUS_INITDONE_MASK) !=
        MC_DRAMSTATUS_INITDONE_MASK); // Wait for Init to Finish

    /* Disable the chip selects if there is no DIMM. */
    if (!dimm_found) {
        mc_reg->MemAddrMap0 = MC_MEMADDRMAP0_VALID_REPLACE_VAL(mc_reg->MemAddrMap0, 0);
        mc_reg->MemAddrMap1 = MC_MEMADDRMAP1_VALID_REPLACE_VAL(mc_reg->MemAddrMap1, 0);
    }

    /* Do write-leveling */
    if (ddr_freq != 160000000) {
        ddr3_write_level(ddr3_width, dimm_found);
    }

    return 0;
}
