/*
 * Beep behavior on POR.
 *
 * This u-boot function controls the beeping behavior on POR. All cards should have
 * either a beeper or a speaker. Both the beeper and the speaker can be driven
 * by the same GPIO. So, to avoid having to check what card we are running on,
 * we will try to drive both the beeper and the speaker, one of which will be
 * absent.
 *
 * On POR, the printer should beep briefly.
 *
 * If a panel cannot be detected, beep 5 times for 1 second each, pause for 5
 * seconds and repeat.
 */

#include <common.h>
#include <i2c.h>
#include <asm/arch/gpio.h>
#include <asm/arch/regAddrs.h>

/*
 * The buzzer line is connected to GPIO 155 which is configured as an output
 * pin from DPWM 9. The DPWM control registers are accessible beginning at
 * 0xF9304000. Each PWM can be enabled or dissabled by setting on clearing the
 * corresponding bit in the register at 0xF9304000.
 */
#define DPWM_TOP_REGS_BASE 0xF9304000
#define DPWM_ENABLE ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0x0))
#define BEEP_PWM_ENABLE_BIT (0x1 << 9)

// The following registers are the control registers for DPWM 9.
#define BEEP_PWM_CFG ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0xA00))
#define BEEP_PWM_COUNT0 ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0xA04))
#define BEEP_PWM_COUNT1 ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0xA08))
#define BEEP_PWM_COUNT2 ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0xA0C))
#define BEEP_PWM_INT_EN ((volatile unsigned int *) (DPWM_TOP_REGS_BASE + 0xA10))

struct {
    unsigned int dpwm_enable;
    unsigned int beep_pwm_cfg;
    unsigned int beep_pwm_count0;
    unsigned int beep_pwm_count1;
    unsigned int beep_pwm_count2;
    unsigned int beep_pwm_int_en;
} pwm_registers;

/*
 * The DAC is configured with control registers that can be written to over the
 * i2c bus. The DAC is on bus 1 at address 001100.
 */
#define DAC_I2C_BUS_NUM 1
#define DAC_I2C_ADDRESS 0x18

/*
 * The following are the control registers needed to configure the DAC to route
 * the buzzer line (via AIN2) to the speaker (on the left channel output). The
 * DAC pages its registers. The active page is changed by writing to the page
 * control register.
 */
#define PAGE_CONTROL_REGISTER 0x00
#define PAGE1_CLASS_D_SPEAKER_AMP 0x20
#define PAGE1_OUTPUT_MIXING 0x23
#define PAGE1_LEFT_ANALOG_VOL_TO_SPEAKER 0x26
#define PAGE1_CLASS_D_SPEAKER_DRIVER 0x2A

/*
 * Volume control:
 * Attenuates from 0 db (0) to -78 db (117).
 */
#define ATTENUATION 24

/*
 * Gain
 * 0 - 6 db
 * 1 - 12 db
 * 2 - 18 db
 * 3 - 24 db
 */
#define GAIN 0

/* Resets the DAC by toggling the reset line via a GPIO. */
static void dac_reset(void) {
    gpio_direction(AUDIO_ENABLE, 1);
    gpio_set(AUDIO_ENABLE, 0);
    /* Must be held low for at least 10 ns */
    udelay(1);
    gpio_set(AUDIO_ENABLE, 1);
}

/* Setup the DAC to drive the speaker from the buzzer line. */
static void dac_setup(void) {
    unsigned char buff;
    int prev_i2c_bus_num;

    /* Reset the DAC before we do anything. */
    dac_reset( );

    /* Save the the current i2c bus so that we can change it and put it back later. */
    prev_i2c_bus_num = i2c_get_bus_num( );
    i2c_set_bus_num(DAC_I2C_BUS_NUM);

    /*
     * Change to page 1. If the i2c call fails, don't do any of the other i2c
     * calls because the DAC isn't there.
     */
    buff = 1;
    if (!i2c_write(DAC_I2C_ADDRESS, 0, 1, &buff, 1)) {
        /* Setting bit 7 powers the Class-D Speaker Amplifier. */
        i2c_read(DAC_I2C_ADDRESS, PAGE1_CLASS_D_SPEAKER_AMP, 1, &buff, 1);
        buff |= (1 << 7);
        i2c_write(DAC_I2C_ADDRESS, PAGE1_CLASS_D_SPEAKER_AMP, 1, &buff, 1);

        /* Setting bit 4 routes AIN2 to the left-channel mixer amplifier. */
        i2c_read(DAC_I2C_ADDRESS, PAGE1_OUTPUT_MIXING, 1, &buff, 1);
        buff |= (1 << 4);
        i2c_write(DAC_I2C_ADDRESS, PAGE1_OUTPUT_MIXING, 1, &buff, 1);

        /*
         * Setting bit 7 routes the left-channel analog volume control output to
         * the Class-D Speaker Driver. Bits 6:0 control the gain.
         */
        buff = (1 << 7) | (ATTENUATION & 0x7f);
        i2c_write(DAC_I2C_ADDRESS, PAGE1_LEFT_ANALOG_VOL_TO_SPEAKER, 1, &buff, 1);

        /* Setting bit 2 un-mutes the Class-D Speaker Driver. Bits 4:3 control the gain. */
        i2c_read(DAC_I2C_ADDRESS, PAGE1_CLASS_D_SPEAKER_DRIVER, 1, &buff, 1);
        buff = (1 << 2) | ((GAIN << 3) & 0x18) | (buff & 0xa3);
        i2c_write(DAC_I2C_ADDRESS, PAGE1_CLASS_D_SPEAKER_DRIVER, 1, &buff, 1);
    }

    i2c_set_bus_num(prev_i2c_bus_num);
}

/* Return the DAC to the state it was in before. */
static void dac_teardown(void) {
    dac_reset( );
}

/**
 * Configure DPWM 9 to produce a 50% duty cycle 500 Hz square wave when enabled.
 */
static void pwm_setup(void) {
    // Store the state of all of the registers that will be overwritten so that
    // they can be restored later.
    pwm_registers.dpwm_enable = *DPWM_ENABLE;
    pwm_registers.beep_pwm_cfg = *BEEP_PWM_CFG;
    pwm_registers.beep_pwm_count0 = *BEEP_PWM_COUNT0;
    pwm_registers.beep_pwm_count1 = *BEEP_PWM_COUNT1;
    pwm_registers.beep_pwm_count2 = *BEEP_PWM_COUNT2;
    pwm_registers.beep_pwm_int_en = *BEEP_PWM_INT_EN;

    // Configure the DPWM.
    *DPWM_ENABLE |= BEEP_PWM_ENABLE_BIT;
    *BEEP_PWM_CFG = 0x20;
    *BEEP_PWM_COUNT0 = 0x0;
    *BEEP_PWM_COUNT1 = 100;
    *BEEP_PWM_COUNT2 = 100;
    *BEEP_PWM_INT_EN = 0x0;
}

/**
 * Return the PWM to the state it was in before.
 */
void pwm_teardown(void) {
    *DPWM_ENABLE = pwm_registers.dpwm_enable;
    *BEEP_PWM_CFG = pwm_registers.beep_pwm_cfg;
    *BEEP_PWM_COUNT0 = pwm_registers.beep_pwm_count0;
    *BEEP_PWM_COUNT1 = pwm_registers.beep_pwm_count1;
    *BEEP_PWM_COUNT2 = pwm_registers.beep_pwm_count2;
    *BEEP_PWM_INT_EN = pwm_registers.beep_pwm_int_en;
}

/* Turns the DPWM on. */
static void beep_on(void) {
    *BEEP_PWM_CFG |= 0x1;
}

/* Turns the DPWM off. */
static void beep_off(void) {
    *BEEP_PWM_CFG &= ~0x1;
}

/* Beep one time for on_us followed by off_us of silence. */
static void beep_once(int on_us, int off_us) {
    beep_on( );
    udelay(on_us);
    beep_off( );
    udelay(off_us);
}

void por_beep(void) {
    pwm_setup( );
    dac_setup( );

    // 100 ms beep on every POR.
    beep_once(100000, 0);

    dac_teardown( );
    pwm_teardown( );
}

/* Beep pattern for missing op panel. */
void no_panel_beep_loop(void) {
    pwm_setup( );
    dac_setup( );

    // 5 1 second beeps (1/2 second on, 1/2 second off), 5 second pause, repeat
    while (1) {
        beep_once(500000, 500000);
        beep_once(500000, 500000);
        beep_once(500000, 500000);
        beep_once(500000, 500000);
        beep_once(500000, 5500000);
    }
}
