/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * (C) Copyright 2002
 * David Mueller, ELSOFT AG, <d.mueller@elsoft.ch>
 *
 * (C) Copyright 2003
 * Texas Instruments, <www.ti.com>
 * Kshitij Gupta <Kshitij@ti.com>
 *
 * (C) Copyright 2004
 * ARM Ltd.
 * Philippe Robin, <philippe.robin@arm.com>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */


#include <common.h>
#include <mmc.h>
#include <spi_flash.h>
#include <config.h>
#include <lxk_panel.h>
#include <i2c.h>
#include "boardEnv/mvBoardEnvLib.h"
#include "cpu/mvCpu.h"
#include "gpp/mvGpp.h"
#include "gpp/mvGppRegs.h"

DECLARE_GLOBAL_DATA_PTR;

/*
 * Allow for some inaccuracy in udelay and sample-to-sample variation.
 * For now, just make it reliable. Can tweak later to reduce boot time
 * when more hardware is available for testing.
 */
#define MDELAY_FUDGE_FACTOR 2

void code_verification_failure(void) {
    while (1);
}

const char * ypp_get_revision(void) {
    uint32_t revision = 0;

    mvGppTypeSet(0, MV_GPP28, MV_GPP_IN & MV_GPP28);
    mvGppPolaritySet(0, MV_GPP28, MV_GPP_IN_ORIGIN & MV_GPP28);

    mvGppTypeSet(0, MV_GPP29, MV_GPP_IN & MV_GPP29);
    mvGppPolaritySet(0, MV_GPP29, MV_GPP_IN_ORIGIN & MV_GPP29);

    mvGppTypeSet(2, MV_GPP65, MV_GPP_IN & MV_GPP65);
    mvGppPolaritySet(2, MV_GPP65, MV_GPP_IN_ORIGIN & MV_GPP65);

    mvGppTypeSet(2, MV_GPP66, MV_GPP_IN & MV_GPP66);
    mvGppPolaritySet(2, MV_GPP66, MV_GPP_IN_ORIGIN & MV_GPP66);

    revision |= mvGppValueGet(0, MV_GPP28) >> 25; // GPP28 is bit 28 of gpp bank 0 and is bit 3 of the card rev, so shift right 25
    revision |= mvGppValueGet(0, MV_GPP29) >> 27; // GPP29 is bit 29 of gpp bank 0 and is bit 2 of the card rev, so shift right 27
    revision |= mvGppValueGet(2, MV_GPP65);       // GPP65 is bit 1 of gpp bank 2 and is bit 1 of the card rev, so no shift
    revision |= mvGppValueGet(2, MV_GPP66) >> 2;  // GPP66 is bit 2 of gpp bank 2 and is bit 0 of the card rev, so shift right 2

    switch(revision) {
        case 0:  return "00a";  break;
        case 1:  return "00b";  break;
        case 2:  return "00c";  break;
        case 3:  return "00d";  break;
        case 4:  return "11a";  break;
        case 5:  return "00e";  break;
        case 6:  return "00f";  break;
        case 8:  return "77a";  break;
        case 9:  return "77b";  break;
        case 10: return "77c";  break;
        default:
        case 12: return "77d";  break;
    }
}

const char * card_get_type(void) {
    uint32_t boardId;

    boardId = mvBoardIdGet();
    if(boardId == YPP_CARD_ID)
        return "ypp";
    else
        return "adb";
}

const char * card_get_revision(void) {
    uint32_t boardId;

    boardId = mvBoardIdGet();
    if(boardId == YPP_CARD_ID)
        return ypp_get_revision();
    else
        return "40a";
}

static void imgpipe_reset(void) {
    if (strcmp(card_get_revision( ), "00d") > 0) {
        mvGppTypeSet(0, (MV_GPP1 | MV_GPP4), MV_GPP_OUT & (MV_GPP1 | MV_GPP4));
        mvGppValueSet(0, (MV_GPP1 | MV_GPP4), (MV_GPP1 | MV_GPP4));
    }
}

static void power_on_hdd(void) {
    if ((strcmp(card_get_revision( ), "00d") > 0) && (strcmp(card_get_revision( ), "11a") != 0)){
        mvGppTypeSet(0, MV_GPP25, MV_GPP_OUT & MV_GPP25);
        mvGppValueSet(0, MV_GPP25, MV_GPP25);
    }
}

static void reset_isp(void) {
    mvGppValueSet(1, MV_GPP60, MV_GPP60);
    mvGppTypeSet(1, MV_GPP60, MV_GPP_OUT & MV_GPP60);

    udelay(10);

    mvGppValueSet(1, MV_GPP60, 0);

    mdelay(10);

    mvGppValueSet(1, MV_GPP60, MV_GPP60);
}

static void reset_amp_slave(void) {
    /*
     * Set both the Reset and Recovery PIOs high, set both as an output.
     * Then toggle reset low, then high.
     */
    mvGppTypeSet(0, (MV_GPP9 | MV_GPP12), (MV_GPP_OUT & (MV_GPP9 | MV_GPP12)));
    mvGppValueSet(0, MV_GPP12, MV_GPP12);
    mvGppValueSet(0, MV_GPP9, 0);

    mvGppValueSet(0, MV_GPP9, MV_GPP9);
    udelay(250);
    mvGppValueSet(0, MV_GPP9, 0);
}

#ifdef CONFIG_LXK_PANEL
void lxk_reset_panel(void) {
    const char *ptr = card_get_revision();
    MV_U32 gpp_group;
    MV_U32 gpp_num;

    if (!ptr || (strnlen(ptr, 4) > 3) || !strncmp(ptr, "40a", 3)) {
        printf("Panel reset not supported on this hw.\n");
        return;
    }

    if (!strncmp(ptr, "00a", 3) || !strncmp(ptr, "00b", 3)) {
        printf("Controller rev %s, reseting panel with mpp24, requires rework.\n", ptr);
        gpp_num = MV_GPP24;
        gpp_group = 0;
    }
    else {
        printf("Controller rev %s, reseting panel with mpp53.\n", ptr);
        gpp_num = MV_GPP53;
        gpp_group = 1;
    }

    /* configure panel reset pin as an output */
    mvGppTypeSet(gpp_group, gpp_num, MV_GPP_OUT & gpp_num);

    /* assert and deassert panel reset */
    mvGppValueSet(gpp_group, gpp_num, 0);
    mdelay(15 * MDELAY_FUDGE_FACTOR);
    mvGppValueSet(gpp_group, gpp_num, gpp_num);
    /*
     * Voltage/reset supervisor chip on new panels will insert
     * 280ms delay before deasserting reset output.
     */
    mdelay((280 + 15) * MDELAY_FUDGE_FACTOR);
}
#endif

/*
 * Construct the boot command for the specified configuration.
 */
static int mmc_get_bootcmd_for_part(int dev_num, int part, disk_partition_t *pinfo, char *cmd, int len, block_dev_desc_t * dev_desc, int boot_recover) {
    if (strcmp((const char *)pinfo->name, "Kernel") != 0) /* The kernel partition is on partition 2 for both EXT2 and CRAM */
        return 1;

    /* If the device uses SSP partitioning then it is using CRAM */
    if (dev_desc->part_type == PART_TYPE_SSP) {
        int partition;
        ulong part_size;
        ulong part_sizeB;
        disk_partition_t pinfoB;
        const uint32_t main_addr = 0x100000;
        uint32_t cramfs_addr;
        const char format2[] = "setenv cramfsaddr %#x;"
                               "mmc dev %d;"
                               "mmc read %#x %#llx %#lx;"
                               "sha256verify %#x %#lx 1;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "mmc read %#x %#llx %#lx;"
                               "sha256verify %#x %#lx 1;"
                               "cramfsload %#x /main.img;"
                               "source %#x;"
                               "loop.l 0xd0018270 1";

        /* Search for recovery partition */
        for (partition = 1; partition < 35; partition++) {
            if (get_partition_info(dev_desc, partition, &pinfoB) == 0) {
                if (strcmp((const char *)pinfoB.name, "RecoveryKernel") == 0 && pinfoB.data_len != 0) {
                    break;
                }
            }
        }

        /* If there is no recovery partition then just make it the same as the primary */
        if (partition == 35) {
            memcpy(&pinfoB, pinfo, sizeof(disk_partition_t));
        }

        /* Boot from the recovery partition */
        if (boot_recover) {
            memcpy(pinfo, &pinfoB, sizeof(*pinfo));
        }

        /* Load the kernel CRAM partition right in front of U-Boot allowing space for the malloc area */
        cramfs_addr = ((gd->start_addr_sp - max(pinfo->data_len, pinfoB.data_len)) >> 20) << 20;
        part_size = (pinfo->data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;
        part_sizeB = (pinfoB.data_len + (dev_desc->blksz - 1)) / dev_desc->blksz;

        sprintf(cmd, format2, cramfs_addr,
                dev_num,
                cramfs_addr, pinfo->start, part_size,
                cramfs_addr, pinfo->data_len - pinfo->fcert_len + cramfs_addr,
                main_addr,
                main_addr,
                cramfs_addr, pinfoB.start, part_sizeB,
                cramfs_addr, pinfoB.data_len - pinfoB.fcert_len + cramfs_addr,
                main_addr,
                main_addr);
    }
    else {
        const uint32_t uImage_addr     = 0x400000;
        const uint32_t uInitramfs_addr = 0x800000;
        const char format[] = "mmc dev %d;"
                              "ext2load mmc %d:%d %#x /boot/boot.scr;"
                              "source %#x;"
                              "ext2load mmc %d:%d %#x /boot/uImage;"
                              "ext2load mmc %d:%d %#x /boot/uInitramfs;"
                              "bootm %#x %#x";

        sprintf(cmd, format, dev_num,
                dev_num, part, uImage_addr,
                uImage_addr,
                dev_num, part, uImage_addr,
                dev_num, part, uInitramfs_addr,
                uImage_addr, uInitramfs_addr);
    }

    return 0;
}


static int sdmmc_get_bootcmd(int dev_num, int boot_recover) {
    block_dev_desc_t *dev_desc = NULL;
    struct mmc *mmc;

    /* Initialize needed devices */
    mmc = find_mmc_device(dev_num);
    if (mmc) {
        int rc;
        rc = mmc_init(mmc);
        if (rc)
        {
            return 0;
        }
    }

    dev_desc = get_dev("mmc", dev_num);
    if (dev_desc) {
        int part;

        /*
         * Found a device -- go scan it for partitions
         * Restrict the search to the first few partitions.
         */
        for (part = 1; part < 5; part++) {
            disk_partition_t info;

            if (get_partition_info(dev_desc, part, &info) == 0) {
                if (info.size || info.data_len) {
                    char buf[300];

                    /* Found a valid partition */
                    if (mmc_get_bootcmd_for_part(dev_num, part, &info, buf, sizeof(buf), dev_desc, boot_recover) == 0) {
                        setenv_ulong("MMC_DEVICE_NUM", dev_num);
                        setenv_ulong("MMC_RD_BLK", mmc->read_bl_len);
                        setenv_ulong("bootCs", dev_num);
                        setenv("bootcmd", buf);
                        return 1;
                    }
                }
            }
        }
    }

    return 0;
}

static int is_sdmmc_dev_present(unsigned int dev) {
    struct mmc *mmc;

    /* Initialize needed devices */
    mmc = find_mmc_device(dev);
    if (!mmc)
        return 0;

    if (mmc_init(mmc))
        return 0;

    return 1;
}

static int get_sdmmc_mux(void) {
    int sdmmc_mux;

    if (strcmp(card_get_revision( ), "00d") < 0) {
        mvGppTypeSet(0, (MV_GPP19 | MV_GPP18), MV_GPP_IN & (MV_GPP19 | MV_GPP18));
        mvGppPolaritySet(0, (MV_GPP19 | MV_GPP18), MV_GPP_IN_INVERT & (MV_GPP19 | MV_GPP18));

        sdmmc_mux = mvGppValueGet(0, MV_GPP19 | MV_GPP18);
        sdmmc_mux >>= 18;

        /* The values for soldered down and DC eMMC are backwards */
        if (sdmmc_mux == 1)
            sdmmc_mux = 2;
        else if (sdmmc_mux == 2)
            sdmmc_mux = 1;

        sdmmc_mux = (sdmmc_mux + 2) % 3;
    }
    else {
        mvGppTypeSet(0, MV_GPP18, MV_GPP_IN & MV_GPP18);
        mvGppPolaritySet(0, MV_GPP18, MV_GPP_IN_INVERT & MV_GPP18);

        sdmmc_mux = mvGppValueGet(0, MV_GPP18);
        sdmmc_mux >>= 18;
    }

    return sdmmc_mux;
}

void set_sdmmc_mux(unsigned int sdmmc_mux) {
    if (strcmp(card_get_revision( ), "00d") < 0) {
        sdmmc_mux = (sdmmc_mux + 1) % 3;

        /* The values for soldered down and DC eMMC are backwards */
        if (sdmmc_mux == 1)
            sdmmc_mux = 2;
        else if (sdmmc_mux == 2)
            sdmmc_mux = 1;

        mvGppValueSet(0, (MV_GPP19 | MV_GPP18), (~sdmmc_mux & 0x3) << 18);
        mvGppTypeSet(0, (MV_GPP19 | MV_GPP18), MV_GPP_OUT & (MV_GPP19 | MV_GPP18));
    }
    else {
        mvGppValueSet(0, MV_GPP18, (~sdmmc_mux & 0x1) << 18);
        mvGppTypeSet(0, MV_GPP18, MV_GPP_OUT & MV_GPP18);
    }
}

static void setup_bootcmd(int ignore_dc, int boot_recover) {
    struct spi_flash *sf;
    int cft_boot = 0;
    int sdmmc_mux;
    char buf[64];

    sf = spi_flash_probe(CONFIG_SF_DEFAULT_BUS, CONFIG_SF_DEFAULT_CS, CONFIG_SF_DEFAULT_SPEED, CONFIG_SF_DEFAULT_MODE);
    if (sf) {
        int rc;

        rc = spi_flash_read(sf, 0xff000, 0x10, buf);
        if (!rc) {
            if (memcmp(buf, "cftrecover", strlen("cftrecover")) == 0) {
                const char *cp = (const char *)0x3f0000;
                const char cft_cmd[] =
                    "setenv CFTRECOVER 1;"
                    "sf probe;sf erase 0xff000 0x1000;"
                    "setenv cramfsaddr 0x2000208;"
                    "sha256verify 0x2000208 0x2000000 1;"
                    "cramfsload 0x100000 /ucmdline;bootm 0x100000;"
                    "cramfsload 0x400000 /uImage;"
                    "cramfsload 0x1000000 /uInitramfs;"
                    "cramfsload 0xf00000 /devicetree-uImage-$DTB.dtb;"
                    "bootm 0x400000 0x1000000 0xf00000;"
                    "bootm 0x400000 0x1000000";

                do {
                    invalidate_dcache_range((unsigned long)cp, (unsigned long)cp + CONFIG_SYS_CACHELINE_SIZE);
                } while (memcmp(cp, "beginboot", strlen("beginboot")) != 0);

                cft_boot = 1;
                setenv("bootcmd", cft_cmd);
                icache_invalid( );
            }
        }

        spi_flash_free(sf);
    }

    sdmmc_mux = get_sdmmc_mux( );
    set_sdmmc_mux(sdmmc_mux);

    if (!cft_boot) {
        if ((sdmmc_mux == 0) && (!ignore_dc)) {
            if (sdmmc_get_bootcmd(1, boot_recover) == 0)
                sdmmc_get_bootcmd(sdmmc_mux, boot_recover);
        }
        else
            sdmmc_get_bootcmd(sdmmc_mux, boot_recover);
    }
    else {
        if ((strcmp(card_get_revision( ), "00d") < 0) && (is_sdmmc_dev_present(2)))
            setenv_ulong("bootCs", 2);
        else
            setenv_ulong("bootCs", 0);
    }

    if (strcmp(card_get_revision( ), "00d") < 0)
        sprintf(buf, "%s,%s,%s",
                is_sdmmc_dev_present(0) ? "eMMC" : "",
                is_sdmmc_dev_present(1) ? "eMMC" : "",
                is_sdmmc_dev_present(2) ? "SD" : "");
    else
        sprintf(buf, "%s,%s",
                is_sdmmc_dev_present(0) ? "eMMC" : "",
                is_sdmmc_dev_present(1) ? "eMMC" : "");
    setenv("BOOTDEV", buf);
}

static void reset_usb_audio(void) {
    mvGppValueSet(1, MV_GPP54, MV_GPP54);
    mvGppTypeSet(1, MV_GPP54, MV_GPP_OUT & MV_GPP54);

    udelay(10);

    mvGppValueSet(1, MV_GPP54, 0);

    udelay(10);

    mvGppValueSet(1, MV_GPP54, MV_GPP54);
}

static void setup_dtb(void)
{
    if (strcmp(card_get_revision( ), "00d") > 0)
        setenv("DTB", "armada-xp-yappy");
    else
        setenv("DTB", "armada-xp-yapp");
}

struct audio_map {
    uint8_t addr;
    uint8_t val;
};

static const struct audio_map audio_settings[] = {
    { 0x00, 0x00 }, { 0x01, 0x01 }, { 0x04, 0x01 }, { 0x1b, 0xc0 },
    { 0x1c, 0x01 }, { 0x40, 0x0c }, { 0x00, 0x01 }, { 0x23, 0x11 },
    { 0x26, 0xff }, { 0x00, 0x00 }, { 0x44, 0x7f }, { 0x45, 0x00 },
    { 0x46, 0xb6 }, { 0x00, 0x01 }, { 0x20, 0x06 }, { 0x2a, 0x00 },
    { 0x22, 0x70 }, { 0x00, 0x00 }, { 0x19, 0x00 }, { 0x1a, 0x01 },
    { 0x33, 0x00 }, { 0x0b, 0x81 }, { 0x0c, 0x81 }, { 0x0d, 0x00 },
    { 0x0e, 0x01 }, { 0x00, 0x01 }, { 0x2a, 0x04 }, { 0x20, 0x86 },
    { 0x26, 0x90 }, { 0x00, 0x00 }, { 0x3f, 0x90 }, { 0x41, 0x20 },
    { 0x42, 0x20 }, { 0x40, 0x04 },
};

static void setup_dac(void) {
    int i, oldbus;

    oldbus = i2c_get_bus_num( );
    i2c_set_bus_num(CONFIG_AUDIO_I2C_BUS);

    for (i=0; i<ARRAY_SIZE(audio_settings); ++i)
        i2c_write(CONFIG_AUDIO_I2C_ADDR,
                audio_settings[i].addr, sizeof(audio_settings[i].addr),
                (uint8_t *)&audio_settings[i].val, sizeof(audio_settings[i].val));

    i2c_set_bus_num(oldbus);
}

static void por_beep(void) {
    int i;

    mvGppTypeSet(1, MV_GPP50, MV_GPP_OUT & MV_GPP50);
    for (i=0; i<200; ++i) {
        mvGppValueSet(1, MV_GPP50, MV_GPP50);
        udelay(250);
        mvGppValueSet(1, MV_GPP50, 0);
        udelay(250);
    }
}

/* Beep pattern for missing op panel. */
void no_panel_beep_loop(void) {
    int i;

    // 5 1 second beeps (1/2 second on, 1/2 second off), 5 second pause, repeat
    while (1) {
        for (i=5; i; i--) {
            por_beep( );
            mdelay(900);
        }

        mdelay(5000);
    }
}

int board_late_init(void) {
    int i;
    char *asic_rev;
    const char *ptr;
    DECLARE_GLOBAL_DATA_PTR;
    char buf[128];
    int ignore_dc = 0;
    int boot_recover = 0;

#ifdef CONFIG_LXK_PANEL
    char *keys;

    /* 
     * Calling  lxk_panel_init() here seems redundant but is harmless.
     * Called earlier during stdio_init()                      .
     * TBD: Is there any case that could bypass stdio_init() and require this call?
     */ 
    lxk_panel_init();

    lxk_panel_status(LXK_PANEL_BLANK);

    keys = getenv("PORKEYS");
    if (keys) {
        unsigned int por_keys = simple_strtoul(keys, NULL, 0);
        if (por_keys == NM_HOME)
            ignore_dc = 1;
        else if (por_keys == (NM_2 | NM_7 | NM_8))
            boot_recover = 1;
    }

    setenv_ulong("PANELTYPE_ENV", lxk_panel_get_type());

    ptr = lxk_panel_get_subtype();
    if (ptr)
        setenv("PANELSUBTYPE", (char *)ptr);
#endif

    if (boot_recover)
        reset_amp_slave( );

    reset_usb_audio( );
    imgpipe_reset( );
    power_on_hdd( );
    reset_isp( );

    setup_dac( );
    por_beep( );

    setup_bootcmd(ignore_dc, boot_recover);
    printf("bootcmd: %s\n", getenv("bootcmd"));

    setenv_ulong("DRAM_SIZE", gd->ram_size);

    buf[0] = 0;
    for (i=0; i<CONFIG_NR_DRAM_BANKS; ++i) {
        sprintf(buf, "%s,%ld", buf, gd->bd->bi_dram[i].size >> 20);
    }
    setenv("DRAM_CONFIG", &buf[1]); /* Remove leading comma */

    sprintf(buf, "console=ttyS0,115200 earlyprintk=serial,ttyS0,115200");
    setenv("bootargs", buf);

    ptr = card_get_type( );
    if (ptr)
        setenv("CARDTYPE", (char *)ptr);

    ptr = card_get_revision( );
    if (ptr)
        setenv("CARDREV", (char *)ptr);

    if (mvCtrlRevGet( ) == MV_78XX0_B0_REV)
        asic_rev = "B0";
    else
        asic_rev = "A0";
    setenv("ASICREV", asic_rev);

    setenv_ulong("CPUMHZ", mvCpuPclkGet( ) / 1000000);
    setenv_ulong("BUSFREQ", CONFIG_SYS_TCLK);

    setup_dtb();

    setenv("initrd_high", "ffffffff");
    setenv("fdt_high", "ffffffff");

    return 0;
}

