/*
 * PLL calculations and initialization ported from
 * Marvell dove framebuffer driver for Linux kernel 3.0
 */
#include <dovefbreg.h>
#include <common.h>
#include <asm/io.h>
#include <div64.h>
#include <lcd.h>
#include <lxk_panel.h>
#include <lcd.h>

//#define VERBOSE_PLL_DEBUG

static int fixed_full_div = 1;
static const int full_div_val = 7;
static const int lcd_ref_clk = 25000000;
static const int lvds_delay_ticks = 2;

static uint32_t get_lcd_k_div_by_ld(uint32_t tar_freq, uint32_t l_div)
{
	uint32_t k = 1;
	uint64_t f_out;

	/* Calculate K according to F-out. */
	f_out = tar_freq / l_div;
	do_div(f_out, 1000000);

	/* K is calculated according to (tar_freq / ld). */
	if((f_out >= 700) && (f_out <= 4000))
		k = 1;
	else if((f_out >= 350) && (f_out <= 2000))
		k = 2;
	else if((f_out >= 175) && (f_out <= 1000))
		k = 4;
	else if((f_out >= 87) && (f_out <= 500))
		k = 8;
	else
		k = 16;

	return k;
}

static void set_lcd_clock_dividers(void *reg_base, uint32_t f_in,
		uint32_t m_div, uint32_t n_div, uint32_t k_div, uint32_t full_div, uint32_t is_half_div)
{
	uint64_t f_vco;
	uint32_t reg;
	uint32_t pll_vco;
	uint32_t apll_lpf;
	int i;

	/* Calculate K according to Fvco. */
	f_vco = f_in * n_div;
	do_div(f_vco, m_div);
	do_div(f_vco, 1000000);

	pll_vco = (f_vco - 481) / 220;

	/* Set APLL LPF */
	if (n_div <= 4)
		apll_lpf = 12;
	else if (n_div <= 22)
		apll_lpf = 7;
	else if (n_div <= 117)
		apll_lpf = 4;
	else
		apll_lpf = 1;

	printf("N = %d, M = %d, K = %d, full_div = %d, half = %d, pll_vco = %d, pll_lpf = %d.\n",
			n_div, m_div, k_div, full_div, is_half_div, pll_vco, apll_lpf);

	/* Clear SMPN */
	reg = readl(reg_base + LCD_CLK_CFG1_REG);
	reg &= ~LCD_SMPN_EN_MASK;
	reg |= LCD_SMPN_EN(0);
	writel(reg, reg_base + LCD_CLK_CFG1_REG);

	/* Set PLL Power Down */
	reg = readl(reg_base + LCD_CLK_CFG0_REG);
	reg &= ~LCD_PLL_PWR_DOWN_MASK;
	reg |= LCD_PLL_PWR_DOWN(1);
	writel(reg, reg_base + LCD_CLK_CFG0_REG);

	/* Set N, M, K */
	reg = readl(reg_base + LCD_CLK_CFG0_REG);
	reg &= ~(LCD_PLL_NDIV_MASK | LCD_PLL_MDIV_MASK | LCD_PLL_KDIV_MASK | LCD_PLL_LPF_MASK | LCD_PLL_VCO_BAND_MASK);
	reg |= LCD_PLL_NDIV(n_div - 1);
	reg |= LCD_PLL_MDIV(m_div - 1);
	/*
	 * Legal k_div values and encodings are:
	 * 1->1, 2->2, 4->3, 8->4, and 16->5.
	 */
	/* reg |= LCD_PLL_KDIV(clamp((ilog2(k_div) + 1), 1, 5)); */

	switch(k_div) {
	case 1:
		reg |= LCD_PLL_KDIV(1);
		break;
	case 2:
		reg |= LCD_PLL_KDIV(2);
		break;
	case 4:
		reg |= LCD_PLL_KDIV(3);
		break;
	case 8:
		reg |= LCD_PLL_KDIV(4);
		break;
	case 16:
		reg |= LCD_PLL_KDIV(5);
		break;
	default:		
		break;
	}
	
	reg |= LCD_PLL_VCO_BAND(pll_vco);
	reg |= LCD_PLL_LPF(apll_lpf);
	writel(reg, reg_base + LCD_CLK_CFG0_REG);

	/* Set half divider */
	reg = readl(reg_base + LCD_CLK_CFG1_REG);
	reg &= ~(LCD_FULL_DIV_MASK | LCD_HALF_DIV_MASK);
	reg |= LCD_FULL_DIV(full_div);
	reg |= LCD_HALF_DIV(is_half_div);
	writel(reg, reg_base + LCD_CLK_CFG1_REG);

	/* Clear PLL Power Down */
	reg = readl(reg_base + LCD_CLK_CFG0_REG);
	reg &= ~LCD_PLL_PWR_DOWN_MASK;
	reg |= LCD_PLL_PWR_DOWN(0);
	writel(reg, reg_base + LCD_CLK_CFG0_REG);

	/* Wait 0.5mSec */
	mdelay(10);

	/* Set SMPN */
	reg = readl(reg_base + LCD_CLK_CFG1_REG);
	reg &= ~LCD_SMPN_EN_MASK;
	reg |= LCD_SMPN_EN(1);
	writel(reg, reg_base + LCD_CLK_CFG1_REG);

	return;
}

/*
** Calculate the best PLL parameters to get the closest output frequency.
** out_freq = (Fin * N / M) / X;
**  OR
** out_freq = (Fin * N / M) / (X + 0.5)
*/
static void calc_best_clock_div(uint32_t tar_freq, uint32_t f_in,
		uint32_t *m_div, uint32_t *n_div, uint32_t *k_div, uint32_t *x_div, uint32_t *is_half_div, uint32_t *lcd_div)
{
	uint64_t best_rem = 0xFFFFFFFFFFFFFFFFll;
	uint32_t best_m = 0;
	uint32_t best_n = 0;
	uint32_t best_x = 0;
	uint32_t best_k = 0;
	uint32_t best_lcd_div = 0;
	uint32_t half_div = 0;
	uint64_t rem;
	uint64_t temp;
	uint32_t temp_32;
	uint32_t n, m, x, k, ld;
	int override = 0; 	/* Used to mark special cases where the LCD */
	uint32_t n_max;
	uint32_t x_start, x_end;
	uint32_t ld_end;

	if (fixed_full_div) {
		x_start = full_div_val;
		x_end = x_start + 1;
	} else {
		x_start = 1;
		x_end = 64;
	}
	ld_end = 2;

	n_max = 480;

	/* Look for the best N & M values assuming that we will NOT use the
	** HALF divider.
	*/

	for(ld = 1; ld < ld_end; ld++) {
		for(x = x_start; x < x_end; x++) {
			k = get_lcd_k_div_by_ld((tar_freq * x), ld);
			for (n = 1; n < n_max; n++) {
				for(m = 1; m < 4; m++) {
					/*
					 * N * K must be smaller than 480
					 * 700 < F_in * N / (M * K) < 4000
					 */
					temp_32 = (((f_in * n) / 1000000) / m / k);
					if ((temp_32 < (700 / k)) || (temp_32 > (4000 / k)))
						continue;
					temp = (uint64_t)f_in * (uint64_t)n;
					do_div(temp, m);
					/* Fin * N / M Must be < 1500MHz */
					if(temp > 1500000000)
						continue;
					do_div(temp, k);
					do_div(temp, x);
					do_div(temp, ld);
					rem = abs64(tar_freq - temp);
					/* It's better not to select k = 1. */
					if ((rem < best_rem) ||
					    ((override == 1) && (rem == best_rem)) ||
					    ((rem == best_rem) && (k < best_k))) {
						best_rem = rem;
						best_m = m;
						best_n = n;
						best_x = x;
						best_k = k;
						best_lcd_div = ld;
					}
					if ((best_rem == 0) && (override == 0))
						break;
				}
			}
		}
	}

	/* Look for the best N & M values assuming that we will use the
	** HALF divider.
	*/
	if ((fixed_full_div == 0) && (best_rem != 0)) {
		for(ld = 1; ld < ld_end; ld++) {
			/* Half div can be between 5.5 & 31.5, prescale by 10 */
			for (x = 55; x <= 315; x += 10) {
				k = get_lcd_k_div_by_ld(((tar_freq * (x / 10)) + (tar_freq / 2)), ld);
				for (n = 1; n < n_max; n++) {
					for(m = 1; m < 4; m++) {
						temp = (uint64_t)f_in * (uint64_t)n * 10;
						do_div(temp, m);
						/* Fin * N / M Must be < 1500MHz */
						if(temp > 15000000000ll)
							continue;
						do_div(temp, x);
						do_div(temp, k);
						do_div(temp, ld);
						rem = abs64(tar_freq - temp);
						if ((rem < best_rem) ||
						    ((override == 1) && (rem == best_rem)) ||
						    ((rem == best_rem) && (k < best_k))) {
							half_div = 1;
							best_rem = rem;
							best_m = m;
							best_n = n;
							best_x = x / 10;
							best_k = k;
							best_lcd_div = ld;
							half_div = 1;
						}
						if ((best_rem == 0) && (override == 0))
							break;
					}
				}
			}
		}
	}

	*is_half_div = half_div;
	*m_div = best_m;
	*n_div = best_n;
	*x_div = best_x;
	*k_div = best_k;
	*lcd_div = best_lcd_div;
}

/*
 * The hardware clock divider has an integer and a fractional
 * stage:
 *
 *	clk2 = clk_in / integer_divider
 *	clk_out = clk2 * (1 - (fractional_divider >> 12))
 *
 * Calculate integer and fractional divider for given clk_in
 * and clk_out.
 */
static void set_clock_divider(void *reg_base, uint32_t pixelclk_hz)
{
	uint32_t x = 0;
	uint32_t full_div;
	uint32_t m_div;
	uint32_t n_div;
	uint32_t k_div;
	uint32_t is_half_div;
	uint32_t lcd_div;


	/*
	 * Check input values.
	 */
	if (!pixelclk_hz) {
		printf("Requested pixelclk_hz %d is invalid.\n", pixelclk_hz);
		return;
	}

	calc_best_clock_div(pixelclk_hz, lcd_ref_clk, &m_div, &n_div,
			&k_div, &full_div, &is_half_div, &lcd_div);

#ifdef VERBOSE_PLL_DEBUG
	printf("pixelclk_hz = %d Hz\n", pixelclk_hz);
	printf("LCD-ref-clk = %d.\n", lcd_ref_clk);
	printf("LCD-Div = %d.\n", lcd_div);
#endif
	set_lcd_clock_dividers(reg_base, lcd_ref_clk, m_div, n_div, k_div, full_div, is_half_div);

	/*
	 * Set setting to reg.
	 */
	x = readl(reg_base + LCD_CFG_SCLK_DIV);
	x &= ~0xFFFF;
	x |= 0x80000000; /* Use PLL/AXI clock. */
	x |= lcd_div;

	writel(x, reg_base + LCD_CFG_SCLK_DIV);
}

/*
 * Wrapper for ported pll code.
 */
int lcd_pll_init(void *reg_base, uint32_t pixelclk_hz)
{
	int rc = 0;

	if((pixelclk_hz > 100000000) || (pixelclk_hz < 5000000)) { /* > 100mHz or < 5mHz */
		printf("%s: Error! Requested pixelclock frequency %d Hz is invalid.\n", __func__, pixelclk_hz);
		rc = -1;
		goto done;
	}

	set_clock_divider(reg_base, pixelclk_hz);

	/* Allow PLL to stabilize */
	mdelay(10);

	/*
	 * clocking setup.
	 */
	writel(0x80000001,   reg_base +  LCD_CFG_SCLK_DIV); // 1a8

done:
	return rc;

}

int lcd_lvds_init(void *reg_base, uint32_t pixelclk_hz, lvds_bit_mapping_t lvds_option)
{
	uint32_t reg = 0;
	int rc = 0;

	fixed_full_div = 1;

	switch(lvds_option) {
	case LVDS_UNUSED:
		fixed_full_div = 0;
		break;
	case LVDS_BPP18:
		reg = LCD_LVDS_CLK_EN | LCD_LVDS_CFG_SER_EN(1);
		reg |= LCD_LVDS_CFG_TICK_DRV(lvds_delay_ticks);
		reg |= LCD_LVDS_CFG_PIN_CNT_18;
		break;
	case LVDS_BPP24_OPTION_1:
		reg = LCD_LVDS_CLK_EN | LCD_LVDS_CFG_SER_EN(1);
		reg |= LCD_LVDS_CFG_TICK_DRV(lvds_delay_ticks);
		reg |= LCD_LVDS_CFG_PIN_CNT_24;
		reg |= LCD_LVDS_CFG_24BIT_OPT1;
		break;
	case LVDS_BPP24_OPTION_2:
		reg = LCD_LVDS_CLK_EN | LCD_LVDS_CFG_SER_EN(1);
		reg |= LCD_LVDS_CFG_TICK_DRV(lvds_delay_ticks);
		reg |= LCD_LVDS_CFG_PIN_CNT_24;
		reg |= LCD_LVDS_CFG_24BIT_OPT2;
		break;
	default:
		rc = -1;
		goto done;
	}

	writel(reg,   reg_base + LCD_LVDS_CLK_CFG); // f0ac

	/*
	 * For Armada, need to enable the LVDS IOPADs in a separate register space.
	 */
	writel( 0xffe00000, (void *) 0xd00182f0);

done:
	return rc;
}

