#include <common.h>

/*
 * The order of these macros should match the order of addresses
 * in setup.c
 */
/* ASIC GPIO Blocks */
#define BLOCKA(x) ((0 * 32) + (x))
#define BLOCKB(x) ((1 * 32) + (x))
#define BLOCKC(x) ((2 * 32) + (x))
#define BLOCKD(x) ((3 * 32) + (x))
#define BLOCKE(x) ((4 * 32) + (x))
#define BLOCKF(x) ((5 * 32) + (x))
#define BLOCKG(x) ((6 * 32) + (x))
#define BLOCKH(x) ((7 * 32) + (x))

/* FPGA GPIO Blocks */
#define BLOCK1B(x) ((0 * 32) + (x))
#define BLOCK1C(x) ((1 * 32) + (x))
#define BLOCKX1(x) ((2 * 32) + (x))
#define BLOCKX2(x) ((3 * 32) + (x))
#define BLOCKX3(x) ((4 * 32) + (x))
#define BLOCKS0(x) ((5 * 32) + (x))
#define BLOCKS1(x) ((6 * 32) + (x))
#define BLOCK2A(x) ((7 * 32) + (x))
#define BLOCKK0(x) ((8 * 32) + (x))

/* Generic signal names */
enum signal_id_t {
    INVALID   = -1,
    I2C0_DATA = 0,
    I2C0_CLK,
    I2C1_DATA,
    I2C1_CLK,
    SD_POWER,
    OP_PANEL_IRQ,
    COVER_DETECT,
    BIN_FULL,
    BSPI_CLK,
    BSPI_CS,
    BSPI_TXD,
    BSPI_RXD,
    VIDEO_SIG,
    CARD_REV0,
    CARD_REV1,
    CARD_REV2,
    CARD_REV3,
    NETWORK_SUPPORT,
    NETWORK_GIGABIT,
    eMMC_ENABLE,
    SDMMC0_CLK,
    SDMMC0_DATA0,
    SDMMC0_DATA1,
    SDMMC0_DATA2,
    SDMMC0_DATA3,
    PWR_EN_USB_FRNT,
    ENET_RESET,
    ENET_25MHZ,
    WIRE_POWER_EN_A9,
    WIRE_POWER_EN_D0,
    OC_USB_FRNT,
    CAGE_FAN,
    VIDEO_CLK,
    AUDIO_ENABLE,
    PANEL_RESET,
    RECOVERY
};

/*
 * Some GPIOs are not valid within a block so use the valid field
 * as a bit mask to show which ones can be used.
 */
struct gpio_block_config_t {
    uint32_t base;
    uint32_t valid;
};

struct gpio_mapping_t {
    enum signal_id_t id;
    uint32_t gpio;
};

#define GPIO_PINCFG_FUNC_SHIFT 0
#define GPIO_PINCFG_FUNC_MASK (0xf << GPIO_PINCFG_FUNC_SHIFT)

#define GPIO_PINCFG_SR_SHIFT  4
#define GPIO_PINCFG_SR_MASK   (0x3 << GPIO_PINCFG_SR_SHIFT)

#define GPIO_PINCFG_PU_SHIFT  6
#define GPIO_PINCFG_PU_MASK   (0x1 << GPIO_PINCFG_PU_SHIFT)

/*
 * To keep from abusing the code store, the setup for the pin
 * configuration is
 * 0-3 function
 *     0 gpio input
 *     1 - 7 alternate functions
 *     8 gpio output of 0
 *     9 gpio output of 1
 * 4-5 slew rate
 * 6   internal pull up enable
 */
#define S(pu_enable, slew_rate, func_sel) \
    (((pu_enable << GPIO_PINCFG_PU_SHIFT) & GPIO_PINCFG_PU_MASK) | ((slew_rate << GPIO_PINCFG_SR_SHIFT) & GPIO_PINCFG_SR_MASK) | ((func_sel << GPIO_PINCFG_FUNC_SHIFT) & GPIO_PINCFG_FUNC_MASK))

int gpio_direction(enum signal_id_t signal, int direction);
int gpio_sample(enum signal_id_t signal);
int gpio_set(enum signal_id_t signal, uint32_t value);
int gpio_mux(enum signal_id_t signal, uint32_t mux);

int gpio_init_pins(const struct gpio_mapping_t *lGpioMapping, uint32_t lGpioMappingSize, const uint8_t *pinSetup, uint32_t pinSetupSize);
int gpio_block_init(const struct gpio_block_config_t *lGpioConfig, uint32_t lGpioConfigSize);
