/***** -*- C -*- **************************************
 * $Id$
 *
 * Copyright (C) 2015 Ricoh Company, Ltd.  All Rights Reserved.
 *	 ORDER			 : Type-SVE1C board touch panel driver
 *	 PROGRAM NAME	 : svinput.ko
 *	 FILE NAME		 : svinput_core.c
 *	 VERSION		 : $Revision$
 *	 DESIGNER		 : Hideaki Yamamoto
 *	 AUTHOR 		 : $Author$
 *-----------------------------------------------------
 *		 HISTORY
 *	 Hideaki Yamamoto - 24 Aug 2015: Created.
 *	 $Log$
 ******************************************************/
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <asm/gpio.h>
#include <asm/io.h>

#include "svinput.h"
#include "sv_touchpanel.h"
#include "sv_keyboard.h"
#include "sv_backlight.h"
#include "sv_led.h"
#include "sv_power.h"
#include "sv_buzzer.h"
#include "sv_fwup.h"

#define OPEREQ_READ_ERR_MAX (3)	/* 3回連続してopereqイベントで異常データを読んだら、バスリセットする */

workqueue_struct_t *ptr_svinput_wqueue = NULL;
svinput_work_t sv_work_struct;
power_mtrig_work_t pw_mt_work_struct;
power_opereq_work_t pw_or_work_struct;
power_scr_work_t pw_scr_work_struct;
power_opereq_poll_work_t pw_or_poll_work_struct;

static irqreturn_t svinput_mtrig_isr(int irq, void *dev_id);
static irqreturn_t svinput_opereq_isr(int irq, void *dev_id);


DEFINE_SPINLOCK(sv_slock);
DEFINE_MUTEX(sv_mutex);

/* request_irqした後は割込有効化なので初期値TRUEとしておく */
static u32 s_opereq_irq_stat = SVINPUT_TRUE;
static u32 s_mtrig_irq_stat = SVINPUT_TRUE;

int already_svinput_boot = 0;
int is_cheetah_ret = -1;
int do_panel_reboot = 0;
unsigned char is_gaudi_model = 0;
int is_getting_up = 0;

extern int s_thine233_power;
int do_reply_hb = 0;

extern wait_queue_head_t hb_wait_queue;

/* for i2c retry dump */
void __iomem *clk_stretch = NULL;

int init_converter(svinput_priv_t *priv);
int exchange_boot_reason(svinput_priv_t *priv);

/* sv_mutexで排他中の処理内でのイベント待ち */
/* argp_condの値を-1する */
int __svinput_wait_event_interruptible_timeout(wait_queue_head_t *argp_wait_queue, int *argp_cond, long timeout)
{
	int ret=0;
	mutex_unlock(&sv_mutex);
	ret = wait_event_interruptible_timeout(*argp_wait_queue, *argp_cond, timeout);
	mutex_lock(&sv_mutex);
	*argp_cond=*argp_cond-1; /* タイムアウト時には後から加算されるので、減算しておく */
	return ret;
}

/* svinput_wait_event_interruptible_timeoutへのイベント送出 */
/* argp_condの値を+1する */
void __svinput_wake_up_interruptible(wait_queue_head_t *argp_wait_queue, int *argp_cond)
{
	*argp_cond=*argp_cond+1;
	wake_up_interruptible(argp_wait_queue);
	return;
}

void enable_opereq_irq(void)
{
	/* unmask irq and enabling GPIOE[23] edge detection */
	unsigned long flags = 0;
	int error = -EINVAL;

	if (s_opereq_irq_stat) {
		DPRINTK(VT100_BLUE"Skip enabling...\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	FUNC_START;
	DPRINTK("ENABLE OPEREQ Interrupt \n");
	
	spin_lock_irqsave(&sv_slock, flags);
	/* clear previous int */
	writel((1 << 23), sv_work_struct.priv->gpio_base + GPIOE_EDR_REG_OFFSET);
	error = request_irq(sv_work_struct.priv->opereq_irq, svinput_opereq_isr, IRQF_TRIGGER_FALLING, SVINPUT_OPEREQ_GPIO_NAME, &(sv_work_struct.priv->opu_client->dev));
	if (error) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Unable to request opereq IRQ. (%d)\n", error);
		goto EXIT_FUNC;
	}
	spin_unlock_irqrestore(&sv_slock, flags);
	s_opereq_irq_stat = SVINPUT_TRUE;
 EXIT_FUNC:
	FUNC_END;
}
EXPORT_SYMBOL(enable_opereq_irq);

void disable_opereq_irq(void)
{
	/* mask irq and disabling GPIOE[23] edge detection */
	unsigned long flags = 0;

	if (!s_opereq_irq_stat) {
		DPRINTK(VT100_BLUE"Skip disabling...\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	FUNC_START;
	DPRINTK("DISABLE OPEREQ Interrupt \n");
	
	spin_lock_irqsave(&sv_slock, flags);
	free_irq(sv_work_struct.priv->opereq_irq, &(sv_work_struct.priv->opu_client->dev));
	spin_unlock_irqrestore(&sv_slock, flags);
	s_opereq_irq_stat = SVINPUT_FALSE;
EXIT_FUNC:
	FUNC_END;
}
EXPORT_SYMBOL(disable_opereq_irq);

void enable_mtrig_irq(void)
{
	/* unmask irq and enabling GPIOE[27] edge detection */
	unsigned long flags = 0;
	int error = -EINVAL;

	if (s_mtrig_irq_stat) {
		DPRINTK(VT100_BLUE"Skip enabling...\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	FUNC_START;
	DPRINTK("ENABLE MLTTRG Interrupt \n");
	
	spin_lock_irqsave(&sv_slock, flags);
	/* clear previous int */
	writel((1 << 27), sv_work_struct.priv->gpio_base + GPIOE_EDR_REG_OFFSET);
	error = request_irq(sv_work_struct.priv->mtrig_irq, svinput_mtrig_isr, IRQF_TRIGGER_FALLING, SVINPUT_MTRIG_IN_GPIO_NAME, &(sv_work_struct.priv->opu_client->dev));
	if (error) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Unable to request mtrig IRQ. (%d)\n", error);
		goto EXIT_FUNC;
	}
	spin_unlock_irqrestore(&sv_slock, flags);
	s_mtrig_irq_stat = SVINPUT_TRUE;
	
 EXIT_FUNC:
	FUNC_END;
}
EXPORT_SYMBOL(enable_mtrig_irq);

void disable_mtrig_irq(void)
{
	/* mask irq and disabling GPIOE[27] edge detection */
	unsigned long flags = 0;

	if (!s_mtrig_irq_stat) {
		DPRINTK(VT100_BLUE"Skip disabling...\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	FUNC_START;
	DPRINTK("DISABLE MLTTRG Interrupt \n");

	spin_lock_irqsave(&sv_slock, flags);
	free_irq(sv_work_struct.priv->mtrig_irq, &(sv_work_struct.priv->opu_client->dev));
	spin_unlock_irqrestore(&sv_slock, flags);
	s_mtrig_irq_stat = SVINPUT_FALSE;
 EXIT_FUNC:
	FUNC_END;
}
EXPORT_SYMBOL(disable_mtrig_irq);

void i2c_reg_dump(void)
{
	void  __iomem  *twsimv  = NULL;	/*  from 0xd401_1000  */
	unsigned long   value;

	/* remap */
	twsimv  = ioremap(0xd4011000, 0x800);

	/* dump reg */
	FORCE_DPRINTK(KERN_EMERG "[svinput]dump twsi regs");
	value = readl(twsimv);
	FORCE_DPRINTK(KERN_EMERG "[svinput]ibmr       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x08);
	FORCE_DPRINTK(KERN_EMERG "[svinput]idbr       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x10);
	FORCE_DPRINTK(KERN_EMERG "[svinput]icr        : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x18);
	FORCE_DPRINTK(KERN_EMERG "[svinput]isr        : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x20);
	FORCE_DPRINTK(KERN_EMERG "[svinput]isar       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x28);
	FORCE_DPRINTK(KERN_EMERG "[svinput]ilcr       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x30);
	FORCE_DPRINTK(KERN_EMERG "[svinput]iwcr       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x40);
	FORCE_DPRINTK(KERN_EMERG "[svinput]wfifo      : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x44);
	FORCE_DPRINTK(KERN_EMERG "[svinput]wfifo_wptr : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x48);
	FORCE_DPRINTK(KERN_EMERG "[svinput]wfifo_rptr : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x50);
	FORCE_DPRINTK(KERN_EMERG "[svinput]rfifo      : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x54);
	FORCE_DPRINTK(KERN_EMERG "[svinput]rfifo_wptr : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x58);
	FORCE_DPRINTK(KERN_EMERG "[svinput]rfifo_rptr : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x60);
	FORCE_DPRINTK(KERN_EMERG "[svinput]fifo_tshld : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x70);
	FORCE_DPRINTK(KERN_EMERG "[svinput]rev0       : 0x%08x\n", (unsigned int)value);
	value = readl(twsimv+0x74);
	FORCE_DPRINTK(KERN_EMERG "[svinput]rev1       : 0x%08x\n", (unsigned int)value);

	/* unmap */
	iounmap(twsimv);
}

void i2c_bus_reset(void)
{
	void  __iomem  *twsiicr = NULL;		/*  from 0xd401_1010  */
	void  __iomem  *twsiisr = NULL;		/*  from 0xd401_1018  */
	void  __iomem  *twsiilcr = NULL;	/*  from 0xd401_1028  */
	void  __iomem  *twsiiwcr = NULL;	/*  from 0xd401_1030  */
	unsigned long   value;
	
	FUNC_START;
	
	i2c_reg_dump();
	
	/* THCV233 Power off */
	gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 0);
	gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 0);
	s_thine233_power = 0;
	
	/* disable i2c1_irq */
	disable_irq(39);
	
	/* remap */
	twsiicr = ioremap(0xd4011010, 4);
	twsiisr = ioremap(0xd4011018, 4);
	twsiilcr = ioremap(0xd4011028, 4);
	twsiiwcr = ioremap(0xd4011030, 4);
	
	/* set ur bit*/
	value = readl(twsiicr);
	value |= 0x00004000;
	writel(value, twsiicr);
	
	/* reset icr reg */
	value = readl(twsiicr);
	value &= ~0x40098000;
	value |= ICR_SDA_GFE_VAL | ICR_GPIO_EN_VAL | ICR_MODE_VAL;
	writel(value, twsiicr);
	
	/* clear isr reg */
	value = 0x00000000;
	writel(value, twsiisr);
	
	/* set ilcr */
	value = readl(twsiilcr);
	value &= ~0x0003fe00;
	value |= ILCR_FLV_VAL;
	writel(value, twsiilcr);
	
	/* set iwcr */
	value = readl(twsiiwcr);
	value &= ~0x1f;
	value |= IWCR_COUNT_VAL;
	writel(value, twsiiwcr);
	
	/* unset ur bit*/
	value = readl(twsiicr);
	value &= (~(0x00004000));
	writel(value, twsiicr);
	
	msleep(10);
	
	/* unmap */
	iounmap(twsiicr);
	iounmap(twsiisr);
	iounmap(twsiilcr);
	iounmap(twsiiwcr);
	
	/* enable i2c1_irq */
	enable_irq(39);
	
	/* THCV233 Power on and init */
	gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 1);
	gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 1);
	s_thine233_power = 1;
	mdelay(10);
	init_converter(NULL);
	
	FUNC_END;
}

int i2c_master_sendrecv(const struct i2c_client *client, const char *sndbuf, int sndcount, char *rcvbuf, int rcvcount)
{
	int ret;
	struct i2c_adapter *adap = client->adapter;
	struct i2c_msg msg[2];
	
	if(s_thine233_power == 0)
	{
		ret = -1;
		DPRINTK("thine power off. i2c master sendrecv skip...\n");
		
		return ret;
	}
	
	msg[0].addr = msg[1].addr = client->addr;
	msg[0].flags = msg[1].flags = client->flags & I2C_M_TEN;
	msg[1].flags |= I2C_M_RD;
	msg[0].len = sndcount;
	msg[0].buf = (char *)sndbuf;
	msg[1].len = rcvcount;
	msg[1].buf = (char *)rcvbuf;

	ret = i2c_transfer(adap, msg, 2);
	
	if (ret != 2)
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't sendrecv with thine:%s, errno : %d\n", __FUNCTION__, ret);
		i2c_bus_reset();
	}

	return (ret == 2) ? 0 : -1;
}

/* 233のレジスタアクセス(read) */
int svinput_thine_read(struct i2c_client *client, u8 reg_addr, u8 *reg_val)
{
	int rc = 0;
	u8 val = {0xff};
	int cnt_snd = 0;
	int cnt_rcv = 0;
	
	if(s_thine233_power == 0)
	{
		rc = -1;
		DPRINTK("thine power off. thine read skip...\n");
		
		return rc;
	}
	
 RETRY_THINE_RD:
	
	rc = i2c_master_send(client, &reg_addr, 1);/* レジスタアドレスを送出して */
	if (rc >= 0)
	{
		rc = i2c_master_recv(client, &val, 1);
		if (rc != 1)
		{
			DPRINTK(KERN_EMERG "Can't recv from thine:%s, errno : %d\n", __FUNCTION__, rc);
			i2c_bus_reset();
			
			if(++cnt_rcv < 240)
			{
				msleep(500);
				goto RETRY_THINE_RD;
			}
			else
			{
				rc = -1;
			}
		}
		else
		{
			rc = 1;
			(*reg_val) = val;
		}
	}
	else
	{
		DPRINTK(KERN_EMERG "Can't send to thine : %s, errno : %d\n", __FUNCTION__, rc);
		i2c_bus_reset();
		
		if(++cnt_snd < 240)
		{
			msleep(500);
			goto RETRY_THINE_RD;
		}
		else
		{
			rc = -1;
		}
	}

	return rc;
}
EXPORT_SYMBOL(svinput_thine_read);

/* 233のレジスタアクセス(write) */
int svinput_thine_write(struct i2c_client *client, u8 reg_addr, u8 reg_val)
{
	int rc = 0;
	u8 write_data[2] = {0};
	int cnt = 0;
	
	if(s_thine233_power == 0)
	{
		rc = -1;
		DPRINTK("thine power off. thine write skip...\n");
		
		return rc;
	}
	
 RETRY_THINE_WR:
	
	write_data[0] = reg_addr;
	write_data[1] = reg_val;
	rc = i2c_master_send(client, write_data, 2);
	if (rc != 2)
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't send to thine : %s, errno : %d\n", __FUNCTION__, rc);
		i2c_bus_reset();
		
		if(++cnt < 240)
		{
			msleep(500);
			goto RETRY_THINE_WR;
		}
	}
	else
	{
		DPRINTK("Send thine to 0x%02x 0x%02x\n", reg_addr, reg_val);
	}

	return rc;
}
EXPORT_SYMBOL(svinput_thine_write);

static int get_opereq_value(void)
{
	/* OPEREQ_N : GPIOE[23] */
	volatile u32 gpiod_plr = 0x00000000;
	gpiod_plr = readl(sv_work_struct.priv->gpio_base + GPIOD_PLR_REG_OFFSET);/* read plr */
	return (int)((gpiod_plr >> 20) & 0x00000001);
}

static void svinput_wqueue_handler(work_struct_t *work)
{
	int32_t rc = 0;
	svinput_work_t *ptr_work = NULL;
	uint8_t recv_buf[I2C_RECV_BUF_SIZE] = {0};
	uint8_t send_buf[I2C_SEND_BUF_SIZE] = {0};
	volatile u8 reg_val = 0;
	volatile int opereq_n = 0;
	int cnt = 0;
	int i=0;
	static unsigned int err_cnt = 0;	/* 3回連続で異常データを受けたらバスリセットする */

	FUNC_START;
	mutex_lock(&sv_mutex);
	//DPRINTK("lock mutex\n");

	ptr_work = (svinput_work_t*)work;

	rc = i2c_master_recv(ptr_work->priv->opu_client, (char*)(recv_buf), sizeof(recv_buf));
	if (rc < 0) {
		DPRINTK("i2c_master_recv returns %d\n", rc);
		goto EXIT_FUNC;
	}
	
	/* ST2以外のイベントに対してはPAK送信しない */
	if ( recv_buf[0] == ST2 ) {
		err_cnt=0;
		send_buf[0] = send_buf[1] = send_buf[2] = PAK;
		rc = i2c_master_send(ptr_work->priv->opu_client, send_buf, 3);
		if (rc < 0) {
			DPRINTK("i2c_master_send returns %d\n", rc);
			goto EXIT_FUNC;
		}

		switch (recv_buf[1])
			{
			case EVT_TPANEL:
				touchpanel_wqueue_handler(work, (svinput_slv_tp_cmd_t*)recv_buf);
				break;
			case EVT_KEY:
				keyboard_wqueue_handler(work, (svinput_slv_kbd_cmd_t*)recv_buf);
				break;
			case EVT_TPANEL_CALIB_REQ:
			case EVT_TPANEL_CALIB_RET:
				touchpanel_wqueue_calib_handler(work, (uint8_t*)recv_buf);
				break;
			case EVT_HEART_BEAT:
				do_reply_hb = 1;
				wake_up_interruptible(&hb_wait_queue);
				//DPRINTK("wake up hb_wait_queue\n");
				break;
			case EVT_BUZ_END:
				buzzer_wqueue_handler((svinput_slv_buz_end_cmd_t*)recv_buf);
				break;
			default:
				DPRINTK("receive invalid command: 0x%02x\n", recv_buf[1]);
				break;
			}
	}
	else if ( (recv_buf[0] == recv_buf[1]) && (recv_buf[1] == recv_buf[2]) && ((recv_buf[2] == PNK)||(recv_buf[2] == PAK)) ) {
		/* NOP */
		FORCE_DPRINTK(KERN_NOTICE "[svinput] %02x recv. NOP for Seq recovery \n",recv_buf[2]);
	}
	else {
		err_cnt++;
		if ( err_cnt >= OPEREQ_READ_ERR_MAX ) {
			err_cnt=0;
			i2c_bus_reset();
			goto EXIT_FUNC;
		}
		FORCE_DPRINTK(KERN_NOTICE"[%s] recv rc:%d\n", __FUNCTION__,rc);
		FORCE_DPRINTK(KERN_NOTICE"[%s] %02x  %02x  %02x  %02x \n",  __FUNCTION__, recv_buf[0], recv_buf[1], recv_buf[2], recv_buf[3]);

		if ( (recv_buf[0] == recv_buf[1]) && (recv_buf[1] == recv_buf[2]) && (recv_buf[2] == 0xff) ) {
			/* readした内容が0xffの場合は、無視する */
			FORCE_DPRINTK(KERN_NOTICE"[%s] All 0xff maybe data nothing. NOP \n", __FUNCTION__);
		}
		else {
			/* 信号破損。 PNKを返す。*/
			FORCE_DPRINTK(KERN_NOTICE "[svinput]Recv error data. Send PNK \n");
			send_buf[0] = send_buf[1] = send_buf[2] = PNK;
			rc = i2c_master_send(ptr_work->priv->opu_client, send_buf, 3);
			if (rc < 0) {
				DPRINTK("i2c_master_send returns %d\n", rc);
				goto EXIT_FUNC;
			}
		}
	}

	udelay(SVINPUT_234_REG_POLLING_INTERVAL);
	svinput_thine_read(ptr_work->priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, (u8 *)&reg_val);
	for (cnt = 0; cnt < SVINPUT_234_REG_POLLING_MAX; cnt++) {
		opereq_n = get_opereq_value();
		if (opereq_n != 0) {
			break;
		}
		//DPRINTK("reg_val=0x%02x\tOPRE_REQ=%s\n", (int)reg_val, (opereq_n == 1) ? "Hi" : "Lo");
		udelay(SVINPUT_234_REG_POLLING_INTERVAL);
		svinput_thine_read(ptr_work->priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, (u8 *)&reg_val);
	}

 EXIT_FUNC:
	mutex_unlock(&sv_mutex);
	//DPRINTK("unlock mutex\n");
	FUNC_END;
	return;
}

/* チェックサム計算 posは1から始まってるので注意 */
unsigned char calc_sum(unsigned char *buf, int sum_pos)
{
	int sum = 0;
	int pos = 0;

	for (pos = 1; pos < sum_pos; pos++) {
		sum += buf[pos];
	}

	return (unsigned char)(sum & 0x000000ff);
}

/**
 * by hideaki at 2016/02/10
 * spin_unlockしたタイミングで意図せず割込を拾ってしまう
 * 事が有るので割込が有効な時だけボトムハーフを実行する
 */

/* opereq割り込み処理関数(トップハーフ ) */
static irqreturn_t svinput_opereq_isr(int irq, void *dev_id)
{
	svinput_priv_t *priv = NULL;

	++is_getting_up;

	if (s_opereq_irq_stat) {
		priv = (svinput_priv_t*)dev_id;
		
		/* ボトムハーフ実行 */
		if(is_cheetah_ret == CHEETAH)
		{
			power_opereq_isr(irq, dev_id);
		}
		else if(is_cheetah_ret == JACKAL)
		{
			queue_work(ptr_svinput_wqueue, (work_struct_t*)&sv_work_struct);
		}
		else
		{
			
		}
	}
	
	return IRQ_HANDLED;
}

/* mtrig割り込み処理関数(トップハーフ ) */
static irqreturn_t svinput_mtrig_isr(int irq, void *dev_id)
{
	++is_getting_up;
	if (s_mtrig_irq_stat) {
		power_mtrig_isr(irq, dev_id);
	}

	return IRQ_HANDLED;
}

/* scr_reboot割り込み処理関数(トップハーフ ) */
static irqreturn_t svinput_scr_rboot_isr(int irq, void *dev_id)
{
	power_scr_reboot_isr(irq, dev_id);
	return IRQ_HANDLED;
}

int wait_opereq_assert(svinput_priv_t *priv)
{
	int cnt = 0;
	u8 opereq_n = 0xff;
	u8 port_chk = 0x00;
	int rc = -1;
	
	if (!priv) {
		priv = sv_work_struct.priv;
	}
	
	/* 最大100ms待ち */
	for (cnt = 0; cnt < 100; cnt++) {
		mdelay(1);
		if( (svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &opereq_n)) >= 0 )
		{
			if( (svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR2, &port_chk)) >= 0 )
			{
				rc = 1;
				opereq_n &= 0x04;
				port_chk &= 0x04;
				opereq_n >>= 2;
				port_chk >>= 2;
			}
		}
		
		if (rc >= 0 && opereq_n == 0 && port_chk != 0) {
			FORCE_DPRINTK(KERN_EMERG "[svinput]opereq_n=%d, port_chk=%d\n", opereq_n, port_chk);
			break;
		}
	}
	
	if (opereq_n != 0 || port_chk == 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]opereq_n=%d, port_chk=%d\n", opereq_n, port_chk);
		rc = -1;
	}

	return rc;
}

int opereq_assert_check(struct i2c_client *client)
{
	int rc		= -1;
	u8 opereq_n	= 0xff;
	u8 port_chk	= 0x00;
	
	FUNC_START;
	
	if( (svinput_thine_read(client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &opereq_n)) >= 0 )
	{
		if( (svinput_thine_read(client, (u8)SVINPUT_CNV_GPIO_REG_ADDR2, &port_chk)) >= 0 )
		{
			rc = 1;
			opereq_n &= 0x04;
			port_chk &= 0x04;
			opereq_n >>= 2;
			port_chk >>= 2;
		}
	}
	
	if (rc < 0) /* read failed */
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]Fail to read thine regs\n");
	}
	else if (opereq_n == 0 && port_chk != 0) /* opereq assert */
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]opereq assert\n");
		rc = 1;
	}
	else if (opereq_n != 0 && port_chk != 0) /* opereq negate */
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]opereq negate\n");
		rc = 0;
	}
	else /* thine power off */
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]thine233 power off\n");
		rc = -1;
	}
	
	FUNC_END;
	
	return rc;
}
EXPORT_SYMBOL(opereq_assert_check);

/* Cheetah機か否かを判別(Cheetah：1, Other：0, エラー：-1) */
int is_cheetah(void)
{
	/* dtsからgui device 種類を取得する処理 */
	/* gui_dev_typeという変数に、デバイスツリーに定義された gui-device-type の値（0:4line, 1:touch panel, 2:cheetah）がセットされる */
	struct device_node *np;
	int of_r32_ret, gui_dev_type;
	
	np = of_find_compatible_node(NULL, NULL, "ricoh,gui-device-type");
	if(!np){
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't find compatible node %s\n", __FUNCTION__);
		return -1;
	}
	
	of_r32_ret = of_property_read_u32(np, "gui-device-type", &gui_dev_type); //gui_dev_type 0:4line, 1:touch panel, 2:cheetah
	if(of_r32_ret < 0){
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't read device type %s\n", __FUNCTION__);
		return -1;
	}
	DPRINTK("gui_dev_type %d\n", gui_dev_type);
	
	if(gui_dev_type == 2){
		return 1;
	}
	else{
		return 0;
	}
}

static int svinput_open(struct inode *inode, struct file *file)
{
	int rc = 0;

	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_TOUCHPANEL_MINOR:
		rc = touchpanel_open(inode, file);
		break;
	case SVINPUT_KEYBORD_MINOR:
		rc = keyboard_open(inode, file);
		break;
	case SVINPUT_LED_MINOR:
		rc = led_open(inode, file);
		break;
	case SVINPUT_BACKLIGHT_MINOR:
		rc = backlight_open(inode, file);
		break;
	case SVINPUT_POWMNG_MINOR:
		rc = power_open(inode, file);
		break;
	case SVINPUT_BUZZER_MINOR:
		rc = buzzer_open(inode, file);
		break;
	case SVINPUT_FWUP_MINOR:
		rc = fwup_open(inode, file);
		break;
	default:
		break;
	}

	return rc;
}

static int svinput_release(struct inode *inode, struct file *file)
{
	int rc = 0;

	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_TOUCHPANEL_MINOR:
		rc = touchpanel_release(inode, file);
		break;
	case SVINPUT_KEYBORD_MINOR:
		rc = keyboard_release(inode, file);
		break;
	case SVINPUT_LED_MINOR:
		rc = led_release(inode, file);
		break;
	case SVINPUT_BACKLIGHT_MINOR:
		rc = backlight_release(inode, file);
		break;
	case SVINPUT_POWMNG_MINOR:
		rc = power_release(inode, file);
		break;
	case SVINPUT_BUZZER_MINOR:
		rc = buzzer_release(inode, file);
		break;
	case SVINPUT_FWUP_MINOR:
		rc = fwup_release(inode, file);
		break;
	default:
		break;
	}

	return 0;
}

static ssize_t svinput_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	ssize_t rc = 0;

	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_TOUCHPANEL_MINOR:
		rc = touchpanel_read(file, buffer, count, ppos);
		break;
	case SVINPUT_KEYBORD_MINOR:
		rc = keyboard_read(file, buffer, count, ppos);
		break;
	case SVINPUT_POWMNG_MINOR:
		rc = power_read(file, buffer, count, ppos);
		break;
	default:
		break;
	}

	return rc;
}

static ssize_t svinput_write(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	ssize_t rc = 0;

	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_FWUP_MINOR:
		rc = fwup_write(file, buffer, count, ppos);
		break;
	default:
		break;
	}

	return rc;
}

static unsigned int svinput_poll(struct file *file, poll_table * wait)
{
	unsigned int rc = 0;

	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_TOUCHPANEL_MINOR:
		rc = touchpanel_poll(file, wait);
		break;
	case SVINPUT_KEYBORD_MINOR:
		rc = keyboard_poll(file, wait);
		break;
	case SVINPUT_POWMNG_MINOR:
		rc = power_poll(file, wait);
		break;
	default:
		break;
	}

	return rc;
}

static long svinput_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	long rc = 0;

	mutex_lock(&sv_mutex);
	switch (iminor(file->f_dentry->d_inode)) {
	case SVINPUT_TOUCHPANEL_MINOR:
		rc = touchpanel_ioctl(file, cmd, arg);
		break;
	case SVINPUT_LED_MINOR:
		rc = led_ioctl(file, cmd, arg);
		break;
	case SVINPUT_BACKLIGHT_MINOR:
		rc = backlight_ioctl(file, cmd, arg);
		break;
	case SVINPUT_POWMNG_MINOR:
		rc = power_ioctl(file, cmd, arg);
		break;
	case SVINPUT_BUZZER_MINOR:
		rc = buzzer_ioctl(file, cmd, arg);
		break;
	default:
		break;
	}
	mutex_unlock(&sv_mutex);

	return rc;
}

static struct file_operations svinput_fops = {
	.open		= svinput_open,
	.release	= svinput_release,
	.read		= svinput_read,
	.write		= svinput_write,
	.poll		= svinput_poll,
	.unlocked_ioctl 	 = svinput_ioctl,
};

static void svinput_platform_release(struct device *dev)
{
	FUNC_START;
	FUNC_END;
	return;
}

static const struct platform_device_id svinput_id[] = {
	{ SVINPUT_NAME, 0 },
	{ }
};
MODULE_DEVICE_TABLE(platform, svinput_id);

static struct platform_driver svinput_driver = {
	.driver = {
		.name = SVINPUT_NAME,
	},
	.id_table = svinput_id,
};

static struct platform_device svinput_device = {
	.name = SVINPUT_NAME,
	.dev = {
		.release = svinput_platform_release,
	}
};

int init_converter(svinput_priv_t *priv)
{
	int err = 0;
	int init_retry_cnt = 0;
	u8 reg_val = 0;
	struct i2c_client *client = NULL;

	FUNC_START;

	if (priv) {
		/**
		 * THCV233との通信向けに使うclient構造体を作成
		 */
		client = kzalloc(sizeof(struct i2c_client), GFP_KERNEL);
		client->flags = priv->opu_client->flags;
		client->addr = (SVINPUT_CNV_WR_ADDR >> 1);
		client->adapter = priv->opu_client->adapter;
		client->dev = priv->opu_client->dev;
		client->irq = priv->opu_client->irq;
		snprintf(client->name, I2C_NAME_SIZE, SVINPUT_CNV_DEVNAME);
		priv->cnv_client = client;
	} else {
		client = sv_work_struct.priv->cnv_client;
	}

	/**
	 * 233の初期化 Phase-1
	 */
	DPRINTK("Phase-1\n");
	err = svinput_thine_write(client, (u8)SVINPUT_CNV_DEVID_REG_ADDR, (u8)(0x80 | (SVINPUT_I2C_OPU_WR_ADDR >> 1)));
	if (err != 2) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't init 233:%s\n", __FUNCTION__);
		goto EXIT_FUNC;
	}

	/**
	 * OPUの通信速度設定
	 */
	DPRINTK("Setup opu speed\n");
	for (init_retry_cnt = 0; init_retry_cnt < SVINPUT_CNV_INIT_RETRY_MAX; init_retry_cnt++) {
		err = svinput_thine_write(client, (u8)SVINPUT_CNV_SCLWH_REG_ADDR, (u8)SVINPUT_CNV_SCLWH_REG_VAL);
		if (err == 2) {
			err = svinput_thine_write(client, (u8)SVINPUT_CNV_SCLWL_REG_ADDR, (u8)SVINPUT_CNV_SCLWL_REG_VAL);
			if (err == 2) {
				mdelay(SVINPUT_CNV_INIT_FREQSET_RETRY_WAIT_TIME);
				break;
			}
		} else {
//			if ((init_retry_cnt % 5) == 0) {
//				gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 0);
//				mdelay(SVINPUT_CNV_INIT_MTRIG_LO_TIME);
//				gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 1);
//				mdelay(SVINPUT_CNV_INIT_WAIT_TIME);
//			} else {
				mdelay(SVINPUT_CNV_INIT_FREQSET_RETRY_WAIT_TIME);
//			}
			continue;
		}
	}
	/* ループを抜けてもエラーだったら周波数設定リトライMAX越えてる */
	if (err != 2) {
		goto EXIT_FUNC;
	}

	/**
	 * 233の初期化 Phase-2
	 */
	DPRINTK("Phase-2\n");
	/* read GPIO */
	svinput_thine_read(client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
	/* enabling interrupt */
	err = svinput_thine_write(client, (u8)SVINPUT_CNV_GPIOINT_REG_ADDR, (u8)SVINPUT_CNV_GPIOINT_REG_VAL);
	if (err < 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't set gpio interrupt %s\n", __FUNCTION__);
		goto EXIT_FUNC;
	}
	/* enabling master interrupt */
	err = svinput_thine_write(client, (u8)SVINPUT_CNV_MASINT_REG_ADDR, (u8)SVINPUT_CNV_MASINT_REG_VAL);
	if (err < 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't set master interrupt %s\n", __FUNCTION__);
		goto EXIT_FUNC;
	}
	/* enabling slave interrupt */
	err = svinput_thine_write(client, (u8)SVINPUT_CNV_SLVINT_REG_ADDR, (u8)SVINPUT_CNV_SLVINT_REG_VAL);
	if (err < 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't set slave interrupt %s\n", __FUNCTION__);
		goto EXIT_FUNC;
	}
	else
	{
		err = 0;
	}
	
	if(is_cheetah_ret == JACKAL)
	{
		/**
		 * OPU側OPE_REQ_Nの読込
		 * 最大100ms待ち 操作部側OPE_REQ_NがLoでOK
		 */
		DPRINTK("Read opu side OPE_REQ_N  (for Jackal) \n");
		for (init_retry_cnt = 0; init_retry_cnt < SVINPUT_CNV_OPEREADY_WAIT_CNT; init_retry_cnt++) {
			svinput_thine_read(client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
			if ((reg_val & 0x04) == 0) {
				DPRINTK("reg_val=%d\n", reg_val);
				break;
			}
			mdelay(1);
		}
		if (init_retry_cnt == SVINPUT_CNV_OPEREADY_WAIT_CNT) {
			err = -1;
		} else {
			err = 0;
		}
	}
	else
	{
		DPRINTK("Read opu side OPE_REQ_N  (for Cheetah) \n");
		wait_opereq_assert(priv);
	}
	
	DPRINTK(" err = %d, reg_val = 0x%08x \n", err, reg_val);
	
	FUNC_END;
	/************** これで終了 **************/

 EXIT_FUNC:
	return err;
}

int exchange_version(svinput_priv_t *priv, int directtion, int do_mtrig)
{
	int	err		= -EINVAL;
	int snd_cnt	= 0;
	int recv_cnt = 0;
	int len = 0;
	u8 	reg_val = 0x00;
	unsigned char sum = 0x00;
	svinput_slvl_ver_cmd_t 	slave_ver;
	svinput_mas_ver_cmd_t 	master_ver = {
		.hdr 	= ST1,
		.cmd 	= 0x03,
		.len 	= 0x03,
		.mjcd 	= 0x31,
		.micdh 	= 0x30,
		.micdl 	= 0x31,
		.sum 	= 0x98,
	};
	char recv_buf[I2C_RECV_BUF_SIZE] 	= {0};
	char pack_buf[I2C_ACK_BUF_SIZE] 	= {PAK, PAK, PAK};
	
	FUNC_START;
	DPRINTK("_____ START Version Exchange _____\n");

	if(!priv)
	{
		priv = sv_work_struct.priv;
	}

	snd_cnt =0;
	
RETRY_MTRIG:

	if( already_svinput_boot != 0 )
	{
		disable_mtrig_irq();
	}

	if(do_mtrig != 0)
	{
	//	if( already_svinput_boot != 0 )
	//	{
	//		disable_mtrig_irq();
	//	}
		
		/* MTRIG send */
		FORCE_DPRINTK(KERN_EMERG "[svinput]MTRIG send\n");
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 0);
		msleep(60);
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 1);
	}

	/* OPEREQ assert wait */
	FORCE_DPRINTK(KERN_EMERG "[svinput]OPEREQ assert wait\n");
	err = wait_opereq_assert(priv);
	if (err == -1)
	{
		int retry_count_boot = 600;
		int	retry_count_power_on_from_ctl = 180;
		int	retry_count_power_on_from_opu = 5;
		if (is_cheetah_ret == JACKAL) {
			/* Jackalの時はいずれもリトライ5回(引継ぎ時点の実装合わせ)) */
			retry_count_boot = 5;
			retry_count_power_on_from_ctl = 5;
			retry_count_power_on_from_opu = 5;
		}
		
		/* 起動時、Cheetahからの応答を10min待つ(MTリトライ間隔1秒) */
		if(already_svinput_boot == 0){
			if (++snd_cnt < retry_count_boot) {/* wait max 10min */
				/* wait_opereq_assert()で最大100ms待つのでリトライ間隔は100ms+900ms=1sec */
				msleep(900);
				if(do_mtrig != 0){
					FORCE_DPRINTK(KERN_EMERG VT100_YELLOW"[svinput]Retry MTRIG send\n"VT100_NORM);
				}
				goto RETRY_MTRIG;
			} else {
				FORCE_DPRINTK(KERN_EMERG "[svinput]OPEREQ not assert\n");
				err = -EINVAL;
				goto EXIT_FUNC;
			}
		/* CTL要因復帰時、Cheetahからの応答を3min待つ(MTリトライ間隔1秒) */
		} else if(do_mtrig != 0){
			if (++snd_cnt < retry_count_power_on_from_ctl) {/* wait max 3min */
				msleep(900);
				FORCE_DPRINTK(KERN_EMERG VT100_YELLOW"[svinput]Retry MTRIG send\n"VT100_NORM);
				goto RETRY_MTRIG;
			} else {
				FORCE_DPRINTK(KERN_EMERG "[svinput]OPEREQ not assert\n");
				err = -EINVAL;
				goto EXIT_FUNC;
			}
		} else {
			/**
			 * Cheetah要因復帰の時、Cheetahから送信されたマルチトリガを検知した後、
			 * 500ms待ってもopereqアサートされない場合、何もせずに処理をやめる
			 */
			if (++snd_cnt < retry_count_power_on_from_opu) {/* wait max 500ms */
				goto RETRY_MTRIG;
			} else {
				err = FAKE_MTRIG;
				goto EXIT_FUNC;
			}
		}
	}
	
	snd_cnt =0;
RETRY_MAS_VER_SEND:
RETRY_SEND:
	
	/* MAS->SLV notify version */
	FORCE_DPRINTK(KERN_EMERG "[svinput]MAS->SLV notify version\n");
	/* Cheetah用処理フロー */
	if (is_cheetah_ret == CHEETAH) {
		if (directtion == 1)
		{
			master_ver.mjcd = 0x71; /* Cheetah MjCD is 0x71 */
			master_ver.sum = 0xd8;
		}
		else
		{
			master_ver.mjcd = 0x31; /* Cheetah MjCD is 0x31 */
			master_ver.sum = 0x98;
		}
		err = i2c_master_sendrecv(priv->opu_client, (char*)(&master_ver), sizeof(master_ver), recv_buf, sizeof(recv_buf));
		if (err < 0)
		{
			msleep(500);
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send master version\n"VT100_NORM);
			//goto EXIT_FUNC;
			FORCE_DPRINTK(KERN_EMERG "[i2c_master_sendrecv][%s]clk_stretch=0x%x\n", __func__, readl(clk_stretch));
			goto RETRY_SEND;
		}
		else
		{
			DPRINTK("hdr  =0x%02x\n", master_ver.hdr);
			DPRINTK("cmd  =0x%02x\n", master_ver.cmd);
			DPRINTK("len  =0x%02x\n", master_ver.len);
			DPRINTK("mjcd =0x%02x\n", master_ver.mjcd);
			DPRINTK("micdh=0x%02x\n", master_ver.micdh);
			DPRINTK("micdl=0x%02x\n", master_ver.micdl);
			DPRINTK("sum  =0x%02x\n", master_ver.sum);
		}
		if (recv_buf[0] != PAK)
		{
			DPRINTK("recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);

			if (snd_cnt <= 10)
			{
				snd_cnt++;
				msleep(20);
				goto RETRY_SEND;
			}
			else
			{
				FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave ack but no problem\n");
				FORCE_DPRINTK(KERN_EMERG "[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
			}
		}
	}
	/* Jackal用処理フロー */
	else if (is_cheetah_ret == JACKAL)
	{
		if (directtion == 1)
		{
			//DPRINTK("[svinput]Directtion Bit is Hi\n"); // test用(17/2/3)
			master_ver.mjcd = 0x71; /* Jackal MjCD is 0x71 */
//			master_ver.micdl = 0x32; /* Heart Beat ON */
			master_ver.micdl = 0x30; /* Heart Beat OFF */
			master_ver.sum = calc_sum((char*)(&master_ver), sizeof(master_ver) - 1);
		}
		else if (directtion == 0)
		{
			//DPRINTK("[svinput]Directtion Bit is Lo\n"); // test用(17/2/3)
			master_ver.mjcd = 0x31; /* Jackal MjCD is 0x31 */
//			master_ver.micdl = 0x32; /* Heart Beat ON */
			master_ver.micdl = 0x30; /* Heart Beat OFF */
			master_ver.sum = calc_sum((char*)(&master_ver), sizeof(master_ver) - 1);
		}
		else
		{
			DPRINTK("Invalid Directtion Bit\n"); // test用(17/2/3)
			err = -EINVAL;
			goto EXIT_FUNC;
		}
		
		err = i2c_master_sendrecv(priv->opu_client, (char*)(&master_ver), sizeof(master_ver), recv_buf, sizeof(recv_buf));
		if (err == -1)
		{
			msleep(500);
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send master version\n"VT100_NORM);
			//goto EXIT_FUNC;
			FORCE_DPRINTK(KERN_EMERG "[i2c_master_sendrecv][%s]clk_stretch=0x%x\n", __func__, readl(clk_stretch));
			goto RETRY_MAS_VER_SEND;
		}
		else if (err == 0)
		{
			DPRINTK("hdr  =0x%02x\n", master_ver.hdr);
			DPRINTK("cmd  =0x%02x\n", master_ver.cmd);
			DPRINTK("len  =0x%02x\n", master_ver.len);
			DPRINTK("mjcd =0x%02x\n", master_ver.mjcd);
			DPRINTK("micdh=0x%02x\n", master_ver.micdh);
			DPRINTK("micdl=0x%02x\n", master_ver.micdl);
			DPRINTK("sum  =0x%02x\n", master_ver.sum);
		}
		else
		{
			DPRINTK("ERROR!! Invalid Return Value (i2c_master_sendrecv)\n"); // test用(17/2/3)
			err = -EINVAL;
			goto EXIT_FUNC;
		}
		
		if (recv_buf[0] != PAK)
		{
			DPRINTK("recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);

			if (snd_cnt <= 10)
			{
				DPRINTK("Retry Master Ver send\n"); // test用(17/2/3)
				snd_cnt++;
				msleep(20);
				goto RETRY_MAS_VER_SEND;
			}
			else if ( snd_cnt == 11 )
			{
				/* Konjiki-P1 FT00197 省エネ制御関係のcmdがJackalに届いていない疑いがあるので、最後に一度bus resetしてから再送する。 */
				FORCE_DPRINTK(KERN_EMERG "[svinput]Retry Master Ver send last one after bus reset\n");
				snd_cnt++;
				i2c_bus_reset();
				msleep(500);
				goto RETRY_MAS_VER_SEND;
			}
			else
			{
				FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave PAK\n");
				FORCE_DPRINTK(KERN_EMERG "[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
			}
		}
		else {
			DPRINTK("PAK recv OK\n");	//test用：PAK応答あり (17/2/3)
		}
		svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
		DPRINTK("OPEREQ_N(OPU)=%s\n", (reg_val & 0x04) ? "Hi" : "Lo");
	}

RETRY_SLV_VER_RECV:

	/* SLV->MAS notify version */
	FORCE_DPRINTK(KERN_EMERG "[svinput]SLV->MAS notify version\n");
	/* Cheetah用処理フロー */
	if (is_cheetah_ret == CHEETAH) {
		err = i2c_master_recv(priv->opu_client, (char*)(&slave_ver), sizeof(slave_ver));
		if (err < 0)
		{
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't recv slave version errno : %d\n", err);
			goto EXIT_FUNC;
		}
		else
		{
			DPRINTK("hdr  =0x%02x\n", slave_ver.hdr);
			DPRINTK("cmd  =0x%02x\n", slave_ver.cmd);
			DPRINTK("len  =0x%02x\n", slave_ver.len);
			DPRINTK("mjcd =0x%02x\n", slave_ver.mjcd);
			DPRINTK("micdh=0x%02x\n", slave_ver.micdh);
			DPRINTK("micdl=0x%02x\n", slave_ver.micdl);
			sum = calc_sum((char*)(&slave_ver), 6);
			DPRINTK("sum  =0x%02x, slvae_ver.sum=0x%02x\n", sum, slave_ver.ver1);
			if ((sum != slave_ver.ver1) || (slave_ver.mjcd == 0) || (slave_ver.micdh == 0) || (slave_ver.micdl == 0))
			{
				if (snd_cnt <= 10)
				{
					snd_cnt++;
					msleep(20);
					goto RETRY_SEND;
				}
				else
				{
					FORCE_DPRINTK(KERN_EMERG "[svinput]invalid command recieved\n");
					FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", slave_ver.hdr);
					FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", slave_ver.cmd);
					FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", slave_ver.len);
					FORCE_DPRINTK(KERN_EMERG "[svinput]mjcd=0x%02x\n", slave_ver.mjcd);
					FORCE_DPRINTK(KERN_EMERG "[svinput]micdh=0x%02x\n", slave_ver.micdh);
					FORCE_DPRINTK(KERN_EMERG "[svinput]micdl=0x%02x\n", slave_ver.micdl);
					FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x, slvae_ver.sum=0x%02x\n", sum, slave_ver.ver1);
					err = -EINVAL;
					goto EXIT_FUNC;
				}
			}
		}
		err = i2c_master_send(priv->opu_client, pack_buf, sizeof(pack_buf));
		if (err < 0)
		{
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send ack\n"VT100_NORM);
			goto EXIT_FUNC;
		}
	}
	/* Jackal用処理フロー */
	else if (is_cheetah_ret == JACKAL) {
		len = sizeof(slave_ver);
		err = i2c_master_recv(priv->opu_client, (char*)(&slave_ver), len);
		if (err != len)
		{
			if (recv_cnt < 4)
			{
				DPRINTK("Retry recv Slave Ver\n");
				recv_cnt++;
				msleep(20);
				goto RETRY_SLV_VER_RECV;
			}
			else {
				FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't recv slave version errno : %d\n", err);
				goto EXIT_FUNC;
			}
		}
		else
		{
			sum = calc_sum((char*)(&slave_ver), 19);
			if ((sum != slave_ver.sum) || (slave_ver.mjcd == 0) || (slave_ver.micdh == 0) || (slave_ver.micdl == 0))
			{
				if (recv_cnt < 4)
				{
					DPRINTK("Retry recv Slave Ver\n");
					recv_cnt++;
					msleep(20);
					goto RETRY_SLV_VER_RECV;
				}
				else
				{
					FORCE_DPRINTK(KERN_EMERG "[svinput]invalid command recieved\n");
					FORCE_DPRINTK(KERN_EMERG "[svinput]hdr  =0x%02x\n", slave_ver.hdr);
					FORCE_DPRINTK(KERN_EMERG "[svinput]cmd  =0x%02x\n", slave_ver.cmd);
					FORCE_DPRINTK(KERN_EMERG "[svinput]len  =0x%02x\n", slave_ver.len);
					FORCE_DPRINTK(KERN_EMERG "[svinput]mjcd =0x%02x\n", slave_ver.mjcd);
					FORCE_DPRINTK(KERN_EMERG "[svinput]micdh=0x%02x\n", slave_ver.micdh);
					FORCE_DPRINTK(KERN_EMERG "[svinput]micdl=0x%02x\n", slave_ver.micdl);
					FORCE_DPRINTK(KERN_EMERG "[svinput]sum  =0x%02x, slvae_ver.sum=0x%02x\n", sum, slave_ver.sum);
					err = -EINVAL;
					goto EXIT_FUNC;
				}
			}
			else {
				DPRINTK("mjcd =0x%02x(%c)\n", slave_ver.mjcd, slave_ver.mjcd);
				DPRINTK("micdh=0x%02x(%c)\n", slave_ver.micdh, slave_ver.micdh);
				DPRINTK("micdl=0x%02x(%c)\n", slave_ver.micdl, slave_ver.micdl);
				DPRINTK("ver1 =0x%02x(%c)\n", slave_ver.ver1, slave_ver.ver1);
				DPRINTK("ver2 =0x%02x(%c)\n", slave_ver.ver2, slave_ver.ver2);
				DPRINTK("ver3 =0x%02x(%c)\n", slave_ver.ver3, slave_ver.ver3);
				DPRINTK("pno1 =0x%02x(%c)\n", slave_ver.pno1, slave_ver.pno1);
				DPRINTK("pno2 =0x%02x(%c)\n", slave_ver.pno2, slave_ver.pno2);
				DPRINTK("pno3 =0x%02x(%c)\n", slave_ver.pno3, slave_ver.pno3);
				DPRINTK("pno4 =0x%02x(%c)\n", slave_ver.pno4, slave_ver.pno4);
				DPRINTK("pno5 =0x%02x(%c)\n", slave_ver.pno5, slave_ver.pno5);
				DPRINTK("pno6 =0x%02x(%c)\n", slave_ver.pno6, slave_ver.pno6);
				DPRINTK("pno7 =0x%02x(%c)\n", slave_ver.pno7, slave_ver.pno7);
				DPRINTK("pno8 =0x%02x(%c)\n", slave_ver.pno8, slave_ver.pno8);
				DPRINTK("posfx=0x%02x(%c)\n", slave_ver.pnosfx, slave_ver.pnosfx);
				DPRINTK("initkey=0x%02x\n", slave_ver.initkey);
				DPRINTK("sum  =0x%02x, slvae_ver.sum=0x%02x\n", sum, slave_ver.sum);
			}
		}
		len = sizeof(pack_buf);
		err = i2c_master_send(priv->opu_client, pack_buf, len);
		if (err != len)
		{
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send PAK\n"VT100_NORM);
			goto EXIT_FUNC;
		}
		else {
			DPRINTK("PAK send OK\n"); /* test用：PAK送信OK */
		}
		if(is_cheetah_ret == JACKAL) {
			err = 0; 
		}
	}

	power_set_ucom_ver(&slave_ver);

EXIT_FUNC:

	FUNC_END;
	return err;
}

int signal_status_notify(svinput_priv_t *priv, int sst, int ver_exch)
{
	int err		= -EINVAL;
	int snd_cnt	= 0;
	u8  reg_val = 0x00;
	svinput_mas_lp_cmd_t master_lpow = {
		.hdr	= ST1,
		.cmd	= 0x1a,
		.len	= 0x01,
		.sst	= 0x10,
		.sum	= 0x2b,
	};
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	
	FUNC_START;
	DPRINTK("_____ START SignalStatus Notify _____\n");
	
	if (!priv)
	{
		priv = sv_work_struct.priv;
	}
	
	if(ver_exch != 0)
	{
		err = exchange_version(priv, 1, 1);
		if (err < 0)
		{
			goto EXIT_FUNC;
		}
		
		
		DPRINTK("<<<  exchange boot reason(signal status notify) >>>\n");
		err = exchange_boot_reason(priv);
		if (err < 0)
		{
			DPRINTK("<<<  exchange boot reason(signal status notify) err:%d >>>\n", err);
			goto EXIT_FUNC;
		}
		
	}
	
RETRY_SEND:
	
	/* MAS->SLV notify low-power mode */
	if ((is_cheetah_ret == JACKAL) && (sst & 0x80)) {
		//DPRINTK("[svinput]POKUSB is masked OK\n"); // test用(17/2/3)
		sst = sst & 0x7f; //jackalではPOKUSBはマスクする
	}
	FORCE_DPRINTK(KERN_EMERG "[svinput]MAS->SLV notify low-power mode 0x%02x\n", sst);
	master_lpow.sst = sst;
	master_lpow.sum = calc_sum((char *)(&master_lpow), 4);
	err = i2c_master_sendrecv(priv->opu_client, (char*)(&master_lpow), sizeof(master_lpow), recv_buf, sizeof(recv_buf));
	if (err == -1)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send master lowpoer\n"VT100_NORM);
		FORCE_DPRINTK(KERN_EMERG "[i2c_master_sendrecv][%s]clk_stretch=0x%x\n", __func__, readl(clk_stretch));
		goto RETRY_SEND;
	}
	else if (err == 0)
	{
		DPRINTK("hdr=0x%02x\n", master_lpow.hdr);
		DPRINTK("cmd=0x%02x\n", master_lpow.cmd);
		DPRINTK("len=0x%02x\n", master_lpow.len);
		DPRINTK("sst=0x%02x\n", master_lpow.sst);
		DPRINTK("sum=0x%02x\n", master_lpow.sum);
	}
	if(recv_buf[0] != PAK)
	{
		DPRINTK("recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
		
		if(snd_cnt <= 10)
		{
			snd_cnt++;
			mdelay(20);
			goto RETRY_SEND;
		}
		else if ( snd_cnt == 11 )
		{
			/* Konjiki-P1 FT00197 省エネ制御関係のcmdがJackalに届いていない疑いがあるので、最後に一度bus resetしてから再送する。 */
			FORCE_DPRINTK(KERN_EMERG "[svinput]Retry Signal Status Notify last one after bus reset\n");
			snd_cnt++;
			i2c_bus_reset();
			msleep(500);
			goto RETRY_SEND;
		}
		else
		{
			FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave ack but no problem\n");
			FORCE_DPRINTK(KERN_EMERG "[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
		}
	}
	
	/* read opereq */
	svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
	DPRINTK("OPEREQ_N(OPU)=%s\n", (reg_val & 0x04) ? "Hi" : "Lo");
	
EXIT_FUNC:
	
	FUNC_END;
	return err;
}

int exchange_boot_reason(svinput_priv_t *priv)
{
	int err		= -EINVAL;
	int snd_cnt	= 0;
	int recv_cnt = 0;
	size_t len = 0;

	unsigned char sum = 0x00;
	svinput_slvl_ver_cmd_t *slave_ver;
	svinput_slv_boot_cmd_t slave_boot;
	svinput_mas_boot_cmd_t master_boot = {
		.hdr = ST1,
		.cmd = 0x1c,
		.len = 0x01,
	};
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	char pack_buf[I2C_ACK_BUF_SIZE] = {PAK, PAK, PAK};
	
	FUNC_START;
	DPRINTK("_____ START BootReason Exchange _____\n");

	if(!priv)
	{
		priv = sv_work_struct.priv;
	}
	
RETRY_RECV:
	
	/* SLV->MAS notify boot reason */
	FORCE_DPRINTK(KERN_EMERG "[svinput]SLV->MAS notify boot reason\n");
	len = sizeof(slave_boot);
	err = i2c_master_recv(priv->opu_client, (char*)(&slave_boot), len);
	if(err != len)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't recv slave boot errno : %d\n", err);
		goto EXIT_FUNC;
	}
	sum = calc_sum((char*)(&slave_boot), 4);
	if((sum != slave_boot.sum) || (slave_boot.hdr == 0))
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]<<<  maybe slv ver retry recv >>>\n");
		FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", slave_boot.hdr);
		FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", slave_boot.cmd);
		FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", slave_boot.len);
		FORCE_DPRINTK(KERN_EMERG "[svinput]cau=0x%02x\n", slave_boot.cau);
		FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x, slave_boot.sum=0x%02x\n", sum, slave_boot.sum);
		// Retry Exchange Version ...
		
		slave_ver = (svinput_slvl_ver_cmd_t *)(&slave_boot);
		sum = calc_sum((char*)(slave_ver), 19);
		if (sum == slave_ver->sum){
			FORCE_DPRINTK(KERN_EMERG"[svinput]<<<  send PAK for exchange boot reason  >>>\n");
			len = sizeof(pack_buf);
			err = i2c_master_send(priv->opu_client, pack_buf, len);
			if(err == len)
			{
				// Success Exchange Version...
			}
		}
		if(recv_cnt < 10)
		{
			recv_cnt++;
			msleep(20);
			goto RETRY_RECV;
		}
		else
		{
			FORCE_DPRINTK(KERN_EMERG "[svinput]invalid command recieved\n");
			FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", slave_boot.hdr);
			FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", slave_boot.cmd);
			FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", slave_boot.len);
			FORCE_DPRINTK(KERN_EMERG "[svinput]cau=0x%02x\n", slave_boot.cau);
			FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x, slave_boot.sum=0x%02x\n", sum, slave_boot.sum);
			err = -EINVAL;
			goto EXIT_FUNC;
		}
	}
	DPRINTK("hdr=0x%02x\n", slave_boot.hdr);
	DPRINTK("cmd=0x%02x\n", slave_boot.cmd);
	DPRINTK("len=0x%02x\n", slave_boot.len);
	DPRINTK("cau=0x%02x\n", slave_boot.cau);
	DPRINTK("sum=0x%02x, slave_boot.sum=0x%02x\n", sum, slave_boot.sum);

	len = sizeof(pack_buf);
	err = i2c_master_send(priv->opu_client, pack_buf, len);
	if(err != len)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send ack\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	
	power_set_boot_cmd(&slave_boot);
	
RETRY_SEND:
	
	/* MAS->SLV notify boot reason */
	FORCE_DPRINTK(KERN_EMERG "[svinput]MAS->SLV notify boot reason\n");
	master_boot.cau = slave_boot.cau;
	master_boot.sum = calc_sum((char*)(&master_boot), sizeof(master_boot) - 1);
	err = i2c_master_sendrecv(priv->opu_client, (char*)(&master_boot), sizeof(master_boot), recv_buf, sizeof(recv_buf));
	if(err == -1)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send master boot\n"VT100_NORM);
		//goto EXIT_FUNC;
		FORCE_DPRINTK(KERN_EMERG "i2c_master_sendrecv][%s]clk_stretch=0x%x\n", __func__, readl(clk_stretch));
		goto RETRY_SEND;
	}
	else if(err == 0)
	{
		DPRINTK("hdr=0x%02x\n", master_boot.hdr);
		DPRINTK("cmd=0x%02x\n", master_boot.cmd);
		DPRINTK("len=0x%02x\n", master_boot.len);
		DPRINTK("cau=0x%02x\n", master_boot.cau);
		DPRINTK("sum=0x%02x\n", master_boot.sum);
	}
	if(recv_buf[0] != PAK)
	{
		DPRINTK("recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
		
		/* ここでboot_reasonを受信した場合は、PAK応答を再送する。 */
		if ( (recv_buf[0] == ST2) && (recv_buf[1]==0x1b) ) {
			FORCE_DPRINTK(KERN_EMERG "[svinput][%s]recv boot_reason after PAK. retry send PAK\n",__func__);
			len = sizeof(pack_buf);
			err = i2c_master_send(priv->opu_client, pack_buf, len);
			if(err != len)
			{
				FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send ack\n"VT100_NORM);
				goto EXIT_FUNC;
			}
		}
		
		if(snd_cnt <= 10)
		{
			snd_cnt++;
			mdelay(20);
			goto RETRY_SEND;
		}
		else if ( snd_cnt == 11 )
		{
			/* Konjiki-P1 FT00197 省エネ制御関係のcmdがJackalに届いていない疑いがあるので、最後に一度bus resetしてから再送する。 */
			FORCE_DPRINTK(KERN_EMERG "[svinput]Retry boot_reason last one after bus reset\n");
			snd_cnt++;
			i2c_bus_reset();
			msleep(500);
			goto RETRY_SEND;
		}
		else
		{
			FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave ack but no problem\n");
			FORCE_DPRINTK(KERN_EMERG "[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
		}
	}
	
EXIT_FUNC:
	
	FUNC_END;
	return err;
}

int init_opu(svinput_priv_t *priv, int do_mtrig)
{
	int	cnt			= 0;
	int	err			= -EINVAL;
	int	opereq_n	= 0;
	u8 	reg_val		= 0x00;
	int len = 0;
	unsigned char sum = 0x00;
	svinput_slvl_ver_cmd_t slave_ver;
	svinput_mas_ver_cmd_t master_ver = {
		.hdr	= ST1,
		.cmd	= 0x03,
		.len	= 0x03,
		.mjcd 	= 0x31,
		.micdh	= 0x30,
		.micdl 	= 0x31,
		.sum 	= 0x98,
	};
	svinput_slv_boot_cmd_t slave_boot;
	svinput_mas_boot_cmd_t master_boot = {
		.hdr	= ST1,
		.cmd	= 0x1c,
		.len	= 0x01,
	};
	svinput_mas_lp_cmd_t master_lpow = {
		.hdr	= ST1,
		.cmd	= 0x1a,
		.len	= 0x01,
		.sst	= 0x10,
		.sum	= 0x2b,
	};
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	char pack_buf[I2C_ACK_BUF_SIZE] = {PAK, PAK, PAK};
	
	
	FUNC_START;
	
	if (!priv) {
		priv = sv_work_struct.priv;
	}
	
	is_cheetah_ret = is_cheetah();
	
	if(is_cheetah_ret == CHEETAH)
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]****** init opu cheetah ******\n");
		
		/* exchange_version */
		err = exchange_version(priv, 0, do_mtrig);
		if (err < 0) {
			goto EXIT_FUNC;
		} else if (err == FAKE_MTRIG){
			FORCE_DPRINTK(KERN_EMERG "[svinput]Skip init opu cheetah\n");
			goto EXIT_FUNC;
		}
		
		/* exchange bootreason */
		exchange_boot_reason(priv);
		if (err < 0) {
			goto EXIT_FUNC;
		}
		
		/*  signal_status_notify */
		if(already_svinput_boot == 0 || do_panel_reboot == 1)
		{
			err = signal_status_notify(priv, 0x90, 1);
			already_svinput_boot = 1;
			do_panel_reboot = 0;
			if (err < 0) {
				goto EXIT_FUNC;
			}
		}
		
		/* read opereq */
		svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
		DPRINTK("OPEREQ_N(OPU)=%s\n", (reg_val & 0x04) ? "Hi" : "Lo");
	}
	else if(is_cheetah_ret == JACKAL)
	{
		
		DPRINTK("****** init opu pumbaa ******\n");
		
		/* exchange_version */
		err = exchange_version(priv, 0, do_mtrig);
		if (err < 0) {
			DPRINTK("Function Fail:exchange_version\n");
			goto EXIT_FUNC;
		}
		else if (err == 0) {
			//DPRINTK("[svinput]Function Success:exchange_version\n"); // test用(17/2/3)
		}
		else {
			DPRINTK("ERROR!! Invalid Return Value (exchange_version)\n"); // test用(17/2/3)
			err = -EINVAL;
			goto EXIT_FUNC;
		}

		/* exchange bootreason */
		err = exchange_boot_reason(priv);
		if (err < 0) {
			DPRINTK("Function Fail:exchange_boot_reason\n");
			goto EXIT_FUNC;
		}
		else if (err == 0) {
			//DPRINTK("[svinput]Function Success:exchange_boot_reason\n"); // test用(17/2/3)
		}
		else {
			DPRINTK("ERROR!! Invalid Return Value (exchange_boot_reason)\n"); // test用(17/2/3)
			err = -EINVAL;
			goto EXIT_FUNC;
		}

		/*  signal_status_notify */
		if (already_svinput_boot == 0)
		{
			err = signal_status_notify(priv, 0x10, 0);
			already_svinput_boot = 1;
			if (err < 0) {
				DPRINTK("Function Fail:signal_status_notify\n");
				goto EXIT_FUNC;
			}
			else if (err == 0) {
				//DPRINTK("[svinput]Function Success:signal_status_notify\n"); // test用(17/2/3)
			}
			else {
				DPRINTK("ERROR!! Invalid Return Value (signal_status_notify)\n"); // test用(17/2/3)
				err = -EINVAL;
				goto EXIT_FUNC;
			}
		}
		
	}
	else
	{
		DPRINTK("ERROR!! Invalid Return Value (is_cheetah)\n"); // test用(17/2/3)
		err = -1;
		goto EXIT_FUNC;
	}
	
	DPRINTK("All opu initialization is Okay!\n");
	
	err = 0;
	
EXIT_FUNC:
	
	FUNC_END;
	
	return err;
}

static int svinput_probe(struct i2c_client *client, const struct i2c_device_id *idp)
{
	svinput_priv_t *priv = NULL;
	int err = -EINVAL;
	int error = -EINVAL;
	u32 reg_val = 0x00000000;
	u8 opereq_val = 0x00;
	void __iomem *twsi_base = NULL;
	
	FUNC_START;
	
	/*---- デバイス情報作成 ----*/
	priv = kzalloc(sizeof(svinput_priv_t), GFP_KERNEL);
	if (!priv) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]failed to allocate driver data\n");
		error = -ENOMEM;
		goto error6;
	}
	sv_work_struct.priv = priv;

	/* デバイス構造体にプライベートデータをセット  */
	dev_set_drvdata(&client->dev, priv);

	/* クライアント情報セット */
	priv->opu_client = client;

	/*---- デバイス登録(InputDeviceの登録) ----*/
	priv->svinput_major = register_chrdev(SVINPUT_MAJOR_NUM, SVINPUT_DEVICE_NAME, &svinput_fops);
	if (priv->svinput_major < 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]svinput_probe:unable to get major %d for touch screen devs\n", priv->svinput_major);
		error = -EBUSY;
		goto error4;
	} else if (priv->svinput_major == 0) {
		priv->svinput_major = SVINPUT_MAJOR_NUM;
		FORCE_DPRINTK(KERN_EMERG "[svinput]svinput_major=[%d]\n", priv->svinput_major);
	}
	/* デバイスクラス生成 */
	priv->svinput_class = class_create(THIS_MODULE, SVINPUT_DEVICE_NAME);
	if (IS_ERR(priv->svinput_class)) {
		error = -EBUSY;
		goto error3;
	}
	/* デバイスクラスからデバイス生成 touch secreen device */
	device_create(priv->svinput_class, NULL, MKDEV(priv->svinput_major,0),"%s%d", SVINPUT_DEVICE_NAME);

	/* ドライバ登録 */
	err = platform_driver_register(&svinput_driver);
	if (err != 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]platform_driver_register error:svinput");
		error = -EBUSY;
		goto error2;
	}

	/* デバイス登録 */
	err = platform_device_register(&svinput_device);
	if (err != 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]platform_device_register error:svinput");
		error = -EBUSY;
		goto error1;
	}

	/* s.nakamura add */
	{
		const char model_name[] = "RICOH Color IJ P model K";
		char* get_model_name = NULL;
		int length = strlen(model_name);
		struct device_node* p_node = of_find_all_nodes(NULL);
		
		get_model_name = of_get_property(p_node, "model", NULL);
		if (NULL != get_model_name) {
			if(0 == strncmp(model_name, (const char*)get_model_name, length)) {
				printk("[svinput] is gaudi model : %s \n", get_model_name);
				is_gaudi_model = 1;
			}
		}
	}
	
	/* I2C Ch.0の追加設定 */
	DPRINTK("Setup I2C Ch.0...\n");
	twsi_base = ioremap_nocache(TWSI_BASE_ADDR, TWSI_REMAP_SIZE);
	/* ILCR */
	reg_val = readl(twsi_base + OFFSET_TWSI_ILCR);
	reg_val &= ~0x0003fe00;
	reg_val |= ILCR_FLV_VAL;
	writel(reg_val, twsi_base + OFFSET_TWSI_ILCR);
	/* ICR */
	reg_val = readl(twsi_base + OFFSET_TWSI_ICR_ADDR);
	reg_val &= ~0x40098000;
	reg_val |= ICR_SDA_GFE_VAL | ICR_GPIO_EN_VAL | ICR_MODE_VAL;
	writel(reg_val, twsi_base + OFFSET_TWSI_ICR_ADDR);
	/* IWCR */
	reg_val = readl(twsi_base + OFFSET_TWSI_IWCR_ADDR);
	reg_val &= ~0x1f;
	reg_val |= IWCR_COUNT_VAL;
	writel(reg_val, twsi_base + OFFSET_TWSI_IWCR_ADDR);
	iounmap(twsi_base);

	/* Map GPIO register */
	priv->gpio_base = ioremap_nocache(GPIO_REG_BASE, GPIO_REG_SIZE);

	is_cheetah_ret = is_cheetah();

	/* THCV233初期化 */
	DPRINTK("Initialize THCV233...\n");
	err = init_converter(priv);
	if (err != 0) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Can't initialize 233:svinput\n");
		error = -1;
		goto error1;
	}
	
	/* OPU初期化 */
	DPRINTK("Initializing OPU...\n");
	err = init_opu(priv, 1);
	if(err != 0)
	{
		FORCE_DPRINTK(KERN_EMERG "Can't initialize opu:svinput\n");
		error = -1;
		goto error1;
	}
	
	/**
	 * 操作部初期化のタイミングでOPEREQ_Nがアサートされちゃうので、
	 * 操作部の初期化が終了した後に割込みと遅延処理の初期化を行う様にした
	 * WorkQueueの生成後に割込みハンドラを登録する様に変更 2015/11/20
	 */
	DPRINTK("Setup workqueue...\n");
	ptr_svinput_wqueue = create_workqueue(SVINPUT_WQUEUE_NAME);
	if (!ptr_svinput_wqueue) {
		FORCE_DPRINTK(KERN_WARNING "[svinput]Can't create workqueue_struct! orz!!!\n");
		error = -1;
		goto error5;
	}
	INIT_WORK((work_struct_t*)(&sv_work_struct), svinput_wqueue_handler);
	sv_work_struct.touch_cnt = 0;
	
	/* マルチトリガ受信による遅延処理の初期化 */
	INIT_WORK((work_struct_t *)(&pw_mt_work_struct), power_mtrig_wqueue_handler);
	
	/* OPEREQ受信による遅延処理の初期化 */
	INIT_WORK((work_struct_t *)(&pw_or_work_struct), power_opereq_wqueue_handler);
	
	/* SCR_Reboot受信による遅延処理の初期化 */
	INIT_WORK((work_struct_t *)(&pw_scr_work_struct), power_scr_wqueue_handler);
	
	/* Opereqポーリング遅延処理の初期化 */
	INIT_WORK((work_struct_t *)(&pw_or_poll_work_struct), power_or_poll_wqueue_handler);
	
	DPRINTK("Setup Interrupt...\n");
	priv->opereq_irq = gpio_to_irq(SVINPUT_OPEREQ_GPIO_NO);
	error = request_irq(priv->opereq_irq, svinput_opereq_isr, IRQF_TRIGGER_FALLING, SVINPUT_OPEREQ_GPIO_NAME, &(priv->opu_client->dev));
	if (error) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Unable to request opereq IRQ. (%d)\n", error);
		goto error5;
	}
	priv->mtrig_irq = gpio_to_irq(SVINPUT_MTRIG_IN_GPIO_NO);
	error = request_irq(priv->mtrig_irq, svinput_mtrig_isr, IRQF_TRIGGER_FALLING, SVINPUT_MTRIG_IN_GPIO_NAME, &(priv->opu_client->dev));
	if (error) {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Unable to request mtrig IRQ. (%d)\n", error);
		goto error5;
	}
	/* mtrigの割込みはデフォルトdisable */
	disable_mtrig_irq();
	if (is_gaudi_model) {
		DPRINTK("SVINPUT_SCR_REBOOT_GPIO_NO is not set \n");
	} else {
		priv->scr_reboot_irq = gpio_to_irq(SVINPUT_SCR_REBOOT_GPIO_NO);
		error = request_irq(priv->scr_reboot_irq, svinput_scr_rboot_isr, IRQF_TRIGGER_FALLING, SVINPUT_SCR_REBOOT_GPIO_NAME, &(priv->opu_client->dev));
		if (error) {
			FORCE_DPRINTK(KERN_EMERG "[svinput]Unable to request scr reboot IRQ. (%d)\n", error);
			goto error5;
		}
	}
	
	/* read opereq */
	svinput_thine_read(priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &opereq_val);
	if( (opereq_val & 0x04) == 0 )
	{
		power_opereq_wqueue_handler(NULL);
	}
	
	backlight_probe(client, idp);
	led_probe(client, idp);
	power_probe(client, idp);
	buzzer_probe(client, idp);
	touchpanel_probe(client, idp);
	fwup_probe(client, idp);
	

	FORCE_DPRINTK(KERN_EMERG VT100_RICHBLUE"[svinput]Succeed to connect OPU\n"VT100_NORM);

	FUNC_END;
	return 0;

error1:
	platform_driver_unregister(&svinput_driver);
error2:
	class_destroy(priv->svinput_class);
error3:
	unregister_chrdev(priv->svinput_major, SVINPUT_DEVICE_NAME);
error4:
	free_irq(priv->opereq_irq, &(priv->opu_client->dev));
	free_irq(priv->mtrig_irq, &(priv->opu_client->dev));
error5:
	dev_set_drvdata(&priv->opu_client->dev, NULL);
	kfree(priv);
error6:

	FORCE_DPRINTK(KERN_EMERG VT100_CRIMSON"[svinput]Failed to connect OPU\n"VT100_NORM);

	FUNC_END;
	return error;
}

static int svinput_remove(struct i2c_client *client)
{
	svinput_priv_t *priv = NULL;

	FUNC_START;
	power_remove(client);
	priv = dev_get_drvdata(&client->dev);
	iounmap(priv->gpio_base);
	kfree(priv->cnv_client);
	disable_irq(priv->opereq_irq);
	free_irq(priv->opereq_irq, &(client->dev));
	free_irq(priv->mtrig_irq, &(client->dev));
	platform_device_unregister(&svinput_device);
	platform_driver_unregister(&svinput_driver);
	kfree(priv);
	class_destroy(priv->svinput_class);
	unregister_chrdev(priv->svinput_major, SVINPUT_DEVICE_NAME);
	dev_set_drvdata(&client->dev, NULL);

	/* ワークキューを破棄 */
	flush_workqueue(ptr_svinput_wqueue);
	destroy_workqueue(ptr_svinput_wqueue);
	ptr_svinput_wqueue = NULL;
	sv_work_struct.priv = NULL;
	kfree(priv);
	priv = NULL;
	FUNC_END;
	return 0;
}

int wait_opepanel_disconnect(void)
{
	void __iomem *pscr = NULL;
	volatile unsigned long value = 0;
	int cnt	= 0;
	
	FORCE_DPRINTK(KERN_EMERG "[svinput]Waiting opu disconnection\n");
	pscr = ioremap(0xd4291184, 4);
	for (cnt = 0; cnt < 120; cnt++) {/* wait max 60sec */
		value = readl(pscr);
		value &= 0x00000001;
		if (value == 0) {
			break;
		}
		msleep(500);
	}
	if (value != 0){
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't disconnect opu, send force_shutdown to opu\n"VT100_NORM);
		disable_mtrig_irq();
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 0);
		msleep(500);
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 1);
		for (cnt = 0; cnt < 420; cnt++) {/* wait max 210sec */
			value = readl(pscr);
			value &= 0x00000001;
			if (value == 0) {
				break;
			}
			msleep(500);
		}
	}
	
	iounmap(pscr);
	
	return value;
}

static void svinput_shutdown(struct i2c_client *client)
{
	int ret = -1;
	
	power_shutdown();

	if (is_cheetah_ret == CHEETAH) {
		/* wait disconnect usb2h */
		ret = wait_opepanel_disconnect();
	} else {
		ret = 0;
	}
	
	if (ret != 0) {
		/* エラーが有ってもなにもできないのでこのまま */
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't disconnect opu, but force shutdown\n"VT100_NORM);
	} else {
		FORCE_DPRINTK(KERN_EMERG "[svinput]Disconnected opu usb\n");
	}
	
	return;
}

static const struct i2c_device_id svinput_i2c_id[] = {
	{ SVINPUT_NAME, 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, svinput_i2c_id);

static struct i2c_driver svinput_i2c_driver = {
	.driver = {
		.name = SVINPUT_NAME,
	},
	.probe	  = svinput_probe,
	.remove   = svinput_remove,
	.id_table = svinput_i2c_id,
	.shutdown = svinput_shutdown,
};

struct i2c_client *s_i2c_client_opu = NULL;

static int __init svinput_init(void)
{
	int ret = 0;
	struct i2c_adapter *adapter = NULL;
	struct i2c_board_info info = {
		.type = {'\0'},
		.flags = (unsigned short)0,
		.addr = (unsigned short)(SVINPUT_I2C_OPU_WR_ADDR >> 1),
		.platform_data = NULL,
		.archdata = NULL,
		.of_node = NULL,
		.irq = 0,
	};

	FUNC_START;

	ret = i2c_add_driver(&svinput_i2c_driver);
	if (ret) {
		FORCE_DPRINTK(KERN_WARNING "[svinput]I2C: Driver registration failed, module not inserted.\n");
		goto exit_init;
	}
	strncpy(info.type, SVINPUT_NAME, I2C_NAME_SIZE);
	adapter = i2c_get_adapter(SVINPUT_I2C_CHANNEL);
	s_i2c_client_opu = i2c_new_device(adapter, &info);

	/* for debug */
	/* clk_stretch */
	clk_stretch = ioremap_nocache( 0xd4019000, 4 );

	FUNC_END;

 exit_init:
	return ret;
}

static void __exit svinput_exit(void)
{
	struct i2c_adapter *adapter = NULL;

	FUNC_START;
	adapter = i2c_get_adapter(SVINPUT_I2C_CHANNEL);
	i2c_put_adapter(adapter);
	i2c_unregister_device(s_i2c_client_opu);
	s_i2c_client_opu = NULL;
	i2c_del_driver(&svinput_i2c_driver);
	FUNC_END;

	return;
}

module_init(svinput_init);
module_exit(svinput_exit);

MODULE_AUTHOR("RICOH Company, LTD.");
MODULE_LICENSE("GPL");
