#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <asm/gpio.h>
#include <linux/list.h>

#include "svinput.h"
#include "sv_keyboard.h"

static int s_keyboard_is_opened = SVINPUT_FALSE;

static DECLARE_WAIT_QUEUE_HEAD(kbd_wait_queue);

static LIST_HEAD(keyinfo_list_head);

extern int is_ctl_ready;

DEFINE_MUTEX(keyinfo_list_mutex);

#define UCOMCODE_TO_INDEX(x) ((((x >> 4) & 0x0f) * MAX_KEYSCAN_COL) + (x & 0x0f))
static const sv_ctl_key_code_t key_code_tbl[MAX_KEYSCAN_COL * MAX_KEYSCAN_ROW] = {
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_RESERVED,   OPE_KEY_HOME,       OPE_KEY_SIMPLE_WIN, OPE_KEY_STS_INFO,
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_LOGIN_OUT,  OPE_KEY_ENESAVE,
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_COUNTER,    OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,
	OPE_KEY_RESERVED, OPE_KEY_RESERVED, OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,   OPE_KEY_STP,
};

static ope_input_event_t cur_key = {
	.time = {
		.tv_sec = 0,
		.tv_usec = 0,
	},
	.type = (uint32_t)SV_EV_KEY,
	.param.key = {
		.code  = (uint16_t)0,
		.status = (uint16_t)0,
	},
};

static struct timeval tv = {
	.tv_sec = 0,
	.tv_usec = 0,
};

typedef struct key_event_info
{
	struct list_head list;
	ope_input_event_t cur_key;
} key_event_info_t;

void keyboard_wqueue_handler(work_struct_t *work, svinput_slv_kbd_cmd_t *slv_kbd_cmd)
{
	svinput_work_t *ptr_work = NULL;
	unsigned int key_cnt = 0;
	ptr_work = (svinput_work_t*)work;
	key_event_info_t *new_data = NULL;
	uint16_t key_code = 0;
	
	FUNC_START;
	do_gettimeofday(&tv);
	
//	DPRINTK("keylen = %d\n", slv_kbd_cmd->len);
	for(key_cnt = 0; key_cnt < slv_kbd_cmd->len; key_cnt+=2){
		
		key_code = key_code_tbl[UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[key_cnt+1])];
		if ((key_code == OPE_KEY_RESERVED) || (key_code == OPE_KEY_INVALID)) {
			DPRINTK("Not Support Key. key_code: 0x%x, key_buf: 0x%x\n", key_code, slv_kbd_cmd->key_buf[key_cnt+1]);
			continue;
		}
		if (!is_ctl_ready) {
			DPRINTK("Not Ready CTL. key_code: 0x%x, key_buf: 0x%x\n", key_code, slv_kbd_cmd->key_buf[key_cnt+1]);
			continue;
		}
		
		new_data = kmalloc(sizeof(key_event_info_t), GFP_KERNEL);
		if (new_data == NULL){
			return -ENOMEM;
		}
		INIT_LIST_HEAD(&new_data->list);
		do_gettimeofday(&(new_data->cur_key.time));
		new_data->cur_key.type = SV_EV_KEY;
		new_data->cur_key.param.key.code = key_code;
//		DPRINTK("form ucom=0x%02x, index=%d\n", slv_kbd_cmd->key_buf[key_cnt+1], UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[key_cnt+1]));
		
		switch (slv_kbd_cmd->key_buf[key_cnt]) {
		case SVKBD_PUSH:
			new_data->cur_key.param.key.status = KEY_PUSH;
			DPRINTK("(%ld.%06ld)form ucom=0x%02x, index=%d, Press \n", tv.tv_sec, tv.tv_usec, slv_kbd_cmd->key_buf[key_cnt+1], UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[key_cnt+1]));
			break;
		case SVKBD_RELEASE:
			new_data->cur_key.param.key.status = KEY_RELEASE;
			DPRINTK("(%ld.%06ld)form ucom=0x%02x, index=%d, Release \n", tv.tv_sec, tv.tv_usec, slv_kbd_cmd->key_buf[key_cnt+1], UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[key_cnt+1]));
			break;
		default:
//			DPRINTK("(%ld.%06ld)form ucom=0x%02x, index=%d, unexpected error \n", tv.tv_sec, tv.tv_usec, slv_kbd_cmd->key_buf[key_cnt+1], UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[key_cnt+1]));
			/* FIX No.0004 */
			if (NULL != new_data){
				kfree(new_data);
			}
			return;
		}
		if (s_keyboard_is_opened) {
			mutex_lock(&keyinfo_list_mutex);
//			DPRINTK("add list:%d\n", key_cnt);
			list_add_tail(&new_data->list, &keyinfo_list_head);
			mutex_unlock(&keyinfo_list_mutex);
		}
	}
	if (s_keyboard_is_opened) {
//		DPRINTK("wake up: %s\n", __func__);
		wake_up_interruptible(&kbd_wait_queue);
	}
	FUNC_END;
}

int keyboard_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_keyboard_is_opened = SVINPUT_TRUE;
	FUNC_END;
	return rc;
}

int keyboard_release(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	/* NOP keyboardcloseKvȃP[X͂ȂB(Olibrelease鎖͂) */
	FUNC_END;
	return rc;
}

ssize_t keyboard_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	int rc = 0;
	
	key_event_info_t *ptr;
	
	FUNC_START;
	do_gettimeofday(&tv);
	
	if ((list_empty(&keyinfo_list_head) != 0) && (file->f_flags & O_NONBLOCK)) {
		count = -EAGAIN;
		goto EXIT_FUNC;
	}
	/* wait_queuewake up  L[񃊃XgɃCxgς܂܂wait */
//	DPRINTK("list_empty:%d\n", list_empty(&keyinfo_list_head));

	wait_event_interruptible(kbd_wait_queue, (list_empty(&keyinfo_list_head) == 0));
	mutex_lock(&keyinfo_list_mutex);
	ptr = list_entry(keyinfo_list_head.next, key_event_info_t, list);

//	DPRINTK("list_entry\n");
	rc = copy_to_user(buffer, &(ptr->cur_key), sizeof(ope_input_event_t));
	if (ptr->cur_key.param.key.status == KEY_PUSH) {
		DPRINTK("(%ld.%06ld)key.code=0x%04x, key.status=PUSH \n", tv.tv_sec, tv.tv_usec, ptr->cur_key.param.key.code);
	} else {
		DPRINTK("(%ld.%06ld)key.code=0x%04x, key.status=RELEASE \n", tv.tv_sec, tv.tv_usec, ptr->cur_key.param.key.code);
	}
	list_del(keyinfo_list_head.next);
//	DPRINTK("list_del\n");
	kfree(ptr);
//	DPRINTK("cmplete free\n");
	count = (rc == 0) ? sizeof(ope_input_event_t) : 0;

 EXIT_FUNC:
	mutex_unlock(&keyinfo_list_mutex);
	FUNC_END;
	return count;
}

unsigned int keyboard_poll(struct file *file, poll_table * wait)
{
	unsigned int rc = 0;

//	FUNC_START;
	poll_wait(file, &kbd_wait_queue, wait);

	if (list_empty(&keyinfo_list_head) == 0) {
		rc = (POLLIN | POLLRDNORM);
	}

//	FUNC_END;
	return rc;
}

