/*****************************************************************************
 Copyright(c) 2010 DTS INSIGHT CORPORATION
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.

 Note:

 Modification history
 +-------------- Historical number (000 to 999)
 |    +--------- Modified System Version
 |    |     +--- Classification of New, Modify, Append, Delete
 v    v     v
 No  Ver  Class Date                Description
 ---+-----+----+------------+--------------------------------------------------
 000 01.00 New  2012/10/01      New
 001 01.10 Mod  2013/01/28      Fixed #13066 #13113
 002 02.12 MOD  2014/12/12      Fixed #13672

*****************************************************************************/

/* Include specification ****************************************************/
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <asm/io.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
#include <linux/moduleparam.h>
#endif

#include <smt/SMTAPI.h>
#include "smtif.h"
#include "smt.h"
#include "smtoutput.h"

/* Extern Function prototypes ******************************************************/
extern void smt_register_api(void);
extern void smt_unregister_api(void);
extern int smt_register_chrdev(int* major, int* minor);
extern void smt_unregister_chrdev(int major, int minor);
extern int smt_setup_chrdev(int devno);
extern void smt_cleanup_chrdev(void);

/* static Function prototypes ******************************************************/
static int smt_init_module(void);
static void smt_cleanup_module(void);
unsigned long smt_disable_int(void);
void smt_enable_int(unsigned long flag);

/* Internal variables *******************************************************/
struct smt_device *smt_device = NULL;					// allocated in smt_init_module
#define SMTDEV_MAJOR		250							// SMT device file major number
#define DRIVER_NAME "SMT API Library device driver"		// SMT Device file name 
static int major = SMTDEV_MAJOR;						// SMT Device file major number
static int minor = 0;									// SMT Device file minor number


bool osproc = 1;										// output process info switch(1=ON/0=OFF)
bool osirq =  1;										// IRQIN/OUT output switch(1=ON/0=OFF)
bool oscall = 1;										// OSCALL output switch(1=ON/0=OFF)

// module parameter
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
module_param(major, int, S_IRUGO);
module_param(osproc, bool, S_IRUGO);
module_param(osirq, bool, S_IRUGO);
module_param(oscall, bool, S_IRUGO);
#else
MODULE_PARM(major,"i");
MODULE_PARM(osproc, "0-1i");
MODULE_PARM(osirq, "0-1i");
MODULE_PARM(oscall, "0-1i");
#endif

/****************************************************************************
1.Function: SMT driver register

2.Restriction, Cautions:

3.Argument
I/O|Variable Name			   |Explanation
---+---------------------------+---------------------------------------------

4.Return Value:
		  -ENOMEM				Out of memory
		  -EFAULT				Bad address
		  -EINVAL				Invalid argument

*****************************************************************************/
static int __init smt_init_module(void)
{
	int result;
	unsigned long saved;

	printk("%s\nVersion %s, Compiled %s %s\n", DRIVER_NAME, DRIVER_VERSION, __DATE__, __TIME__);
	result = smt_register_chrdev(&major, &minor);
	if(result < 0){
		printk(KERN_WARNING "%s: cannot get major %d\n" , MODULE_NAME, major);
		return (result);
	}
	printk(KERN_INFO "%s: major number %d\n", MODULE_NAME, major);

	smt_device = kmalloc(sizeof(struct smt_device), GFP_KERNEL);
	if (!smt_device) {
		if (smt_device) {
			smt_cleanup_chrdev();
			kfree(smt_device);
			smt_device = NULL;
		}
		smt_unregister_chrdev(major, minor);
		return(-ENOMEM);
	}
	memset(smt_device,0,sizeof(struct smt_device));					// device struct zero clear
	result = smt_setup_chrdev(MKDEV(major, minor));					// character device initialize 
	if(result < 0){
		if (smt_device) {
			smt_cleanup_chrdev();
			kfree(smt_device);
			smt_device = NULL;
		}
		smt_unregister_chrdev(major, minor);
		return(-EBUSY);
	}
	if( _SMT_Init()!=_SMT_OK ){
		return(-ENOMEM);
	}
	spin_lock_init(&smt_device->lock);									//  spin lock init
	saved = smt_disable_int();
	smt_register_api();													// kernel land API regist
	smt_enable_int(saved);
	return( 0 );
}

/****************************************************************************
1.Function: SMT driver unregist

2.Restriction, Cautions:

3.Argument
I/O|Variable Name			   |Explanation
---+---------------------------+---------------------------------------------

4.Return Value:

*****************************************************************************/
static void __exit smt_cleanup_module(void)
{
	unsigned long saved;

	saved = smt_disable_int();
	smt_unregister_api();										// unmap io address
	smt_enable_int(saved);
	if (smt_device) {
		smt_cleanup_chrdev();
		kfree(smt_device);
		smt_device = NULL;
	}
	smt_unregister_chrdev(major, minor);
}
module_init(smt_init_module);
module_exit(smt_cleanup_module);
MODULE_LICENSE("GPL v2");

/****************************************************************************
1.Function: disable interrupt and preemption

2.Restriction, Cautions:

3.Argument
I/O|Variable Name			   |Explanation
---+---------------------------+---------------------------------------------

4.Return Value:

*****************************************************************************/
unsigned long smt_disable_int(void)
{
	unsigned long flag;

	spin_lock_irqsave(&smt_device->lock,flag);
	return( flag );
}

/****************************************************************************
1.Function: enable interrupt and preemption

2.Restriction, Cautions:

3.Argument
I/O|Variable Name			   |Explanation
---+---------------------------+---------------------------------------------

4.Return Value:

*****************************************************************************/
void smt_enable_int(unsigned long flag)
{
	spin_unlock_irqrestore(&smt_device->lock, flag);
}
