/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/**
 * \file lpp_api.h
 *
 * \brief LowPowerProcessor CPU API.
 *
 * The common API used to interface low power system functions.
 *
 **/

#ifndef INC_LPP_API_H
#define INC_LPP_API_H

#include <stdbool.h>

/*
 * Description of our power modes: (see lpp_power_level_t)
 * <lpp_power_wake>  ~= ACPI S1
 *      AP is powered, but in (sleep) lower power mode.  We assume our PLL is on, so that we can quickly transition.
 *      On LPP entry we change from existing -->  <lpp_power_service>
 *      On LPP exit we change from <lpp_power_service> --> <lpp_power_wake>...  Power on AP.
 *
 * <lpp_power_service>  ~= ACPI S3
 *      LPP is NOT in WFI, and depending on configuration, is somewhere between S1 <= S3 <= WFI in capabilities.
 *      LPP exits S3 in 2 paths:
 *      <lpp_power_service> --> <lpp_power_wfi>
 *      <lpp_power_service> --> <lpp_power_wake>
 *
 * <lpp_power_wfi>      ~= ACPI S4
 *      LPP in WFI.  Usually PLL(s) in bypass, DDR in self refresh.... but this is configurable..
 *      exit wfi instruction and automatic transition from <lpp_power_wfi> --> <lpp_power_service>
 *
 */

/**********************  ITEMS ONLY for LPP OS BUILD *********************************************/
#ifdef HAVE_LPP_OS
#include "lpp_config_api.h"

void _lpp_assert(const char *, const char *, int);
void _lpp_xassert(const char *, const char *, int, unsigned long int);

// for now assertions will be applied in both debug and release builds
#define ASSERT(e) ((e) ? (void)0 : _lpp_assert(#e, __FILE__, __LINE__))
#define XASSERT(e,errvalue) ((e) ? (void)0 : _lpp_xassert(#e, __FILE__,__LINE__,(errvalue)))
#endif
/********************  END ITEMS ONLY for LPP OS BUILD *******************************************/

#define LOWPOWER_COMMAND_DATECODE        0x20150903  // every definition change of commands should roll this define.

/**
 * definition of sleep levels
 */
typedef enum
{
    e_lpp_sleep_level_wake          = 0,    // equivalent to pwr_mgr_on_e
    e_lpp_sleep_level_light         = 1,    // equivalent to pwr_mgr_reduced_power_e, still DRAM execution.
    e_lpp_sleep_level_deep          = 2,    // equivalent to pwr_mgr_lowest_power_e, transfer execution into LPP module (SRAM)
    e_lpp_sleep_level_suspend_1     = 0x10, // custom implementation of pwr_mgr_reduced_power_e.  Objective to sit in WFI, get out of DRAM. (Suspend)
    e_lpp_sleep_level_suspend_2     = 0x11, // custom implementation of pwr_mgr_reduced_power_e.  suspend threadx, ints disabled, tight loop, watch A53 wfi = ddr sr + bypass core pll
    e_lpp_sleep_level_deep_rdy      = 0x80, // ready response to e_lpp_sleep_level_deep - sync - AP=suspend to ram completed; LPP=suspend to LCM complete
    e_lpp_sleep_level_suspend_rdy   = 0x81, // ready response to e_lpp_sleep_level_suspend - in TBD ms.
    e_lpp_sleep_level_suspend1_auto = 0x91, // ready response to e_lpp_sleep_level_suspend - system is configured in default in SUSPEND_1 AUTO mode.
    e_lpp_sleep_level_suspend2_auto = 0x92, // ready response to e_lpp_sleep_level_suspend - system is configured in default in SUSPEND_2 AUTO mode.
                                            //   Only IPC interrupt is left enabled, waiting for 'e_lpp_sleep_level_wake'.
} asic_low_power_sleep_level_t;
/**
 * definition of low power commands
 */
#define  LPP_OEM_CMD_START          0x80
typedef enum
{
// 32 system cmd
    e_lpp_sys_initialize = 0,            // 0x00 CMD: u32 = datecode 'LPP_COMMAND_DATECODE'
    e_lpp_sys_sleep_level,               // 0x01 CMD: u32 = asic_low_power_sleep_level_t
    e_lpp_sys_current_ticks,             // 0x02 CMD: u32 = current system ticks
    e_lpp_sys_alarm_ticks,               // 0x03 CMD: u32 = wakeup alarm in X ticks
    e_lpp_sys_alarm_ticks_rsp,           // 0x04 CMD: u32 = RESPONSE if wakeup was alarm
    e_lpp_sys_min_sleep_ticks,           // 0x05 CMD: u32 = debug - minimum ticks in low power, helps form duty cycle
    e_lpp_sys_sleep_ticks_rsp,           // 0x06 CMD: u32 = RESPONSE # of ticks in previous sleep...
    e_lpp_sys_mode_flags,                // 0x07 CMD: u32 = operation/mode flags for low power
    e_lpp_sys_mode_flags_rsp,            // 0x08 CMD: u32 = RESPONSE flags from low power
    e_lpp_sys_irq_x      ,               // 0x09 CMD: u32 = irq # wake
    e_lpp_sys_irq_x_rsp,                 // 0x0A CMD: u32 = RESPONSE irq # TRIGGERED wake
    e_lpp_sys_debug_uart_num,            // 0x0B CMD: u32 = debug uart to use; 0xFF is NO UART for LPP ;(
    e_lpp_sys_power_service,             // 0x0C CMD: u32 = power configuration for mode 'service'  (DDR, PLL, ...)
    e_lpp_sys_power_wfi,                 // 0x0D CMD: u32 = power configuration for mode 'wfi'      (DDR, PLL, ...)
    e_lpp_sys_power_wake,                // 0x0E CMD: u32 = power configuration for mode 'wake'     (DDR, PLL, ...)
    e_lpp_sys_power_global,              // 0x0F CMD: u32 = power configuration for global LP mode.  Misc knobs for LPP
    e_lpp_sys_usec_rsp,                  // 0x10 CMD: u32 = # of usec terminal count rollovers
    e_lpp_sys_ddr_sr_idle_clks,          // 0x11 CMD: u32 = # of ddr clks to auto enter ddr sr, when enabled
    e_lpp_sys_ddr_mc_flags,              // 0x12 CMD: u32 = ddr / mc power saving options
    e_lpp_sys_avs,                       // 0x13 CMD: u32 = avs configuration as defined in 'lpp_power_avs_t'
    e_lpp_sys_cmd_list1,                 // 0x14 CMD: u32 = bit masks of commands to auto populate from list1, 32 possible commands
    e_lpp_sys_cmd_list2,                 // 0x15 CMD: u32 = bit masks of commands to auto populate from list2, 32 possible commands
    e_lpp_sys_power_cmd_builder,         // 0x16 CMD: u32 = customize/modify and existing power command: wfi, wake, service
    e_lpp_sys_measure_idle_time,         // 0x17 CMD: u32 = lpp to start timer for X seconds and report total time idle
    e_lpp_sys_prereq_idle_time,          // 0x18 CMD: u32 = 2 settings: 0xAAAA_BBBB, where AAAA is useconds idle prior to polled sleep level 1; BBBB is useconds idle prior to level 2
                                         //
    e_lpp_sys_poll_reg = 0x1B,           // 0x1B CMD: u32 = define a specific register to poll
    e_lpp_sys_poll_mask,                 // 0x1C CMD: u32 = define the bits in the specific register to poll
    e_lpp_sys_poll_val,                  // 0x1D CMD: u32 = define the target value of a specific register to poll
    e_lpp_sys_power_debug,               // 0x1E CMD: u32 = power configuration for debug LP mode.  Misc knobs for LPP
    e_lpp_sys_test_cmd_lock,             // 0x1F CMD: u32 = lock existing commands, rerun test.
// 4 usbd 2.0 cmd
    e_lpp_usb2d_endpoint_naki = 0x20,    // 0x20 CMD: u32 = end points bit mask:  tx31:16; rx15:0
    e_lpp_usb2d_rsp,                     // 0x21 CMD: u32 = RESPONSE - USBSTS at time of wake...
    e_lpp_usb2d_intr,                    // 0x22 CMD: u32 = override the INTR register, default - not needed.
    e_lpp_usb2d_4_TBD,                   // 0x23 CMD:
// 4 usbd 3.0 cmd
    e_lpp_usb3d_endpoint_naki = 0x24,    // 0x24 CMD: u32 = end points bit mask:  tx31:16; rx15:0
    e_lpp_usb3d_rsp,                     // 0x25 CMD: u32 = RESPONSE - USBSTS at time of wake...
    e_lpp_usb3d_intr,                    // 0x26 CMD: u32 = override the INTR register, default - not needed.
    e_lpp_usb3d_4_TBD,                   // 0x27 CMD:
// 4 eth cmd
    e_lpp_eth_memory_buffer = 0x30,      // 0x30 CMD: u32 = shared memory pointer
    e_lpp_eth_2_TBD,                     // 0x31 CMD:
    e_lpp_eth_3_TBD,                     // 0x32 CMD:
    e_lpp_eth_4_TBD,                     // 0x33 CMD:
// 2 gpio X cmd                          //
    e_lpp_gpio_pin_x = 0x40,             // 0x40 CMD: u32 = bank/pin; see lpp_gpio_config_t for definition
    e_lpp_gpio_pin_x_rsp,                // 0x41 CMD: u32 = RESPONSE bank/pin that triggered wake
//                                            // 0xXX CMD: u32 =
    e_lpp_oem_cmd_start = LPP_OEM_CMD_START,  // 0xXX CMD: TBD
//  oem custom commands.....
//  defined in oem/<oem name>/<proj name>/lowpower/config/lpp_config.h  -- asic_low_power_oem_cmd_t
//
    e_lpp_sys_cmd_skip   = 0xFD,        // just skip this command
    e_lpp_sys_cmd_divide = 0xFE,        // divides incoming commands from later response commands
    e_lpp_sys_cmd_last = 0x800000FF,
} asic_low_power_cmd_t;

/**
 * definition of low power command table
 */
typedef struct
{
  asic_low_power_cmd_t  cmd;
  uint32_t              value;
} asic_low_power_table_t;

#define ASIC_LOW_POWER_MAX_CMDS     (LPP_MAILBOX_SIZE / sizeof(uint32_t))

/**
 * lpp_gpio_config bit flags
 * GPIO bank, pin, ack bitmask: xxxxxxxx xxxxxxxx 000abbbb 00pppppp; allows 64 pins/bank, 16 banks.
 */
typedef struct
{
    uint32_t    pin:6;              // b5:0   AP asking LPP wake on PIN #
    uint32_t    reserved6:2;        // b7:6
    uint32_t    bank:4;             // b11:8  AP asking LPP wake on the pin in BANK
    uint32_t    ack:1;              // b12    AP asking LPP to ACK the bank:pin on interrupt
    uint32_t    reserved13:3;       // b15:13
    uint32_t    local_data:4;       // b19:16 used for local state data
    uint32_t    reserved20:4;       // b23:20
    uint32_t    data:8;             // b31:24
} lpp_gpio_config_t;

/**
 * lpp_usbd_config_t bit flags
 */
typedef struct
{
    uint32_t    rx16:16;            // b15:0   RX EP 15:0 to wake on; set NAKI
    uint32_t    tx16:16;            // b31:16  TX EP 15:0;to wake on; set NAKI
} lpp_usbd_config_t;

/**
 * lpp_power_config bit flags
 */
typedef struct
{
    uint32_t    core0_pd:1;         // b0     AP asking LPP to power down core0
    uint32_t    core0_down:1;       // b1     LPP has PD core0, internal written
    uint32_t    reserved2:6;        // b7:2   AP asking LPP
    uint32_t    ddr_sr:1;           // b8     AP asking LPP to put DDR into self refresh
    uint32_t    ddr_pll_bp:1;       // b9     AP asking LPP when in self refresh to put DDR PLL into bypass
    uint32_t    ddr_pll_pd:1;       // b10    AP asking LPP when DDR PLL in bypass to power down PLL
    uint32_t    ddr_asr:1;          // b11    AP asking LPP to put DDR into auto self refresh..
    uint32_t    ddr_dfc:2;          // b13:12 AP asking LPP to put DDR into DDR25: 0 No change, 1 DDR3 25, 2 DDR4 25, 3 exit DDR25
    uint32_t    ddr_halt_sched:1;   // b14    AP asking LPP to halt MC scheduler prior to DDR SR, monitor MC status to exit
    uint32_t    ddr_mc_pad:1;       // b15    AP asking LPP
    uint32_t    core_pll_bp:1;      // b16    AP asking LPP to put Core PLL into bypass
    uint32_t    core_pll_pd:1;      // b17    AP asking LPP when Core PLL is in bypass, to power down
    uint32_t    reserved18:2;       // b19:18 AP asking LPP
    uint32_t    clk_div:2;          // b21:20 AP asking LPP to change clock div
    uint32_t    avs:1;              // b22    AP asking LPP to change AVS control and/or VddLimit registers, as defined in cmd 'e_lpp_sys_avs' and 'lpp_power_avs_t'
    uint32_t    sys_clk_gate:1;     // b23    AP asking LPP to gate SYS CLK
    uint32_t    sys_pll_pd:1;       // b24    AP asking LPP to gate+bp+pd SYS CLK
    uint32_t    reserved25:7;       // b31:25 AP asking LPP
} lpp_power_control_t;

/**
 * lpp_power_command_builder_t - option to build/modify existing
 * lpp_power_table_t: .config_wfi, .config_srvc or .config_wake
 */
typedef struct
{
    uint32_t    block:8;            // b7:0   bits to set in command
    uint32_t    mask:8;             // b15:8  bits to mask off of existing bits
    uint32_t    shift:8;            // b23:16 shift to place 'block' into command
    uint32_t    cmd:4;              // b27:24 bitmask for which command(s): bit 0 = wake; bit 1 service; bit 2 wfi; bit 3 global
    uint32_t    reserved28:4;       // b31:28
} lpp_power_command_builder_t;

/**
 * lpp_power_global bit flags
 */
typedef struct
{                                   // default is usually all 0s
    uint32_t    cold_boot:1;        // b0  AP asking LPP to cold boot on wake
    uint32_t    reserved1:1;        // b1  not defined yet
    uint32_t    sync_to_ap:1;       // b2  AP asking LPP to send sync on deep sleep ready
    uint32_t    skip_sync_ap:1;     // b3  AP asking LPP NOT TO WAIT for AP to send sync on deep sleep ready (suspend ram complete) - debug
    uint32_t    core123_pd:1;       // b4  AP asking LPP to power off core 1,2,3 prior to core 0 is powered off, default-assumption is core 0 will turn off core 1-3.
    uint32_t    gpu_pd:1;           // b5  AP asking LPP to power off GPU after core 0 is powered off, default-assumption is core 0 will turn off GPU.
    uint32_t    skip_l2_pd:1;       // b6  AP asking LPP to NOT power off/on L2 cache when core 0 is powered off/on, leave it on(as is)
    uint32_t    skip_complex_pd:1;  // b7  AP asking LPP to NOT power off/on core complex (other) when core 0 is powered off/on, leave it on(as is)
    uint32_t    max_div_apbusclk:1; // b8  AP asking LPP set max div in clk gen -- impacts our (apmu) usec timer..
    uint32_t    max_div_ipsbusclk:1;// b9  AP asking LPP set max div in clk gen
    uint32_t    skip_gpio_driver:1; // b10 AP asking LPP to not load gpio driver, it might want gpio left as is...
    uint32_t    skip_wake_intr:1;   // b11 AP asking LPP to not exit LP mode on interrupt, but like to poll interrupt to wake AP
    uint32_t    reserved12:4;       // b15:12
    uint32_t    wfi_poll_ap_wfi:1;  // b16 AP asking LPP to poll bits instead of WFI. Poll AP cpu WFI bits
    uint32_t    wfi_poll_mc_idle:1; // b17 AP asking LPP to poll bits instead of WFI. Poll MC Idle
    uint32_t    wfi_poll_custom_down:1;// b18 AP asking LPP to poll register when AP PD
    uint32_t    wfi_poll_custom_up:1;  // b19 AP asking LPP to poll register when AP PU
    uint32_t    wfi_poll_optimize:1;// b20 AP asking LPP to optimize transition from WFI POLLING to SERVICE. It bypasses normal sequence.  Use with caution.
    uint32_t    reserved21:11;      // b31:21
} lpp_power_global_t;

/**
 * lpp_power_debug bit flags - used to test / debug
 */
typedef struct
{                                   // default is usually all 0s
    uint32_t    sys_pll_bp:1;       // b0  AP asking LPP to put SYS PLL into bypass
    uint32_t    sys_pll_pd:1;       // b1  AP asking LPP to power down, TBD if to PU on wake...
    uint32_t    scan_pll_bp:1;      // b2  AP asking LPP to put SCAN PLL into bypass
    uint32_t    scan_pll_pd:1;      // b3  AP asking LPP to power down, TBD if to PU on wake...
    uint32_t    lvds_pll_bp:1;      // b4  AP asking LPP to put LVDS (both) PLL into bypass
    uint32_t    lvds_pll_pd:1;      // b5  AP asking LPP to power down, TBD if to PU on wake...
    uint32_t    reserved6:2;        // b7:6
    uint32_t    lvds_vid_pd:1;      // b8  AP asking LPP to power down, TBD
    uint32_t    lvds_scif0_pd:1;    // b9  AP asking LPP to power down, TBD
    uint32_t    lvds_scif1_pd:1;    // b10 AP asking LPP to power down, TBD
    uint32_t    lvds_lcd_pd:1;      // b11 AP asking LPP to power down, TBD
    uint32_t    gpio_vsense_pd:1;   // b12 AP asking LPP to power down, TBD
    uint32_t    lcd_clk_gate:1;     // b13 AP asking LPP to clock gate, TBD
    uint32_t    phy_pd:1;           // b14 AP asking LPP to power down PHYs (USB, PCIe, SATA)
    uint32_t    reserved15:1;       // b15
    uint32_t    periodic_status:1;  // b16 display status every minute
    uint32_t    reserved17:1;       // b17
    uint32_t    reserved18:1;       // b18
    uint32_t    reserved19:1;       // b19
    uint32_t    reserved20:1;       // b20
    uint32_t    reserved21:1;       // b21
    uint32_t    pll_skip_lock:1;    // b22:1 debug to poll PLL LOCK
    uint32_t    pll_poll_lock:1;    // b23:1 debug to poll PLL LOCK
    uint32_t    pll_skip_50us:1;    // b24:1 debug to skip PLL 50us LOCK delay
    uint32_t    poll_skip_wake_tmr:1;// b25 Polling mode, don't wake on AP timer
    uint32_t    poll_skip_resume:1; // b26 Polling mode, keep AP powered down (measure static power)
    uint32_t    poll_trace:1;       // b27 Polling trace
    uint32_t    poll_chk_memory:1;  // b28 Polling will checksum memory on exit
    uint32_t    poll_heartbeat:1;   // b29 Polling will exit on heartbeat
    uint32_t    wfi_log:1;          // b30 R4 is only doing logging in WFI state
    uint32_t    a53_force_wfi:1;    // b31 a debug a53 test app is running - baremetal...
} lpp_power_debug_t;

/**
 * lpp_power_avs configuration --
 */
typedef struct
{
    uint32_t    vdd_limit:6;        // b5:0  directly written to AP::MPMU::AVS::VddLimit [high_vdd_limit and low_vdd_limit]  0-51 is valid..
    uint32_t    reserved6:2;        // b7:6  unused - help keep command easy to read limit value
    uint32_t    set_vdd_limit:1;    // b8    apply vdd_limit to AP::MPMU::AVS::VddLimit [high_vdd_limit and low_vdd_limit]  0-51 is valid..
    uint32_t    deassert_enable:1;  // b9    deassert AP::MPMU::AVS::Control:enable --- '1: AVS is on.....'
    uint32_t    assert_enable:1;    // b10   assert   AP::MPMU::AVS::Control:enable --- '1: AVS is on.....'
    uint32_t    deassert_rst_b:1;   // b11   deassert AP::MPMU::AVS::Control:rst_b --- 'This active-low reset...'
    uint32_t    assert_rst_b:1;     // b12   assert   AP::MPMU::AVS::Control:rst_b --- 'This active-low reset...'
    uint32_t    reserved13:19;      // b31:13 unused
} lpp_power_avs_t;

/**
 * lpp_power_mc configuration --
 */
typedef struct
{
    uint32_t    ddr_clk:1;          // b0    AP asking LPP ddr clk
    uint32_t    reserved2:3;        // b3:1  AP asking LPP
    uint32_t    phy14_cke:1;        // b4    AP asking LPP MC_CH0_PHY_CONTROL_14_PHY_CKE_PWRDWN_REPLACE_VAL
    uint32_t    phy14_nrst:1;       // b5    AP asking LPP MC_CH0_PHY_CONTROL_14_PHY_RESETN_PWRDWN_REPLACE_VAL
    uint32_t    phy5_ac_pd:1;       // b6    AP asking LPP MC_CH0_PHY_CONTROL_5_MC_*_PD_REPLACE_VAL
    uint32_t    phy5_ck_pd:1;       // b7    AP asking LPP MC_CH0_PHY_CONTROL_5_MC_*_PD_REPLACE_VAL
    uint32_t    phy5_dq_pd:1;       // b8    AP asking LPP MC_CH0_PHY_CONTROL_5_MC_*_PD_REPLACE_VAL
    uint32_t    phy5_qsp_pd:1;      // b9    AP asking LPP MC_CH0_PHY_CONTROL_5_MC_*_PD_REPLACE_VAL
    uint32_t    phy5_qsn_pd:1;      // b10   AP asking LPP MC_CH0_PHY_CONTROL_5_MC_*_PD_REPLACE_VAL
    uint32_t    phy15_pu:1;         // b11   AP asking LPP MC_CH0_PHY_CONTROL_15_PHY_PU_REPLACE_VAL -- VDAC power up.
    uint32_t    phy17_vreg:1;       // b12   AP asking LPP MC_CH0_PHY_CONTROL_17_VREG_PU_REPLACE_VAL
    uint32_t    phy15_pd:1;         // b13   AP asking LPP MC_CH0_PHY_CONTROL_15_PHY_REG_PD_MASK_SHIFT -- Charge-Pump power down.
    uint32_t    reserved14:27;      // b31:14 unused
} lpp_power_mc_t;

/**
 * lpp_power_idle_t configuration of PREREQ usecond period of
 * IDLE prior to engaging controls
 */
typedef struct
{
    uint32_t    level2:16;          // b15:0  AP asking LPP to measure 'level2' useconds of idle PRIOR
    uint32_t    level1:16;          // b31:16 AP asking LPP to measure 'level1' useconds of idle PRIOR (probably 0)
} lpp_power_idle_t;

/**
 * Wake modes:
 */
typedef enum
{
    LPP_WAKE_MODE_UNKNOWN=0,
    LPP_WAKE_MODE_NORMAL,       // full/immediate wakeup is requested
    LPP_WAKE_MODE_IO,           // quick wake / resleep is requested --> AKA IO wake, wake to handle IO, and resleep
    LPP_WAKE_MODE_OTHER
} lpp_wake_mode_t;

/**
 * definition of cmds sent to each driver..
 */
typedef enum
{
    lpp_driver_start,
    lpp_driver_stop,
    lpp_driver_release,
    lpp_driver_abort,
} lpp_driver_cmd_t;

/**
 * definition of cmds sent to power controls driver..The
 * implementation/definition can vary per ASIC..
 */
typedef enum
{                                    // The following levels are just a STATE, the definition is ACTUALLY determined dynamically in a schema
    lpp_driver_power_wake,           // WAKE:    ~= ACPI S1:   AP is powered, but some power controls enabled, control is being transitioned AP to/from LPP
    lpp_driver_power_service,        // SERVICE: ~= ACPI S2/3: AP is (usually) NOT powered, DRAM usually in self refresh, LPP is exe from SRAM
    lpp_driver_power_wfi,            // WFI:     ~= ACPI S4:   AP is (usually) NOT powered, DRAM usually in self refresh, PLL bypass, LPP in WFI
    lpp_driver_power_wfi_poll,       // WFI:     ~= ACPI S4:   AP is (usually) NOT powered, DRAM usually in self refresh, PLL bypass, LPP polling for activity
    lpp_driver_power_unknown         // unknown..
} lpp_power_level_t;

#define LPP_LOG_ID_STRS  "on","cON","cBP","sON","dON","SR","nSR","pad","Halt","nHalt","Idle","nIdle","wTmr","wIntr","WFI","nWFI","level","lWAKE","lSRV","lWFI","uart","beat","PollS","PollX","Susp0","Resm0","cstup","cstdn","cstif","t1","t2","t3","t4","t5","t6"
typedef enum
{
    e_lpp_log_on             = 0,    //
    e_lpp_log_core_pll_on,
    e_lpp_log_core_pll_bp,
    e_lpp_log_sys_pll_on,
    e_lpp_log_ddr_pll_on,
    e_lpp_log_ddr_sr,
    e_lpp_log_ddr_nsr,
    e_lpp_log_mc_pad,
    e_lpp_log_mc_halt,
    e_lpp_log_mc_nhalt,
    e_lpp_log_poll_idle,
    e_lpp_log_mc_nidle,
    e_lpp_log_timer_wake,
    e_lpp_log_intr_wake,
    e_lpp_log_ap_wfi,
    e_lpp_log_ap_nwfi,
    e_lpp_log_power_level,
    e_lpp_log_wake_ready,
    e_lpp_log_service_ready,
    e_lpp_log_wfi_ready,
    e_lpp_log_uart_char,
    e_lpp_log_heartbeat,
    e_lpp_log_poll_start,
    e_lpp_log_poll_exit,
    e_lpp_log_poll_suspend,
    e_lpp_log_poll_resume,
    e_lpp_log_reg_up,
    e_lpp_log_reg_down,
    e_lpp_log_reg_info,
    e_lpp_log_tst1,
    e_lpp_log_tst2,
    e_lpp_log_tst3,
    e_lpp_log_tst4,
    e_lpp_log_tst5,
    e_lpp_log_tst6,
} lpp_log_id_t;
#ifndef LOW_POWER_EXCLUDE_API
#include "error_types.h"
/**
 * definition of driver table
 */
typedef error_type_t (*LPP_PFN_DRV)( lpp_driver_cmd_t cmd);
typedef struct lpp_driver_table_s
{
    LPP_PFN_DRV driver;
    char        *name;
} lpp_driver_table_t;

// MPMU IRQ is shared for 6 interrupts. AP::MPMU::MPMU_misc::IntEnReg 0xD0627000; IntPendReg 04; IntAckReg 08
#define INTNUM_MPMU_IRQ_RTC_Alarm2      (INTNUM_MPMU_IRQ | 0x200000)        // 0x200025
#define INTNUM_MPMU_IRQ_RTC_Alarm1      (INTNUM_MPMU_IRQ | 0x100000)        // 0x100025
#define INTNUM_MPMU_IRQ_Timer3          (INTNUM_MPMU_IRQ | 0x080000)        // 0x080025
#define INTNUM_MPMU_IRQ_Timer2          (INTNUM_MPMU_IRQ | 0x040000)        // 0x040025
#define INTNUM_MPMU_IRQ_Timer1          (INTNUM_MPMU_IRQ | 0x020000)        // 0x020025
#define INTNUM_MPMU_IRQ_Timer0          (INTNUM_MPMU_IRQ | 0x010000)        // 0x010025

/**
 * Defines for: transition from AP to LPP commands, and current
 * status of LPP
 */
#define LPP_FLAGS_SLEEPING      (1<<0)       // default - Tell LPP enter sleep loop
#define LPP_FLAGS_WFI_SKIP      (1<<1)       // debug - Tell LPP skip using WFI instruction
#define LPP_FLAGS_WAKE_WFI_SKIP (1<<2)       // debug - Tell LPP to immediately wake when we skip WFI - allows time transition measure doing full power controls

#define LPP_FLAGS_LOG_ON        (1<<4)       // debug - Tell LPP to log events for timestamp

//#define LPP_FLAGS_DEBUG_QUIET   (1<<8)       // debug - Tell LPP control serial output (on/off)
#define LPP_FLAGS_DEBUG_SOFT    (1<<9)       // debug - Tell LPP control serial output (on/off)
#define LPP_FLAGS_DEBUG_LOUD    (1<<10)      // debug - Tell LPP control serial output (on/off)
#define LPP_FLAGS_LPP_MARKER    (1<<11)      // Tell LPP when we enter & exit LPP, display marker 'lp><'
                                             //
#define LPP_FLAGS_SLEEP_DELAY   (1<<12)      // debug - Tell LPP delay on sleep entry
#define LPP_FLAGS_WAKE_DELAY    (1<<13)      // debug - Tell LPP delay on sleep exit
#define LPP_FLAGS_EXIT_DELAY    (1<<14)      // debug - Tell LPP delay on sleep exit

#define LPP_FLAGS_DEBUG_PAUSE   (1<<16)      // debug - Tell LPP to pause between power controls, wait on <sp>
#define LPP_FLAGS_SLEEP_PAUSE   (1<<17)      // debug - Tell LPP to pause on entry, allows for testing...
#define LPP_FLAGS_EXIT_PAUSE    (1<<18)      // debug - Tell LPP to pause on wake, allows for testing...
#define LPP_FLAGS_MEM_CHECK     (1<<19)      // debug - Tell LPP to test memory @ sleep & exit, allows for testing...

#define LPP_FLAGS_DEBUG_INWAKE  (1<<20)      // debug - in unrecognized debug UART input will wake
#define LPP_FLAGS_DEBUG_SKIP_ER (1<<21)      // debug - 'debug_space_active' be continual.  allows for mode to skip over spin/error loops

#define LPP_FLAGS_DEBUG_TRACE_D (1<<28)      // debug - Tell LPP display main.c bread crumbs
#define LPP_FLAGS_DEBUG_TRACE_C (1<<29)      // debug - Tell LPP display power control bread crumbs
#define LPP_FLAGS_DEBUG_TRACE_B (1<<30)      // debug - Tell LPP display power control control function trace
#define LPP_FLAGS_DEBUG_TRACE_A (1<<31)      // debug - Tell LPP display power control mode function trace

/**
 * Defines for dispatching tasks
 */
#define LPP_TASK_SCHEDULE_NEVER     0xFFFFFFFF
#define LPP_TASK_SCHEDULE_RELOAD    0xFFFFFFFE
#define LPP_TASK_SCHEDULE_IMMEDIATE 0

typedef uint32_t (*PFN_TASK)( void *user_data );

/**
 * lpp_get_ticks - return current tick (10ms) count
 *
 * @return uint32_t
 */
uint32_t lpp_get_ticks( void );

/**
 * lpp_wakeup_ticks - when a driver requests a full system wakeup
 * @param wakeup_in_ticks
 */
void lpp_set_wakeup_ticks(uint32_t wakeup_in_ticks);
/**
 * lpp_set_min_sleep_ticks - usually in debug to form duty cycle
 * to measure power averages...
 * @param wakeup_in_ticks
 */
void lpp_set_min_sleep_ticks(uint32_t min_sleep_ticks);

/**
 * lpp_get_wakeup_usec - return the active alarm value
 * @return uint32_t - active alarm value
 */
uint64_t lpp_get_wakeup_usec( void );
/**
 * lpp_get_min_sleep_ticks - return the active min sleep value
 * @return uint32_t - active min sleep value
 */
uint32_t lpp_get_min_sleep_ticks( void );

/**
 * \brief Register function (tasklet) for processing.  Commonly
 *        used to process the 'bottom half' of an ISR outside of
 *        interrupt context. A device may also register a timed
 *        function for periodic polling or blinking on LED?.
 *
 * @param useconds - when to schedule in future.
 *  LPP_TASK_SCHEDULE_IMMEDIATE - for execution immediately
 *  LPP_TASK_SCHEDULE_NEVER     - to remove task from list
 *  other                   - 1 - 0xFFFF FFF0 = # of ticks
 * @param handler - user callback
 * @param [in] user_data - Opaque user context returned to the user in the callback.
 *
 * @return bool - false if list is full, failed to add
 */
bool lpp_task_attach(uint32_t useconds, PFN_TASK handler, void *user_data);

/**
 * \brief called remove scheduled task
 *
 */
bool lpp_task_detach(PFN_TASK handler);

/**
 * \brief called periodically to handle registered tasks
 *
 */
void lpp_task_dispatch();

/**
 * lpp_wfi_skip - check restrictions to enter WFI
 *
 * @author mikee (3/1/2012)
 */
bool lpp_wfi_skip();

/**
 * lpp_wakeup_system - being asked to issue 'mode' wake up from
 * 'name'
 *
 * @author mikee (3/4/2013)
 *
 * @param mode - type of wakeup
 * @param name - source of request - debug only
 */
void lpp_wakeup_system( lpp_wake_mode_t mode, char *name );

/**
 * lpp_get_wakeup - return reason of wakeup
 *
 * @author mikee (9/25/2013)
 *
 * @return lpp_wake_mode_t
 */
lpp_wake_mode_t lpp_get_wakeup( void );

/**
 * lpp_set_mailbox_cmd
 *
 * @author mikee (12/2/2013)
 *
 * @param command
 * @param value
 */
void lpp_set_mailbox_cmd(asic_low_power_cmd_t command, uint32_t value);
/**
 * lpp_get_mailbox_cmd
 *
 * @author mikee (12/2/2013)
 *
 * @param command
 * @param value
 *
 * @return error_type_t
 */
error_type_t lpp_get_mailbox_cmd(asic_low_power_cmd_t command, uint32_t *value);
/**
 * lpp_print_mailbox_cmd - print all mailbox cmds
 *
 * @author mikee (12/3/2013)
 */
void lpp_print_mailbox_cmd();
/**
 * lpp_get_mailbox_mcmd - special implementation to find
 * multiple entries of same cmd..
 */
error_type_t lpp_get_mailbox_mcmd(bool first, uint32_t cmd, uint32_t *value);


/**
 * Manipulation of LPP system flags
 */
void lpp_system_clr_flag(uint32_t flag);
void lpp_system_set_flag(uint32_t flag);
uint32_t lpp_system_get_flag(uint32_t flag);

/**
 * \brief busy wait for a specified number of microseconds.
 * Delay is done by checking the soft 64b usec counter.
 * \param delayInUs[in] The delay required in u secs.
 */
void lpp_delay_us(uint32_t delayInUs);

/**
 * \brief get current usec count from running usec timer.
 *
 * @return uint32_t current usecond count
 */
uint32_t lpp_get_usec( void );

/**
 * \brief get current u64 usec count + usec, from running usec
 *        timer.
 *
 * @return uint32_t current usecond count + usec
 */
uint64_t lpp_get_usec64( uint32_t usec );

/**
 * \brief get usec uptime of low power module.
 *
 * @return uint64_t useconds in low power module
 */
uint64_t lpp_get_up_time( void );

typedef void (*lpp_gpio_isr_t)( uint32_t bank, uint32_t pin,  void *user_data);
/**
 * lpp_gpio_isr_attach -
 */
error_type_t lpp_gpio_isr_attach( uint32_t bank, uint32_t pin, lpp_gpio_isr_t isr_callback, void *user_data );
/**
 * lpp_gpio_isr_detach -
 */
error_type_t lpp_gpio_isr_detach( uint32_t bank, uint32_t pin);

/**
 * lpp_log - timestamp an event
 */
void lpp_log(lpp_log_id_t id, uint32_t value);
/**
 * lpp_log_print - print event log
 */
void lpp_log_print();

/**
 * lpp_service_watchdog
 *
 * @author mikee (3/16/2015)
 *
 * @param user_data -- unused. common API of tasket
 *
 * @return uint32_t -1 is NOT enabled;-2 if enabled
 */
uint32_t lpp_service_watchdog( void *user_data );
/**
 * lpp_trigger_watchdog - platform reset
 *
 * @author mikee (3/16/2015)
 */
void lpp_trigger_watchdog(void);

#endif   // LOW_POWER_EXCLUDE_API
#endif   // #ifndef INC_LPP_API_H

