/*
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
*/
/*
Low Power Idle - modprobe low_power_idle
Terms:
    R4 = LPP (low power processor) = SP (secure processor)
    A53 = AP (application processor)

Mode    Name Description

wake    e_lpp_level_wake
        Manual LPP, will also exit/wake from, any other mode.

        If in any LP Mode, a WAKE command will exit/wake up from respective mode.
        While in this mode, LP commands can manually be sent to LPP to define a custom power schema.
        For example:
            Display existing queue of commands:
            lp_cmd.sh

            Manually recreate Mode 1: (SUSPEND 1)
            lp_cmd.sh 0x1=0x10           --- enters suspend_1
            lp_cmd.sh 0x1=0x00           --- exit low power mode 1


light   e_lpp_level_light   -- when: lp_flags.sh set lp_light_enable 0    --- LP_AP_LIGHT_DEFAULT
        R4 Suspend_1
        Driver will instruct R4 Processor to suspend (WFI), waiting for wakeup command.

        lp_mode.sh light    --- enters e_lpp_level_light
        lp_mode.sh wake     --- exit low power mode

light   e_lpp_level_light   -- when: lp_flags.sh set lp_light_enable 1    --- LP_AP_LIGHT_DEFAULT
        ** only a test mode, this has limited abilities..
        Driver will execute e_lpp_level_light as well as implement its own CPUIDLE extention C-State.
        When system is idle, the DDR will be placed in auto self-refresh, and then the primary core enters WFI.
        When the primary core exits WFI, the power controls are restored.
        The non primary cores would already be off.

        The DDR is configured for auto exit.

        lp_mode.sh light    --- enters e_lpp_level_light
        lp_mode.sh wake     --- exit low power mode

        The level of power controls are defined in:
            lpi.lpp[e_lpp_level_light].wfi
            lpi.lpp[e_lpp_level_light].asr_clocks


        lp_flags.sh show light_sleep_flags
        lp_flags.sh show light_asr_flags

        To change the default 'sr_idle_clks' setting, use:

        lp_flags.sh set light_asr_flags 0x00000200

        default: LP_AP_AUTO_SR_IDLE_CLKS        0x00000100

        To change the default 'light_sleep_flags' setting, use:
        lp_flags.sh set light_sleep_flags 0x????????

        #define LP_AP_FLAGS_DEFAULT

deep    e_lpp_level_deep
        R4 enters DEEP sleep (LP module) -- needs about 96KB of SRAM
        Driver will instruct R4 Processor to monitor A53 CPUs and Memory Controller
        While IDLE (A53s are in WFI and MC is Idle), the R4 will
            put DDR into Halt self refresh and bypass DDR PLL.
            powerdown DDR pads
            gate sys clk
        The level of power controls are defined in:
            lpi.lpp[e_lpp_level_deep].*
        When NOT IDLE, controls are restored.
        R4 will exit this mode upon receiving wakeup command.

        lp_mode.sh deep     --- enters e_lpp_level_deep
        lp_mode.sh wake     --- exit low power mode

        lp_flags.sh show deep_sleep_flags
        lp_flags.sh show deep_mc_flags

        To change the default 'deep_sleep_flags' setting, use:

        lp_flags.sh set deep_sleep_flags 0x????????

hibernate  e_lpp_level_hibernate
        R4 enters DEEP sleep (LP module) -- needs about 96KB of SRAM
        Driver will instruct R4 Processor to monitor system resourcs (interrupts, gpio, ...)
        Communication I/O (network, USB, ..) are expected to be off.
        While in sleep, the R4 will
            put DDR into self refresh and bypass DDR PLL.
            powerdown DDR pads
            gate sys clk
            bypass Core PLL
            wfi
            restore...
        The level of power controls are defined in:
            lpi.lpp[e_lpp_level_hibernate].*

        R4 will exit this mode upon receiving wakeup command.

        lp_mode.sh hibernate --- enters e_lpp_level_hibernate
        lp_mode.sh wake      --- exit low power mode

softoff e_lpp_level_softoff
        R4 enters DEEP sleep (LP module) -- needs about 96KB of SRAM
        Driver will instruct R4 Processor to monitor system resourcs (interrupts, gpio, ...)
        Communication I/O (network, USB, ..) are expected to be off.
        While in sleep, the R4 will
            put DDR into self refresh (or reset) and bypass and powerdown DDR PLL.
            powerdown DDR pads
            gate sys clk
            bypass and powerdown Core PLL
            wfi
            restore...
        The level of power controls are defined in:
            lpi.lpp[e_lpp_level_softoff].*

        R4 will exit this mode upon receiving wakeup command and cold boot SoC

        lp_mode.sh softoff   --- enters e_lpp_level_softoff
        lp_mode.sh wake      --- exit low power mode

*/

#define DEBUG       0
//#define DEBUG       1
//#define EXPERIMENT  1

#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/mm.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/ioctl.h>
#include <linux/dma-mapping.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/slab.h>
#include <linux/firmware.h>
#include <linux/delay.h>
#include <linux/cpu_pm.h>
#include <asm/system_misc.h>

#define MY_DRIVER_NAME                  "low_power_idle"
#define LOW_POWER_IDLE_BUILD            0x20150904
#define LPI_PRINTK                      if (lpi.debug)printk
#define LPI_DEBUG_WR_MC_PC0_REG(x)      if (DEBUG)writel((x),MC_PC0_REG);
#define LPI_DEBUG_WR_MC_PC0_REG_INC()   if (DEBUG)writel((readl(MC_PC0_REG)+1),MC_PC0_REG);
#define LPI_DEBUG_WR_MC_PC1_REG(x)      if (DEBUG)writel((x),MC_PC1_REG);
#define LPI_DEBUG_WR_MC_PC2_REG(x)      if (DEBUG)writel((x),MC_PC2_REG);

#define LOW_POWER_EXCLUDE_API
#include "lpp_api.h"
#include "low_power_idle.h"
#include "low_power_idle_defaults.h"
#include "ipc_api.h"
#include "ipc_well_known_ports.h"
#define IPC0_DEVICE             0

MODULE_LICENSE("Dual BSD/GPL");
#if 0
extern int         (*pegmatite_sleep)(int c_state, int wake_us);
extern unsigned int *pegmatite_cstate_3_residency;
#endif
extern void pegmatite_cpu_suspend(void);
extern void gic_cpu_if_down(void);


unsigned char open_count;
struct miscdevice miscdev;

static ipc_drvr_handle  port_handle;
static volatile unsigned int ipc_response_cmd;
static volatile unsigned int ipc_response_param;

static lp_control_t lpi = {
    .mode                               = e_lpp_level_wake,
    .mode_name[e_lpp_level_wake]        = LPI_MODE_LPP_WAKE_STR,
    .mode_name[e_lpp_level_light]       = LPI_MODE_LPP_LIGHT_STR,
    .mode_name[e_lpp_level_deep]        = LPI_MODE_LPP_DEEP_STR,
    .mode_name[e_lpp_level_suspend]     = LPI_MODE_LPP_SUSPEND_STR,
    .mode_name[e_lpp_level_hibernate]   = LPI_MODE_LPP_HIBERNATE_STR,
    .mode_name[e_lpp_level_softoff]     = LPI_MODE_LPP_SOFTOFF_STR,
    .mode_name[e_lpp_level_rtos]        = LPI_MODE_LPP_RTOS1_STR,

    .sleep_name[e_lpp_level_wake]       = LPI_MODE_LPP_WAKE_SX,
    .sleep_name[e_lpp_level_light]      = NULL,
    .sleep_name[e_lpp_level_deep]       = LPI_MODE_LPP_DEEP_SX,
    .sleep_name[e_lpp_level_suspend]    = LPI_MODE_LPP_SUSPEND_SX,
    .sleep_name[e_lpp_level_hibernate]  = LPI_MODE_LPP_HIBERNATE_SX,
    .sleep_name[e_lpp_level_softoff]    = LPI_MODE_LPP_SOFTOFF_SX,
    .sleep_name[e_lpp_level_rtos]       = NULL,

    .debug                              = DEBUG,
    .light_lpi_enable                   = LP_AP_LIGHT_DEFAULT,

    .lpp[e_lpp_level_light].wfi         = LP_AP_FLAGS_DEFAULT,
    .lpp[e_lpp_level_light].asr_clocks  = LP_AP_AUTO_SR_IDLE_CLKS,
    .lpp[e_lpp_level_light].mc          = LP_LPP_FLAGS_HIBERNATE_MC,

    .lpp[e_lpp_level_deep].avs          = LP_LPP_FLAGS_AVS_DEFAULT,
    .lpp[e_lpp_level_deep].global       = LP_LPP_FLAGS_DEEP_GLOBAL,
    .lpp[e_lpp_level_deep].mc           = LP_LPP_FLAGS_DEEP_MC,
    .lpp[e_lpp_level_deep].service      = LP_LPP_FLAGS_DEEP_SERVICE,
    .lpp[e_lpp_level_deep].wake         = 0,
    .lpp[e_lpp_level_deep].wfi          = LP_LPP_FLAGS_DEEP_SLEEP,
    .lpp[e_lpp_level_deep].debug        = LP_LPP_FLAGS_DEEP_DBG,

    .lpp[e_lpp_level_suspend].avs       = LP_LPP_FLAGS_AVS_DEFAULT,
    .lpp[e_lpp_level_suspend].global    = LP_LPP_FLAGS_SUSPEND_GLOBAL,
    .lpp[e_lpp_level_suspend].mc        = LP_LPP_FLAGS_SUSPEND_MC,
    .lpp[e_lpp_level_suspend].service   = LP_LPP_FLAGS_SUSPEND_SERVICE,
    .lpp[e_lpp_level_suspend].wake      = 0,
    .lpp[e_lpp_level_suspend].wfi       = LP_LPP_FLAGS_SUSPEND_SLEEP,
    .lpp[e_lpp_level_suspend].debug     = LP_LPP_FLAGS_SUSPEND_DBG,

    .lpp[e_lpp_level_hibernate].avs     = LP_LPP_FLAGS_AVS_DEFAULT,
    .lpp[e_lpp_level_hibernate].global  = LP_LPP_FLAGS_HIBERNATE_GLOBAL,
    .lpp[e_lpp_level_hibernate].mc      = LP_LPP_FLAGS_HIBERNATE_MC,
    .lpp[e_lpp_level_hibernate].service = LP_LPP_FLAGS_HIBERNATE_SERVICE,
    .lpp[e_lpp_level_hibernate].wake    = 0,
    .lpp[e_lpp_level_hibernate].wfi     = LP_LPP_FLAGS_HIBERNATE_SLEEP,
    .lpp[e_lpp_level_hibernate].debug   = LP_LPP_FLAGS_HIBERNATE_DBG,

    .lpp[e_lpp_level_softoff].avs       = LP_LPP_FLAGS_AVS_DEFAULT,
    .lpp[e_lpp_level_softoff].global    = LP_LPP_FLAGS_SOFTOFF_GLOBAL,
    .lpp[e_lpp_level_softoff].mc        = LP_LPP_FLAGS_SOFTOFF_MC,
    .lpp[e_lpp_level_softoff].service   = LP_LPP_FLAGS_SOFTOFF_SERVICE,
    .lpp[e_lpp_level_softoff].wake      = 0,
    .lpp[e_lpp_level_softoff].wfi       = LP_LPP_FLAGS_SOFTOFF_SLEEP,
    .lpp[e_lpp_level_softoff].debug     = LP_LPP_FLAGS_SOFTOFF_DBG,
};
static void         *restore_sleep_pfn = NULL;



static unsigned int low_power_idle_gated=0;     // we set this when the PM runtime system tells us 'not sleep'
static unsigned int low_power_idle_gated_counter=0;
static unsigned int low_power_idle_smp_active;
static unsigned int low_power_idle_counter=0;
static unsigned int low_power_idle_c_states[10];
//static unsigned int low_power_idle_APCPUClkStatus=0;

// define LP command queue
typedef struct ipc_cmd_s {
    int     cmd;
    int     value;
} ipc_cmd_t;

struct {
    ipc_cmd_t cmd[10];
    int     index;
} lp_cmd;

#define HISTORY_SIZE 128
struct {
    int     c_state;
    int     predicted_sleep_length;
    unsigned int     wfi_time;
    unsigned int     wake_time;
} lp_history[HISTORY_SIZE];
int lp_history_index = 0;

static ktime_t ktime_previous;

//static void __iomem *apmu_base = NULL;
static void __iomem *mpmu_base = NULL;
static void __iomem *mc_base = NULL;
static void __iomem *ciu_base = NULL;

static struct of_device_id of_mpmu_table[] = {
	{.compatible = "marvell,pegmatite-mpmu"},
	{ /* end of list */ },
};
static struct of_device_id of_mc_table[] = {
	{.compatible = "marvell,mckinley5"},
	{ /* end of list */ },
};
static const struct of_device_id of_ciu_table[] = {
	{.compatible = "marvell,pegmatite-ciu"},
	{ /* end of list */ },
};

static void send_low_power_cmd(uint8_t command, void *buffer, uint16_t length, bool wait, uint8_t wait_cmd);
//volatile u32 halting = 1;

static void map_cpuidle_fn(void *fn)
{
#if 0
    /*
     * redirect the cpuidle SR handler
     */
    if (pegmatite_sleep != NULL) {
        printk("pegmatite_sleep already hooked! Replacing..\n");
    }
    restore_sleep_pfn = pegmatite_sleep;
    pegmatite_sleep = fn;
#endif
}
static void unmap_cpuidle_fn(void)
{
#if 0
    /*
     * restore the cpuidle SR handler
     */
    pegmatite_sleep = restore_sleep_pfn;
    restore_sleep_pfn = NULL;
#endif
}

static void cpu_suspend(void)
{
    static u32 suspend_count = 0;
    u32 timer_enable=0;
    u32 timer_disable=0;

    suspend_count++;

    LPI_DEBUG_WR_MC_PC2_REG(suspend_count);

    /* Suspend */
    local_fiq_disable();
    local_irq_disable();

    cpu_pm_enter();

    cpu_cluster_pm_enter();

    gic_cpu_if_down(); // If I don't do this, I will get CPU not in WFI error when using IPS timers..

    //halting = 1;
    //while (halting);
    //halting = 1;
    // we can read sys count peek 0xd40fc000 16 4; peek 0xd40fd000 16 4
    // hack from arch_timer_reg_read_cp15
    asm volatile("mrc p15, 0, %0, c14, c2, 1" : "=r" (timer_enable));
    // hack disable from arch_timer_reg_write_cp15
    asm volatile("mcr p15, 0, %0, c14, c2, 1" : : "r" (timer_disable));

    LPI_DEBUG_WR_MC_PC0_REG(0xDEAD0000);
    LPI_DEBUG_WR_MC_PC1_REG(0xDEAD);
    pegmatite_cpu_suspend();
    LPI_DEBUG_WR_MC_PC1_REG(01);
    LPI_DEBUG_WR_MC_PC0_REG_INC();
    outer_resume();
    LPI_DEBUG_WR_MC_PC0_REG_INC();

    //while (halting);

    // now restore values: CNTPCT, CNTP_CVAL  (certainly could be a better way to sync new time..)
    //crash set_timer(0,cnt); //asm volatile("mcrr p15, 0, %0, %1, c14" :: "r" (ct_l), "r" (ct_h));
    //set_timer(2,cmp); //asm volatile("mcrr p15, 2, %0, %1, c14" :: "r" (cv_l), "r" (cv_h));
    //asm volatile("mcr p15, 0, %0, c14, c2" :: "r" (timer_enable));
    asm volatile("mcr p15, 0, R0, c14, c2");

    // hack from arch_timer_reg_write_cp15
    asm volatile("mcr p15, 0, %0, c14, c2, 1" :: "r" (timer_enable));
    LPI_DEBUG_WR_MC_PC0_REG_INC();

    /* Resume */
    cpu_cluster_pm_exit();
    LPI_DEBUG_WR_MC_PC0_REG_INC();

    cpu_pm_exit();
    LPI_DEBUG_WR_MC_PC0_REG_INC();

    local_irq_enable();
    LPI_DEBUG_WR_MC_PC0_REG_INC();
    local_fiq_enable();
    LPI_DEBUG_WR_MC_PC0_REG_INC();
    LPI_DEBUG_WR_MC_PC1_REG(0);
}
/**
 *  lp_sleep - cpu_idle extension - self refresh handler.
 *
 * @author mikee (5/1/2015)
 */
static int lp_sleep(int c_state, int wake_us)
{
    unsigned int core=0, sys=0;
    unsigned int auto_sr_val=0;
    unsigned int my_flags = 0;
    ktime_t ktime_now;
    int sleep_time_us = 0;

    // if LPP is REQUIRING us to not enter S2 SUSPEND, at most take S1.
    if ((c_state == 3) && readl(CIU_CPU_CSTATE_NOT_3))
    {
        c_state = 2;
    }
    low_power_idle_c_states[c_state]++;

    // store CSTATE for LPP
    writel(c_state, CIU_CPU_CSTATE);
    // store next known timer for LPP to wake us on.
    // the governor will pick CSTATE from PREDICTED sleep, which is likely less than known next timer.
    writel(wake_us, CIU_CPU_TIMER);

    switch (lpi.mode) {
    case e_lpp_level_deep:
    case e_lpp_level_suspend:
    case e_lpp_level_rtos:
        // using LPP for SR and other power controls, all we do is WFI..
        low_power_idle_counter++;
        if (lpi.debug) {
            lp_history[lp_history_index].c_state = c_state;
            lp_history[lp_history_index].predicted_sleep_length = sleep_time_us;
            ktime_now = ktime_get();
            lp_history[lp_history_index].wake_time = ktime_us_delta(ktime_now,ktime_previous);
        }

        if ((c_state == 3) && (lpi.lpp[lpi.mode].wfi & LP_FLAGS_CORE0_PD))
        {
            // our target sleep is big enough, and we are willing to suspend cpu0
            cpu_suspend();
        }
        else
        {
            asm volatile ("wfi");
        }

        if (lpi.debug) {
            ktime_previous = ktime_get();
            lp_history[lp_history_index].wfi_time = (unsigned int)ktime_us_delta(ktime_previous,ktime_now);
            lp_history_index = (lp_history_index+1) & (HISTORY_SIZE-1);
            writel(lp_history[lp_history_index].wfi_time, CIU_CPU_SLEPT_TIME);
        }
        /* clear current CSTATE for LPP */
        writel(0, CIU_CPU_CSTATE);
        writel(c_state, CIU_CPU_CSTATE_B);
        return 0;
    case e_lpp_level_light:
        break;
    case e_lpp_level_softoff:
        // we will need to do more than this, but this is a start...
        while (1)
        {
            asm volatile ("wfi");
        }
    default:
        asm volatile ("wfi");
        return 0;
    }

    if (low_power_idle_gated)
    {
        low_power_idle_gated_counter++;
        asm volatile ("wfi");
        return 0;
    }

	// safety net. note, just because CPUs are reported offline doesn't
	// mean they aren't accessing DRAM (as in the case where they just went
	// offline)
	if( num_online_cpus()>1 )
	{
		if( !low_power_idle_smp_active )
			printk(KERN_ERR "Error: SMP active while low_power_idle_enable = %d!\n",num_online_cpus());
		low_power_idle_smp_active++;
		asm volatile ("wfi");
                LPI_DEBUG_WR_MC_PC0_REG(0xF003);
        return 0;
	}

    low_power_idle_counter++;
    /*
     * my_flags could be all 0, in which case, all we do is WFI.
     */

    my_flags = lpi.lpp[e_lpp_level_light].wfi;
    if (my_flags & LP_FLAGS_DDR_AUTO_SR) {
        // store existing value, usually 0
        // since we are letting MC decide when IDLE, we don't care about spurious DDR access, it will go idle when ready.
        auto_sr_val = readl(MC_AUTO_SR_REG);
        writel(lpi.lpp[e_lpp_level_light].asr_clocks,MC_AUTO_SR_REG);
    }
    else if (my_flags & LP_FLAGS_DDR_SR) {
        // Write SR to the AP::MC::USER_COMMAND_0, this has an automatic exit if a source trys an access
        // We are relying on our cache to keep the remaining instructions from immediately bumping us out of SR
        writel(USER_COMMAND_0_SR_ENTER,MC_SR_REG);
    }

    // AP::MPMU::MPMU_misc::SYSPLL_Enable -- gate Sys PLL clk
    if (my_flags & LP_FLAGS_SYS_CLK_GATE) {
        sys = readl(SYS_CLK_GATE_REG);
        writel((sys & ~CLK_EN),SYS_CLK_GATE_REG);
    }

    // Read the AP::MPMU::CORE_PLL::fixed_mode_ssc_mode
    if (my_flags & LP_FLAGS_CORE_PLL_BP) {
        core = readl(CORE_PLL_EN_REG);
        writel((core | FIXED_MODE_SSC_MODE_BP),CORE_PLL_EN_REG);
    }

    // we are finished with our power down sequence, so WFI
    asm volatile ("wfi");

    // return to full speed (non bypass)
    if (my_flags & LP_FLAGS_CORE_PLL_BP) {
        writel(core,CORE_PLL_EN_REG);
    }

    // restore sys pll en
    if (my_flags & LP_FLAGS_SYS_CLK_GATE) {
        writel(sys,SYS_CLK_GATE_REG);
    }

    // disable to auto SR timer, exit SR (it should have already done AUTO exit)
    if (my_flags & LP_FLAGS_DDR_AUTO_SR) {
        writel(auto_sr_val,MC_AUTO_SR_REG);
        writel(USER_COMMAND_0_SR_EXIT,MC_SR_REG);
    }
    else if (my_flags & LP_FLAGS_DDR_SR) {
        writel(USER_COMMAND_0_SR_EXIT,MC_SR_REG);
    }
    LPI_DEBUG_WR_MC_PC0_REG(0);
    return 0;
}

/**
 * send_low_power_cmd - send IPC command, and optionally wait for
 * response.
 *
 * @author mikee (5/14/2015)
 *
 * @param command
 * @param buffer
 * @param length
 * @param wait
 * @param wait_cmd
 */
static void send_low_power_cmd(uint8_t command, void *buffer, uint16_t length, bool wait, uint8_t wait_cmd)
{
#if 0  /*  RICOH del  */
    ipc_response_cmd = 0xffff;
    ipc_send( port_handle, command, buffer, length );
    if (wait) {
        while (ipc_response_cmd != wait_cmd) {
            if (wait) LPI_PRINTK("waiting...");
            wait = false;
        }
        if (!wait) LPI_PRINTK("\n\nResponse command received: %d....:\n",wait_cmd);
    }
#endif
}

/**
 * send_low_power_cmd_q - send all queued commands
 *
 * @author mikee (5/14/2015)
 */
static void send_low_power_cmd_q(bool *sync)
{
    int index=0;
    if (sync) *sync = false;
    // send any manually queued commands
    for (index = 0;index < lp_cmd.index;++index) {
        LPI_PRINTK("LPP: cmd %#x = %#x\n", lp_cmd.cmd[index].cmd, lp_cmd.cmd[index].value);
        send_low_power_cmd(lp_cmd.cmd[index].cmd, (void *)lp_cmd.cmd[index].value,0,false,0);
        if (lp_cmd.cmd[index].cmd == e_lpp_sys_power_global) {
            // determine is deep sleep has a sync response to wait for...
            if (lp_cmd.cmd[index].value & 4) {
                // sync_to_ap
                if (sync) *sync = true;
            }
            else
            {
                // allow for multiple command, keep the last..
                if (sync) *sync = false;
            }
        }
    }
    lp_cmd.index = 0;
}

/**
 * show_low_power_cmds - display our list of queued command to
 * be sent to LPP. Used in e_lpp_level_wake
 *
 * @author mikee (3/11/2015)
 *
 * @param dev
 * @param attr
 * @param buf
 *
 * @return ssize_t
 */
static ssize_t show_low_power_cmds(struct device *dev, struct device_attribute *attr, char *buf)
{
    int index;

    printk("%s: Cmd table pending contains:\n", __func__);
    for (index = 0;index < lp_cmd.index;++index) {
        printk("       + cmd %#x = %#x\n", lp_cmd.cmd[index].cmd, lp_cmd.cmd[index].value);
    }
    return 0;
}
/**
 * store_low_power_cmds - add new LPP power command to queue.
 * Used in e_lpp_level_wake
 *
 * @author mikee (3/11/2015)
 *
 * @param dev
 * @param attr
 * @param buf
 * @param count
 *
 * @return ssize_t
 */
static ssize_t store_low_power_cmds(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int cmd, value, index;
    bool wait = true;
    bool send_q_cmd = true;

    sscanf(buf, "%x=%x", &cmd, &value);

    if (lpi.mode) {
        LPI_PRINTK("%s: Warning.  Currently in Low Power Mode: %s\n", __func__, lpi.mode_name[lpi.mode]);
        //return count;
    }

    // if cmd is 'e_lpp_sys_sleep_level', this is trigger.
    if (cmd == e_lpp_sys_sleep_level)
    {
        if (value == e_lpp_sleep_level_deep_rdy) {
            // e_lpp_sleep_level_deep_rdy - AP sync to LP telling it to proceed with power off
            wait = false;
            send_q_cmd = true; //false;
        }
        if (value == 255) {
            // undefined - just send blank/tickle
            wait = false;
            send_q_cmd = false;
        }
        if (send_q_cmd) {
            // send commands from our storage
            send_low_power_cmd_q(&wait);
        }

        LPI_PRINTK("LPP: cmd %#x = %#x\n", cmd, value);
        if (value == 0x102) {
            // test/debug -- send immediate 'sync to lp' = deep ready
            send_low_power_cmd(e_lpp_sys_sleep_level, (void *)e_lpp_sleep_level_deep, 0, wait, 1);
            send_low_power_cmd(e_lpp_sys_sleep_level, (void *)e_lpp_sleep_level_deep_rdy, 0, false, 1);
        }
        else
        {
            send_low_power_cmd(e_lpp_sys_sleep_level, (void *)value, 0, wait, 1);
        }
        lp_cmd.index = 0;
    }
    else if (cmd == 255)
    {
        // just a hack to clear pending table..
        LPI_PRINTK("%s: Flushing all commands\n", __func__);
        lp_cmd.index = 0;
    }
    else
    {
        LPI_PRINTK("%s: Add cmd to pending table\n", __func__);
        lp_cmd.cmd[lp_cmd.index].cmd = cmd;
        lp_cmd.cmd[lp_cmd.index].value = value;
        lp_cmd.index++;
        LPI_PRINTK("%s: Cmd table pending contains:\n", __func__);
        for (index = 0;index < lp_cmd.index;++index) {
            LPI_PRINTK("       + cmd %#x = %#x\n", lp_cmd.cmd[index].cmd, lp_cmd.cmd[index].value);
        }
    }

    return count;
}
static void print_sleep_history(void)
{
    int i=0;
    unsigned int time;
    if (lpi.debug)
    {
        i=lp_history_index;
        time=0;
        do
        {
            printk("%03d: cstate: %d;  target sleep %08d us; wfi %08d; wake %08d; time %08d\n", i, lp_history[i].c_state,lp_history[i].predicted_sleep_length,lp_history[i].wfi_time,lp_history[i].wake_time,time);
            time += (lp_history[i].wake_time+lp_history[i].wfi_time);
            i = (i+1) & (HISTORY_SIZE-1);
        } while (i != lp_history_index);
    }
}
static void print_command_sleep_flags(void)
{
    printk("state WFI bit flags are:\n");
    printk("LP_FLAGS_CORE0_PD             = %#08x\n", LP_FLAGS_CORE0_PD);
    printk("LP_FLAGS_DDR_SR               = %#08x\n", LP_FLAGS_DDR_SR);
    printk("LP_FLAGS_DDR_PLL_BP           = %#08x\n", LP_FLAGS_DDR_PLL_BP);
    printk("LP_FLAGS_DDR_AUTO_SR          = %#08x\n", LP_FLAGS_DDR_AUTO_SR);
    printk("LP_FLAGS_DDR_HALT_MC          = %#08x\n", LP_FLAGS_DDR_HALT_MC);
    printk("LP_FLAGS_DDR_MC_PAD           = %#08x\n", LP_FLAGS_DDR_MC_PAD);
    printk("LP_FLAGS_CORE_PLL_BP          = %#08x\n", LP_FLAGS_CORE_PLL_BP);
    printk("LP_FLAGS_AVS_CONTROL          = %#08x\n", LP_FLAGS_AVS_CONTROL);
    printk("LP_FLAGS_SYS_CLK_GATE         = %#08x\n", LP_FLAGS_SYS_CLK_GATE);
    printk("low_power_idle_counter        = %d\n", low_power_idle_counter);
    printk("low_power_idle_c_states       = %d:%d:%d:%d:%d:\n\n\n", low_power_idle_c_states[0], low_power_idle_c_states[1], low_power_idle_c_states[2], low_power_idle_c_states[3], low_power_idle_c_states[4]);
    memset(low_power_idle_c_states,0,sizeof(low_power_idle_c_states));
    print_sleep_history();
}
static ssize_t show_light_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("light_sleep_flags = %#08x\n", lpi.lpp[e_lpp_level_light].wfi);
    printk("light_asr_flags   = %#08x\n", lpi.lpp[e_lpp_level_light].asr_clocks);
    print_command_sleep_flags();
    printk("low_power_idle_smp_active     = %d\n", low_power_idle_smp_active);
    low_power_idle_smp_active = 0;
    //printk("low_power_idle_APCPUClkStatus = %#x\n", low_power_idle_APCPUClkStatus);

    return 0;
}
static ssize_t store_light_sleep_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_light].wfi);
    printk("light_sleep_flags = %#x\n", lpi.lpp[e_lpp_level_light].wfi);
    return count;
}
static ssize_t show_light_asr_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("light_asr_flags   = %#08x\n", lpi.lpp[e_lpp_level_light].asr_clocks);
    return 0;
}
static ssize_t store_light_asr_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_light].asr_clocks);
    printk("light_asr_flags   = %#08x\n", lpi.lpp[e_lpp_level_light].asr_clocks);
    return count;
}

/*
 * Deep Sleep
 */
static void handle_new_cmd(const char *buf, unsigned int *flag, asic_low_power_cmd_t cmd)
{
    sscanf(buf, "%x", flag);
    if (lpi.mode==e_lpp_level_deep) {
        send_low_power_cmd(cmd,(void*)*flag,0,false,0);
        send_low_power_cmd(e_lpp_sys_sleep_level,(void*)e_lpp_sleep_level_deep_rdy,0,false,0);
    }
    return;
}
static ssize_t show_deep_wake_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].wake         = %#08x\n", lpi.lpp[e_lpp_level_deep].wake);
    return 0;
}
static ssize_t store_deep_wake_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].wake, e_lpp_sys_power_wake);
    printk("lpp[e_lpp_level_deep].wake = %#x\n", lpi.lpp[e_lpp_level_deep].wake);
    return count;
}
static ssize_t show_deep_service_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].service      = %#08x\n", lpi.lpp[e_lpp_level_deep].service);
    return 0;
}
static ssize_t store_deep_service_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].service, e_lpp_sys_power_service);
    printk("lpp[e_lpp_level_deep].service = %#x\n", lpi.lpp[e_lpp_level_deep].service);
    return count;
}
static ssize_t show_deep_global_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].global       = %#08x\n", lpi.lpp[e_lpp_level_deep].global);
    return 0;
}
static ssize_t store_deep_global_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].global, e_lpp_sys_power_global);
    printk("lpp[e_lpp_level_deep].global = %#x\n", lpi.lpp[e_lpp_level_deep].global);
    return count;
}
static ssize_t show_deep_mc_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].mc           = %#08x\n", lpi.lpp[e_lpp_level_deep].mc);
    return 0;
}
static ssize_t store_deep_mc_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].mc, e_lpp_sys_ddr_mc_flags);
    printk("lpp[e_lpp_level_deep].mc = %#x\n", lpi.lpp[e_lpp_level_deep].mc);
    return count;
}
static ssize_t show_deep_avs_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].avs          = %#08x\n", lpi.lpp[e_lpp_level_deep].avs);
    return 0;
}
static ssize_t store_deep_avs_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].avs, e_lpp_sys_avs);
    printk("lpp[e_lpp_level_deep].avs = %#x\n", lpi.lpp[e_lpp_level_deep].avs);
    return count;
}
static ssize_t show_deep_dbg_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].debug        = %#08x\n", lpi.lpp[e_lpp_level_deep].debug);
    return 0;
}
static ssize_t store_deep_dbg_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].debug, e_lpp_sys_power_debug);
    printk("lpp[e_lpp_level_deep].debug = %#x\n", lpi.lpp[e_lpp_level_deep].debug);
    return count;
}
static ssize_t display_deep_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_deep].wfi          = %#08x\n", lpi.lpp[e_lpp_level_deep].wfi);
    show_deep_wake_flags(dev,attr,buf);
    show_deep_service_flags(dev,attr,buf);
    show_deep_global_flags(dev,attr,buf);
    show_deep_mc_flags(dev,attr,buf);
    show_deep_avs_flags(dev,attr,buf);
    show_deep_dbg_flags(dev,attr,buf);
    return 0;
}
static ssize_t show_deep_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    display_deep_sleep_flags(dev,attr,buf);
    print_command_sleep_flags();
    //printk("low_power_idle_APCPUClkStatus = %#x\n", low_power_idle_APCPUClkStatus);
    return 0;
}
static ssize_t store_deep_sleep_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_deep].wfi, e_lpp_sys_power_wfi);
    printk("lpp[e_lpp_level_deep].wfi = %#x\n", lpi.lpp[e_lpp_level_deep].wfi);
    return count;
}
/*
 * Deep SUSPEND Sleep
 */
static ssize_t show_suspend_wake_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].wake      = %#08x\n", lpi.lpp[e_lpp_level_suspend].wake);
    return 0;
}
static ssize_t store_suspend_wake_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].wake, e_lpp_sys_power_wake);
    printk("lpp[e_lpp_level_suspend].wake = %#x\n", lpi.lpp[e_lpp_level_suspend].wake);
    return count;
}
static ssize_t show_suspend_service_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].service   = %#08x\n", lpi.lpp[e_lpp_level_suspend].service);
    return 0;
}
static ssize_t store_suspend_service_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].service, e_lpp_sys_power_service);
    printk("lpp[e_lpp_level_suspend].service = %#x\n", lpi.lpp[e_lpp_level_suspend].service);
    return count;
}
static ssize_t show_suspend_global_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].global    = %#08x\n", lpi.lpp[e_lpp_level_suspend].global);
    return 0;
}
static ssize_t store_suspend_global_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].global, e_lpp_sys_power_global);
    printk("lpp[e_lpp_level_suspend].global = %#x\n", lpi.lpp[e_lpp_level_suspend].global);
    return count;
}
static ssize_t show_suspend_mc_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].mc        = %#08x\n", lpi.lpp[e_lpp_level_suspend].mc);
    return 0;
}
static ssize_t store_suspend_mc_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].mc, e_lpp_sys_ddr_mc_flags);
    printk("lpp[e_lpp_level_suspend].mc = %#x\n", lpi.lpp[e_lpp_level_suspend].mc);
    return count;
}
static ssize_t show_suspend_avs_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].avs       = %#08x\n", lpi.lpp[e_lpp_level_suspend].avs);
    return 0;
}
static ssize_t store_suspend_avs_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].avs, e_lpp_sys_avs);
    printk("lpp[e_lpp_level_suspend].avs = %#x\n", lpi.lpp[e_lpp_level_suspend].avs);
    return count;
}
static ssize_t show_suspend_dbg_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].debug     = %#08x\n", lpi.lpp[e_lpp_level_suspend].debug);
    return 0;
}
static ssize_t store_suspend_dbg_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].debug, e_lpp_sys_power_debug);
    printk("lpp[e_lpp_level_suspend].debug = %#x\n", lpi.lpp[e_lpp_level_suspend].debug);
    return count;
}
static ssize_t show_suspend_cstate3(struct device *dev, struct device_attribute *attr, char *buf)
{
#if 0
    if (pegmatite_cstate_3_residency)
    {
        printk("      lpp[e_lpp_level_suspend].cstate3     = %d useconds\n", *pegmatite_cstate_3_residency);
    }
#endif
    return 0;
}
static ssize_t store_suspend_cstate3(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
#if 0
    if (pegmatite_cstate_3_residency)
    {
        sscanf(buf, "%d", pegmatite_cstate_3_residency);
        printk("lpp[e_lpp_level_suspend].cstate3 = %d useconds\n", *pegmatite_cstate_3_residency);
    }
#endif
    return count;
}
static ssize_t display_suspend_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_suspend].wfi       = %#08x\n", lpi.lpp[e_lpp_level_suspend].wfi);
    show_suspend_wake_flags(dev,attr,buf);
    show_suspend_service_flags(dev,attr,buf);
    show_suspend_global_flags(dev,attr,buf);
    show_suspend_mc_flags(dev,attr,buf);
    show_suspend_avs_flags(dev,attr,buf);
    show_suspend_dbg_flags(dev,attr,buf);
    return 0;
}
static ssize_t show_suspend_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    display_suspend_sleep_flags(dev,attr,buf);
    print_command_sleep_flags();
    //printk("low_power_idle_APCPUClkStatus = %#x\n", low_power_idle_APCPUClkStatus);
    return 0;
}
static ssize_t store_suspend_sleep_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    handle_new_cmd(buf, &lpi.lpp[e_lpp_level_suspend].wfi, e_lpp_sys_power_wfi);
    printk("lpp[e_lpp_level_suspend].wfi = %#x\n", lpi.lpp[e_lpp_level_suspend].wfi);
    return count;
}
/*
 * SoftOff Sleep
 */
static ssize_t display_softoff_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].wfi       = %#08x\n", lpi.lpp[e_lpp_level_softoff].wfi    );
    printk("      lpp[e_lpp_level_softoff].wake      = %#08x\n", lpi.lpp[e_lpp_level_softoff].wake   );
    printk("      lpp[e_lpp_level_softoff].service   = %#08x\n", lpi.lpp[e_lpp_level_softoff].service);
    printk("      lpp[e_lpp_level_softoff].global    = %#08x\n", lpi.lpp[e_lpp_level_softoff].global );
    printk("      lpp[e_lpp_level_softoff].mc        = %#08x\n", lpi.lpp[e_lpp_level_softoff].mc     );
    printk("      lpp[e_lpp_level_softoff].avs       = %#08x\n", lpi.lpp[e_lpp_level_softoff].avs    );
    printk("      lpp[e_lpp_level_softoff].debug     = %#08x\n", lpi.lpp[e_lpp_level_softoff].debug  );
    return 0;
}
static ssize_t show_softoff_wake_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].wake         = %#08x\n", lpi.lpp[e_lpp_level_softoff].wake);
    return 0;
}
static ssize_t store_softoff_wake_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].wake);
    printk("lpp[e_lpp_level_softoff].wake = %#x\n", lpi.lpp[e_lpp_level_softoff].wake);
    return count;
}
static ssize_t show_softoff_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    display_softoff_flags(dev,attr,buf);
    print_command_sleep_flags();
    //printk("low_power_idle_APCPUClkStatus = %#x\n", low_power_idle_APCPUClkStatus);
    return 0;
}
static ssize_t store_softoff_sleep_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].wfi);
    printk("lpp[e_lpp_level_softoff].wfi = %#x\n", lpi.lpp[e_lpp_level_softoff].wfi);
    return count;
}
static ssize_t show_softoff_service_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].service      = %#08x\n", lpi.lpp[e_lpp_level_softoff].service);
    return 0;
}
static ssize_t store_softoff_service_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].service);
    printk("lpp[e_lpp_level_softoff].service = %#x\n", lpi.lpp[e_lpp_level_softoff].service);
    return count;
}
static ssize_t show_softoff_global_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].global       = %#08x\n", lpi.lpp[e_lpp_level_softoff].global);
    return 0;
}
static ssize_t store_softoff_global_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].global);
    printk("lpp[e_lpp_level_softoff].global = %#x\n", lpi.lpp[e_lpp_level_softoff].global);
    return count;
}
static ssize_t show_softoff_mc_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].mc           = %#08x\n", lpi.lpp[e_lpp_level_softoff].mc);
    return 0;
}
static ssize_t store_softoff_mc_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].mc);
    printk("lpp[e_lpp_level_softoff].mc = %#x\n", lpi.lpp[e_lpp_level_softoff].mc);
    return count;
}
static ssize_t show_softoff_avs_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].avs          = %#08x\n", lpi.lpp[e_lpp_level_softoff].avs);
    return 0;
}
static ssize_t store_softoff_avs_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].avs);
    printk("lpp[e_lpp_level_softoff].avs = %#x\n", lpi.lpp[e_lpp_level_softoff].avs);
    return count;
}
static ssize_t show_softoff_dbg_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_softoff].debug        = %#08x\n", lpi.lpp[e_lpp_level_softoff].debug);
    return 0;
}
static ssize_t store_softoff_dbg_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_softoff].debug);
    printk("lpp[e_lpp_level_softoff].debug = %#x\n", lpi.lpp[e_lpp_level_softoff].debug);
    return count;
}
/*
 * Hibernate Sleep
 */
static ssize_t display_hibernate_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].wfi     = %#08x\n", lpi.lpp[e_lpp_level_hibernate].wfi    );
    printk("      lpp[e_lpp_level_hibernate].wake    = %#08x\n", lpi.lpp[e_lpp_level_hibernate].wake   );
    printk("      lpp[e_lpp_level_hibernate].service = %#08x\n", lpi.lpp[e_lpp_level_hibernate].service);
    printk("      lpp[e_lpp_level_hibernate].global  = %#08x\n", lpi.lpp[e_lpp_level_hibernate].global );
    printk("      lpp[e_lpp_level_hibernate].mc      = %#08x\n", lpi.lpp[e_lpp_level_hibernate].mc     );
    printk("      lpp[e_lpp_level_hibernate].avs     = %#08x\n", lpi.lpp[e_lpp_level_hibernate].avs    );
    printk("      lpp[e_lpp_level_hibernate].debug   = %#08x\n", lpi.lpp[e_lpp_level_hibernate].debug  );
    return 0;
}
static ssize_t show_hibernate_wake_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].wake         = %#08x\n", lpi.lpp[e_lpp_level_hibernate].wake);
    return 0;
}
static ssize_t store_hibernate_wake_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].wake);
    printk("lpp[e_lpp_level_hibernate].wake = %#x\n", lpi.lpp[e_lpp_level_hibernate].wake);
    return count;
}
static ssize_t show_hibernate_sleep_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    display_hibernate_flags(dev,attr,buf);
    print_command_sleep_flags();
    //printk("low_power_idle_APCPUClkStatus = %#x\n", low_power_idle_APCPUClkStatus);
    return 0;
}
static ssize_t store_hibernate_sleep_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].wfi);
    printk("lpp[e_lpp_level_hibernate].wfi = %#x\n", lpi.lpp[e_lpp_level_hibernate].wfi);
    return count;
}
static ssize_t show_hibernate_service_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].service      = %#08x\n", lpi.lpp[e_lpp_level_hibernate].service);
    return 0;
}
static ssize_t store_hibernate_service_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].service);
    printk("lpp[e_lpp_level_hibernate].service = %#x\n", lpi.lpp[e_lpp_level_hibernate].service);
    return count;
}
static ssize_t show_hibernate_global_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].global       = %#08x\n", lpi.lpp[e_lpp_level_hibernate].global);
    return 0;
}
static ssize_t store_hibernate_global_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].global);
    printk("lpp[e_lpp_level_hibernate].global = %#x\n", lpi.lpp[e_lpp_level_hibernate].global);
    return count;
}
static ssize_t show_hibernate_mc_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].mc           = %#08x\n", lpi.lpp[e_lpp_level_hibernate].mc);
    return 0;
}
static ssize_t store_hibernate_mc_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].mc);
    printk("lpp[e_lpp_level_hibernate].mc = %#x\n", lpi.lpp[e_lpp_level_hibernate].mc);
    return count;
}
static ssize_t show_hibernate_avs_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].avs          = %#08x\n", lpi.lpp[e_lpp_level_hibernate].avs);
    return 0;
}
static ssize_t store_hibernate_avs_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].avs);
    printk("lpp[e_lpp_level_hibernate].avs = %#x\n", lpi.lpp[e_lpp_level_hibernate].avs);
    return count;
}
static ssize_t show_hibernate_dbg_flags(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("      lpp[e_lpp_level_hibernate].debug        = %#08x\n", lpi.lpp[e_lpp_level_hibernate].debug);
    return 0;
}
static ssize_t store_hibernate_dbg_flags(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.lpp[e_lpp_level_hibernate].debug);
    printk("lpp[e_lpp_level_hibernate].debug = %#x\n", lpi.lpp[e_lpp_level_hibernate].debug);
    return count;
}




static ssize_t store_low_power_debug(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.debug);
    printk("lp_debug_enable = %s\n", (lpi.debug)?"enabled":"disabled");
    return count;
}
static ssize_t show_low_power_debug(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("lp_debug_enable = %s\n", (lpi.debug)?"enabled":"disabled");
    return 0;
}
static ssize_t store_light_lpi_enable(struct device *dev, struct device_attribute *attr, char *buf, size_t count)
{
    sscanf(buf, "%x", &lpi.light_lpi_enable);
    printk("lp_light_enable = %s\n", (lpi.light_lpi_enable)?"enabled":"disabled");
    return count;
}
static ssize_t show_light_lpi_enable(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("lp_light_enable = %s\n", (lpi.light_lpi_enable)?"enabled":"disabled");
    return 0;
}

/**
 *
 */
static ssize_t show_low_power_idle(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("\n");
    if (lpi.mode < e_lpp_level_levels)
    {
        printk("Current low power mode = %s \n",lpi.mode_name[lpi.mode]);
    }
    else
    {
        printk("Current low power mode = %d : %s \n",lpi.mode, LPI_MODE_LPP_UNKNOWN_STR);
        return 0;
    }
    printk("Write to this node to enable a level of sleep\n");
    printk("   Exit any Sleep mode                 :  lp_mode.sh %s \n",LPI_MODE_LPP_WAKE_STR);
    printk("   Enter LIGHT Sleep mode              :  lp_mode.sh %s \n",LPI_MODE_LPP_LIGHT_STR);
    printk("    schema:     lp_light_enable         = %s\n",(lpi.light_lpi_enable)?"enabled":"disabled");
    if (lpi.light_lpi_enable)
    {
        printk("      lpp[e_lpp_level_light].wfi         = %#08x\n", lpi.lpp[e_lpp_level_light].wfi);
        printk("      lpp[e_lpp_level_light].asr_clocks  = %#08x\n", lpi.lpp[e_lpp_level_light].asr_clocks);
    }
    else
    {
        printk("      R4 will WFI\n");
    }
    printk("   Enter DEEP  Sleep mode              :  lp_mode.sh %s \n",LPI_MODE_LPP_DEEP_STR);
    printk("    schema:\n");
    display_deep_sleep_flags(dev,attr,buf);
    printk("   Enter SUSPEND Sleep mode            :  lp_mode.sh %s \n",LPI_MODE_LPP_SUSPEND_STR);
    printk("    schema:\n");
    display_suspend_sleep_flags(dev,attr,buf);

    printk("   Enter HIBERNATION Sleep mode        :  lp_mode.sh %s \n",LPI_MODE_LPP_HIBERNATE_STR);
    printk("    schema:\n");
    display_hibernate_flags(dev,attr,buf);

    printk("   Enter SOFT OFF Sleep mode           :  lp_mode.sh %s \n",LPI_MODE_LPP_SOFTOFF_STR);
    printk("    schema:\n");
    display_softoff_flags(dev,attr,buf);

    return 0;
};

#include <linux/regulator/consumer.h>
static int reg_imaging_enable = 2; // default on. PI is on at init, so to PD, we need to enable and disable (kick it)
static struct regulator *reg_imaging = NULL;

static ssize_t store_reg_imaging_enable(struct device *dev, struct device_attribute *attr, char *buf, size_t count)
{
    int enable;
    int ret = 0;
    sscanf(buf, "%x", &enable);
    if (!reg_imaging)
    {
        reg_imaging = regulator_get(dev, "pegmatite_imgpipe");
    }
    if (!reg_imaging)
    {
        printk("error on get reg_imagings\n");
        return count;
    }
    if (enable && !reg_imaging_enable)
    {
        ret = regulator_enable(reg_imaging);
        enable = 1;
    }
    if (!enable && reg_imaging_enable)
    {
        if (reg_imaging_enable==2)
        {
            // one time special interim case.
            ret = regulator_enable(reg_imaging);
        }
        ret = regulator_disable(reg_imaging);
    }
    if (ret) printk("regulator error %d\n", ret);

    reg_imaging_enable = enable;
    printk("reg_imaging_enable = %s\n", (reg_imaging_enable) ? "enabled" : "disabled");
    return count;
}
static ssize_t show_reg_imaging_enable(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("reg_imaging_enable = %s\n", (reg_imaging_enable)?"enabled":"disabled");
    return 0;
}

static int reg_gpu_enable = 2; // default on. PI is on at init, so to PD, we need to enable and disable (kick it)
static struct regulator *reg_gpu = NULL;
static ssize_t store_reg_gpu_enable(struct device *dev, struct device_attribute *attr, char *buf, size_t count)
{
    int enable;
    int ret = 0;
    sscanf(buf, "%x", &enable);
    if (!reg_gpu)
    {
        reg_gpu = regulator_get(dev, "pegmatite_gpu");
    }
    if (!reg_gpu)
    {
        printk("error on get reg_gpu\n");
        return count;
    }
    if (enable && !reg_gpu_enable)
    {
        ret = regulator_enable(reg_gpu);
        enable = 1;
    }
    if (!enable && reg_gpu_enable)
    {
        if (reg_gpu_enable==2)
        {
            // one time special interim case.
            ret = regulator_enable(reg_gpu);
        }
        ret = regulator_disable(reg_gpu);
    }
    if (ret) printk("regulator error %d\n", ret);

    reg_gpu_enable = enable;
    printk("reg_gpu_enable = %s\n", (reg_gpu_enable) ? "enabled" : "disabled");
    return count;
}
static ssize_t show_reg_gpu_enable(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("reg_gpu_enable = %s\n", (reg_gpu_enable)?"enabled":"disabled");
    return 0;
}

static void low_power_mode_clean_up(void)
{
    // disable all
    // unhook
    if ((lpi.mode == e_lpp_level_light) ||
        (lpi.mode == e_lpp_level_deep)   ||
        (lpi.mode == e_lpp_level_suspend)   ||
        (lpi.mode == e_lpp_level_rtos)   ||
        (lpi.mode == e_lpp_level_softoff)) {
        unmap_cpuidle_fn();
        mb();
        LPI_PRINTK("number of times gated by PM: %d\n", low_power_idle_gated_counter);
        LPI_PRINTK("low_power_idle_counter:      %u\n", low_power_idle_counter);
    }
}

static void disable_all(void)
{
    if (!lpi.mode) {
        return;
    }
    low_power_mode_clean_up();

    // send power:wake
    LPI_PRINTK("\n\nSend WAKE command\n");
    send_low_power_cmd(e_lpp_sys_sleep_level,(void*)e_lpp_sleep_level_wake,0,true,1);
    lpi.mode = e_lpp_level_wake;
    msleep(500);
    printk("\n\nAWAKE\n");
}

static void handle_light_sleep(void)
{
    int cs_count=0;
    if (lpi.light_lpi_enable == 1) {
        printk("\n\n*** %d CPUs are online. ***\n",num_online_cpus());
        // r4 suspend_1 --> wfi
        // linux hooks cpu_idle routine and manages DDR SR & ....
        if (lpi.mode) {
            disable_all();
        }

        cs_count  = (readl(MC_DRAM_STATUS_REG) & MC_DRAM_STATUS_INIT_DONE00_MASK)?1:0;
        cs_count += (readl(MC_DRAM_STATUS_REG) & MC_DRAM_STATUS_INIT_DONE01_MASK)?1:0;
        cs_count += (readl(MC_DRAM_STATUS_REG) & MC_DRAM_STATUS_INIT_DONE02_MASK)?1:0;
        // some configurations of multiple nCS DDR are not supported for MC Auto exit of SR, so we can't continue!
        if (cs_count > 1)
        {
            printk(KERN_ERR "Warning: Self Refresh ASR does not support all modes of multi_nCS DDR.\n");
            return;
        }

        // trigger mode
        send_low_power_cmd(e_lpp_sys_sleep_level,(void*)e_lpp_sleep_level_suspend_1,0,false,1);
        LPI_PRINTK("\n\nLPP suspend wfi\n");
        lpi.mode=e_lpp_level_light;

        low_power_idle_counter=0;
        low_power_idle_gated_counter=0;
        low_power_idle_smp_active=0;
        /*
         * redirect the cpuidle SR handler
         */
        map_cpuidle_fn(lp_sleep);

        printk("\n\nSLEEP MODE AP: %s\n",lpi.mode_name[e_lpp_level_light]);
        LPI_PRINTK("lp_flags.sh show light_sleep_flags = %#x\n",lpi.lpp[e_lpp_level_light].wfi);
        mb();
        return;
    }
    if (lpi.light_lpi_enable == 0) {
        // letting CPUIDLE handle SR, we only tell R4 to WFI
        printk("\n\n*** %d CPUs are online. ***\n",num_online_cpus());
        // any access to this 'store' disables the manual LPP command mode.
        lp_cmd.index = 0;
        // r4 suspend_1 --> wfi
        if (lpi.mode) {
            disable_all();
        }
        send_low_power_cmd(e_lpp_sys_sleep_level,(void*)e_lpp_sleep_level_suspend_1,0,true,1);

        printk("\n\nSLEEP MODE R4: %s\n",lpi.mode_name[e_lpp_level_light]);
        lpi.mode=e_lpp_level_light;
        return;
    }
    printk("\n\nUNKNOWN LIGHT SLEEP MODE\n");
    return;
}

static void handle_deep_sleep(void)
{
    if (num_online_cpus() > 1)
    {
        printk("\n\n*** please disable non-boot CPUs, %d CPUs are online. ***\n",num_online_cpus());
        return;
    }
    // linux hooks cpu_idle routine and trigger LPP to manage DDR SR...
    if (lpi.mode) {
        disable_all();
    }

    // AP power down NOT allowed in this mode
    lpi.lpp[e_lpp_level_deep].wfi &= ~LP_FLAGS_CORE0_PD;

    // trigger transition -- assumes R4 will sync to use when ready, and wait for our sync to it. *DEFAULT LPP will SYNC*
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep,0,true,1);
    // reset mailbox AFTER R4 sync resp
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)0xFFFFFFFF,0,false,1);
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)LOWPOWER_COMMAND_DATECODE,0,false,1);
    send_low_power_cmd(e_lpp_sys_power_service, (void*)lpi.lpp[e_lpp_level_deep].service,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_wfi,     (void*)lpi.lpp[e_lpp_level_deep].wfi,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_global,  (void*)lpi.lpp[e_lpp_level_deep].global,0,false,0);
    send_low_power_cmd(e_lpp_sys_ddr_mc_flags,  (void*)lpi.lpp[e_lpp_level_deep].mc,0,false,0);
    send_low_power_cmd(e_lpp_sys_avs,           (void*)lpi.lpp[e_lpp_level_deep].avs,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_debug,   (void*)lpi.lpp[e_lpp_level_deep].debug,0,false,0);

    // send any commands queue for options/overrides
    send_low_power_cmd_q(NULL);

    // send AP sync to activate LPP LP mode, it will now monitor system, looking to engage shutdowns..
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep_rdy,0,false,1);

    lpi.mode=e_lpp_level_deep;
    LPI_PRINTK("\n\nLPP ENTERED e_lpp_level_deep\n");

    low_power_idle_counter=0;
    low_power_idle_gated_counter=0;
    low_power_idle_smp_active=0;
    /*
     * redirect the cpuidle SR handler
     */
    map_cpuidle_fn(lp_sleep);

    printk("\n\nSLEEP MODE %s\n",lpi.mode_name[e_lpp_level_deep]);
    LPI_PRINTK("lp_flags.sh show deep_sleep_flags   = %#x\n",lpi.lpp[e_lpp_level_deep].wfi);
    LPI_PRINTK("lp_flags.sh show deep_mc_flags      = %#x\n",lpi.lpp[e_lpp_level_deep].mc);
    mb();
}

static void handle_suspend_sleep(void)
{
    if (num_online_cpus() > 1)
    {
        printk("\n\n*** please disable non-boot CPUs, %d CPUs are online. ***\n",num_online_cpus());
        return;
    }
    printk("\n\n\nSuspend Mode is still under development!!\n\n\n");
    // linux hooks cpu_idle routine and trigger LPP to manage DDR SR...
    if (lpi.mode) {
        disable_all();
    }

    // suspend mode ALWAYS requests try AP power down
    lpi.lpp[e_lpp_level_suspend].wfi |= LP_FLAGS_CORE0_PD;

    // trigger transition -- assumes R4 will sync to use when ready, and wait for our sync to it. *DEFAULT LPP will SYNC*
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep,0,true,1);
    // reset mailbox AFTER R4 sync resp
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)0xFFFFFFFF,0,false,1);
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)LOWPOWER_COMMAND_DATECODE,0,false,1);
    send_low_power_cmd(e_lpp_sys_power_service, (void*)lpi.lpp[e_lpp_level_suspend].service,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_wfi,     (void*)lpi.lpp[e_lpp_level_suspend].wfi,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_global,  (void*)lpi.lpp[e_lpp_level_suspend].global,0,false,0);
    send_low_power_cmd(e_lpp_sys_ddr_mc_flags,  (void*)lpi.lpp[e_lpp_level_suspend].mc,0,false,0);
    send_low_power_cmd(e_lpp_sys_avs,           (void*)lpi.lpp[e_lpp_level_suspend].avs,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_debug,   (void*)lpi.lpp[e_lpp_level_suspend].debug,0,false,0);

    // send set of interrupts to wake only AP, LPP would remain in mode until we tell it to wake
#ifdef LP_LPP_SUSPEND_DEFAULT_WAKE_INT_0
    LPI_PRINTK("e_lpp_sys_irq_x             %#x\n",        LP_LPP_SUSPEND_DEFAULT_WAKE_INT_0);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_SUSPEND_DEFAULT_WAKE_INT_0,0,false,0);
#endif
#ifdef LP_LPP_SUSPEND_DEFAULT_WAKE_INT_1
    LPI_PRINTK("e_lpp_sys_irq_x             %#x\n",        LP_LPP_SUSPEND_DEFAULT_WAKE_INT_1);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_SUSPEND_DEFAULT_WAKE_INT_1,0,false,0);
#endif

    // send any commands queue for options/overrides
    send_low_power_cmd_q(NULL);

    // send AP sync to activate LPP LP mode, it will now monitor system, looking to engage shutdowns..
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep_rdy,0,false,1);

    lpi.mode=e_lpp_level_suspend;
    LPI_PRINTK("\n\nLPP ENTERED e_lpp_level_suspend\n");

    low_power_idle_counter=0;
    low_power_idle_gated_counter=0;
    low_power_idle_smp_active=0;
    /*
     * redirect the cpuidle SR handler
     */
    map_cpuidle_fn(lp_sleep);

    printk("\n\nSLEEP MODE %s\n",lpi.mode_name[e_lpp_level_suspend]);
    LPI_PRINTK("lp_flags.sh show deep_sleep_flags   = %#x\n",lpi.lpp[e_lpp_level_suspend].wfi);
    LPI_PRINTK("lp_flags.sh show deep_mc_flags      = %#x\n",lpi.lpp[e_lpp_level_suspend].mc);
    mb();
}

static void handle_hibernate_sleep(void)
{
    // hibernate and suspend NEED the non-boot cpus off at this point.
    if (num_online_cpus() > 1)
    {
        printk("\n\n*** please disable non-boot CPUs, %d CPUs are online. ***\n",num_online_cpus());
        return;
    }
    // linux hooks cpu_idle routine and trigger LPP to manage DDR SR...
    if (lpi.mode) {
        disable_all();
    }

    // trigger transition -- assumes R4 will sync to use when ready, and wait for our sync to it. *DEFAULT LPP will SYNC*
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep,0,true,1);
    // reset mailbox AFTER R4 sync resp
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)0xFFFFFFFF,0,false,1);
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)LOWPOWER_COMMAND_DATECODE,0,false,1);
    send_low_power_cmd(e_lpp_sys_power_service, (void*)lpi.lpp[e_lpp_level_hibernate].service,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_wfi,     (void*)lpi.lpp[e_lpp_level_hibernate].wfi,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_global,  (void*)lpi.lpp[e_lpp_level_hibernate].global,0,false,0);
    send_low_power_cmd(e_lpp_sys_ddr_mc_flags,  (void*)lpi.lpp[e_lpp_level_hibernate].mc,0,false,0);
    send_low_power_cmd(e_lpp_sys_avs,           (void*)lpi.lpp[e_lpp_level_hibernate].avs,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_debug,   (void*)lpi.lpp[e_lpp_level_hibernate].debug,0,false,0);
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_TIMER
    printk("e_lpp_sys_alarm_ticks       %d 10ms ticks\n",LP_LPP_TEST_HIBERNATE_WAKE_TIMER);
    send_low_power_cmd(e_lpp_sys_alarm_ticks,     (void*)LP_LPP_TEST_HIBERNATE_WAKE_TIMER,0,false,0);
#endif
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_GPIO_0
    printk("e_lpp_gpio_pin_x            %#x\n",        LP_LPP_TEST_HIBERNATE_WAKE_GPIO_0);
    send_low_power_cmd(e_lpp_gpio_pin_x,        (void*)LP_LPP_TEST_HIBERNATE_WAKE_GPIO_0,0,false,0);
#endif
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_INT_0
    printk("e_lpp_sys_irq_x             %#x\n",        LP_LPP_TEST_HIBERNATE_WAKE_INT_0);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_HIBERNATE_WAKE_INT_0,0,false,0);
#endif
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_INT_1
    printk("e_lpp_sys_irq_x             %#x\n",        LP_LPP_TEST_HIBERNATE_WAKE_INT_1);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_HIBERNATE_WAKE_INT_1,0,false,0);
#endif
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_INT_2
    printk("e_lpp_sys_irq_x             %#x\n",        LP_LPP_TEST_HIBERNATE_WAKE_INT_2);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_HIBERNATE_WAKE_INT_2,0,false,0);
#endif
#ifdef LP_LPP_TEST_HIBERNATE_WAKE_INT_3
    printk("e_lpp_sys_irq_x             %#x\n",        LP_LPP_TEST_HIBERNATE_WAKE_INT_3);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_HIBERNATE_WAKE_INT_3,0,false,0);
#endif
    // send any commands queue for options/overrides
    // this would likely include the setup for specific GPIO pin(s) or IRQ#(s)
    send_low_power_cmd_q(NULL);

    lpi.mode=e_lpp_level_hibernate;
    printk("\n\nSLEEP MODE %s\n",lpi.mode_name[e_lpp_level_hibernate]);
    LPI_PRINTK("lp_flags.sh show hibernate_sleep_flags   = %#x\n",lpi.lpp[e_lpp_level_hibernate].wfi);

    // send AP sync that we are ready to be shut off  - it will wait for us to WFI
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep_rdy,0,false,1);


    ipc_response_cmd = 0xffff;

    cpu_suspend();

    lpi.mode=e_lpp_level_wake;
    // clean up if keep...
    bool wait = true;
    //ipc_response_cmd = 0xffff;
    while (ipc_response_cmd != e_lpp_sys_sleep_level) {
        if (wait) LPI_PRINTK("waiting...");
        wait = false;
    }

    LPI_PRINTK("\n\nLPP WOKE from HIBERNATE\n");
    LPI_PRINTK("\n\nHibernate Wake command received....:\n");
}

static void handle_softoff_sleep(void)
{
    if (num_online_cpus() > 1)
    {
        printk("\n\n*** please disable non-boot CPUs, %d CPUs are online. ***\n",num_online_cpus());
        return;
    }
    // linux hooks cpu_idle routine and trigger LPP to manage DDR SR...
    if (lpi.mode) {
        disable_all();
    }

    // trigger transition -- assumes R4 will sync to use when ready, and wait for our sync to it. *DEFAULT LPP will SYNC*
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep,0,true,1);
    // reset mailbox AFTER R4 sync resp
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)0xFFFFFFFF,0,false,1);
    send_low_power_cmd(e_lpp_sys_initialize,    (void*)LOWPOWER_COMMAND_DATECODE,0,false,1);
    send_low_power_cmd(e_lpp_sys_power_service, (void*)lpi.lpp[e_lpp_level_softoff].service,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_wfi,     (void*)lpi.lpp[e_lpp_level_softoff].wfi,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_global,  (void*)lpi.lpp[e_lpp_level_softoff].global,0,false,0);
    send_low_power_cmd(e_lpp_sys_ddr_mc_flags,  (void*)lpi.lpp[e_lpp_level_softoff].mc,0,false,0);
    send_low_power_cmd(e_lpp_sys_avs,           (void*)lpi.lpp[e_lpp_level_softoff].avs,0,false,0);
    send_low_power_cmd(e_lpp_sys_power_debug,   (void*)lpi.lpp[e_lpp_level_softoff].debug,0,false,0);
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_TIMER
    printk("e_lpp_sys_alarm_ticks       %d 10ms ticks\n",LP_LPP_TEST_SOFTOFF_WAKE_TIMER);
    send_low_power_cmd(e_lpp_sys_alarm_ticks,   (void*)LP_LPP_TEST_SOFTOFF_WAKE_TIMER,0,false,0);
#endif
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_GPIO_0
    printk("e_lpp_gpio_pin_x            %#x\n",LP_LPP_TEST_SOFTOFF_WAKE_GPIO_0);
    send_low_power_cmd(e_lpp_gpio_pin_x,        (void*)LP_LPP_TEST_SOFTOFF_WAKE_GPIO_0,0,false,0);
#endif
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_INT_0
    printk("e_lpp_sys_irq_x             %#x\n",LP_LPP_TEST_SOFTOFF_WAKE_INT_0);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_SOFTOFF_WAKE_INT_0,0,false,0);
#endif
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_INT_1
    printk("e_lpp_sys_irq_x             %#x\n",LP_LPP_TEST_SOFTOFF_WAKE_INT_1);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_SOFTOFF_WAKE_INT_1,0,false,0);
#endif
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_INT_2
    printk("e_lpp_sys_irq_x             %#x\n",LP_LPP_TEST_SOFTOFF_WAKE_INT_2);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_SOFTOFF_WAKE_INT_2,0,false,0);
#endif
#ifdef LP_LPP_TEST_SOFTOFF_WAKE_INT_3
    printk("e_lpp_sys_irq_x             %#x\n",LP_LPP_TEST_SOFTOFF_WAKE_INT_3);
    send_low_power_cmd(e_lpp_sys_irq_x,         (void*)LP_LPP_TEST_SOFTOFF_WAKE_INT_3,0,false,0);
#endif
    // send any commands queue for options/overrides
    // this would likely include the setup for specific GPIO pin(s) or IRQ#(s)
    send_low_power_cmd_q(NULL);

    lpi.mode=e_lpp_level_softoff;
    printk("\n\nSLEEP MODE %s\n",lpi.mode_name[e_lpp_level_softoff]);
    LPI_PRINTK("lp_flags.sh show softoff_sleep_flags   = %#x\n",lpi.lpp[e_lpp_level_softoff].wfi);

    /*
     * redirect the cpuidle SR handler
     */
    map_cpuidle_fn(lp_sleep);

    mb();

    // send AP sync that we are ready to be shut off  - it will wait for us to WFI
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_deep_rdy,0,false,1);
}

// LPP RTOS can handle some simple power commands - experimentation...
static void handle_rtos_sleep(void)
{
#if 1
    printk("RTOS -- not supported, TBD\n");
    return;
#endif
    if (num_online_cpus() > 1)
    {
        printk("\n\n*** please disable non-boot CPUs, %d CPUs are online. ***\n",num_online_cpus());
        return;
    }
    // any access to this 'store' disables the manual LPP command mode.
    lp_cmd.index = 0;
    // r4 suspend_2 --> r4 manually handles pll bp, ddr sr...
    if (lpi.mode) {
        disable_all();
    }
    // for now, we just use settings in e_lpp_level_deep configuration.
    send_low_power_cmd(e_lpp_sys_power_wfi,     (void*)lpi.lpp[e_lpp_level_deep].wfi,0,false,0);
    send_low_power_cmd(e_lpp_sys_ddr_mc_flags,  (void*)lpi.lpp[e_lpp_level_deep].mc,0,false,0);
    send_low_power_cmd(e_lpp_sys_sleep_level,   (void*)e_lpp_sleep_level_suspend_2,0,true,1);
    /*
     * redirect the cpuidle SR handler
     */
    map_cpuidle_fn(lp_sleep);

    printk("\n\nSLEEP MODE R4 TEST: %s\n",lpi.mode_name[e_lpp_level_rtos]);
    LPI_PRINTK("RTOS sleep flags   = %#x\n",lpi.lpp[e_lpp_level_deep].wfi);
    LPI_PRINTK("RTOS mc flags      = %#x\n",lpi.lpp[e_lpp_level_deep].mc);
    lpi.mode=e_lpp_level_rtos;
    return;
}

static ssize_t store_low_power_idle(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    struct device_node *np;
    int value = e_lpp_level_levels;
    int i,len;
    char name[256];

    memcpy(name,buf,256);
    name[255] = 0;
    len = strlen(name);

    name[len-1] = (name[len-1] == '\n')?0:name[len-1];
    len = strlen(name);

    for (i = e_lpp_level_wake; i < e_lpp_level_levels; ++i)
    {
        if (!strnicmp(name, lpi.mode_name[i],len))
        {
            value = i;
            break;
        }
        //LPI_PRINTK("\n\n%d: %s : %s : %s\n",i,name,lpi.sleep_name[i],lpi.mode_name[i]);
        if (lpi.sleep_name[i] == NULL)
        {
            continue;
        }
        if (!strnicmp(name, lpi.sleep_name[i],len))
        {
            value = i;
            break;
        }
    }
    LPI_PRINTK("\n\nSelected mode: %d\n",value);
    if (port_handle == NULL) {
        printk("IPC handle = null!\n");
        return count;
    }

    if (!mc_base)
    {
        np = of_find_matching_node(NULL, of_mc_table);
        if (np) {
            mc_base = of_iomap(np, 0);
        }
        else{
            printk("mc_base error\n");
            return count;
        }
    }
    if (!ciu_base)
    {
        np = of_find_matching_node(NULL, of_ciu_table);
        if (np) {
            ciu_base = of_iomap(np, 0);
        }
        else{
            printk("ciu_base error\n");
            return count;
        }
    }
    if (!mpmu_base)
    {
        np = of_find_matching_node(NULL, of_mpmu_table);
        if (np) {
            mpmu_base = of_iomap(np, 0);
        }
        else{
            printk("mpmu_base error\n");
            return count;
        }
    }


    switch (value)
    {
    default:
        printk("\n\nLPI_MODE UNKNOWN - Force exit.  Use: wake, light, deep, softoff, hibernate.\n");
    case e_lpp_level_wake:
        disable_all();
        lpi.mode=e_lpp_level_wake;
        return count;
    case e_lpp_level_light:
        handle_light_sleep();
        return count;
    case e_lpp_level_deep:
        handle_deep_sleep();
        return count;
    case e_lpp_level_suspend:
        handle_suspend_sleep();
        return count;
    case e_lpp_level_hibernate:
        handle_hibernate_sleep();
        return count;
    case e_lpp_level_softoff:
        handle_softoff_sleep();
        return count;
    case e_lpp_level_rtos:
        handle_rtos_sleep();
        return count;
    }

    return count;
};
DEVICE_ATTR(lp_mode         , S_IWUSR | S_IRUSR, show_low_power_idle, store_low_power_idle);
DEVICE_ATTR(lp_cmd          , S_IWUSR | S_IRUSR, show_low_power_cmds, store_low_power_cmds);
DEVICE_ATTR(lp_debug_enable , S_IWUSR | S_IRUSR, show_low_power_debug, store_low_power_debug);
DEVICE_ATTR(lp_light_enable , S_IWUSR | S_IRUSR, show_light_lpi_enable, store_light_lpi_enable);

DEVICE_ATTR(light_sleep_flags  , S_IWUSR | S_IRUSR, show_light_sleep_flags  , store_light_sleep_flags);  /* SLEEP, AKA WFI flags */
DEVICE_ATTR(light_asr_flags    , S_IWUSR | S_IRUSR, show_light_asr_flags    , store_light_asr_flags);    /* ASR clocks flags */

DEVICE_ATTR(deep_sleep_flags  , S_IWUSR | S_IRUSR, show_deep_sleep_flags  , store_deep_sleep_flags);  /* SLEEP, AKA WFI flags */
DEVICE_ATTR(deep_wake_flags   , S_IWUSR | S_IRUSR, show_deep_wake_flags   , store_deep_wake_flags);
DEVICE_ATTR(deep_service_flags, S_IWUSR | S_IRUSR, show_deep_service_flags, store_deep_service_flags);
DEVICE_ATTR(deep_global_flags , S_IWUSR | S_IRUSR, show_deep_global_flags , store_deep_global_flags);
DEVICE_ATTR(deep_mc_flags     , S_IWUSR | S_IRUSR, show_deep_mc_flags     , store_deep_mc_flags);
DEVICE_ATTR(deep_avs_flags    , S_IWUSR | S_IRUSR, show_deep_avs_flags    , store_deep_avs_flags);
DEVICE_ATTR(deep_dbg_flags    , S_IWUSR | S_IRUSR, show_deep_dbg_flags    , store_deep_dbg_flags);

DEVICE_ATTR(suspend_sleep_flags  , S_IWUSR | S_IRUSR, show_suspend_sleep_flags  , store_suspend_sleep_flags);  /* SLEEP, AKA WFI flags */
DEVICE_ATTR(suspend_wake_flags   , S_IWUSR | S_IRUSR, show_suspend_wake_flags   , store_suspend_wake_flags);
DEVICE_ATTR(suspend_service_flags, S_IWUSR | S_IRUSR, show_suspend_service_flags, store_suspend_service_flags);
DEVICE_ATTR(suspend_global_flags , S_IWUSR | S_IRUSR, show_suspend_global_flags , store_suspend_global_flags);
DEVICE_ATTR(suspend_mc_flags     , S_IWUSR | S_IRUSR, show_suspend_mc_flags     , store_suspend_mc_flags);
DEVICE_ATTR(suspend_avs_flags    , S_IWUSR | S_IRUSR, show_suspend_avs_flags    , store_suspend_avs_flags);
DEVICE_ATTR(suspend_dbg_flags    , S_IWUSR | S_IRUSR, show_suspend_dbg_flags    , store_suspend_dbg_flags);
DEVICE_ATTR(suspend_cstate3      , S_IWUSR | S_IRUSR, show_suspend_cstate3      , store_suspend_cstate3);

DEVICE_ATTR(softoff_sleep_flags  , S_IWUSR | S_IRUSR, show_softoff_sleep_flags  , store_softoff_sleep_flags);  /* SLEEP, AKA WFI flags */
DEVICE_ATTR(softoff_wake_flags   , S_IWUSR | S_IRUSR, show_softoff_wake_flags   , store_softoff_wake_flags);
DEVICE_ATTR(softoff_service_flags, S_IWUSR | S_IRUSR, show_softoff_service_flags, store_softoff_service_flags);
DEVICE_ATTR(softoff_global_flags , S_IWUSR | S_IRUSR, show_softoff_global_flags , store_softoff_global_flags);
DEVICE_ATTR(softoff_mc_flags     , S_IWUSR | S_IRUSR, show_softoff_mc_flags     , store_softoff_mc_flags);
DEVICE_ATTR(softoff_avs_flags    , S_IWUSR | S_IRUSR, show_softoff_avs_flags    , store_softoff_avs_flags);
DEVICE_ATTR(softoff_dbg_flags    , S_IWUSR | S_IRUSR, show_softoff_dbg_flags    , store_softoff_dbg_flags);

DEVICE_ATTR(hibernate_sleep_flags  , S_IWUSR | S_IRUSR, show_hibernate_sleep_flags  , store_hibernate_sleep_flags);  /* SLEEP, AKA WFI flags */
DEVICE_ATTR(hibernate_wake_flags   , S_IWUSR | S_IRUSR, show_hibernate_wake_flags   , store_hibernate_wake_flags);
DEVICE_ATTR(hibernate_service_flags, S_IWUSR | S_IRUSR, show_hibernate_service_flags, store_hibernate_service_flags);
DEVICE_ATTR(hibernate_global_flags , S_IWUSR | S_IRUSR, show_hibernate_global_flags , store_hibernate_global_flags);
DEVICE_ATTR(hibernate_mc_flags     , S_IWUSR | S_IRUSR, show_hibernate_mc_flags     , store_hibernate_mc_flags);
DEVICE_ATTR(hibernate_avs_flags    , S_IWUSR | S_IRUSR, show_hibernate_avs_flags    , store_hibernate_avs_flags);
DEVICE_ATTR(hibernate_dbg_flags    , S_IWUSR | S_IRUSR, show_hibernate_dbg_flags    , store_hibernate_dbg_flags);

DEVICE_ATTR(reg_imaging_enable , S_IWUSR | S_IRUSR, show_reg_imaging_enable, store_reg_imaging_enable);
DEVICE_ATTR(reg_gpu_enable     , S_IWUSR | S_IRUSR, show_reg_gpu_enable,     store_reg_gpu_enable);
//DEVICE_ATTR(reg_upc     , S_IWUSR | S_IRUSR, show_light_lpi_enable, store_light_lpi_enable);


static struct attribute * low_power_idle_dev_attrs[] = {
    &dev_attr_lp_mode.attr,
    &dev_attr_lp_cmd.attr,
    &dev_attr_lp_debug_enable.attr,
    &dev_attr_lp_light_enable.attr,

    &dev_attr_deep_sleep_flags.attr,
    &dev_attr_deep_wake_flags.attr,
    &dev_attr_deep_service_flags.attr,
    &dev_attr_deep_global_flags.attr,
    &dev_attr_deep_mc_flags.attr,
    &dev_attr_deep_avs_flags.attr,
    &dev_attr_deep_dbg_flags.attr,

    &dev_attr_suspend_sleep_flags.attr,
    &dev_attr_suspend_wake_flags.attr,
    &dev_attr_suspend_service_flags.attr,
    &dev_attr_suspend_global_flags.attr,
    &dev_attr_suspend_mc_flags.attr,
    &dev_attr_suspend_avs_flags.attr,
    &dev_attr_suspend_dbg_flags.attr,
    &dev_attr_suspend_cstate3.attr,

    &dev_attr_softoff_sleep_flags.attr,
    &dev_attr_softoff_wake_flags.attr,
    &dev_attr_softoff_service_flags.attr,
    &dev_attr_softoff_global_flags.attr,
    &dev_attr_softoff_mc_flags.attr,
    &dev_attr_softoff_avs_flags.attr,
    &dev_attr_softoff_dbg_flags.attr,

    &dev_attr_hibernate_sleep_flags.attr,
    &dev_attr_hibernate_wake_flags.attr,
    &dev_attr_hibernate_service_flags.attr,
    &dev_attr_hibernate_global_flags.attr,
    &dev_attr_hibernate_mc_flags.attr,
    &dev_attr_hibernate_avs_flags.attr,
    &dev_attr_hibernate_dbg_flags.attr,

    &dev_attr_light_sleep_flags.attr,
    &dev_attr_light_asr_flags.attr,

    &dev_attr_reg_imaging_enable.attr,
    &dev_attr_reg_gpu_enable.attr,
//    &dev_attr_reg_upc.attr,
    NULL,
};

static struct attribute_group low_power_idle_attr_group = {
    .attrs = low_power_idle_dev_attrs,
};

static const struct attribute_group * low_power_idle_attr_groups[] = {
    &low_power_idle_attr_group,
    NULL,
};

static struct class low_power_idle_class =
{
    .name       = MY_DRIVER_NAME,
    .owner      = THIS_MODULE,
    .dev_groups = low_power_idle_attr_groups,
};

static struct device * port_dev;

/* State information that is tracked on a per-client basis */
struct instance_state {
    uint32_t ioctl_access_cnt;  /* simply an example of something that could be tracked. */
};

/*
 * This function is called whenever a client opens the driver's device node.
 */
static int driver_open(struct inode *ind, struct file *filp)
{
    struct instance_state *s;

    open_count++;
    s = kmalloc(sizeof(struct instance_state), GFP_KERNEL);
    if (!s)
        return -ENOMEM;

    s->ioctl_access_cnt = 0;
    filp->private_data = s;

    return 0;
}


/*
 * This function is called whenever a client closes its connection to the driver.
 */
static int driver_close(struct inode *ind, struct file *filp)
{
    open_count--;
    kfree(filp->private_data); /* Free instance_state */
    filp->private_data = NULL;
    return 0;
}


/*
 * Handles all ioctl() calls.  There is nothing magic about the
 * cmd numbers -- they simply need to be unique within a particular driver.
 */
static long driver_ioctl(struct file *filp, unsigned int cmd, unsigned long args)
{
    int32_t rc = -EINVAL;
    struct instance_state *s = (struct instance_state *)filp->private_data;

    s->ioctl_access_cnt++;

    switch (cmd) {
        /* IOCTL handlers */
        default:
            break;
    }

    return rc;
}

#if 0
/*
 * A handy place to dump out some driver status information.  Called
 * whenever a client reads the drivers proc node.
 */
static int driverReadProc(
        char *buf,
        char **start,
        off_t offset,
        int len,
        int *eof,
        void *data
        )
{
    char *obuf = buf;

    buf += sprintf(buf, "\n-- Driver Information--\n");

    return(buf - obuf);
}
#endif

/*
 * Specifies which file operations are supported by the driver.
 */
static struct file_operations driver_fops = {
         owner:          THIS_MODULE,
         unlocked_ioctl: driver_ioctl,
         open:           driver_open,
         release:        driver_close,
};


/*
 * Stub release function -- called during rmmod.
 */
static void driver_release (struct device *dev){}


/*
 * Called by the kernel to determine whether the hardware exists that is
 * managed by this driver.
 */
static int driver_probe(struct platform_device *pdev)
{

    miscdev.minor = MISC_DYNAMIC_MINOR;
    miscdev.name  = MY_DRIVER_NAME;
    miscdev.fops  = &driver_fops;
    misc_register(&miscdev);

#if 0
    /* Create the /proc node for our device */
    create_proc_read_entry(MY_DRIVER_NAME, 0, NULL, (read_proc_t *)driverReadProc, NULL);
#endif

    /*
     * Create a class
     */
    class_register(&low_power_idle_class);
    port_dev = device_create(&low_power_idle_class, NULL, MKDEV(0, 0), NULL, "sleep");
    if (IS_ERR(port_dev)) {
        pr_err("%s: failed to create device for low_power_idle\n", __func__);
    }

    /*
     * The kernel complains during rmmod if the release member of the device
     * structure is NULL, so go ahead and provide a stub.
     */
    if (!pdev->dev.release)
        pdev->dev.release = driver_release;

    return(0);
};



/*
 * Called when a driver is unloaded.
 */
static int driver_remove(struct platform_device *pdev)
{

#if 0
    /*
     * Remove read_proc
     */
    remove_proc_entry(MY_DRIVER_NAME, NULL);
#endif

    misc_deregister(&miscdev);

    /*
     * Remove class
     */
    device_unregister(port_dev);
    put_device(port_dev);

    class_unregister(&low_power_idle_class);

    return 0;
}

/*
 * Defines a platform driver structure that can be passed to
 * platform_driver_register() to associate a dtiver with a named device.
 */
static struct platform_device low_power_idle_device  = {
    .name = MY_DRIVER_NAME,
    .id   = 0,
    .num_resources = 0,
    .resource = 0,
};

/*
 * Defines a platform driver structure that can be passed to
 * platform_driver_register() to associate a driver with a named device.
 */
static struct platform_driver low_power_idle_driver = {
    .probe  = driver_probe,
    .remove = driver_remove,
    .driver = {
        .name   = MY_DRIVER_NAME,
        .owner  = THIS_MODULE,
    },
};

/*
 *
 */
void recv_callback(ipc_drvr_handle handle, void *user_param, uint8_t command, void *buffer, uint16_t length)
{
    ipc_response_cmd = command;
    ipc_response_param = (int)buffer;
    if (!ipc_response_cmd) {
        printk("LPP API Datecode            = %x\n", ipc_response_param );
        if (LOWPOWER_COMMAND_DATECODE != ipc_response_param) {
            printk("\n\n**Warning: LPP Power API Datecode mismatch.  Our datecode is = %x\n", LOWPOWER_COMMAND_DATECODE );
        }
    }
    else {
        printk("Low Power Idle RX IPC cmd: %d; param: %#x\n", command, ipc_response_param );
    }
    if (lpi.mode && ipc_response_cmd==e_lpp_sys_sleep_level && ipc_response_param==e_lpp_sleep_level_wake) {
        printk("Exiting LPD MODE: %s\n", lpi.mode_name[lpi.mode]);
        low_power_mode_clean_up();
        lpi.mode = e_lpp_level_wake;
    }
}

/*
 * Called when the driver is loaded (insmod).
 */
static int __init low_power_idle_init(void)
{
    int error;

    /* Register the platform device with the kernel */
    error = platform_device_register(&low_power_idle_device);
    if (error)
        return error;

    // Get a handle for the IPC interface
    port_handle = ipc_attach(IPC0_DEVICE, IPC_PORT_POWER_MANAGER, recv_callback, NULL);
    if (port_handle == NULL)
    {
        printk("%s: ipc_attach failed\n", __func__);
    }
    else
    {
        printk("\n\n");
        printk("LOW_POWER_IDLE_BUILD        = %x\n",LOW_POWER_IDLE_BUILD);
        printk("LOW_POWER_IDLE_API_DATECODE = %x\n",LOWPOWER_COMMAND_DATECODE);
        send_low_power_cmd(e_lpp_sys_initialize,(void*)LOWPOWER_COMMAND_DATECODE,0,true,0);
    }

    // initialize a timestamp..
    ktime_previous = ktime_get();

    if (DEBUG) printk("\n\n*** LPI DEBUG is ENABLED ***\n\n\n");
#ifdef EXPERIMENT
    printk("\n\n\nLOW_POWER_IDLE EXPERIMENT BUILD\n\n\n");
#endif
    //printk("\n\nls \/sys\/class\/low_power_idle\/sleep\n");
    //printk("cat \/sys\/class\/low_power_idle\/sleep\/mode\n\n");

    /*
     * Register the platform driver with the kernel.  This will cause the
     * platform driver's probe function to be invoked.
     */
    return platform_driver_register(&low_power_idle_driver);
}


/*
 * Called when the driver is unloaded (rmmod).
 */
static void __exit low_power_idle_exit(void)
{
    disable_all();
    if (port_handle) ipc_detach(port_handle);
    platform_device_unregister(&low_power_idle_device);
    platform_driver_unregister(&low_power_idle_driver);
}


/* Register module init & exit functions */
module_init(low_power_idle_init);
module_exit(low_power_idle_exit);

