/*
 * phy-mv-ehci.c - USB PHY driver for Marvell based chip
 *
 * Copyright (c) 2008 Philipp Zabel <philipp.zabel@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/usb.h>
#include <linux/workqueue.h>
#include <linux/of_gpio.h>
#include <linux/of.h>
#include <linux/usb/phy.h>
#include <linux/clk.h>

#define PHY_PLL0  0
#define   PLL_READY_SHIFT 31
#define   REFDIV_SHIFT 0
#define   REFDIV_MASK 0x7f
#define   FBDIV_SHIFT 16
#define   FBDIV_MASK 0x1ff
#define   SEL_LPFR_SHIFT 28
#define   SEL_LPFR_MASK  0x3

#define PHY_PLL1  4
#define   PU_PLL         (1 << 0)
#define   PU_PLL_BY_REG  (1 << 1)
#define   PLL_SUSPEND_EN (1 << 3)

#define CAL_CNTL  8
#define   IMPCAL_START  (1 << 13)
#define   PLLCAL_START  (1 << 22)
#define   IMPCAL_DONE   (1 << 23)
#define   PLLCAL_DONE   (1 << 31)
#define   IMPCAL_VTH_SHIFT 8
#define   IMPCAL_VTH_MASK 0x7
#define   IMPCAL_VTH_39_OHMS 0x06

#define TX_CCR0  0xc
#define   PU_ANA        (1 << 24)
#define   PU_BY_REG     (1 << 25)

#define   MASK_AMP                 (0x7UL << 20)
#define   VAL_AMP_SIMVA_A          (0x2UL << 20)
#define   VAL_AMP_SIMVA_B          (0x3UL << 20)
#define   VAL_AMP_SIMVA_C          (0x4UL << 20)
#define   VAL_AMP_SIMVA_D          (0x5UL << 20)
#define   MASK_HS_SR_SEL           (0x3UL << 4)
#define   VAL_HS_SR_SEL_SIMVA_4    (0x0UL << 4)
#define   VAL_HS_SR_SEL_SIMVA_6    (0x2UL << 4)
#define   MASK_IMP_SEL_LS          (0xfUL << 16)
#define   VAL_IMP_SEL_LS_A         (0x3UL << 16)
#define   VAL_IMP_SEL_LS_B         (0xeUL << 16)
#define   MASK_DRV_EN_LS           (0xfUL << 12)
#define   VAL_DRV_EN_LS_A          (0x3UL << 12)
#define   VAL_DRV_EN_LS_B          (0x7UL << 12)

#define RX_CC0   0x14
#define   VAL_SQ_DET_SEL           (0x1UL << 28)
#define   MASK_SQ_DET_EN           (0x1UL << 15)
#define   VAL_SQ_DET_EN            (0x0UL << 15)

#define RX_CC1   0x18
#define   REG_SQCAL_START (1 << 4)
#define   REG_SQCAL_DONE  (1 << 31)
#define   VAL_EXT_SQ_AMP_CAL_EN    (0x1UL << 3)
#define   MASK_EXT_SQ_AMP_CAL      (0x7UL << 0)
#define   VAL_EXT_SQ_AMP_CAL       (0x1UL << 0)

#define DIG_CR_0 0x1c
#define   PLL_LOCK_BYPASS (1 << 7)

static DEFINE_MUTEX(pll_lock);
static int phys_running = 0;

struct phy_ehci_mv_data {
	struct usb_phy		phy;
	void __iomem *		reg;
	struct clk *		clk;
	u32			index;
	int			running;
};

static void start_pll(struct phy_ehci_mv_data *pdata) {
	mutex_lock(&pll_lock);

	if (phys_running == 0) {
		void __iomem *reg = pdata->reg;
		u32 val;

		val = readl(reg + PHY_PLL0);
		val &= ~(SEL_LPFR_MASK << SEL_LPFR_SHIFT);
		val |= (0 << SEL_LPFR_SHIFT);
		val &= ~(FBDIV_MASK << FBDIV_SHIFT);
		val |= (96 << FBDIV_SHIFT);
		val &= ~(REFDIV_MASK << REFDIV_SHIFT);
		val |= (5 << REFDIV_SHIFT);
		writel(val, reg + PHY_PLL0);

		val = readl(reg + DIG_CR_0);
		val |= PLL_LOCK_BYPASS;
		writel(val, reg + DIG_CR_0);

		val = readl(reg + PHY_PLL1);
		val &= ~PLL_SUSPEND_EN;
		val |= PU_PLL_BY_REG;
		writel(val, reg + PHY_PLL1);

		val = readl(reg + PHY_PLL1);
		val |= PU_PLL;
		writel(val, reg + PHY_PLL1);

		msleep(1);
		val = readl(reg + CAL_CNTL);
		val &= ~(PLLCAL_START | IMPCAL_START);
		writel(val, reg + CAL_CNTL);

		/* VTH -- set Impedence Calibration Threshold to 39 ohms */
		msleep(1);
		val = readl(reg + CAL_CNTL) & ~(IMPCAL_VTH_MASK << IMPCAL_VTH_SHIFT);
		val |= (PLLCAL_START | IMPCAL_START | (IMPCAL_VTH_39_OHMS << IMPCAL_VTH_SHIFT));
		writel(val, reg + CAL_CNTL);

		msleep(1);
		val = readl(reg + RX_CC1);
		val &= ~REG_SQCAL_START;
		writel(val, reg + RX_CC1);

		msleep(1);
		val = readl(reg + RX_CC1);
		val |= REG_SQCAL_START;
		writel(val, reg + RX_CC1);

		while ((readl(reg + CAL_CNTL) & (PLLCAL_DONE | IMPCAL_DONE)) != (PLLCAL_DONE | IMPCAL_DONE));
		while ((readl(reg + RX_CC1) & REG_SQCAL_DONE) == 0);

		while ((readl(reg + PHY_PLL0) & (1 << PLL_READY_SHIFT)) == 0);
	}

	++phys_running;

	mutex_unlock(&pll_lock);
}

static void stop_pll(struct phy_ehci_mv_data *pdata) {
	mutex_lock(&pll_lock);

	--phys_running;

	mutex_unlock(&pll_lock);
}

static int phy_ehci_mv_init(struct usb_phy *phy) {
	struct phy_ehci_mv_data *pdata =
		container_of(phy, struct phy_ehci_mv_data, phy);

	if (pdata->running == 0) {
		u32 val;
		u32 val_amp;
		u32 val_hs_sr_sel;
		u32 val_imp_sel_ls;
		u32 val_drv_en_ls;
		int gui_dev_type;

		clk_prepare_enable(pdata->clk);
		start_pll(pdata);

		val = readl(pdata->reg + (pdata->index * 0x40) + TX_CCR0);

		{//get gui-device type from dts
			//gui_dev_type(0:4line, 1:touch panel, 2:cheetah)
			struct device_node *np;
			int of_r32_ret;

			np = of_find_compatible_node(NULL, NULL, "ricoh,gui-device-type");
			if (!np) {
				printk(KERN_EMERG "%s(%d): of_find_compatible_node() failed! \n", __func__, __LINE__);
			}

			of_r32_ret = of_property_read_u32(np, "gui-device-type", &gui_dev_type); //gui_dev_type 0:4line, 1:touch panel, 2:cheetah
			if (of_r32_ret < 0) {
				printk(KERN_EMERG "%s(%d): of_property_read_u32() failed! \n", __func__, __LINE__);
			}
		}

		if (gui_dev_type == 2) { /*cheetah, MF sh1v*/
			switch(pdata->index) {
			case 0:
				//0xD074000C:0x03BE7F6F
				val_amp = VAL_AMP_SIMVA_B;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
				val_imp_sel_ls = VAL_IMP_SEL_LS_A;
				val_drv_en_ls = VAL_DRV_EN_LS_A;
				break;
			case 1:
				//0xD074004C:0x03DE7F4F
				val_amp = VAL_AMP_SIMVA_C;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_4;
				val_imp_sel_ls = VAL_IMP_SEL_LS_A;
				val_drv_en_ls = VAL_DRV_EN_LS_A;
				break;
			case 2:
				//0xD074008C:0x03BE7F6F
				val_amp = VAL_AMP_SIMVA_B;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
				val_imp_sel_ls = VAL_IMP_SEL_LS_B;
				val_drv_en_ls = VAL_DRV_EN_LS_B;
				break;
			default:
				//default 0x03BE7F6F
				val_amp = VAL_AMP_SIMVA_B;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
				val_imp_sel_ls = VAL_IMP_SEL_LS_B;
				val_drv_en_ls = VAL_DRV_EN_LS_B;
				break;
			}
		} else if (gui_dev_type == 0) { /*P-ki s1v*/
			switch(pdata->index) {
			case 0:
				//0xD074000C:0x03CE7F4F
				val_amp = VAL_AMP_SIMVA_C;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_4;
				val_imp_sel_ls = VAL_IMP_SEL_LS_A;
				val_drv_en_ls = VAL_DRV_EN_LS_A;
				break;
			case 1:
				//0xD074004C:0x03CE7F4F
				val_amp = VAL_AMP_SIMVA_C;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_4;
				val_imp_sel_ls = VAL_IMP_SEL_LS_A;
				val_drv_en_ls = VAL_DRV_EN_LS_A;
				break;
			case 2:
				//0xD074008C:0x03BE7F6F
				val_amp = VAL_AMP_SIMVA_B;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
				val_imp_sel_ls = VAL_IMP_SEL_LS_B;
				val_drv_en_ls = VAL_DRV_EN_LS_B;
				break;
			default:
				//default 0x03BE7F6F
				val_amp = VAL_AMP_SIMVA_B;
				val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
				val_imp_sel_ls = VAL_IMP_SEL_LS_B;
				val_drv_en_ls = VAL_DRV_EN_LS_B;
				break;
			}
		} else {
			//default 0x03BE7F6F
			val_amp = VAL_AMP_SIMVA_B;
			val_hs_sr_sel = VAL_HS_SR_SEL_SIMVA_6;
		}
		val &= (~MASK_AMP);
		val |= val_amp;
		val &= (~MASK_HS_SR_SEL);
		val |= val_hs_sr_sel;
		val &= (~MASK_IMP_SEL_LS);
		val |= val_imp_sel_ls;
		val &= (~MASK_DRV_EN_LS);
		val |= val_drv_en_ls;

		val |= (PU_BY_REG | PU_ANA);
		writel(val, pdata->reg + (pdata->index * 0x40) + TX_CCR0);

		if (pdata->index == 2) {
			/* RX_CC0 */
			val = 0x00009657;
			writel(val, pdata->reg + (pdata->index * 0x40) + RX_CC0);
			/* RX_CC1 */
			val = 0x00000004;
			writel(val, pdata->reg + (pdata->index * 0x40) + RX_CC1);
		} else {
			/* RX_CC0 */
			val = readl(pdata->reg + (pdata->index * 0x40) + RX_CC0);
			val |= VAL_SQ_DET_SEL;
			val &= (~MASK_SQ_DET_EN);
			val |= VAL_SQ_DET_EN;
			writel(val, pdata->reg + (pdata->index * 0x40) + RX_CC0);
			/* RX_CC1 */
			val = readl(pdata->reg + (pdata->index * 0x40) + RX_CC1);
			val |= VAL_EXT_SQ_AMP_CAL_EN;
			val &= (~MASK_EXT_SQ_AMP_CAL);
			val |= VAL_EXT_SQ_AMP_CAL;
			writel(val, pdata->reg + (pdata->index * 0x40) + RX_CC1);
		}

		pdata->running = 1;
	}

	return 0;
}

static void phy_ehci_mv_shutdown(struct usb_phy *phy) {
	struct phy_ehci_mv_data *pdata =
		container_of(phy, struct phy_ehci_mv_data, phy);

	if (pdata->running == 1) {
		u32 val;

		val = readl(pdata->reg + (pdata->index * 0x40) + TX_CCR0);
		val &= ~(PU_BY_REG | PU_ANA);
		writel(val, pdata->reg + (pdata->index * 0x40) + TX_CCR0);

		stop_pll(pdata);
		clk_disable_unprepare(pdata->clk);

		pdata->running = 0;
	}
}

/* platform driver interface */
static int phy_ehci_mv_probe(struct platform_device *pdev)
{
	struct phy_ehci_mv_data *pdata;
	void __iomem *reg;
	struct resource *res;
	int err;

	pdata = devm_kzalloc(&pdev->dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	reg = devm_ioremap(&pdev->dev, res->start, resource_size(res));
	if (IS_ERR(reg))
		return PTR_ERR(reg);

	pdata->clk = devm_clk_get(&pdev->dev, "phyclk");
	if (IS_ERR(pdata->clk))
		return PTR_ERR(pdata->clk);

	platform_set_drvdata(pdev, pdata);

	pdata->phy.label = "phy-ehci-mv";
	pdata->phy.dev = &pdev->dev;
	pdata->phy.init = phy_ehci_mv_init;
	pdata->phy.shutdown = phy_ehci_mv_shutdown;
	pdata->reg = reg;
	of_property_read_u32(pdev->dev.of_node, "phy-index", &pdata->index);

	err = usb_add_phy_dev(&pdata->phy);
	if (err) {
		dev_err(&pdev->dev, "can't register transceiver, err: %d\n",
			err);
		return err;
	}

	return 0;
}

static int phy_ehci_mv_remove(struct platform_device *pdev)
{
	struct phy_ehci_mv_data *pdata = platform_get_drvdata(pdev);

	usb_remove_phy(&pdata->phy);

	return 0;
}

MODULE_ALIAS("platform:phy-ehci-mv");

static const struct of_device_id phy_ehci_mv_of_dt_ids[] = {
	{ .compatible = "marvell,phy-ehci-mv" },
	{},
};
MODULE_DEVICE_TABLE(of, phy_ehci_mv_of_dt_ids);

static struct platform_driver phy_ehci_mv_driver = {
	.driver = {
		.name	= "phy-ehci-mv",
		.owner	= THIS_MODULE,
		.of_match_table	= of_match_ptr(phy_ehci_mv_of_dt_ids),
	},
	.probe		= phy_ehci_mv_probe,
	.remove		= phy_ehci_mv_remove,
};

module_platform_driver(phy_ehci_mv_driver);

MODULE_DESCRIPTION("Marvell USB PHY driver");
MODULE_LICENSE("GPL");
