/*
 *  I2C adapter for the SCCP I2C bus access.
 *
 *  Copyright (C) 2002 Intrinsyc Software Inc.
 *  Copyright (C) 2004-2005 Deep Blue Solutions Ltd.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/time.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/firmware.h>

#include <asm/irq.h>

/* SCCP register offsets */
#define SCCP_INT_ADDR         0x38
#define SCCP_CPU_CTRL_ADDR    0x100
#define SCCP_IPTR_ADDR        0x104

#define SCCP_SRAM_ADDR        0x1000

#define SCCP_SRAM_CMDA_ADDR    0x1080
#define SCCP_SRAM_CMDB_ADDR   0x1084

#define SCCP_SRAM_SCLK_ADDR    0x1090
#define SCCP_SRAM_SDATA_ADDR   0x1094

#define SCCP_SRAM_CLK_STRETCH_TIMEOUT   0x1098

/* Register field settings */
#define SCCP_CPU_CTRL_PRIORITY  0x4
#define SCCP_CPU_CTRL_RUN       0x1

/* Instruction Ptr start (ucode dependent) */
#define SCCP_IPTR               0x2a

/* Interrupt defines */
#define SCCP_INT_SUCCESS      0x1
#define SCCP_INT_ADDR_NACK    0x2
#define SCCP_INT_DATA_NACK    0x4

#define TIMEOUT 		20 /* ms */

/* Error interrupts - determined by SCCP ucode */
#define ERR_ADDR_NOACK      1
#define ERR_WRDATA_NOACK    2
#define ERR_DATA_LOW        3
#define ERR_INVALID_CMD     4
#define ERR_CLK_STRETCH_TIMEOUT 5

#define SCCP_INT_ERR_ADDR_NOACK     (1UL << ERR_ADDR_NOACK)
#define SCCP_INT_ERR_WRDATA_NOACK   (1UL << ERR_WRDATA_NOACK)
#define SCCP_INT_ERR_DATA_LOW       (1UL << ERR_DATA_LOW)
#define SCCP_INT_ERR_INVALID_CMD    (1UL << ERR_INVALID_CMD)
#define SCCP_INT_ERR_CLK_STRETCH_TIMEOUT (1UL << ERR_CLK_STRETCH_TIMEOUT)

#define SCCP_INT_ERR                (SCCP_INT_ERR_ADDR_NOACK | SCCP_INT_ERR_WRDATA_NOACK | SCCP_INT_ERR_DATA_LOW | SCCP_INT_ERR_INVALID_CMD | SCCP_INT_ERR_CLK_STRETCH_TIMEOUT)

/*
 * Command A bits 15:9  = 7-bit I2C Device Select field (or I2CAddr[9:3] for 10-bit mode)
 * Command A bits  4:2  = 7-bit mode: <unused>, 10-bit mode: I2CAddr[2:0]
 */
#define CMDA_SEVEN_BIT_ADDR(x) ((x & 0x7f) << 9)
#define CMDA_TEN_BIT_ADDR(x) ((x & 0x3f8) << 6) | ((x & 0x7) << 2);

/*
 * Command A bit   8    = 0 for standard 100kHz speed, 1 for fast 400kHz speed
 */
#define CMDA_FAST_MODE 1 << 8

/*
 * Command A bit   7:5  = R/W (0 = write, 0x1 = read, 0x2 - write read, 0x3 - read 10bit)
 */
#define CMDA_ANY_BIT_WRITE 0 << 5
#define CMDA_SEVEN_BIT_READ 1 << 5
#define CMDA_ANY_BIT_WRITE_READ 2 << 5
#define CMDA_TEN_BIT_READ 3 << 5

/*
 * Command A bit   1    = 0=7-bit mode, 1=10-bit mode
 */
#define CMDA_TEN_BIT_MODE 1 << 1

/*
 * Command A bit   0    = <Reserved>
 */
#define CMDA_SPECIAL_RSVD_BIT 1 << 0

/*
 * Command B bits 15:8  = number of bytes to be read (currently max is 64 bytes)
 *                        (e.g. 0x40 = 64-byte transfer, 0x00 = 0-byte transfer)
 */
#define CMDB_READ_LEN(x) (x << 8)

/*
 * Command B bits  7:0  = number of bytes to be written
 *                        (e.g. 0x40 = 64-byte transfer, 0x00 = 0-byte transfer)
 */
#define CMDB_WRITE_LEN(x) (x)

static char *err_sources[] = {
	[ERR_ADDR_NOACK] = "slave address byte not acknowledged ",
	[ERR_WRDATA_NOACK] = "write data byte not acknowledged",
	[ERR_DATA_LOW] = "data line is low prior to transactions",
	[ERR_INVALID_CMD] = "invalid command",
};

#define SCCP_MAX_XFER_BYTES  64
#define SCCP_FW_FILENAME     "sccp.bin"

/**
 * struct i2c_sccp_dev - private i2c-sccp data
 * @dev: driver model device node
 * @base: IO registers pointer
 * @cmd_complete: tx completion indicator
 * @lock: protect this struct and IO registers
 * @cmd_err: run time hadware error code
 * @msgs: points to an array of messages currently being transfered
 * @msgs_num: the number of elements in msgs
 * @msg_idx: the element index of the current message in the msgs array
 * @msg_err: error status of the current transfer
 * @err_source: error interrupts
 * @irq: interrupt number for the i2c master
 * @adapter: i2c subsystem adapter node
 * @fast_mode: i2c speed (400khz or 100khz)
 */
struct i2c_sccp_dev {
	struct device		*dev;
	void __iomem		*base;
	struct clk		*clk;
	struct completion	cmd_complete;
	struct mutex		lock;
	int			cmd_err;
	struct i2c_msg  	*msgs;
	int			msgs_num;
	int			msg_idx;
	int			msg_err;
	u32			err_source;
	int			irq;
	struct i2c_adapter	adapter;
	int			fast_mode;
	int			sclk;
	int			sdata;
	int			clk_stretch_timeout;
};


/**
 * i2c_sccp_init() - initialize the SCCP hardware (to emulate an I2C master)
 * @dev: device private data
 *
 * This functions configures and enables the I2C master.
 * This function is called during I2C init function, and in case of timeout at
 * run time.
 */
static int i2c_sccp_init(struct i2c_sccp_dev *dev)
{
	u32 i;
	const struct firmware *fw;
	unsigned short * buf;
	int ret;

	/* Write '0' to clear any existing interrupts */
	writel(0x0, dev->base + SCCP_INT_ADDR);

	/* Give CPU priority to access SRAM */
	writel(SCCP_CPU_CTRL_PRIORITY, dev->base + SCCP_CPU_CTRL_ADDR);

	/* Instruction pointer start address in SRAM (specific to microcode) */
	writel(SCCP_IPTR, dev->base + SCCP_IPTR_ADDR);

	ret = request_firmware(&fw, SCCP_FW_FILENAME, dev->dev);
	if (ret < 0) {
		dev_err(dev->dev, "firmware file (%s) not loaded\n", SCCP_FW_FILENAME);
		return ret;
	} else {
		buf = kzalloc(fw->size, GFP_KERNEL);
		if(!buf) {
			dev_err(dev->dev, "Could not allocate firmware buffer\n");
			return -ENOMEM;
		} else {
			memcpy(buf, fw->data, fw->size);
		}
	}

	/* Write microcode to SCCP SRAM */
	for (i = 0; i < fw->size / 2; i++) {
		writel(be16_to_cpu(buf[i]), dev->base + SCCP_SRAM_ADDR + (i*4));
	}
	release_firmware(fw);
	kfree(buf);

	/* Tell the microcode which SCCP pins to use */
	writel(dev->sclk, dev->base + SCCP_SRAM_SCLK_ADDR);
	writel(dev->sdata, dev->base + SCCP_SRAM_SDATA_ADDR);

	/* Set the clock stretching timeout */
	writel(dev->clk_stretch_timeout, dev->base + SCCP_SRAM_CLK_STRETCH_TIMEOUT);

	/* Set Run bit */ 
	writel( (readl(dev->base + SCCP_CPU_CTRL_ADDR) | SCCP_CPU_CTRL_RUN),dev->base + SCCP_CPU_CTRL_ADDR);

	return 0;
}

/*
 * Wait for SCCP idle
 */
static int i2c_sccp_wait_idle(struct i2c_sccp_dev *dev)
{
	int timeout = TIMEOUT;

	/* Command location in SRAM will be 0 when a command is complete */
	while (readl(dev->base + SCCP_SRAM_CMDA_ADDR)!=0) {
		if (timeout <= 0) {
			dev_warn(dev->dev, "timeout waiting for SCCP idle\n");
			return -ETIMEDOUT;
		}
		timeout--;
		mdelay(1);
	}

	return 0;
}

/*
 * Issue I2C master read or write transaction.
 * This function errors if the message length exceeds the max buffer size (determined by ucode).
 */
static void i2c_sccp_xfer_msg(struct i2c_sccp_dev *dev)
{
	struct i2c_msg *msgs = dev->msgs;
	u32 addr = msgs[dev->msg_idx].addr;
	u32 buf_len, i;
	u16 cmda = 0, cmdb = 0;
	char *buf;

	if (msgs[dev->msg_idx].len == 0) {
		dev_err(dev->dev,"%s: invalid message length = %u\n", __func__, msgs[dev->msg_idx].len);
		dev->msg_err = -EINVAL;
		return;
	}

	buf = msgs[dev->msg_idx].buf;
	if(msgs[dev->msg_idx].len > SCCP_MAX_XFER_BYTES)
		buf_len = SCCP_MAX_XFER_BYTES;
	else
		buf_len = msgs[dev->msg_idx].len;

	if(msgs[dev->msg_idx].flags & I2C_M_TEN) {
		/* Set 10-bit mode */
		cmda |= CMDA_TEN_BIT_MODE;

		/* Set the 10-bit address */
		cmda |= CMDA_TEN_BIT_ADDR(addr);

		/* If this is a read, set the 10-bit read bits */
		if(msgs[dev->msg_idx].flags & I2C_M_RD)
			cmda |= CMDA_TEN_BIT_READ;
	} else {
		/* Set the 7-bit address */
		cmda |= CMDA_SEVEN_BIT_ADDR(addr);

		/* If this is a read, set the 7-bit read bits */
		if(msgs[dev->msg_idx].flags & I2C_M_RD)
			cmda |= CMDA_SEVEN_BIT_READ;
	}

	/* If fast_mode, set speed to 400khz */
	if(dev->fast_mode)
		cmda |= CMDA_FAST_MODE;

	/*
	 * Setting reserved bit 0 to avoid all 0's in cmda (cmda=0 => not executed by SCCP)
	 * for special case:  (7bit write, I2C Addr=0x000, std speed)
	 */
	cmda |= CMDA_SPECIAL_RSVD_BIT;

	if(msgs[dev->msg_idx].flags & I2C_M_RD) {
		/* Set the read length */
		cmdb = CMDB_READ_LEN(buf_len);
	} else {
		/*
		 * If a write transfer, write data (before issuing cmd)
		 * Data starts at base SRAM address
		 * Each SRAM word holds 2 bytes, MS Byte sent first
		 */
		for (i = 0; i<buf_len; i=i+2) {
			u32 val = buf[i] << 8;
			if (i < (buf_len-1))
				val |= buf[i+1];
			writel(val, dev->base + SCCP_SRAM_ADDR+(i*2));
		}

		msgs[dev->msg_idx].buf += buf_len;
		msgs[dev->msg_idx].len -= buf_len;

		/* Set the write length */
		cmdb = CMDB_WRITE_LEN(buf_len);
	}

	/* Write cmdb first (SCCP microcode is polling for nonzero cmd at cmda SRAM addr) */
	writel(cmdb, dev->base + SCCP_SRAM_CMDB_ADDR);
	writel(cmda, dev->base + SCCP_SRAM_CMDA_ADDR);
}

/*
 * Called when the current message index is a read.  Copy data
 * out of sccp sram into the target buffer
 */
static void i2c_sccp_read(struct i2c_sccp_dev *dev)
{
	struct i2c_msg *msgs = dev->msgs;
	int len, i;
	char *buf;

	buf = msgs[dev->msg_idx].buf;
	if(msgs[dev->msg_idx].len > SCCP_MAX_XFER_BYTES)
		len = SCCP_MAX_XFER_BYTES;
	else
		len = msgs[dev->msg_idx].len;

	/*
	 * Data starts at base SRAM address
	 * Each SRAM word holds 2 bytes, MS Byte sent first
	 */
	for (i=0; i<len; i=i+2) {
		/* if odd number of read bytes, last bytes will be in LSB (not MSB as with write) */
		if (i == (len-1)) {
			*buf++ = readl(dev->base + SCCP_SRAM_ADDR+(i*2)) & 0x00ff;
		} else {
			*buf++   = (readl(dev->base + SCCP_SRAM_ADDR+(i*2)) & 0xff00) >> 8;
			*buf++ = (readl(dev->base + SCCP_SRAM_ADDR+(i*2)) & 0x00ff);
		}
	}

	msgs[dev->msg_idx].buf = buf;
	msgs[dev->msg_idx].len -= len;
}

static int i2c_sccp_handle_err(struct i2c_sccp_dev *dev)
{
	unsigned long err_source = dev->err_source;
	int i;

	for_each_set_bit(i, &err_source, ARRAY_SIZE(err_sources))
		dev_dbg(dev->dev,"%s: %s\n", __func__, err_sources[i]);

	return -EREMOTEIO;
}

/*
 * Prepare controller for a transaction and call i2c_sccp_xfer_msg
 */
static int i2c_sccp_xfer(struct i2c_adapter *adap, struct i2c_msg msgs[], int num)
{
	struct i2c_sccp_dev *dev = i2c_get_adapdata(adap);
	int ret;

	mutex_lock(&dev->lock);

	reinit_completion(&dev->cmd_complete);
	dev->msgs = kzalloc(sizeof(struct i2c_msg) * num, GFP_KERNEL);
	if(!dev->msgs) {
		ret = -ENOMEM;
		goto done;
	}
	memcpy(dev->msgs, msgs, sizeof(struct i2c_msg) * num);
	dev->msgs_num = num;
	dev->cmd_err = 0;
	dev->msg_idx = 0;
	dev->msg_err = 0;
	dev->err_source = 0;

	ret = i2c_sccp_wait_idle(dev);
	if (ret < 0)
		goto done;

	/* Kick off the first message */
	i2c_sccp_xfer_msg(dev);

	/* wait for tx to complete */
	ret = wait_for_completion_timeout(&dev->cmd_complete, HZ);
	if (ret == 0) {
		dev_err(dev->dev, "controller timed out\n");
		i2c_sccp_init(dev);
		ret = -ETIMEDOUT;
		goto done;
	}

	if (dev->msg_err) {
		ret = dev->msg_err;
		goto done;
	}

	/* no error */
	if (likely(!dev->cmd_err)) {
		ret = num;
		goto done;
	}

	/* We have an error */
	if (dev->cmd_err) {
		ret = i2c_sccp_handle_err(dev);
		goto done;
	}
	ret = -EIO;

done:
	if(dev->msgs)
		kfree(dev->msgs);

	mutex_unlock(&dev->lock);

	return ret;
}

/*
 * Interrupt service routine. This gets called whenever a SCCP interrupt
 * occurs.
 */
static irqreturn_t i2c_sccp_isr(int this_irq, void *dev_id)
{
	struct i2c_sccp_dev *dev = dev_id;
	int done = 0;
	u32 irq_stat;

	/* Read the pending interrupts */
	irq_stat = readl(dev->base + SCCP_INT_ADDR);

	/* Clear pending interrupts */
	writel(0, dev->base + SCCP_INT_ADDR);
	dev_dbg(dev->dev, "%s: irq_stat=0x%x\n", __func__, irq_stat);

	if (irq_stat & SCCP_INT_ERR)  {
		dev->cmd_err = 1;
		dev->err_source = irq_stat;
		done = 1;
	} else if (irq_stat & SCCP_INT_SUCCESS) {

		/*
		 * If status is success, and this message was a read,
		 * then copy the data from the sccp sram into the
		 * target buffer
		 */
		if (dev->msgs[dev->msg_idx].flags & I2C_M_RD)
			i2c_sccp_read(dev);

		/*
		 * If the current msg length is 0, we are done with it.
		 * If its the last message then call complete and return.
		 * If its not the last message, increment the msg index
		 * and kick off the next transfer
		 *
		 * If the current msg length is not 0, we need to kick
		 * off another transfer to continue it
		 */
		if (dev->msgs[dev->msg_idx].len == 0) {
			if (dev->msg_idx == dev->msgs_num - 1) {
				done = 1;
			} else {
				dev->msg_idx++;
				i2c_sccp_xfer_msg(dev);
			}
		} else {
			i2c_sccp_xfer_msg(dev);
		}
	}

	if (done || dev->msg_err)
		complete(&dev->cmd_complete);

	return IRQ_HANDLED;
}

static ssize_t show_fast_mode(struct device *dev, struct device_attribute *devattr, char *buf)
{
	struct i2c_sccp_dev *data = dev_get_drvdata(dev);
	return snprintf(buf, PAGE_SIZE, "%d\n", data->fast_mode);
}

static ssize_t store_fast_mode(struct device *dev, struct device_attribute *devattr, const char *buf, size_t count)
{
	struct i2c_sccp_dev *data = dev_get_drvdata(dev);
 	int val;

 	sscanf(buf, "%d", &val);
 	if(val == 0) {
 		data->fast_mode = 0;
 	}
 	else if(val == 1) {
 		data->fast_mode = 1;
 	}
 	else {
 		dev_err(data->dev, "Invalid mode setting");
 	}

	return count;
}

static DEVICE_ATTR(fast_mode, S_IRUGO | S_IWUSR, show_fast_mode, store_fast_mode);

static u32 i2c_sccp_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR;
}

static const struct i2c_algorithm i2c_sccp_algorithm = {
	.master_xfer	= i2c_sccp_xfer,
	.functionality	= i2c_sccp_func,
};

static struct of_device_id i2c_sccp_dt_ids[] = {
	{ .compatible = "mrvl,sccp-i2c" },
	{}
};
MODULE_DEVICE_TABLE(of, i2c_sccp_dt_ids);

static int i2c_sccp_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct i2c_sccp_dev *dev;
	int ret = 0;

	dev = devm_kzalloc(&pdev->dev, sizeof(struct i2c_sccp_dev), GFP_KERNEL);
	if (!dev) {
		return -ENOMEM;
	}
	dev->dev = get_device(&pdev->dev);

	dev->base = of_iomap(np, 0);
	if (!dev->base) {
		dev_err(&pdev->dev, "ioremap failed for sccp i2c base address\n");
		return -ENODEV;
	}

	dev->irq = irq_of_parse_and_map(np, 0);
	if (!dev->irq) {
		dev_err(&pdev->dev, "failed to map sccp i2c irq\n");
		return -ENODEV;
	}

	dev->clk = of_clk_get(np, 0);
	if (IS_ERR(dev->clk) || clk_prepare_enable(dev->clk)) {
		dev_err(&pdev->dev, "error getting sccp i2c clock\n");
		return PTR_ERR(dev->clk);
	}

	if (of_get_property(np, "i2c-fast-mode", NULL))
		dev->fast_mode = 1;

	/* Read sccp pin to use as sclk, default to 6 */
	if (of_property_read_u32(np, "sclk", &dev->sclk)) {
		dev->sclk = 6;
	} else if ( dev->sclk < 0 || dev->sclk > 15) {
		dev_err(&pdev->dev, "invalid value for sclk %d\n", dev->sclk);
		ret =  -EINVAL;
		goto error_clk;
	}

	/* Read sccp pin to use as sdata, default to 5 */
	if (of_property_read_u32(np, "sdata", &dev->sdata)) {
		dev->sdata = 5;
	} else if ( dev->sdata < 0 || dev->sdata > 15) {
		dev_err(&pdev->dev, "invalid value for sdata %d\n", dev->sdata);
		ret = -EINVAL;
		goto error_clk;
	}

	/* Read clock stretch timeout, default to 0 (disabled) */
	if (of_property_read_u32(np, "clock-stretch-timeout", &dev->clk_stretch_timeout)) {
		dev->clk_stretch_timeout = 0;
	}

	init_completion(&dev->cmd_complete);
	mutex_init(&dev->lock);
	platform_set_drvdata(pdev, dev);

	i2c_set_adapdata(&dev->adapter, dev);
	dev->adapter.owner = THIS_MODULE;
	dev->adapter.class = I2C_CLASS_HWMON;
	strlcpy(dev->adapter.name, "I2c SCCP adapter",
			sizeof(dev->adapter.name));
	dev->adapter.algo = &i2c_sccp_algorithm;
	dev->adapter.nr = -1;
	dev->adapter.dev.parent = &pdev->dev;
	dev->adapter.dev.of_node = pdev->dev.of_node;

	/* Initialize the sccp hardware */
	ret = i2c_sccp_init(dev);
	if (ret) {
		dev_err(&pdev->dev, "failed to initialize the sccp block %d\n", ret);
		goto error_clk;
	}

	ret = devm_request_irq(&pdev->dev, dev->irq, i2c_sccp_isr, IRQF_SHARED, pdev->name, dev);
	if (ret) {
		dev_err(&pdev->dev, "failure requesting irq %d\n", dev->irq);
		goto error_clk;
	}

	ret = i2c_add_numbered_adapter(&dev->adapter);
	if (ret) {
		dev_err(&pdev->dev, "failure adding adapter\n");
	}

	ret = device_create_file(dev->dev, &dev_attr_fast_mode);
	if (ret) {
		dev_err(&pdev->dev, "failure adding sysfs fast_mode entry\n");
	}

	return ret;

error_clk:
	clk_disable_unprepare(dev->clk);
	return ret;
}

static int i2c_sccp_remove(struct platform_device *pdev)
{
	struct i2c_sccp_dev *dev = platform_get_drvdata(pdev);

	device_remove_file(dev->dev, &dev_attr_fast_mode);

	platform_set_drvdata(pdev, NULL);
	i2c_del_adapter(&dev->adapter);
	put_device(&pdev->dev);

	/* Fix - need to disable SCCP RUN bit?   */

	clk_disable_unprepare(dev->clk);

	return 0;
}

static struct platform_driver i2c_sccp_driver = {
	.probe		= i2c_sccp_probe,
	.remove		= i2c_sccp_remove,
	.driver		= {
		.name	= "sccp-i2c",
		.owner	= THIS_MODULE,
		.of_match_table = i2c_sccp_dt_ids,
	},
};

static int __init i2c_adap_sccp_init(void)
{
	return platform_driver_register(&i2c_sccp_driver);
}

static void __exit i2c_adap_sccp_exit(void)
{
	platform_driver_unregister(&i2c_sccp_driver);
}

MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:sccp-i2c");

subsys_initcall(i2c_adap_sccp_init);
module_exit(i2c_adap_sccp_exit);
