/*
 *  GPIO Edge IRQ to Level Emulation
 *
 *  Copyright (C) 2015 Lexmark International
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/irqdomain.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include "gpiolib.h"

#define DRIVER_NAME "gpio-edge-to-level"

struct gpio_e2l {
	struct device *dev;
	struct irq_chip chip;
	struct irq_domain *domain;
	struct work_struct work;
	struct gpio_desc *gpio;
	int irq;
	int virq;
	bool masked;
	bool requested;
	unsigned int current_flow_type;
};

static int gpio_e2l_is_active(struct gpio_e2l *e2l, int level)
{
	return (level && (e2l->current_flow_type & IRQF_TRIGGER_RISING)) ||
		(!level && (e2l->current_flow_type & IRQF_TRIGGER_FALLING));
}

static void gpio_e2l_work(struct work_struct *work)
{
	struct gpio_e2l *e2l = container_of(work, struct gpio_e2l, work);
	struct irq_desc *desc = irq_to_desc(e2l->virq);
	int active;
	int level;

	if (e2l->masked)
		return;

	level = gpiod_get_value_cansleep(e2l->gpio);

	/* Desc lock protects current_flow_type mask */
	raw_spin_lock_irq(&desc->lock);
	active = gpio_e2l_is_active(e2l, level);
	raw_spin_unlock_irq(&desc->lock);

	if (active) {
		/*
		 * IRQ handlers assume IRQs and preemption disabled even if
		 * locally re-enabled.
		 */
		local_irq_disable();
		preempt_disable();

		/*
		 * The level flow handler will mask/unmask which will
		 * reschedule this work so there is no need to loop.
		 */
		generic_handle_irq_desc(e2l->virq, desc);

		preempt_enable();
		local_irq_enable();
	}
}

static irqreturn_t gpio_e2l_irq(int irq, void *data)
{
	struct gpio_e2l *e2l = data;

	schedule_work(&e2l->work);

	return IRQ_HANDLED;
}

static void gpio_e2l_mask(struct irq_data *data)
{
	struct gpio_e2l *e2l = irq_data_get_irq_chip_data(data);

	/*
	 * No locking on mask/unmask because they're only called while holding
	 * the irq_desc lock
	 *
	 * Don't cancel work here as this will be called from our work via the
	 * level flow handler
	 */
	if (!e2l->masked)
		disable_irq(e2l->irq);
	e2l->masked = true;
}

/*
 * Unmask is the only time polling needs to be triggered other than in response
 * to an interrupt edge. This happens when the client enables the interrupt, on
 * every IRQ call via the level flow handler, and on system resume when all
 * IRQs are unmasked. This may need to be revisited with suspend/resume hooks
 * if this doesn't work when the IRQ is configured as a wakeup source.
 */
static void gpio_e2l_unmask(struct irq_data *data)
{
	struct gpio_e2l *e2l = irq_data_get_irq_chip_data(data);

	/*
	 * No locking on mask/unmask because they're only called while holding
	 * the irq_desc lock
	 */
	if (e2l->masked)
		enable_irq(e2l->irq);
	e2l->masked = false;
	schedule_work(&e2l->work);
}

/*
 * Request the gpio and get the IRQ. We defer this until set_type() to allow
 * other clients to use it earlier if necessary. gpio_free() might sleep, so
 * we have to leak the requested gpio if there is a subsequent failure.
 */
static int gpio_e2l_gpio_to_irq(struct gpio_e2l *e2l)
{
	int rc;

	rc = gpiod_request(e2l->gpio, dev_name(e2l->dev));
	if (rc)
		return rc;
	rc = gpiod_direction_input(e2l->gpio);
	if (rc)
		return rc;

	return gpiod_to_irq(e2l->gpio);
}

static int gpio_e2l_set_type(struct irq_data *data, unsigned int flow_type)
{
	struct gpio_e2l *e2l = irq_data_get_irq_chip_data(data);
	unsigned int new_flow_type = 0;
	int rc;

	if (flow_type & IRQ_TYPE_EDGE_BOTH || !(flow_type & IRQ_TYPE_LEVEL_MASK)) {
		dev_err(e2l->dev, "Only level irqs are supported\n");
		return -EINVAL;
	}

	if (flow_type & IRQ_TYPE_LEVEL_HIGH)
		new_flow_type |= IRQF_TRIGGER_RISING;
	if (flow_type & IRQ_TYPE_LEVEL_LOW)
		new_flow_type |= IRQF_TRIGGER_FALLING;

	if (e2l->current_flow_type == new_flow_type)
		return 0;

	if (e2l->irq == -1) {
		e2l->irq = gpio_e2l_gpio_to_irq(e2l);
		if (e2l->irq < 0)
			return e2l->irq;
	}

	if (e2l->requested)
		free_irq(e2l->irq, e2l);

	/*
	 * This function is called with the desc lock held which protects the
	 * current_flow_type.
	 */

	rc = request_irq(e2l->irq, gpio_e2l_irq,
			 new_flow_type, dev_name(e2l->dev), e2l);
	if (rc) {
		dev_err(e2l->dev, "IRQ request failed %d\n", rc);
		return rc;
	}
	e2l->current_flow_type = new_flow_type;
	e2l->requested = true;
	/* Match disable depth with local masked state */
	if (e2l->masked)
		disable_irq(e2l->irq);
	schedule_work(&e2l->work);

	return 0;
}

int gpio_e2l_map(struct irq_domain *d, unsigned int virq, irq_hw_number_t hw)
{
	struct gpio_e2l *e2l = d->host_data;

	e2l->virq = virq;
	/* Handle level IRQ will take care of calling mask from from the
	 * downstream generic_handle_irq */
	irq_set_chip_and_handler(virq, &e2l->chip, handle_level_irq);
	irq_set_chip_data(virq, e2l);
	set_irq_flags(virq, IRQF_VALID);

	return 0;
}

static const struct irq_domain_ops gpio_e2l_domain_ops = {
	.map	= gpio_e2l_map,
	.xlate	= irq_domain_xlate_onecell,
};

static const struct of_device_id gpio_e2l_of_match[] = {
	{ .compatible = "linux," DRIVER_NAME, },
	{ }
};
MODULE_DEVICE_TABLE(of, gpio_e2l_of_match);

static int gpio_e2l_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct gpio_e2l *e2l;
	int gpio;

	e2l = devm_kzalloc(&pdev->dev, sizeof(*e2l), GFP_KERNEL);
	if (!e2l)
		return -ENOMEM;
	e2l->dev = &pdev->dev;
	INIT_WORK(&e2l->work, gpio_e2l_work);
	e2l->irq = -1;

	gpio = of_get_gpio(node, 0);
	if (!gpio_is_valid(gpio))
		return gpio == -EPROBE_DEFER ? gpio : -ENODEV;
	e2l->gpio = gpio_to_desc(gpio);

	e2l->chip.name = devm_kasprintf(e2l->dev, GFP_KERNEL, "GPIO-E2L@%d", gpio);
	if (!e2l->chip.name)
		return -ENOMEM;
	e2l->chip.irq_mask	= gpio_e2l_mask,
	e2l->chip.irq_unmask	= gpio_e2l_unmask,
	e2l->chip.irq_set_type	= gpio_e2l_set_type,

	e2l->domain = irq_domain_add_nomap(node, ~0, &gpio_e2l_domain_ops, e2l);
	if (!e2l->domain)
		return -ENOMEM;
	irq_create_direct_mapping(e2l->domain);

	return 0;
}

static struct platform_driver gpio_e2l_driver = {
	.driver	= {
		.name		= DRIVER_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= gpio_e2l_of_match,
	},
	.probe	= gpio_e2l_probe,
};

static int __init gpio_e2l_init(void)
{
	return platform_driver_register(&gpio_e2l_driver);
}
postcore_initcall(gpio_e2l_init);
