/*
 * Driver for the Marvell 61xx CDMA Controller
 *
 * Derived from drivers/dma/dw_dmacs.h
 *
 * Copyright (C) 2005-2007 Atmel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#ifndef MV61_CDMA_H
#define MV61_CDMA_H

#include <linux/err.h>
#include <linux/platform_data/mv61_cdma.h>
#include <asm/atomic.h>

#define CDMAINT_OFFSET	0x0
#define CTSR_OFFSET	0x4
#define CDMAPR_OFFSET	0x8
#define CTR0_OFFSET	0xc
#define CTR1_OFFSET	0x10

#define  CDMA_INTACK_RESERVED1_MASK 0xfffffff0

#define  CDMA_INTPEND_TERMCNTPEND_MASK 0x1
#define  CDMA_INTPEND_TIMEOUTPEND_MASK 0x2
#define  CDMA_INTPEND_CHAINDONEPEND_MASK 0x4
#define  CDMA_INTPEND_CLEARCOMPLETEPEND_MASK 0x8

#define  CDMA_STATUS_DATABUFEMPTY_MASK 0x4
#define  CDMA_STATUS_RDCMDBUFEMPTY_MASK 0x10
#define  CDMA_STATUS_WRCMDBUFEMPTY_MASK 0x40
#define  CDMA_STATUS_FIFOEMPTY_MASK 0x10000

#define  CDMA_STATUS_BYTESREMAIN_MASK 0xffff
#define  CDMA_STATUS_BYTESREMAIN_SHIFT 0
#define  CDMA_STATUS_BYTESREMAIN_MASK_SHIFT(reg) (((reg) & CDMA_STATUS_BYTESREMAIN_MASK) >> CDMA_STATUS_BYTESREMAIN_SHIFT)

#define  CDMA_DESTADDRSTAT_CURRDESTADDR_MASK 0xffffffff
#define  CDMA_DESTADDRSTAT_CURRDESTADDR_SHIFT 0
#define  CDMA_DESTADDRSTAT_CURRDESTADDR_MASK_SHIFT(reg) (((reg) & CDMA_DESTADDRSTAT_CURRDESTADDR_MASK) >> CDMA_DESTADDRSTAT_CURRDESTADDR_SHIFT)

#define  CDMA_CFG_ENABLE_MASK 0x1
#define  CDMA_CFG_HALT_MASK 0x800000

#define  CDMA_CFG_FILL_MASK 0x2
#define  CDMA_CFG_FILL_SHIFT 1
#define  CDMA_CFG_FILL_MASK_SHIFT(reg) (((reg) & CDMA_CFG_FILL_MASK) >> CDMA_CFG_FILL_SHIFT)
#define  CDMA_CFG_FILL_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_FILL_MASK) | (((uint32_t)val) << CDMA_CFG_FILL_SHIFT))

#define  CDMA_CFG_SRCENDIAN_MASK 0x4
#define  CDMA_CFG_SRCENDIAN_SHIFT 2
#define  CDMA_CFG_SRCENDIAN_MASK_SHIFT(reg) (((reg) & CDMA_CFG_SRCENDIAN_MASK) >> CDMA_CFG_SRCENDIAN_SHIFT)
#define  CDMA_CFG_SRCENDIAN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_SRCENDIAN_MASK) | (((uint32_t)val) << CDMA_CFG_SRCENDIAN_SHIFT))

#define  CDMA_CFG_DESTENDIAN_MASK 0x8
#define  CDMA_CFG_DESTENDIAN_SHIFT 3
#define  CDMA_CFG_DESTENDIAN_MASK_SHIFT(reg) (((reg) & CDMA_CFG_DESTENDIAN_MASK) >> CDMA_CFG_DESTENDIAN_SHIFT)
#define  CDMA_CFG_DESTENDIAN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_DESTENDIAN_MASK) | (((uint32_t)val) << CDMA_CFG_DESTENDIAN_SHIFT))

#define  CDMA_CFG_SRCPID_MASK 0x3f0
#define  CDMA_CFG_SRCPID_SHIFT 4
#define  CDMA_CFG_SRCPID_MASK_SHIFT(reg) (((reg) & CDMA_CFG_SRCPID_MASK) >> CDMA_CFG_SRCPID_SHIFT)
#define  CDMA_CFG_SRCPID_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_SRCPID_MASK) | (((uint32_t)val) << CDMA_CFG_SRCPID_SHIFT))

#define  CDMA_CFG_DESTPID_MASK 0x3f000
#define  CDMA_CFG_DESTPID_SHIFT 12
#define  CDMA_CFG_DESTPID_MASK_SHIFT(reg) (((reg) & CDMA_CFG_DESTPID_MASK) >> CDMA_CFG_DESTPID_SHIFT)
#define  CDMA_CFG_DESTPID_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_DESTPID_MASK) | (((uint32_t)val) << CDMA_CFG_DESTPID_SHIFT))

#define  CDMA_CFG_FLOWCTRL_MASK 0x700000
#define  CDMA_CFG_FLOWCTRL_SHIFT 20
#define  CDMA_CFG_FLOWCTRL_MASK_SHIFT(reg) (((reg) & CDMA_CFG_FLOWCTRL_MASK) >> CDMA_CFG_FLOWCTRL_SHIFT)
#define  CDMA_CFG_FLOWCTRL_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_FLOWCTRL_MASK) | (((uint32_t)val) << CDMA_CFG_FLOWCTRL_SHIFT))

#define  CDMA_CFG_DATAUNITSIZE_MASK 0x3000000
#define  CDMA_CFG_DATAUNITSIZE_SHIFT 24
#define  CDMA_CFG_DATAUNITSIZE_MASK_SHIFT(reg) (((reg) & CDMA_CFG_DATAUNITSIZE_MASK) >> CDMA_CFG_DATAUNITSIZE_SHIFT)
#define  CDMA_CFG_DATAUNITSIZE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_DATAUNITSIZE_MASK) | (((uint32_t)val) << CDMA_CFG_DATAUNITSIZE_SHIFT))

#define  CDMA_CFG_WRDELAY_MASK 0xf0000000
#define  CDMA_CFG_WRDELAY_SHIFT 28
#define  CDMA_CFG_WRDELAY_MASK_SHIFT(reg) (((reg) & CDMA_CFG_WRDELAY_MASK) >> CDMA_CFG_WRDELAY_SHIFT)
#define  CDMA_CFG_WRDELAY_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CFG_WRDELAY_MASK) | (((uint32_t)val) << CDMA_CFG_WRDELAY_SHIFT))

#define  CDMA_INTEN_TERMCNTEN_MASK 0x1
#define  CDMA_INTEN_TERMCNTEN_SHIFT 0
#define  CDMA_INTEN_TERMCNTEN_MASK_SHIFT(reg) (((reg) & CDMA_INTEN_TERMCNTEN_MASK) >> CDMA_INTEN_TERMCNTEN_SHIFT)
#define  CDMA_INTEN_TERMCNTEN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_INTEN_TERMCNTEN_MASK) | (((uint32_t)val) << CDMA_INTEN_TERMCNTEN_SHIFT))

#define  CDMA_INTEN_TIMEOUTEN_MASK 0x2
#define  CDMA_INTEN_TIMEOUTEN_SHIFT 1
#define  CDMA_INTEN_TIMEOUTEN_MASK_SHIFT(reg) (((reg) & CDMA_INTEN_TIMEOUTEN_MASK) >> CDMA_INTEN_TIMEOUTEN_SHIFT)
#define  CDMA_INTEN_TIMEOUTEN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_INTEN_TIMEOUTEN_MASK) | (((uint32_t)val) << CDMA_INTEN_TIMEOUTEN_SHIFT))

#define  CDMA_INTEN_CHAINDONEEN_MASK 0x4
#define  CDMA_INTEN_CHAINDONEEN_SHIFT 2
#define  CDMA_INTEN_CHAINDONEEN_MASK_SHIFT(reg) (((reg) & CDMA_INTEN_CHAINDONEEN_MASK) >> CDMA_INTEN_CHAINDONEEN_SHIFT)
#define  CDMA_INTEN_CHAINDONEEN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_INTEN_CHAINDONEEN_MASK) | (((uint32_t)val) << CDMA_INTEN_CHAINDONEEN_SHIFT))

#define  CDMA_INTEN_CLEARCOMPLETEEN_MASK 0x8
#define  CDMA_INTEN_CLEARCOMPLETEEN_SHIFT 3
#define  CDMA_INTEN_CLEARCOMPLETEEN_MASK_SHIFT(reg) (((reg) & CDMA_INTEN_CLEARCOMPLETEEN_MASK) >> CDMA_INTEN_CLEARCOMPLETEEN_SHIFT)
#define  CDMA_INTEN_CLEARCOMPLETEEN_REPLACE_VAL(reg,val) (((reg) & ~CDMA_INTEN_CLEARCOMPLETEEN_MASK) | (((uint32_t)val) << CDMA_INTEN_CLEARCOMPLETEEN_SHIFT))

#define  CDMA_CONTROL_SRCBURSTSIZE_MASK 0x3
#define  CDMA_CONTROL_SRCBURSTSIZE_SHIFT 0
#define  CDMA_CONTROL_SRCBURSTSIZE_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_SRCBURSTSIZE_MASK) >> CDMA_CONTROL_SRCBURSTSIZE_SHIFT)
#define  CDMA_CONTROL_SRCBURSTSIZE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_SRCBURSTSIZE_MASK) | (((uint32_t)val) << CDMA_CONTROL_SRCBURSTSIZE_SHIFT))

#define  CDMA_CONTROL_DESTBURSTSIZE_MASK 0x30
#define  CDMA_CONTROL_DESTBURSTSIZE_SHIFT 4
#define  CDMA_CONTROL_DESTBURSTSIZE_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_DESTBURSTSIZE_MASK) >> CDMA_CONTROL_DESTBURSTSIZE_SHIFT)
#define  CDMA_CONTROL_DESTBURSTSIZE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_DESTBURSTSIZE_MASK) | (((uint32_t)val) << CDMA_CONTROL_DESTBURSTSIZE_SHIFT))

#define  CDMA_CONTROL_SRCXFERWIDTH_MASK 0x300
#define  CDMA_CONTROL_SRCXFERWIDTH_SHIFT 8
#define  CDMA_CONTROL_SRCXFERWIDTH_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_SRCXFERWIDTH_MASK) >> CDMA_CONTROL_SRCXFERWIDTH_SHIFT)
#define  CDMA_CONTROL_SRCXFERWIDTH_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_SRCXFERWIDTH_MASK) | (((uint32_t)val) << CDMA_CONTROL_SRCXFERWIDTH_SHIFT))

#define  CDMA_CONTROL_DESTXFERWIDTH_MASK 0xc00
#define  CDMA_CONTROL_DESTXFERWIDTH_SHIFT 10
#define  CDMA_CONTROL_DESTXFERWIDTH_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_DESTXFERWIDTH_MASK) >> CDMA_CONTROL_DESTXFERWIDTH_SHIFT)
#define  CDMA_CONTROL_DESTXFERWIDTH_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_DESTXFERWIDTH_MASK) | (((uint32_t)val) << CDMA_CONTROL_DESTXFERWIDTH_SHIFT))

#define  CDMA_CONTROL_SRCADDRINC_MASK 0x1000
#define  CDMA_CONTROL_SRCADDRINC_SHIFT 12
#define  CDMA_CONTROL_SRCADDRINC_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_SRCADDRINC_MASK) >> CDMA_CONTROL_SRCADDRINC_SHIFT)
#define  CDMA_CONTROL_SRCADDRINC_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_SRCADDRINC_MASK) | (((uint32_t)val) << CDMA_CONTROL_SRCADDRINC_SHIFT))

#define  CDMA_CONTROL_DESTADDRINC_MASK 0x2000
#define  CDMA_CONTROL_DESTADDRINC_SHIFT 13
#define  CDMA_CONTROL_DESTADDRINC_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_DESTADDRINC_MASK) >> CDMA_CONTROL_DESTADDRINC_SHIFT)
#define  CDMA_CONTROL_DESTADDRINC_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_DESTADDRINC_MASK) | (((uint32_t)val) << CDMA_CONTROL_DESTADDRINC_SHIFT))

#define  CDMA_CONTROL_TRANSSIZE_MASK 0xffff0000
#define  CDMA_CONTROL_TRANSSIZE_SHIFT 16
#define  CDMA_CONTROL_TRANSSIZE_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_TRANSSIZE_MASK) >> CDMA_CONTROL_TRANSSIZE_SHIFT)
#define  CDMA_CONTROL_TRANSSIZE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_TRANSSIZE_MASK) | (((uint32_t)val) << CDMA_CONTROL_TRANSSIZE_SHIFT))

#define  CDMA_CONTROL_OWNWRITEDISABLE_A0_MASK 0x20000
#define  CDMA_CONTROL_OWNWRITEDISABLE_A0_SHIFT 17
#define  CDMA_CONTROL_OWNWRITEDISABLE_A0_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_OWNWRITEDISABLE_A0_MASK) >> CDMA_CONTROL_OWNWRITEDISABLE_A0_SHIFT)
#define  CDMA_CONTROL_OWNWRITEDISABLE_A0_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_OWNWRITEDISABLE_A0_MASK) | (((uint32_t)val) << CDMA_CONTROL_OWNWRITEDISABLE_A0_SHIFT))

#define  CDMA_CONTROL_OWNWRITEDISABLE_MASK 0x8000
#define  CDMA_CONTROL_OWNWRITEDISABLE_SHIFT 15
#define  CDMA_CONTROL_OWNWRITEDISABLE_MASK_SHIFT(reg) (((reg) & CDMA_CONTROL_OWNWRITEDISABLE_MASK) >> CDMA_CONTROL_OWNWRITEDISABLE_SHIFT)
#define  CDMA_CONTROL_OWNWRITEDISABLE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_CONTROL_OWNWRITEDISABLE_MASK) | (((uint32_t)val) << CDMA_CONTROL_OWNWRITEDISABLE_SHIFT))

#define  CDMA_SRCADDR_SRCADDR_MASK 0xffffffff
#define  CDMA_SRCADDR_SRCADDR_SHIFT 0
#define  CDMA_SRCADDR_SRCADDR_MASK_SHIFT(reg) (((reg) & CDMA_SRCADDR_SRCADDR_MASK) >> CDMA_SRCADDR_SRCADDR_SHIFT)
#define  CDMA_SRCADDR_SRCADDR_REPLACE_VAL(reg,val) (((reg) & ~CDMA_SRCADDR_SRCADDR_MASK) | (((uint32_t)val) << CDMA_SRCADDR_SRCADDR_SHIFT))

#define  CDMA_DESTADDR_DESTADDR_MASK 0xffffffff
#define  CDMA_DESTADDR_DESTADDR_SHIFT 0
#define  CDMA_DESTADDR_DESTADDR_MASK_SHIFT(reg) (((reg) & CDMA_DESTADDR_DESTADDR_MASK) >> CDMA_DESTADDR_DESTADDR_SHIFT)
#define  CDMA_DESTADDR_DESTADDR_REPLACE_VAL(reg,val) (((reg) & ~CDMA_DESTADDR_DESTADDR_MASK) | (((uint32_t)val) << CDMA_DESTADDR_DESTADDR_SHIFT))

#define  CDMA_TIMERCONTROL_COUNT_MASK 0x7ff
#define  CDMA_TIMERCONTROL_COUNT_SHIFT 0
#define  CDMA_TIMERCONTROL_COUNT_MASK_SHIFT(reg) (((reg) & CDMA_TIMERCONTROL_COUNT_MASK) >> CDMA_TIMERCONTROL_COUNT_SHIFT)
#define  CDMA_TIMERCONTROL_COUNT_REPLACE_VAL(reg,val) (((reg) & ~CDMA_TIMERCONTROL_COUNT_MASK) | (((uint32_t)val) << CDMA_TIMERCONTROL_COUNT_SHIFT))

#define  CDMA_TIMERCONTROL_TIMEBASE_MASK 0x7000
#define  CDMA_TIMERCONTROL_TIMEBASE_SHIFT 12
#define  CDMA_TIMERCONTROL_TIMEBASE_MASK_SHIFT(reg) (((reg) & CDMA_TIMERCONTROL_TIMEBASE_MASK) >> CDMA_TIMERCONTROL_TIMEBASE_SHIFT)
#define  CDMA_TIMERCONTROL_TIMEBASE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_TIMERCONTROL_TIMEBASE_MASK) | (((uint32_t)val) << CDMA_TIMERCONTROL_TIMEBASE_SHIFT))

#define  CDMA_TIMERCONTROL_TIMERENABLE_MASK 0x8000
#define  CDMA_TIMERCONTROL_TIMERENABLE_SHIFT 15
#define  CDMA_TIMERCONTROL_TIMERENABLE_MASK_SHIFT(reg) (((reg) & CDMA_TIMERCONTROL_TIMERENABLE_MASK) >> CDMA_TIMERCONTROL_TIMERENABLE_SHIFT)
#define  CDMA_TIMERCONTROL_TIMERENABLE_REPLACE_VAL(reg,val) (((reg) & ~CDMA_TIMERCONTROL_TIMERENABLE_MASK) | (((uint32_t)val) << CDMA_TIMERCONTROL_TIMERENABLE_SHIFT))

#define DESCRIPTOR_OWNLENGTH_LENGTH_MASK    0x0000ffff
#define DESCRIPTOR_OWNLENGTH_OWN_MASK       0x01000000

#define DESCRIPTOR_NEXTCTRL_NEXT_MASK       0xfffffffc
#define DESCRIPTOR_NEXTCTRL_INT_MASK        0x00000002
#define DESCRIPTOR_NEXTCTRL_STOP_MASK       0x00000001

#define MV61_DMA_MAX_NR_PCHANNELS	(12)
#define MV61_DMA_MAX_NR_VCHANNELS	(96)
#define MV61_SUBCHAIN_MAX_SIZE \
	(CDMA_CONTROL_TRANSSIZE_MASK >> CDMA_CONTROL_TRANSSIZE_SHIFT)

/* dev_vdbg uses KERN_DEBUG, but we need higher priority */
#define __dev_vdbg(dev, format, arg...)	if(vdebug) dev_printk(KERN_ALERT, dev , format , ##arg)
#define __dev_vrdbg(dev, format, arg...)	if(vresidue) dev_printk(KERN_ALERT, dev , format , ##arg)

#define SHOWVARW(var)  printk(KERN_NOTICE "%-33s = 0x%08x\n","   "#var, \
((unsigned int)(var)))
#define SHOWVARWA(var)  printk(KERN_NOTICE "%-33s = 0x%08x @%p\n","   "#var, \
((unsigned int)(var)), &var)

/**
 * enum mv61_alignment - shift for alignment masking
 * @MV61_PDMA_FLUSH: assert halt to explicitly flush fifo
 * @MV61_PDMA_CLEAR: write default state back to all regs
 */
enum mv61_term_flags {
	MV61_PDMA_FLUSH   = (1 << 0),
	MV61_PDMA_CLEAR = (1 << 1),
};

struct mv61_dma;
struct mv61_vdma;

/* Hardware register definitions. */

/**
 * enum mv61_alignment - shift for alignment masking
 * @MV61_VDMA_ALIGN8:  byte alignment
 * @MV61_VDMA_ALIGN16: short alignment
 * @MV61_VDMA_ALIGN32: word alignmnet
 */

enum mv61_alignment {
	MV61_VDMA_ALIGN8,
	MV61_VDMA_ALIGN16,
	MV61_VDMA_ALIGN32,
};

#define MV61_MEMCPY_ALIGN	MV61_VDMA_ALIGN32
#define MV61_MEMFILL_ALIGN	MV61_VDMA_ALIGN32
 
/**
 * struct mv61_pdma_chan_regs - auto-generated regs
 *
 * The reg overlay structure is auto-generated by the RTL build tools.
 * Per Kernel Coding Style, use the underlying struct instead of the 
 * typedef. See CDMA_regstructs.h.
 */
typedef struct mv61_pdma_chan_regs
{
  volatile uint32_t CFG;  ///< 0x0 [R/W]: DMA Configuration Register
  volatile uint32_t Control;  ///< 0x4 [R/W]: DMA Channel Control Register
  volatile uint32_t Status;  ///< 0x8 [R]: DMA Status Register
  volatile uint32_t reserved0;
  volatile uint32_t CPR;  ///< 0x10 [R]: CDMA Parameter Register
  volatile uint32_t CDR;  ///< 0x14 [R/W]: CDMA Descriptor Register
  volatile uint32_t CNDAR;  ///< 0x18 [R]: CDMA Next Descriptor Address Register
  volatile uint32_t FillValue;  ///< 0x1c [R/W]: Fill Value Register
  volatile uint32_t intEn;  ///< 0x20 [R/W]: Interrupt Enable Register
  volatile uint32_t intPend;  ///< 0x24 [R]: Interrupt Pending Register
  volatile uint32_t intAck;  ///< 0x28 [W]: Interrupt Acknowledge Register
  volatile uint32_t intForce;  ///< 0x2c [W]: Interrupt Force Register
  volatile uint32_t TimerControl;  ///< 0x30 [R/W]: DMA Timer Control
  volatile uint32_t TimeOutStat;  ///< 0x34 [R]: Timeout Count Status Register
  volatile uint32_t CRBAR;  ///< 0x38 [R]: CDMA Read Burst Address Register
  volatile uint32_t CRBLR;  ///< 0x3c [R]: CDMA Read Burst Length Register
  volatile uint32_t CWBAR;  ///< 0x40 [R]: CDMA Write Burst Address Register
  volatile uint32_t CWBLR;  ///< 0x44 [R]: CDMA Write Burst Length Register
  volatile uint32_t CWBRR;  ///< 0x48 [R]: CDMA Write Bytes Remain Register
  volatile uint32_t CSRR;  ///< 0x4c [R/W]: DMA Save/Restore Control Register
  volatile uint32_t CSRLI;  ///< 0x50 [R/W]: Save/Restore Lower DMA ID Register
  volatile uint32_t CSRUI;  ///< 0x54 [R/W]: Save/Restore Upper DMA ID Register
  volatile uint32_t CRSL;  ///< 0x58 [R]: DMA Lower Request Status Register
  volatile uint32_t CRSU;  ///< 0x5c [R]: DMA Upper Request Stautus Register
  volatile uint32_t CAFL;  ///< 0x60 [R/W]: DMA Lower Ack Force Register
  volatile uint32_t CAFU;  ///< 0x64 [R/W]: DMA Upper Ack Force Register
} mv61_pdma_chan_regs_t;

/**
 * struct mv61_vreg_defs - default reg settings for the virtual channel
 * @valid: structure contains valid data
 * @CFG: DMA Configuration Register
 * @Control: DMA Channel Control Register
 * @SrcAddr: DMA Source Address Register
 * @DestAddr: DMA Destination Address Register
 * @TimerControl: DMA Timer and Timebase Control
 *
 * This subset of register settings for the virtual channel is initialized
 * from platform data passed in when the channel is requested. It is used
 * to load the physical channel regs when the dispatcher grants access.
 * 
 * Channels obtained for memcpy/memset using dma_find_channel will not use this.
 */
struct mv61_vreg_defs {
	bool valid;
	u32 CFG;
	u32 Control;
	dma_addr_t SrcAddr;
	dma_addr_t DestAddr;	
	u32 TimerControl;
};

/**
 * struct mv61_tx_regs - reg values common to all blocks of a transfer
 * @CFG: dma configuration
 * @FillValue: data for memset
 * @intEn: interrupt enables
 * @Timercontrol: timebase and idle-bus timeout
 */
struct mv61_tx_regs {
	u32 CFG;
	u32 FillValue;
	u32 intEn;
	u32 TimerControl;
};

/**
 * struct mv61_stat_regs - hw status regs to save at end of transfer
 * @Status: 
 * @SrcAddrStat: 
 * @DestAddrStat: 
 * @LLIR: physical address of next lli in chain
 */
struct mv61_stat_regs {
	u32 Status;
	u32 SrcAddrStat;
	u32 DestAddrStat;
	u32 LLIR;
	u32 CPR;
	u32 CDR;
	u32 Control;
};

/**
 * struct mv61_lli - reg values used by hardware for one block of a transfer
 * @SrcAddr: physical address of source for this block
 * @DestAddr: physical address of destination for this block
 * @LLIR: physical address of next lli in chain
 * @Control: control register for this block
 */
struct mv61_lli {
	/* FIRST values the hardware uses, do not change order */
	u32			OwnLength;
	dma_addr_t		SrcAddr;
	dma_addr_t		DestAddr;
	u32			NextCtrl;
};

/**
 * struct mv61_chain - linked-list descriptor for one block of a transfer
 * @lli: the reg values used by hardware
 * @chain_node: links in subchain for transaction or child transaction.
 * @synced: lli member is currently synced for device
 * @phys: physical address of this descriptor and its lli member
 *
 * Transactions for non-shared channels will only have one subchain,
 * consisting of one mv61_desc link for the first scatterlist
 * entry, plus an mv61_chain link for each additional scatterlist entry.
 * Large transactions for shared channels will be broken into smaller pieces
 * to avoid monopolizing the hardware channel. Each piece will have its own
 * mv61_desc top link, plus an mv61_chain link for each additional scatterlist
 * entry in its piece of the transfer.
 *
 * Some transactions will use many of these descriptors, especially if they
 * require wrapping. For efficiency, it is designed to fit in a single cache
 * line.
 *
 */
struct mv61_chain {
	/* FIRST values the hardware uses */
	struct mv61_lli		lli;
	/* THEN values for driver housekeeping */
	struct list_head	chain_node;
	int			synced;
	dma_addr_t		phys;
};


/**
 * struct mv61_desc - top level transaction control structure
 * @lli: first hardware descriptor must be directly loaded by software
 * @subchain: top of hardware descriptor subchain
 * @all_chains_node: used for cleanup to avoid mem leaks
 * @status: TBD
 * @txregs: reg values common to all blocks of a transfer
 * @txd: transaction descriptor for dmaengine API
 * @len: total bytes to transfer
 * @sublen: total bytes in this subchain
 * @desc_node: TBD
 * @tx_list: TBD
 * @chains: number of mv61_desc nodes in this chain
 * @links: number of mv61_chain nodes in this subchain
 *
 * DMA-loaded hardware descriptors must be mapped for dma and sync'ed.
 * Since the first descriptor is directly loaded by software, it does not need
 * to be mapped or sync'ed, so put it here. The lli members will never be modified
 * after they are created, so they don't need lock protection.
 */
struct mv61_desc {
	struct mv61_lli			lli;
	dma_addr_t			lli_phys;
	struct list_head		subchain;
	struct list_head		all_chains_node;
	u32				status;
	struct mv61_tx_regs		txregs;
	struct dma_async_tx_descriptor	txd;
	size_t				len;
	size_t				sublen;
	struct list_head		desc_node;
	struct list_head		tx_list;
	struct mv61_desc		*active;
	int				chains;
	int				links;
	struct mv61_chain		*paused_link;
};

/**
 * struct mv61_pdma_chan - physical channel control
 * @ch_regs: virtual base address of channel's mv61_pdma_chan_regs
 * @vtype: channel type, same as id/type of virtual controllers
 * @index: this physical channel's index
 * @mv61p: top level control structure
 */
struct mv61_pdma_chan {
	void __iomem		*ch_regs;
	enum mv61_vdma_type	vtype;
	int			index;
	struct mv61_dma		*mv61p;

	/* these elements must be protected by struct mv61_dma.biglock */
};


/**
 * struct mv61_vdma_chan - virtual channel control
 * @chan: channel control structure used by dmaengine API
 * @mv61v: virtual controller instance that owns this channel
 * @def: default reg subset settings to load into a physical channel
 * @wrap: word aligned byte count for dest address to wrap
 * @lock: protect everything except def and chan (which has its own lock)
 * @irqs: TBD
 * @completed: TBD
 * @active_list: TBD
 * @complete_list: completed subtransactions 
 * @queue: TBD
 * @hwstat: 
 * @vcstatus: 
 *
 * Channel index into mv61_dma_vpmap.v_to_p will be offset for each instance.
 * This channel index within its instance is contained in chan.chan_id.
 */
struct mv61_vdma_chan {
	struct dma_chan		chan;
	struct mv61_vdma	*mv61v;
	
	struct mv61_vreg_defs	def;
	int			wrap;
	
	spinlock_t		lock;

	/* these elements are all protected by lock */
	u32			irqs;
	dma_cookie_t		completed;
	dma_cookie_t		started;
	struct list_head	active_list;
	struct list_head	complete_list;
	struct list_head	queue;
	struct mv61_stat_regs	hwstat;
	int			residue;
	enum dma_status		status;

};

/* not sure which will be fastest */ 
#if 1
#define DISPATCH_TABLE_TYPE	u32
#define DISPATCH_REGSHIFT	(5)
#endif
#if 0
#define DISPATCH_TABLE_TYPE	u16
#define DISPATCH_REGSHIFT	(4)
#endif
#if 0
#define DISPATCH_TABLE_TYPE	u8
#define DISPATCH_REGSHIFT	(3)
#endif

#define DISPATCH_REGWIDTH	(4 * sizeof(DISPATCH_TABLE_TYPE))
#define DISPATCH_MASK		(DISPATCH_REGWIDTH - 1)

#if (MV61_DMA_MAX_NR_PCHANNELS > 32)
/* If we ever hit this, the ASIC architecture should be seriously reconsidered .... */
#error The mv61_dma_dispatch pchan flag logic must be extended to support more channels.
#endif

/**
 * struct mv61_dma_dispatch - streamlined pool manager dispatch data
 * @pchans: bit per physical channel, indicates channel is available
 * @vchanw: bit per virtual channel, indicates waiting for a physical channel
 * @chancnt: number of shared channels
 * @regcnt: number of vchanw entries in use
 * @vlast: index of last vchan dispatched
 * @vwaitcnt: nr of transactions submitted but not yet dispatched
 *
 * The data in the structure will be used by the top-half interrupt handler
 * to implement "round robin" arbitration. The next waiting virtual channel 
 * will be dispatched to the next available physical channel whenever a 
 * transaction completes, even if more transactions are queued.
 *
 * TBD whether to lock here or at a higher level.
 */
struct mv61_dma_dispatch {
	/* these elements must be protected by struct mv61_dma.biglock */
	u32			pchans; /* pchan sharing status */
	DISPATCH_TABLE_TYPE	vchanw[MV61_DMA_MAX_NR_VCHANNELS / 
						DISPATCH_REGWIDTH];
	int 			vlast; /* index of last vchan dispatched */
	int			chancnt; 
	int			regcnt;
	int			vwaitcnt[0]; /* vchan waiting refcnt */
};

/**
 * struct mv61_dma_vpmap - map of virtual <-> physical channel assignment
 * @p_to_v: table of virtual channels assigned to physical channels
 * @voffset: index offset of virtual controller in v_to_p table
 * @v_to_p: table of physical channels assigned to virtual channels
 *
 * TBD whether to lock here or at a higher level.
 */
struct mv61_dma_vpmap {
	/* these elements must be protected by struct mv61_dma.biglock */
	struct mv61_vdma_chan	*p_to_v[MV61_DMA_MAX_NR_PCHANNELS];
	int			voffset[MV61_NR_VDMA_CONTROLLERS];
	struct mv61_pdma_chan	*v_to_p[0];
};

/**
 * struct mv61_dma - top level control structure
 * @device: physical platform device's "device" member 
 * @ch_regs[]: base addresses of each channel's register bank
 * @CDMAInt: base address of top interrupt status register
 * @pchannels: total number of physical channels available
 * @irq_call_cnt: cumulative count of irq handler calls for debug
 * @tasklet: bottom half tasklet invoked by interrupt handler
 * @dispatch: pool manager dispatch data
 * @desc_cachep: slab cache for the transaction descriptors
 * @lli_cachep: slab cache for the dma linked list descriptors
 * @mv61v: control structure for each of the virtual dma devices
 * @all_chains_lock: protect all_chains list using spin_lock_bh
 * @all_chains: used for cleanup to avoid mem leaks
 * @biglock: protect physical channels, mapping, dispatch
 * @chan[]: physical channel control structures
 *
 * The all_chains list head is in top struct mv61_dma, but it is never touched 
 * in the irq handler, only the tasklet. Really just need atomic list ops.
 * The kernel does provide smp-safe versions with internal spin_locks, but they 
 * still aren't bottom-half-safe, so just handle it here.
 */
struct mv61_dma {
	struct device		*dev;
	void __iomem		*ch_regs[MV61_DMA_MAX_NR_PCHANNELS];
	void __iomem		*CDMAInt;
	int			pchannels;
	u32			irq_call_cnt;
	struct tasklet_struct	tasklet;
	
	struct mv61_dma_dispatch *dispatch;
	struct mv61_dma_vpmap	*vpmap;
	struct kmem_cache	*desc_cachep;
	struct kmem_cache	*chain_cachep;
        struct mv61_vdma	*mv61v[MV61_NR_VDMA_CONTROLLERS];

	spinlock_t		all_chains_lock;
	struct list_head	all_chains;

	spinlock_t		biglock;
	int reva;
	struct mv61_pdma_chan	chan[0];
};

/**
 * struct mv61_vdma - top level virtual control structure
 * @dma: control structure used by dmaengine API 
 * @mv61p: physical dma controller
 * @vtype: id and usage type for each virtual controller instance
 * @chan[]: virtual channel control structures
 *
 * Each usage type gets its own virtual controller instance with
 * unique capabilities. 
 */
struct mv61_vdma {
	struct dma_device	dma;
	struct mv61_dma		*mv61p;
	enum mv61_vdma_type	vtype;
	struct mv61_vdma_chan	chan[0];
};

/*
 * wrappers for the reg field macros in auto-generated CDMA_regmasks.h
 * use the var version for manipulating fields in stored register copies
 * use the reg version for directly manipulating register fields
 */
#define MV61_CDMA_VAR_WR_FIELD(var, field, val) var = \
		(field##_REPLACE_VAL(var, (val & (field##_MASK >> field##_SHIFT))))

#define MV61_CDMA_VAR_RD_FIELD(var, field) (field##_MASK_SHIFT(var))

#endif

/*----------------------------------------------------------------------*/
/* Declarations for non-static cdma_main.c functions and module_params  */

extern int vdebug;
extern int wtrace;
extern int vresidue;
extern int ignore_fifo_residue;
extern int dumppausedesc;

int mv61vc_dostart(struct mv61_vdma_chan *mv61vc, struct mv61_desc *desc);
dma_cookie_t mv61vc_tx_submit(struct dma_async_tx_descriptor *tx);
void mv61vc_handle_error(struct mv61_vdma *mv61v, struct mv61_vdma_chan *mv61vc);

/*----------------------------------------------------------------------*/
/* Declarations for cdma_cyclic.c functions */

void mv61vc_cyclic_dma_tasklet_handler(struct mv61_dma 	*mv61p,
				struct mv61_vdma	*mv61v,
				struct mv61_vdma_chan	*mv61vc);

#if 0
void mv61_cyclic_dma_interrupt_handler(struct mv61_dma *mv61p, 
				struct mv61_pdma_chan *mv61pc,
				struct mv61_pdma_chan_regs *pcregs,
				struct mv61_vdma_chan *mv61vc);
#endif

struct dma_async_tx_descriptor *mv61vc_prep_dma_cyclic(struct dma_chan *chan, 
				dma_addr_t buf_addr, size_t buf_len,
				size_t period_len, 
				enum dma_transfer_direction direction,
				unsigned long flags);

/*----------------------------------------------------------------------*/
/* Declarations for cdma_pausing.c functions */

int mv61vc_resume_dma(struct mv61_vdma_chan *mv61vc, enum mv61_dma_flow_ctrl	
								flowcontrol);
int mv61vc_check_residue(struct mv61_vdma_chan *mv61vc);
int mv61vc_check_residue_running(struct mv61_vdma_chan *mv61vc);


/*----------------------------------------------------------------------*/
/* Declarations for cdma_util.c functions */

struct mv61_desc *mv61vc_first_active(struct mv61_vdma_chan *mv61vc);
bool mv61_tx_list_next(struct mv61_desc *desc);
void mv61_retire_tx(struct mv61_vdma_chan *mv61vc);
struct mv61_chain *mv61_link_get(struct mv61_vdma_chan *mv61vc);
struct mv61_desc *mv61_desc_get(struct mv61_vdma_chan *mv61vc);
void mv61_link_put(struct mv61_vdma *mv61v, struct dma_chan *chan, 
							struct mv61_chain *link);
void mv61_chain_put(struct mv61_vdma *mv61v, struct mv61_desc *desc);
void mv61_desc_put(struct mv61_desc *desc);
dma_cookie_t
mv61vc_assign_cookie(struct mv61_vdma_chan *mv61vc, struct mv61_desc *desc);
int mv61vc_prep_slave_lli(struct mv61_lli *lli, 
					struct mv61_vdma_chan *mv61vc,
					enum dma_transfer_direction direction,
					enum mv61_dma_flow_ctrl	flowcontrol,
					u32 mem, u32 len, size_t offset); 

int mv61_clear_pchannel(struct mv61_pdma_chan *mv61pc);

struct mv61_pdma_chan *mv61_vpmap_v_to_p(struct mv61_vdma_chan *mv61vc);
struct mv61_vdma_chan *mv61_vpmap_p_to_v(struct mv61_pdma_chan *mv61pc);
struct mv61_vdma_chan *mv61_vpmap_pair(struct mv61_vdma_chan *mv61vc, 
						struct mv61_pdma_chan *mv61pc);
struct mv61_vdma_chan *mv61_vpmap_unpair(struct mv61_vdma_chan *mv61vc, 
						struct mv61_pdma_chan *mv61pc);
void mv61vc_pause_dump(struct mv61_vdma_chan *mv61vc);
void mv61_link_dump(struct mv61_chain *link, struct mv61_desc *desc);
void mv61_desc_dump(struct mv61_desc *desc, char *note);
void mv61_chain_dump(struct mv61_desc *desc);
void mv61_tx_dump(struct mv61_desc *desc);
void mv61_dump_pchan(struct mv61_pdma_chan_regs *pcregs);
void mv61_cdma_dumpall(struct mv61_dma *mv61p);

/*----------------------------------------------------------------------*/
/* simple pointer translation functions */

/**
 * txd_to_mv61_desc - get our transaction control structure 
 * @txd: transaction control structure used by dmaengine API
 */
static inline struct mv61_desc *
txd_to_mv61_desc(struct dma_async_tx_descriptor *txd)
{
	return container_of(txd, struct mv61_desc, txd);
}

/**
 * dchan_to_mv61_vdma_chan - get top virtual channel control structure 
 * @chan: channel control structure used by dmaengine API
 */
static inline struct mv61_vdma_chan *dchan_to_mv61_vdma_chan(struct dma_chan *chan)
{
	return container_of(chan, struct mv61_vdma_chan, chan);
}

/**
 * ddev_to_mv61_vdma - get top virtual controller instance structure 
 * @ddev: device control structure used by dmaengine API
 */
static inline struct mv61_vdma *ddev_to_mv61_vdma(struct dma_device *ddev)
{
	return container_of(ddev, struct mv61_vdma, dma);
}

/**
 * desc_to_mv61_vdma - get top virtual controller instance structure 
 * @desc: top transaction control structure
 */
static inline struct mv61_vdma *desc_to_mv61_vdma(struct mv61_desc *desc)
{
	return ddev_to_mv61_vdma(desc->txd.chan->device);
}

/**
 * chan2dev - get vdma device 
 * @chan: channel control structure  used by dmaengine API
 *
 * There may be up to three different virtual cdma controller devices
 * registered, each with unique capabilities and channel pools.
 */
static inline struct device *chan2dev(struct dma_chan *chan)
{
	return &chan->dev->device;
}

/**
 * chan2parent - get parent device 
 * @chan: channel control structure  used by dmaengine API
 */
static inline struct device *chan2parent(struct dma_chan *chan)
{
	return chan->dev->device.parent;
}

