/*
 * Marvell Pegmatite SoC timer handling.
 *
 * Timer 0 is used as free-running clocksource, while timer 1 is
 * used as clock_event_device.
 */

#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/timer.h>
#include <linux/clockchips.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/irq.h>
#include <linux/module.h>
#include <linux/sched_clock.h>

/*
 * Timer block registers.
 */
#define TMR_CCR         0x0000 // Timer Clock Control Register
#define TMR_T0_M0       0x0004 // Timer 0 Match Register 0
#define TMR_T0_M1       0x0008 // Timer 0 Match Register 1
#define TMR_T0_M2       0x000c // Timer 0 Match Register 2
#define TMR_T1_M0       0x0010 // Timer 1 Match Register 0
#define TMR_T1_M1       0x0014 // Timer 1 Match Register 1
#define TMR_T1_M2       0x0018 // Timer 1 Match Register 2
#define TMR_T2_M0       0x001c // Timer 2 Match Register 0
#define TMR_T2_M1       0x0020 // Timer 2 Match Register 1
#define TMR_T2_M2       0x0024 // Timer 2 Match Register 2
#define TMR_CR0         0x0028 // Timer 0 Count Registers
#define TMR_CR1         0x002c // Timer 1 Count Registers
#define TMR_CR2         0x0030 // Timer 2 Count Registers
#define TMR_SR0         0x0034 // Timer 0 Status Register
#define TMR_SR1         0x0038 // Timer 1 Status Register
#define TMR_SR2         0x003c // Timer 2 Status Register
#define TMR_IER0        0x0040 // Timer 0 Interrupt Enable Register
#define TMR_IER1        0x0044 // Timer 1 Interrupt Enable Register
#define TMR_IER2        0x0048 // Timer 2 Interrupt Enable Register
#define TMR_PLVR0       0x004c // Timer 0 Preload Value Register
#define TMR_PLVR1       0x0050 // Timer 1 Preload Value Register
#define TMR_PLVR2       0x0054 // Timer 2 Preload Value Register
#define TMR_PLCR0       0x0058 // Timer 0 Preload Control Register
#define TMR_PLCR1       0x005c // Timer 1 Preload Control Register
#define TMR_PLCR2       0x0060 // Timer 2 Preload Control Register
#define TMR_ICR0        0x0074 // Timer 0 Interrupt Clear Register
#define TMR_ICR1        0x0078 // Timer 1 Interrupt Clear Register
#define TMR_ICR2        0x007c // Timer 2 Interrupt Clear Register
#define TMR_CER         0x0084 // Timers Count Enable Register
#define TMR_CMR         0x0088 // Timers Count Mode Register
#define TMR_ILR0        0x008c // Timer 0 Interrupt Length Register
#define TMR_ILR1        0x0090 // Timer 1 Interrupt Length Register
#define TMR_ILR2        0x0094 // Timer 2 Interrupt Length Register
#define TMR_CVWR0       0x00a4 // Timer 0 Counters Value Write for Read Request Register
#define TMR_CVWR1       0x00a8 // Timer 1 Counters Value Write for Read Request Register
#define TMR_CVWR2       0x00ac // Timer 2 Counters Value Write for Read Request Register
#define TMR_CRSR        0x00b0 // Timers Count Restart Register

#define MAX_DELTA		(0xfffffffe)
#define MIN_DELTA		(16)

#define TMR_CR_THRESHOLD 100

static void __iomem *timer_base;

/*
 * FIXME: the timer needs some delay to stablize the counter capture
 */
static inline uint32_t timer_read(void)
{
	u32 v1, v2, v3;

	do {
		v1 = __raw_readl(timer_base + TMR_CR1);
		v2 = __raw_readl(timer_base + TMR_CR1);
		v3 = __raw_readl(timer_base + TMR_CR1);
	} while (!likely(((v2 - v1) < TMR_CR_THRESHOLD) && ((v3 - v2) < TMR_CR_THRESHOLD)));

	return v2;
}

static u64 notrace pegmatite_read_sched_clock(void)
{
	return timer_read();
}

static irqreturn_t timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *c = dev_id;

	/*
	 * Clear pending interrupt status.
	 */
	__raw_writel(0x01, timer_base + TMR_ICR0);

	/*
	 * Disable timer 0.
	 */
	__raw_writel(0x02, timer_base + TMR_CER);

	c->event_handler(c);

	return IRQ_HANDLED;
}

static int timer_set_next_event(unsigned long delta,
				struct clock_event_device *dev)
{
	unsigned long flags;

	local_irq_save(flags);

	/*
	 * Disable timer 0.
	 */
	__raw_writel(0x02, timer_base + TMR_CER);

	/*
	 * Clear and enable timer match 0 interrupt.
	 */
	__raw_writel(0x01, timer_base + TMR_ICR0);
	__raw_writel(0x01, timer_base + TMR_IER0);

	/*
	 * Setup new clockevent timer value.
	 */
	__raw_writel(delta - 1, timer_base + TMR_T0_M0);

	/*
	 * Enable timer 0.
	 */
	__raw_writel(0x03, timer_base + TMR_CER);

	local_irq_restore(flags);

	return 0;
}

static void timer_set_mode(enum clock_event_mode mode,
			   struct clock_event_device *dev)
{
	unsigned long flags;

	local_irq_save(flags);
	switch (mode) {
	case CLOCK_EVT_MODE_ONESHOT:
	case CLOCK_EVT_MODE_UNUSED:
	case CLOCK_EVT_MODE_SHUTDOWN:
		/* disable the matching interrupt */
		__raw_writel(0x00, timer_base + TMR_IER0);
		break;
	case CLOCK_EVT_MODE_RESUME:
	case CLOCK_EVT_MODE_PERIODIC:
		break;
	}
	local_irq_restore(flags);
}

static struct clock_event_device ckevt = {
	.name		= "Pegmatite clockevent",
	.features	= CLOCK_EVT_FEAT_ONESHOT,
	.rating		= 200,
	.set_next_event	= timer_set_next_event,
	.set_mode	= timer_set_mode,
};

static cycle_t clksrc_read(struct clocksource *cs)
{
	return timer_read();
}

static struct clocksource cksrc = {
	.name		= "Pegmatite clocksource",
	.rating		= 200,
	.read		= clksrc_read,
	.mask		= CLOCKSOURCE_MASK(32),
	.flags		= CLOCK_SOURCE_IS_CONTINUOUS,
};

static void __init timer_config(void)
{
	__raw_writel(0x0, timer_base + TMR_CER); /* disable */

	__raw_writel(0, timer_base + TMR_CCR);

	/* set timer 0 to periodic mode, and timer 1 to free-running mode */
	__raw_writel(0x2, timer_base + TMR_CMR);

	__raw_writel(0x1, timer_base + TMR_PLCR0); /* periodic */
	__raw_writel(0x7, timer_base + TMR_ICR0);  /* clear status */
	__raw_writel(0x0, timer_base + TMR_IER0);

	__raw_writel(0x0, timer_base + TMR_PLCR1); /* free-running */
	__raw_writel(0x7, timer_base + TMR_ICR1);  /* clear status */
	__raw_writel(0x0, timer_base + TMR_IER1);

	/* enable timer 1 counter */
	__raw_writel(0x2, timer_base + TMR_CER);
}

static struct irqaction timer_irq = {
	.name		= "timer",
	.flags		= IRQF_DISABLED | IRQF_TIMER | IRQF_IRQPOLL,
	.handler	= timer_interrupt,
	.dev_id		= &ckevt,
};

void __init timer_init(int irq, unsigned long rate)
{
	timer_config();

	sched_clock_register(pegmatite_read_sched_clock, 32, rate);

	ckevt.cpumask = cpumask_of(0);
	ckevt.irq = irq;

	setup_irq(irq, &timer_irq);
	clocksource_register_hz(&cksrc, rate);
	clockevents_config_and_register(&ckevt, rate,
					MIN_DELTA, MAX_DELTA);
}

#ifdef CONFIG_OF
static void __init pegmatite_timer_init(struct device_node *np)
{
	struct clk *clk;
	unsigned long rate;
	int irq, ret;

	clk = of_clk_get(np, 0);
	if (IS_ERR(clk)) {
		ret = PTR_ERR(clk);
		pr_err("failed to get clock for clocksource (%d)\n", ret);
		goto out;
	}

	ret = clk_prepare_enable(clk);
	if (ret) {
		pr_err("failed to enable timer clock for clocksource (%d)\n",
		       ret);
		goto err_clk_enable;
	}
	rate = clk_get_rate(clk);

	irq = irq_of_parse_and_map(np, 0);
	if (!irq) {
		ret = -EINVAL;
		goto err;
	}
	timer_base = of_iomap(np, 0);
	if (!timer_base) {
		ret = -ENOMEM;
		goto err;
	}
	timer_init(irq, rate);
	return;

err:
	clk_disable_unprepare(clk);
err_clk_enable:
	clk_put(clk);
out:
	pr_err("Failed to get timer from device tree with error:%d\n", ret);
}

CLOCKSOURCE_OF_DECLARE(pegmatite_timer, "marvell,pegmatite-timer", pegmatite_timer_init);
#endif
