/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/
#include <linux/module.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/of_device.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/kernel.h>
#include <linux/version.h>

#include <linux/sched.h> /* omo */

#include "ipc_api.h"
char *gdbgveripcuseriface="ipc detect signal port automatically, build time xxx:"__DATE__":"__TIME__;
MODULE_LICENSE("Dual BSD/GPL");

#if 0
#define OUTPUTDBG(fmt, ...) printk(KERN_EMERG, fmt, __VA_ARGS__)
#else
#define OUTPUTDBG(fmt, ...)
#endif
extern void addDbgLog(char ind1, unsigned short ind2, unsigned int ind3, unsigned int ind4);
#if 1
#define _IPC_MEMLOG_MUST addDbgLog
#define _IPC_MEMLOG_DBG(fmt, ...)
#else
#define _IPC_MEMLOG_MUST addDbgLog
#define _IPC_MEMLOG_DBG addDbgLog
#endif

#define _FILE_NAME_ "ipc_user_iface.c"

#define IPC_NAME "ipc"
#define IPC_PORT_NAME (IPC_NAME "_port")
#define IPC_IFACE_NAME "ipc_iface"
#define PREFIX IPC_IFACE_NAME ": "

#define ENTER() pr_debug(PREFIX "ENTER %s\n", __FUNCTION__)
#define EXIT()  pr_debug(PREFIX "EXIT  %s:%d\n", __FUNCTION__, __LINE__)

static ssize_t port_export_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count);
static ssize_t port_unexport_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count);
static ssize_t port_read(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t port_write(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,13,0)
static DEVICE_ATTR(export,   S_IWUSR, NULL, port_export_set);
static DEVICE_ATTR(unexport, S_IWUSR, NULL, port_unexport_set);

static struct attribute *ipc_dev_attrs[] =
{
    &dev_attr_export.attr,
    &dev_attr_unexport.attr,
    NULL,
};

static DEVICE_ATTR(rw,   S_IWUSR | S_IRUSR, port_read, port_write);

static struct attribute *ipc_port_attrs[] =
{
    &dev_attr_rw.attr,
    NULL,
};

static struct attribute_group ipc_dev_attr_group = {
    .attrs = ipc_dev_attrs
};

static const struct attribute_group *ipc_dev_attr_groups[] = {
    &ipc_dev_attr_group,
    NULL
};

static struct attribute_group ipc_port_attr_group = {
    .attrs = ipc_port_attrs
};

static const struct attribute_group *ipc_port_attr_groups[] = {
    &ipc_port_attr_group,
    NULL
};

static struct class ipc_class =
{
    .name        = IPC_NAME,
    .owner       = THIS_MODULE,
    .dev_groups  = ipc_dev_attr_groups,
};

static struct class ipc_port_class =
{
    .name        = IPC_PORT_NAME,
    .owner       = THIS_MODULE,
    .dev_groups  = ipc_port_attr_groups,
};

#else
static struct device_attribute ipc_dev_attrs[] =
{
    __ATTR(export,   S_IWUSR, NULL, port_export_set),
    __ATTR(unexport, S_IWUSR, NULL, port_unexport_set),
    __ATTR_NULL,
};

static struct class ipc_class =
{
    .name        = IPC_NAME,
    .owner       = THIS_MODULE,
    .dev_attrs   = ipc_dev_attrs,
};

static struct device_attribute ipc_port_attrs[] =
{
    __ATTR(rw,   S_IWUSR | S_IRUSR, port_read, port_write),
    __ATTR_NULL,
};

static struct class ipc_port_class =
{
    .name        = IPC_PORT_NAME,
    .owner       = THIS_MODULE,
    .dev_attrs   = ipc_port_attrs,
};

#endif

uint8_t port_number_to_data_type(uint8_t port_number)
{
	uint8_t data_type;
	
	switch(port_number) {
	case 0x1:
		data_type=1;//leci
		break;
	case 0x2:
		data_type=2;//rsci
		break;
	case 0x3:
		data_type=3;//fax
		break;
	case 0x4:
		data_type=4; /* mnakamura */
		break;
	case 0x8:
	case 0x9:
	case 0xa:
	case 0xb:
	case 0xc:
	case 0xd:
	case 0xe:
	case 0xf:
		data_type=0;//signal
		break;
	default:
		printk(KERN_EMERG "ipc error wrong port!!!!!");
		break;
	}
	
	return data_type;
}

uint8_t data_type_to_port_number(uint8_t data_type)
{
	uint8_t port_number;
	
	switch(data_type) {
	case 0x0://signal
		port_number=0x0; //0x0 MEANS SIGNAL!
		break;
	case 0x1://leci
		port_number=1;
		break;
	case 0x2://rsci
		port_number=2;
		break;
	case 0x3://fax
		port_number=3;
		break;
	case 0x4:/* mnakamura */
		port_number=4;
		break;
	}
	
	return port_number;
}

uint32_t get_device_index(struct device *dev)
{
    void *data = dev_get_drvdata(dev);
    return (uint32_t)data;
}

void recv_callback(ipc_drvr_handle handle, void *user_param, uint8_t data_type, uint32_t command, void *buffer, uint16_t length)
{
    if ((buffer != NULL) && (length > 0))
    {
        pr_alert("IPC_IFACE (%s.%d) received cmd %d, buffer 0x%p, len %d\n", ipc_get_device_name(((uint32_t)user_param) >> 8), ((uint32_t)user_param) & 0xFF, command, buffer, length);
        if (command == 255)
        {
            pr_alert("\"%s\"\n", (char*)buffer);
        }
    }
    else
    {
        pr_alert("IPC_IFACE (%s.%d) received cmd %d, param %d\n", ipc_get_device_name(((uint32_t)user_param) >> 8), ((uint32_t)user_param) & 0xFF, command, (int)buffer);
    }
}

#define MAX_PORT	(16)
#define MAX_PORT_ARY_SIZE	(MAX_PORT+1)

#define PORT_DATA_BUF_SIZE 16
static struct port_recv_data_st {
	struct port_buf_st {
		unsigned int buf[PORT_DATA_BUF_SIZE];
		short wpos;
		short rpos;
		int cnt;
	} port_buf;
	wait_queue_head_t wqh_nonempty;
	int flag_nonempty;
	wait_queue_head_t wqh_nonfull;
	int flag_nonfull;
	struct semaphore sem;
	bool is_exported;
} g_recv_port_data[MAX_PORT_ARY_SIZE];

static int add_port_buf(struct port_buf_st *pbufst, unsigned int data)
{
	if (pbufst->cnt >= PORT_DATA_BUF_SIZE) {
		//buffer full
		return -1;
	} else {
		pbufst->buf[pbufst->wpos] = data;
		pbufst->wpos++;
		if (pbufst->wpos >= PORT_DATA_BUF_SIZE) {
			pbufst->wpos = 0;
		}
		pbufst->cnt++;
	}
	return pbufst->cnt;
}

static int rmv_port_buf(struct port_buf_st *pbufst, unsigned int *data)
{
	if (pbufst->cnt == 0) {
		//buffer full
		//ugh! assert cnt < 0
		return -1;
	} else {
		*data = pbufst->buf[pbufst->rpos];
		pbufst->rpos++;
		if (pbufst->rpos >= PORT_DATA_BUF_SIZE) {
			pbufst->rpos = 0;
		}
		pbufst->cnt--;
	}
	return pbufst->cnt;
}

void add_data_port_buf(uint8_t port_number, uint32_t command)
{
	down(&g_recv_port_data[port_number].sem);
	
	//producer wait buffer become nonfull
	while(g_recv_port_data[port_number].port_buf.cnt >= PORT_DATA_BUF_SIZE) {
		
	    up(&g_recv_port_data[port_number].sem);
		wait_event_timeout(g_recv_port_data[port_number].wqh_nonfull, (g_recv_port_data[port_number].flag_nonfull != 0), 100);
	    down(&g_recv_port_data[port_number].sem);
		g_recv_port_data[port_number].flag_nonfull = 0;
	}
	
	//producer add data to port buffer
	add_port_buf(&g_recv_port_data[port_number].port_buf, command);
_IPC_MEMLOG_DBG('C', __LINE__, port_number, g_recv_port_data[port_number].port_buf.cnt);
	
	//wakeup consumer
	g_recv_port_data[port_number].flag_nonempty=1;
	wake_up(&g_recv_port_data[port_number].wqh_nonempty);
	
	up(&g_recv_port_data[port_number].sem);
}

void recv_callback_ricoh(ipc_drvr_handle handle, void *user_param, uint8_t data_type, uint32_t command, void *buffer, uint16_t length)
{
//	printk(KERN_EMERG "IPC_IFACE (%s.%d) received cmd %x, buffer 0x%p, len %d\n", ipc_get_device_name(((uint32_t)user_param) >> 8), ((uint32_t)user_param) & 0xFF, command, buffer, length);
	
	uint8_t port_number;
	
	port_number = data_type_to_port_number(data_type); //tarnslate
	
	if (port_number != 0) {
		//non signal data
_IPC_MEMLOG_DBG('C', __LINE__, port_number, g_recv_port_data[port_number].port_buf.cnt);
		add_data_port_buf(port_number, command);
	} else {
		//signal data
		int i;
_IPC_MEMLOG_DBG('C', __LINE__, port_number, g_recv_port_data[port_number].port_buf.cnt);
			for(i=0x8; i<0x10; i++) { //add signal data to SIGNAL port buffers (SIGNAL port buffers are 0x8-0xf)
			if (g_recv_port_data[i].is_exported == true) {
				add_data_port_buf(i, command);
			}
		}
	}
_IPC_MEMLOG_DBG('C', __LINE__, port_number, g_recv_port_data[port_number].port_buf.cnt);
}

extern uint8_t ipc_get_port_number(ipc_drvr_handle handle);

ssize_t ipc_recv_sai(struct device *dev, uint32_t *buf, uint8_t port_number)
{
	OUTPUTDBG("sai:%s(%d)%s:sta\n", _FILE_NAME_, __LINE__, __func__);
	ENTER();
	
_IPC_MEMLOG_DBG('R', __LINE__, port_number, 0x0);
	down(&g_recv_port_data[port_number].sem);
	
_IPC_MEMLOG_DBG('R', __LINE__, port_number, 0x0);
	//consumer wait buffer become nonempty
	while(g_recv_port_data[port_number].port_buf.cnt == 0) {
	    up(&g_recv_port_data[port_number].sem);
		wait_event_timeout(g_recv_port_data[port_number].wqh_nonempty, (g_recv_port_data[port_number].flag_nonempty != 0), 100);
	    down(&g_recv_port_data[port_number].sem);
		g_recv_port_data[port_number].flag_nonempty = 0;
	}
	
_IPC_MEMLOG_DBG('R', __LINE__, port_number, 0x0);
	//consumer get data from port buffer
	rmv_port_buf(&g_recv_port_data[port_number].port_buf, buf);
	
	//wakeup producer
	g_recv_port_data[port_number].flag_nonfull=1;
	wake_up(&g_recv_port_data[port_number].wqh_nonfull);
	
	up(&g_recv_port_data[port_number].sem);
	
_IPC_MEMLOG_DBG('R', __LINE__, port_number, *buf);
	EXIT();
	OUTPUTDBG("sai:%s(%d)%s:fin ret 4\n", _FILE_NAME_, __LINE__, __func__);
	return 4;
}

extern void ipc_printDbgLog(void);
static ssize_t port_read(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint32_t *pintbuf;
    uint32_t data;
    ssize_t ret;
	uint8_t port_number;

	port_number = ipc_get_port_number(dev_get_drvdata(dev));
_IPC_MEMLOG_MUST('R', 1, port_number, 0);
    if (buf == NULL) {
_IPC_MEMLOG_MUST('E', __LINE__, port_number, 0);
        return 0;
    }
    pintbuf = (uint32_t *)buf;
    OUTPUTDBG("sai:%s(%d)%s:1\n", _FILE_NAME_, __LINE__, __func__);

    ret = ipc_recv_sai(dev, &data, port_number);

    if (ret != 0) {
//        *pintbuf = data;
    	
    	*(buf+0) = (char)data;
    	*(buf+1) = (char)(data >> 8);
    	*(buf+2) = (char)(data >> 16);
    	*(buf+3) = (char)(data >> 24);
    	
        OUTPUTDBG("sai:%s(%d)%s:fin ok %x, %d\n", _FILE_NAME_, __LINE__, __func__, *pintbuf, ret);
    }
_IPC_MEMLOG_MUST('R', 2, port_number, *pintbuf);

    return ret;
}

static unsigned int sLastdata = 0;

static ssize_t port_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int cmd = 0;
    unsigned int param = 0;
    ipc_error_type_t err;
	uint8_t port_number;
	uint8_t data_type;
	
	port_number = ipc_get_port_number(dev_get_drvdata(dev));
_IPC_MEMLOG_MUST('W', 1, port_number, *((uint32_t *)buf));
_IPC_MEMLOG_MUST('W', 2, port_number, count);
    if (buf == NULL) {
_IPC_MEMLOG_MUST('E', __LINE__, port_number, 0);
        return 0;
    }
    if (count != 4) {
_IPC_MEMLOG_MUST('E', __LINE__, port_number, 0);
        return 0;
    }
    {
        int val;
        val=*((int *)buf);
        if (val == 0x0a313233)
        {
            ipc_printDbgLog();
			return count;
        }
    }
    OUTPUTDBG("sai:%s(%d)%s:sta:%d,%s\n", _FILE_NAME_, __LINE__, __func__, count, buf);

    ENTER();

	param = 0;
	param = param | (*(buf+0) << 0);
	param = param | (*(buf+1) << 8);
	param = param | (*(buf+2) << 16);
	param = param | (*(buf+3) << 24);
	param = param & 0x00ffffff;
_IPC_MEMLOG_DBG('W', __LINE__, param, 0);

	if (port_number == 8) {
		//merge R4.8, 9, 10, 11 signal data
		param = sLastdata & (~0x00001000) | (param & 0x00001000);
		sLastdata = param; //save last signal data of R4.8
	}
	if (port_number == 9) {
		//merge R4.8, 9, 10, 11 signal data
		param = sLastdata & (~0x00000100) | (param & 0x00000100);
		sLastdata = param; //save last signal data of R4.9
	}
	if (port_number == 10) {
		//merge R4.8, 9, 10, 11 signal data
		param = sLastdata & (~0x00800000) | (param & 0x00800000);
		sLastdata = param; //save last signal data of R4.8
	}
	if (port_number == 11) {
		//merge R4.8, 9, 10, 11 signal data
		param = sLastdata & (~0x00400000) | (param & 0x00400000);
		sLastdata = param; //save last signal data of R4.8
	}
	
	data_type = port_number_to_data_type(port_number);
    err = ipc_send(dev_get_drvdata(dev), cmd, (void *)&param, 0, data_type); //translted from port_number to data_type

    if ( err == e_IPC_NO_LISTENER )
    {
        pr_alert( PREFIX "message sent, but nothing is attached to that port on the remote side.\n" );
    }

    EXIT();
    return count;
}

static ssize_t port_export_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    unsigned long port_number;
    uint32_t device_index = get_device_index(dev);
	register unsigned first;
	void *vp;

    ENTER();

    if ((kstrtoul(buf, 10, &port_number) == 0) && (port_number > 0) && (port_number < 16))
    {
        ipc_drvr_handle  port_handle;
        struct device   *port_dev;

    	if ( (0 < port_number) && (port_number < MAX_PORT) ) {
			g_recv_port_data[port_number].port_buf.cnt=0;
			g_recv_port_data[port_number].port_buf.wpos=0;
			g_recv_port_data[port_number].port_buf.rpos=0;
			g_recv_port_data[port_number].is_exported=true;
    	}
    	init_waitqueue_head(&g_recv_port_data[port_number].wqh_nonempty);
    	init_waitqueue_head(&g_recv_port_data[port_number].wqh_nonfull);
	    sema_init( &g_recv_port_data[port_number].sem,  1 );
    	
        port_handle = ipc_attach(device_index, (uint8_t)port_number, recv_callback_ricoh, (void*)((device_index << 8) | port_number));

        port_dev = device_create(&ipc_port_class, dev, MKDEV(0,0), port_handle, "%s.%d", dev_name(dev), (int)port_number);
        if (IS_ERR( port_dev ))
        {
            pr_err(PREFIX "failed to create device for exported port %d\n", (int)port_number);
        }
    }
	
    if (port_number == 250)
	{
		vp = ioremap(0x1fa00000, 4096*5);
		first=0x1fa00000;
//		asm("mcr        p15, 0, %0, c7, c6, 1" : : "r" (v));
		asm("mcr	p15, 0, %0, c7, c6, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}
    if (port_number == 251)
	{
//		printk(KERN_EMERG "mcr c7 c10 6\n");
		vp = ioremap(0x1fa00000, 4096*5);
		first=0x1fa00000;
//		asm("mcr        p15, 0, %0, c7, c10, 1" : : "r" (v));
		asm volatile("mcr	p15, 0, %0, c7, c10, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}
    if (port_number == 252)
	{
		vp = ioremap(0x3ef00000, 4096*5);
		first=0x3ef00000;
//		asm("mcr        p15, 0, %0, c7, c6, 1" : : "r" (v));
		asm("mcr	p15, 0, %0, c7, c6, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}
    if (port_number == 253)
	{
//		printk(KERN_EMERG "mcr c7 c10 6\n");
		vp = ioremap(0x3ef00000, 4096*5);
		first=0x3ef00000;
//		asm("mcr        p15, 0, %0, c7, c10, 1" : : "r" (v));
		asm volatile("mcr	p15, 0, %0, c7, c10, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}
    if (port_number == 254)
	{
		vp = ioremap(0x7ef00000, 4096*5);
		first=0x7ef00000;
//		asm("mcr        p15, 0, %0, c7, c6, 1" : : "r" (v));
		asm("mcr	p15, 0, %0, c7, c6, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}
    if (port_number == 255)
	{
//		printk(KERN_EMERG "mcr c7 c10 6\n");
		vp = ioremap(0x7ef00000, 4096*5);
		first=0x7ef00000;
//		asm("mcr        p15, 0, %0, c7, c10, 1" : : "r" (v));
		asm volatile("mcr	p15, 0, %0, c7, c10, 1 @ flush_pgd"
        "DSB ;"
        "ISB ;"
		    : : "r" (vp));
		iounmap(vp);
	}

    EXIT();

    return count;
}

static int match_port( struct device *dev, void *data )
{
    if (strcmp(dev_name(dev), (char*)data) == 0)
        return 1;
    return 0;
}

static ssize_t port_unexport_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    unsigned long port_number;

    ENTER();

    if ((kstrtoul(buf, 10, &port_number) == 0) && (port_number > 0) && (port_number < 256))
    {
        struct device *port_dev;
        char          *port_name;

        port_name = kmalloc(strlen(dev_name(dev)) + count + 2, GFP_KERNEL );
        if (port_name != NULL)
        {
            sprintf(port_name, "%s.%d", dev_name(dev), (int)port_number);
            port_dev = device_find_child( dev, port_name, match_port);
            if (port_dev != NULL)
            {
                ipc_detach(dev_get_drvdata(port_dev));
                device_unregister(port_dev);
                put_device(port_dev);
            }
            kfree(port_name);
        }
    }

    EXIT();
    return count;
}

static int ipc_user_iface_init(void)
{
    int retval = 0;
    int i;
    int num_devices = ipc_get_num_devices();

    ENTER();

    pr_debug(PREFIX "loading driver\n");

	for (i = 0; i < MAX_PORT; i++) {
		g_recv_port_data[i].is_exported=false;
	}

    class_register(&ipc_class);

    for (i = 0; i < num_devices; i++)
    {
        device_create(&ipc_class, NULL, MKDEV(0, 0), (void *)i, ipc_get_device_name(i));
    }

    class_register(&ipc_port_class);

    EXIT();
    return retval;
}

static void ipc_user_iface_exit(void)
{
    struct class_dev_iter iter;
    struct device *dev;

    ENTER();

    class_unregister(&ipc_port_class);

    class_dev_iter_init(&iter, &ipc_class, NULL, NULL);
    while ((dev = class_dev_iter_next(&iter)))
    {
        device_unregister(dev);
        put_device(dev);
    }
    class_dev_iter_exit(&iter);

    class_unregister(&ipc_class);

    pr_debug(PREFIX "removed IPC driver\n");

    EXIT();
}

module_init(ipc_user_iface_init);
module_exit(ipc_user_iface_exit);

