/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>


extern int pegmatite_reg_check(void);

static struct regulator *img_regulator = 0;

struct imagepower_drvr {
	int Major;
	int Minor;
	struct cdev Cdev;
	dev_t Dev;
	struct class *Class;
	struct device *dev;
	struct platform_device *pdev; // = dma_alloc_dev
	//dma_alloc_dev_t *dma_alloc_dev;
};

static struct imagepower_drvr imagepower_drvr;

static int imagepower_open(struct inode *inode, struct file *filep)
{
	int retval = -EBUSY;
    
	if (!filep->private_data) {
		filep->private_data = img_regulator;
		if (img_regulator && !regulator_enable( img_regulator )) {
			//printk(KERN_ERR "Imagepower enabled image regulator %p\n", img_regulator);
			retval = 0;
		} else { 
			printk(KERN_ERR "Imagepower can't enable image regulator %p\n", img_regulator);
		}
	}
	return retval;
}
static int imagepower_close(struct inode *inode, struct file *filep)
{
	struct regulator *r = filep->private_data;
	if (r) {
		regulator_disable(r);
		filep->private_data = 0;
	}
	return 0;
}
static int imagepower_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr)
{
	return -EINVAL;
}
static int imagepower_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
	return -EINVAL;
}

struct file_operations imagepower_fops =
{
	.owner = THIS_MODULE,
	.open = imagepower_open,
	.release = imagepower_close,
	.read = imagepower_read,
	.write = imagepower_write,
};

static int imagepower_platform_probe(struct platform_device *pdev)
{
    struct device *dev;

    //printk(KERN_ERR "probe imagepower %p\n", pdev);
    if (!(dev = (struct device *)kzalloc(sizeof(*dev), GFP_KERNEL))) {
        dev_err(&pdev->dev, "imagepower kzalloc failed\n");
        return(-ENOMEM);
    }
    if (pegmatite_reg_check())
	    img_regulator = regulator_get(imagepower_drvr.dev, "pegmatite_imgpipe");//&pdev->dev, "pegmatite_imgpipe");

    //printk(KERN_ERR "probe regulator_get %p\n", img_regulator);
    
    
    platform_set_drvdata(pdev, dev);
    return 0;
}

static int imagepower_platform_remove(struct platform_device *pdev)
{
    // remove dev handle and class
    cdev_del(&imagepower_drvr.Cdev);
    device_destroy(imagepower_drvr.Class, MKDEV(imagepower_drvr.Major, imagepower_drvr.Minor));
    class_destroy(imagepower_drvr.Class);


    if (pdev)
    {
        struct regulator *r = platform_get_drvdata(pdev);
	regulator_put(r);
	img_regulator = 0;
        platform_set_drvdata(pdev, NULL);
    }
    return 0;
}

static struct platform_driver imagepower_platform_driver =
{
	.probe = imagepower_platform_probe,
	.remove  = imagepower_platform_remove,
	.driver  = {
		.name  = "pegmatite-imagepower",
		.owner = THIS_MODULE,
	},
};




static int __init imagepower_platform_init(void)
{
	int err;
	struct device *dev;

	imagepower_drvr.Minor = 1;
	imagepower_drvr.Class = class_create(THIS_MODULE, "imagepower");
	if (IS_ERR(imagepower_drvr.Class)) {
		printk(KERN_ERR "can't register imagepower character driver CLASS \n");
		return PTR_ERR(imagepower_drvr.Class);
	}

	if ((err = alloc_chrdev_region(&imagepower_drvr.Dev, 0, 1, "imagepower devices"))) {
		class_destroy(imagepower_drvr.Class);
		printk(KERN_ERR "can't register imagepower character driver %d\n", err);
		return(err);
	}

	imagepower_drvr.Major = MAJOR(imagepower_drvr.Dev);

	imagepower_drvr.dev = device_create(imagepower_drvr.Class, NULL, MKDEV(imagepower_drvr.Major, 1),
					    NULL, "imagepower");

	cdev_init( &imagepower_drvr.Cdev, &imagepower_fops);
	imagepower_drvr.Cdev.owner = THIS_MODULE;
	imagepower_drvr.Cdev.ops   = &imagepower_fops;

	if (cdev_add(&imagepower_drvr.Cdev, MKDEV(imagepower_drvr.Major, 1), 1) < 0) {
		printk(KERN_ERR "can't add imagepower device driver\n");
		return(-1);
	}

	if ( (err = platform_driver_register(&imagepower_platform_driver) ) )
		goto exit;

	if (pegmatite_reg_check())  // force this if probe isn't called.  Todo: why isn't probe being called.
	    img_regulator = regulator_get(imagepower_drvr.dev, "pegmatite_imgpipe");
	if (!img_regulator)
		printk(KERN_ERR "driver init failed to get regulator %p\n", img_regulator);
    

	//printk(KERN_ERR "imagepower driver init OK \n");
	return 0;
exit:
	printk(KERN_ERR "imagepower driver init failure\n");
	return err;
}
module_init(imagepower_platform_init);

static void __exit imagepower_platform_exit(void)
{
    platform_driver_unregister(&imagepower_platform_driver);
    regulator_put(img_regulator);
    img_regulator = 0;
}
module_exit(imagepower_platform_exit);

MODULE_AUTHOR("Copyright (c) 2015 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell Image power island Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION("2015_June_1");

