/*
**************************************************************************************
*
* Copyright (c) 2013 Marvell International, Ltd. All Rights Reserved
*
**************************************************************************************
*
* Marvell GPL License
*
* This file licensed to you in accordance with the terms and conditions of the General
* Public License Version 2, June 1991 (the "GPL License"). You can redistribute it
* and/or modify it under the terms of the GPL License; either version 2 of the License,
* or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful, but WITHOUT ANY
* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
* PARTICULAR PURPOSE. See the GPL License for more details.
*
* You should have received a copy of the GNU General Public License along with this
* program. If not, see http://www.gnu.org/licenses/.
*
**************************************************************************************
*/


/** 
 *  \brief The hips driver is used to control power to the hips channels.  As long
 *  as there is any open handle to a given hips driver the corresponding
 *  channel is powered on.  When the open count drops to 0 the channel
 *  is powered off.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/cdev.h>
//#include <linux/device.h>
//#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of_device.h>
#include <linux/of_address.h>

#include "hips_pll_api.h"

#include "HiPS_regmasks.h"
#include "HiPS_regstructs.h"

#define HIPS_DRIVER_NAME "hips_pll"
#define HIPS_NUM_DEVICES 8

struct _hips_dev {
    struct mutex lock;
    struct cdev char_dev;
    struct platform_device *platform_dev;
    dev_t dev_num;
    int open_count;
    int channel;
};

typedef struct _hips_driver {
    dev_t dev_num_base;
    struct class *Class;
    hips_dev* hips_devices[HIPS_NUM_DEVICES];
    HIPS_REGS_t *pHIPS_REGS;
    int refcnt;
    uint32_t phys_addr; // physical address of the subblock 
    uint32_t addr_size; // size of address space in the subblock
}hips_driver;

static hips_driver hips_drv = {0}; 

///////////////////////////////////////////////////////////
/// Common Routines

static int hips_open(hips_dev* mydev)
{
    BUG_ON(hips_drv.pHIPS_REGS == 0);
    BUG_ON(mydev == 0);

    mutex_lock(&mydev->lock);

    mydev->open_count++;
    hips_drv.pHIPS_REGS->Chan_Enable |= 1<<mydev->channel;  // enable the channel.

    mutex_unlock(&mydev->lock);

    return 0;
}

static int hips_close(hips_dev* mydev)
{
    BUG_ON(hips_drv.pHIPS_REGS == 0);
    BUG_ON(mydev == 0);

    mutex_lock(&mydev->lock);

    if(--mydev->open_count == 0)
    {
        // disable the channel when last open handle closes
        hips_drv.pHIPS_REGS->Chan_Enable &= ~(1<<mydev->channel);
    }

    mutex_unlock(&mydev->lock);

    return 0;
}

///////////////////////////////////////////////////////////
/// Kernel Driver Direct Interface

hips_dev* hips_kopen( unsigned int channel )
{
    hips_dev* mydev = 0;
    printk(KERN_DEBUG "%s\n", __FUNCTION__ );

    if(channel <= HIPS_NUM_DEVICES)
    {
        mydev = hips_drv.hips_devices[channel];
        if(hips_open(mydev) != 0)
        {
            mydev = 0;
        }
    }

    return mydev;
}
EXPORT_SYMBOL(hips_kopen);

int hips_kclose( hips_dev* mydev )
{
    printk(KERN_DEBUG "%s\n", __FUNCTION__ );
    BUG_ON(mydev == 0);
    BUG_ON(mydev != hips_drv.hips_devices[mydev->channel]);

    return hips_close(mydev);
}
EXPORT_SYMBOL(hips_kclose);

///////////////////////////////////////////////////////////
/// Character Driver Interface

static int hips_cdev_open( struct inode *inode, struct file *filep )
{
    hips_dev* mydev;
    printk(KERN_DEBUG "%s\n", __FUNCTION__ );

    mydev = (hips_dev *)container_of(inode->i_cdev, hips_dev, char_dev);
    BUG_ON(mydev != hips_drv.hips_devices[mydev->channel]);
    filep->private_data = mydev;

    return hips_open(mydev);
}

static int hips_cdev_release( struct inode *inode, struct file *filep )
{
    hips_dev *mydev;
    printk(KERN_DEBUG "%s\n", __FUNCTION__ );

    mydev = filep->private_data;

    return hips_close(mydev);
}

static const struct file_operations hips_fops = {
    .owner = THIS_MODULE,
    .open = hips_cdev_open,
    .release = hips_cdev_release
};

///////////////////////////////////////////////////////////
/// Platform Driver Interface

static int hips_device_init(hips_dev *mydev, int id)
{
    if (id >= HIPS_NUM_DEVICES) {
        printk(KERN_ERR "to many hips devices found\n");
        return(-EIO);
    }

    if (IS_ERR(device_create(hips_drv.Class, NULL, mydev->dev_num,
                                       NULL, "hips%d", id))) {
        printk(KERN_ERR "failed to create device: hips%d\n", id);
        return(-1);
    }

    cdev_init( &mydev->char_dev, &hips_fops);
    mydev->char_dev.owner = THIS_MODULE;
    mydev->char_dev.ops   = &hips_fops;

    if (cdev_add(&mydev->char_dev, mydev->dev_num, 1) < 0) {
        printk(KERN_ERR "can't add hips%d device driver\n", id);
        return(-1);
    }

    mydev->open_count = 0;
    mydev->channel = id;
    mutex_init(&mydev->lock);

    return(0);
}

static int hips_platform_probe(struct platform_device *pdev)
{
    int              retval = 0;
    int instance;
    struct resource	 res;
    hips_dev*        mydev;
    bool mem_region_locked = false;
    bool mem_region_mapped = false;
    struct device_node *node;

    printk(KERN_DEBUG "%s: device '%s' registration begun\n", __func__, pdev->name);

    node = pdev->dev.of_node; 
    if(!node)
    {
        dev_err(&pdev->dev, "Could not find device node\n");
        return -ENODEV;
    }

    // get the variable instance - so we know which block instance it is
    if (of_property_read_u32(node, "instance", &instance))
    {
        dev_err(&pdev->dev, "reading instance failed\n");
        return -ENODEV;
    }

    if(instance >= HIPS_NUM_DEVICES || hips_drv.hips_devices[instance] != 0)
    {
        dev_err(&pdev->dev, " device '%s' already exists or out-of-bounds (instance=%d)\n", pdev->name, instance);
        return -EEXIST;
    }

    if (!(mydev = (hips_dev *)kzalloc(sizeof(hips_dev), GFP_KERNEL))) {
        dev_err(&pdev->dev, "hips_pll kzalloc failed\n");
        return(-ENOMEM);
    }

    mydev->dev_num = MKDEV(MAJOR(hips_drv.dev_num_base), instance);

    // we use same register set for all driver device instances, so only map for first device
    if(!hips_drv.refcnt)
    {
        retval = of_address_to_resource(node, 0, &res);
        if (retval < 0)
        {
            dev_err(&pdev->dev, "hips_pll can't get resource for node (retval=0x%x)\n", retval);
            goto failed;
        }

        if (!request_mem_region(res.start, resource_size(&res), "hipspll"))
        {
            dev_err(&pdev->dev, "hipspll request_mem_region failed\n");
            retval = -EBUSY;
            goto failed;
        } 
        mem_region_locked = true;

        hips_drv.pHIPS_REGS = (HIPS_REGS_t*)of_iomap(node, 0); // get virtual address
        if(hips_drv.pHIPS_REGS == NULL)
        {
            dev_err(&pdev->dev, "hipspll of_iomap failed\n");
            goto failed;
        }
        mem_region_mapped = true;
        hips_drv.phys_addr = res.start;
        hips_drv.addr_size = resource_size(&res);

        BUG_ON(retval != 0);
        BUG_ON(hips_drv.refcnt != 0);
    }

    if ((retval = hips_device_init(mydev, instance)))
    {
        dev_err(&pdev->dev, "hips_platform_probe failed\n");
        goto failed;
    }

    // success

    hips_drv.refcnt++;

    // Remember the platform and device ID's for this instance...
    mydev->platform_dev = pdev;
    hips_drv.hips_devices[instance] = mydev;

    platform_set_drvdata(pdev, mydev);
    printk(KERN_INFO "hips%d: device probe complete\n", instance);

    return 0;

failed:

    if(mem_region_locked)
    {
        if(mem_region_mapped)
        {
            iounmap(hips_drv.pHIPS_REGS);
            hips_drv.pHIPS_REGS = 0;
            hips_drv.phys_addr = 0;
            hips_drv.addr_size = 0;
        }
        release_mem_region(res.start, resource_size(&res));
    }
    kfree(mydev); 
    return retval;
}


static int hips_platform_remove(struct platform_device *pdev)
{
    hips_dev *mydev;
    struct device_node *node;
    int instance;

    printk(KERN_INFO "%s: driver platform remove begun\n", __func__);

    node = pdev->dev.of_node; 
    if(!node)
    {
        dev_err(&pdev->dev, "Could not find device node\n");
        return -ENODEV;
    }

    // get the variable instance - so we know which block instance it is
    if (of_property_read_u32(node, "instance", &instance))
    {
        dev_err(&pdev->dev, "reading instance failed\n");
        return -ENODEV;
    }

    if(instance >= HIPS_NUM_DEVICES || hips_drv.hips_devices[instance] == 0)
    {
        dev_err(&pdev->dev, " device '%s' does not exist or out-of-bounds (instance=%d)\n", pdev->name, instance);
        return -EEXIST;
    }

    mydev = (hips_dev*)platform_get_drvdata(pdev);

    mutex_destroy(&mydev->lock);

    cdev_del(&mydev->char_dev);
    device_destroy(hips_drv.Class, mydev->dev_num);

    hips_drv.refcnt--;
    if(!hips_drv.refcnt)
    {
        BUG_ON(!hips_drv.pHIPS_REGS);
        BUG_ON(!hips_drv.phys_addr);
        BUG_ON(!hips_drv.addr_size);
        iounmap(hips_drv.pHIPS_REGS);    
        release_mem_region(hips_drv.phys_addr, hips_drv.addr_size);
    }

    printk(KERN_INFO "hips%d: device remove complete\n", instance);

    hips_drv.hips_devices[instance] = 0;

    kfree(mydev);
    platform_set_drvdata(pdev, NULL);

	return 0;
}

static int hips_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    printk(KERN_INFO "%s: driver platform suspend %d\n", __func__, state.event);
	return (0);
}

static int hips_platform_resume(struct platform_device *pdev)
{
    printk(KERN_INFO "%s: driver platform resume begun\n", __func__);
    return(0);
}

#ifdef CONFIG_OF
static const struct of_device_id mrvl_hipspll_dt_match[] = {

    { .compatible = "marvell-hipspll"},
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_hipspll_dt_match);
#endif

static struct platform_driver hips_platform_driver =
{
    .probe   = hips_platform_probe,
    .remove  = hips_platform_remove,
    .suspend = hips_platform_suspend,
    .resume  = hips_platform_resume,
    .driver  = {
        .name  = HIPS_DRIVER_NAME,
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_hipspll_dt_match),
    }
};

static int __init hips_init(void)
{
    int retcode = 0;
    printk(KERN_INFO "hips_init\n");

    memset(&hips_drv, 0, sizeof(hips_driver));

    hips_drv.Class = class_create(THIS_MODULE, HIPS_DRIVER_NAME);
    if (IS_ERR(hips_drv.Class)) 
    {
        retcode = PTR_ERR(hips_drv.Class);
        printk(KERN_ERR "unable to create hips_pll class %d\n", retcode);
        goto EXIT;
    }

    retcode = alloc_chrdev_region( &hips_drv.dev_num_base, 0, HIPS_NUM_DEVICES, HIPS_DRIVER_NAME );
    if( retcode != 0 ) 
    {
        printk(KERN_ERR "%s: error allocating chrdev region\n", __func__);
        goto EXIT_CHRDEV;
    }

    if ((retcode = platform_driver_register(&hips_platform_driver)))
    {
        printk(KERN_ERR "%s: error registering hips platform driver\n", __func__);
        goto EXIT_PLATFORM;
    }

    return 0;

EXIT_PLATFORM:
    unregister_chrdev_region(hips_drv.dev_num_base, HIPS_NUM_DEVICES);
EXIT_CHRDEV:
    class_destroy(hips_drv.Class);
EXIT:

    return retcode;
}

static void __exit hips_exit(void)
{
    platform_driver_unregister(&hips_platform_driver);
    unregister_chrdev_region(hips_drv.dev_num_base, HIPS_NUM_DEVICES);
    class_destroy(hips_drv.Class);
    printk(KERN_INFO "hips_exit\n");
}


module_init(hips_init);
module_exit(hips_exit);

MODULE_AUTHOR("Jay Shoen");
MODULE_DESCRIPTION("HiPS PLL Driver");
MODULE_LICENSE("GPL");

