/*

Neonode zForce Touchscreen Specification v3.1

*/
#ifndef _ZFORCE_H_
#define _ZFORCE_H_

#include <common.h>

#define DEFAULT_CMD_TIMEOUT_MS (1000)
#define ZF_GPIO_OUTPUT 1

#define ZF_GPIO_INPUT 0

#define DUAL_ENABLE 0x01

#define ZF_DEFAULT_VERBOSITY 0

#define PACKED __attribute__ ((__packed__))



typedef struct {
   uint8_t FrameStart;
   uint8_t DataSize;
} PACKED zf_packet_hdr_t;

typedef struct {
   uint8_t CommandID;
   uint8_t CommandData[0];
} PACKED zf_packet_data_t;

typedef struct {
   zf_packet_hdr_t PacketHdr;
   zf_packet_data_t PacketData;
} PACKED zf_packet_t;

typedef struct {
   uint8_t CommandID;
   int	pending;
} cmd_status_t;

typedef struct {
    uint16_t major;
    uint16_t minor;
    uint16_t build;
    uint16_t revision;
    uint8_t  protocol_major;
    uint8_t  protocol_minor;
} zf_version_t;

#define ZF_FRAME_START (0xee)

#define TXBUFFSIZE	16
#define RXBUFFSIZE	256
#define RESPBUFFSIZE	256
#define NOTIFBUFFSIZE	256

#if (ZF_DEFAULT_VERBOSITY > 0)
#define PRINTZFDBG(...)   if(zf && zf->verbose) printf(__VA_ARGS__)
#else
#define PRINTZFDBG(...)
#endif

typedef struct {
    int version;
    int type;
    int subtype;
    int cfg;
} zf_id_straps_t;

#define SUBTYPE_FLAG_LVDS_OPTION1 (1 << 1)

typedef struct {
    int port;
    int bit;
} zf_gpio_t;

/**
 * struct zf_ctrl - top zforce control structure
 * @straps: state of id straps read during init
 * @fw_version: version info extracted from status report
 * @pre_api5_fw; flag to indicate early, incomplete firmware
 * @verbose: enable verbose debug
 * @refcount: client / reference counter (not sure if this will ever exceed 1)
 * @cmdstat: struct used for tracking/identifying command/response packets
 * @packet_cachep: slab cache pointer for packets not dispositioned by irq handler
 * @txbuf: buffer for sending packet to zforce
 * @rxbuf: raw buffer for receiving packet from zforce
 * @respbuf: buffer for packet identified as command response
 * It is illegal to send multiple commands to the zforce controller without
 * waiting for a response.
 */
typedef struct zf_ctrl {
   zf_id_straps_t	straps;
   zf_version_t   fw_version;
   int            pre_api5_fw;
   int			   verbose;
   int			   refcount;
   cmd_status_t	cmdstat;
   unsigned char  *packets;
   uint8_t		   txbuf[TXBUFFSIZE];
   uint8_t		   rxbuf[RXBUFFSIZE];
   uint8_t		   respbuf[RXBUFFSIZE];
} zf_ctrl_t;

/**
 * enum zf_command - zforce command/response IDs
 */
enum zf_command {
    ZFSEND_DEACTIVATE            = 0x00,
    ZFSEND_ACTIVATE              = 0x01,
    ZFSEND_SETRESOLUTION         = 0x02,
    ZFSEND_SETCONFIGURATION      = 0x03,
    ZFSEND_SETSCANNINGFREQUENCY  = 0x08,
    ZFSEND_LOWSIGNALSREQUEST     = 0x0D,
    ZFSEND_FIXEDPULSESTRENGTH    = 0x0F,
    ZFSEND_FORCECALIBRATION      = 0x1A,
    ZFSEND_LEDLEVELREQUEST       = 0x1C,
    ZFSEND_SETACTIVELEDS         = 0x1D,
    ZFSEND_STATUSCOMMAND         = 0x1E,
    ZFSEND_FORCEDLEDLEVELREQUEST = 0x20,
    ZFSEND_OPENSHORTREQUEST      = 0x21,
    ZFSEND_SETGPIOCONTROL        = 0x23,
    ZFSEND_GETGPIOCONTROL        = 0x24,
    // Proximity may not be supported
    ZFSEND_PROXIMITY             = 0x26,
    ZFSEND_BOOTLOADER            = 0x2B,
    ZFSEND_SETPWMCONTROL         = 0xE0,
};

/**
 * enum lcd_type - uivc display types
 */
typedef enum lcd_type {
    LCD_TYPE_4p3 = 0,
    LCD_TYPE_7p0 = 1,
    LCD_TYPE_10p1 = 2,
    LCD_TYPE_RESERVED = 3,
} zf_display_types_t;

/**
 * uivc display cfg flags
 */
// native10 only relevant on LCD_TYPE_10p1
#define LCD_CFG_NATIVE10 (1 << 0)

typedef struct {
    uint16_t width;
    uint16_t height;
} PACKED zf_setresolution_t;

typedef struct {
    uint32_t flags;
} PACKED zf_setconfiguration_t;

typedef struct {
    uint8_t port;
    uint8_t direction;
    uint8_t state;
} PACKED zf_setgpiocontrol_t;

typedef struct {
    uint8_t port;
} PACKED zf_getgpiocontrol_t;

typedef struct {
    uint8_t enable;
    uint8_t frequency;
    uint8_t duty;
} PACKED zf_setpwmcontrol_t;

/**
 * enum zf_notify - zforce command/response IDs
 *
 * Most received packet IDs will match a sent command ID. Only need
 * values for notification and error response IDs
 */
enum zf_notify {
    ZFRECV_BOOTCOMPLETE        = 0x07,
    ZFRECV_GETGPIOCONTROLRESPONSE = 0x24,
    ZFRECV_OVERRUN                = 0x25,
    ZFRECV_PROXIMITY              = 0x26,
    ZFRECV_INVALIDCOMMAND         = 0xFE,
 };

typedef struct {
    uint8_t result;
} PACKED zf_boot_t;

typedef struct {
    uint16_t major;
    uint16_t minor;
    uint16_t build;
    uint16_t revision;
    uint8_t  active;
    uint16_t scanning_counter;
    uint16_t prepared_touch_packages;
    uint16_t sent_touch_packages;
    uint16_t invalid_touch_counter;
    uint32_t configuration;
    uint16_t idle_freq;
    uint16_t finger_freq;
    uint16_t stylus_freq;
    uint16_t resolution_width;
    uint16_t resolution_height;
    uint16_t physical_width;
    uint16_t physical_height;
    uint8_t first_active_led_x;
    uint8_t last_active_led_x;
    uint8_t first_active_led_y;
    uint8_t last_active_led_y;
    uint8_t max_signal_strength;
    uint8_t max_size_enabled;
    uint8_t max_size;
    uint8_t min_size_enabled;
    uint8_t min_size;
    uint8_t device_type; /* reserved */
    uint8_t protocol_major;
    uint8_t protocol_minor;
    uint8_t reserved[131-44];
} PACKED zf_statusresponse_t;

typedef struct {
    uint8_t port;
    uint8_t direction;
    uint8_t out_state;
    uint8_t in_state;
} PACKED zf_getgpiocontrolresponse_t;

/* TODO: This could replace several of the structures above for simplification */
typedef struct {
    uint8_t result;
} PACKED zf_simple_response_t;

extern int i2c(uint16_t addr, uint8_t *sbuf, uint8_t slen, uint8_t *rbuf, uint8_t rlen);
int zf_neonode_init(zf_ctrl_t *zf, vga_panel_data *vpd, mv_mmp_gen2_display_info_t *dynamic_lcd);
int zf_deinit(zf_ctrl_t *zf);
int zf_data_ready_handler(zf_ctrl_t *zf);

int zf_send(zf_ctrl_t *zf, uint8_t cmd, uint8_t *sbuf, uint8_t slen);
int zf_recv(zf_ctrl_t *zf, uint8_t *rbuf, uint8_t *rlen, size_t space);
int zf_cmd_transaction(zf_ctrl_t *zf, uint8_t cmd, uint8_t *sbuf, uint8_t slen, int wait_response, char *dbgtext);
int zf_version_check(zf_ctrl_t *zf, const zf_version_t *min_ver);
int zf_protocol_check(zf_ctrl_t *zf, const zf_version_t *min_ver);
int zf_probe(vga_panel_data *vpd, mv_mmp_gen2_display_info_t *dynamic_lcd);

#endif /* _ZFORCE_H_ */
