#include <malloc.h>
#include <asm/errno.h>
#include <common.h>
#include <i2c.h>
#include <mv_mmp_panel.h>
#include "mv_mmp_panel_common.h"
#include "zforce.h"

#define I2CADDR_ZFORCE (0x50 >> 1)

/* GPIO mapping for strap bits */
static const zf_gpio_t uivc_version[] = {
   {4, 3}, /* HWV1 */
   {4, 4}, /* HWV2 */
   {4, 5}, /* HWV3 */
   {4, 6}, /* HWV4 */
};

static const zf_gpio_t uivc_type[] = {
   {2, 4}, /* SPA4 */
   {2, 3}, /* SPA3 */
};

static const zf_gpio_t uivc_subtype[] = {
   {2, 6}, /* SPA6 */
   {2, 5}, /* SPA5 */
};

static const zf_gpio_t uivc_cfg[] = {
   {3, 7}, /* SPA10 */
   {4, 2}, /* SPA9  */
   {4, 7}, /* SPA8  */
   {2, 7}, /* SPA7  */
};

static const zf_gpio_t lrx_dn = {1, 7};
static const zf_gpio_t disp_sby = {2, 0};
static const zf_gpio_t en_5vsw = {4, 1};
static const zf_gpio_t pwm_bl = {1, 3};
static const zf_gpio_t bpp18 = {2, 1};
static const zf_gpio_t bl_sel = {1, 6};

/* Not perfect due to lack of revision consistency by vendor. Update when versions are corrected. */
static const zf_version_t min_identifiable_ver = {
    .major = 1,
    .minor = 0,
    .build = 0,
    .revision = 0,
    .protocol_major = 5,
    .protocol_minor = 0,
};


#ifndef CONFIG_PANEL_I2C_BUS
#error CONFIG_PANEL_I2C_BUS must be defined in board config
#endif
static mv_mmp_panel_i2c_info_t i2c_info = {
	.bus_num = -1,
	.dev_addr = I2CADDR_ZFORCE,
	.read_retries = 0,
	.write_retries = 0,
	.delay = 1000,
	.protocol = PROTOCOL_ZFORCE,
};

zf_ctrl_t *zf = NULL;

/**
 * zf_print_buf - dump a raw buffer
 * @zf: top zforce control structure
 * @buf: destination for touch event string
 * @size: buffer size
 * @dbgtext: text string for verbose debug messages
 */
void zf_print_buf(zf_ctrl_t *zf, uint8_t *buf, int size, char *dbgtext) {
   int i;

   PRINTZFDBG("%s - printing %s buffer...", __FUNCTION__, (dbgtext ? dbgtext : "undescribed"));
   for (i = 0; i < size; i++) {
      if (i % 8 == 0)
         PRINTZFDBG("\n0x");
      PRINTZFDBG("%02x", (buf)[i]);
   }
   PRINTZFDBG("\nbuffer end\n");
}


/**
 * zf_send - create and transmit a packet to zforce
 * @zf: top zforce control structure
 * @cmd: zforce command ID
 * @sbuf: buffer for command-specific data
 * @slen: sbuf size in bytes
 * @space: rbuff size
 *
 */
int zf_send(zf_ctrl_t *zf, uint8_t cmd, uint8_t *sbuf, uint8_t slen)
{
   int mlen;
   int rc;
   zf_packet_t *tpkt = (zf_packet_t *)zf->txbuf;

   PRINTZFDBG("%s - entered with cmd 0x%02x, slen %d\n", __FUNCTION__, cmd, slen);

   mlen = sizeof(zf_packet_t) + slen;
   tpkt->PacketHdr.FrameStart = ZF_FRAME_START;
   tpkt->PacketHdr.DataSize = sizeof(tpkt->PacketData.CommandID) + slen;
   tpkt->PacketData.CommandID = cmd;

   /* some commands do not require any data */
   if(slen > 0)
      memcpy(tpkt->PacketData.CommandData, sbuf, slen);

   zf_print_buf(zf, (uint8_t *)&tpkt->PacketData, tpkt->PacketHdr.DataSize, "tx");
   rc = mv_mmp_panel_i2c_write(&i2c_info, (uint8_t *)tpkt, mlen);

   if(rc) {
      printf("%s:%d - I2C error rc=0x%08x!\n", __FUNCTION__, __LINE__, rc);
      return -1;
   }
   else {
      return 0;
   }
}

/**
 * zf_recv - receive a packet from zforce
 * @zf: top zforce control structure
 * @rbuf: buffer for packet payload without header
 * @rlen: updated to report payload size at exit
 * @space: rbuff size
 *
 */
int zf_recv(zf_ctrl_t *zf, uint8_t *rbuf, uint8_t *rlen, size_t space)
{
   int rc;
   zf_packet_hdr_t rhdr;

   PRINTZFDBG("%s - entered, zf->cmdstat.pending = %d  zf->cmdstat.CommandID = 0x%02x\n",
                                __FUNCTION__, zf->cmdstat.pending, zf->cmdstat.CommandID);

   memset(&rhdr, 0, sizeof(rhdr));

   /*
    * we have to split the read into two transactions
    * because we can read back variable lengths of data
    *
    * the first transaction will get the length of the payload
    */
   rc = mv_mmp_panel_i2c_read(&i2c_info, 0, 0, (uint8_t *) &rhdr, sizeof(rhdr));
   if(rc) {
       printf("%s:%d - I2C error rc=0x%08x!\n", __FUNCTION__, __LINE__, rc);
      return -1;
   }

   PRINTZFDBG("%s - Received packet header 0x%02x 0x%02x\n", __FUNCTION__, rhdr.FrameStart, rhdr.DataSize);
   if (rhdr.FrameStart != ZF_FRAME_START) {
      printf("%s - unexpected start byte 0x%x\n", __FUNCTION__, rhdr.FrameStart);
      return -1;
   }

   if(rhdr.DataSize > space) {
      printf("%s - not enough room in buffer\n", __FUNCTION__);
      return -1;
   }

   *rlen = rhdr.DataSize;

   /*
    * the second transaction will read the remainder of the payload
    */
   rc = mv_mmp_panel_i2c_read(&i2c_info, 0, 0, rbuf, *rlen);
   if(rc) {
      printf("%s:%d - I2C error, rc=0x%08x!\n", __FUNCTION__, __LINE__, rc);
      return -1;
   }

   zf_print_buf(zf, (uint8_t *)rbuf, *rlen, "rx");

   return 0;
}


/**
 * zf_rx_packet_handler - common handler to parse received or injected packets
 * @zf: top zforce control structure
 *
 */
static int zf_rx_packet_handler(zf_ctrl_t *zf, zf_packet_data_t *pkt, int rlen, int maxlen)
{
   PRINTZFDBG("%s - entered!\n", __FUNCTION__);

   // compiler will complain about bounds check
   if(rlen > 0) {
      if(zf->cmdstat.pending &&
            ((pkt->CommandID == zf->cmdstat.CommandID) ||
             (pkt->CommandID == ZFRECV_OVERRUN) ||
             (pkt->CommandID == ZFRECV_INVALIDCOMMAND))) {
         //save response for command issuer
         memcpy(zf->respbuf, zf->rxbuf, rlen);
         zf->cmdstat.pending = 0;
      }
      else if(pkt->CommandID == ZFRECV_BOOTCOMPLETE) {
#if (ZF_DEFAULT_VERBOSITY > 0)
         zf_boot_t *boot = (zf_boot_t *)pkt->CommandData;
         PRINTZFDBG("%s - zForce boot %s\n", __FUNCTION__, (!boot->result ? "succeeded." : "failed!"));
#endif
      }
      /* else ignore packet */
    }

   return 0;
}


/**
 * zf_data_ready_handler
 */
int zf_data_ready_handler(zf_ctrl_t *zf)
{
    uint8_t rlen = 0;
    int rc = 0;

    PRINTZFDBG("%s:%d - entered\n", __FUNCTION__, __LINE__);
    rc = zf_recv(zf, zf->rxbuf, &rlen, sizeof(zf->rxbuf));
    if(!rc)
    {
       zf_rx_packet_handler(zf, (zf_packet_data_t *)&zf->rxbuf[0], rlen, sizeof(zf->rxbuf));
    }
    else {
       printf("%s:%d: Error reading zForce response!\n",
                                         __func__, __LINE__);
    }

    PRINTZFDBG("%s - returning rc = %d\n", __FUNCTION__, rc);

    return rc;
}


/**
 * zf_wait_command_response - wait for response packet from interrupt handler
 * @zf: top zforce control structure
 * @timeout_ms: timeout in milliseconds
 *
 * Response packet may be interleaved between other notification packets.
 * Interrupt handler will compare incoming IDs to the ID that was sent
 * to identify a reponse.
 *
 */
int zf_wait_command_response(zf_ctrl_t *zf, long timeout_ms) {
   int rc = 0;

   PRINTZFDBG("%s - entered\n", __FUNCTION__);
   rc = mv_mmp_panel_wait_irq_gen2(UIBC_UIVC_IRQ, timeout_ms);

   if(!rc) {
      zf_data_ready_handler(zf);
   }
   else {
      printf("%s:%d: Timeout waiting for uivc irq\n",
                                            __func__, __LINE__);
      rc = -ETIMEDOUT;
   }

   return rc;
}

/**
 * zf_cmd_transaction - send a zforce command and wait for response
 * @zf: top zforce control structure
 * @cmd: zforce command ID
 * @sbuf: buffer for command-specific data
 * @slen: sbuf size in bytes
 * @rbuf: response buffer pointer (may be removed soon)
 * @wait_response: flag indicating response is expected (true for most commands)
 * @dbgtext: text string for verbose debug messages
 *
 */
int zf_cmd_transaction(zf_ctrl_t *zf, uint8_t cmd, uint8_t *sbuf, uint8_t slen, int wait_response, char *dbgtext) {
   int rc;
   zf_packet_data_t *pkt;

   PRINTZFDBG("%s - entered!\n", __FUNCTION__);

   if(dbgtext)
      PRINTZFDBG("%s - sending %s command!\n", __FUNCTION__, dbgtext);
   zf->cmdstat.CommandID = cmd;

   if(wait_response)
      zf->cmdstat.pending = 1;

   zf_send(zf, zf->cmdstat.CommandID, sbuf, slen);

   if(!wait_response) {
      rc = 0;
      goto done;
   }

   PRINTZFDBG("%s - waiting for %s response!\n", __FUNCTION__, dbgtext);

   rc = zf_wait_command_response(zf, DEFAULT_CMD_TIMEOUT_MS);
   if(rc)
      printf("%s - timeout waiting for %s response\n", __FUNCTION__, dbgtext);

   //keep the debug printf sane for now
   if(zf->verbose)
      udelay(2000);

   pkt = (zf_packet_data_t *)&zf->respbuf[0];

   switch(pkt->CommandID) {
      case ZFRECV_INVALIDCOMMAND:
      case ZFRECV_OVERRUN:
         rc = -1;
         break;
      default:
         break;
   }

   if(rc)
      rc = -1;

done:
   PRINTZFDBG("%s - returning rc = %d\n", __FUNCTION__, rc);
   return rc;
}

/**
 * zf_gpio_output - configure a zforce gpio output pin
 * @zf: top zforce control structure
 * @port: zforce controller gpio port
 * @bit: gpio bit number within port
 * @val: output logic value
 *
 * Any pin configured as an input will
 * generate an interrupt and notification packet on any change. The packet may or
 * may not be paired with a query command.
 *
 */
int zf_gpio_output(zf_ctrl_t *zf, int port, int bit, int val) {
   zf_getgpiocontrol_t get_state;
   zf_setgpiocontrol_t new_state;
   zf_packet_data_t *rpkt = (zf_packet_data_t *)zf->respbuf;
   zf_getgpiocontrolresponse_t *old_state = NULL;
   uint8_t mask;

   PRINTZFDBG("%s - entered!\n", __FUNCTION__);

   get_state.port = port;
   mask = 1 << bit;

   zf_cmd_transaction(zf, ZFSEND_GETGPIOCONTROL, (uint8_t *)&get_state, sizeof(get_state), 1, "get port state");

   old_state = (zf_getgpiocontrolresponse_t *)rpkt->CommandData;
   new_state.port = old_state->port;
   new_state.direction = (old_state->direction & ~mask) | (ZF_GPIO_OUTPUT << bit);
   new_state.state = (old_state->out_state & ~mask) | ((val ? 1 : 0) << bit);

   zf_cmd_transaction(zf, ZFSEND_SETGPIOCONTROL, (uint8_t *)&new_state, sizeof(new_state), 1, "set port state");

   return 0;
}


/**
 * zf_gpio_input - configure a zforce gpio input pin
 * @zf: top zforce control structure
 * @port: zforce controller gpio port
 * @bit: gpio bit number within port
 *
 * Any pin configured as an input will
 * generate an interrupt and notification packet on any change. The packet may or
 * may not be paired with a query command.
 *
 */
int zf_gpio_input(zf_ctrl_t *zf, int port, int bit) {
    zf_getgpiocontrol_t get_state;
    zf_setgpiocontrol_t new_state;
    zf_packet_data_t *rpkt = (zf_packet_data_t *)zf->respbuf;
    volatile zf_getgpiocontrolresponse_t *old_state = NULL;
    uint8_t mask;
    int val;

    PRINTZFDBG("%s - entered!\n", __FUNCTION__);

    if(zf->pre_api5_fw) {
        printf("%s: Error! zforce fw too old.\n", __FUNCTION__);
        return -1;
    }

    get_state.port = port;
    mask = 1 << bit;

    PRINTZFDBG("%s:%d - reading port %d bit %d\n", __FUNCTION__, __LINE__, port, bit);
    zf_cmd_transaction(zf, ZFSEND_GETGPIOCONTROL, (uint8_t *)&get_state, sizeof(get_state), 1, "get port state");

    old_state = (zf_getgpiocontrolresponse_t *)rpkt->CommandData;
    PRINTZFDBG("%s:%d - old direction 0x%02x out_state 0x%02x in_state 0x%02x\n", __FUNCTION__, __LINE__,
                                 old_state->direction, old_state->out_state, old_state->in_state);

    if((old_state->direction & mask) == (ZF_GPIO_INPUT << bit)) {
       val = (old_state->in_state & mask) ? 1 : 0;
    }
    else {
       new_state.port = old_state->port;
       new_state.direction = (old_state->direction & ~mask) | (ZF_GPIO_INPUT << bit);
       //new_state.state = (old_state->out_state & ~mask);
       new_state.state = (old_state->out_state | mask);

       PRINTZFDBG("%s:%d - new direction 0x%02x out_state 0x%02x\n", __FUNCTION__, __LINE__,
                                               new_state.direction, new_state.state);
       zf_cmd_transaction(zf, ZFSEND_SETGPIOCONTROL, (uint8_t *)&new_state, sizeof(new_state), 1, "set port state");
       zf_cmd_transaction(zf, ZFSEND_GETGPIOCONTROL, (uint8_t *)&get_state, sizeof(get_state), 1, "get port state");

       old_state = (zf_getgpiocontrolresponse_t *)rpkt->CommandData;
       PRINTZFDBG("%s:%d - updated direction 0x%02x out_state 0x%02x in_state 0x%02x\n", __FUNCTION__, __LINE__,
                                               old_state->direction, old_state->out_state, old_state->in_state);
       val = (old_state->in_state & mask) ? 1 : 0;
    }
    return val;
}


/**
 * zf_get_id_straps - retrieve panel id
 * @zf: top zforce control structure
 *
 */
int zf_get_id_straps(zf_ctrl_t *zf) {
    int i;

    memset (&zf->straps, 0, sizeof(zf->straps));

    PRINTZFDBG("%s - entered!\n", __FUNCTION__);
    PRINTZFDBG("%s - *************************************************************************\n", __FUNCTION__);
    for(i = (ARRAY_SIZE(uivc_version) - 1); i >= 0; i--) {
        zf->straps.version |= zf_gpio_input(zf, uivc_version[i].port, uivc_version[i].bit) << i;
    }
    for(i = (ARRAY_SIZE(uivc_type) - 1); i >= 0; i--) {
        zf->straps.type |= zf_gpio_input(zf, uivc_type[i].port, uivc_type[i].bit) << i;
    }
    for(i = (ARRAY_SIZE(uivc_subtype) - 1); i >= 0; i--) {
        zf->straps.subtype |= zf_gpio_input(zf, uivc_subtype[i].port, uivc_subtype[i].bit) << i;
    }
    for(i = (ARRAY_SIZE(uivc_cfg) - 1); i >= 0; i--) {
        zf->straps.cfg |= zf_gpio_input(zf, uivc_cfg[i].port, uivc_cfg[i].bit) << i;
    }
    printf("%s - UIVC version = %d, type = %d, subtype = %d, cfg = 0x%02x\n", __FUNCTION__,
                                                                  zf->straps.version,
                                                                  zf->straps.type,
                                                                  zf->straps.subtype,
                                                                  zf->straps.cfg);

    return 0;
}


/*
 * zf_version_get - get zforce fw version from status report
 * @zf: top zforce control structure
 *
 */
static int zf_version_get(zf_ctrl_t *zf) {
    int rc;
    zf_packet_data_t *rpkt = (zf_packet_data_t *)zf->respbuf;
    volatile zf_statusresponse_t *status = NULL;
    zf_version_t *version = &zf->fw_version;

    rc = zf_cmd_transaction(zf, ZFSEND_STATUSCOMMAND, NULL, 0, 1, "status");
    if(rc) {
       printf("%s: Error! Unable to read zForce status\n", __FUNCTION__);

       return -1;
    }

    status = (zf_statusresponse_t *)rpkt->CommandData;

    version->major = status->major;
    version->minor = status->minor;
    version->build = status->build;
    version->revision = status->revision;
    version->protocol_major = status->protocol_major;
    version->protocol_minor = status->protocol_minor;

    return 0;
}

/**
 * zf_version_check - compare zForce firmware version to a minimum requirement
 * @zf: top zforce control structure
 * @min_ver: pointer to minimum version required
 *
 * Returns 0 if version >= requirement
 */
int zf_version_check(zf_ctrl_t *zf, const zf_version_t *min_ver) {
    zf_version_t *version = &zf->fw_version;

    if(version->major > min_ver->major) {
       return 0;
    }
    else if(version->major == min_ver->major) {
       if(version->minor > min_ver->minor) {
          return 0;
       }
       else if(version->minor == min_ver->minor) {
          if(version->build > min_ver->build) {
             return 0;
          }
          else if(version->build == min_ver->build) {
             if(version->revision >= min_ver->revision) {
                return 0;
             }
          }
       }
    }

    return -1;
}

/**
 * zf_protocol_check - compare zForce protocol version to a minimum requirement
 * @zf: top zforce control structure
 * @min_ver: pointer to minimum version required
 *
 * Returns 0 if version >= requirement
 */
int zf_protocol_check(zf_ctrl_t *zf, const zf_version_t *min_ver) {
   zf_version_t *version = &zf->fw_version;

    if(version->protocol_major > min_ver->protocol_major) {
       return 0;
    }
    else if(version->protocol_major == min_ver->protocol_major) {
       if(version->protocol_minor >= min_ver->protocol_minor) {
          return 0;
       }
    }

    return -1;
}

/**
 * zf_wait_boot_packet - flush boot notification packet
 * @zf: top zforce control structure
 *
 * Must be called after interrupts are enabled.
 *
 * Will only get a boot notification packet after a hard
 * reset. Until gpio reset is implemented, report success
 * unconditionally to avoid breaking warm boot.
 */
int zf_wait_boot_packet(zf_ctrl_t *zf) {
   int rc;
   rc = zf_wait_command_response(zf, 200);
   if(rc)
      printf("%s: zForce boot packet missing, normal on early hw after warm boot\n", __FUNCTION__);
   return 0;
}

/**
 * zf_vga_gen2_mode - use straps information to retrieve display timing
 * @straps: data read from straps
 */
const mv_mmp_gen2_display_info_t *zf_vga_gen2_mode(zf_id_straps_t *straps) {
	MV_MMP_PANEL_TYPE		panel_type;
	int i;

	switch(straps->type) {
	case LCD_TYPE_4p3:
		panel_type = MV_MMP_PANEL_TYPE_480X272_NON_SEQ;
		break;
	case LCD_TYPE_7p0:
		panel_type = MV_MMP_PANEL_TYPE_800X480;
		break;
	case LCD_TYPE_10p1:
		if(zf->straps.cfg & LCD_CFG_NATIVE10) {
            panel_type = MV_MMP_PANEL_TYPE_1024X600;
		}
		else {
            panel_type = MV_MMP_PANEL_TYPE_800X480_SCALED;
		}
		break;
	default:
		return NULL;
	}

   for(i = 0; i < mv_mmp_display_info_count; i++) {
		if(mv_mmp_display_info[i].mv_mmp_type == panel_type) {
			return &mv_mmp_display_info[i];
		}
	}

	return NULL;
}

/**
 * zf_neonode_init - touch and gpio hardware initialization
 * @zf: top zforce control structure
 *
 * Must be called after interrupts are enabled.
 */
int zf_neonode_init(zf_ctrl_t *zf, vga_panel_data *vpd, mv_mmp_gen2_display_info_t *dynamic_lcd) {
   int rc;
   int errline = -1;
   const mv_mmp_gen2_display_info_t *display = NULL;

   zf_setresolution_t res;
   zf_setpwmcontrol_t pwm;
   zf_setconfiguration_t cfg;

   zf->verbose = ZF_DEFAULT_VERBOSITY;

   PRINTZFDBG("%s - entered!\n", __FUNCTION__);

   rc = zf_wait_boot_packet(zf);
   if(rc) {
       printf("%s: Error! zForce boot error!\n", __FUNCTION__);
       goto done;
   }

   rc = zf_cmd_transaction(zf, ZFSEND_ACTIVATE, NULL, 0, 1, "activate");
   if(rc) {
      printf("%s: Error! zForce failed to activate\n", __FUNCTION__);
      goto done;
   }

   rc = zf_version_get(zf);
   if(rc) {
      printf("%s: Error! Unable to read zForce status\n", __FUNCTION__);
       goto done;
   }
   printf("zForce fw version = %d.%d.%d.%d, protocol version = %d.%d\n",
              zf->fw_version.major,
              zf->fw_version.minor,
              zf->fw_version.build,
              zf->fw_version.revision,
              zf->fw_version.protocol_major,
              zf->fw_version.protocol_minor);


   PRINTZFDBG("%s - calling zf_get_id_straps!\n", __FUNCTION__);
   rc = zf_protocol_check(zf, &min_identifiable_ver);
   if(!rc) {
       rc = zf_get_id_straps(zf);
   }
   else {
       printf("%s%d: deprecated zForce firmware does not support display size detection.\n", __FUNCTION__, __LINE__);
       zf->pre_api5_fw = 1;
       /* temp override hack because GP inputs are not working */
       //zf->straps.type = LCD_TYPE_7p0;
       zf->straps.type = LCD_TYPE_4p3;
       zf->straps.version = 0;
       zf->straps.subtype = 0;
       zf->straps.cfg = 0;
   }

   if(vpd->lvds == LVDS_UNKNOWN) {
       /*
        * If lvds type was not identified by eeprom, fall back to straps.
        * 18 bit is not directly supported by straps, but option 2 is downward
        * compatible.
        */
       if(zf->straps.subtype & SUBTYPE_FLAG_LVDS_OPTION1)
           vpd->lvds = LVDS_BPP24_OPTION_1;
       else
           vpd->lvds = LVDS_BPP24_OPTION_2;
   }

   if(!dynamic_lcd) {
       printf("%s:%d Using straps to identify display timing.\n", __func__, __LINE__);
       display = zf_vga_gen2_mode(&zf->straps);
   }
   else {
       printf("%s:%d Using eeprom to identify display timing.\n", __func__, __LINE__);
       display = dynamic_lcd;
   }

   rc = mv_mmp_vga_translate(vpd, display);
   if(rc) {
      printf("%s:%d zForce display timing detection failed.\n", __func__, __LINE__);
      rc = -1;
      goto done;
   }
   else {
      if(display->name) {
	 printf("%s Found display %s.\n", __func__, display->name);
      }
   }

   res.width = vpd->xres;
   res.height = vpd->yres;

   rc = zf_cmd_transaction(zf, ZFSEND_SETRESOLUTION, (uint8_t *)&res, sizeof(res), 1, "resolution");
   if(rc) {
      printf("%s%d: zForce Error!\n", __FUNCTION__, __LINE__);
      goto done;
   }

   rc = zf_gpio_output(zf, en_5vsw.port, en_5vsw.bit, 1); // EN_5VSW (enable 5v switched)
   if(rc) {
      errline = __LINE__;
      goto done;
   }

   /*
    * There's no input strap for bit depth. If not specified by eeprom, assume 24
    * for legacy compatibility.
    */
   switch(vpd->bit_depth) {
   case 16:
   case 18:
      rc = zf_gpio_output(zf, bpp18.port, bpp18.bit, 1); // BPP18 (select bpp18)
      break;
   default:
      vpd->bit_depth = 24;
      rc = zf_gpio_output(zf, bpp18.port, bpp18.bit, 0); // BPP18 (select bpp24)
      break;
   }
   if(rc) {
      errline = __LINE__;
      goto done;
   }

   rc = zf_gpio_output(zf, lrx_dn.port, lrx_dn.bit, 1); // LRX_DN (enable lvds)
   if(rc) {
      errline = __LINE__;
      goto done;
   }
   rc = zf_gpio_output(zf, disp_sby.port, disp_sby.bit, 1); // DISP_SBY (enable display)
   if(rc) {
      errline = __LINE__;
      goto done;
   }
   rc = zf_gpio_output(zf, bl_sel.port, bl_sel.bit, vpd->backlight_select); // BL_SEL (backlight driver type)
   if(rc) {
      errline = __LINE__;
      goto done;
   }

   // Set sane defaults if backlight duty was not detected from eeprom
   if(vpd->backlight_duty_max == 0)
      vpd->backlight_duty_max = BACKLIGHT_DUTY_MAX_DEFAULT;
   if((vpd->backlight_duty_init == 0) || (vpd->backlight_duty_init > vpd->backlight_duty_max))
      vpd->backlight_duty_init = vpd->backlight_duty_max;

   // enable backlight pwm
   pwm.enable = 1;
   pwm.frequency = 0; // 1 kHz
   pwm.duty = (uint8_t)(vpd->backlight_duty_init);

   rc = zf_cmd_transaction(zf, ZFSEND_SETPWMCONTROL, (uint8_t *)&pwm, sizeof(pwm), 1, "pwm");
   if(rc) {
      errline = __LINE__;
      goto done;
   }

   cfg.flags = 0;
   rc = zf_cmd_transaction(zf, ZFSEND_SETCONFIGURATION, (uint8_t *)&cfg, sizeof(cfg), 1, "configuration");
   if(rc) {
      errline = __LINE__;
      goto done;
   }

   rc = zf_cmd_transaction(zf, ZFSEND_STATUSCOMMAND, NULL, 0, 1, "status");
   if(rc) {
      printf("%s%d: zForce Error!\n", __FUNCTION__, __LINE__);
      goto done;
   }

done:
   if(errline > 0)
      printf("%s: Error! zForce communication error reported at line %d\n", __FUNCTION__, errline);

   PRINTZFDBG("%s - returning rc = %d\n", __FUNCTION__, rc);
   return rc;
}

/**
 * zf_deinit - disable zforce touch controller and release resources
 * This will also disable GPIO control.
 */
int zf_deinit(zf_ctrl_t *zf) {
   zf_cmd_transaction(zf, ZFSEND_DEACTIVATE, NULL, 0, 1, "deactivate");
   if(zf)
   {
      free(zf);
      zf = NULL;
   }

   return 0;
}

/**
 * zf_probe - detect and init zForce controller
 */
int zf_probe(vga_panel_data *vpd, mv_mmp_gen2_display_info_t *dynamic_lcd) {
   int rc;

   MV_MMPPANELDBG("%s - entered!\n", __FUNCTION__);

   zf = calloc(1, sizeof(*zf));
   if(zf == NULL)
   {
      printf("%s:%d Unable to malloc zf ctrl\n", __func__, __LINE__);
      return -1;
   }
   else {
      /* buffer was preinitialized to zero by calloc */
      rc = zf_neonode_init(zf, vpd, dynamic_lcd);
      if(rc)
         free(zf);
   }

   if(rc) {
      printf("%s:%d zForce initialization failed.\n", __func__, __LINE__);
      return -1;
   }
   printf("%s:%d zForce initialization completed.\n", __func__, __LINE__);
   return 0;
}
