#include <common.h>
#include <libfdt.h>
#include <fdt_support.h>
#include <errno.h>
#include <dovefbreg.h>
#include <lcd.h>
#include <lxk_panel.h>
#include "lxk_panel_common.h"
#include "lxk_eeprom_info.h"

#ifdef CONFIG_LXK_PANEL_VGA_PEGMATITE
#define LCD_BASE (0xD008b000)
#else
#define LCD_BASE (0xD00e0000)
#endif
#define REF_RATE_HZ (60)
#define LCD_MAX_COLS (1024)
#define LCD_MAX_ROWS (600)
#define LCD_COLOR_DEPTH (LCD_COLOR16)

#define SHOWREGW(base, reg)  serial_printf("offset 0x%04x = 0x%08x\n", \
						(unsigned int)reg, \
						((u32)readl(base + reg)))

int lcd_color_fg;
int lcd_color_bg;

void *lcd_console_address;             /* Start of console buffer      */

short console_col;
short console_row;

extern void lxk_reset_panel(void);

void lcd_initcolregs (void)
{
}

void lcd_setcolreg(ushort regno, ushort red, ushort green, ushort blue)
{
}

void lcd_enable (void)
{
}

vidinfo_t panel_info = 
{
       vl_col  : LCD_MAX_COLS,
       vl_row  : LCD_MAX_ROWS,
       vl_bpix : LCD_COLOR_DEPTH,
       cmap    : NULL,
       priv    : NULL,
       refresh : NULL,
};

extern int lxk_armada_dumb_panel_init(void *reg_base, void *buffer);

const char backlight_compat[] = "lxk-backlight";
const char backlight_name[] = "backlight0";
#define MAX_BACKLIGHT_LEVEL_INDEX	100
const char lcd_name[] = "pxa168-fb";
const char gamma_name[] = "gamma";

#if defined(CONFIG_OF_BOARD_SETUP)
#ifdef CONFIG_LXK_PANEL_VGA_PEGMATITE
#define TIMING_NODE "/lcd@d008b000/display-timings/timing0"
#define LCD_NODE "/lcd@d008b000"
#else
#define TIMING_NODE "/soc/internal-regs/lcd@e0000/display-timings/timing0"
#define LCD_NODE "/soc/internal-regs/lcd@e0000"
#endif

static int fdt_fixup_gamma(void *blob, vga_panel_data *vpd)
{
	int lcd_node;
	int gamma_node;
	int ret;
	gamma_table_t *gamma;

	if(!vpd->gamma)
		return 0;

	gamma = vpd->gamma;

	/*
	 * Search for lcd node by path.
	 * Could also use fdt_node_offset_by_compatible(), but already using path
	 * for timing fixup, so be consistent.
	 */ 
	lcd_node = fdt_path_offset(blob, LCD_NODE);
	if(lcd_node < 0) {
		printf("Unable to find or access lcd node '%s': %s\n",
				lcd_name, fdt_strerror(lcd_node));
		return -EIO;
	}
	/* Search for gamma subnode, and add if not found. */
	gamma_node = fdt_subnode_offset(blob, lcd_node, gamma_name);
	if(gamma_node == -FDT_ERR_NOTFOUND) {
		gamma_node = fdt_add_subnode(blob, lcd_node, gamma_name);
	}
	if (gamma_node < 0) {
		printf("Cannot access or create node '%s': %s\n", gamma_name,
		       fdt_strerror(gamma_node));
		return -EIO;
	}
	/* Gamma LUT data in 3 arrays of bytes. */
	ret = fdt_setprop(blob, gamma_node, "red", gamma->red, sizeof(gamma->red));
	if(ret < 0) {
		printf("Error setting red gamma: %s\n", fdt_strerror(ret));
		return -EIO;
	}
	ret = fdt_setprop(blob, gamma_node, "green", gamma->green, sizeof(gamma->green));
	if(ret < 0) {
		printf("Error setting blue gamma: %s\n", fdt_strerror(ret));
		return -EIO;
	}
	ret = fdt_setprop(blob, gamma_node, "blue", gamma->blue, sizeof(gamma->blue));
	if(ret < 0) {
		printf("Error setting blue gamma: %s\n", fdt_strerror(ret));
		return -EIO;
	}
			
	return 0;
}

static int fdt_fixup_backlight(void *blob, vga_panel_data *vpd)
{
	int node;
	fdt32_t levels[MAX_BACKLIGHT_LEVEL_INDEX + 1];
	fdt32_t default_level;
	fdt32_t select;
	int i;
	int ret;
	
	/* Check for existing backlight node */
	node = fdt_node_offset_by_compatible(blob, -1, backlight_compat);
	if (node == -FDT_ERR_NOTFOUND) {
		/* If node doesn't exist, create one. */
		node = fdt_add_subnode(blob, 0, backlight_name);
		if (node < 0) {
			printf("Cannot create node '%s': %s\n", backlight_name,
			       fdt_strerror(node));
			return -EIO;
		}

		if (fdt_setprop_string(blob, node, "compatible", backlight_compat)) {
			printf("Cannot set compatible\n");
			return -EIO;
		}
	} else if (node < 0) {
		printf("Cannot access node '%s': %s\n", backlight_name,
		       fdt_strerror(node));
		return -EIO;
	}

	/* Create or modify the brightness level mapping */
	for(i = 0; i <= MAX_BACKLIGHT_LEVEL_INDEX; i++) {
		levels[i] = cpu_to_fdt32((vpd->backlight_duty_max * i)/MAX_BACKLIGHT_LEVEL_INDEX);
	}
	default_level = cpu_to_fdt32(MAX_BACKLIGHT_LEVEL_INDEX);
	
	ret = fdt_setprop(blob, node, "brightness-levels", levels, sizeof(levels));
	if(ret < 0) {
		printf("Error setting brightness-levels: %s\n", fdt_strerror(ret));
		return -EIO;
	}
	ret = fdt_setprop(blob, node, "default-brightness-level", &default_level, sizeof(default_level));
	if(ret < 0) {
		printf("Error setting default-brightness-level: %s\n", fdt_strerror(ret));
		fdt_strerror(ret);
		return -EIO;
	}

	select = cpu_to_fdt32(vpd->backlight_select);
	ret = fdt_setprop(blob, node, "select", &select, sizeof(select));
	if(ret < 0) {
		printf("Error setting backlight driver type select: %s\n", fdt_strerror(ret));
		fdt_strerror(ret);
		return -EIO;
	}

	return 0;
}
	
void ft_lcd_setup(void *blob, bd_t *bd)
{
	vga_panel_data *vpd;

	if(!panel_info.priv)
		return;

	vpd = (vga_panel_data *)panel_info.priv;

	if (vpd->dt) {
		fdt_fixup_display_timings(blob, TIMING_NODE, vpd->dt);
#ifdef CONFIG_LXK_PANEL_VGA_PEGMATITE
		/* ignore scalars if zero */
		if(vpd->hscalar > 0) {
			do_fixup_by_path_u32(blob, LCD_NODE, "upscale_h", (u32)vpd->hscalar, 1);
		}
		else if(vpd->hscalar < 0) {
			do_fixup_by_path_u32(blob, LCD_NODE, "downscale_h", (u32)(-vpd->hscalar), 1);
		}

		if(vpd->vscalar > 0) {
			do_fixup_by_path_u32(blob, LCD_NODE, "upscale_v", (u32)vpd->vscalar, 1);
		}
		else if(vpd->vscalar < 0) {
			do_fixup_by_path_u32(blob, LCD_NODE, "downscale_v", (u32)(-vpd->vscalar), 1);
		}
#endif
	}
	fdt_fixup_backlight(blob, vpd);
	fdt_fixup_gamma(blob, vpd);
}
#endif

/**
 * lxk_vram_size - get video buffer size
 * 
 * This will return the buffer size allocated by lcd_setmem(), which
 * happens before the real size is known.
 */
unsigned int lxk_vram_size(void)
{
	unsigned int size;
	int line_length = (LCD_MAX_COLS * NBITS(LCD_COLOR_DEPTH)/ 8);

	size = line_length * LCD_MAX_ROWS;

	/* Round up to nearest full page */
	size = (size + (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1);
	
	return size;
}

/**
 * lxk_eeprom_version_ge - test for minimum eeprom version
 * @major - minimum version major
 * @minor - minimum version minor
 * @vpi - pointer to eeprom data image
 */
int lxk_eeprom_version_ge(int major, int minor, vga_panel_info *vpi) {
	if((vpi->version_major > major) ||
				((vpi->version_major == major) &&
				(vpi->version_minor >= minor))) {
					
		return 1;
	}
	else {
		return 0;
	}
}

/**
 * lxk_eeprom_version_eq - test for exact eeprom version
 * @major - required version major
 * @minor - required version minor
 * @vpi - pointer to eeprom data image
 */
int lxk_eeprom_version_eq(int major, int minor, vga_panel_info *vpi) {
	if((vpi->version_major == major) ||
				(vpi->version_minor == minor)) {
					
		return 1;
	}
	else {
		return 0;
	}
}

/**
 * lxk_eeprom_version_blank - test for blank or corrupted version
 * @vpi - pointer to eeprom data image
 *
 * Blank or brownout-corrupted state is usually 0xff, so make it illegal in the
 * version fields.
 */
int lxk_eeprom_version_blank(vga_panel_info *vpi) {
    if (vpi->version_major == 0xff) {
        return 1;
    }

    /* v1.255 is valid */
    if (vpi->version_minor == 0xff && vpi->version_major != 1) {
        return 1;
    }

    return 0;
}

/**
 * lxk_eeprom_checksum - verify eeprom contents if supported
 * @vpi - pointer to eeprom data image
 */
int lxk_eeprom_checksum(vga_panel_info *vpi) {
  
  if(!vpi) {
     printf("%s:%d: Null pointer error!\n", __func__, __LINE__);
     return -1;
  }
  
  if(lxk_eeprom_version_ge(2, 3, vpi)) {
     uint16_t eechecksum = 0;
     int check_len = offsetof(vga_panel_info, check_msb);
     int i;
     uint8_t *data = (uint8_t *)vpi;

     for(i=0;i<check_len;i++) {
        /* 
	 * Assume data could be placed after checksum bytes
	 * in later versions.
	 */
        if((i == offsetof(panel_info_t, check_msb)) ||
          		(i == offsetof(panel_info_t, check_lsb))) {
           continue;
        }
        /* panel_info.check_*sb included in sum, this time probably non-zero */
        eechecksum += data[i] & 0x0ff; 
     }
     eechecksum += (data[offsetof(panel_info_t, check_msb)] & 0x0ff) << 8;
     eechecksum += data[offsetof(panel_info_t, check_lsb)] & 0x0ff;
     if(eechecksum != (uint16_t)LXK_EEPROM_CHECKSUM_MAGIC) {
        printf("%s:%d: Uicc eeprom may be corrupted!\n",__func__, __LINE__);
        return -1;
     }
     
     return 0;
  }
  else {
     printf("%s:%d: Checksum not supported by this eeprom revision.\n",
     						__func__, __LINE__);
     return 0;
  }
}
  
void lxk_armada_dump_regs(void *reg_base)
{
#ifdef __LXK_VGA_MV61FB_DEBUG__
   unsigned int
      i;

	printf("******** ARMADA LCD BRIDGE registers ********\n");

	for (i=0; i<24; i++) {
	   SHOWREGW(reg_base, LCD_WIN_CTRL(0) + i*4);
	}

	printf("\n\n******** ARMADA LCD registers *****\n");

	SHOWREGW(reg_base, LCD_SPU_ADV_REG);  // 0x84

	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_Y0); // 0xc0
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_U0);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_V0);
	SHOWREGW(reg_base, LCD_CFG_DMA_START_ADDR_0);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_Y1);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_U1);
	SHOWREGW(reg_base, LCD_SPU_DMA_START_ADDR_V1);
	SHOWREGW(reg_base, LCD_CFG_DMA_START_ADDR_1);

	SHOWREGW(reg_base, LCD_SPU_DMA_PITCH_YC); // 0xe0
	SHOWREGW(reg_base, LCD_SPU_DMA_PITCH_UV);
	SHOWREGW(reg_base, LCD_SPUT_DMA_OVSA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_DMA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_DZM_HPXL_VLN);
	SHOWREGW(reg_base, LCD_CFG_GRA_START_ADDR0);
	SHOWREGW(reg_base, LCD_CFG_GRA_START_ADDR1);
	SHOWREGW(reg_base, LCD_CFG_GRA_PITCH);
	SHOWREGW(reg_base, LCD_SPU_GRA_OVSA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_GRA_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPU_GZM_HPXL_VLN);
	SHOWREGW(reg_base, LCD_SPUT_V_H_TOTAL);
	SHOWREGW(reg_base, LCD_SPU_V_H_ACTIVE);
	SHOWREGW(reg_base, LCD_SPU_H_PORCH);
	SHOWREGW(reg_base, LCD_SPU_V_PORCH);
	SHOWREGW(reg_base, LCD_SPU_BLANKCOLOR);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_Y);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_U);
	SHOWREGW(reg_base, LCD_SPU_COLORKEY_V);

	SHOWREGW(reg_base, LCD_SPU_GAMMA_RDDAT);
	SHOWREGW(reg_base, LCD_SPU_PALETTE_RDDAT);

	SHOWREGW(reg_base, LCD_SPU_SMPN_CTRL);

	SHOWREGW(reg_base, LCD_SPU_DMA_CTRL0);
	SHOWREGW(reg_base, LCD_SPU_DMA_CTRL1);
	SHOWREGW(reg_base, LCD_SPU_SRAM_CTRL);
	SHOWREGW(reg_base, LCD_SPU_SRAM_WRDAT);
	SHOWREGW(reg_base, LCD_SPU_SRAM_PARA1);
	SHOWREGW(reg_base, LCD_CFG_SCLK_DIV);
	SHOWREGW(reg_base, LCD_SPU_CONTRAST);
	SHOWREGW(reg_base, LCD_SPU_SATURATION);
	SHOWREGW(reg_base, LCD_SPU_CBSH_HUE);
	SHOWREGW(reg_base, LCD_SPU_DUMB_CTRL);
	SHOWREGW(reg_base, SPU_IOPAD_CONTROL);
	SHOWREGW(reg_base, SPU_IRQ_ENA);
	SHOWREGW(reg_base, SPU_IRQ_ISR);

	SHOWREGW(reg_base, LCD_CLK_CFG0_REG);
	SHOWREGW(reg_base, LCD_CLK_CFG1_REG);
	SHOWREGW(reg_base, LCD_LVDS_CLK_CFG); 

	/*
	 * LVDS I/O pad registers are not in the video section, but very important.
	 */
	SHOWREGW(0xd0000000, 0x182f0);


	printf("********************************\n");
#endif
}

void lxk_armada_paint_buffer(void *buffer, unsigned int bufsize)
{ 
#ifdef __LXK_VGA_MV61FB_DEBUG__
	/*
	 * Paint buffer with some dummy data for debug.
	 */
	int i;
	short tmp;
	short *ip = (short *)buffer;

	for (i = 0; i < (bufsize >> 1); i++) {
		tmp = (i & 0x0ff);
		tmp |= tmp << 8;
		if(i & 32) {
			ip[i] = tmp;
		}
		else {
			if(i & 8)
				ip[i] = 0x7fef;
			else
				ip[i] = 0xfbff;
		}
	}
#else
	memset(buffer, 0, bufsize);
#endif
}

/*
 */
void lcd_ctrl_init (void *buffer)
{
	void *reg_base = (void *)LCD_BASE;
	LXK_PANEL_TYPE display_type;

#ifdef CONFIG_LXK_PANEL
	lxk_reset_panel();
#endif

	display_type = lxk_panel_get_type();
	LXKPANELDBG("%s:%d: panel display type = %d\n", __func__, __LINE__, display_type);

	if(display_type == LXK_PANEL_TYPE_INVALID) {
	        printf("%s:%d: no vga panel found\n", __func__, __LINE__);
		return;
	}
		
	switch(display_type) {
//	case LXK_PANEL_TYPE_320X240:
//		lxk_mv61fb_smart_panel_init(reg_base, panel, buffer);
//		break;
	case LXK_PANEL_TYPE_480X272:
	case LXK_PANEL_TYPE_800X480:
	case LXK_PANEL_TYPE_800X480_SCALED:
	case LXK_PANEL_TYPE_480X272_FIXW:
	case LXK_PANEL_TYPE_800X480_FIXW:
	case LXK_PANEL_TYPE_480X272_NON_SEQ:
	case LXK_PANEL_TYPE_1024X600:
	case LXK_PANEL_TYPE_DYNAMIC:
		lxk_armada_dumb_panel_init(reg_base, buffer);
		break;
	default:
		return;
	}
}																																																	
