#include <common.h>
#include <stdarg.h>
#include <i2c.h>
#include <asm/io.h>
#include <lcd.h>
#include <lxk_panel.h>
#include <stdio_dev.h>
#include <video_font.h>
#include "atmelpnl_switch.h"
#include "lxk_panel_common.h"

const gpio_map_t lxk_common_gpio_map[] = {
    { MCU_GPIOA_OER_REG, MCU_GPIOA_OUT_REG },
    { MCU_GPIOB_OER_REG, MCU_GPIOB_OUT_REG },
    { MCU_GPIOC_OER_REG, MCU_GPIOC_OUT_REG },
    { MCU_GPIOD_OER_REG, MCU_GPIOD_OUT_REG },
};

/* This common map supports all current panels except LED and 2line */
const key_map_t lxk_common_key_map[] = {
    { NM_RETURN, ATM_SW_ROW4COL2 }, /* Back */
    { NM_HOME,   ATM_SW_ROW3COL2 }, /* Home */
    { NM_STOP,   ATM_SW_ROW2COL2 }, /* Stop */
    { NM_UP,     ATM_SW_ROW3COL3 }, /* Up */
    { NM_DOWN,   ATM_SW_ROW4COL3 }, /* Down */
    { NM_LEFT,   ATM_SW_ROW1COL3 }, /* Left */
    { NM_RIGHT,  ATM_SW_ROW2COL3 }, /* Right */
    { NM_SELECT, ATM_SW_ROW5COL0 }, /* Select */
    { NM_1,      ATM_SW_ROW2COL1  }, /* 1 */
    { NM_2,      ATM_SW_ROW0COL1 }, /* 2 */
    { NM_3,      ATM_SW_ROW0COL0 }, /* 3 */
    { NM_4,      ATM_SW_ROW1COL0 }, /* 4 */
    { NM_5,      ATM_SW_ROW3COL1 }, /* 5 */
    { NM_6,      ATM_SW_ROW2COL0 }, /* 6 */
    { NM_7,      ATM_SW_ROW3COL0 }, /* 7 */
    { NM_8,      ATM_SW_ROW4COL0 }, /* 8 */
    { NM_9,      ATM_SW_ROW1COL2 }, /* 9 */
    { NM_0,      ATM_SW_ROW4COL1 }, /* 0 */
    { NM_STAR,   ATM_SW_ROW1COL1 }, /* * */
    { NM_POUND,  ATM_SW_ROW0COL2 }, /* # */
    { 0xff,      0xff }  /* End of Key Map */
};

/*
 * This is closely based on eeprom_read() from /common/cmd_eeprom.c.
 * It was ported here to remove dependencies on board config settings.
 */
int lxk_panel_eeprom_read (unsigned dev_addr, unsigned offset, unsigned addrlen, uchar *buffer, unsigned cnt)
{
	unsigned end = offset + cnt;
	unsigned blk_off;

	/* Read data until done or would cross a page boundary.
	 * We must write the address again when changing pages
	 * because the next page may be in a different device.
	 */
	while (offset < end) {
		volatile unsigned len;
		unsigned maxlen;
		uint8_t chip;
		volatile unsigned int addr;
		volatile unsigned int alen = addrlen;

		switch(addrlen) {
		case 1:
			blk_off = offset & 0xFF;	/* block offset */

			chip = offset >> 8;		/* block number */
			addr = blk_off;			/* block offset */
			break;
		case 2:
			blk_off = offset & 0xFF;	/* block offset */

			chip = offset >> 16;		/* block number */
			addr = offset & 0x0FF00;	/* upper address octet */
			addr |= blk_off;		/* lower address octet */
			break;
		default:
			return -1;
		}

		chip |= dev_addr;		/* insert device address */

		len = end - offset;

		maxlen = 0x100 - blk_off;
		if (maxlen > I2C_RXTX_LEN)
			maxlen = I2C_RXTX_LEN;
		if (len > maxlen)
			len = maxlen;
		if (i2c_read(chip, addr, alen, buffer, len)) {
			return -1;
		}

		buffer += len;
		offset += len;
	}

	return 0;
}

int lxk_panel_i2c_write(lxk_panel_i2c_info_t *i2c_info,
                        unsigned char *msg, int msglen) {
    int rc = 0;
    int retries = i2c_info->write_retries + 1;
    unsigned int
       orig_bus_num = I2C_GET_BUS();

    /* The 'volatile' is needed because on some writes trans_addr_len was being read as non-zero despite the hardcode
     * below. This made the underlying i2c framwork incorrectly submit the data.
     */
    volatile unsigned int trans_addr;
    volatile unsigned int trans_addr_len;

    if(i2c_info->bus_num == -1)
        i2c_info->bus_num = CONFIG_PANEL_I2C_BUS;
    if(i2c_info->bus_num == -1)
        return -1;

    if (!msglen || !msg) goto done;


    switch (i2c_info->protocol) {
    case PROTOCOL_ZFORCE:
    case PROTOCOL_RAW:
    case PROTOCOL_GEN1:
        /* offset address not used, command is embedded in msg */
        trans_addr = 0;
        trans_addr_len = 0;
        I2C_SET_BUS(i2c_info->bus_num);

        while (retries--) {
            rc = i2c_write(i2c_info->dev_addr, trans_addr, trans_addr_len, msg, msglen);
            mdelay(1);
            if (!rc) break;
        }
        /* Restore I2C bus number */
        I2C_SET_BUS(orig_bus_num);
        break;
    case PROTOCOL_GEN2:
        {
            /*
             * The uboot i2c api allows up to 4 byte offset address, sent MSB first.
             * Use the first byte for control info.
             * The remaining bytes are unused, command is embedded in msg
             */

            /* For GEN2, we must reconstruct the data packet to include the
             * header information (UIBC_WRITE | msglen & UIBC_DATA_SIZE_MASK).
             * This adds 1 to the size of the message we are sending out via
             * i2c.
             */
            unsigned char g2packet[msglen + 1];
            trans_addr = 0;
            trans_addr_len = 0;
            /* Subtract 1 from msglen to get length of actual data */
            g2packet[0] = UIBC_WRITE | ((msglen - 1) & UIBC_DATA_SIZE_MASK);
            /* copy original payload into new packet */
            memcpy(&g2packet[1], msg, msglen);

            I2C_SET_BUS(i2c_info->bus_num);

            while (retries--) {
                rc = i2c_write(i2c_info->dev_addr, trans_addr, trans_addr_len, g2packet, sizeof(g2packet));
                mdelay(1);
                if (!rc) break;
            }
            /* Restore I2C bus number */
            I2C_SET_BUS(orig_bus_num);
            break;
        }
    default:
        LXKPANELDBG("%s:%d: Error! Unknown protocol!\n", __func__, __LINE__);
        goto done;
    }

done:
    return rc;
}

int lxk_panel_i2c_read(lxk_panel_i2c_info_t *i2c_info,
                       unsigned int addr, int addrlen,
                       unsigned char *rcv, int rcvlen) {
    int rc = 0;
    int retries = i2c_info->read_retries + 1;
    int
       orig_bus_num = I2C_GET_BUS();
    unsigned int trans_addr;
    unsigned int trans_addr_len;

    if(i2c_info->bus_num == -1)
        i2c_info->bus_num = CONFIG_PANEL_I2C_BUS;
    if(i2c_info->bus_num == -1)
        return -1;

    if (!rcvlen || !rcv) goto done;

    switch (i2c_info->protocol) {
    case PROTOCOL_ZFORCE:
    case PROTOCOL_RAW:
        trans_addr = 0;
        trans_addr_len = 0;
        break;
    case PROTOCOL_EEPROM:
        if (addrlen > 2) {
            rc = -1;
            goto done;
        }
        trans_addr = addr;
        trans_addr_len = addrlen;
        break;
    case PROTOCOL_GEN1:
        if (addrlen > (sizeof(int))) {
            rc = -1;
            goto done;
        }
        trans_addr = addr;
        trans_addr_len = addrlen;
        break;
    case PROTOCOL_GEN2:
        /*
         * The uboot i2c api allows up to 4 byte offset address, sent MSB first.
         * Use the first byte for control info.
         */
        if (addrlen > (sizeof(int) - 1)) {
            rc = -1;
            goto done;
        }
        trans_addr = addr;
        trans_addr |= UIBC_READ << 8;
        trans_addr |= (rcvlen & UIBC_DATA_SIZE_MASK) << 8;
        trans_addr_len = addrlen + 1;
        break;
    default:
        LXKPANELDBG("%s:%d: Error! Unknown protocol!\n", __func__, __LINE__);
        goto done;
    }

    I2C_SET_BUS(i2c_info->bus_num);

    while (retries--) {
        if(i2c_info->protocol == PROTOCOL_EEPROM) {
        	rc = lxk_panel_eeprom_read(i2c_info->dev_addr, trans_addr, trans_addr_len, rcv, rcvlen);
        }
        else {
        	rc = i2c_read(i2c_info->dev_addr, trans_addr, trans_addr_len, rcv, rcvlen);
        	udelay(1000);
        }
        if (!rc) break;
    }

    /*
     * Restore I2C bus number
     */
    I2C_SET_BUS(orig_bus_num);

done:
    return rc;
}

int lxk_panel_read_buttons(lxk_panel_i2c_info_t *i2c_info, const key_map_t *map, unsigned int buttons, int toggle) {
    unsigned int btns = 0;
    unsigned char
       dat,
       cmd,
       more = 0,
       sw_state = 0,
       btn_val = 0, i;
    int rc;

    int timeout = 100;

    /*
     * Toggling previous buttons is only used for svcerr support on LED panels.
     * Not sure WHY it's used, but preserving functionality.
     */
    if (toggle) btns = buttons;

    /* Need to add this delay in order for all key
     * presses be stored in the key FIFO regs */
    udelay(1000 * 100);

    while (timeout--) {   /* An infinite loop here will hang if there is no panel attached... */
        udelay(1000 * 100);
        /* Read contents of the FIFO buffer */
        switch (i2c_info->protocol) {
        case PROTOCOL_GEN1:
            cmd = MCU_KEY_FIFO_REG;
            break;
        case PROTOCOL_GEN2:
            cmd = UIBC_KEY_SCAN_FIFO;
            break;
        default:
            LXKPANELDBG("%s:%d: Error! Unknown or unsupported protocol!\n", __func__, __LINE__);
            goto done;
        }
        rc = lxk_panel_i2c_read(i2c_info, cmd, 1, &dat, 1);
        if (rc) {
            LXKPANELDBG("%s:%d: Unable to read KEY_FIFO, rc: %d\n", __func__, __LINE__, rc);
            goto done;
        }

        /* Fetch the more flag and fetch the actual button value */
        more = (dat >> 7) & 0x01;
        sw_state = (dat >> 6) & 0x01;
        btn_val = dat & 0x3F;

        /* Look for the logical button equivalent of the HW button data we got */
        for (i = 0; map[i].logic_btn != 0xff; i++) {
            if (map[i].hw_btn == btn_val) {
                /* Check the switch state */
                if (sw_state) {
                    btns |= map[i].logic_btn;
                } else if (toggle) {
                    /*
                     * Toggling previous buttons is only used for svcerr support on LED panels.
                 * Not sure WHY it's used, but preserving functionality.
                 */
                    btns ^= map[i].logic_btn;
                }
            }
        }

        /* Check the "more" flag */
        if (more == 0) break;
    } /* End of while */

done:
    return (btns);
}

unsigned int lxk_panel_buttons_alias(unsigned int buttons) {
    unsigned int btns = buttons;
    /*
     * 4 button Clear All NVRAM button push, registers as 6 buttons on Vertical Btn Map.
     */
#define PORBTN_CLEAR_ALL_NVRAM (NM_2 | NM_6 | NM_8 | NM_7)
    if (btns == (NM_2 | NM_3 | NM_4 | NM_6 | NM_7 | NM_8)) btns = PORBTN_CLEAR_ALL_NVRAM;

    return (btns);
}

/*-----------------------------------------------------------------------------
 *
 * Name:        initKeyScan
 *
 * Description: Initialize the key scanning functionality of the mcu
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:       Steps for initialization:
 *              1. Enable the row and column switch scanning I/O specified
 *                 in the SW_ENAB register.
 *              2. Bit 6 - 0: Enable switch scanning
 *                         1: Disable switch scanning (default setting)
 *              3. Bits 5:3 - Select number of switch columns.  Columns
 *                 enabled are SW_COL through 0
 *                 000 - SWCol 0
 *                 001 - SWCol 1 to 0
 *                 010 - SWCol 2 to 0
 *                 011 - SWCol 3 to 0
 *                 100 - SWCol 4 to 0
 *                 101 - SWCol 5 to 0
 *                 110 - SWCol 6 to 0
 *                 111 - SWCol 7 to 0
 *              4. Bits 2:0 - Select number of switch rows.  Rows enabled
 *                 are SW_ROW through 0
 *                 000 - SWRow 0
 *                 001 - SWRow 1 to 0
 *                 010 - SWRow 2 to 0
 *                 011 - SWRow 3 to 0
 *                 100 - SWRow 4 to 0
 *                 101 - SWRow 5 to 0
 *                 110 - SWRow 6 to 0
 *                 111 - SWRow 7 to 0
 *
 *              The bit 6 enable was defined by the uC api v1 but not implemented. It is still
 *              maintained in the uC api v2, so continue setting it for compatibility.
 *
 *---------------------------------------------------------------------------*/
int lxk_panel_keyscan_init(lxk_panel_i2c_info_t *i2c_info, unsigned char row, unsigned char col) {
    int rc = 0;
    unsigned char
       msg[2];

    /* Enable switch scanning       */
    msg[0] = MCU_SW_ENAB_REG;
    msg[1] = (row - 1) & 0x7;
    msg[1] |= ((col - 1) & 0x7) << 3;
    msg[1] |= MCU_SW_ENAB_ON; /* no effect on BB or LB1 api, but keep it common */
    rc = lxk_panel_i2c_write(i2c_info, msg, sizeof(msg));
    if (!rc) goto done;

    msg[0] = MCU_IRQ_MASK_REG;
    lxk_panel_i2c_read(i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
    msg[1] |= MCU_IRQ_MASK_KEYSCAN;
    rc = lxk_panel_i2c_write(i2c_info, msg, sizeof(msg));
    if (!rc) goto done;

    msg[0] = MCU_KEY_RATE_REG;
    msg[1] = 0x10;
    rc = lxk_panel_i2c_write(i2c_info, msg, sizeof(msg));

done:
    return rc;
}

int lxk_panel_i2c_probe(lxk_panel_i2c_info_t *i2c_info) {
    int rc = 0;
    int retries = i2c_info->read_retries + 1;
    int
       orig_bus_num = I2C_GET_BUS();

    LXKPANELDBG("%s:%d: probing i2c address 0x%02x\n", __func__, __LINE__, i2c_info->dev_addr << 1);

    if(i2c_info->bus_num == -1)
        i2c_info->bus_num = CONFIG_PANEL_I2C_BUS;
    if(i2c_info->bus_num == -1)
        return -1;

    I2C_SET_BUS(i2c_info->bus_num);

    while (retries--) {
        switch (i2c_info->protocol) {
        case PROTOCOL_GEN1:
        case PROTOCOL_RAW:
            rc = i2c_probe(i2c_info->dev_addr);
            break;
        case PROTOCOL_EEPROM:
            {
                unsigned char data;
                rc = lxk_panel_i2c_read(i2c_info, 0, 1, &data, 1);
            }
            break;
        case PROTOCOL_GEN2:
            /* probe not supported, use dummy version read */
            {
                int cmd = UIBC_FW_VERSION_MAJ;
                unsigned char data;
                rc = lxk_panel_i2c_read(i2c_info, cmd, 1, &data, 1);
            }
            break;
        case PROTOCOL_ZFORCE:
            LXKPANELDBG("%s:%d: Error! Probing not supported by zforce protocol!\n", __func__, __LINE__);
            rc = -1;
            retries = 0;
            break;
        default:
            LXKPANELDBG("%s:%d: Error! Unknown protocol!\n", __func__, __LINE__);
            rc = -1;
            retries = 0;
            break;
        }
        udelay(1000);
        if (!rc) break;
    }

    /*
     * Restore I2C bus number
     */
    I2C_SET_BUS(orig_bus_num);

    return rc;
}

/*-----------------------------------------------------------------------------
 *
 * Description: Resets the mcu by sending the reset command
 *
 * Input:       None
 *
 * Output:
 *
 * Notes:    The uC performs a software reset when the I2C master writes the
 *           reset command = 0x99.
 *           This function may be called before the panel type has been detected,
 *           but i2c_info will need to be pre-initialized for the panel we're
 *           looking for.
 *---------------------------------------------------------------------------*/

int lxk_panel_reset_mcu(lxk_panel_i2c_info_t *i2c_info) {
    unsigned char cmd;
    int rc;
    int retries = 5;

    /* Send the reset command = 0x99 */
    cmd = MCU_RESET_CMD;
    while (retries) {
        rc = lxk_panel_i2c_write(i2c_info, &cmd, 1);

        udelay(10 * 1000);

        /* Verify that reset was successful. */
        if (!rc) rc = lxk_panel_i2c_probe(i2c_info);

        if (rc) {
            retries--;
        } else {
            break;
        }
    }

    return (rc);
}

/*-----------------------------------------------------------------------------
 *
 * Name:        lxk_panel_disp_ctrl
 *
 * Description: Detect and control the reset ouputs
 *
 * Input:       None
 *
 * Output:      None
 *
 * Notes:       Eeprom must be read before this is called. For early or blank
 *              eeprom, reasonable default values for flags2 and flags3 should
 *              already be set.
 *---------------------------------------------------------------------------*/
int lxk_panel_disp_ctrl(lxk_panel_i2c_info_t *i2c_info, vga_panel_info *vpi, reset_type_t which, int value) {
    int rc = 0;
    unsigned char msg[2];
    uint8_t pio, pio_block;

    switch (which) {
    case PRIMARY_RESET_OUT:
        pio = PIO(vpi->flags2);
        pio_block = PIO_BLOCK(vpi->flags2);
        break;
    case SECONDARY_RESET_OUT:
        if (!PIO_ENABLED(vpi->flags3)) return rc;
        pio = PIO(vpi->flags3);
        pio_block = PIO_BLOCK(vpi->flags3);
        break;
    default:
        return rc;
    }

    /* set the desired pin level */
    msg[0] = lxk_common_gpio_map[pio_block].out_reg;
    /* read current value */
    if (!rc)
        rc = lxk_panel_i2c_read(i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
    /* set appropriate bit */
    SET_BIT(msg[1], pio, (value ? 1 : 0));
    if (!rc)
        rc = lxk_panel_i2c_write(i2c_info, msg, 2);

    /* configure the pin as an ouput */
    msg[0] = lxk_common_gpio_map[pio_block].oer_reg;
    /* read current value */
    if (!rc)
        rc = lxk_panel_i2c_read(i2c_info, msg[0], 1, &msg[1], sizeof(msg[1]));
    /* set appropriate bit */
    SET_BIT(msg[1], pio, OER_OUTPUT);
    if (!rc)
        rc = lxk_panel_i2c_write(i2c_info, msg, 2);

    return rc;
}

