#ifndef LXK_GEN2_EEPROM_H
#define LXK_GEN2_EEPROM_H

/* We want to keep this header file common */
#ifdef GENERATE_PNL_EEPROM
#include "display_timing.h"
#else
#include <linux/types.h>
#include <linux/stddef.h>
#include <video/display_timing.h>
#endif

#ifndef ENTRY_DEFAULT
#define ENTRY_DEFAULT(entry, val) \
	.entry.min = val, \
	.entry.typ = val, \
	.entry.max = val
#endif

#define MAX_RECORDS	4
#define MAX_NAME_SIZE	32
#define LXK_GEN2_MAGIC_NUMBER	0x2014
#define DEVICE_SIZE 1024

#define val16(hi, lo) (((hi & 0x0ff) << 8) | (lo & 0x0ff))

/*
 * If the struct definitions in video/display_timing.h change, we may
 * need a translation layer for compatibility. For now, just typdef them as-is.
 */
typedef struct timing_entry lxk_gen2_timing_entry;
typedef struct display_timing linux_display_timing;

/*
 * LXK_GEN2_DISPLAY_TIMING record layout
 * This is similar to typedef struct lxk_gen2_display_info_t in include/lxk_panel.h, but
 * allocates fixed space in record for strings instead of using pointers. It is also
 * intended for use by a translation layer to maintain layout stability.
 */
typedef struct {
	char name[MAX_NAME_SIZE];	// 32 bytes
	char vendor[16];		// 16 bytes
	char model[16];			// 16 bytes
	linux_display_timing 	dt;	// 112 bytes
} lxk_gen2_display_nv;			// 176 bytes total content per record

typedef struct {
	u8			volts;
	u8			centivolts;
} lxk_panel_voltage;

/*
 * PANEL_HW record layout
 * Unused entries should be cleared to zero
 */
typedef struct {
	u8			bl_duty_max;
	u8			bl_duty_init;
	u8			bl_fb_millivolts;
	u8			proximity_gain;
	u8			bit_depth;
	u8			lvds_option;
	lxk_panel_voltage	vcom;
	lxk_panel_voltage	vgh;
	lxk_panel_voltage	vgl;
	lxk_panel_voltage	avdd;
	lxk_panel_voltage	dvdd;
	u8			bl_select;
} lxk_gen2_panel_hw;

/*
 * LCD gamma LUT per channel or common
 * Overkill to wrap single array in a struct, but stay consistent with
 * other record types.
 */
typedef struct {
	u8 data[256];
} gamma_lut_single;

/*
 * Defined record types.
 * The EDID structs are defined in include/edid.h and correspond to VESA/HDMI EDID v1.3.
 * More record types may be added as new requirements surface.
 */
typedef enum {
	UNUSED_ENTRY = 0,
	LXK_GEN2_DISPLAY_TIMING,	// record layout defined by typedef struct lxk_gen2_display_nv
	LINUX_DISPLAY_TIMING,		// record layout defined by typedef struct linux_display_timing
	EDID_INFO,			// record layout defined by struct edid1_info in include/edid.h
	EDID_DETAILED,			// record layout defined by struct edid_detailed_timing in include/edid.h
	PANEL_HW,			// record layout defined by typedef struct lxk_gen2_panel_hw
	RAW,				// arbitrary data
	GAMMA_LUT_COMMON,		// gamma LUT for all channels defined by typedef struct gamma_lut 
	GAMMA_LUT_RED,			// gamma LUT for red channel defined by typedef struct gamma_lut 
	GAMMA_LUT_GREEN,		// gamma LUT for green channel defined by typedef struct gamma_lut 
	GAMMA_LUT_BLUE,			// gamma LUT for blue channel defined by typedef struct gamma_lut 
	MAX_RECORD_TYPES,               // always last entry, add new record types above this entry
} __attribute__((packed)) DISPLAY_RECORD_TYPE;

/*
 * Each record has a corresponding directory entry.
 */
typedef struct {
	DISPLAY_RECORD_TYPE	type;
	u8			size_hi;	// size of record data MSB
	u8			size_lo;	// size of record data LSB
	u8			offset_hi;	// offset of record within data section MSB
	u8			offset_lo;	// offset of record within data section LSB
	u8			checksum_hi;	// record checksum MSB
	u8			checksum_lo;	// record checksum LSB
	u8			reserved;
} lxk_gen2_display_record_directory_entry_nv;

/*
 * Directory / Header layout
 * Everything in the directory/header is coded to be endian-independent. However,
 * the records in the data section are memory images and must be consistent. All current
 * products are little-endian for the foreseable future ....
 */
typedef struct {
	u8	magic_hi;					// magic number MSB
	u8	magic_lo;					// magic number LSB
	u8	data_size_hi;					// data section size MSB
	u8	data_size_lo;					// data section size LSB
	u8	layout_version_major;				// increment if this struct changes
	u8	layout_version_minor;
	u8	content_version_major;				// increment if content changes
	u8	content_version_minor;
	u8	content_flags;					// TBD
	u8	reserved[21];
	lxk_gen2_display_record_directory_entry_nv directory[MAX_RECORDS];
	// checksum bytes must be immediately before the data section
	u8	checksum_hi;					// record_directory checksum MSB
	u8	checksum_lo;					// record_directory checksum LSB
	// zero-length array must be last entry, allows struct to be variable size
	u8	data[0];					// contains variable length display records
} lxk_gen2_display_info_eeprom;

#define RECORD_DIRECTORY_CHECKSUM_SIZE (sizeof(lxk_gen2_display_info_eeprom) - 2)

#endif

