#include <common.h>
#include <asm/dw_i2c_regmasks.h>
#include <asm/dw_i2c_regstructs.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/board_priv.h>

static unsigned int i2c_bus_num __attribute__ ((section (".data"))) = CONFIG_I2C_DEFAULT_BUS;

static uint32_t i2c_bus_address[] = {
	APB_I2C0_BASE,
	APB_I2C1_BASE,
	APB_I2C2_BASE,
	APB_I2C3_BASE
};

unsigned int i2c_get_bus_num(void) {
	return i2c_bus_num;
}

int i2c_set_bus_num(unsigned int bus) {
	i2c_bus_num = bus;
	return 0;
}

static uint32_t i2c_get_bus(unsigned char chip) {
	if (i2c_bus_num < CONFIG_SYS_MAX_I2C_BUS)
		return i2c_bus_address[i2c_bus_num];
	else
		return i2c_bus_address[0];
}

static int i2cwriteb(volatile struct I2C_REGS_s *i2c, unsigned char byte) {
	unsigned int timeout = 1000;

	while ((!(i2c->IC_STATUS & I2C_IC_STATUS_TFE_MASK)) && timeout) {
		--timeout;
		udelay(1);
	}

	if (timeout == 0)
		return -1;

	i2c->IC_DATA_CMD = byte & I2C_IC_DATA_CMD_DAT_MASK;

	return 0;
}

int i2c_read(unsigned char chip, unsigned int addr, int alen, unsigned char *buf, int len) {
	volatile struct I2C_REGS_s *i2c = (volatile struct I2C_REGS_s *)i2c_get_bus(chip);
	unsigned int timeout;
	int bytes_requested=0;
	int bytes_read=0;

	i2c->IC_ENABLE = 0;
	i2c->IC_TAR = chip & 0x7f;
	i2c->IC_ENABLE = 1;

	if ((alen < 0) || (alen > 4)) {
		return 1;
	}

	while (alen) {
		alen--;
		i2cwriteb(i2c, (addr >> (alen * 8)) & 0xff);
	}


	/*
	 * If the command FIFO runs dry, the transaction will stop.  Avoid undesired
	 * restart operations (which breaks communication with certain devices) by
	 * making sure it doesn't run dry.
	 * Queue up as many read commands now as possible.
	 * NOTE: This algorithm assumes that the CPU can keep up with the I2C data
	 * transfer rate.
	 */
	while (bytes_requested < len && (i2c->IC_STATUS & I2C_IC_STATUS_TFNF_MASK)) {
		i2c->IC_DATA_CMD = I2C_IC_DATA_CMD_CMD_MASK;
		bytes_requested++;
	}

	while (bytes_read < len) {
		timeout = 1000;

		while ((!(i2c->IC_STATUS & I2C_IC_STATUS_RFNE_MASK)) && timeout) {
			--timeout;
			udelay(1);
		}

		if (timeout == 0)
			return -1;

		while (i2c->IC_STATUS & I2C_IC_STATUS_RFNE_MASK) {
			*buf++ = (unsigned char)(i2c->IC_DATA_CMD & I2C_IC_DATA_CMD_DAT_MASK);
			bytes_read++;
		}

		/*
		 * If there is now room in the command fifo for another command, and if
		 * needed, go ahead and queue it up.
		 */
		while (bytes_requested < len && (i2c->IC_STATUS & I2C_IC_STATUS_TFNF_MASK)) {
			i2c->IC_DATA_CMD = I2C_IC_DATA_CMD_CMD_MASK;
			bytes_requested++;
		}
	}

	/*
	 * Make sure that the transaction finishes without errors.
	 */
	while (i2c->IC_STATUS & I2C_IC_STATUS_MST_ACTIVITY_MASK) {
		udelay(1);
	}

	return 0;
}

int i2c_write(unsigned char chip, unsigned int addr, int alen, unsigned char *buf, int len) {
	volatile struct I2C_REGS_s *i2c = (volatile struct I2C_REGS_s *)i2c_get_bus(chip);

	i2c->IC_ENABLE = 0;
	i2c->IC_TAR = chip & 0x7f;
	i2c->IC_ENABLE = 1;

	if ((alen < 0) || (alen > 4)) {
		return 1;
	}

	while (alen) {
		alen--;
		i2cwriteb(i2c, (addr >> (alen * 8)) & 0xff);
	}

	while (len) {
		i2cwriteb(i2c, *buf++);
		len--;
	}

	/*
	 * Make sure that the transaction finishes without errors.
	 */
	while (i2c->IC_STATUS & I2C_IC_STATUS_MST_ACTIVITY_MASK) {
		udelay(1);
	}

	if (i2c->IC_RAW_INTR_STAT & I2C_IC_RAW_INTR_STAT_TX_ABRT_MASK) {
		return -1;
	}

	return 0;
}

void dw_i2c_init(uint32_t base) {
	volatile struct I2C_REGS_s *i2c = (volatile struct I2C_REGS_s *)base;
	uint32_t bus_freq = board_get_bus_freq_hz( ) / 1000;

	i2c->IC_ENABLE = 0;
	i2c->IC_TAR = 0;

	/* TODO change hard coded values to be based off of speed */
	i2c->IC_SS_SCL_HCNT = (bus_freq/200) & I2C_IC_SS_SCL_HCNT_IC_SS_SCL_HCNT_MASK;
	i2c->IC_SS_SCL_LCNT = (bus_freq/200) & I2C_IC_SS_SCL_LCNT_IC_SS_SCL_LCNT_MASK;

	/* TODO tune these values */
	i2c->IC_FS_SCL_HCNT = (bus_freq/800) & I2C_IC_FS_SCL_HCNT_IC_FS_SCL_HCNT_MASK;
	i2c->IC_FS_SCL_LCNT = (bus_freq/800) & I2C_IC_FS_SCL_LCNT_IC_FS_SCL_LCNT_MASK;

	i2c->IC_CON = I2C_IC_CON_MASTER_MODE_MASK | (0x1 << I2C_IC_CON_SPEED_SHIFT) | I2C_IC_CON_IC_SLAVE_DISABLE_MASK | I2C_IC_CON_IC_RESTART_MASK;
}

void i2c_init(int speed, int slaveaddr) {
	int i;

	for (i=0; i<CONFIG_SYS_MAX_I2C_BUS; ++i)
		dw_i2c_init(i2c_bus_address[i]);
}

int i2c_probe(unsigned char chip) {
	unsigned char byte;
	/* read a byte and see if the addr acked or nacked */
	return i2c_read(chip, 0, 0, &byte, 1);
}
