#include <common.h>
#include <asm/io.h>
#include <div64.h>
#include "asm/arch/regAddrs.h"
#include <dovefbreg.h>
#include <lcd.h>
#include <lxk_panel.h>
#include <lcd.h>

extern int is_Gr2_RevA(void);
extern bool isFPGA(void);

extern unsigned int board_get_core_freq_hz(void);

//#define VERBOSE_PLL_DEBUG

#define XCLK_MIN_PHASE_DIV 1
#define XCLK_MAX_PHASE_DIV 16
#define XCLK_PREDIV_OPTION 10

#define VCO_MAX_FREQ_MHZ 2500
#define VCO_MIN_FREQ_MHZ 2300

#define FB_DIV_MAX 511
#define FB_DIV_MIN 1

#define PLL_LOCK_TIMEOUT_MS 10000
#define PLL_LOCK_DELAY_MS 10

#define SHOWREG(reg)  printf("%-33s @ 0x%08x = 0x%08x\n",#reg,(unsigned int)(&(reg)),((unsigned int)(reg)))

typedef struct {
    uint32_t ctrl[4];
    uint32_t imp_ctrl[2];
    uint32_t pll_ctrl[5];
    uint32_t status;
    uint32_t imp_status;
    uint32_t pll_status;
    uint32_t bist_status;
    uint32_t reserved;
    uint32_t d_cfg[42];
} lvds_phy_t;

typedef struct {
	int hidiv;
	int lodiv;
	int prediv;
} xclk_div_t;

static volatile lvds_phy_t *lvds = NULL;
static volatile unsigned int *lcd_xclk_config = NULL;
static volatile unsigned int *lcd_xclk_status = NULL;
static volatile unsigned int *apbus_clk_lcd_config = NULL;
static volatile unsigned int *lcd_cfg_sclk_div = NULL;

static int lcd_pll_kvco_select(uint32_t vco_freq_mhz)
{
	if(vco_freq_mhz >= 3600)
		return 7;
	else if(vco_freq_mhz >= 3380)
		return 6;
	else if(vco_freq_mhz >= 3150)
		return 5;
	else if(vco_freq_mhz >= 2920)
		return 4;
	else if(vco_freq_mhz >= 2690)
		return 3;
	else if(vco_freq_mhz >= 2440)
		return 2;
	else if(vco_freq_mhz >= 2200)
		return 1;
	else
		return 0;
}

static int lcd_pll_intpi_select(uint32_t vco_freq_mhz)
{
	if(vco_freq_mhz >= 3500)
		return 0x0c;
	else if(vco_freq_mhz >= 3000)
		return 0x0a;
	else if(vco_freq_mhz >= 2500)
		return 0x08;
	else
		return 0x06;
}

static int lcd_pll_icp_select(uint32_t input_freq_hz)
{
	if(input_freq_hz >= 67500000u)
		return 4;
	else if(input_freq_hz >= 55000000u)
		return 5;
	else if(input_freq_hz >= 47500000u)
		return 6;
	else if(input_freq_hz >= 40000000u)
		return 7;
	else if(input_freq_hz >= 35000000u)
		return 8;
	else
		return 9;
}

static int lcd_pll_clkout_div_select(int output_div) {
	if(is_Gr2_RevA()) {
	    	switch(output_div) {
	    	case 1:
	    	case 2:
	    	case 3:
	    	case 4:
	    	case 8:
	    		return (output_div - 1);
	    	default:
	    		printf("%s invalid post divider %d. Defaulting to 8\n", __func__, output_div);
	    		return 7;
	    	}
	}
	else {
	    	switch(output_div) {
	    	case 1:
	    	case 2:
	    	case 3:
	    	case 4:
	    		return (output_div - 1);
	    	case 8:
	    		return 4;
	    	case 16:
	    		return 5;
	    	case 32:
	    		return 6;
	    	default:
	    		printf("%s invalid post divider %d. Defaulting to 32\n", __func__, output_div);
	    		return 6;
	    	}
	}
}

#ifdef VERBOSE_PLL_DEBUG
void lcd_lvds_pll_dump(void)
{
	printf("%s:\n", __func__);
	SHOWREG(apbus_clk_lcd_config[0]);       
	SHOWREG(lcd_xclk_config[0]);       
	SHOWREG(lcd_xclk_status[0]);       
	SHOWREG(lcd_cfg_sclk_div[0]);
	SHOWREG(lvds->ctrl[0]);       
	SHOWREG(lvds->ctrl[1]);       
	SHOWREG(lvds->ctrl[2]);       
	SHOWREG(lvds->ctrl[3]);       
	SHOWREG(lvds->imp_ctrl[0]);       
	SHOWREG(lvds->imp_ctrl[1]);       
	SHOWREG(lvds->pll_ctrl[0]);       
	SHOWREG(lvds->pll_ctrl[1]);       
	SHOWREG(lvds->pll_ctrl[2]);       
	SHOWREG(lvds->pll_ctrl[3]);       
	SHOWREG(lvds->pll_ctrl[4]);       
	SHOWREG(lvds->status);       
	SHOWREG(lvds->imp_status);       
	SHOWREG(lvds->pll_status);       
	SHOWREG(lvds->bist_status);

	printf("lvds mux regs:\n");
	print_buffer((ulong)lvds->d_cfg, (void *)lvds->d_cfg, sizeof(lvds->d_cfg[0]), ARRAY_SIZE(lvds->d_cfg), 7);
}
#endif

static void lcd_pointer_init(void)
{
	if(lvds)
		return;

	lcd_cfg_sclk_div = (volatile unsigned int *)(AP_LCD2_BASE + LCD_CFG_SCLK_DIV);
        lvds = (volatile lvds_phy_t *)(AP_LVDS_BASE + 0x300);	

	if(is_Gr2_RevA()) {
		lcd_xclk_config = (volatile unsigned int *)(AP_APMU_BASE + 0x02e8);
		lcd_xclk_status = (volatile unsigned int *)(AP_APMU_BASE + 0x02f0);
		apbus_clk_lcd_config = (volatile unsigned int *)(AP_APMU_BASE + 0x0250);
    	}
	else {
		lcd_xclk_config = (volatile unsigned int *)(AP_APMU_BASE + 0x0518);
		lcd_xclk_status = (volatile unsigned int *)(AP_APMU_BASE + 0x0520);
		apbus_clk_lcd_config = (volatile unsigned int *)(AP_APMU_BASE + 0x0488);
	}
}

static unsigned int lcd_xclk_select(xclk_div_t *xclk, unsigned int xclk_divider)
{
	if(xclk_divider > (XCLK_MAX_PHASE_DIV * 2))
		xclk_divider = XCLK_MAX_PHASE_DIV * 2;

	if(xclk_divider < (XCLK_MIN_PHASE_DIV * 2)) {
		xclk_divider = 1;
		xclk->prediv = 1;
		xclk->hidiv = xclk->lodiv = 0;

		return xclk_divider;
	}
	
	xclk->hidiv = xclk->lodiv = xclk_divider / 2;
	if((xclk_divider & 1) && (xclk->hidiv < XCLK_MAX_PHASE_DIV)){
		/* adjust for odd divisors, won't be symmetric */
		xclk->hidiv++;
	}

	xclk->prediv = 1;

	xclk_divider = (xclk->hidiv + xclk->lodiv) * xclk->prediv;

	return xclk_divider;
}

static void lcd_xclk_init(xclk_div_t *xclk)
{
	uint32_t val;

#ifdef VERBOSE_PLL_DEBUG
	printf("%s: hidiv=%d lodiv=%d\n", __func__, xclk->hidiv, xclk->lodiv);
#endif

	/* enable clocks */
	*apbus_clk_lcd_config |= 2;

	//*lcd_xclk_config = (xclk->hidiv << 16) | (xclk->lodiv << 8) | (1 << 1);
	val = (xclk->hidiv << 16) | (xclk->lodiv << 8);
	if(xclk->prediv == 10) {
		val |= 1 << 2;
	}
	*lcd_xclk_config = val;
	*lcd_xclk_status = (1 << 1);
}

static void lcd_pll_dump_op_point(uint32_t core_freq_hz, int xclk_div, int refdiv, int fb_div, int output_div)
{
	uint32_t input_freq_hz;
	uint32_t output_freq_hz;
	uint32_t dotclk_freq_hz;
	uint32_t vco_freq_mhz;

	input_freq_hz = core_freq_hz / (refdiv * xclk_div);
	vco_freq_mhz = ((input_freq_hz / 1000) * fb_div * 4) / 1000;
	output_freq_hz = ((vco_freq_mhz * 1000) / output_div) * 1000;
	dotclk_freq_hz = output_freq_hz / 7;

#ifdef VERBOSE_PLL_DEBUG
	printf("core_freq_hz=%d refdiv=%d xclk_div=%d\n"
		"input_freq_hz=%d\n"
		"fb_div=%d, x4=%d\n"
		"vco_freq_mhz=%d\n"
		"output_div=%d\n"
		"output_freq_hz=%d\n"
		"dotclk_freq_hz=%d\n",
		core_freq_hz, refdiv, xclk_div, input_freq_hz,
		fb_div, (fb_div * 4), vco_freq_mhz, output_div,
		output_freq_hz, dotclk_freq_hz);
#else
	printf("output_freq_hz=%d dotclk_freq_hz=%d\n",
		output_freq_hz, dotclk_freq_hz);
#endif
}

static int initialize_pll(uint32_t core_freq_hz, int xclk_div, int refdiv, int fb_div, int output_div)
{

    uint32_t val;
    int timeout_ms;
    int locked = 0;
    uint32_t input_freq_hz;
    uint32_t vco_freq_mhz;
    int clkout_div_sel;
    int kvco;
    int intpi;
    int intp_cap;
    int icp;
    int reserve_ir;

    if (isFPGA())   // No PLLs in the FPGAs
        return 0;

    input_freq_hz = core_freq_hz / (refdiv * xclk_div);
    vco_freq_mhz = ((input_freq_hz / 1000) * fb_div * 4) / 1000;

    kvco = 0x0f & lcd_pll_kvco_select(vco_freq_mhz);
    intpi = 0x0f & lcd_pll_intpi_select(vco_freq_mhz);
    intp_cap = 4;
    icp = 0x0f & lcd_pll_icp_select(input_freq_hz);
    clkout_div_sel = 0x07 & lcd_pll_clkout_div_select(output_div);
    reserve_ir = (1 << 4); //POST_SEL
    reserve_ir |= ((clkout_div_sel >> 2) & 1) << 5; //clkout_div_sel[2]
    
#ifdef VERBOSE_PLL_DEBUG
    printf("kvco=%d intpi=%d intp_cap=%d icp=%d\n", kvco, intpi, intp_cap, icp);
#endif
    // reset lvds_6tx_pll and configure the phy
    lvds->ctrl[0] |= (1 << 23); // PU
    lvds->ctrl[0] &= ~(3 << 13);  // D_NBITS configure for 7 bit lvds
    lvds->ctrl[2] |= (0x3f << 26); // enable TX pad drivers
    lvds->imp_ctrl[1] |= 1; // lvds_6_tx samples D  on posedge D_CK
    lvds->ctrl[3] |= (1 << 31); // TX_SR

    //configure pll
    val = lvds->pll_ctrl[2];
    val |= (1 << 28); //RESET_PLL
    val |= (1 << 27); //RESET_PI
    val |= (1 << 29); //RESET_SCC
    val &= ~(1 << 30); //SSC_EN
    val &= ~(1 << 0); //PI_EN
    lvds->pll_ctrl[2] = val;

    val = lvds->imp_ctrl[0];
    val &= ~(0xff << 11);
    val |= reserve_ir << 11;
    lvds->imp_ctrl[0] = val;

    val = lvds->pll_ctrl[0];
    val &= ~((0x1ff << 10) | (3 << 1));
    val |= ((fb_div & 0x1ff) << 10);
    val |= (clkout_div_sel & 3) << 1; //clkout_div_sel[1:0]
    lvds->pll_ctrl[0] = val;

    val = lvds->pll_ctrl[2];
    val &= ~(0x1ff << 2);
    val |= (refdiv << 2);
    lvds->pll_ctrl[2] = val;

    val = lvds->pll_ctrl[1];
    val &= ~((0x0f << 17) | (7 << 21) | (0x0f << 24) | (0x0f << 28)); // ICP, INTP_CAP, INTPI, KVCO
    val |= kvco << 28;
    val |= intpi << 24;
    val |= intp_cap << 21;
    val |= icp << 17;
    lvds->pll_ctrl[1] = val;

    // bring pll out of reset
    lvds->ctrl[0] |= (1 << 23); // PU
    lvds->pll_ctrl[2] &= ~((1 << 28)); //RESET_PLL

    *lcd_cfg_sclk_div = 0x80000001;

    timeout_ms = PLL_LOCK_TIMEOUT_MS;
    while(timeout_ms > 0) {
        mdelay(PLL_LOCK_DELAY_MS);
        timeout_ms -= PLL_LOCK_DELAY_MS;
        if(lvds->pll_status & (1 << 5)) {
            locked = 1;
            break;
        }
    }

    if(locked) {
        return 0;
    }
    else {
        printf("Timeout waiting for LCD LVDS PLL to lock!\n");
        return -1;
    }
}

static int lcd_output_div_select(int requested_output_div)
{
	if(is_Gr2_RevA() && (requested_output_div > 8)) {
		printf("%s: Requested frequency is out of supported range.\n", __func__);
		return 8;
	}
	else if(requested_output_div > 32) {
		printf("%s: Requested frequency is out of supported range.\n", __func__);
		return 32;
	}
	else if(requested_output_div > 16) {
		return 32;
	}
	else if(requested_output_div > 8) {
		return 16;
	}
	else if(requested_output_div > 4) {
		return 8;
	}
	else if(requested_output_div > 3) {
		return 4;
	}
	else if(requested_output_div > 2) {
		return 3;
	}
	else if(requested_output_div > 1) {
		return 2;
	}
	else {
		return 1;
	}
}

static int setup_pll(uint32_t pixelclk_hz)
{
	xclk_div_t xclk;
	int xclk_divider;
	int xclk_div_temp;
	int output_div;
	int fb_div;
	int ref_div;
	uint32_t core_freq_hz = board_get_core_freq_hz();
	uint32_t input_freq_hz = 38000000; // start approximation at optimum input frequency for pll
	uint32_t target_vco_freq_mhz;
	uint32_t vco_freq_mhz;
	uint32_t target_output_freq_hz;

	core_freq_hz = board_get_core_freq_hz();
	target_output_freq_hz = pixelclk_hz * 7; // LVDs requires 7x clock
	
	output_div = lcd_output_div_select((VCO_MIN_FREQ_MHZ * 1000) / (target_output_freq_hz / 1000));
	target_vco_freq_mhz = ((target_output_freq_hz / 1000) * output_div) / 1000;
	if(target_vco_freq_mhz > VCO_MAX_FREQ_MHZ) {
		printf("%s: Adjusting frequency to fit vco bandwidth.\n", __func__);
		target_vco_freq_mhz = VCO_MAX_FREQ_MHZ;
	}
	else if(target_vco_freq_mhz < VCO_MIN_FREQ_MHZ) {
		printf("%s: Adjusting frequency to fit vco bandwidth.\n", __func__);
		target_vco_freq_mhz = VCO_MIN_FREQ_MHZ;
	}

	// first approximation
	fb_div = (target_vco_freq_mhz * 1000) / ((input_freq_hz / 1000) * 4);
	if(fb_div > FB_DIV_MAX)
		fb_div = FB_DIV_MAX;
	else if(fb_div < FB_DIV_MIN)
		fb_div = FB_DIV_MIN;
	
	input_freq_hz = ((target_vco_freq_mhz * 1000) / (fb_div * 4)) * 1000;
	xclk_divider = core_freq_hz / input_freq_hz;
	for (ref_div = 1; ref_div <= 511; ref_div++) {
		xclk_div_temp = xclk_divider / ref_div;
		if(xclk_div_temp <= (XCLK_MAX_PHASE_DIV * 2))
			break;
	}
	xclk_divider = lcd_xclk_select(&xclk, xclk_div_temp);

#ifdef VERBOSE_PLL_DEBUG
	printf("%s: First iteration operating point:\n", __func__);
	lcd_pll_dump_op_point(core_freq_hz, xclk_divider, ref_div, fb_div, output_div);   
#endif

	// Recalculate fb_div to reduce error if possible.
	input_freq_hz = core_freq_hz / (xclk_divider * ref_div);
	fb_div = (target_vco_freq_mhz * 1000) / ((input_freq_hz / 1000) * 4);

	// Final sanity check and adjustment to legal bounds.
	vco_freq_mhz = ((input_freq_hz / 1000) * fb_div * 4) / 1000;
	if(vco_freq_mhz < VCO_MIN_FREQ_MHZ)
		fb_div++;
	if(fb_div > FB_DIV_MAX)
		fb_div = FB_DIV_MAX;
	else if(fb_div < FB_DIV_MIN)
		fb_div = FB_DIV_MIN;
	
	printf("%s: Second iteration operating point:\n", __func__);
	lcd_pll_dump_op_point(core_freq_hz, xclk_divider, ref_div, fb_div, output_div);  

	lcd_xclk_init(&xclk);

	initialize_pll(core_freq_hz, xclk_divider, ref_div, fb_div, output_div);

	return 0;
}

/*
 * D_CFG Reg Index  Channel   Bit    TI'94 Bit  TI'822 Bit   Usage-Opt2   LDD/mux Bit-Opt2   Usage-Opt1   LDD/mux Bit-Opt1
 *       0            0        0         D7         D6        G2            10                G0            8
 *       1            0        1         D6         D5        R7            7                 R5            5
 *       2            0        2         D4         D4        R6            6                 R4            4
 *       3            0        3         D3         D3        R5            5                 R3            3
 *       4            0        4         D2         D2        R4            4                 R2            2
 *       5            0        5         D1         D1        R3            3                 R1            1
 *       6            0        6         D0         D0        R2            2                 R0            0
 *       7            1        0        D18         D13       B3            19                B1            17
 *       8            1        1        D15         D12       B2            18                B0            16
 *       9            1        2        D14         D11       G7            15                G5            13
 *      10            1        3        D13         D10       G6            14                G4            12
 *      11            1        4        D12         D9        G5            13                G3            11
 *      12            1        5         D9         D8        G4            12                G2            10
 *      13            1        6         D8         D7        G3            11                G1            9
 *      14            2        0        D26         D20      DENA           26               DENA           26
 *      15            2        1        D25         D19      VSYNC          25               VSYNC          25
 *      16            2        2        D24         D18      HSYNC          24               HSYNC          24
 *      17            2        3        D22         D17       B7            23                B5            21
 *      18            2        4        D21         D16       B6            22                B4            20
 *      19            2        5        D20         D15       B5            21                B3            19
 *      20            2        6        D19         D14       B4            20                B2            18
 *      21            3        0        D23         RSV      rsvd           28               rsvd           28
 *      22            3        1        D17         D26       B1            17                B7            23
 *      23            3        2        D16         D25       B0            16                B6            22
 *      24            3        3        D11         D24       G1            9                 G7            15
 *      25            3        4        D10         D23       G0            8                 G6            14
 *      26            3        5         D5         D22       R1            1                 R7            7
 *      27            3        6        D27         D21       R0            0                 R6            6
 *      28            4        0                           CLK high         27             CLK high         27 
 *      29            4        1                           CLK high         27             CLK high         27 
 *      30            4        2                            CLK low         28              CLK low         28
 *      31            4        3                            CLK low         28              CLK low         28
 *      32            4        4                            CLK low         28              CLK low         28
 *      33            4        5                           CLK high         27             CLK high         27
 *      34            4        6                           CLK high         27             CLK high         27
 *
 * Note: 18bpp, aka Format-3 or Option-3, is identical to Option-2 except LVDS pair 3 is unused.
 */
static void setup_lvds_mux(lvds_bit_mapping_t lvds_option) {
    const char mux[3][35] = {
                         { 10,  7,  6,  5,  4,  3,  2,
                           19, 18, 15, 14, 13, 12, 11,
                           26, 25, 24, 23, 22, 21, 20,
                           28, 28, 28, 28, 28, 28, 28,
                           27, 27, 28, 28, 28, 27, 27 }, /* BPP18 */

                         {  8,  5,  4,  3,  2,  1, 0,
                           17, 16, 13, 12, 11, 10, 9,
                           26, 25, 24, 21, 20, 19, 18,
                           28, 23, 22, 15, 14,  7,  6,
                           27, 27, 28, 28, 28, 27, 27 }, /* Option 1 */

                         { 10,  7,  6,  5,  4,  3,  2,
                           19, 18, 15, 14, 13, 12, 11,
                           26, 25, 24, 23, 22, 21, 20,
                           28, 17, 16,  9,  8,  1,  0,
                           27, 27, 28, 28, 28, 27, 27 }, /* Option 2 */
                      };
    int i, j;

#ifdef VERBOSE_PLL_DEBUG
    printf("%s:\n", __func__);
#endif

    switch(lvds_option) {
    case LVDS_BPP18:
        i = 0;
        break;
    case LVDS_BPP24_OPTION_1:
        i = 1;
        break;
    case LVDS_BPP24_OPTION_2:
        i = 2;
        break;
    default:
        printf("%s: Unrecognized LVDS option.\n", __func__);
        return;
    }

    for (j = 0; j < ARRAY_SIZE(mux[i]); j++) {
        lvds->d_cfg[j] = mux[i][j] & 0x01f;
    }
}

int lcd_lvds_init(void *lcd_reg_base, uint32_t pixelclk_hz, lvds_bit_mapping_t lvds_option)
{
	xclk_div_t xclk;
	int xclk_divider_tmp = board_get_core_freq_hz() /35000000ul;

#ifdef VERBOSE_PLL_DEBUG
	printf("%s:\n", __func__);
#endif

	lcd_pointer_init();

	// preinit xclk to a dummy frequency so we can init the lvds mux
	lcd_xclk_select(&xclk, xclk_divider_tmp);
	lcd_xclk_init(&xclk);

	setup_lvds_mux(lvds_option);

	return 0;
}

int lcd_pll_init(void *lcd_reg_base, uint32_t pixelclk_hz)
{
	lcd_pointer_init();

	/* choose and initialize one of the hardcoded configurations */
	setup_pll(pixelclk_hz);

#ifdef VERBOSE_PLL_DEBUG
	lcd_lvds_pll_dump();
#endif

	return 0;
}

