#include <common.h>
#include <asm/arch/board_priv.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/apb_config_regstructs.h>
#include <asm/arch/efuse_regstructs.h>
#include <asm/sizes.h>
#include <pci_ids.h>

/* U-boot does not have any PCIE register structures so just defining what we need */
#define PCIE_EP_REG_SIZE            SZ_128M
#define PCIE_APPERATURE_SIZE        SZ_32M
#define PCIE_INBOUND_APPERATURES    3
#define PCIE_OUTBOUND_APPERATURES   8
#define PCIE_LCL_CFG_OFFSET         (0x00008000)
#define PCIE_APP_REGS_OFFSET        (0x00010000)

/* pcie configuration registers */
#define PCIE_CLASSREVID_REG         (0x0008 / sizeof (unsigned int))
#define PCIE_LMSICTRL_REG           (0x0050 / sizeof (unsigned int))
#define PCIE_LCFGBAR0MASK_REG       (0x1010 / sizeof (unsigned int))
#define PCIE_LCFGBAR1MASK_REG       (0x1014 / sizeof (unsigned int))

/* pcie application registers */
#define PCIE_ACTRL_REG              (0x0000 / sizeof (unsigned int))
    #define PCIE_APP_CTRL_REG_LTSSM_EN  (1 << 15) /* 0x8000 */
#define PCIE_INADDRXLT_REG0         (0x0014 / sizeof (unsigned int))
#define PCIE_BAR0_REG               (0x0020 / sizeof (unsigned int))
#define PCIE_PHY_REGACCESS_REG      (0x1000 / sizeof (unsigned int))
#define PCIE_PHY_READWRITE_REG      (0x1004 / sizeof (unsigned int))
    #define PCIE_PHY_RW_WDATA(x)        ((x & 0xffff) << 0)
    #define PCIE_PHY_RW_ADDR(x)         ((x & 0xff) << 16)
    #define PCIE_PHY_RW_WRITE           (1 << 24)
    #define PCIE_PHY_RW_READ            (0 << 24)
#define PCIE_PHY_READWRITEDONE_REG  (0x1008 / sizeof (unsigned int))
    #define PCIE_PHY_RW_DONE            (1 << 16)
    #define PCIE_PHY_RW_RDATA(x)        ((x & 0xffff) << 0)
#define PCIE_PHY_READWRITESTART_REG (0x100C / sizeof (unsigned int))
    #define PCIE_PHY_RW_START           (1 << 0)
#define PCIE_OUTADDRXLT_REG0        (0x1010 / sizeof (unsigned int))

int pcie_is_endpoint(void)
{
    struct EFUSE_REGS_s *efuse = (struct EFUSE_REGS_s *)APB_EFUSE_BASE;
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;
    int rc=0;

    /* check to see if VCF is enabled */
    if ((efuse->EfuseStatus63to32_Bank1 & 0xf) == 0x5)
    {
       printf("PCIe: using eFuse for controller type\n");
       /* check efuse info for RC/EP */
       if ((efuse->EfuseStatus63to32_Bank2 & (1<<24)) == 0) rc=1;
    }
    else
    {
       printf("PCIe: using bootstrap config for controller type\n");
       /* check config bootstrap for RC/EP */
       if ((apb_config->ASR & (1<<1)) == 0) rc=1;
    }

    return(rc);
}

void pcie_phy_soft_reset(int reset)
{
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;
    u32 fcrr;

    /* Read the asic ver reg */
    fcrr = apb_config->FCRR;

    if (reset)
        fcrr |= (1<<1);
    else
        fcrr &= ~(1<<1);

    apb_config->FCRR = fcrr;
}

int pcie_phy_write(unsigned int base, u16 addr, u16 data)
{
   volatile unsigned int *appRegs = (unsigned int *) (base + PCIE_APP_REGS_OFFSET);
   int done=0, i=0;
   int rc=0;
   int temp;

   temp = PCIE_PHY_RW_ADDR(addr) |
          PCIE_PHY_RW_WDATA(data) |
          PCIE_PHY_RW_WRITE;

   appRegs[PCIE_PHY_READWRITE_REG] = temp;

   appRegs[PCIE_PHY_READWRITESTART_REG] = PCIE_PHY_RW_START;

   done = appRegs[PCIE_PHY_READWRITEDONE_REG];
   while (((done & PCIE_PHY_RW_DONE) == 0) && (i < 5))
   {
       i++;
       mdelay(10);
       done = appRegs[PCIE_PHY_READWRITEDONE_REG];
   }

   if ((done & PCIE_PHY_RW_DONE) == 0)
   {
        printf("%s: PCIe PHY reg write failed to complete at 0x%x\n", __func__, addr);
        rc = -1;
   }

   return rc;
}

int pcie_phy_read(unsigned int base, u16 addr, u16 *data)
{
   volatile unsigned int *appRegs = (unsigned int *) (base + PCIE_APP_REGS_OFFSET);
   int done=0, i=0;
   int rc=0;

   appRegs[PCIE_PHY_READWRITE_REG] = (PCIE_PHY_RW_ADDR(addr) | PCIE_PHY_RW_READ);

   appRegs[PCIE_PHY_READWRITESTART_REG] = PCIE_PHY_RW_START;

   done = appRegs[PCIE_PHY_READWRITEDONE_REG];
   while (((done & PCIE_PHY_RW_DONE) == 0) && (i < 5))
   {
       i++;
       mdelay(10);
       done = appRegs[PCIE_PHY_READWRITEDONE_REG];
   }

   *data = PCIE_PHY_RW_RDATA(appRegs[PCIE_PHY_READWRITEDONE_REG]);

   if ((done & PCIE_PHY_RW_DONE) == 0)
   {
        printf("%s: PCIe PHY reg read failed to complete at 0x%x\n", __func__, addr);
        rc = -1;
   }

   return rc;
}

int pcie_phy_write_verify(unsigned int base, u8 addr, u16 wdata)
{
   int rc=0;
   u16 rdata;

   rc = pcie_phy_write(base, addr, wdata);
   if (rc < 0) goto fail;
   rc = pcie_phy_read (base, addr, &rdata);
   if (rc < 0) goto fail;

   if (wdata != rdata)
   {
       printf("%s: PCIe PHY reg write fail at 0x%x: Exp:0x%x Rec:0x%x\n", __func__, addr, wdata, rdata);
       rc = -1;
   }
fail:
   return (rc);
}

int pcie_ref_clock_check(unsigned int base)
{
   u16 data=0;
   int rc=0;

   rc = pcie_phy_read(base, 0x82, &data);
   if (rc < 0) goto fail;
   if ((data & (0x6)) != 0x6)
       rc = -1;
fail:
   return(rc);
}

int pcie_pll_lock_check(unsigned int base)
{
   u16 data=0;
   int rc=0;
   int i=0;

   rc = pcie_phy_read(base, 0x01, &data);
   if (rc < 0) goto fail;
   while (((data & (0x100)) != 0x100) && (i < 5))
   {
       i++;
       mdelay(10);
       rc = pcie_phy_read(base, 0x01, &data);
       if (rc < 0) goto fail;
   }
   if ((data & (0x100)) != 0x100)
       rc = -1;
fail:
   return(rc);
}

int pcie_pclk_check(unsigned int base)
{
   u16 data=0;
   int rc=0;

   rc = pcie_phy_read(base, 0x83, &data);
   if (rc < 0) goto fail;
   if ((data & (0x1)) != 0x1)
       rc = -1;
fail:
   return(rc);
}

int pcie_cal_done_check(unsigned int base)
{
   u16 data=0;
   int rc=0;
   int i=0;

   rc = pcie_phy_read(base, 0x02, &data);
   if (rc < 0) goto fail;
   while (((data & (0x4000)) != 0x4000) && (i < 5))
   {
       i++;
       mdelay(10);
       rc = pcie_phy_read(base, 0x02, &data);
       if (rc < 0) goto fail;
   }
   if ((data & (0x4000)) != 0x4000)
       rc = -1;
fail:
   return(rc);
}

int pcie_rx_term_enb(unsigned int base)
{
   u16 data=0;
   int rc=0;

   rc = pcie_phy_read(base, 0x48, &data);
   if (rc < 0) goto fail;
   data &= ~(1<<2);
   rc = pcie_phy_write(base, 0x48, data);
fail:
   return(rc);
}

int pcie_init_phy(unsigned int base)
{
   int rc=0;
   volatile unsigned int *appRegs = (unsigned int *) (base + PCIE_APP_REGS_OFFSET);

   {
       /* de-assert soft reset to PHY */
       pcie_phy_soft_reset(0);

       /* set ClkDiv to 3 per Marvell */
       appRegs[PCIE_PHY_REGACCESS_REG] = 3;

       /* 8-bit, 250MHz PIPE */
       rc = pcie_phy_write_verify(base, 0xc1, 0x0025);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0xc8, 0x0005);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0x80, 0x0101);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0x81, 0x3001);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0xc5, 0x011f);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0x80, 0x1000);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0x81, 0x0011);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0x01, 0xfc60);
       if (rc < 0) goto fail;
       rc = pcie_phy_write(base, 0x02, 0x0240);
       if (rc < 0) goto fail;
       rc = pcie_phy_write_verify(base, 0xc1, 0x0024);
       if (rc < 0) goto fail;

       /* wait 10ms for PHY clock to lock */
       mdelay(10);

       /* check for ref clock */
       rc = pcie_ref_clock_check(base);
       if (rc < 0)
       {
           printf("PCIe Ref Clock not detected\n");
           goto fail;
       }

       /* check for PLL lock */
       rc = pcie_pll_lock_check(base);
       if (rc < 0)
       {
           printf("PCIe PLL not locked\n");
           goto fail;
       }

       /* check for pclk */
       rc = pcie_pclk_check(base);
       if (rc < 0)
       {
           printf("PCIe PCLK not detected\n");
           goto fail;
       }

       /* check for calibration sequence complete */
       rc = pcie_cal_done_check(base);
       if (rc < 0)
       {
           printf("PCIe Calibration Not Complete\n");
           goto fail;
       }
       /* enable Rx termination */
       rc = pcie_rx_term_enb(base);
   }
fail:
   return(rc);
}

void pcie_ep_preinit(unsigned int base)
{
   u32 i, class;
   volatile unsigned int *appRegs = (unsigned int *) (base + PCIE_APP_REGS_OFFSET);
   volatile unsigned int *cfgRegs = (unsigned int *) (base + PCIE_LCL_CFG_OFFSET);

   /*
    * enable the LTSSM in the core to start Link training sequences
    */
   appRegs[PCIE_ACTRL_REG] = PCIE_APP_CTRL_REG_LTSSM_EN;

   /*
    * fix up the class register
    */
   class = cfgRegs[PCIE_CLASSREVID_REG];
   class |= (PCI_CLASS_SYSTEM_OTHER << 16);
   cfgRegs[PCIE_CLASSREVID_REG] = class;

   /*
    * fix up BAR0 mask register (it's write only) to set size
    */
   cfgRegs[PCIE_LCFGBAR0MASK_REG] = PCIE_EP_REG_SIZE - 1;
   cfgRegs[PCIE_LCFGBAR1MASK_REG] = 0xffffffff;
   for (i=0; i<PCIE_INBOUND_APPERATURES; i++)
      appRegs[PCIE_INADDRXLT_REG0 + i] = (APB_BASE + (i * PCIE_APPERATURE_SIZE));
   /* TODO: this needs to be addressed. change some of the hard-coded values for BAR0Mask and InAddrXlt.
             Device driver needs to set BAR0 in app reg space.  This is fixed in RevB */
   appRegs[PCIE_BAR0_REG] = 0xe0000000;

   /* Set bits 19:17 to 111 to enable up to 128 MSI interrupts */
   cfgRegs[PCIE_LMSICTRL_REG] = (0x000e0000 | cfgRegs[PCIE_LMSICTRL_REG]);

   /* setup the outbound address translate regs for APB accesses */
   for (i=0; i<PCIE_OUTBOUND_APPERATURES; i++)
      appRegs[PCIE_OUTADDRXLT_REG0 + i] = (APB_BASE + (i * PCIE_APPERATURE_SIZE));
}

int pcie_init ( void )
{
   unsigned int pcie_base = PCIE_BASE;
   int endpoint=0;

   endpoint = pcie_is_endpoint();
   if (endpoint)
   {
      if (pcie_init_phy(pcie_base) < 0)
      {
         printf("PCIe: failed to init phy\n");
         return(0);
      }
      pcie_ep_preinit(pcie_base);
   }

   return 1;
}
