/*----------------------------------------------------------------------------
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License Version 2.
 *---------------------------------------------------------------------------*/
#ifndef LINUX_FLASH_H
#define LINUX_FLASH_H 1

#define MAX_BAD_BLOCKS (20)


#define MAX_BBT_ENTRIES      (0x100)      /* Maximum bad block entries    - 256 */
#define MAX_BBT_SIZE         (MAX_BBT_ENTRIES * sizeof(BadBlockTable_t))

/*
 * Total number of blocks contained in the parts
 */
/* #define MAX_BLOCKS (1024) */

#define FLASH_PAGE_SIZE             512

#define MBR_MAGIC (0x000055aa)
#define OFFSET_MBR (32 * 1024)      // Needs to be removed and replaced with
                                    // MBR_BLOCK_OFFSET(blocksize).

#define MBR_BLOCK_INDEX				(2)	// Master Boot Record Block Index.
#define MBR_BLOCK_OFFSET(blocksize)	(blocksize * MBR_BLOCK_INDEX)

/*
 * Is the flash image shuffled down?
 */
#define SHUFFLE_MAGIC 0x75d1e43f

/*
 * Format of the partition table entries.
 */
typedef struct {
   unsigned int
      Start,
      Length;
} PartitionTable_t;

/*
 * These reflect what the actual flash parts are implementing.
 */

/*
 * NOTE: In the past, max partitions was used by both the kernel and
 * the applications to indicate how many partitions are on the device.
 * Now, this value is only really supposed to be used in the kernel to
 * set maximum sizes for staticly allocated data structures.  The number
 * of actual partitions is variable and read from the device by the kernel
 * when the device is initialized.  Applications that need to know should
 * ask the kernel through a service call.
 *
 * NOTE: Although we could make the kernel structures that depend on this
 * dynamic too, the amount of memory wasted having static maximal
 * allocations is probably smaller than the amount of buggy code that
 * would be needed to maintain the dynamic structures in the first place.
 */

#define MAX_POSSIBLE_PARTITIONS (61)
#define MAX_PARTITIONS (60)  /* This is capped at 60 */
                             /* This is how many fit in the MBR size.      */
                             /* NOTE: If you change this, check that the   */
                             /* MBR structure _Reserved_ size is OK below. */

#define PARTITIONING_DEFAULT_8MEG  (16)	/* Defaults for how many partitions */
#define PARTITIONING_DEFAULT_16MEG (32) /* to use on various devices.       */

#if (PARTITIONING_DEFAULT_8MEG > MAX_PARTITIONS || PARTITIONING_DEFAULT_16MEG > MAX_PARTITIONS)
#error MAX_PARTITIONS is not big enough!
#endif

#define MAX_FLASH_SIZE              (256 * 1024 * 1024)
#define NUMBER_DEVICES    3  /* number of logical 'disks' implemented in flash */

#define TARGET_NAME_LENGTH 16

/*
 * This is the format of the Master Boot Record which is found in Block 2, Page 0
 */
typedef struct {
   PartitionTable_t
      PartitionTable[MAX_PARTITIONS];
   int   /* Pushes MAGIC to end of 512-byte page */
      _Reserved_[(FLASH_PAGE_SIZE - 4
                  - (MAX_PARTITIONS*sizeof(PartitionTable_t)))/4];
   int
      Magic;
} masterBootRecord_t, MasterBootRecord_t;

/*----------------------------------------------------------------------------
 * Boot Block Definitions
 *
 * This section defines the format of the data which is found at Block 0/
 * at Page 0
 *---------------------------------------------------------------------------*/
typedef struct {
   unsigned char
      UAA[6],
      _Reserved_[10];
} NetworkBoot_t;

/*
 * Leave these as signed entries so that we can check for -1 and leave loops
 * early.
 */
typedef struct {
   short
      Block,
      Remap;
} BadBlockTable_t;

#define FLASH_FLAG_CURRENT_A    (1<<0)
#define FLASH_FLAG_FLASHABLE_A  (1<<1)
#define FLASH_FLAG_CURRENT_B    (1<<2)
#define FLASH_FLAG_FLASHABLE_B  (1<<3)

#define BOOT_BLOCK_MAGIC (0x5fa97381)
#define BOOT_BLOCK_DATA_MAGIC (0x8a427341)
/*
 * This structure defines what is found at the very base of the flash.  Its
 * length is 256 bytes
 * NOTE:
 * NOTE: If you add ints or shorts to this structure, be sure to add
 * NOTE: swapping of the new members over in lxk/flash-low.c.
 * NOTE:
 */
typedef struct {
   int
      Opcodes[16];
   unsigned int
      Magic;
   int
      DeviceId,
      BlockSize,
      DeviceSize;

   NetworkBoot_t
      NetworkBoot;
   int
      ProjectId,
      numberPartitions,
      numberBadBlocks,
      offsetBadBlockTable,
      Version;
   char
      targetName[TARGET_NAME_LENGTH];
   unsigned int
      PageSize;
   int FlashFlags;
   int
      _Reserved2_[9];
   BadBlockTable_t
      BBT[MAX_BAD_BLOCKS];

} bootBlock_t, BootBlock_t;

typedef enum {
   DEVICETYPE_ROM,
   DEVICETYPE_FLASH
} DeviceType_e;

#endif
