#include <common.h>
#include <asm/arch/gpio.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/apb_config_regmasks.h>
#include <asm/arch/apb_config_regstructs.h>
#include <asm/arch/efuse_regstructs.h>
#include <asm/arch/efuse_regmasks.h>
#include <asm/arch/LSCAN_regmasks.h>
#include <asm/arch/LSCAN_regstructs.h>

#include "card.h"
#include "ddr.h"
#include "regulator.h"

static const uint8_t card05GpioSetup[] = {
    S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), // Block A  0 - 15
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 5), S(0, 0, 5), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0),
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), // Block A 16 - 31
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0),

    S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), // Block B  0 - 15
    S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), // Block B 16 - 31
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 9), S(0, 0, 1),

    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 2), S(0, 0, 2), // Block C  0 - 15
    S(0, 0, 0), S(0, 0, 0), S(0, 0, 9), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), // Block C 16 - 31
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),

    S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 1), S(0, 0, 1), // Block D  0 - 15
    S(0, 0, 5), S(0, 0, 5), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),
    S(0, 0, 4), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(1, 0, 0), S(0, 0, 0), S(0, 0, 0), // Block D 16 - 31
    S(0, 0, 1), S(0, 0, 1), S(1, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0),

    S(0, 0, 1), S(0, 0, 0), S(0, 0, 9), S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 0), S(0, 0, 1), // Block E  0 - 15
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 1, 1), // Block E 16 - 31
    S(0, 0, 1), S(0, 0, 0), S(0, 1, 1), S(0, 0, 0), S(0, 1, 0), S(0, 0, 1), S(0, 0, 0), S(0, 0, 0),

    S(0, 0, 0), S(0, 0, 0), S(0, 1, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), // Block F  0 - 15
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 1, 1),
    S(0, 0, 9), S(0, 0, 1), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 2), S(0, 0, 9), S(0, 0, 2), // Block F 16 - 31
    S(0, 1, 2), S(0, 0, 2), S(0, 0, 3), S(0, 0, 1), S(0, 0, 0), S(0, 0, 0), S(0, 0, 1), S(0, 0, 0),

    S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 0, 0), S(0, 0, 1), S(0, 0, 5), // Block G  0 - 15
    S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5),
    S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 4), S(0, 0, 0), S(0, 0, 5), S(0, 0, 5), S(0, 0, 0), // Block G 16 - 31
    S(0, 0, 9), S(0, 0, 5), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 1, 1), S(0, 0, 6), S(0, 0, 6),

    S(0, 0, 6), S(0, 0, 2), S(0, 1, 2), S(0, 0, 2), S(0, 0, 6), S(0, 0, 2), S(0, 0, 2), S(0, 0, 0), // Block H  0 - 15
    S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 9), S(0, 0, 0), S(0, 0, 9), S(0, 0, 9), S(0, 0, 0),
    S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), // Block H 16 - 31
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0)
};


/* Mapping of signal name to physical gpio number */
static const struct gpio_mapping_t card05GpioMapping[] = {
    { OP_PANEL_IRQ, BLOCKA(0) },
    { BSPI_CLK, BLOCKA(3) },
    { BSPI_CS,  BLOCKA(4) },
    { BSPI_TXD, BLOCKA(5) },
    { BSPI_RXD, BLOCKA(6) },
    { OC_USB_FRNT, BLOCKA(8) },
    { PWR_EN_USB_FRNT, BLOCKA(9) },
    { SDMMC0_CLK,   BLOCKB(0) },
    { SDMMC0_DATA0, BLOCKB(2) },
    { SDMMC0_DATA1, BLOCKB(3) },
    { SDMMC0_DATA2, BLOCKB(4) },
    { SDMMC0_DATA3, BLOCKB(5) },
    { ENET_RESET,   BLOCKB(30) },
    { ENET_25MHZ,   BLOCKB(31) },
    { VIDEO_CLK, BLOCKC(11) },
    { VIDEO_SIG, BLOCKD(23) },
    { AUDIO_ENABLE, BLOCKE(2) },
    { eMMC_ENABLE,  BLOCKF(2) },
    { CARD_REV1, BLOCKG(26) },
    { CARD_REV2, BLOCKG(27) },
    { CARD_REV3, BLOCKG(28) }
};

static int is_ddr_x16(void) {
    struct EFUSE_REGS_s *efuse = (struct EFUSE_REGS_s *)APB_EFUSE_BASE;

    /* Populate the efuse status registers */
    efuse->AUTO_Control = 1;
    while (efuse->TOP_Status & EFUSE_TOP_STATUS_BUSY_MASK);

    return ((efuse->EfuseStatus31to0_Bank1 & EFUSE_DDR3_WIDTH) || (efuse->EfuseStatus95to64_Bank2 & EFUSE_DDR3_WIDTHv2));
}

// LS_Clk_Acc_GalvoP period counted in 240mhz clocks
#define LVDS_12P6 19
#define LVDS_13P3 18
#define LVDS_14P1 17
#define LVDS_15P0 16
#define LVDS_16P0 15
#define LVDS_17P1 14
#define LVDS_18P5 13
#define LVDS_20P0 12

#define LVDS_REF_PERIOD LVDS_13P3

int clk_lvds_ui_init(void) {
    struct MIRROR1_REGS_s *mirror1 = (struct MIRROR1_REGS_s *)DEC_DEC_LSCAN_MIRROR1_BASE;

    // Enable reference clock, only used by FIN224C serializer

    // disable LS_Clk_Acc_GalvoP
    mirror1->mcfg2 = MIRROR1_MCFG2_GALVOR_IDLE_REPLACE_VAL(mirror1->mcfg2, 0);
    mirror1->mcfg2 = MIRROR1_MCFG2_GALVOP_DIS_REPLACE_VAL(mirror1->mcfg2, 1);

    // set frequency
    mirror1->mcfg1 = MIRROR1_MCFG1_MODE_REPLACE_VAL(mirror1->mcfg1, 0x1);
    mirror1->mcfg1 = MIRROR1_MCFG1_CONTROL_REPLACE_VAL(mirror1->mcfg1, (LVDS_REF_PERIOD - 2));

    // set duty cycle
    mirror1->galvop_rise0 = MIRROR1_GALVOP_RISE0_ENABLE_MASK;
    mirror1->galvop_fall0 = MIRROR1_GALVOP_FALL0_ENABLE_MASK | (LVDS_REF_PERIOD / 2);

    // enable LS_Clk_Acc_GalvoP
    mirror1->mcfg2 = MIRROR0_MCFG2_GALVOP_DIS_REPLACE_VAL(mirror1->mcfg2, 0);

    // Wait for FIN224C serdes PLLs to stabilize.
    udelay(200);

    return 0;
}

int is_phy_gigabit(void) {
    const char *card = card_get_type( );

    return (strcmp(card, "card05") != 0);
}

const char * card_get_type(void) {
    if (is_ddr_x16( ))
        return "card05";
    else
        return "card05a";
}

const char * card_get_revision(void) {
    uint32_t revision = 0;

    gpio_direction(CARD_REV1, 0);
    gpio_direction(CARD_REV2, 0);
    gpio_direction(CARD_REV3, 0);

    revision |= gpio_sample(CARD_REV1) << 2;
    revision |= gpio_sample(CARD_REV2) << 1;
    revision |= gpio_sample(CARD_REV3);

    if (strcmp(card_get_type( ), "card05a") == 0) {
        switch (revision) {
            case 0:  return "00a";  break;
            case 1:  return "00b";  break;
            case 2:  return "00c";  break;
            case 3:  return "00d";  break;
            case 4:  return "00e";  break;
            case 5:
            case 6:  return "00f";  break;
            default:
            case 7:  return "00k";  break;
        }
    }
    else {
        switch (revision) {
            case 1:  return "00a";  break;
            case 2:  return "00b";  break;
            case 3:  return "00c";  break;
            case 4:  return "00d";  break;
            case 5:
            case 6:  return "00e";  break;
            default:
            case 7:  return "00i";  break;
        }
    }
}

int board_supports_lcd_vsync_in(void) {
    const char *revision = card_get_revision( );
    /*
     * This only means the board layout supports vsync_in.
     * There may be display-specific population differences.
     * To support "illegal" combinations of development hardware, the
     * driver for displays using vsync in may need to be verified by
     * setting a timeout.
     */
    if (revision) /* return true for versions 00b and later */
        return (strncmp(revision, "00a", 3));
    else
        return 0;
}

static void gpio_fix_for_overcurrent_usb_host(void) {
    const char *revision = card_get_revision( );
    int old_board = 0;

    /*
     * The old boards need to have GPIOA(9) set output High
     * to fix the over current circuit.
     * The new boards should be fixed and have GPIOA(9)
     * set to function 1.
    */

    if (strcmp(card_get_type( ), "card05a") == 0) {
        if (strncmp(revision, "00e", 3) < 0 ) {
            old_board = 1;
        }
    } else {
        if (strncmp(revision, "00d", 3) < 0 ) {
            old_board = 1;
        }
    }

    if(old_board) {
        gpio_set(OC_USB_FRNT, 0);
        gpio_direction(OC_USB_FRNT, 0);
        gpio_mux(OC_USB_FRNT, 0);

        gpio_set(PWR_EN_USB_FRNT, 1);
        gpio_direction(PWR_EN_USB_FRNT, 1);
        gpio_mux(PWR_EN_USB_FRNT, 0);
    }
}

int card_init(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    gpio_init_pins(card05GpioMapping,
                   ARRAY_SIZE(card05GpioMapping),
                   card05GpioSetup,
                   ARRAY_SIZE(card05GpioSetup));

    gpio_fix_for_overcurrent_usb_host();

    // DCmotor4 encoder inputs are routed to the UPC block
    apb_config->PINCR = APB_CONFIG_PINCR_UPCENCODERSEL_REPLACE_VAL(apb_config->PINCR, 0x4);

    // Set drive characteristics for LVDS inputs
    apb_config->PDCR2 = APB_CONFIG_PDCR2_ILVDS_ODT_REPLACE_VAL(apb_config->PDCR2, 0xf);

    //turn off the 16ma override on lcd outputs
    apb_config->PDCR2 = APB_CONFIG_PDCR2_GPIOC_LEDB_REPLACE_VAL(apb_config->PDCR2, (~0));

    return 0;
}

void card_dram_init(void) {
    int ddr_type = is_ddr_x16( ) ? DDR3_x16 : DDR3_x32;

    initDDR3_nt5cb128m16dp_be_MHz400(ddr_type);
}

#define VOLTAGE_DROPOFF_ADJUSTMENT 10
#define TPS53819_I2C_BUS_NUM       1
#define TPS53819_I2C_ADDR          0x10
/* The core voltage regulator is configured early (before PLL is ramped) */
void card_config_core_voltage(char *asic_rev) {
    config_regulator(asic_rev,
                     VOLTAGE_DROPOFF_ADJUSTMENT,
                     TPS53819_I2C_BUS_NUM,
                     TPS53819_I2C_ADDR);
}

