#include <common.h>
#include <asm/arch/gpio.h>
#include <asm/arch/regAddrs.h>
#include <asm/arch/apb_config_regmasks.h>
#include <asm/arch/apb_config_regstructs.h>
#include <asm/arch/LSCAN_regmasks.h>
#include <asm/arch/LSCAN_regstructs.h>

#include "card.h"
#include "ddr.h"
#include "regulator.h"

static void gpio_fix_for_overcurrent_usb_host(void);

static const uint8_t card01GpioSetup[] = {
    S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 0, 1), // Block A  0 - 15
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 5), S(0, 0, 5), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0),
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), // Block A 16 - 31
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0),

    S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), // Block B  0 - 15
    S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(1, 1, 0), S(1, 1, 0), S(1, 1, 0), S(1, 1, 0),
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), // Block B 16 - 31
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 9), S(0, 0, 1),

    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 2), S(0, 0, 2), // Block C  0 - 15
    S(1, 0, 0), S(1, 0, 0), S(0, 0, 9), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), // Block C 16 - 31
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1),

    S(0, 1, 1), S(0, 0, 1), S(1, 1, 0), S(0, 1, 0), S(0, 0, 0), S(0, 0, 0), S(0, 1, 1), S(0, 0, 1), // Block D  0 - 15
    S(0, 1, 5), S(0, 1, 5), S(1, 0, 0), S(0, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 0), S(1, 0, 0),
    S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), // Block D 16 - 31
    S(0, 0, 1), S(1, 0, 0), S(0, 0, 1), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0),

    S(0, 1, 1), S(0, 1, 0), S(0, 1, 9), S(0, 1, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 0), S(0, 0, 1), // Block E  0 - 15
    S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1),
    S(0, 1, 1), S(0, 1, 1), S(0, 1, 0), S(0, 1, 0), S(0, 1, 0), S(0, 1, 0), S(0, 1, 0), S(0, 1, 1), // Block E 16 - 31
    S(0, 1, 1), S(0, 1, 0), S(0, 1, 0), S(0, 1, 0), S(0, 1, 0), S(0, 1, 1), S(0, 1, 0), S(0, 1, 0),

    S(0, 1, 0), S(0, 1, 0), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 0), S(0, 1, 1), S(0, 1, 1), // Block F  0 - 15
    S(0, 1, 0), S(0, 1, 0), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 1, 1), S(0, 0, 1), S(0, 1, 1),
    S(0, 0, 1), S(0, 0, 1), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), S(0, 1, 2), // Block F 16 - 31
    S(0, 1, 2), S(0, 1, 2), S(0, 0, 3), S(0, 0, 1), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0), S(0, 0, 0),

    S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), S(0, 0, 0), S(0, 0, 0), S(0, 1, 5), // Block G  0 - 15
    S(0, 1, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5),
    S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 1, 4), S(0, 1, 0), S(0, 1, 5), S(0, 1, 5), S(0, 1, 5), // Block G 16 - 31
    S(0, 1, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(0, 0, 5), S(1, 1, 0), S(0, 0, 6), S(0, 0, 6),

    S(0, 0, 6), S(0, 0, 2), S(1, 1, 0), S(0, 1, 2), S(0, 1, 6), S(0, 0, 2), S(0, 0, 2), S(0, 0, 0), // Block H  0 - 15
    S(0, 0, 0), S(0, 0, 0), S(0, 1, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0),
    S(1, 0, 0), S(0, 0, 1), S(0, 0, 1), S(0, 0, 1), S(0, 1, 1), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), // Block H 16 - 31
    S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0), S(1, 0, 0)
};


/* Mapping of signal name to physical gpio number */
static const struct gpio_mapping_t card01GpioMapping[] = {
    { OP_PANEL_IRQ, BLOCKA(0) },
    { BSPI_CLK, BLOCKA(3) },
    { BSPI_CS,  BLOCKA(4) },
    { BSPI_TXD, BLOCKA(5) },
    { BSPI_RXD, BLOCKA(6) },
    { OC_USB_FRNT, BLOCKA(8) },
    { PWR_EN_USB_FRNT, BLOCKA(9) },
    { SDMMC0_CLK,   BLOCKB(0) },
    { SDMMC0_DATA0, BLOCKB(2) },
    { SDMMC0_DATA1, BLOCKB(3) },
    { SDMMC0_DATA2, BLOCKB(4) },
    { SDMMC0_DATA3, BLOCKB(5) },
    { ENET_RESET,   BLOCKB(30) },
    { ENET_25MHZ,   BLOCKB(31) },
    { VIDEO_CLK,    BLOCKC(11) },
    { VIDEO_SIG,    BLOCKD(23) },
    { eMMC_ENABLE,  BLOCKD(16) },
    { AUDIO_ENABLE, BLOCKE(2)  }
};

// LS_Clk_Acc_GalvoP period counted in 240mhz clocks
#define LVDS_12P6 19
#define LVDS_13P3 18
#define LVDS_14P1 17
#define LVDS_15P0 16
#define LVDS_16P0 15
#define LVDS_17P1 14
#define LVDS_18P5 13
#define LVDS_20P0 12

#define LVDS_REF_PERIOD LVDS_13P3

int clk_lvds_ui_init(void) {
    struct MIRROR1_REGS_s *mirror1 = (struct MIRROR1_REGS_s *)DEC_DEC_LSCAN_MIRROR1_BASE;

    // Enable reference clock, only used by FIN224C serializer

    // disable LS_Clk_Acc_GalvoP
    mirror1->mcfg2 = MIRROR1_MCFG2_GALVOR_IDLE_REPLACE_VAL(mirror1->mcfg2, 0);
    mirror1->mcfg2 = MIRROR1_MCFG2_GALVOP_DIS_REPLACE_VAL(mirror1->mcfg2, 1);

    // set frequency
    mirror1->mcfg1 = MIRROR1_MCFG1_MODE_REPLACE_VAL(mirror1->mcfg1, 0x1);
    mirror1->mcfg1 = MIRROR1_MCFG1_CONTROL_REPLACE_VAL(mirror1->mcfg1, (LVDS_REF_PERIOD - 2));

    // set duty cycle
    mirror1->galvop_rise0 = MIRROR1_GALVOP_RISE0_ENABLE_MASK;
    mirror1->galvop_fall0 = MIRROR1_GALVOP_FALL0_ENABLE_MASK | (LVDS_REF_PERIOD / 2);

    // enable LS_Clk_Acc_GalvoP
    mirror1->mcfg2 = MIRROR0_MCFG2_GALVOP_DIS_REPLACE_VAL(mirror1->mcfg2, 0);

    // Wait for FIN224C serdes PLLs to stabilize.
    udelay(200);

    return 0;
}

int is_phy_gigabit(void) {
    return 1;
}

const char * card_get_type(void) {
    return "card01";
}

#define CARDREV_PIN 11
const char * card_get_revision(void) {
    uint32_t revision;

    revision = mv61x0_adc_read(CARDREV_PIN);

    switch (revision) {
        case 0xf:  return "02b";  break;
        case 0x5:
        case 0x4:  return "12a";  break;
        case 0xb:
        case 0xa:  return "12b";  break;
        case 0x9:
        case 0x8:  return "12c";  break;
        case 0x7:
        case 0x6:  return "12d";  break;
        case 0x3:
        case 0x2:  return "12e";  break;
        case 0x1:
        case 0x0:  return "12f";  break;
        default:
        case 0xd:
        case 0xc:  return "12g";  break;
    }
}

int board_supports_lcd_vsync_in(void) {
    const char *revision = card_get_revision( );
    /*
     * This only means the board layout supports vsync_in.
     * There may be display-specific population differences.
     * To support "illegal" combinations of development hardware, the
     * driver for displays using vsync in may need to be verified by
     * setting a timeout.
     */
    if(revision) /* return true for versions 12a and later */
        return (strncmp(revision, "02b", 3) && strncmp(revision, "01a", 3));
    else
        return 0;
}

static void gpio_fix_for_overcurrent_usb_host(void) {
    const char *revision = card_get_revision( );
    /*
     * The old boards need to have GPIOA(9) set output High
     * to fix the over current circuit.
     * The new boards should be fixed and have GPIOA(9)
     * set to function 1.
    */
    if(revision) {
        if (strncmp(revision, "12c", 3) < 0) {
            gpio_set(OC_USB_FRNT, 0);
            gpio_direction(OC_USB_FRNT, 0);
            gpio_mux(OC_USB_FRNT, 0);

            if (strncmp(revision, "12b", 3) < 0) {
                gpio_set(PWR_EN_USB_FRNT, 1); //12a
            } else {
                gpio_set(PWR_EN_USB_FRNT, 0); //12b
            }
            gpio_direction(PWR_EN_USB_FRNT, 1);
            gpio_mux(PWR_EN_USB_FRNT, 0);
        }
    }
}

int card_init(void) {
    struct APB_CONFIG_REGS_s *apb_config = (struct APB_CONFIG_REGS_s *)APB_CONFIG_BASE;

    gpio_init_pins(card01GpioMapping,
                   ARRAY_SIZE(card01GpioMapping),
                   card01GpioSetup,
                   ARRAY_SIZE(card01GpioSetup));

    gpio_fix_for_overcurrent_usb_host();

    // DCmotor4 encoder inputs are routed to the UPC block
    apb_config->PINCR = APB_CONFIG_PINCR_UPCENCODERSEL_REPLACE_VAL(apb_config->PINCR, 0x4);

    // Set drive characteristics for LVDS inputs
    apb_config->PDCR2 = APB_CONFIG_PDCR2_ILVDS_ODT_REPLACE_VAL(apb_config->PDCR2, 0xf);

    //turn off the 16ma override on lcd outputs
    apb_config->PDCR2 = APB_CONFIG_PDCR2_GPIOC_LEDB_REPLACE_VAL(apb_config->PDCR2, (~0));

    return 0;
}

void card_dram_init(void) {
    int ddr_type = DDR3_x16;
    const char *card_rev = card_get_revision( );

    if ((strcmp(card_rev, "02b") == 0) || (strcmp(card_rev, "12d") > 0))
        ddr_type = DDR3_x32;

    initDDR3_nt5cb128m16dp_be_MHz400(ddr_type);
}

#define VOLTAGE_DROPOFF_ADJUSTMENT 10
#define TPS53819_I2C_BUS_NUM       1
#define TPS53819_I2C_ADDR          0x10
/* The core voltage regulator is configured early (before PLL is ramped) */
void card_config_core_voltage(char *asic_rev) {
    config_regulator(asic_rev,
                     VOLTAGE_DROPOFF_ADJUSTMENT,
                     TPS53819_I2C_BUS_NUM,
                     TPS53819_I2C_ADDR);
}

