#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/kthread.h>
#include <asm/gpio.h>
#include <asm/io.h>
#include <linux/workqueue.h>


#include "svinput.h"
#include "sv_power.h"

extern int init_converter(svinput_priv_t *priv);
extern int init_opu(svinput_priv_t *priv, int do_mtrig);
extern int exchange_boot_reason(svinput_priv_t *priv);
extern void enable_opereq_irq(void);
extern void disable_opereq_irq(void);
extern void enable_mtrig_irq(void);
extern void disable_mtrig_irq(void);
extern int i2c_master_sendrecv(const struct i2c_client *client, const char *sndbuf, int sndcount, char *rcvbuf, int rcvcount);
extern u8 svinput_thine_read(struct i2c_client *client, u8 reg_addr, u8 *reg_val);
extern unsigned char calc_sum(unsigned char *buf, int sum_pos);
extern int opereq_assert_check(struct i2c_client *client);

static DECLARE_WAIT_QUEUE_HEAD(power_poll_queue);

static int s_power_is_opened = 0;
static struct i2c_client *s_client = NULL;
svinput_slvl_ver_cmd_t s_ucom_ver;
svinput_slv_boot_cmd_t s_boot_cmd;

static int s_mtrig_asserted = 0;
static int s_wakeup_from_opu = 0;
static int s_scr_reboot_evt = 0;
static int s_opepanel_disconnect = 0;
static int s_opereq_polling = 0;
int s_thine233_power = 1;

extern power_mtrig_work_t  pw_mt_work_struct;
extern power_opereq_work_t pw_or_work_struct;
extern power_opereq_work_t pw_scr_work_struct;
extern power_opereq_poll_work_t pw_or_poll_work_struct;
extern workqueue_struct_t *ptr_svinput_wqueue;
extern svinput_work_t sv_work_struct;
extern int already_svinput_boot;
extern int do_panel_reboot;
extern int is_cheetah_ret;
extern int is_getting_up;
extern int enable_scr_interrupt;

/** *************************************************************************
 * User action, ADF open/close & paper set
 * begin
 */
#define SCAN_SENS_MON_INTERVAL (500) /* 500ms */
#define MTRG_INTERVAL_MIN (800) /* 1sec */
#define SCAN_DOC_MON_PORT (50) /* IO_PAD50 : GPIOB[18] */
#define SCAN_AKS_MON_PORT (51) /* IO_PAD51 : GPIOB[19] */
#define SCAN_PONSENS_PORT (147)/* IO_PAD147 : GPIOE[19] */
DEFINE_MUTEX(user_act_mutex);

typedef enum {
	SCAN_ENG_ST_UNDEFINED = -1,
	SCAN_ENG_ST_OFF = 0,
	SCAN_ENG_ST_ON = 1,
	SCAN_ENG_ST_ANCHOR = 2,
} scan_eng_stat_t;

static scan_eng_stat_t s_scan_eng_status = SCAN_ENG_ST_UNDEFINED;
static int s_scan_eng_evt_fire_flag = 0;
static struct task_struct *s_scan_sens_kthread = NULL;

/* ADF open/close sens helpers */
static void scan_set_eng_status(scan_eng_stat_t st)
{
	/* setterはセンサーの状態確認処理と排他する */
	mutex_lock(&user_act_mutex);
	s_scan_eng_status = st;
	/**
	 * SCAN_ENG_ST_ONでスキャナがOnになる場合にはセンサーの電源制御を邪魔しないように0(Active)にする
	 */
	gpio_set_value(SCAN_PONSENS_PORT, (s_scan_eng_status == SCAN_ENG_ST_ON) ? 1 : 0);
	mutex_unlock(&user_act_mutex);
}

static scan_eng_stat_t scan_get_eng_status(void)
{
	return s_scan_eng_status;
}

static int scan_sensor_monitoring_thread(void *arg)
{
	volatile scan_eng_stat_t st = SCAN_ENG_ST_UNDEFINED;
	struct timeval tv_start = {
		.tv_sec = 0,
		.tv_usec = 0,
	};
	struct timeval tv_end = {
		.tv_sec = 0,
		.tv_usec = 0,
	};
	int proc_time = 0, cnt = 0;
	int prev_val_aks = 0, val_aks = 0;
	int prev_val_doc = 0, val_doc = 0;
	int tmp_val_aks[2] = {0};
	int tmp_val_doc[2] = {0};

	FUNC_START;

	prev_val_aks = gpio_get_value(SCAN_AKS_MON_PORT);
	prev_val_doc = gpio_get_value(SCAN_DOC_MON_PORT);

	while (!kthread_should_stop()) {
		do_gettimeofday(&tv_start);
		mutex_lock(&user_act_mutex);
		st = scan_get_eng_status();

		/* スキャナセンサーのポートはON中にONと言われても悪影響はないので常にONする所作とする */
		gpio_set_value(SCAN_PONSENS_PORT, 1);
		mdelay(1);
		tmp_val_aks[cnt] = gpio_get_value(SCAN_AKS_MON_PORT);
		tmp_val_doc[cnt] = gpio_get_value(SCAN_DOC_MON_PORT);
		if (st == SCAN_ENG_ST_OFF) {
			gpio_set_value(SCAN_PONSENS_PORT, 0);
		}

		/* 2回連続で同じ値なら値確定 */
		if ((tmp_val_aks[0] + tmp_val_aks[1]) == 0) {
			val_aks = 0;
		} else if ((tmp_val_aks[0] + tmp_val_aks[1]) == 2) {
			val_aks = 1;
		} else {
			val_aks = -1;/* チャタ除去中 */
		}
		if ((tmp_val_doc[0] + tmp_val_doc[1]) == 0) {
			val_doc = 0;
		} else if ((tmp_val_doc[0] + tmp_val_doc[1]) == 2) {
			val_doc = 1;
		} else {
			val_doc = -1;/* チャタ除去中 */
		}

		if (((val_doc == 0) && (val_doc != prev_val_doc)) || ((val_aks != -1) && (val_aks != prev_val_aks))) {
			s_scan_eng_evt_fire_flag++;
			FORCE_DPRINTK(VT100_GREEN"[svinput]s_scan_eng_evt_fire_flag=%d\n"VT100_NORM, s_scan_eng_evt_fire_flag);
			wake_up_interruptible(&power_poll_queue);
		}

		/* 値が確定している時だけ前の値を更新する */
		cnt = (cnt + 1) % 2;
		if ((val_aks != -1) && (prev_val_aks != val_aks)) {
			FORCE_DPRINTK(VT100_RICHBLUE"[svinput]AKS %s\n"VT100_NORM, (val_aks == 0) ? "Open" : "Close");
			prev_val_aks = val_aks;
		}
		if ((val_doc != -1) && (prev_val_doc != val_doc)) {
			FORCE_DPRINTK(VT100_RICHBLUE"[svinput]DOC %s\n"VT100_NORM, (val_doc == 0) ? "Set" : "Unset");
			prev_val_doc = val_doc;
		}

		mutex_unlock(&user_act_mutex);
		do_gettimeofday(&tv_end);
		proc_time = ((tv_end.tv_sec - tv_start.tv_sec) * 1000) + ((tv_end.tv_usec - tv_start.tv_usec) / 1000);
		if (proc_time < 0) {
			proc_time = 0;
		} else if (proc_time > SCAN_SENS_MON_INTERVAL) {
			proc_time = SCAN_SENS_MON_INTERVAL;
		}
		msleep(SCAN_SENS_MON_INTERVAL - proc_time);
	}

	FUNC_END;

	return 0;
}
/** 
 * end
 */

void power_probe(struct i2c_client *client, const struct i2c_device_id *idp)
{
	s_client = client;

	/* ********** Scanner sensor ********** */
	FORCE_DPRINTK(VT100_YELLOW"[svinput]Creating kernel thread...\n"VT100_NORM);
	scan_set_eng_status(SCAN_ENG_ST_ON);
	s_scan_sens_kthread = kthread_run(scan_sensor_monitoring_thread, NULL, "SCNSENMON");
	msleep(10);
	/* ********** Scanner sensor ********** */

	return;
}

int power_remove(struct i2c_client *client)
{
	return 0;
}

void power_shutdown(void)
{
	FUNC_START;
	s_scan_sens_kthread = NULL;
	FUNC_END;
	return;
}

int power_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_power_is_opened = SVINPUT_FALSE;
	FUNC_END;
	return rc;
}

int power_release(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_power_is_opened = SVINPUT_FALSE;
	FUNC_END;
	return rc;
}

ssize_t power_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	int rc = 0;
	static const u8 key_buf_on[6] = {0x01, 0x01, 0x00, 0x00, 0x26, 0x01};
	static const u8 key_buf_off[6] = {0x01, 0x01, 0x00, 0x00, 0x26, 0x00};
	static const u8 null_buf[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
	static const u8 key_buf_user_act[6] = {0x01, 0x01, 0x00, 0x00, 0x31, 0x01};
	static const u8 key_ope_network_error[6] = {0x01, 0x01, 0x00, 0x00, 0x8c, 0x01};
	static const u8 key_ope_shutdown[6] = {0x01, 0x01, 0x00, 0x00, 0x8d, 0x01};
	struct timeval tv = {
		.tv_sec = 0,
		.tv_usec = 0,
	};

	FUNC_START;
	DPRINTK("[svinput]svinput_power_read start\n");
	
	do_gettimeofday(&tv);
	
	if (count >= 6) {
		if (s_mtrig_asserted == 2) {
			FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)notify_eco_key_on\n", tv.tv_sec, tv.tv_usec);
			rc = copy_to_user(buffer, key_buf_on, 6);
			s_mtrig_asserted--;
		} else if (s_mtrig_asserted == 1) {
			FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)notify_eco_key_off\n", tv.tv_sec, tv.tv_usec);
			rc = copy_to_user(buffer, key_buf_off, 6);
			s_mtrig_asserted--;
		} else if ((s_mtrig_asserted == 0) && (s_scan_eng_evt_fire_flag != 0)) {
			rc = copy_to_user(buffer, key_buf_user_act, 6);
			s_scan_eng_evt_fire_flag = 0;
		} else if ((s_mtrig_asserted == 0) && (s_scan_eng_evt_fire_flag == 0) && (s_scr_reboot_evt != 0)) {
			FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld) key_ope_network_error\n", tv.tv_sec, tv.tv_usec);
			rc = copy_to_user(buffer, key_ope_network_error, 6);
			s_scr_reboot_evt = 0;
		} else if ((s_mtrig_asserted == 0) && (s_scan_eng_evt_fire_flag == 0) && (s_scr_reboot_evt == 0) && (s_opepanel_disconnect != 0)) {
			FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld) key_ope_shutdown\n", tv.tv_sec, tv.tv_usec);
			rc = copy_to_user(buffer, key_ope_shutdown, 6);
			s_opepanel_disconnect = 0;
		} else {
			rc = copy_to_user(buffer, null_buf, 6);
		}
		count = 6;
	} else {
		count = 0;
	}
	DPRINTK("[svinput]svinput_power_read end\n");
	FUNC_END;

	return count;
}

unsigned int power_poll(struct file *file, poll_table * wait)
{
	int rc = 0;
	
	poll_wait(file, &power_poll_queue, wait);
	if (s_mtrig_asserted) {
		DPRINTK("[svinput]power_poll s_mtrig_asserted : %d\n", s_mtrig_asserted);
		rc = (POLLIN | POLLRDNORM);
	} else if (s_scan_eng_evt_fire_flag) {
		DPRINTK(VT100_GREEN"[svinput]AKS or DOC-set : %d"VT100_NORM"\n", s_scan_eng_evt_fire_flag);
		rc = (POLLIN | POLLRDNORM);
	} else if (s_scr_reboot_evt) {
		DPRINTK("[svinput]power_poll s_scr_reboot_evt : %d\n", s_scr_reboot_evt);
		rc = (POLLIN | POLLRDNORM);
	} else if (s_opepanel_disconnect) {
		DPRINTK("[svinput]power_poll s_opepanel_disconnect : %d\n", s_opepanel_disconnect);
		rc = (POLLIN | POLLRDNORM);
	}
	
	
	return rc;
}

long power_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int rc = -1;
	u8 reg_val = 0x00;
	char recv_buf[I2C_RECV_BUF_SIZE] = {0};
	svinput_mas_lp_cmd_t master_lpow_cmd = {
		.hdr = ST1,
		.cmd = 0x1a,
		.len = 0x01,
		.sst = 0x01,
		.sum = 0x1c,
	};
	struct timeval tv = {
		.tv_sec = 0,
		.tv_usec = 0,
	};

	FUNC_START;
	if (!s_client) {
		rc = -1;
		DPRINTK(VT100_RED"[svinput]invalid i2c client...\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	
	do_gettimeofday(&tv);

	switch (cmd) {
	/* 操作部電源OFF要求 */
	case PW_IOC_OPEPANEL_OFF:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_OFF\n", tv.tv_sec, tv.tv_usec);
		rc = i2c_master_send(s_client, (char*)(&master_lpow_cmd), sizeof(master_lpow_cmd));
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG "[svinput]Can't send power management command (-_-;\n");
		}
		rc = i2c_master_recv(s_client, recv_buf, sizeof(recv_buf));
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave ack(-_-;\n");
		} else {
			/*rc = (recv_buf[0] == PAK) ? 0 : -1;*/
			rc = 0;
		}
		break;
	/* Key uCom's version */
	case PW_IOC_MC_VERSIONS:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld,%ld)PW_IOC_MC_VERSIONS\n", tv.tv_sec, tv.tv_usec);
		rc = copy_to_user((void __user *)arg, &s_ucom_ver, MFPOPE_SIZE_SVERDAT);
		if (rc != 0) {
			rc = -1;
		}
		break;
	case PW_IOC_OPEPANEL_ENERGYSAVE:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_ENERGYSAVE\n", tv.tv_sec, tv.tv_usec);
		/**
		 * やる事は以下
		 * 1. Key uComに省エネ状態への遷移を指示
		 * 2. 233 MainLink/SubLink Power Down
		 * 3. ope_reqのIRQ無効化＆MTのIRQ有効化
		 */
		if(s_thine233_power != 0)
		{
			rc = signal_status_notify(NULL, 0x11, 0);
			if(rc < 0)
			{
				rc = -1;
				FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Failed to execute energy save command (-_-;\n");
				break;
			}
			mdelay(1);/* 通信が234を経由して相手に届くのを待つ */
			gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 0);
			gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 0);
			s_thine233_power = 0;
		}
		
		disable_opereq_irq();
		
		/* I2Cコマンドが相手に届くのを待つ */
		msleep(10);

		FORCE_DPRINTK(KERN_EMERG"[svinput]sst command sended\n");

		enable_mtrig_irq();
		break;
	case PW_IOC_OPEPANEL_ON:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_ON\n", tv.tv_sec, tv.tv_usec);
		/**
		 * やる事は以下
		 * 1. 233 MainLink/SubLink Power Up
		 * 2. 233 初期化
		 * 3. key uCom初期化
		 * 4. MTのIRQ無効化＆ope_reqのIRQ有効化
		 */
		gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 1);
		gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 1);
		s_thine233_power = 1;
		mdelay(10);/* magic? */
		rc = init_converter(NULL);
		if(rc != 0)
		{
			rc = -1;
			FORCE_DPRINTK(KERN_EMERG "[svinput]Failed to execute thine init function (-_-;\n");
			break;
		}
		/* version not exchange, POKUSB ON, SDMODE OFF, PONOPE ON */
		rc = signal_status_notify(NULL, 0x90, 1);
		if(rc != 0)
		{
			rc = -1;
			FORCE_DPRINTK(KERN_EMERG "[svinput]Failed to execute opepanel on command (-_-;\n");
			break;
		}
		
		disable_mtrig_irq();
		
		if(s_wakeup_from_opu != 0)
		{
			s_wakeup_from_opu = 0;
		}
		
		/* I2Cコマンドが相手に届くのを待つ */
		msleep(10);
		
		/* Opereq読み捨て */
		svinput_thine_read(sv_work_struct.priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
		enable_opereq_irq();
		
		if (is_cheetah_ret) {
			if (gpio_get_value(62) == 0) {
				gpio_set_value(62, 1);
				msleep(50);
			}
		}
		++is_getting_up;
		
		break;
	case PW_IOC_OPEPANEL_POKUSBON:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_POKUSBON\n", tv.tv_sec, tv.tv_usec);
		
		/* CTL要因復帰の時 */
		if(s_wakeup_from_opu == 0)
		{
			disable_mtrig_irq();
			gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 1);
			gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 1);
			s_thine233_power = 1;
			mdelay(10);/* magic? */
			rc = init_converter(NULL);
			if (rc != 0)
			{
				rc = -1;
				FORCE_DPRINTK(KERN_EMERG "[svinput]Failed to execute thine init function (-_-;\n");
				break;
			}
			rc = init_opu(NULL, 1);
		}
		/* 操作部要因復帰の時 */
		else
		{
			s_wakeup_from_opu = 0;
		}
		
		/* version not exchange, POKUSB ON, SDMODE OFF, PONOPE ON */
		if (rc != FAKE_MTRIG) {
			rc = signal_status_notify(NULL, 0x91, 0);
		}
		if(rc < 0)
		{
			rc = -1;
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send pokusb command\n"VT100_NORM);
			break;
		}
		
		enable_mtrig_irq();
		
		break;
	/* ********** Scanner sensor ********** */
	case PW_IOC_SCAN_ENGON:
		FORCE_DPRINTK(KERN_NOTICE "[svinput](%ld.%ld)PW_IOC_SCAN_ENGON\n", tv.tv_sec, tv.tv_usec);
		++is_getting_up;
		scan_set_eng_status(SCAN_ENG_ST_ON);
		rc = 0;
		break;
	/* ********** Scanner sensor ********** */
	case PW_IOC_SCAN_ENGOFF:
		FORCE_DPRINTK(KERN_NOTICE "[svinput](%ld.%ld)PW_IOC_SCAN_ENGOFF\n", tv.tv_sec, tv.tv_usec);
		scan_set_eng_status(SCAN_ENG_ST_OFF);
		rc = 0;
		break;
	case PW_IOC_OPEPANEL_SHUTDOWN:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_SHUTDOWN\n", tv.tv_sec, tv.tv_usec);
		/* Master→SLAVE notify SST (PONOPE OFF, POKUSB OFF, SDMODE ON) */
		signal_status_notify(NULL, 0x01, 0);
		break;
	case PW_IOC_OPEPANEL_FORCE_OFF:
		FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)PW_IOC_OPEPANEL_FORCE_OFF\n", tv.tv_sec, tv.tv_usec);
		disable_mtrig_irq();
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 0);
		msleep(500);
		gpio_set_value(SVINPUT_MTRIG_OUT_GPIO_NO, 1);
		
		rc = wait_opepanel_disconnect();
		if (rc != 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't disconnect opu, but force shutdown\n"VT100_NORM);
		} else {
			FORCE_DPRINTK(KERN_EMERG "[svinput]Disconnected opu usb\n"VT100_NORM);
		}
		
		s_opepanel_disconnect = 1;
		wake_up_interruptible(&power_poll_queue);
		
		break;
	case PW_IOC_OPEPANEL_CHECK_GETTINGUP:
		rc = (is_getting_up) ? 1 : 0;
		break;
	case PW_IOC_OPEPANEL_RESET_GETTINGUP_FLAG:
		is_getting_up = 0;
		rc = 0;
		break;
	default:
		break;
	}

 EXIT_FUNC:
	FUNC_END;
	return rc;
}

void power_set_ucom_ver(svinput_slvl_ver_cmd_t *ucom_ver)
{
	memcpy(&s_ucom_ver, ucom_ver, sizeof(s_ucom_ver));
}

void power_set_boot_cmd(svinput_slv_boot_cmd_t *slv_boot)
{
	memcpy(&s_boot_cmd, slv_boot, sizeof(s_boot_cmd));
}

irqreturn_t power_mtrig_isr(int irq, void *dev_id)
{
	FUNC_START;
	
	is_getting_up = 1;
	
	/* ボトムハーフ実行 */
	queue_work(ptr_svinput_wqueue, (work_struct_t*)&pw_mt_work_struct);

	FUNC_END;

	return IRQ_HANDLED;
}

void power_mtrig_wqueue_handler(work_struct_t *work)
{
	int rc = -1;
	u8 reg_val = 0x00;

	struct timeval tv_now = {
		.tv_sec = 0,
		.tv_usec = 0,
	};
	static struct timeval tv_prev = {
		.tv_sec = 0,
		.tv_usec = 0,
	};

	int mtrg_interval_time = 0;
	
	do_gettimeofday(&tv_now);
	if ((tv_prev.tv_sec == 0) && (tv_prev.tv_usec == 0)){
		mtrg_interval_time = MTRG_INTERVAL_MIN;
	}
	else{
		mtrg_interval_time = ((tv_now.tv_sec - tv_prev.tv_sec) * 1000) + ((tv_now.tv_usec - tv_prev.tv_usec) / 1000);
	}
	
	/* 前回のMT検知時刻から800ms経過している場合はCheetahとのネゴシエーションを開始する */
	if(mtrg_interval_time >= MTRG_INTERVAL_MIN){
		tv_prev = tv_now;
	}
	/**
	 * システム時刻変更などによりtv_prev > tv_nowになった場合、
	 * tv_prevをセットし直し、一度MTボトムハーフ処理をスキップする
	 */
	else if (mtrg_interval_time < 0){
		printk("[svinput]Invalid Time Value\n");
		tv_prev = tv_now;
		printk(KERN_EMERG "[svinput]skip %s\n",__func__);
		return;
	}
	/* 前回のMT検知時刻から800ms経っていない場合は以降の処理をスキップする */
	else{
		printk(KERN_EMERG "[svinput]skip %s\n",__func__);
		return;
	}

	FUNC_START;

	FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)power_mtrig_wqueue_handler start\n", tv_now.tv_sec, tv_now.tv_usec);

	gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 1);
	gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 1);
	s_thine233_power = 1;
	mdelay(10);/* magic? */
	rc = init_converter(NULL);
	if (rc != 0) {
		rc = -1;
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Failed to execute thine init function (-_-;\n"VT100_NORM);
	}
	
	rc = init_opu(NULL, 0);
	if (rc == FAKE_MTRIG){
		printk(KERN_EMERG "[svinput]skip %s\n",__func__);
		return;
	}
	else if (rc != 0) {
		rc = -1;
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Failed to execute opu init function (-_-;\n"VT100_NORM);
	}
	
	FORCE_DPRINTK(KERN_EMERG "[svinput]boot reason 0x%02x\n", s_boot_cmd.cau);
	
	if (s_boot_cmd.cau == 0x10 || s_boot_cmd.cau == 0x11 || s_boot_cmd.cau == 0x12)
	{
		s_mtrig_asserted = 2;
		s_wakeup_from_opu = 1;
		wake_up_interruptible(&power_poll_queue);
	}
	else if (s_boot_cmd.cau == 0x0f)
	{
		/* I2Cコマンドが相手に届くのを待つ */
		msleep(10);
		
		/* Opereq読み捨て */
		svinput_thine_read(sv_work_struct.priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
		disable_mtrig_irq();
		enable_opereq_irq();
	}
	
	FUNC_END;
}

irqreturn_t power_opereq_isr(int irq, void *dev_id)
{
	FUNC_START;

	/* ボトムハーフ実行 */
	queue_work(ptr_svinput_wqueue, (work_struct_t*)&pw_or_work_struct);

	FUNC_END;

	return IRQ_HANDLED;
}

void power_opereq_wqueue_handler(work_struct_t *work)
{
	int rc 		= -1;
	int snd_cnt	= 0;
	u8 reg_val = 0x00;
	unsigned char sum = 0x00;
	uint8_t recv_buf[I2C_RECV_BUF_SIZE] = {0};
	uint8_t send_buf[I2C_SEND_BUF_SIZE] = {0};
	svinput_priv_t *priv = sv_work_struct.priv;
	svinput_mas_boot_cmd_t master_boot = {
		.hdr = ST1,
		.cmd = 0x1c,
		.len = 0x01,
	};
	struct timeval tv = {
		.tv_sec = 0,
		.tv_usec = 0,
	};


	FUNC_START;
	
	do_gettimeofday(&tv);
	
	FORCE_DPRINTK(KERN_EMERG "[svinput](%ld.%ld)power_opereq_wqueue_handler start\n", tv.tv_sec, tv.tv_usec);
	
	if (do_panel_reboot == 1)
	{
		DPRINTK("[svinput]panel rebooting. opereq wqueue handler skip...\n");
		goto EXIT_FUNC;
	}

	/* SLV->MAS 1コマンド受信 */
	rc = i2c_master_recv(priv->opu_client, (char*)(recv_buf), sizeof(recv_buf));
	if (rc < 0)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't recv slave command\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	else
	{
		sum = calc_sum((char*)(recv_buf), 4);
		
		if((sum != recv_buf[4]) || (recv_buf[0] == 0))
		{
			FORCE_DPRINTK(KERN_EMERG "[svinput]invalid command recieved\n");
			FORCE_DPRINTK(KERN_EMERG "[svinput]hdr=0x%02x\n", recv_buf[0]);
			FORCE_DPRINTK(KERN_EMERG "[svinput]cmd=0x%02x\n", recv_buf[1]);
			FORCE_DPRINTK(KERN_EMERG "[svinput]len=0x%02x\n", recv_buf[2]);
			FORCE_DPRINTK(KERN_EMERG "[svinput]dat=0x%02x\n", recv_buf[3]);
			FORCE_DPRINTK(KERN_EMERG "[svinput]sum=0x%02x, slvae.sum=0x%02x\n", sum, recv_buf[4]);
			goto EXIT_FUNC;
		}
		else
		{
			DPRINTK("[svinput]hdr=0x%02x\n", recv_buf[0]);
			DPRINTK("[svinput]cmd=0x%02x\n", recv_buf[1]);
			DPRINTK("[svinput]len=0x%02x\n", recv_buf[2]);
			DPRINTK("[svinput]dat=0x%02x\n", recv_buf[3]);
			DPRINTK("[svinput]sum=0x%02x, slvae.sum=0x%02x\n", sum, recv_buf[4]);
		}
	}
	send_buf[0] = send_buf[1] = send_buf[2] = PAK;
	rc = i2c_master_send(priv->opu_client, send_buf, 3);
	if(rc < 0)
	{
		FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't send pak\n"VT100_NORM);
		goto EXIT_FUNC;
	}
	
	FORCE_DPRINTK(KERN_EMERG "[svinput]recieve cmd 0x%02x\n", recv_buf[1]);
	
	if (recv_buf[1] == 0x1b) /* notify boot reason */
	{
		power_set_boot_cmd((svinput_slv_boot_cmd_t *)&recv_buf);
	
RETRY_SEND:
		
		/* MAS->SLV notify boot reason */
		DPRINTK("[svinput]MAS->SLV notify boot reason\n")
		master_boot.cau = recv_buf[3];
		master_boot.sum = calc_sum((char*)(&master_boot), sizeof(master_boot) - 1);
		rc = i2c_master_sendrecv(priv->opu_client, (char*)(&master_boot), sizeof(master_boot), recv_buf, sizeof(recv_buf));
		if(rc < 0)
		{
			FORCE_DPRINTK(KERN_EMERG VT100_RED"[svinput]Can't recv slave boot\n"VT100_NORM);
			//goto EXIT_FUNC;
			FORCE_DPRINTK(KERN_EMERG "[i2c_master_sendrecv][%s]clk_stretch=0x%x\n", __func__, readl(clk_stretch) );
			goto RETRY_SEND;

		}
		else
		{
			DPRINTK("[svinput]hdr=0x%02x\n", master_boot.hdr);
			DPRINTK("[svinput]cmd=0x%02x\n", master_boot.cmd);
			DPRINTK("[svinput]len=0x%02x\n", master_boot.len);
			DPRINTK("[svinput]cau=0x%02x\n", master_boot.cau);
			DPRINTK("[svinput]sum=0x%02x\n", master_boot.sum);
		}
		if(recv_buf[0] != PAK)
		{
			DPRINTK("[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
			
			if(snd_cnt <= 10)
			{
				snd_cnt++;
				mdelay(20);
				goto RETRY_SEND;
			}
			else
			{
				FORCE_DPRINTK(KERN_EMERG "[svinput]Can't recv slave ack but no problem\n");
				FORCE_DPRINTK(KERN_EMERG "[svinput]recv_buf[0]=0x%02x, recv_buf[1]=0x%02x, recv_buf[2]=0x%02x\n", recv_buf[0], recv_buf[1], recv_buf[2]);
			}
		}
		
		FORCE_DPRINTK(KERN_EMERG "[svinput]boot reason 0x%02x\n", s_boot_cmd.cau);
		
		if(s_boot_cmd.cau == 0x10 || s_boot_cmd.cau == 0x11 || s_boot_cmd.cau == 0x12)
		{
			s_mtrig_asserted = 2;
			wake_up_interruptible(&power_poll_queue);
		}
	}
	else if (recv_buf[1] == 0x1f) /* notify opepanel reboot */
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]recieve event 0x%02x\n", recv_buf[3]);
		
		if (recv_buf[3] == 0x00 || recv_buf[3] == 0x01 || recv_buf[3] == 0x02) /* reconnect */
		{
			mdelay(1);/* 通信が234を経由して相手に届くのを待つ */
			
			gpio_set_value(SVINPUT_233_MAIN_GPIO_NO, 0);
			gpio_set_value(SVINPUT_233_SUB_GPIO_NO, 0);
			s_thine233_power = 0;
			
			do_panel_reboot = 1;
			s_opereq_polling = 0;
			
			disable_opereq_irq();
			enable_mtrig_irq();
		}
	}
	else if(recv_buf[1] == 0x1e) /* notify opepanel status */
	{
		if (recv_buf[3] == 0x13) /* recovery mode */
		{
			FORCE_DPRINTK(KERN_EMERG "[svinput]opepanel recovery mode in\n");
			s_opereq_polling = 1;
			
			/* opereqポーリング開始 */
			queue_work(ptr_svinput_wqueue, (work_struct_t*)&pw_or_poll_work_struct);
		}
	}

EXIT_FUNC:
	
	/* Opereq読み捨て */
	svinput_thine_read(sv_work_struct.priv->cnv_client, (u8)SVINPUT_CNV_GPIO_REG_ADDR, &reg_val);
	
	FUNC_END;

}

irqreturn_t power_scr_reboot_isr(int irq, void *dev_id)
{
	FUNC_START;
	
	/* ボトムハーフ実行 */
	/* scr割り込み許可フラグが立っていることを確認 */
	if (enable_scr_interrupt != 0){
		queue_work(ptr_svinput_wqueue, (work_struct_t*)&pw_scr_work_struct);
	}
	FUNC_END;

	return IRQ_HANDLED;
}

void power_scr_wqueue_handler(work_struct_t *work)
{
	u32 val = 0x00000000;
	void __iomem *tmp_io;
	
	FUNC_START;
	
	/* SCR_PER_EN(GPIOE[30]) H:ON L:OFF */
	tmp_io = ioremap(0xd4019400, 4);
	val = ((readl(tmp_io)) & (BIT(30)));
	/* scrマイコンの電源が入っていることを確認 */
	if(val != 0)
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]scr pwr check on (enable_scr_interrupt:%d)\n", enable_scr_interrupt);
		s_scr_reboot_evt = 1;
		wake_up_interruptible(&power_poll_queue);
	}
	else
	{
		FORCE_DPRINTK(KERN_EMERG "[svinput]scr pwr check off\n");
	}
	
	iounmap(tmp_io);
	
	FUNC_END;
}

void power_or_poll_wqueue_handler(work_struct_t *work)
{
	int ret = 0;
	
	FUNC_START;
	
	while(1)
	{
		ret = opereq_assert_check(sv_work_struct.priv->cnv_client);
		
		if(ret == 1) /* assert */
		{
			power_opereq_wqueue_handler((work_struct_t*)&pw_or_work_struct);
		}
		
		if( s_opereq_polling == 0 )
		{
			break;
		}
		
		msleep(1000);
	}
	
	FUNC_END;
}
