#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <asm/gpio.h>

#include "svinput.h"
#include "sv_keyboard.h"

static int sv_kbd_info_num = 0;
static int s_keyboard_is_opened = SVINPUT_FALSE;

static DECLARE_WAIT_QUEUE_HEAD(kbd_wait_queue);

#define UCOMCODE_TO_INDEX(x) ((((x >> 4) & 0x0f) * MAX_KEYSCAN_COL) + (x & 0x0f))
static const sv_ctl_key_code_t key_code_tbl[MAX_KEYSCAN_COL * MAX_KEYSCAN_ROW] = {
	OPE_KEY_1, OPE_KEY_7,          OPE_KEY_CLR,     OPE_KEY_HOME, OPE_KEY_SIMPLE_WIN, OPE_KEY_STS_INFO,
	OPE_KEY_2, OPE_KEY_8,          OPE_KEY_START,   OPE_KEY_F1,   OPE_KEY_LOGIN_OUT,  OPE_KEY_ENESAVE,
	OPE_KEY_3, OPE_KEY_9,          OPE_KEY_STP,     OPE_KEY_F2,   OPE_KEY_INT,        OPE_KEY_RESERVED,
	OPE_KEY_4, OPE_KEY_0,          OPE_KEY_COUNTER, OPE_KEY_F3,   OPE_KEY_CHECK,      OPE_KEY_RESERVED,
	OPE_KEY_5, OPE_KEY_KPASTERISK, OPE_KEY_PRGM,    OPE_KEY_F4,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,
	OPE_KEY_6, OPE_KEY_SHARP,      OPE_KEY_RESET,   OPE_KEY_F5,   OPE_KEY_RESERVED,   OPE_KEY_RESERVED,
};

static ope_input_event_t cur_key = {
	.time = {
		.tv_sec = 0,
		.tv_usec = 0,
	},
	.type = (uint32_t)SV_EV_KEY,
	.param.key = {
		.code  = (uint16_t)0,
		.status = (uint16_t)0,
	},
};

void keyboard_wqueue_handler(work_struct_t *work, svinput_slv_kbd_cmd_t *slv_kbd_cmd)
{
	svinput_work_t *ptr_work = NULL;

	ptr_work = (svinput_work_t*)work;

	++(ptr_work->kbd_cnt);
	do_gettimeofday(&(cur_key.time));
	cur_key.param.key.code = key_code_tbl[UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[1])];
	DPRINTK("form ucom=0x%02x, index=%d\n", slv_kbd_cmd->key_buf[1], UCOMCODE_TO_INDEX(slv_kbd_cmd->key_buf[1]));
	switch (slv_kbd_cmd->key_buf[0]) {
	case SVKBD_PUSH:
		cur_key.param.key.status = KEY_PUSH;
		DPRINTK("Press\n");
		break;
	case SVKBD_RELEASE:
		cur_key.param.key.status = KEY_RELEASE;
		DPRINTK("Release\n");
		break;
	default:
		/* FIX No.0004 */
		return;
	}
	if (s_keyboard_is_opened) {
		sv_kbd_info_num++;
		wake_up_interruptible(&kbd_wait_queue);
	} else {
		sv_kbd_info_num = 0;
	}
}

int keyboard_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_keyboard_is_opened = SVINPUT_TRUE;
	FUNC_END;
	return rc;
}

int keyboard_release(struct inode *inode, struct file *file)
{
	int rc = 0;
	FUNC_START;
	s_keyboard_is_opened = SVINPUT_FALSE;
	FUNC_END;
	return rc;
}

ssize_t keyboard_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	int rc = 0;

	FUNC_START;
	if ((sv_kbd_info_num == 0) && (file->f_flags & O_NONBLOCK)) {
		count = -EAGAIN;
		goto EXIT_FUNC;
	}
	wait_event_interruptible(kbd_wait_queue, (sv_kbd_info_num != 0));
	rc = copy_to_user(buffer, &cur_key, sizeof(ope_input_event_t));
	count = (rc == 0) ? sizeof(ope_input_event_t) : 0;
	sv_kbd_info_num--;
	if (sv_kbd_info_num < 0) {
		sv_kbd_info_num = 0;
	}

 EXIT_FUNC:
	FUNC_END;
	return count;
}

unsigned int keyboard_poll(struct file *file, poll_table * wait)
{
	unsigned int rc = 0;

	FUNC_START;
	poll_wait(file, &kbd_wait_queue, wait);

	if (sv_kbd_info_num != 0) {
		rc = (POLLIN | POLLRDNORM);
	}

	FUNC_END;
	return rc;
}
