/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

/**
 * \file stepper-api.h
*/

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/kfifo.h>
#include <linux/mutex.h>
#include <linux/semaphore.h>
#include <linux/kernel.h>
#include <linux/of_device.h>
#include <linux/kthread.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mach/irq.h>
#include <asm/gpio.h>

#include "stepper-api.h"
#include "stepper.h"
#include "STMotor_regheaders.h"
#include "stepper_cdma.h"

#define DRIVER_NAME "g2_stepper_api"

#if ASIC_REV <= ASIC_REV_A0
    #define ASIC_NAME_STR "a0"
#else
    #define ASIC_NAME_STR "b0"
#endif

#define NUM_STEPPERS 6
#define MAX_TRIGGERS 20
#define PWM_T_VALUE 5556
#define NUM_STEPS 32
#define RUN_FOREVER_STEPS 0xffffffff

// settings for the connects register
//
#define MODE_PWM1 0
#define MODE_PWM2 1
#define MODE_REG0 2
#define MODE_REG1 3
#define MODE_SEQ  4
#define MODE_LS   5

typedef struct stmotor_s
{
    uint32_t motor_id;

    /* Configured motor parameters */
    uint32_t         *accel_table;          // Accel table pointer set by motor params
    uint32_t          accel_table_entries;  // Number of values in the accel_table
    uint32_t         *deaccl_table;         // if null no table
    uint32_t          deaccl_table_entries;
    stmotor_accel_t   accel_select;         // Accel method set by motor params
    uint16_t          preholding_steps;
    uint16_t          postholding_steps;
    int               max_index;            // Max allowed accel index (speed limiter)

    int               accel_index;

        /* Motor state variables */
    stmotor_state_t   motor_state;
    uint32_t          ref_position;         // Tracks mech reference position

    /* Current move variables */
    uint32_t          pending_steps;        // Pending steps
    bool              ls_requested;         // Line start pulses requested
    uint32_t          enable_ls_at_stepnum; // Turn on ls at step number
    uint8_t           direction;            // Direction
    uint8_t           num_triggers;
    uint8_t           triggers_reported;    // the number of triggers reported.
    int8_t           trig_order[MAX_TRIGGERS];
    stmotor_trigger_t trigger[MAX_TRIGGERS];              // Trigger for current move
    stepper_cmd_packet_t  cmd_packet;       // Command packet
    uint32_t          stepper_cmd_deaccl;

} stmotor_spec_t;

typedef struct 
{
    uint32_t      phys_addr;
    void __iomem *virt_addr;
    uint32_t      addr_size;
    uint32_t      instance;
    uint32_t      irq;
} DEVICE_DATA;

/**
 * @brief this is the registers that are used for sending cmd 
 *        blocks to the stepper.  Should map to registers in the
 *        stepper block.
 */
typedef struct
{
    uint32_t pwm_t;
    uint32_t pwm_p;
    int32_t acc_incr;
    uint32_t pwm_m;
    uint32_t cmd;
    uint32_t cmd_issue;
} stepper_cmd_t;

/*
  Storage for the steppers.
*/
static stmotor_spec_t stmotor[STEPPER_NUM_MOTORS];

static struct task_struct *status_thread;

/**
 * @brief scan for status changes from the low level driver.
 * 
 * 
 * @param unused 
 * 
 * @return int 
 */
static int st_status_thread(void *unused)
{
    uint32_t buf[10];
    void *cb_user_data;
    TriggerCbFuncPtr trigger_cb;
    uint32_t motor_id;
    uint32_t buf_val;
    stmotor_trigger_cond_t trig;
    int ret;
    int trigger_order;

    while (1)
    {
        //
        // see if we need to stop.
        //
        if (kthread_should_stop())
        {
            return 0;
        }
        ret = stepper_status((char *)buf, 40);
        if (ret == 0)
        {
            // we had a timeout, simply go around and try it again.
            continue;
        }
        //      printk("Got the status, %x %x %x\n", buf[0], buf[1], buf[2]);
        //
        // got a status, call the notify stuff
        //
        switch (buf[1])
        {
            case 1:
                // the move is complete, clean up.

                buf_val = buf[2];
                motor_id = 0;
                while (buf_val > 0)
                {
                    if (buf_val & 1)
                    {
                        break;
                    }
                    motor_id++;
                    buf_val >>= 1;
                }
                stmotor[motor_id].num_triggers = 0;
                break;

            case 2: // this is a notify

                buf_val = buf[2];
                motor_id = 0;
                while (buf_val > 0)
                {
                    if (buf_val & 1)
                    {
                        break;
                    }
                    motor_id++;
                    buf_val >>= 1;
                }
       //         printk("motor id calc %d reported %d\n", motor_id, stmotor[motor_id].triggers_reported);
                trigger_order = stmotor[motor_id].trig_order[stmotor[motor_id].triggers_reported] -1;
        //        printk("num trig %d order %d\n", stmotor[motor_id].num_triggers, trigger_order);
        //        printk("total number of triggers %d\n", stmotor[motor_id].num_triggers);
                if (trigger_order < 0 && stmotor[motor_id].num_triggers > 0)
                {
                    trigger_order = stmotor[motor_id].num_triggers -1;     // we got a trigger after we started moving, handle that here.
                }
                if (trigger_order >= 0 && stmotor[motor_id].trigger[trigger_order].callback)
                {
                    trig = stmotor[motor_id].trigger[trigger_order].type;
                    stmotor[motor_id].trigger[trigger_order].type = TRIG_NULL;
                    cb_user_data = stmotor[motor_id].trigger[trigger_order].cb_user_data;
                    trigger_cb = stmotor[motor_id].trigger[trigger_order].callback;
                    stmotor[motor_id].trigger[trigger_order].callback = NULL;
                    stmotor[motor_id].trigger[trigger_order].cb_user_data = 0;
          //          printk("do trigger callback %x\n", cb_user_data);
                    trigger_cb(&stmotor[motor_id], cb_user_data);   // notify them of the event
                    if (stmot_ioctl(motor_id, GET_MTR_IDLE, 0))
                    {
                        stmot_ioctl(motor_id, DISABLE_MOTOR, 0);
                    } 
                    stmotor[motor_id].triggers_reported++;

                } else
                {
                    if (stmot_ioctl(motor_id, GET_MTR_IDLE, 0)) // if the motor block is idle, disable the motor
                    {
           //             printk("*** disable the motor\n");
                        stmot_ioctl(motor_id, DISABLE_MOTOR, 0);
                    }
                }
            break;
        }
    }
    return 0;
}

stmotor_t *smot_step_create_motor(
                       stmotor_pin_connect_t    *stmotor_connects,
                       stmotor_pin_sequence_t   *stmotor_sequences,
                       uint32_t                 number_usteps )
{
    int i, reg;

    // check input
    // 
    if (number_usteps != 1 && number_usteps != 2 && number_usteps != 4 && number_usteps != 8 && number_usteps != 16)
    {
        printk(KERN_ERR "illegal value for u-steps %d, should be one of 1,2,4,8,16\n", number_usteps);
        return NULL;    // illegal value
    }
    // initialize this entry
    memset(&stmotor[stmotor_connects->block_num], 0, sizeof(stmotor_spec_t));
    stmotor[stmotor_connects->block_num].motor_id = stmotor_connects->block_num;
    //
    // open this motor number
    //
    if (stmotor_open(stmotor_connects->block_num))
    {
        printk(KERN_ERR "Unable to open motor %d\n", stmotor_connects->block_num);
        return NULL;
    }
    //
    // set the idle setting
    //
    stmot_ioctl(stmotor_connects->block_num, SET_STEP_ENA_IDLE, 0x3f);
    //
    // set the number of u steps


    stmot_ioctl(stmotor_connects->block_num, SET_NUM_USTEPS, number_usteps);

    // setup the sequences
    // 
    stmot_ioctl(stmotor_connects->block_num, SET_SEQUENCE, (unsigned long)&stmotor_sequences->pin_seq[0]);
    //
    // now setup the output mode
    //
    reg = 0;
    for (i = 0; i < 6; i++)
    {
        switch (stmotor_connects->pin_cfg[i])
        {
            case STEP_OUTPUT_MODE_PWM1:
                reg |= MODE_PWM1<<(i*4);
                break;      
            case STEP_OUTPUT_MODE_PWM2:
                reg |= MODE_PWM2<<(i*4);
                break;
            case STEP_OUTPUT_MODE_REG0:
                reg |= MODE_REG0<<(i*4);
                break;
            case STEP_OUTPUT_MODE_REG1:
                reg |= MODE_REG1<<(i*4);
                break;
            case STEP_OUTPUT_MODE_SEQ:
                reg |= MODE_SEQ<<(i*4);
                break;
            case STEP_OUTPUT_MODE_LS:
                reg |= MODE_LS<<(i*4);
                break;
            default:
                break;
        }
    }
    stmot_ioctl(stmotor_connects->block_num, SET_MTR_CONNECTS, reg);    // set the connects

#if 0
    for (i = 0; i < stmotor[motor_id].num_triggers; i++)
    {
        printk("callback i %d callback %x\n",i, stmotor[motor_id].trigger[trigger_order].callback);
    }
#endif
    return &stmotor[stmotor_connects->block_num];       // return the handle

}

void smot_step_set_motor_deaccl_table(stmotor_t *motor_handle, uint32_t num_table_entries, uint32_t *deaccl_table)
{
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }
    smtr->deaccl_table = deaccl_table;
    smtr->deaccl_table_entries = num_table_entries;

}

void smot_step_set_motor_move_params(stmotor_t  *motor_handle,
                                     const  stmotor_move_param_t *move_params)
{
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }

    smtr->accel_select        = move_params->accel_select;
    smtr->accel_table         = move_params->accel_table;
    smtr->accel_table_entries = move_params->accel_table_entries;
    smtr->max_index           = move_params->accel_table_entries;
    smtr->accel_index         = 0;

    smtr->preholding_steps  = move_params->preholding_steps;
    smtr->postholding_steps = move_params->postholding_steps;
    smtr->cmd_packet.pwm_m  = move_params->pwm_const;
  //  printk("motor move params pwm_const %d\n", smtr->cmd_packet.pwm_m);


    stmot_ioctl(smtr->motor_id, SET_TORQUE_TABLE, (unsigned long)move_params->tvt_table);
 //   printk("Return from %s\n", __func__);
}


bool smot_step_set_speed(stmotor_t    *motor_handle,
                         uint32_t      accel_index)
{

    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle || accel_index > 100000)
    {
        return false;       // the passed in value is bogus.
    }
    smtr->accel_index = accel_index;
    return true;
}



void smot_step_cfg_line_start(stmotor_t                *motor_handle,
                              stmotor_line_start_src_t  ls_src,
                              uint16_t                  ls_incr)
{
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;

    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }
    stmot_ioctl(smtr->motor_id, SET_LINE_START_REG, (ls_incr -1)| (ls_src << 20));

}
/**
 * \brief get location of the requested trigger in the table of 
 *        registered trigger events.
 * 
 * 
 * @param smtr 
 * @param trigger 
 * 
 * @return int 
 * @retval 0 no trigger found 
 * @retval >0 location in table. 
 */
static int find_trigger(stmotor_spec_t *smtr, stmotor_trigger_cond_t trigger, uint32_t param)
{
    int i;

    for (i = 0; i < smtr->num_triggers; i++)
    {
        // is this the correct trigger?  If the trigger is state then we check param also to find the trigger.
        // logic cases type matches (A) type is TRIG_STATE(B) param matches (C) so I want
        // A(not)B  + ABC   reduces to A & ((not)B + C)
        if ((smtr->trigger[i].type == trigger) &&
            ((trigger != TRIG_STATE) ||
             (smtr->trigger[i].param == param)))
        {
            return i+1;
        }
    }
    return 0;
}
/**
 * @brief Fill out a trigger entry in the cmd_blk entry. 
 *        Increment the size of the allocation.
 * 
 * 
 * @param smtr Pointer to the motor info block
 * @param alloc_size The number of entries in the cmd_blk
 * @param cmd_blk Pointer to the stepper command block
 */
static void add_trigger_cmd(stmotor_spec_t *smtr, uint32_t *alloc_size, stepper_cmd_t *cmd_blk)
{
    cmd_blk[*alloc_size].acc_incr = 0;
    cmd_blk[*alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[*alloc_size].cmd_issue = 1;         // issue the cmd_blk.
    cmd_blk[*alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
    cmd_blk[*alloc_size].pwm_p = smtr->accel_table[0];
    cmd_blk[*alloc_size].cmd = STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 7);   // put on an illegal command for notification
    *alloc_size += 1;
}

/**
 * @brief Create the cmd block for a given set of motor 
 *        parameters
 * 
 * 
 * @param smtr The motor handle
 * @param cmd pointer to where the completed cdma blocks are to 
 *            be placed.
 * @param num_entries The number of entries in the cmd array
 * @param enable_ls Is line sync to be enabled during this move
 * @param ls_at_stepnum If enable_ls is true this is the step 
 *                      num to enable line sync
 * @param move_direction The direction of the move.
 * 
 * @return int 
 */
static int smot_create_cdma_cmd(stmotor_spec_t *smtr,  stepper_cmd_t **cmd, uint32_t *num_entries,
                                bool enable_ls, uint32_t ls_at_stepnum, stmotor_dir_t move_direction)
{
    stepper_cmd_t *cmd_blk = NULL;
    uint32_t alloc_size = 0;
    uint32_t current_step_count = 0;
    int32_t acc_table_index;
    uint32_t cmd_value;
    uint32_t ramp_count = 0;
    uint32_t num_steps;
    bool set_ls = false;
    uint32_t move_steps;
    bool first_time = true;
    bool set_trig = false;
    bool state_change = false;
    int trig_location;
    int current_trig_num = 0;   // the number of triggers that have been processed.

    smtr->motor_state = MOTOR_ACCEL;
    memset(smtr->trig_order, 0, MAX_TRIGGERS);

    move_steps = smtr->pending_steps;
    // check for pre-hold
    //
    if (smtr->preholding_steps > 0)
    {
        cmd_blk = kmalloc(sizeof(stepper_cmd_t), 0);
        if (!cmd_blk)
        {
            printk(KERN_ERR "failed malloc 0\n");
            return -1;
        }
        cmd_blk[alloc_size].acc_incr = 0;
        cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
        cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
        cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
        cmd_blk[alloc_size].pwm_p = smtr->accel_table[0];
        cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, smtr->preholding_steps) | 
                STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 0);   // hold command for steps
        smtr->preholding_steps = 0;
        alloc_size++;
    }
    //
    // see if we need a start trigger.
    //
    if ((trig_location = find_trigger(smtr, TRIG_MOTOR_START, 0)))
    {
        if (cmd_blk)    // see if we have an allocation already
        {
            // yes, realloc it
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed malloc 1\n");
                return -1;
            }
        } else
        {
            // no allocation, allowate it now.
            cmd_blk = kmalloc(sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed malloc 2\n");
                return -1;
            }
        }
        add_trigger_cmd(smtr,&alloc_size,cmd_blk);
        smtr->trig_order[current_trig_num++] = trig_location;
    }
    if((trig_location = find_trigger(smtr, TRIG_STATE, MOTOR_ACCEL)))
    {
        //
        // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
        // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
        //
        if (cmd_blk)    // see if we have an allocation already
        {
            // yes, realloc it
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed malloc 3\n");
                return -1;
            }
        } else
        {
            // no allocation, allowate it now.
            cmd_blk = kmalloc(sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed malloc 4\n");
                return -1;
            }
        }
        add_trigger_cmd(smtr,&alloc_size,cmd_blk);
        smtr->trig_order[current_trig_num++] = trig_location;
    }
    //
    // now build the rest of the table.
    acc_table_index = 0;        
    while (current_step_count < move_steps)
    {
        //     printk("In the big loop, cmd_blk %p current_step_count %d move_steps %d\n", 
        //              cmd_blk, current_step_count, move_steps);
        //     printk("acc index %d #alloc %d\n", acc_table_index, alloc_size);
        //
        // we will need to do a cmd block, allocate that here and init stuff.
        if (cmd_blk)
        {
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed 0\n");
                return -1;
            }
        } else
        {
            cmd_blk = kmalloc(sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed 1\n");
                return -1;
            }
        }
        cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
        cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
        cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
        //
        // now decide how many steps to move this time.
        num_steps = NUM_STEPS;
        if (smtr->motor_state == MOTOR_CONSTANT)        //
        {
            num_steps = 0xffff;
            if ((num_steps + current_step_count) > move_steps - ramp_count)
            {
                num_steps = move_steps - current_step_count - ramp_count;
            }
        }
        // check to see if we need to limit the next move to start the ls running.

        if (enable_ls & !set_ls)
        {
            if (ls_at_stepnum < current_step_count + num_steps)
            {
                if (ls_at_stepnum != current_step_count)
                {
                    num_steps = ls_at_stepnum - current_step_count;
                } else
                {
                    set_ls = true;
                }
            }
        }

        //
        // see if we have a trigger that needs to have steps modified.
        //
        if (!set_trig && (trig_location = find_trigger(smtr, TRIG_POSITION, 0)))
        {
            if ((smtr->trigger[trig_location-1].param >= current_step_count) && 
                (smtr->trigger[trig_location-1].param < current_step_count + num_steps))
            {
                // want to add a trigger.  this is the 2nd pass, the number of steps is 0 so we add in a trigger
                // and reallocate the cmd block.
                if((smtr->trigger[trig_location-1].param - current_step_count) == 0)
                {
                    add_trigger_cmd(smtr,&alloc_size,cmd_blk);
                    smtr->trig_order[current_trig_num++] = trig_location;
                    //
                    // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
                    // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
                    //
                    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
                    if (!cmd_blk)
                    {
                        printk(KERN_ERR "failed 2\n");
                        return -1;
                    }
                    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
                    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
                    cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
                    set_trig = true;

                } else
                {
                    // first time we have a move that contains the trigger point.  This makes the smaller move that puts us at the 
                    // exact location of the trigger.
                    num_steps = smtr->trigger[trig_location-1].param - current_step_count;
                }
            }
        }
        // now setup the move depending on what we want to do.
        if (smtr->motor_state == MOTOR_ACCEL)
        {
            //
            // See if we are on the step where we should trigger.
            if ((trig_location = find_trigger(smtr, TRIG_VELOCITY, 0)) && acc_table_index == smtr->trigger[trig_location-1].param)
            {
                //
                // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
                // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
                //
                add_trigger_cmd(smtr,&alloc_size,cmd_blk);
                smtr->trig_order[current_trig_num++] = trig_location;
                //
                // get the next block
                //
                cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
                if (!cmd_blk)
                {
                    printk(KERN_ERR "failed realloc\n");
                    return -1;
                }
                cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
                cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
                cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
                // now continue with the move we want
            }
            // do accelerate
            cmd_blk[alloc_size].pwm_p = smtr->accel_table[acc_table_index];
            // figure the increment to use.
            if (acc_table_index + 1 >= smtr->max_index)
            {
                cmd_blk[alloc_size].acc_incr = 0;
            } else
            {
                cmd_blk[alloc_size].acc_incr = (smtr->accel_table[acc_table_index] - 
                                                smtr->accel_table[acc_table_index + 1]) / num_steps;
            }
            //    printk("index %d table 0 %d table1 %d steps %d val %d\n", acc_table_index, 
            //            smtr->accel_table[acc_table_index], smtr->accel_table[acc_table_index+1],
            //           num_steps, cmd_blk[alloc_size].acc_incr);
            //
            // figure the type of command to execute.
            if (cmd_blk[alloc_size].acc_incr == 0)
            {
                cmd_value = 1;
            } else
            {
                if (smtr->accel_select == ACCEL_PWM_P)
                {
                    cmd_value = 4;      // accelerate pwm_p
                } else
                {
                    cmd_value = 2;
                }
            }
            cmd_blk[alloc_size].cmd = 0;
            cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps) | 
                    STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, cmd_value) |
                    STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, set_ls) | 
                    STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, move_direction);
            //
            // update all the pointers
            acc_table_index++;
            current_step_count += num_steps;
        } else if (smtr->motor_state == MOTOR_DECEL)
        {
            if (state_change && (trig_location = find_trigger(smtr, TRIG_STATE, MOTOR_DECEL)))
            {
                //
                // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
                // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
                //
                add_trigger_cmd(smtr,&alloc_size,cmd_blk);
                smtr->trig_order[current_trig_num++] = trig_location;
                //
                // get the next block
                //
                cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
                if (!cmd_blk)
                {
                    printk(KERN_ERR "failed realloc trigger\n");
                    return -1;
                }
                cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
                cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
                cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
                state_change = false;
                // now continue with the move we want
            }
            //
            // on deaccl we step backwards in the acc table.
            if (current_step_count + num_steps > move_steps)
            {
                num_steps = move_steps - current_step_count;
            }
            cmd_blk[alloc_size].pwm_p = smtr->accel_table[acc_table_index];
            if (acc_table_index == 0)
            {
                cmd_blk[alloc_size].acc_incr = 0;
            } else
            {
                cmd_blk[alloc_size].acc_incr = (smtr->accel_table[acc_table_index - 1] - 
                                                smtr->accel_table[acc_table_index]) / num_steps;
            }

            //      printk("index %d table 0 %d table1 %d steps %d val %d\n", acc_table_index, 
            //              smtr->accel_table[acc_table_index], smtr->accel_table[acc_table_index+1],
            //             num_steps, cmd_blk[alloc_size].acc_incr);
            if (cmd_blk[alloc_size].acc_incr == 0)
            {
                cmd_value = 1;
            } else
            {
                if (smtr->accel_select == ACCEL_PWM_P)
                {
                    cmd_value = 4;      // accelerate pwm_p
                } else
                {
                    cmd_value = 2;
                }
            }
            cmd_blk[alloc_size].cmd = 0;
            cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps) | 
                    STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, cmd_value) |
                    STMOTOR_REG_CMD_ACC_DIR_REPLACE_VAL(0, 1) | 
                    STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, set_ls) | 
                    STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, move_direction);  // deaccelerate
            // save this so if we are aborted, we can find this cmd.  
            // an abort will search for this cmd value.  It will abort till this is found, 
            // which will then cause a normal de-accel
            if (first_time)
            {
                smtr->stepper_cmd_deaccl = cmd_blk[alloc_size].cmd;
            }
            first_time = false;
            acc_table_index--;          // step back, going down the table
            current_step_count += num_steps;
        } else if (smtr->motor_state == MOTOR_CONSTANT)
        {    

            if (state_change && (trig_location = find_trigger(smtr, TRIG_STATE, MOTOR_CONSTANT)))
            {
                //
                // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
                // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
                //
                add_trigger_cmd(smtr,&alloc_size,cmd_blk);
                smtr->trig_order[current_trig_num++] = trig_location;
                //
                // get the next block
                //
                cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
                if (!cmd_blk)
                {
                    printk(KERN_ERR "failed 6\n");
                    return -1;
                }
                cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
                cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
                cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
                state_change = false;
                // now continue with the move we want
            }
            //       printk("program up constant speed num %d ramp %d move %d current %d\n", 
            //              num_steps, ramp_count, move_steps, current_step_count);
            if (smtr->deaccl_table != NULL)
            {
                cmd_blk[alloc_size].pwm_p = smtr->deaccl_table[acc_table_index];
            } else
            {
                cmd_blk[alloc_size].pwm_p = smtr->accel_table[acc_table_index];
            }
            cmd_blk[alloc_size].acc_incr = 0;
            cmd_blk[alloc_size].cmd = 0;
            cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps) | 
                    STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 1) |
                    STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, set_ls) | 
                    STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, move_direction); // step away
            if (move_steps == RUN_FOREVER_STEPS)
            {
                cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_INF_REPLACE_VAL(cmd_blk[alloc_size].cmd, 1) |
                        STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, move_direction);    // turn on inf run
                smtr->motor_state = MOTOR_DECEL;
                state_change = true;
            }
            //         printk("cmd %x\n", cmd_blk[alloc_size].cmd);
            current_step_count += num_steps;
        }


        if (smtr->accel_select != ACCEL_PWM_P)
        {
            cmd_value = cmd_blk[alloc_size].pwm_m;
            cmd_blk[alloc_size].pwm_m = cmd_blk[alloc_size].pwm_p;
            cmd_blk[alloc_size].pwm_p = cmd_value;
        }
        alloc_size++;   // ready for the next one.


        if (smtr->motor_state == MOTOR_ACCEL)
        {
            if (acc_table_index >= smtr->max_index ||
                (smtr->accel_index != 0 && acc_table_index >= smtr->accel_index))   // if we are limiting the speed do that.
            {
                // we are up to the desired speed, hold this speed.
                smtr->motor_state = MOTOR_CONSTANT;
                state_change = true;
                ramp_count = current_step_count;
                if (smtr->deaccl_table != NULL && smtr->deaccl_table_entries != 0)
                {
                    ramp_count = smtr->deaccl_table_entries * NUM_STEPS;
                }
                acc_table_index--;
            }
            if ((current_step_count * 2) > move_steps)
            {
                // have not finished the acceleration, however we need to start slowing down.
                smtr->motor_state = MOTOR_DECEL;
                state_change = true;
            }
        }
        if (smtr->motor_state != MOTOR_DECEL &&
            ramp_count &&
            (move_steps - current_step_count) <= ramp_count)
        {
            smtr->motor_state = MOTOR_DECEL;
            state_change = true;
            if (smtr->deaccl_table != NULL && smtr->deaccl_table_entries != 0)
            {
                acc_table_index = smtr->deaccl_table_entries;
            }
        }
        //
        if (acc_table_index < 0)
        {
            break;      // we are done, finish it up.
        }
    }

    //
    // always place a stop notification at the end.
    //
    {
        //      printk("Input stop trigger\n");
        //
        // We cause this trigger by putting in an illegal command into the table.  Then when the stepper gets to this
        // command it will give an illegal cmd interrupt.  That is captured and returned as the trigger.
        //
        set_trig = false;
        trig_location = find_trigger(smtr,TRIG_STATE, MOTOR_STOPPED);
        if (trig_location)
        {
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed 7\n");
                return -1;
            }
            add_trigger_cmd(smtr,&alloc_size,cmd_blk);

            smtr->trig_order[current_trig_num++] = trig_location;
            set_trig = true;
        }
        trig_location = find_trigger(smtr, TRIG_MOTOR_STOP, 0);
        if (trig_location)
        {
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed 8\n");
                return -1;
            }
            add_trigger_cmd(smtr,&alloc_size,cmd_blk);

            smtr->trig_order[current_trig_num++] = trig_location;
            set_trig = true;
        }
        if (!set_trig)
        {
            cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
            if (!cmd_blk)
            {
                printk(KERN_ERR "failed 9\n");
                return -1;
            }
            add_trigger_cmd(smtr,&alloc_size,cmd_blk);
        }

    }
    if (smtr->postholding_steps > 0)
    {
        cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
        if (!cmd_blk)
        {
            printk(KERN_ERR "failed 10\n");
            return -1;
        }
        cmd_blk[alloc_size].acc_incr = 0;
        cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
        cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd_blk.
        cmd_blk[alloc_size].pwm_m = smtr->cmd_packet.pwm_m;
        cmd_blk[alloc_size].pwm_p = smtr->accel_table[0];
        cmd_blk[alloc_size].cmd = 0;
        cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, smtr->postholding_steps) | 
                STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 0) |
                STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, move_direction);   // hold command for steps
        smtr->postholding_steps = 0;
        alloc_size++;
    }

    *cmd = cmd_blk;
    *num_entries = alloc_size;
    alloc_size = 0;
    #if 0
    while (smtr->trig_order[alloc_size] != 0)
    {
        printk("trig order item %d trig %d\n", alloc_size, smtr->trig_order[alloc_size]);
        alloc_size++;
    }
    #endif
    return 0;
}

void smot_step_move_rel(stmotor_t      *motor_handle,
                        uint32_t        move_steps,
                        stmotor_dir_t   move_direction,
                        bool            enable_ls,
                        uint32_t        ls_at_stepnum)
{
    stepper_cmd_t *cmd;
    int result, num_entries;

    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;

    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }


  //  printk("Move the motor # %d steps %d direction %d\n", smtr->motor_id, move_steps, move_direction);

    // set the direction
    stmot_ioctl(smtr->motor_id, SET_REG0_BIT, move_direction == MOVE_FORWARD);
    stmot_ioctl(smtr->motor_id, SET_REG1_BIT, 0);       // enable the motor
    // set the number of usteps.
    smtr->pending_steps = move_steps;
    //
    // build the move
    //
    result = smot_create_cdma_cmd(smtr,&cmd,&num_entries, enable_ls, ls_at_stepnum, move_direction);
    if (result)
    {
        printk(KERN_ERR "Didn't create the cmd thing\n");
        return;
    }
//    printk("%s about to start num entries %d\n", __func__, num_entries);
    // now send and start the move
    smtr->triggers_reported = 0;
    stmot_write(smtr->motor_id, (const char *)cmd, num_entries * sizeof(stepper_cmd_t));

}

void smot_step_set_location(stmotor_t *motor_handle, int32_t location)
{
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;

    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }
 //   printk("Set location to %d motor %d\n", location, smtr->motor_id);
    smtr->ref_position = location;
    stmot_ioctl(smtr->motor_id, RESET_STEP_CNT, 0);
}

int32_t smot_step_get_location(stmotor_t *motor_handle)
{
    int cur_loc;
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return -1;
    }

    cur_loc = stmot_ioctl(smtr->motor_id, GET_STEP_CNT, 0);
 //   printk("Get location motor %d step_count %d\n", smtr->motor_id, cur_loc);
    return smtr->ref_position + cur_loc;
}

void smot_step_move_abs(stmotor_t    *motor_handle,
                        int           tar_loc_steps,
                        bool          enable_ls,
                        uint32_t      ls_at_stepnum)
{
    int             cur_pos_steps;
    uint32_t        move_steps;
    uint32_t        move_direction;
//    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }

    cur_pos_steps = smot_step_get_location(motor_handle);
   // printk("Move abs Cur loc %d desired loc %d\n", cur_pos_steps, tar_loc_steps);

    /* Using the current absolute position and the desired destination position,
     * calculate a relative move
     */
    if (cur_pos_steps > tar_loc_steps)
    {
        move_direction = MOVE_REVERSE;
        move_steps = cur_pos_steps - tar_loc_steps;
    }
    else
    {
        move_direction = MOVE_FORWARD;
        move_steps = tar_loc_steps - cur_pos_steps;
    }

    smot_step_move_rel(motor_handle,
                       move_steps,
                       move_direction,
                       enable_ls,
                       ls_at_stepnum);
}



void smot_step_move_cont(stmotor_t      *motor_handle,
                         stmotor_dir_t   move_direction,
                         bool            enable_ls,
                         uint32_t        ls_at_stepnum)
{

    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    stepper_cmd_t *cmd;
    int result, num_entries;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }

    smtr->pending_steps        = RUN_FOREVER_STEPS;    
    smtr->direction            = move_direction;
    smtr->enable_ls_at_stepnum = ls_at_stepnum;
    smtr->ls_requested         = enable_ls;
    // set the direction
    stmot_ioctl(smtr->motor_id, SET_REG0_BIT, move_direction == MOVE_FORWARD);
    stmot_ioctl(smtr->motor_id, SET_REG1_BIT, 0);       // enable the motor
    // set the number of usteps.
    smtr->pending_steps = RUN_FOREVER_STEPS;
    //
    // build the move
    //
    result = smot_create_cdma_cmd(smtr,&cmd,&num_entries, enable_ls, ls_at_stepnum, move_direction);
    if (!result)
    {
        return;
    }
    // now send and start the move
    smtr->triggers_reported = 0;
    stmot_write(smtr->motor_id, (const char *)cmd, num_entries * sizeof(stepper_cmd_t));

}

void smot_step_dump(stmotor_t *motor_handle)
{

    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }
    dbg_print_reg(smtr->motor_id);
}



void smot_step_emergency_halt(stmotor_t *motor_handle)
{
    stmotor_spec_t *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }
    stmot_ioctl(smtr->motor_id, EMERGENCY_STOP, 0);

}


bool smot_step_motor_is_idle(stmotor_t *motor_handle)
{
    stmotor_spec_t  *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return false;
    }

    return stmot_ioctl(smtr->motor_id, GET_MTR_IDLE, 0);

}


void smot_step_request_motor_stop(stmotor_t *motor_handle)
{
    stmotor_spec_t  *smtr = (stmotor_spec_t *) motor_handle;
    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }

 //   printk("Request to stop the motor %d\n", smtr->motor_id);

    stmot_ioctl(smtr->motor_id, ABORT_MOTORS, smtr->stepper_cmd_deaccl);         // abort all the motors
}


void smot_step_add_trigger(stmotor_t              *motor_handle,
                           stmotor_trigger_cond_t  type,
                           uint32_t                param1,
                           TriggerCbFuncPtr        trigger_callback,
                           void                   *cb_user_data
                          )
{
    stmotor_spec_t *smtr = (stmotor_spec_t *)motor_handle;

    if (!motor_handle)
    {
        printk(KERN_ERR "Invalid handle\n");
        return;
    }

    smtr->trigger[smtr->num_triggers].type         = type;
    smtr->trigger[smtr->num_triggers].param        = param1;
    smtr->trigger[smtr->num_triggers].cb_user_data = cb_user_data;
    smtr->trigger[smtr->num_triggers].callback     = trigger_callback;
    smtr->num_triggers++;
    //    printk("set trigger callback motor %d  type %x number %d callback %x\n", smtr->motor_id,  type, smtr->num_triggers, trigger_callback);

}

/******************************************************************************
 * Stepper Public API Linux Exports
 *****************************************************************************/

EXPORT_SYMBOL(smot_step_add_trigger);
EXPORT_SYMBOL(smot_step_cfg_line_start);
EXPORT_SYMBOL(smot_step_set_speed);
EXPORT_SYMBOL(smot_step_motor_is_idle);
EXPORT_SYMBOL(smot_step_dump);
EXPORT_SYMBOL(smot_step_set_location);
EXPORT_SYMBOL(smot_step_get_location);
EXPORT_SYMBOL(smot_step_set_motor_move_params);
EXPORT_SYMBOL(smot_step_create_motor);
EXPORT_SYMBOL(smot_step_request_motor_stop);
EXPORT_SYMBOL(smot_step_emergency_halt);
EXPORT_SYMBOL(smot_step_move_rel);
EXPORT_SYMBOL(smot_step_move_abs);
EXPORT_SYMBOL(smot_step_move_cont);
EXPORT_SYMBOL(smot_step_set_motor_deaccl_table);




/******************************************************************************
 * Linux Platform API Functions
 *****************************************************************************/

/**
 * \brief Get Stepper Block Config
 *
 *  Returns hardware config for the specified stepper block.
 **/
bool smot_platform_get_block_cfg(uint32_t         block_num,
                                 volatile void  **reg_base_addr,
                                 uint32_t        *irq_num)
{
    if (block_num >= STEPPER_NUM_MOTORS)
    {
        printk(KERN_ERR "%s: invalid block id = %d\n", __func__, block_num);
        return false;
    }

    *reg_base_addr = (void *) stmot_ioctl(block_num, GET_REG_ADDR, 0);    // 
    *irq_num = stmot_ioctl(block_num, GET_MTR_IRQ, 0);

    return true;
}

static int stepper_api_platform_probe(struct platform_device *pdev)
{
    printk(KERN_ERR "%s: driver platform probe (%s)\n", __func__, ASIC_NAME_STR);
    return (0);
}

static int stepper_api_platform_remove(struct platform_device *pdev)
{
    printk(KERN_ERR "%s: driver platform remove\n", __func__);
    return (0);
}

static int stepper_api_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    printk(KERN_ERR "%s: driver platform suspend\n", __func__);
    return (0);
}

static int stepper_api_platform_resume(struct platform_device *pdev)
{
    printk(KERN_ERR "%s: driver platform resume\n", __func__);
    return(0);
}

static const struct of_device_id mrvl_stepper_api_dt_match[] = {

#if ASIC_REV <= ASIC_REV_A0
    { .compatible = "mrvl,stepper-api-a0"},
#else
    { .compatible = "mrvl,stepper-api-b0"},
#endif

    {},
};
MODULE_DEVICE_TABLE(of, mrvl_stepper_api_dt_match);

static struct platform_driver stepper_api_platform_driver =
{
    .probe   = stepper_api_platform_probe,
    .remove  = stepper_api_platform_remove,
    .suspend = stepper_api_platform_suspend,
    .resume  = stepper_api_platform_resume,
    .driver  = {
        .name  = DRIVER_NAME,
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_stepper_api_dt_match),
    }
};


static int __init stepper_api_init(void)
{
    int retval;
    char my_thread[]="st status";

    printk(KERN_ERR "%s: driver platform init (%s)\n", __func__, ASIC_NAME_STR);

    status_thread = kthread_create(st_status_thread, NULL, my_thread);
    if (status_thread)
    {
      //  printk("Started status thread\n");
        wake_up_process(status_thread);
    }

    retval = platform_driver_register(&stepper_api_platform_driver);
    if (retval)
    {
        printk(KERN_ERR "%s: error registering platform driver (%s)\n", __func__, ASIC_NAME_STR);
        return -ENXIO;
    }
    else
    {
        printk(KERN_ERR "%s: platform registration complete (%s)\n", __func__, ASIC_NAME_STR);
    }

    return 0;
}
module_init(stepper_api_init);


static void __exit stepper_api_exit(void)
{
    printk(KERN_ERR "%s: driver platform exit begun\n", __func__);

    kthread_stop(status_thread);

    platform_driver_unregister(&stepper_api_platform_driver);   

    printk(KERN_ERR "%s: platform remove complete\n", __func__);
}
module_exit(stepper_api_exit);


MODULE_AUTHOR("Copyright (c) 2015 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell Scan Stepper Driver API");

MODULE_LICENSE("GPL");
MODULE_VERSION("2015_May_4");
