/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

/**
 * \file stepper-api.h
 *
 * \brief External interface for the stepper motor driver
 *
 *  Data types and API functions for the stepper motor driver.
 *
 *  The stepper motor driver is at the bottom of the architectural stack (used by
 *  subsystems like scan and print), so we try to limit the motor drivers view to
 *  just the motor and ASIC. The motor driver should not contain scan/print
 *  mechanism details (for example, the connection of physical motor block pins).
 *
 *  Driver Notes:
 *   - This is the motor driver external interface; driver internal definitions
 *     should not be placed in this file.
 *   - This API must work for both Linux Kernel and ThreadX implementations. The
 *     driver must avoid using OS specific constructs in this interface.
 *   - This driver is not intended to be scan or print specific (although it 
 *     may provide features not used by some applications, like line starts).
 *   - This driver is tailored to work with 6 pin motor blocks
 *   - This driver only supports micro-stepping (8 usteps per step)
 *
 *  Notes on Line Starts:
 *    Line starts are typically used in scan applications to provide a 'start of
 *    scan line' signal to the scan subsystem. Coupling the scan line start signal
 *    to the motor block allows scan lines to be captured in sync with actual motor
 *    movements (can eliminate ripple and enable scanning during motor acceleration).
 *    Applications other than scan can probably ignore this feature (simply set 
 *    enable_ls to false in smot_step_move_*).
 *    
 */

#ifndef INC_STEPPER_API_H
#define INC_STEPPER_API_H

#define INVALID_STEP_MOTOR      NULL
#define MOVE_FORWARD            1
#define MOVE_REVERSE            0

// Maximum number of stepper motors supported by this driver.
#define STEPPER_NUM_MOTORS    6

/**
 *  \brief Stepper motor instance type
 *
 *  Client motor instance (handle) returned by create_motor, then used for all
 *  subsequent motor API calls. Note that this type is intentionally opaque in
 *  the motor external interface.
 **/
typedef struct stmotor_s stmotor_t;


/**
 *  \brief Stepper motor block ID
 *
 *  Identifies the ASIC motor block to be used when creating a motor instance 
 *  (for example, STEP_MTR_BLK_0). These values need to come from the platform
 *  code (for example, the dtsi file in Linux). This maps a virtual motor 
 *  instance (see create_motor) to a physical hardware motor block.
 **/
typedef uint32_t stmotor_block_id_t;


/**
 *  \brief Stepper motor pin output mode selection
 *
 *  Selects the signal source for ASIC motor output pins; each pin (0 to 5) in 
 *  stmotor_pin_connect_t will be configured with one of these values.
 **/
typedef enum {
    STEP_OUTPUT_MODE_PWM1 = 0,         // PWM_Ta
    STEP_OUTPUT_MODE_PWM2,             // PWM_Tb
    STEP_OUTPUT_MODE_REG0,             // Single bit output register 0  
    STEP_OUTPUT_MODE_REG1,             // Single bit output register 1
    STEP_OUTPUT_MODE_SEQ,              // Sequence
    STEP_OUTPUT_MODE_LS                // DEBUG: exposes internal line start signal
} stmotor_pin_signal_t;

#define NO_NENABLE_PIN_SOURCE  -1

/**
 *  \brief Stepper motor connections
 *
 *  Table used to specify how the 6 pins of the motor block are actually connected
 *  to the mechanism hardware.  These signals may be connected to the motor block
 *  in a variety of ways, the motor driver needs to be told.
 **/
typedef struct stmotor_connect_s
{
    stmotor_block_id_t   block_num;    // ASIC stepper motor block number
    int                  nEnableSrc;   // ASIC nEnable Pin Source/Mode
    stmotor_pin_signal_t pin_cfg[6];   // ST0 - ST5 pin configuration
} stmotor_pin_connect_t;


/**
 *  \brief Stepper motor output sequences
 *
 *  Table used to specify the stepper controller output sequences for each of the
 *  the 6 motor block pins. Only used for pins configured as STEP_OUTPUT_MODE_SEQ.
 *  There is a difference between rev A and the rest.  Rev A
 *  only supports up to 8 micro-steps, rev B and higher support
 *  up to 16 micro-steps.  For rev A the pin_seq means the
 *  following:
 *  offset  seq Reg
 *  0          0
 *  1          1
 *  2          2
 *  3          3
 *  4          4
 *  5          5
 * 
 *  For revB+ the following is the mapping
 *  offset  seq register
 *  0       0 low
 *  1       1 low
 *  2       2 low
 *  3       3 low
 *  4       4 low
 *  5       5 low
 *  6       0 high
 *  7       1 high
 *  8       2 high
 *  9       3 high
 *  10      4 high
 *  11      5 high
 **/
typedef struct stmotor_sequence_s
{

    uint32_t        pin_seq[12];
} stmotor_pin_sequence_t;


/**
 *  \brief Stepper motor acceleration method
 *
 *  Acceleration can be done using either PWM_M or PWM_P values. This is only
 *  important if you are using line starts, which can be based on either the PWM_M
 *  or the PWM_P clock. If line starts are enabled, then PWM_M should be used to
 *  accelerate so that the line starts respond according to the acceleration.
 **/
typedef enum {
    ACCEL_PWM_P = 0,         // Accel with PWM_P, keep PWM_M constant
    ACCEL_PWM_M              // Accel with PWM_M, keep PWM_P constant
} stmotor_accel_t;


/**
 *  \brief Stepper motor line start source
 *  
 *  Used to configure the PWM clock source of the line start signal.
 **/
typedef enum {
    LS_SRC_STEP = 0,        // Line starts from PWM_P * PWM_M sysclks
    LS_SRC_PWM_M            // Line starts from PWM_M sysclks
} stmotor_line_start_src_t;


/**
 *  \brief Stepper motor torque vector table
 *
 *  The TVT consists of a set of 9 possible PWM_T duty cycles.
 **/
typedef struct {

    uint16_t    tvt[17];
} stmotor_TVT_t;


/**
 *  \brief Motor move parameters
 *
 *  This structure contains all the parameters for a specific motor move.  The
 *  client may maintain a table of these entries in a platform specific motor/mech
 *  table, but a single entry is provided to the motor driver for each move operation.
 *
 *  \notes
 *  See the ASIC programmers guide for a detailed discussion of acceleration schemes.
 *  A sample scan motor table entry might look something like the following:
 *
 *    stmotor_TVT_t fb_300M_tvt = { {1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000} };
 *
 *    // Note that smaller accel numbers are faster!
 *    uint32_t fb_accel8u_300M[200] = {
 *       270,267,265,262,260,257,255,253,250,248,245,243,241,238,236,234,232,230,227,225,
 *       223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,190,188,186,
 *       184,182,181,179,177,176,174,172,171,169,167,166,164,163,161,160,158,157,155,154,
 *       152,151,149,148,146,145,144,142,141,140,138,137,136,134,133,132,131,129,128,127,
 *       126,125,123,122,121,120,119,118,116,115,114,113,112,111,110,109,108,107,106,105,
 *       104,103,102,101,100,99,98,97,96,95,94,93,93,92,91,90,89,88,87,87,
 *       86,85,84,83,83,82,81,80,79,79,78,77,76,76,75,74,74,73,72,72,
 *       71,70,70,69,68,68,67,66,66,65,64,64,63,63,62,61,61,60,60,59,
 *       59,58,57,57,56,56,55,55,54,54,53,53,52,52,51,51,50,50,49,49,
 *       48,48,47,47,47,46,46,45,45,44,44,44,43,43,42,42,42,41,41,40};
 *
 *    stmotor_param_t motor_entry = {ACCEL_PWM_M, 58, 64, 64, fb_300M_tvt, 200, fb_accel8u_300M};
 *
 **/
typedef struct {
    stmotor_accel_t  accel_select;          // Accelerate using either PWM_M or PWM_P
    uint32_t         pwm_const;             // Constant pwm value 
                                            //   (either PWM_P or PWM_M, see stmotor_accel_t)
    uint16_t         preholding_steps;      // Holding steps before move
    uint16_t         postholding_steps;     // Holding steps after move
    stmotor_TVT_t    *tvt_table;            // Torque vector table
    uint32_t         accel_table_entries;   // Number of values in the accel_table
    uint32_t         *accel_table;          // Pointer to array of divisor values used to accelerate
                                            //   (either PWM_P or PWM_M, see stmotor_accel_t)
} stmotor_move_param_t;


/**
 *  \brief Motor move direction type
 *
 *  Used to specify the direction of a motor move.  Note that the motor code
 *  does not define mech specific coordinate systems/polarities.
 **/
typedef enum {
    MOT_REVERSE = 0,           // Move causes decreasing location counts
    MOT_FORWARD                // Move causes increasing location counts
} stmotor_dir_t;


/**
 *  \brief Trigger callback function pointer
 *
 *  Function pointer to a motor trigger callback routine.
 **/
typedef void (*TriggerCbFuncPtr)(stmotor_t *motor_handle, void *cb_user_data);


/**
 *  \brief Internal stepper motor state
 *
 *  Internal state of a motor; type is exposed to support setting of TRIG_STATE
 *  triggers.
 **/
typedef enum {
    MOTOR_STOPPED,               // Motor is stopped/idle
    MOTOR_ACCEL,                 // Accelerating to target speed
    MOTOR_CONSTANT,              // Running at target velocity
    MOTOR_DECEL,                 // Decelerating to target speed
} stmotor_state_t;


/**
 *  \brief Motor trigger conditions
 *
 *  Conditions/events that can be used to generate triggers.
 **/
typedef enum {
    TRIG_NULL,                   // No trigger set (clear trigger)
    TRIG_POSITION,               // Trigger on location value
    TRIG_VELOCITY,               // Trigger on speed
    TRIG_MOTOR_STOP,             // Trigger after complete motor stop
    TRIG_MOTOR_START,            // Trigger on motor start
    TRIG_STATE                   // Trigger on specific motor state
} stmotor_trigger_cond_t;


/**
 *  \brief (API) Set up a new motor
 *
 *  API function to create a new motor instance. This routine must only be called
 *  once for each physical motor to be supported by the motor driver.
 *
 *  \param[in] stmtr_connnects  Pointer to motor pin connection struct
 *  \param[in] stmtr_sequence   Pointer to motor pin sequence struct
 *  \param[in] number_usteps    Number of micro steps in this
 *        motor, allowed values 1,2,4,8,16
 *
 *  \return stmotor_t *
 *  \retval NULL  Motor could not be created
 *  \retval Valid client motor handle
 **/
stmotor_t *smot_step_create_motor(
                       stmotor_pin_connect_t    *stmotor_connects,
                       stmotor_pin_sequence_t   *stmotor_sequences,
                       uint32_t                 number_usteps );


/**
 *  \brief (API) Set motor move parameters
 *
 *  API function to set motor move parameters.  Each platform will typically
 *  support several different motor move profiles, the details must be given
 *  to the motor driver using this function BEFORE:
 *   - doing any motor move with smot_step_move_*
 *   - adjusting the speed with smot_step_set_speed
 *
 *  \param[in] motor_handle  Client motor handle
 *  \param[in] move_params   Const pointer to motor parameter struct
 **/
void smot_step_set_motor_move_params(stmotor_t              *motor_handle,
                                     const stmotor_move_param_t *move_params);
/**
 * @brief Specify a table to be used for the deacceleration of 
 *        the move.
 * This is an optional command.  If this is not used the same 
 * table used for acceleration will be used for deacceleration. 
 * This gives the driver a table to be used for the motor 
 * deacceleration.  The format of the table is the same as the 
 * accelerate table, and the order is slow to fast. 
 * 
 * 
 * @param motor_handle returned from smot_step_create_motor
 * @param num_table_entries Number of entries in the table
 * @param deaccl_table The table itself.
 */
void smot_step_set_motor_deaccl_table(stmotor_t *motor_handle,
                                      uint32_t num_table_entries, 
                                      uint32_t *deaccl_table);
/**
 *  \brief (API) Set motor speed
 *
 *  API function to adjust the motor speed specified in the motor parameters. By 
 *  default, the last entry in the acceleration table will determine how fast we
 *  will move.
 * 
 *  Calling this before the move starts: 
 *    Specifying an accel_index less than the number of entries in the acceleration
 *    table will cut the acceleration short, resulting in a slower speed.
 *
 *  Calling this after the move starts: 
 *    Will result in the motor either speeding up or slowing down, depending on the
 *    current steady state index into the accel table.  Calling
 *    this after the move starts is not supported in the cdma
 *    stepper driver.
 *
 *  \warning
 *  smot_step_set_motor_move_params must be called before this function is used.
 *
 *  \param[in] motor_handle  Client motor handle
 *  \param[in] accel_index   New target acceleration table index
 *
 *  \return BOOL
 *  \retval TRUE     New accel_index has been applied
 *  \retval FALSE    New accel_index invalid, ignored
 **/
bool smot_step_set_speed(stmotor_t    *motor_handle,
                         uint32_t      accel_index);


/**
 * \brief (API) Configure stepper motor line start
 *
 *  Configures the line start signal; values only used when enable_ls is set by
 *  a smot_step_move_* function.
 *
 *  \param[in] motor_handle Client motor handle
 *  \param[in] ls_src       PWM clock source for line start signal
 *  \param[in] ls_incr      Number of PWM clocks between line starts
 **/
void smot_step_cfg_line_start(stmotor_t                *motor_handle,
                              stmotor_line_start_src_t  ls_src,
                              uint16_t                  ls_incr);


/**
 *  \brief (API) Move the stepper motor relative to the current location
 *
 *  Moves the motor the specified number of steps in the specified direction.
 *
 *  \warning
 *   - smot_step_set_motor_move_params must be called before this function
 *   - if enable_ls is true, smot_step_cfg_line_start should be called before
 *     this function
 *
 *  \param[in] motor_handle   Client motor handle (from create_motor)
 *  \param[in] move_steps     Number of steps to move
 *  \param[in] move_direction Direction to move
 *  \param[in] enable_ls      true to enable line starts
 *  \param[in] ls_at_stepnum  Generate ls after taking the specified number of steps
 *                            (only used if enable_ls is true)
 **/
void smot_step_move_rel(stmotor_t      *motor_handle,
                        uint32_t        move_steps,
                        stmotor_dir_t   move_direction,
                        bool            enable_ls,
                        uint32_t        ls_at_stepnum);


/**
 *  \brief (API) Move the stepper motor to an absolute location
 *
 *  Moves the motor to the specified location (in steps, based on the current motor
 *  location). See smot_step_set_location and smot_step_get_location.
 *
 *  \warning
 *   - smot_step_set_motor_move_params must be called before this function
 *   - if enable_ls is true, smot_step_cfg_line_start should be called before
 *     this function
 *
 *  \param[in] motor_handle   Client motor handle (from create_motor)
 *  \param[in] tar_loc_steps  Target location in steps
 *  \param[in] enable_ls      true to enable line starts
 *  \param[in] ls_at_stepnum  Generate ls after taking the specified number of steps
 *                            (only used if enable_ls is true)
 **/
void smot_step_move_abs(stmotor_t    *motor_handle,
                        int           tar_loc_steps,
                        bool          enable_ls,
                        uint32_t      ls_at_stepnum);


/**
 *  \brief (API) Continuous stepper motor move
 *
 *  Moves the motor until it is told to stop (see smot_step_request_motor_stop).
 *
 *  \warning
 *   - smot_step_set_motor_move_params must be called before this function
 *   - if enable_ls is true, smot_step_cfg_line_start should be called before
 *     this function
 *   - Long moves may cause loss of motor location if location counter rolls over
 *
 *  \param[in] motor_handle   Client motor handle (from create_motor)
 *  \param[in] move_direction Direction to move
 *  \param[in] enable_ls      true to enable line starts
 *  \param[in] ls_at_stepnum  Generate ls after taking the specified number of steps
 *                            (only used if enable_ls is true)
 **/
void smot_step_move_cont(stmotor_t      *motor_handle,
                         stmotor_dir_t   move_direction,
                         bool            enable_ls,
                         uint32_t        ls_at_stepnum);


/**
 *  \brief (API) Initiate 'gentle' motor stop
 *
 * Request a gentle motor stop:
 *  - executes a ramp down (as defined by the stmotor_param_t acceleration table)
 *  - executes any programmed post hold steps
 *
 *  \param[in] motor_handle    Client motor handle (from create_motor)
 **/
void smot_step_request_motor_stop(stmotor_t *motor_handle);


/**
 *  \brief (API) Initiate an emergency motor stop
 *
 * Request an 'emergency' motor stop:
 *  - does NOT execute a ramp down
 *  - executes any programmed post hold steps
 *
 * \warning
 * An emergency halt can result is a loss of motor location accuracy.
 *
 *  \param[in] motor_handle    Client motor handle (from create_motor)
 **/
void smot_step_emergency_halt(stmotor_t  *motor_handle);


/**
 *  \brief (API) Check for motor idle (stopped)
 *
 *  Motor driver clients can use this routine to wait for the motor to stop after
 *  requesting a stop (the client will have to poll this function in some sort of
 *  thread friendly way).
 *
 *  \warning
 *  This routine may not accurately detect the motor state after an 'emergency'
 *  stop; in that case the motor may still be moving/coasting after the stop 
 *  state has been set.
 *
 *  \param[in] motor_handle Client motor handle
 *
 *  \return BOOL
 *  \retval TRUE     Motor is in the idle (stopped) state
 *  \retval FALSE    Motor is not idle
 **/
bool smot_step_motor_is_idle(stmotor_t *motor_handle);


/**
 * \brief Set Stepper Motor location
 *
 * Sets the value of the current motor location.  Subsequent moves will increment
 * and decrement the location as needed.
 *
 *  \warning
 *  It is advisable to only call this function while the motor is stopped (idle)
 *
 *  \param[in] motor_handle Client motor handle 
 *  \param[in] location     Desired location value, in steps
 **/
void smot_step_set_location(stmotor_t *motor_handle, int32_t location);


/**
 * \brief Get Stepper Motor location
 *
 *  Gets the value of the current motor location in steps.
 *
 *  \param[in] motor_handle Client motor handle 
 *
 *  \return int32_t 
 *  \retval Current motor location in steps
 **/
int32_t smot_step_get_location(stmotor_t *motor_handle);


/**
 *  \brief (API) Add a motor event trigger
 *
 *  Allows the client to register for a callback when a specific motor
 *  event or condition occurs.
 *
 *  \notes
 *  - triggers are one shot, they go away after they fire
 *  - triggers apply to the current move (or next move if we are stopped), after
 *    a move is done 'unused' triggers are cleared
 *  - Up to 20 triggers may be set on a move.
 *  - If 2 triggers are next to each other.  For example
 *    TRIGGER_STOPPED and TRIGGER_STATE (MOTOR_STOPPED) one of
 *    the triggers may not occur because of interrupt overlap.
 *  - Only one type of a trigger at a time is allowed, ie 2
 *    TRIGGER_POSITION are not allowed.  If 2 are given only the
 *    first registered will happen.
 *
 *  \warning
 *  The provided callback function will be called in interrupt context!
 *
 *  \param[in] motor_handle Client motor handle
 *  \param[in] type         Type of trigger:
 *                            (POSITION, VELOCITY, STATE, etc)
 *  \param[in] param1       Trigger type specific value:
 *                            POSITION: location (in steps) value to cause callback
 *                            STATE:    motor state to cause callback (cast enum stmotor_state_t)
 *                            VELOCITY: motor velocity to cause callback (index in accel table)
 *  \param[in] function_ptr Address of callback function
 *  \param[in] cb_user_data Pointer to user supplied callback data (opaque to driver)
 **/
void smot_step_add_trigger( stmotor_t                *motor_handle, 
                            stmotor_trigger_cond_t   type,
                            uint32_t                 param1,
                            TriggerCbFuncPtr         trigger_callback,
                            void                     *cb_user_data );

/**
 *  \brief (API) Dump stepper motor registers
 *
 *  \param[in] motor_handle Client motor handle
 **/
void smot_step_dump(stmotor_t *motor_handle);

#endif // INC_STEPPER_API_H

