/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

/**
 * \file stepper_cdma.h 
 * This file provides the api for the low level cdma stepper 
 * motor driver. This driver works by taking a list of commands 
 * for the stepper and using the cdma block sending the commands
 * one at a time to the stepper till complete. 
 * This driver provides the following: 
 * 1,  Automatic operation.  Complete move can be programmed and 
 * move will complete without any interrupts or user 
 * intervention. 
 * 2.  Status reporting.  Allows setting notifications at any 
 * location during the move. 
 * 3.  Normal cdma writes to the stepper by default cover the 
 * registers PWM_T to CMD_ISSUE.  This can be changed to cover 
 * any number of registers.  So for example you can also extend 
 * the cdma to cover the OUT_MODE register if desired. 
 * 4.  Syncronization of  motors.  This allows the sync of 
 * running many motors.  This is accomplished by starting each 
 * command block with a hold, and after all motors are 
 * programmed, calling the abort all which will allow them to 
 * start at the same time. 
 * 5.  2 interfaces.  Supplies a kernel and uio interface. 
 * 6.  Supports line sync for scanning. 
 * 7.  Supports many types of stops. 
 * 8.  Provides an ioctl interface to allow config and control 
 * of the steppers. 
 *  
 */
#ifndef STEPPER_CDMA_H
#define STEPPER_CDMA_H

typedef int stmotor_id_t;

/**
 * @brief Call this to initialize a given motor block.
 * 
 * 
 * @param motor_num The number of the motor to open
 * 
 * @return int 
 * @retval 0 successful 
 * @retval -1 motor number not valid 
 */
int stmotor_open(uint32_t motor_num);

/**
 * @brief Do the ioctl commands shown below.
 * 
 * 
 * @param motor_id The motor number to do the ioctl on
 * @param cmd The ioctl command, see the enum list below.
 * @param args Depends on the ioctl command, see below
 * 
 * @return long Depends on the ioctl command, see below.
 */
long stmot_ioctl(uint32_t motor_id, unsigned int cmd, unsigned long args);

/**
 * @brief Send in a command block to send to the stepper. 
 * This command will start running the command block immediatly. 
 *  If the motor is already running then this will append these
 *  commands to the end of the current list.  This will return
 *  when the commands are all appended on the list.
 * 
 * 
 * @param motor_id The motor number to which the commands are to 
 *                 be appended
 * @param buf The buffer with the command list.  This is not 
 *            retained by the driver.  The calling routine can
 *            do anything it wishes with this buffer after
 *            returning from this call.
 * @param count The size of buf in bytes.  This should be 
 *              divisable by the length of the cdma cmd that is
 *              being sent to the stepper block.  For the
 *              default case this should be divisable by 24.
 *              This may be changed by the ioctl SET_DMA_WRAP.
 * 
 * @return ssize_t The number queued.  Should match count.
 */
ssize_t stmot_write(int motor_id, const char *buf, size_t count);

/**
 * @brief Get the status of the motor move.  Blocks till status 
 *        is available, when it returns.  If status takes more
 *        than 1 second, this returns with a timeout error.
 * The data that is returned is a series of 4 byte words, word 0 
 * is the length of the rest of the status in bytes.  Word 1 is 
 * the status code.  Word 2 is data specific to a status 
 * code 
 * Current codes are: 
 * ST_COMPLETE = 1      The commanded move is complete. This is 
 * when the last CDMA block has been sent to the motor.  Move 
 * may still be executing. Last word is the motor block number 
 * that is complete 
 * ST_NOTIFY = 2 A notification has occurred.  word 2 is a bit 
 * mask of which motor block notified.  bit 0 = stepper block 0, 
 * bit 1 = stepper block 1 etc. 
 * 
 * 
 * @param buf Address of buffer to hold the data
 * @param count Size of the data buffer in bytes
 * 
 * @return int 
 * @retval >0 Amount of data returned by call in bytes. 
 * @retval = 0 Timeout of status. 
 */
int stepper_status(char *buf, ssize_t count);

void dbg_print_reg(uint32_t motor_id);

/**
 * List of the ioctls supported by the driver.
 */
typedef enum {
    SET_TORQUE_TABLE,       ///< Pass in a torque table to put into the stepper block. args=pointer to the torque table, 
    SET_MTR_CONNECTS,       ///< Pass in a 32 bit word to set the motor connect registers, args = connect value to store.
    SET_REG0_BIT,           ///< set or clear the reg0 bit in the stepper registers, args = 0 or 1
    SET_REG1_BIT,           ///< set or clear the reg1 bit in the stepper registers, args = 0 or 1
    SET_SEQUENCE,           ///< set the sequence registers in the stepper block. args=pointer to the sequence table to store
    SET_DMA_WRAP,           ///< How many bytes to cdma before wrapping to the beginning.  Defaults to 24 bytes.  Must be divisible by 4. args=wrap value.
    SET_DMA_DESTADDR,       ///< byte offset from the beginning of the stepper block to start the cdma running.  Defaults to 0x4c.  Must be divisible by 4. args=offset value
    SET_STEP_ENA_IDLE,      ///< Set the output line idle settings. args=value to put in the idle register
    SET_NUM_USTEPS,         ///< Set the number of u-steps / step. args=number of u-steps
    START_MOTORS,           ///< Preforms an abort on all requested motors.  Allows sync of motor starts.  Motors should be stopped on a hold for this to work. args=bitmask, 1bit/mtr, 0xffffff, does all
    SET_LINE_START_REG,     ///< Set the line start register in the stepper block. args=the line start register value
    GET_REG_ADDR,           ///< Get the address of the start of the stepper block registers, returns the address of the block 
    GET_MTR_IRQ,            ///< Get the irq of the stepper. returns = the irq number
    GET_MTR_IDLE,           ///< returns true if the motor is idle, false otherwise.
    EMERGENCY_STOP,         ///< shut down the cdma and stop the motor block asap. args=don't care
    DISABLE_MOTOR,          ///< turn this motor block off.  Takes power from the motor windings. args=don't care
    ABORT_MOTORS,           ///< This will abort the motor commands till it finds the passed in command.  Then it returns.  
                            /// This is used when it is desired to skip part of a programmed sequence and start at another.  For example
                            /// this would allow the programming of a accelerate, cruise for a long time, de-accelerate.  When a stop is requested
                            /// this will kill all commands from the current one to the start of the de-accelerate profile.  args=cmd to find, abort all before this
    RESET_STEP_CNT,         ///< Reset the step count register in the stepper block. args=don't care
    GET_STEP_CNT            ///< returns the value of the step count register in the stepper block.
} stioctl_cmds_t;

/**
 * debug commands:  When this driver starts the following files 
 * are created in each stepper directory 
 * (/sys/devices/f815<0-5>000_stepper) for use in debugging 
 * -- dump halt help ioctl move pwm_m stop 
 *  
 * help -- Doing a cat of this file will give a short debug help 
 *          list
 * dump -- Doing a cat of this file will display all motor 
 *          registers
 * halt -- Writting anything to this file will cause an 
 *          emergency stop of the motor.
 * ioctl --Writing to this file will attempt to do an ioctl to 
 *          this motor.  The format is echo <cmd> <args> >ioctl.
 *          cmd is the enum number relative to the ioctl.  See
 *          the structure above for stioctl_cmds_t.  args
 *          depends on the cmd. Again see the documentation in
 *          the stioctl_cmds_t structure.
 * pwm_m --This sets the pwm_m value to use in the commands.  If 
 *          this is set to 0 or not changed the debug system
 *          will figure out a good value for pwm_m.
 * stop --  This causes the current transfer to transistion to 
 *          the de-accelerate phase of the move and continue
 *          there.  Normal stop.
 * move --  Parameters: <move len> <direction> <speed> <accel> 
 *          Move len is the length of the move in steps.
 *          direction is what the direction bit in the cmd
 *          register will be set to.  May not mean move
 *          direction.  speed is the desired top speed in
 *          steps/sec.  And accel is in steps/sec/sec.  The move
 *          command will build an accelerate and de-accelerate
 *          profile for the move and make the move.  The accel
 *          and deaccel profiles are contained within the move 
 *          steps.  The acceleration profile is a 3 step ramp,
 *          approx straight line to the desired speed.
 *  
 *  
 *          As an example of how to use this:
 *          echo "<cmd> <arg>" >ioctl       // to configure the
 *                                      sequence tables etc for
 *                                      this motor
 *          cat dump        // list of all the registers
 *          echo 60 >pwm_m  // set my pwm-m value
 *          echo "10000 0 1000 500" >move  // Move 10000 steps
 *                                          in the direction 0
 *                                          with a max speed of
 *                                          1000steps/second,
 *                                          with an acceleration
 *                                          of 500 steps/sec/sec
 *  
 * 
 */

#endif
