/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file stepper_cdma.c
 *
 * \brief Stepper Servo Scan Motor Driver 
 * This implements a stepper driver for use in both kernal and
 * user space.  This does cdma xfers to the motor driver.  So
 * the entire transfer is constructed before start and then the
 * transfer is sent to the motor driver one command at a time
 * using the cdma block.  This will de-couple the stepper
 * operation from any real-time requirements.
 *
 */
#ifndef __KERNEL__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <posix_ostools.h>
#include "scos.h"
#include "ATypes.h"
#include "memAPI.h"
#include "interrupt_api.h"
#include "regAddrs.h"
#include "intnums.h"
#include "scanalyzer.h"
#include "lassert.h"
#include "scancore.h"
#include "utils.h"
#include "scantypes.h"
#include "hwconfig_api.h"
#include "scandbg.h"
#include "scanhw.h"
#include "scan.h"
#include "scanif.h"
#include "cpu_api.h"
#include "debug.h"
#include "ostools.h"
#include "STMotor_regheaders.h"
#include "mtr6pin.h"
#include "cdma_api.h" //MRVL-Harris for stepper cdma
#else // __KERNEL__
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/kfifo.h>
#include <linux/mutex.h>
#include <linux/kernel.h>
#include <linux/dma-mapping.h>
#include <linux/dmaengine.h>
#include <linux/scatterlist.h>
#include <linux/of.h>
#include <linux/platform_data/mv61_cdma.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/list.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mach/irq.h>
#include <asm/uaccess.h>

#include "STMotor_regheaders.h"
#include "stepper_cdma.h"

// status values
#define ST_NOTIFY 2
#define ST_COMPLETE 1

#define INVALID_STEP_MOTOR_ID        -1

#define DRIVER_NAME  "g2_stepper"
#define STATUS_FNAME "stepper_status"

#if ASIC_REV <= ASIC_REV_A0
    #define ASIC_NAME_STR "a0"
#else
    #define ASIC_NAME_STR "b0"
#endif


#define STEPPER_NUM_MOTORS  6

#endif // __KERNEL__

static void smot_count_reset(stmotor_id_t motor_id, bool value);
void stmotor_close(int motor_id);

#ifdef DEBUG
#define dbg1(args...) printk(args)
//#define dbg1(args...)
//#define dbg2(args...) printk(args)
#define dbg2(args...)
#else
#define dbg1(args...)
#define dbg2(args...)
#endif
/**
 * \brief format of the linked list entry in the cdma list. 
 * Each write to a stepper file is a list of cdma commands. 
 * There can be any number of commands.  This is the linked list 
 * entry for the cdma commands that are passed on.  After the 
 * complete interrupt occurs, this is used to know what to free.
 */
typedef struct linked_list_format_s
{
    struct list_head    list_entry;                 ///< the linked list structure.
    uint32_t            *data_buffer;               ///< the data buffer holding the list.
    struct sg_table     *sg_table;                       ///< The scatter-gather list returned from the cdma driver.

} linked_list_format_t;

//
// define the structure for holding the motor parameters
typedef struct stmotor_params_s
{
    volatile STMOTOR_REG_REGS_t   *stmotor_regs;               ///>stepper motor control register address
    volatile STMOTOR_REG_REGS_t   *phy_stmotor_regs;           ///> physical address of stmotor regs.  Used by cdma
    uint32_t                       size_of_registers;          ///> size of the register region
    uint32_t                       stmotor_irq_num;            ///> IRQ number for this motor
    bool                           stmotor_cdma_enable;         ///< when true we have a cdma channel for this stepper.
    uint32_t                       stmotor_cdma_num;            ///< the number of the cdma channel for this stepper.
#ifndef __KERNEL__
    cdma_handle_t                 *stmotor_cdma_handle;
    cdma_chain_cmds_t             *stmotor_cdma_chain;
#else
    struct dma_chan               *stmotor_dma_chan;            ///< the return from the dmaengine_request_channel, the cdma handle.  
    struct mv61_dma_slave          tx_dma_slave;                ///< this is the cdma block configuration registers.  When a channel is obtained, this is passed in.
    struct list_head               list_head;                   ///< linked list holding cdma data.  
    struct mutex               sem;                        ///< mutex to protect the above linked list.
#endif
    char   misc_name[16];                                       ///< the name of the device file, 
    struct miscdevice miscdev;                                  ///< the device file handle.
} stmotor_params_t;

static stmotor_params_t *stmotors    =  NULL;
static uint32_t stmotor_active = 0;                 ///< bit mask of the currently active motors.
static struct miscdevice status_device;             ///< the device file handle for the status device.
static wait_queue_head_t wait_queue;                ///< used to notify of completion of requests.
static int int_value[10]={0,0,0,0,0,0,0,0,0,0};        ///< last interupt status report
static int int_value_thread[10]={0,0,0,0,0,0,0,0,0,0};        ///< last thread status report

static int8_t  st_max_motors         = -1;
static int8_t  stmotors_created      =  0;


#ifndef __KERNEL__
TX_SEMAPHORE                    stmotor_cdma_semaphore; //MRVL-Harris for stepper cdma
#endif


#if ASIC_REV <= ASIC_REV_A0
  #define TVT_89  TVT_8
  #define STMOTOR_REG_TVT_89_TVT_8_REPLACE_VAL STMOTOR_REG_TVT_8_TVT_8_REPLACE_VAL
#endif

extern int   dbg_printf(const char *pFormat, ...);

#define step_func_enter()             dbg1("STEP@%s: enter\n", __FUNCTION__)
#define step_func_leave()  
#define step_func_infor(msg, args...)
#define step_func_error(msg, args...) dbg1("STEP@%s.%d: " msg, __FUNCTION__, __LINE__, ##args)
#define step_func_debug(msg, args...) dbg1("STEP@%s.%d: " msg, __FUNCTION__, __LINE__, ##args)

static bool smot_step_debug_in_interrupt_enable = false; //only enable this when you want to see some debug message in interrupt

#define ASSERT(cond)  BUG_ON(!(cond))

#define XASSERT(cond,value) \
    do { \
        if( !(cond) ) { \
            printk( KERN_CRIT "XASSERT failed at %s %d value=0x%x\n", \
                    __FILE__, __LINE__, value );\
            panic("BUG!"); \
        } \
    } while(0); 



static void smot_step_set_torque_vector_table(stmotor_id_t motor_id, uint16_t tvt[9])
{
    stmotors[motor_id].stmotor_regs->TVT_01 = STMOTOR_REG_TVT_01_TVT_0_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_01, tvt[0]);
    stmotors[motor_id].stmotor_regs->TVT_01 = STMOTOR_REG_TVT_01_TVT_1_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_01, tvt[1]);
    stmotors[motor_id].stmotor_regs->TVT_23 = STMOTOR_REG_TVT_23_TVT_2_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_23, tvt[2]);
    stmotors[motor_id].stmotor_regs->TVT_23 = STMOTOR_REG_TVT_23_TVT_3_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_23, tvt[3]);
    stmotors[motor_id].stmotor_regs->TVT_45 = STMOTOR_REG_TVT_45_TVT_4_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_45, tvt[4]);
    stmotors[motor_id].stmotor_regs->TVT_45 = STMOTOR_REG_TVT_45_TVT_5_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_45, tvt[5]);
    stmotors[motor_id].stmotor_regs->TVT_67 = STMOTOR_REG_TVT_67_TVT_6_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_67, tvt[6]);
    stmotors[motor_id].stmotor_regs->TVT_67 = STMOTOR_REG_TVT_67_TVT_7_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_67, tvt[7]);
    stmotors[motor_id].stmotor_regs->TVT_89  = STMOTOR_REG_TVT_89_TVT_8_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_89, tvt[8]);

#if ASIC_REV > ASIC_REV_A0
    stmotors[motor_id].stmotor_regs->TVT_89  = STMOTOR_REG_TVT_89_TVT_9_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_89, tvt[9]);
    stmotors[motor_id].stmotor_regs->TVT_10_11  = STMOTOR_REG_TVT_10_11_TVT_10_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_10_11, tvt[10]);
    stmotors[motor_id].stmotor_regs->TVT_10_11  = STMOTOR_REG_TVT_10_11_TVT_11_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_10_11, tvt[11]);
    stmotors[motor_id].stmotor_regs->TVT_12_13  = STMOTOR_REG_TVT_12_13_TVT_12_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_12_13, tvt[12]);
    stmotors[motor_id].stmotor_regs->TVT_12_13  = STMOTOR_REG_TVT_12_13_TVT_13_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_12_13, tvt[13]);
    stmotors[motor_id].stmotor_regs->TVT_14_15  = STMOTOR_REG_TVT_14_15_TVT_14_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_14_15, tvt[14]);
    stmotors[motor_id].stmotor_regs->TVT_14_15  = STMOTOR_REG_TVT_14_15_TVT_15_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_14_15, tvt[15]);
    stmotors[motor_id].stmotor_regs->TVT_16  = STMOTOR_REG_TVT_16_TVT_16_REPLACE_VAL(stmotors[motor_id].stmotor_regs->TVT_16, tvt[16]);
#endif



}



static void smot_step_set_sequence(stmotor_id_t motor_id, uint32_t stmotor_sequence[])
{

    stmotors[motor_id].stmotor_regs->SEQ_0 = stmotor_sequence[0];
    stmotors[motor_id].stmotor_regs->SEQ_1 = stmotor_sequence[1];
    stmotors[motor_id].stmotor_regs->SEQ_2 = stmotor_sequence[2];
    stmotors[motor_id].stmotor_regs->SEQ_3 = stmotor_sequence[3];
    stmotors[motor_id].stmotor_regs->SEQ_4 = stmotor_sequence[4];
    stmotors[motor_id].stmotor_regs->SEQ_5 = stmotor_sequence[5];
#if ASIC_REV > ASIC_REV_A0
    stmotors[motor_id].stmotor_regs->SEQ_0_U = stmotor_sequence[6];
    stmotors[motor_id].stmotor_regs->SEQ_1_U = stmotor_sequence[7];
    stmotors[motor_id].stmotor_regs->SEQ_2_U = stmotor_sequence[8];
    stmotors[motor_id].stmotor_regs->SEQ_3_U = stmotor_sequence[9];
    stmotors[motor_id].stmotor_regs->SEQ_4_U = stmotor_sequence[10];
    stmotors[motor_id].stmotor_regs->SEQ_5_U = stmotor_sequence[11];
#endif
}


void smot_step_block_enable(stmotor_id_t motor_id, bool value)
{
    stmotors[motor_id].stmotor_regs->SMC_CFG = STMOTOR_REG_SMC_CFG_STEP_EN_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CFG, value);
}


static void smot_step_cdma_mode_enable(stmotor_id_t motor_id, bool value)
{
    stmotors[motor_id].stmotor_regs->SMC_CFG = STMOTOR_REG_SMC_CFG_CDMA_MODE_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CFG, value);
}


static void smot_step_soft_reset(stmotor_id_t motor_id, bool value)
{
    stmotors[motor_id].stmotor_regs->SMC_CTRL = STMOTOR_REG_SMC_CTRL_SRESET_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CTRL, value);
}


static void smot_step_command_abort(stmotor_id_t motor_id, bool value)
{
    stmotors[motor_id].stmotor_regs->SMC_CTRL = STMOTOR_REG_SMC_CTRL_CMD_ABRT_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CTRL, value);
}

static void smot_step_set_blk_addr_irq(stmotor_id_t motor_id, uint32_t cdma_num)
{

#ifdef __KERNEL__
    
    stmotors[motor_id].tx_dma_slave.vtype = MV61_VDMA_OWNED;
    stmotors[motor_id].tx_dma_slave.wr_delay = 0;
    stmotors[motor_id].tx_dma_slave.destendian = MV61_DMA_LITTLE_ENDIAN;
    stmotors[motor_id].tx_dma_slave.srcendian = MV61_DMA_LITTLE_ENDIAN;
    stmotors[motor_id].tx_dma_slave.flowctrl = MV61_DMA_MEMORY_TO_PERIPHERAL;
    stmotors[motor_id].tx_dma_slave.dest_pid = cdma_num;
    stmotors[motor_id].tx_dma_slave.dest_addr_inc = true;
    stmotors[motor_id].tx_dma_slave.src_addr_inc = true;
    stmotors[motor_id].tx_dma_slave.dest_width = MV61_DMA_XFER_WIDTH_32BIT;
    stmotors[motor_id].tx_dma_slave.src_width = MV61_DMA_XFER_WIDTH_32BIT;
    stmotors[motor_id].tx_dma_slave.data_unit_size = MV61_DMA_UNIT_SIZE_32BIT;
    stmotors[motor_id].tx_dma_slave.dest_burst = MV61_DMA_BURST1;
    stmotors[motor_id].tx_dma_slave.src_burst = MV61_DMA_BURST1;
    stmotors[motor_id].tx_dma_slave.dest_reg = (dma_addr_t)&(stmotors[motor_id].phy_stmotor_regs->PWM_T);
    stmotors[motor_id].tx_dma_slave.timebase = MV61_TIMEBASE_1MS;
    stmotors[motor_id].tx_dma_slave.timer = 0;
    stmotors[motor_id].tx_dma_slave.wrap = 24;

#endif
}

static void smot_step_set_reg_bit0(stmotor_id_t motor_id, bool bit0)
{
    stmotors[motor_id].stmotor_regs->REG_01 = STMOTOR_REG_REG_01_REG_BIT0_REPLACE_VAL(stmotors[motor_id].stmotor_regs->REG_01, bit0);
}


static void smot_step_set_reg_bit1(stmotor_id_t motor_id, bool bit1)
{
    stmotors[motor_id].stmotor_regs->REG_01 = STMOTOR_REG_REG_01_REG_BIT1_REPLACE_VAL(stmotors[motor_id].stmotor_regs->REG_01, bit1);
}

static void smot_step_int_enable(stmotor_id_t motor_id, uint32_t int_enab)
{
    uint32_t intAckMask;

    intAckMask =
        (   STMOTOR_REG_I_ACK_CMD_MASK |
            STMOTOR_REG_I_ACK_STEP_INT_MASK |
            STMOTOR_REG_I_ACK_FIFO_OVERFLOW_MASK |
            STMOTOR_REG_I_ACK_I_OMODE_MASK |
            STMOTOR_REG_I_ACK_I_USTEPS_MASK |
            STMOTOR_REG_I_ACK_STCNT_UO_MASK |
            STMOTOR_REG_I_ACK_I_INST_MASK |
            STMOTOR_REG_I_ACK_PCNT_UO_MASK |
            STMOTOR_REG_I_ACK_MCNT_UO_MASK
        );
    stmotors[motor_id].stmotor_regs->I_ACK = intAckMask;
    stmotors[motor_id].stmotor_regs->I_ACK = 0;
    stmotors[motor_id].stmotor_regs->I_EN = int_enab;
}


static void smot_step_int_disable(stmotor_id_t motor_id)
{
    stmotors[motor_id].stmotor_regs->I_EN = 0;
}
/**
 * @brief remove the top of the motor command linked list and 
 *        free all the elements.
 * 
 * @param motor_id Motor number of the linked list
 * 
 * @return bool 
 * @retval true removed an entry 
 * @retval false List already empty 
 */
static bool remove_list_top(stmotor_id_t motor_id)
{
    struct linked_list_format_s *list_entry;

    if (list_empty(&stmotors[motor_id].list_head))
    {
        return false;
    }

    mutex_lock(&stmotors[motor_id].sem);
    list_entry = list_entry(stmotors[motor_id].list_head.next, linked_list_format_t, list_entry);
    list_del(&list_entry->list_entry);
    mutex_unlock(&stmotors[motor_id].sem);
    sg_free_table(list_entry->sg_table);
    kfree(list_entry->data_buffer);
    kfree(list_entry);
    return true;
}

/**
 * @brief This is called by the cdma driver when it completes a 
 *        cmd list.  Note the move will most likley not be
 *        complete, just that the cdma is complete.
 */
#ifndef __KERNEL__
void smot_step_motor_cdma_callback(void *cb_data, uint32_t isr_pending_bits) 
#else
void smot_step_motor_cdma_callback(void *cb_data) 
#endif
{
    stmotor_id_t  motor_id;
    uint32_t      *temp;
    
    temp = cb_data;
    for (motor_id=0; motor_id<stmotors_created; motor_id++) 
    {
        if (stmotors[motor_id].stmotor_cdma_num == *temp) 
        {
            break;
        }
    }
#ifdef __KERNEL__
    //
    // clean up the memory allocations.
    //
    remove_list_top(motor_id);
    //
    // if the list is empty, report that we are done with everything
    //
    if (list_empty(&stmotors[motor_id].list_head))
    {
        int_value_thread[1] = ST_COMPLETE;
        int_value_thread[2] = 1 << motor_id;
        int_value_thread[0] = 8;
        wake_up_interruptible(&wait_queue);

    }
    if (smot_step_debug_in_interrupt_enable)
    {
        step_func_infor("stmotors[%d]'s cdma transfer was completed\n", motor_id);
    }
#endif
#ifndef __KERNEL__
    tx_semaphore_put(&stmotor_cdma_semaphore);
#else
#endif
}
irqreturn_t smot_step_motor_interrupt(int irq, void *intRegMask)
{
    uint32_t              isr_pending;
    stmotor_id_t          motor_id = (stmotor_id_t) intRegMask;


    isr_pending = stmotors[motor_id].stmotor_regs->I_PEND & stmotors[motor_id].stmotor_regs->I_EN;  // get the interrupt I am interested in.
  //  printk("interrupts %x\n", stmotors[motor_id].stmotor_regs->I_PEND);
    
    if (isr_pending & STMOTOR_REG_I_ACK_I_INST_MASK)
    {
        int_value[2] |= 1<<motor_id; // let the thing go
        int_value[1] = ST_NOTIFY;
        int_value[0] = 8;
        wake_up_interruptible(&wait_queue);
        
    }
    // ack the interrupt
    stmotors[motor_id].stmotor_regs->I_ACK = isr_pending;
    return IRQ_HANDLED;
}




/**
 *  \brief Motor set to idle or leave from idle
 *
 *  \param[in] stmotor_id    Client Motor ID
 *  \parma[in] seq           idle setting
 **/
static void smot_step_set_idle(stmotor_id_t motor_id, uint32_t seq)
{

    stmotors[motor_id].stmotor_regs->STEP_EN_IDLE = 
        STMOTOR_REG_STEP_EN_IDLE_STEP_EN_IDLE_SMC_REPLACE_VAL(0, seq);

}
/**
 * @brief set the # of u steps in the driver
 * 
 * @param motor_id The motor block
 * @param usteps number of micro steps
 * 
 * @return uint32_t 
 * @retval 0 success 
 * @retval -1 bad usteps value 
 */
uint32_t smot_set_num_usteps(stmotor_id_t motor_id, uint32_t usteps)
{
    if (usteps == 1 || usteps == 2 || usteps == 4 || usteps == 8)
    {
        stmotors[motor_id].stmotor_regs->NUM_USTEPS =  usteps;
    } else

#if ASIC_REV > ASIC_REV_A0
    if (usteps == 16)
    {
        stmotors[motor_id].stmotor_regs->NUM_USTEPS =  usteps;
    } else
#endif
    {
        return -1;
    }
    return 0;
}

void smot_ls_set_reg(stmotor_id_t motor_id, uint32_t ls_val)
{
    stmotors[motor_id].stmotor_regs->LS_INCR =  ls_val;
}


/**
 *  \brief (API) Initialize Stepper motor code
 *
 *  API function to initialize the Stepper motor code, must be called once at system
 *  startup before any other motor routines can be used.
 *
 *  \param[in] num_motors  Number of motors to support
 *
 **/
bool smot_step_init(uint8_t num_motors)
{

    if (stmotors != NULL)
    {
        step_func_error("motor has already been initialized\n");
        ASSERT(0);
        return true;
    }

    stmotors = (stmotor_params_t *)kmalloc(sizeof(stmotor_params_t) * num_motors, GFP_KERNEL);
    if (stmotors == NULL)
    {
        step_func_error("memory allocation failed\n");
        ASSERT(0);
        return true;
    }
    memset(stmotors, 0, (sizeof(stmotor_params_t)*num_motors));
    st_max_motors = num_motors;

    return false;
}


/**
 *  \brief (API) Set up a new motor
 *
 *  API function to create a new motor instance, usually called by the mech code
 *  during system startup.  
 *
 **/
stmotor_id_t smot_create_motor(uint32_t motor_id)
{
  //  stmotor_id_t motor_id = INVALID_STEP_MOTOR_ID;
    if (st_max_motors == -1)
    {
        step_func_error("please call smot_step_init first\n");
        ASSERT(0);
        return INVALID_STEP_MOTOR_ID;
    }

    if (stmotors_created >= st_max_motors)
    {
        step_func_error("no more available motor ID\n");
        ASSERT(0);
        return INVALID_STEP_MOTOR_ID;
    }

    stmotors_created++;

   // printk("Create motor %d\n", stmotors_created);
    step_func_infor("ID=%d ADDR=0x%p IRQ=%d CDMA=%d\n", motor_id, 
                                                             stmotors[motor_id].stmotor_regs,
                                                             stmotors[motor_id].stmotor_irq_num,
                                                             stmotors[motor_id].stmotor_cdma_num);
#ifndef __KERNEL__
    stmotors[motor_id].stmotor_dma_chan = 0;
#endif
    //block reset
    smot_step_block_enable(motor_id, false); //stepping disable
    smot_step_soft_reset(motor_id, true); //soft reset
    smot_count_reset(motor_id, true); //count reset
    smot_step_command_abort(motor_id, true); //command abort
    smot_step_command_abort(motor_id, false);
    smot_step_soft_reset(motor_id, false);
    smot_count_reset(motor_id, false);

    //interrupt config
    if(request_irq(stmotors[motor_id].stmotor_irq_num,
                          smot_step_motor_interrupt,
                          0,
                          "stepperMotor",
                          (void*)motor_id)){
        return -EIO;
    }
    smot_step_int_disable(motor_id);

    stmotors[motor_id].stmotor_regs->NUM_USTEPS = STMOTOR_REG_NUM_USTEPS_NUM_REPLACE_VAL(stmotors[motor_id].stmotor_regs->NUM_USTEPS, 8);

    //set default variables
    smot_step_block_enable(motor_id, true); 
    smot_step_set_idle(motor_id, 0); 
    smot_step_block_enable(motor_id, false); 
#ifdef __KERNEL__
#endif


    return motor_id;
}




static int status_open(struct inode *ind, struct file *filp)
{
    return 0;
}
/**
 * @brief get the status of the motor move
 * When this is called, if there is a pending status that is 
 * obtained.  Otherwise it waits for a status to arrive.  This 
 * will return with a 0 after 1 second if nothing comes in. 
 * 
 * @param buf Where to put the status data
 * @param count The count of the data.
 * 
 * @return int 
 * @retval 0 timeout 
 * @retval >0 the amount of status info in the buffer in bytes. 
 */
int stepper_status(char *buf, ssize_t count)
{
    uint32_t size;
    long ret;
  //  printk("In stepper status value %d\n", int_value[0]);
    if (int_value[0] == 0 && int_value_thread[0] == 0)
    {
        ret = wait_event_interruptible_timeout(wait_queue, int_value[0] != 0 || int_value_thread[0] != 0, HZ);  // time out after a second.
        if (!ret)
        {
            return 0;
        }
    }
 //   printk("values 0 = %d code = %d data = %d\n", int_value[0], int_value[1], int_value[2]);
    if (int_value[0] != 0)
    {
        memcpy(buf, int_value, sizeof(uint32_t) + int_value[0]);
        size = int_value[0];
        memset(int_value,0, sizeof(int_value));
        return sizeof(uint32_t) + size;
    }
    if (int_value_thread[0] != 0)
    {
        memcpy(buf, int_value_thread, sizeof(uint32_t) + int_value_thread[0]);
        size = int_value_thread[0];
        memset(int_value_thread,0, sizeof(int_value_thread));
        return sizeof(uint32_t) + size;
    }
    return 0;

}
EXPORT_SYMBOL(stepper_status);
/**
 * @brief This is the uio status read function.
 * 
 * @param filp The open handle.
 * @param buf Pointer to the status buffer.
 * @param count The size of the buffer
 * @param f_pos Not used
 * 
 * @return ssize_t 
 */
static ssize_t status_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
    int ret;

    while((ret = stepper_status(buf, count)) == 0);
    return ret;

}

static int status_close(struct inode *ind, struct file *filp)
{
    return 0;
}

#define TYPE(dev) (MINOR(dev) >> 4) /* high nibble */
#define NUM(dev) (MINOR(dev) & 0xf) /* low nibble */


int stmotor_open(uint32_t motor_num)
{
    if (motor_num >= STEPPER_NUM_MOTORS)
    {
        return -1;
    }
    stmotors[motor_num].stmotor_cdma_enable = false;
    smot_step_set_blk_addr_irq(motor_num, stmotors[motor_num].stmotor_cdma_num);
    stmotor_active |= 1<<motor_num;
    return 0;

}

EXPORT_SYMBOL(stmotor_open);

static int driver_open(struct inode *ind, struct file *filp)
{
  //  Scull_Dev *dev; /* device information */
    int motor_num = NUM(ind->i_rdev);

    filp->private_data = (void *)motor_num;

    return stmotor_open(motor_num);
}

static void install_new_st_motor_connect(uint32_t motor_num, uint32_t stmotor_connects)
{
    if (motor_num < 0 || motor_num >= STEPPER_NUM_MOTORS)
    {
        return;
    }

    stmotors[motor_num].stmotor_regs->OUT_MODE = stmotor_connects;

}

static void stmot_disable_motor(uint32_t motor_id)
{
    if (motor_id < 0 || motor_id >= STEPPER_NUM_MOTORS)
    {
        return;
    }

    if (stmotors[motor_id].stmotor_cdma_enable)
    {
        dma_release_channel(stmotors[motor_id].stmotor_dma_chan);
        stmotors[motor_id].stmotor_cdma_enable = false;
        smot_step_block_enable(motor_id, false); //stepping disable
    }
}

static void smot_count_reset(stmotor_id_t motor_id, bool value)
{
    stmotors[motor_id].stmotor_regs->SMC_CTRL = STMOTOR_REG_SMC_CTRL_STCNT_RST_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CTRL, value);
}

/**
 * @brief execute the defined 
 * s for a given motor.
 * 
 * 
 * @param motor_id The motor on which to execute this ioctl
 * @param cmd The cmd to execute
 * @param args The arguments for a given command.
 * 
 * @return long 
 * @retval 0 all OK 
 * @retval != 0 bad. 
 */
long stmot_ioctl(uint32_t motor_id, unsigned int cmd, unsigned long args)
{

#if ASIC_REV <= ASIC_REV_A0
    uint32_t sequence[6];
    uint16_t tvt[9];
#else
    uint16_t tvt[17];
    uint32_t sequence[12];
#endif
    int i;
    int ret = 0;
    if (motor_id < 0 || motor_id >= STEPPER_NUM_MOTORS)
    {
        return -2;
    }
   
    #if 0
    static uint32_t last_cmd = 0;
    if(last_cmd != cmd)
    {
        last_cmd = cmd;
        printk("Got ioctl %d cmd %d args %x\n", motor_id, cmd, args);
    }
    #endif
    switch ((stioctl_cmds_t) cmd)
    {
        case SET_TORQUE_TABLE:
            // set the torque table settings.
            if(copy_from_user(tvt, (uint16_t *)args, sizeof(tvt)) != 0)
            {
                // copy from user failed, this must be coming from the kernel 
                // and we can access directly.

                for (i = 0; i < sizeof(tvt)/sizeof(uint16_t); i++)
                {
                    tvt[i] = ((uint16_t *)args)[i];
                }
            }
            smot_step_set_torque_vector_table(motor_id, tvt);
            break;
        case SET_MTR_CONNECTS:
            // set the initial motor connects.
            install_new_st_motor_connect(motor_id, (uint32_t)args);
            break;
        case SET_REG0_BIT:
            smot_step_set_reg_bit0(motor_id, (bool) args);
            break;
        case SET_REG1_BIT:
            smot_step_set_reg_bit1(motor_id, (bool)args);
            break;
        case SET_SEQUENCE:
            //
            // Set the sequence registers.  The input is an array of 32 bit registers 6 long.
            if(copy_from_user(sequence, (uint32_t *)args, sizeof(sequence)) != 0)
            {
                // copy from user failed, must be a copy from kernel

                for (i = 0; i < sizeof(sequence)/sizeof(uint32_t); i++)
                {
                    sequence[i] = ((uint32_t *)args)[i];
                }
            }
            smot_step_set_sequence(motor_id, sequence);
            break;

        case SET_DMA_WRAP:  // must be done before first write, or no-op
            // this is the length of the cdma xfer to the motor block.
            //
            stmotors[motor_id].tx_dma_slave.wrap = (uint32_t) args;
            break;
        case SET_DMA_DESTADDR:      // must be done before first write, or no-op
            // this is the starting offset for the cdma xfer.  This can be used to control odd motors.
            stmotors[motor_id].tx_dma_slave.dest_reg = (dma_addr_t)((uint32_t)stmotors[motor_id].phy_stmotor_regs + (uint32_t)args);
            break;
        case SET_STEP_ENA_IDLE:
            // set the idle motor line setting.
            smot_step_set_idle(motor_id, args);
            break;
        case SET_NUM_USTEPS:

            // set the number of u-steps/step.

            ret = smot_set_num_usteps(motor_id, args);
            break;
            // this actually does an abort.  If you start a sequence with a hold and then moves, this will start the motors.
        case START_MOTORS:
            sequence[0] = (uint32_t)args & stmotor_active;      // get the motors to start
            for (i = 0; i < STEPPER_NUM_MOTORS; i++)
            {
                if ((1<<i)&sequence[0])
                {
                    smot_step_command_abort(i, true);
                }
            }
            break;
        case SET_LINE_START_REG:
            //
            // set the line start register.
            smot_ls_set_reg(motor_id, (uint32_t)args);
            break;
        case GET_REG_ADDR:
            ret = (int32_t) stmotors[motor_id].stmotor_regs;
            break;
        case GET_MTR_IRQ:
            // return the irq number for the motor.
            ret = stmotors[motor_id].stmotor_irq_num;
            break;
        case GET_MTR_IDLE:
            if (!list_empty(&stmotors[motor_id].list_head))
            {
                ret = false;       // still stuff on the list
                break;
            }
            ret = true;
            break;
        case EMERGENCY_STOP:
            if (stmotors[motor_id].stmotor_cdma_enable)
            {
                // kill all the cdma activity on this motor
                dmaengine_terminate_all(stmotors[motor_id].stmotor_dma_chan);   // this does not cause the callback to happen
                // kill the current command at the motor
                smot_step_command_abort(motor_id, true);
                // the motor can queue up a command, kill this command.
                smot_step_command_abort(motor_id, true);
                //
                // get rid of the entry on the list, since the callback was not called.
                //
                while (remove_list_top(motor_id));

                int_value[2] |= 1 << motor_id; // let the thing go
                int_value[1] = ST_NOTIFY;
                int_value[0] = 8;
                wake_up_interruptible(&wait_queue); // notify we are now stopped.
                int_value_thread[2] |= 1 << motor_id; // let the thing go
                int_value_thread[1] = ST_COMPLETE;
                int_value_thread[0] = 8;
                wake_up_interruptible(&wait_queue); // notify we are now stopped.
            }
            break;
        case DISABLE_MOTOR:
            stmot_disable_motor(motor_id);
            break;
        case ABORT_MOTORS:
            i = 0;
            while (stmotors[motor_id].stmotor_regs->COM_CMD != (uint32_t)args &&
                   (stmotors[motor_id].stmotor_regs->COM_STAT & STMOTOR_REG_COM_STAT_EXEC_MASK))
            {
                i++;
                smot_step_command_abort(motor_id, true);    // kill till we find the cmd we want
                smot_step_command_abort(motor_id, false);
            }
            break;
        case RESET_STEP_CNT:
            stmotors[motor_id].stmotor_regs->SMC_CTRL = STMOTOR_REG_SMC_CTRL_STCNT_RST_REPLACE_VAL(stmotors[motor_id].stmotor_regs->SMC_CTRL, 1);
            break;
        case GET_STEP_CNT:
            {
                long cur_loc;
                //
                // get the current step count and sign extend it to 32 bits.
                //
                cur_loc = stmotors[motor_id].stmotor_regs->COM_STP & 0xffffff;
                if ((cur_loc & 0x800000) > 0) 
                {                               
                   cur_loc = -(0x1000000 - cur_loc);  
                }
                return (cur_loc);
            }
            break;

    }
    return ret;
}

EXPORT_SYMBOL(stmot_ioctl);
/**
 * @brief This is connected to a device file in user space.
 * 
 * 
 * @param filp If of the file
 * @param cmd the command to execute
 * @param args The arguments for this command.
 * 
 * @return long 
 */
static long driver_ioctl(struct file *filp, unsigned int cmd, unsigned long args)
{
    int motor_id = (uint32_t) filp->private_data;     // get the motor number
    return stmot_ioctl(motor_id,cmd,args);
}
/**
 * @brief Write to a given motor block 
 * This queues a data block to the stepper.  The input is 
 * copied, so owernship of the data buffer remains with the 
 * calling program. 
 * 
 * @param motor_id The motor block to use
 * @param buf Pointer to the data buffer
 * @param count The size of the buffer in bytes
 * 
 * @return ssize_t 
 * @retval <0 error 
 * @retval >=0 #queued to the stepper. 
 */
ssize_t stmot_write(int motor_id, const char *buf, size_t count)
{

    dma_cap_mask_t mask;
    struct dma_async_tx_descriptor *tx_desc;
    dma_cookie_t cookie;
    int len;
    struct linked_list_format_s *list_entry;

    struct scatterlist *sgl = NULL;

    if (motor_id < 0 || motor_id >= STEPPER_NUM_MOTORS)
    {
        return -2;
    }

#if 0
    printk("motor_id %d addr %p\n", motor_id, stmotors[motor_id].stmotor_regs);
    for (len = 0; len < count/sizeof(uint32_t); len++)
    {
        printk("Cmd setting cnt %d val %x\n", len, ((uint32_t *)buf)[len]);
    }
#endif
 //   printk("list_empty  start write %d motor %d\n", list_empty(&stmotors[motor_id].list_head), motor_id);
    if (!stmotors[motor_id].stmotor_cdma_enable)
    {
        //
        // The cdma is not initialized, get the channel and get things ready to go.
        //
        smot_step_int_disable(motor_id);
        
        smot_step_cdma_mode_enable(motor_id, true);
        smot_step_block_enable(motor_id, true);
        // turn on the cmd error interrupt, if needed
        //
        smot_step_int_enable(motor_id, STMOTOR_REG_I_ACK_I_INST_MASK);      // turn on the interrupt for the illegal instruction
        dma_cap_zero(mask);
        dma_cap_set(DMA_SLAVE, mask);

       // printk("slave %p val %x\n", &stmotors[motor_id].tx_dma_slave, stmotors[motor_id].tx_dma_slave.dest_width);
        stmotors[motor_id].stmotor_dma_chan = dma_request_channel(mask, filter, &(stmotors[motor_id].tx_dma_slave));

        if (stmotors[motor_id].stmotor_dma_chan == NULL)
        {
            printk("ERROR dma channel\n");
            step_func_error("dma_request_channel error\n");
            return -1;
        }
       // printk("dma channel %p\n", stmotors[motor_id].stmotor_dma_chan);
        stmotors[motor_id].stmotor_cdma_enable = true;
        step_func_infor("stmotors[%d].stmotor_sg_table:%p,dma_chan:%p sgl:%p\n", motor_id,
                        stmotors[motor_id].stmotor_sg_table,
                        stmotors[motor_id].stmotor_dma_chan, SGL);

    }

    //
    // get the memory for the linked list entry to hold this info.
    //
    list_entry = kmalloc(sizeof(struct linked_list_format_s), __GFP_WAIT | ZONE_NORMAL);
    if (!list_entry)
    {
        printk("Unable to get a list entry\n");
        return -1;
    }
 //   printk("List entry %p\n", list_entry);
    INIT_LIST_HEAD(&list_entry->list_entry);
    list_entry->sg_table = kmalloc(sizeof(struct sg_table), GFP_DMA);
    if (!list_entry->sg_table)
    {
        printk("failed to get buffer\n");
        kfree(list_entry);
        return -1;
    }
    list_entry->data_buffer =  kmalloc(count, GFP_DMA);
    if (!list_entry->data_buffer )
    {
        printk("Did not get the buffer\n");
        kfree(list_entry->sg_table);
        kfree(list_entry);
        return -1;
    }
    //
    // copy the input info into the list buffer.
    //
    memcpy(list_entry->data_buffer, buf, count);
    sg_alloc_table(list_entry->sg_table, 1, GFP_DMA);
    sgl = list_entry->sg_table->sgl;

    sg_set_buf(sgl, list_entry->data_buffer, count); 
    sgl = sg_next(sgl); 

  //  printk("map the channel device %p %x\n",stmotors[motor_id].stmotor_dma_chan->device,stmotors[motor_id].stmotor_dma_chan->device->dev);
    len = dma_map_sg(stmotors[motor_id].stmotor_dma_chan->device->dev,
                     list_entry->sg_table->sgl,
                     1,
                     DMA_TO_DEVICE);
   // printk("dma map len %d\n", len);
    if (len <= 0) 
    {
        step_func_error("dma_map_sg error\n");
        kfree(list_entry->data_buffer);
        kfree(list_entry->sg_table);
        kfree(list_entry);
        return -1;
    }

    tx_desc = stmotors[motor_id].stmotor_dma_chan->device->device_prep_slave_sg(
        stmotors[motor_id].stmotor_dma_chan,
        list_entry->sg_table->sgl,
        len,
        DMA_TO_DEVICE, DMA_PREP_INTERRUPT | DMA_CTRL_ACK, &len); // &len is for context, which is not used
    if (!tx_desc) step_func_error("dma_prep_slave_sg error\n");

    tx_desc->callback = smot_step_motor_cdma_callback;
    tx_desc->callback_param = &stmotors[motor_id].stmotor_cdma_num;

    // save the list entry
    //
    mutex_lock(&stmotors[motor_id].sem);

    list_add_tail(&list_entry->list_entry, &stmotors[motor_id].list_head);
    mutex_unlock(&stmotors[motor_id].sem);
    //
    // Now send the dma chain to the cdma engine for passing to the stepper motor driver
    //
    cookie = dmaengine_submit(tx_desc);
    stmotors[motor_id].stmotor_dma_chan->device->device_issue_pending(stmotors[motor_id].stmotor_dma_chan);
    return count;
}

EXPORT_SYMBOL(stmot_write);

/**
 * @brief This provides the file write function for the uio 
 *        driver.  Uses standard write api.
 * 
 * @param filp File handle
 * @param buf Pointer to the buffer
 * @param count Size of the buffer
 * @param f_pos ignored
 * 
 * @return ssize_t 
 * @retval <0 error 
 * @retval >=0 number of bytes written. 
 */
static ssize_t driver_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
    int motor_id = (uint32_t)filp->private_data;     // get the motor number

    return stmot_write(motor_id, buf, count);
}

// start of debug stuff
#define CLK_FREQ 6000000
#define MIN_P 20
#define PWM_T_VALUE 5556

typedef struct
{
    int motor_id;
    int m_value;
    uint32_t stop_value;
} my_platform_data_t;

static ssize_t dbg_motor_stop(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int motor_id;
    motor_id = ((my_platform_data_t *)dev->platform_data)->motor_id;

    stmot_ioctl(motor_id, ABORT_MOTORS, ((my_platform_data_t *)dev->platform_data)->stop_value);

    return count;
}

static ssize_t dbg_motor_halt(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int motor_id;


    motor_id = ((my_platform_data_t *)dev->platform_data)->motor_id;

    stmot_ioctl(motor_id, EMERGENCY_STOP, 0);

    return count;
}


static ssize_t dbg_do_ioctl(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int cmd, arg;
    int ret;
    int motor_id;


    motor_id = ((my_platform_data_t *)dev->platform_data)->motor_id;
 
    ret = sscanf(buf,"%d %x", &cmd, &arg);
    if (ret != 2 || cmd < 0 || cmd > GET_STEP_CNT)
    {
        return -1;
    }
    stmot_ioctl(motor_id,cmd, arg);
    return count;

}

/**
 * @brief this is the registers that are used for sending cmd 
 *        blocks to the stepper.  Should map to registers in the
 *        stepper block.
 */
typedef struct
{
    uint32_t pwm_t;
    uint32_t pwm_p;
    int32_t acc_incr;
    uint32_t pwm_m;
    uint32_t cmd;
    uint32_t cmd_issue;
} stepper_cmd_t;

stepper_cmd_t *build_move(int steps, bool direction, uint32_t max_speed, uint32_t accel_value, uint32_t *length, int m_value, uint32_t *stop_value)
{

    stepper_cmd_t *cmd_blk;
    int alloc_size = 0;
    int coast_steps, tmp;
    int num_steps, max_p, acc_incr;
    int ramp_time, step_time_slow, step_time_fast;
    int min_p, p_now, cur_p;

    if (m_value == 0)
    {
        m_value = CLK_FREQ / max_speed / MIN_P;
    }
 //   printk("m_value %d\n", m_value);
    ramp_time = max_speed * 1000 / accel_value;
    step_time_slow = 1000000/20;
    step_time_fast = 1000000/max_speed;
    num_steps = 2000 * ramp_time/(step_time_fast + step_time_slow);   //
    max_p = CLK_FREQ / 20 / m_value;    // figure the max p value
    min_p = CLK_FREQ / max_speed / m_value;
  //  printk("max p %d min p %d\n", max_p, min_p);

    // break into 3 pieces, first 1/6, 2nd 1/6 last 2/3
    //
    p_now = ((max_p - min_p) * 5)/6;
    acc_incr = p_now /((num_steps-1)/6);
    cmd_blk = kmalloc(sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p = max_p;

    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].pwm_p = max_p;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/6) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    alloc_size++;
    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p -= acc_incr * ((num_steps/6)-1);

    cmd_blk[alloc_size].pwm_p = cur_p;
    p_now = ((max_p - min_p)*3)/24;
    acc_incr = p_now /((num_steps-2)/3);
    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/3) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    alloc_size++;


    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p -= acc_incr * ((num_steps/3) -1);

    cmd_blk[alloc_size].pwm_p = cur_p;
    p_now = ((max_p - min_p))/24;
    acc_incr = p_now /((num_steps-3)/2);
    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/2) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    alloc_size++;


    if (num_steps > steps)
    {
        num_steps = steps;
    }

    // now coast.
    coast_steps = steps - num_steps;
    while (coast_steps > 0)
    {
        tmp = coast_steps;
        if (tmp > 0xffff)
        {
            tmp = 0xffff;
        }
        cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
        if (!cmd_blk)
        {
            return NULL;
        }
        cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
        cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
        cmd_blk[alloc_size].pwm_m = m_value;
        cmd_blk[alloc_size].pwm_p = CLK_FREQ / max_speed / m_value;
        cmd_blk[alloc_size].acc_incr = 0;
        cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, tmp) |
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 1) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) |
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
        alloc_size++;
        coast_steps -= tmp;

    }
    // now slow down
    //

    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p = cmd_blk[alloc_size-1].pwm_p;

    cmd_blk[alloc_size].pwm_p = cur_p;
    p_now = ((max_p - min_p))/24;
    acc_incr = p_now /((num_steps-1)/2);
    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/2) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_ACC_DIR_REPLACE_VAL(0,1) |
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    *stop_value = cmd_blk[alloc_size].cmd;
    alloc_size++;



    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p += acc_incr * ((num_steps/2)-1);

    cmd_blk[alloc_size].pwm_p = cur_p;
    p_now = ((max_p - min_p)*3)/24;
    acc_incr = p_now /((num_steps-1)/3);
    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/3) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_ACC_DIR_REPLACE_VAL(0,1) |
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    alloc_size++;


    cmd_blk = krealloc(cmd_blk, (alloc_size + 1) * sizeof(stepper_cmd_t), 0);
    if (!cmd_blk)
    {
        return NULL;
    }
    cur_p += acc_incr * ((num_steps/3)-1);

    cmd_blk[alloc_size].pwm_p = cur_p;
    p_now = ((max_p - min_p)*5)/6;
    acc_incr = p_now /((num_steps-2)/6);
    cmd_blk[alloc_size].pwm_t = PWM_T_VALUE;
    cmd_blk[alloc_size].cmd_issue = 1;         // issue the cmd.
    cmd_blk[alloc_size].pwm_m = m_value;
    cmd_blk[alloc_size].acc_incr = acc_incr;
    cmd_blk[alloc_size].cmd = STMOTOR_REG_CMD_ST_TGT_REPLACE_VAL(0, num_steps/6) | 
            STMOTOR_REG_CMD_CMD_REPLACE_VAL(0, 4) |
            STMOTOR_REG_CMD_LS_EN_REPLACE_VAL(0, 0) | 
            STMOTOR_REG_CMD_ACC_DIR_REPLACE_VAL(0,1) |
            STMOTOR_REG_CMD_DIR_REPLACE_VAL(0, direction);
    alloc_size++;

    *length = alloc_size * sizeof(stepper_cmd_t);
    return cmd_blk;


}

static ssize_t dbg_move(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int  distance  = 0;
    int  direction = 0;
    stepper_cmd_t *cmd_block;
    uint32_t motor_id;
    uint32_t length;
    int max_speed;
    uint32_t accel_value;
    int m_value, stop_value;
    int ret;

    motor_id = ((my_platform_data_t *)dev->platform_data)->motor_id;
    m_value = ((my_platform_data_t *)dev->platform_data)->m_value;
 
    ret = sscanf(buf,"%d %d %d %d", &distance, &direction, &max_speed, &accel_value);
    if (ret != 4)
    {
        printk("Illegal input, want distance direction speed acceleration\n");
        return -1;
    }
    cmd_block = build_move(distance, direction, max_speed, accel_value, &length, m_value, &stop_value);
    ((my_platform_data_t *)dev->platform_data)->stop_value = stop_value;
    for (distance = 0; distance < length/4; distance++)
    {
        printk("#%d %x\n", distance, ((uint32_t *)cmd_block)[distance]);
    }
    stmot_write(motor_id,(const char *)cmd_block, length);
    return count;
}

void dbg_print_reg(uint32_t motor_id)
{

    printk("Stepper Registers motor block #%d\n", motor_id);
    printk("name\t\toffset\tvalue\n");
    printk("SMC_CFG=\t%x\t0x%X\n",    offsetof(STMOTOR_REG_REGS_t, SMC_CFG),  stmotors[motor_id].stmotor_regs->SMC_CFG);
    // SMC_CTRL is write only, not dumping
    printk("LS_INCR=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, LS_INCR),   stmotors[motor_id].stmotor_regs->LS_INCR);
    printk("REG_01=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, REG_01), stmotors[motor_id].stmotor_regs->REG_01);
    printk("NUM_USTEPS=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, NUM_USTEPS),   stmotors[motor_id].stmotor_regs->NUM_USTEPS);
    printk("STEP_EN_IDLE=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, STEP_EN_IDLE), stmotors[motor_id].stmotor_regs->STEP_EN_IDLE);
    printk("TVT_01=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, TVT_01),      stmotors[motor_id].stmotor_regs->TVT_01);
    printk("TVT_23=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, TVT_23),      stmotors[motor_id].stmotor_regs->TVT_23);
    printk("TVT_45=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, TVT_45),      stmotors[motor_id].stmotor_regs->TVT_45);
    printk("TVT_67=\t\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_67),       stmotors[motor_id].stmotor_regs->TVT_67);
    printk("TVT_89=\t\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_89),       stmotors[motor_id].stmotor_regs->TVT_89);

#if ASIC_REV > ASIC_REV_A0
    printk("TVT_10_11=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_10_11),       stmotors[motor_id].stmotor_regs->TVT_10_11);
    printk("TVT_12_13=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_12_13),       stmotors[motor_id].stmotor_regs->TVT_12_13);
    printk("TVT_14_15=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_14_15),       stmotors[motor_id].stmotor_regs->TVT_14_15);
    printk("TVT_16=\t\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, TVT_16),       stmotors[motor_id].stmotor_regs->TVT_16);
#endif

    printk("SEQ_0=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_0),       stmotors[motor_id].stmotor_regs->SEQ_0);
    printk("SEQ_1=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_1),       stmotors[motor_id].stmotor_regs->SEQ_1);
    printk("SEQ_2=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_2),       stmotors[motor_id].stmotor_regs->SEQ_2);
    printk("SEQ_3=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_3),       stmotors[motor_id].stmotor_regs->SEQ_3);
    printk("SEQ_4=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_4),       stmotors[motor_id].stmotor_regs->SEQ_4);
    printk("SEQ_5=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_5),       stmotors[motor_id].stmotor_regs->SEQ_5);

#if ASIC_REV > ASIC_REV_A0
    printk("SEQ_0_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_0_U),       stmotors[motor_id].stmotor_regs->SEQ_0_U);
    printk("SEQ_1_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_1_U),       stmotors[motor_id].stmotor_regs->SEQ_1_U);
    printk("SEQ_2_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_2_U),       stmotors[motor_id].stmotor_regs->SEQ_2_U);
    printk("SEQ_3_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_3_U),       stmotors[motor_id].stmotor_regs->SEQ_3_U);
    printk("SEQ_4_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_4_U),       stmotors[motor_id].stmotor_regs->SEQ_4_U);
    printk("SEQ_5_U=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SEQ_5_U),       stmotors[motor_id].stmotor_regs->SEQ_5_U);
#endif
    printk("OUT_MODE=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, OUT_MODE),     stmotors[motor_id].stmotor_regs->OUT_MODE);
    printk("STEP_INT=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, STEP_INT),     stmotors[motor_id].stmotor_regs->STEP_INT);
    printk("PWM_T=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, PWM_T),       stmotors[motor_id].stmotor_regs->PWM_T);
    printk("PWM_P=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, PWM_P),       stmotors[motor_id].stmotor_regs->PWM_P);
    printk("ACC_INCR=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, ACC_INCR),    stmotors[motor_id].stmotor_regs->ACC_INCR);
    printk("PWM_M=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, PWM_M),       stmotors[motor_id].stmotor_regs->PWM_M);
    printk("CMD=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, CMD),         stmotors[motor_id].stmotor_regs->CMD);
    // CMD_ISSUE is write only, not dumping
    printk("COM_STAT=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, COM_STAT),    stmotors[motor_id].stmotor_regs->COM_STAT);
    printk("COM_PWM_P=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, COM_PWM_P),    stmotors[motor_id].stmotor_regs->COM_PWM_P);
    printk("COM_ACC_INCR=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, COM_ACC_INCR), stmotors[motor_id].stmotor_regs->COM_ACC_INCR);
    printk("COM_PWM_M=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, COM_PWM_M),   stmotors[motor_id].stmotor_regs->COM_PWM_M);
    printk("COM_CMD=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, COM_CMD),      stmotors[motor_id].stmotor_regs->COM_CMD);
    printk("COM_STP=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, COM_STP),      stmotors[motor_id].stmotor_regs->COM_STP);
    printk("COM_TVT=\t%x\t0x%X\n",offsetof(STMOTOR_REG_REGS_t, COM_TVT),      stmotors[motor_id].stmotor_regs->COM_TVT);
    printk("COM_SEQ=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, COM_SEQ),  stmotors[motor_id].stmotor_regs->COM_SEQ);
    printk("I_EN=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, I_EN),        stmotors[motor_id].stmotor_regs->I_EN);
    printk("I_PEND=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, I_PEND),    stmotors[motor_id].stmotor_regs->I_PEND);
    // I_ACK is write only, not dumping
    // I_FORCE is write only, not dumping
    printk("SMC_TRIG=\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, SMC_TRIG),    stmotors[motor_id].stmotor_regs->SMC_TRIG);
    printk("REV0=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, REV0),        stmotors[motor_id].stmotor_regs->REV0);
    printk("REV1=\t\t%x\t0x%X\n", offsetof(STMOTOR_REG_REGS_t, REV1),        stmotors[motor_id].stmotor_regs->REV1);
}

EXPORT_SYMBOL(dbg_print_reg);

static ssize_t dbg_reg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint32_t motor_id;

    motor_id = ((my_platform_data_t *)dev->platform_data)->motor_id;
    dbg_print_reg(motor_id);
    return 0;
}

static const char *cmd_help =                                    \
"stepper test commands:\n"                                       \
"  cat help           - shows command help\n"                    \
"  echo dist dir max_speed accel > move - relative move of selected test motor dist=steps speed=st/sec accel=st/sec/sec dir=cmd direction bit sense\n"  \
"  echo > stop        - gentle stop selected test motor\n"       \
"  echo > halt        - emergency halt selected test motor\n"    \
"  cat dump           - dumps motor block registers\n"           \
"  echo <pwm_m> >pwm_m - initialize the pwm_m value, if this is missing, figure a good one to use" \
"  echo <cmd> <arg> >ioctl - issue an ioctl cmd to the motor, see .h file";

static ssize_t dbg_get_cmd_help(struct device *dev, struct device_attribute *attr,char *buf)
{
    strcpy(buf, cmd_help);
    return strlen(buf);
}
static ssize_t dbg_get_pwm_m(struct device *dev, struct device_attribute *attr,char *buf)
{

    sprintf(buf,"pwm m value = %d\n", ((my_platform_data_t *)dev->platform_data)->m_value);
    return strlen(buf);
}

static ssize_t dbg_set_pwm_m(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int ret;
    int m_value;

 
    ret = sscanf(buf,"%d", &m_value);
    if (ret != 1)
    {
        return -1;
    }
    ((my_platform_data_t *)dev->platform_data)->m_value = m_value;
    return count;

}



static DEVICE_ATTR(help,    (S_IRUSR),         dbg_get_cmd_help, NULL             );
static DEVICE_ATTR(pwm_m,   (S_IWUSR|S_IRUSR), dbg_get_pwm_m,    dbg_set_pwm_m     );
static DEVICE_ATTR(move,    (S_IWUSR),         NULL,             dbg_move     );
static DEVICE_ATTR(stop,    (S_IWUSR),         NULL,             dbg_motor_stop   );
static DEVICE_ATTR(halt,    (S_IWUSR),         NULL,             dbg_motor_halt   );
static DEVICE_ATTR(dump,    (S_IRUSR),         dbg_reg_show,     NULL             );
static DEVICE_ATTR(ioctl,   (S_IWUSR),         NULL,             dbg_do_ioctl     );

void stmotor_close(int motor_id)
{
    if (stmotors[motor_id].stmotor_cdma_enable)
    {
        dma_release_channel(stmotors[motor_id].stmotor_dma_chan);
        stmotors[motor_id].stmotor_cdma_enable = false;
        smot_step_block_enable(motor_id, false); //stepping disable
    }
    if (stmotor_active == (1<<motor_id))
    {
        stmotor_active &= ~(1<<motor_id);
    }
}
/**
 * @brief Called when a file is closed.
 * 
 * @param ind ignored
 * @param filp file handle
 * 
 * @return int 0 = success.
 */
static int driver_close(struct inode *ind, struct file *filp)
{

    int motor_id = (uint32_t)filp->private_data;     // get the motor number
    stmotor_close(motor_id);
    return 0;
}

static const struct file_operations driver_fops = {
         .owner          = THIS_MODULE,
         .unlocked_ioctl = driver_ioctl,
         .open           = driver_open,
         .release        = driver_close,
         .write          = driver_write,
};

static const struct file_operations status_fops = {
         .owner     = THIS_MODULE,
         .open      = status_open,
         .release   = status_close,
         .read      = status_read,
};

///////////////////////////////////////////////////////////
/**
 * @brief called for each stepper in the system based on the dtsi entries
 * 
 * 
 * @param pdev description of the stepper.
 * 
 * @return int 
 * @retval 0 success 
 * @retval <0 error 
 */
static int stepper_mod_platform_probe(struct platform_device *pdev)
{
    int                 retval = 0;
    int                 irq;
    uint32_t            motor_id;
    struct device_node  *node = pdev->dev.of_node;

    uint32_t            phys_addr;
    uint32_t            addr_size;
    void __iomem        *virt_addr;
    struct resource     *reg_addr;

    //printk(KERN_ERR "%s: probe begin (%s)\n", __func__, ASIC_NAME_STR);

    // Not going to get far if we don't have a device node
    if (!node)
    {
        dev_err(&pdev->dev, "%s: Could not find stepper device node\n", __func__);
        return -ENODEV;
    }

    //printk(KERN_ERR "\n\n\n%s: driver registration begun\n", __func__);

    // get the id of the motor we are working on.
    if (of_property_read_u32(node, "motorid", &motor_id))
    {
        dev_err(&pdev->dev, "reading motorid failed\n");
        return -EINVAL;
    }
        // get the interrupt number for this motor block
    irq = platform_get_irq(pdev, 0);
    if (irq < 0)
    {
        dev_err(&pdev->dev, "platform_get_irq failed\n");
        return -ENXIO;
    }
    stmotors[motor_id].stmotor_irq_num = irq;
    //
    // Get the cdma channel for this motor block.
    //
    if (of_property_read_u32(node, "cdmaid", &stmotors[motor_id].stmotor_cdma_num))
    {
        dev_err(&pdev->dev, "reading cdmaid failed\n");
        return -EINVAL;
    }


    printk(KERN_ERR "%s: Probe motor %d (%s)\n", __func__, motor_id, ASIC_NAME_STR);

    // get the physical address of the registers.
    reg_addr = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    if (!reg_addr)
    {
        dev_err(&pdev->dev, "platform_get_resource failed: %s\n", __func__);
        return -ENXIO;
    }
    phys_addr   = reg_addr->start;
    stmotors[motor_id].phy_stmotor_regs = (STMOTOR_REG_REGS_t *)phys_addr;

    addr_size   = resource_size(reg_addr);
    stmotors[motor_id].size_of_registers = addr_size;
    //
    // get the virtual address for the registers.
    //
    virt_addr = ioremap(phys_addr, addr_size);
    if (!virt_addr)
    {
        dev_err(&pdev->dev, "ioremap failed: %s\n", __func__);
        return -ENOMEM;
    }

    stmotors[motor_id].stmotor_regs = (STMOTOR_REG_REGS_t *)virt_addr;

    //
    // now create the device file for this motor block
    //
    memset(&stmotors[motor_id].miscdev, 0, sizeof(struct miscdevice));
    INIT_LIST_HEAD(&stmotors[motor_id].list_head);
    mutex_init(&stmotors[motor_id].sem);
    stmotors[motor_id].miscdev.minor =  0x40 + motor_id;
    sprintf(stmotors[motor_id].misc_name, "stepper%d", motor_id);
    stmotors[motor_id].miscdev.name  = stmotors[motor_id].misc_name;
    stmotors[motor_id].miscdev.fops  = &driver_fops;
    if (misc_register(&stmotors[motor_id].miscdev))
    {
        dev_err(&pdev->dev, "misc_register failed\n");
        iounmap(virt_addr);
        return -ENXIO;
    }
    //
    // now initialize everything to use this block.
    //
    smot_create_motor(motor_id);

    //
    // get the debug stuff in place

    pdev->dev.platform_data = kmalloc(sizeof(my_platform_data_t), GFP_KERNEL);;
    if (!pdev->dev.platform_data)
    {
        iounmap(virt_addr);
        misc_deregister(&stmotors[motor_id].miscdev);
        return -ENXIO;
    }
    ((my_platform_data_t *)pdev->dev.platform_data)->motor_id = motor_id;
    ((my_platform_data_t *)pdev->dev.platform_data)->m_value = 0;

    if ((device_create_file(&pdev->dev, &dev_attr_help)) ||
        (device_create_file(&pdev->dev, &dev_attr_move)) ||
        (device_create_file(&pdev->dev, &dev_attr_pwm_m)) ||
        (device_create_file(&pdev->dev, &dev_attr_stop)) ||
        (device_create_file(&pdev->dev, &dev_attr_halt)) ||
        (device_create_file(&pdev->dev, &dev_attr_dump)) ||
        (device_create_file(&pdev->dev, &dev_attr_ioctl)))
    {
        iounmap(virt_addr);
        kfree(pdev->dev.platform_data);
        misc_deregister(&stmotors[motor_id].miscdev);
        printk(KERN_ERR "%s: device_create_file failed\n", __func__);
    }
    return retval;
}


static int stepper_mod_platform_remove(struct platform_device *pdev)
{
    uint32_t            motor_id;
    struct device_node  *node = pdev->dev.of_node;

    if (of_property_read_u32(node, "motorid", &motor_id))
    {
        dev_err(&pdev->dev, "reading motorid failed\n");
        return -EINVAL;
    }
    mutex_destroy(&stmotors[motor_id].sem);

    kfree(pdev->dev.platform_data);     // get rid of platform data

    // get rid of the device file
    misc_deregister(&stmotors[motor_id].miscdev);
    // free our use of the irq for the system
    free_irq(stmotors[motor_id].stmotor_irq_num, (void *)motor_id);

    iounmap(stmotors[motor_id].stmotor_regs);
    //
    // get rid of the debug files.
    //
    device_remove_file(&pdev->dev, &dev_attr_help);
    device_remove_file(&pdev->dev, &dev_attr_move);
    device_remove_file(&pdev->dev, &dev_attr_pwm_m);
    device_remove_file(&pdev->dev, &dev_attr_stop);
    device_remove_file(&pdev->dev, &dev_attr_halt);
    device_remove_file(&pdev->dev, &dev_attr_dump);
    device_remove_file(&pdev->dev, &dev_attr_ioctl);

    return (0);
}

static int stepper_mod_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    printk(KERN_ERR "%s: driver platform suspend begun\n", __func__);
    return (0);
}

static int stepper_mod_platform_resume(struct platform_device *pdev)
{
    printk(KERN_ERR "%s: driver platform resume begun\n", __func__);
    return(0);
}


static const struct of_device_id mrvl_stepper_mod_dt_match[] = {

#if ASIC_REV <= ASIC_REV_A0
    { .compatible = "mrvl,stepper-mod-a0"},
    { .compatible = "mrvl,stepper-a0"},
#else
    { .compatible = "mrvl,stepper-mod-b0"},
    { .compatible = "mrvl,stepper-bo"},
#endif

    //{ .compatible = "mrvl,stepper-mod"},    // the 3d linux build uses these
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_stepper_mod_dt_match);

static struct platform_driver stepper_mod_platform_driver =
{
    .probe   = stepper_mod_platform_probe,
    .remove  = stepper_mod_platform_remove,
    .suspend = stepper_mod_platform_suspend,
    .resume  = stepper_mod_platform_resume,
    .driver  = {
        .name  = DRIVER_NAME,
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_stepper_mod_dt_match),
    }
};


/**
 * stepper_mod_platform_init: module init
 *
 * Driver load
 */
static int __init stepper_mod_platform_init(void)
{
    int retval;

    printk(KERN_ERR "%s: driver platform init begun (%s)\n", __func__, ASIC_NAME_STR);

    retval = smot_step_init(STEPPER_NUM_MOTORS);
    if (retval)
    {
        printk(KERN_ERR "%s: error initializing driver (%s)\n", __func__, ASIC_NAME_STR);
    }
    else
    {
        retval = platform_driver_register(&stepper_mod_platform_driver);
        if (retval)
        {
            printk(KERN_ERR "%s: error registering platform driver (%s)\n", __func__, ASIC_NAME_STR);
            return -ENXIO;
        }
        else
        {
            printk(KERN_ERR "%s: platform registration complete (%s)\n", __func__, ASIC_NAME_STR);
        }
        //
        // setup the stuff needed by the status thread.
        //
        init_waitqueue_head(&wait_queue);       // what it will wait on.
        memset(&status_device, 0, sizeof(status_device));
        status_device.minor =  0x40 + 0xf;
        status_device.name  = STATUS_FNAME;
        status_device.fops  = &status_fops;
        if(misc_register(&status_device))
        {
            printk(KERN_ERR "%s: error registering status_device (%s)\n", __func__, ASIC_NAME_STR);
            return -ENXIO;
        }

    }

    return retval;
}

module_init(stepper_mod_platform_init);

/**
 * stepper_mod_platform_exit: module exit
 *
 * Driver unload
 */
static void __exit stepper_mod_platform_exit(void)
{
    printk(KERN_ERR "%s: driver platform exit begun\n", __func__);

    //
    // get rid of the status device file
    misc_deregister(&status_device);
    //
    // get rid of everything else.  This will cause the remove to be called to get rid of each motor.
    //
    platform_driver_unregister(&stepper_mod_platform_driver);       

    // now get rid of the data structures.
    kfree(stmotors);

   
    //stepper_onetime_cleanup();
    printk(KERN_ERR "%s: platform remove complete\n", __func__);
}
module_exit(stepper_mod_platform_exit);

MODULE_AUTHOR("Copyright (c) 2015 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell G2 Stepper Driver CDMA");

MODULE_LICENSE("GPL");
MODULE_VERSION("2015_Aug_04");
