/** @file  mlanconfig.c
  *
  * @brief Program to configure addition parameters into the mlandriver
  *
  * Copyright (C) 2008-2014, Marvell International Ltd.
  *
  * This software file (the "File") is distributed by Marvell International
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991
  * (the "License").  You may use, redistribute and/or modify this File in
  * accordance with the terms and conditions of the License, a copy of which
  * is available by writing to the Free Software Foundation, Inc.,
  * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA or on the
  * worldwide web at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about
  * this warranty disclaimer.
  *
  */
/************************************************************************
Change log:
     11/26/2008: initial version
     03/10/2009:  add setuserscan, getscantable etc. commands
     08/11/2009:  add addts, regclass, setra, scanagent etc. commands
************************************************************************/

#include    "mlanconfig.h"
#include    "mlanhostcmd.h"
#include    "mlanmisc.h"

/** mlanconfig version number */
#define MLANCONFIG_VER "M2.0"

/** Initial number of total private ioctl calls */
#define IW_INIT_PRIV_NUM    128
/** Maximum number of total private ioctl calls supported */
#define IW_MAX_PRIV_NUM     1024

/********************************************************
		Local Variables
********************************************************/

/** Private ioctl commands */
enum COMMANDS {
	CMD_HOSTCMD,
	CMD_ARPFILTER,
	CMD_CFG_DATA,
	CMD_CMD52RW,
	CMD_CMD53RW,
	CMD_GET_SCAN_RSP,
	CMD_SET_USER_SCAN,
	CMD_ADD_TS,
	CMD_DEL_TS,
	CMD_QCONFIG,
	CMD_QSTATS,
	CMD_TS_STATUS,
	CMD_WMM_QSTATUS,
	CMD_REGRW,
	CMD_MEMRW,
	CMD_STA_CUSTOM_IE,
	CMD_STA_MGMT_FRAME_TX,
};

static t_s8 *commands[] = {
	"hostcmd",
	"arpfilter",
	"cfgdata",
	"sdcmd52rw",
	"sdcmd53rw",
	"getscantable",
	"setuserscan",
	"addts",
	"delts",
	"qconfig",
	"qstats",
	"ts_status",
	"qstatus",
	"regrdwr",
	"memrdwr",
	"customie",
	"mgmtframetx",
};

static t_s8 *usage[] = {
	"Usage: ",
	"   mlanconfig -v  (version)",
	"   mlanconfig <mlanX> <cmd> [...]",
	"   where",
	"   mlanX : wireless network interface",
	"   cmd : hostcmd",
	"     : customie",
	"     : mgmtframetx",
	"     : arpfilter",
	"     : cfgdata",
	"     : sdcmd52rw, sdcmd53rw",
	"     : getscantable, setuserscan",
	"     : addts, delts, qconfig, qstats, ts_status, qstatus",
	"     : regrdwr, memrdwr",
	"     : additional parameter for hostcmd",
	"     :   <filename> <cmd>",
	"      : additional parameters for customie are:",
	"      :  <index> <mask> <IE buffer>",
	"      : additional parameters for mgmtframetx are:",
	"      : <pkt file>",
	"     : additional parameter for arpfilter",
	"     :   <filename>",
	"     : additional parameter for cfgdata",
	"     :   <type> <filename>",
	"     : additional parameter for sdcmd52rw",
	"     :   <function> <reg addr.> <data>",
	"     : additional parameter for sdcmd53rw",
	"     :   <func> <addr> <mode> <blksiz> <blknum> <data1> ... ...<dataN> ",
	"     : additional parameter for addts",
	"     :   <filename.conf> <section# of tspec> <timeout in ms>",
	"     : additional parameter for delts",
	"     :   <filename.conf> <section# of tspec>",
	"     : additional parameter for qconfig",
	"     :   <[set msdu <lifetime in TUs> [Queue Id: 0-3]]",
	"     :    [get [Queue Id: 0-3]] [def [Queue Id: 0-3]]>",
	"     : additional parameter for qstats",
	"     :   <[get [User Priority: 0-7]]>",
	"     : additional parameter for regrdwr",
	"     :   <type> <offset> [value]",
	"     : additional parameter for memrdwr",
	"     :   <address> [value]",
};

t_s32 sockfd;  /**< socket */
t_s8 dev_name[IFNAMSIZ + 1];	/**< device name */
static struct iw_priv_args *priv_args = NULL;	   /**< private args */
static int we_version_compiled = 0;
				  /**< version compiled */

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/
/**
 *    @brief isdigit for String.
 *
 *    @param x            Char string
 *    @return             MLAN_STATUS_FAILURE for non-digit.
 *                        MLAN_STATUS_SUCCESS for digit
 */
static int
ISDIGIT(t_s8 *x)
{
	unsigned int i;
	for (i = 0; i < strlen(x); i++)
		if (isdigit(x[i]) == 0)
			return MLAN_STATUS_FAILURE;
	return MLAN_STATUS_SUCCESS;
}

/**
 * Check of decimal or hex string
 * @param   num string
 */
#define IS_HEX_OR_DIGIT(num) \
    (strncasecmp("0x", (num), 2)?ISDIGIT((num)):ishexstring((num)))

/**
 *  @brief Get private info.
 *
 *  @param ifname   A pointer to net name
 *  @return 	    MLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int
get_private_info(const t_s8 *ifname)
{
	/* This function sends the SIOCGIWPRIV command, which is handled by the
	   kernel and gets the total number of private ioctl's available in the
	   host driver. */
	struct iwreq iwr;
	int s, ret = MLAN_STATUS_SUCCESS;
	struct iw_priv_args *ppriv = NULL;
	struct iw_priv_args *new_priv;
	int result = 0;
	size_t size = IW_INIT_PRIV_NUM;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		perror("socket[PF_INET,SOCK_DGRAM]");
		return MLAN_STATUS_FAILURE;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ - 1);

	do {
		/* (Re)allocate the buffer */
		new_priv = realloc(ppriv, size * sizeof(ppriv[0]));
		if (new_priv == NULL) {
			printf("Error: Buffer allocation failed\n");
			ret = MLAN_STATUS_FAILURE;
			break;
		}
		ppriv = new_priv;

		iwr.u.data.pointer = (caddr_t) ppriv;
		iwr.u.data.length = size;
		iwr.u.data.flags = 0;

		if (ioctl(s, SIOCGIWPRIV, &iwr) < 0) {
			result = errno;
			ret = MLAN_STATUS_FAILURE;
			if (result == E2BIG) {
				/* We need a bigger buffer. Check if kernel
				   gave us any hints. */
				if (iwr.u.data.length > size) {
					/* Kernel provided required size */
					size = iwr.u.data.length;
				} else {
					/* No hint from kernel, double the
					   buffer size */
					size *= 2;
				}
			} else {
				/* ioctl error */
				perror("ioctl[SIOCGIWPRIV]");
				break;
			}
		} else {
			/* Success. Return the number of private ioctls */
			priv_args = ppriv;
			ret = iwr.u.data.length;
			break;
		}
	} while (size <= IW_MAX_PRIV_NUM);

	if ((ret == MLAN_STATUS_FAILURE) && (ppriv))
		free(ppriv);

	close(s);

	return ret;
}

/**
 *  @brief Get Sub command ioctl number
 *
 *  @param i        command index
 *  @param priv_cnt     Total number of private ioctls availabe in driver
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *  @return 	        MLAN_STATUS_SUCCESS or MLAN_STATUS_FAILURE
 */
static int
marvell_get_subioctl_no(t_s32 i,
			t_s32 priv_cnt, int *ioctl_val, int *subioctl_val)
{
	t_s32 j;

	if (priv_args[i].cmd >= SIOCDEVPRIVATE) {
		*ioctl_val = priv_args[i].cmd;
		*subioctl_val = 0;
		return MLAN_STATUS_SUCCESS;
	}

	j = -1;

	/* Find the matching *real* ioctl */

	while ((++j < priv_cnt)
	       && ((priv_args[j].name[0] != '\0') ||
		   (priv_args[j].set_args != priv_args[i].set_args) ||
		   (priv_args[j].get_args != priv_args[i].get_args))) {
	}

	/* If not found... */
	if (j == priv_cnt) {
		printf("%s: Invalid private ioctl definition for: 0x%x\n",
		       dev_name, priv_args[i].cmd);
		return MLAN_STATUS_FAILURE;
	}

	/* Save ioctl numbers */
	*ioctl_val = priv_args[j].cmd;
	*subioctl_val = priv_args[i].cmd;

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Get ioctl number
 *
 *  @param ifname   	A pointer to net name
 *  @param priv_cmd	A pointer to priv command buffer
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *  @return 	        MLAN_STATUS_SUCCESS or MLAN_STATUS_FAILURE
 */
static int
marvell_get_ioctl_no(const t_s8 *ifname,
		     const t_s8 *priv_cmd, int *ioctl_val, int *subioctl_val)
{
	t_s32 i;
	t_s32 priv_cnt;
	int ret = MLAN_STATUS_FAILURE;

	priv_cnt = get_private_info(ifname);

	/* Are there any private ioctls? */
	if (priv_cnt <= 0 || priv_cnt > IW_MAX_PRIV_NUM) {
		/* Could skip this message ? */
		printf("%-8.8s  no private ioctls.\n", ifname);
	} else {
		for (i = 0; i < priv_cnt; i++) {
			if (priv_args[i].name[0] &&
			    !strcmp(priv_args[i].name, priv_cmd)) {
				ret = marvell_get_subioctl_no(i, priv_cnt,
							      ioctl_val,
							      subioctl_val);
				break;
			}
		}
	}

	if (priv_args) {
		free(priv_args);
		priv_args = NULL;
	}

	return ret;
}

/**
 *  @brief Retrieve the ioctl and sub-ioctl numbers for the given ioctl string
 *
 *  @param ioctl_name   Private IOCTL string name
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *
 *  @return             MLAN_STATUS_SUCCESS or MLAN_STATUS_FAILURE
 */
int
get_priv_ioctl(char *ioctl_name, int *ioctl_val, int *subioctl_val)
{
	int retval;

	retval = marvell_get_ioctl_no(dev_name,
				      ioctl_name, ioctl_val, subioctl_val);

	return retval;
}

/**
 *  @brief Process host_cmd
 *  @param argc		number of arguments
 *  @param argv		A pointer to arguments array
 *  @return      	MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_host_cmd(int argc, char *argv[])
{
	t_s8 cmdname[256];
	t_u8 *buf;
	HostCmd_DS_GEN *hostcmd;
	struct iwreq iwr;
	int ret = MLAN_STATUS_SUCCESS;
	int ioctl_val, subioctl_val;
	FILE *fp = NULL;

	if (get_priv_ioctl("hostcmd",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	if (argc < 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./mlanconfig mlanX hostcmd <hostcmd.conf> <cmdname>\n");
		exit(1);
	}

	if ((fp = fopen(argv[3], "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[3]);
		exit(1);
	}

	buf = (t_u8 *)malloc(MRVDRV_SIZE_OF_CMD_BUFFER);
	if (buf == NULL) {
		printf("Error: allocate memory for hostcmd failed\n");
		fclose(fp);
		return -ENOMEM;
	}
	snprintf(cmdname, sizeof(cmdname), "%s", argv[4]);
	ret = prepare_host_cmd_buffer(fp, cmdname, buf);
	fclose(fp);

	if (ret == MLAN_STATUS_FAILURE)
		goto _exit_;

	hostcmd = (HostCmd_DS_GEN *)buf;
	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = (t_u8 *)hostcmd;
	iwr.u.data.length = le16_to_cpu(hostcmd->size);

	iwr.u.data.flags = 0;
	if (ioctl(sockfd, ioctl_val, &iwr)) {
		fprintf(stderr,
			"mlanconfig: MLANHOSTCMD is not supported by %s\n",
			dev_name);
		ret = MLAN_STATUS_FAILURE;
		goto _exit_;
	}
	ret = process_host_cmd_resp(buf);

_exit_:
	if (buf)
		free(buf);

	return ret;
}

/**
 *  @brief  get range
 *
 *  @return	MLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int
get_range(t_void)
{
	struct iw_range *range;
	struct iwreq iwr;
	size_t buf_len;

	buf_len = sizeof(struct iw_range) + 500;
	range = malloc(buf_len);
	if (range == NULL) {
		printf("Error: allocate memory for iw_range failed\n");
		return -ENOMEM;
	}
	memset(range, 0, buf_len);
	memset(&iwr, 0, sizeof(struct iwreq));
	iwr.u.data.pointer = (caddr_t) range;
	iwr.u.data.length = buf_len;
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);

	if ((ioctl(sockfd, SIOCGIWRANGE, &iwr)) < 0) {
		printf("Get Range Results Failed\n");
		free(range);
		return MLAN_STATUS_FAILURE;
	}
	we_version_compiled = range->we_version_compiled;
	printf("Driver build with Wireless Extension %d\n",
	       range->we_version_compiled);
	free(range);
	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Display usage
 *
 *  @return       NA
 */
static t_void
display_usage(t_void)
{
	t_u32 i;
	for (i = 0; i < NELEMENTS(usage); i++)
		fprintf(stderr, "%s\n", usage[i]);
}

/**
 *  @brief Find command
 *
 *  @param maxcmds  max command number
 *  @param cmds     A pointer to commands buffer
 *  @param cmd      A pointer to command buffer
 *  @return         index of command or MLAN_STATUS_FAILURE
 */
static int
findcommand(t_s32 maxcmds, t_s8 *cmds[], t_s8 *cmd)
{
	t_s32 i;

	for (i = 0; i < maxcmds; i++) {
		if (!strcasecmp(cmds[i], cmd)) {
			return i;
		}
	}

	return MLAN_STATUS_FAILURE;
}

/**
 *  @brief Process arpfilter
 *  @param argc   number of arguments
 *  @param argv   A pointer to arguments array
 *  @return     MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_arpfilter(int argc, char *argv[])
{
	t_u8 *buf;
	struct iwreq iwr;
	t_u16 length = 0;
	int ret = MLAN_STATUS_SUCCESS;
	FILE *fp = NULL;
	int ioctl_val, subioctl_val;

	if (get_priv_ioctl("arpfilter",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	if (argc < 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./mlanconfig mlanX arpfilter <arpfilter.conf>\n");
		exit(1);
	}

	if ((fp = fopen(argv[3], "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[3]);
		return MLAN_STATUS_FAILURE;
	}
	buf = (t_u8 *)malloc(MRVDRV_SIZE_OF_CMD_BUFFER);
	if (buf == NULL) {
		printf("Error: allocate memory for arpfilter failed\n");
		fclose(fp);
		return -ENOMEM;
	}
	ret = prepare_arp_filter_buffer(fp, buf, &length);
	fclose(fp);

	if (ret == MLAN_STATUS_FAILURE)
		goto _exit_;

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = buf;
	iwr.u.data.length = length;
	iwr.u.data.flags = 0;
	if (ioctl(sockfd, ioctl_val, &iwr)) {
		fprintf(stderr,
			"mlanconfig: arpfilter command is not supported by %s\n",
			dev_name);
		ret = MLAN_STATUS_FAILURE;
		goto _exit_;
	}

_exit_:
	if (buf)
		free(buf);

	return ret;
}

/**
 *  @brief Process cfgdata
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_cfg_data(int argc, char *argv[])
{
	t_u8 *buf;
	HostCmd_DS_GEN *hostcmd;
	struct iwreq iwr;
	int ret = MLAN_STATUS_SUCCESS;
	int ioctl_val, subioctl_val;
	FILE *fp = NULL;

	if (get_priv_ioctl("hostcmd",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	if (argc < 4 || argc > 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./mlanconfig mlanX cfgdata <register type> <filename>\n");
		exit(1);
	}

	if (argc == 5) {
		if ((fp = fopen(argv[4], "r")) == NULL) {
			fprintf(stderr, "Cannot open file %s\n", argv[3]);
			exit(1);
		}
	}
	buf = (t_u8 *)malloc(MRVDRV_SIZE_OF_CMD_BUFFER);
	if (buf == NULL) {
		printf("Error: allocate memory for hostcmd failed\n");
		if (argc == 5)
			fclose(fp);
		return -ENOMEM;
	}
	ret = prepare_cfg_data_buffer(argc, argv, fp, buf);
	if (argc == 5)
		fclose(fp);

	if (ret == MLAN_STATUS_FAILURE)
		goto _exit_;

	hostcmd = (HostCmd_DS_GEN *)buf;
	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = (t_u8 *)hostcmd;
	iwr.u.data.length = le16_to_cpu(hostcmd->size);

	iwr.u.data.flags = 0;
	if (ioctl(sockfd, ioctl_val, &iwr)) {
		fprintf(stderr,
			"mlanconfig: MLANHOSTCMD is not supported by %s\n",
			dev_name);
		ret = MLAN_STATUS_FAILURE;
		goto _exit_;
	}
	ret = process_host_cmd_resp(buf);

_exit_:
	if (buf)
		free(buf);

	return ret;
}

/**
 *  @brief read current command
 *  @param ptr      A pointer to data
 *  @param curCmd   A pointer to the buf which will hold current command
 *  @return         NULL or the pointer to the left command buf
 */
static t_s8 *
readCurCmd(t_s8 *ptr, t_s8 *curCmd)
{
	t_s32 i = 0;
#define MAX_CMD_SIZE 64	/**< Max command size */

	while (*ptr != ']' && i < (MAX_CMD_SIZE - 1))
		curCmd[i++] = *(++ptr);

	if (*ptr != ']')
		return NULL;

	curCmd[i - 1] = '\0';

	return ++ptr;
}

/**
 *  @brief parse command and hex data
 *  @param fp       A pointer to FILE stream
 *  @param dst      A pointer to the dest buf
 *  @param cmd      A pointer to command buf for search
 *  @return         Length of hex data or MLAN_STATUS_FAILURE
 */
static int
fparse_for_cmd_and_hex(FILE * fp, t_u8 *dst, t_u8 *cmd)
{
	t_s8 *ptr;
	t_u8 *dptr;
	t_s8 buf[256], curCmd[64];
	t_s32 isCurCmd = 0;

	dptr = dst;
	while (fgets(buf, sizeof(buf), fp)) {
		ptr = buf;

		while (*ptr) {
			/* skip leading spaces */
			while (*ptr && isspace(*ptr))
				ptr++;

			/* skip blank lines and lines beginning with '#' */
			if (*ptr == '\0' || *ptr == '#')
				break;

			if (*ptr == '[' && *(ptr + 1) != '/') {
				if (!(ptr = readCurCmd(ptr, curCmd)))
					return MLAN_STATUS_FAILURE;

				if (strcasecmp(curCmd, (char *)cmd))	/* Not
									   equal
									 */
					isCurCmd = 0;
				else
					isCurCmd = 1;
			}

			/* Ignore the rest if it is not correct cmd */
			if (!isCurCmd)
				break;

			if (*ptr == '[' && *(ptr + 1) == '/')
				return (dptr - dst);

			if (isxdigit(*ptr)) {
				ptr = convert2hex(ptr, dptr++);
			} else {
				/* Invalid character on data line */
				ptr++;
			}
		}
	}

	return MLAN_STATUS_FAILURE;
}

/**
 *  @brief Send an ADDTS command to the associated AP
 *
 *  Process a given conf file for a specific TSPEC data block.  Send the
 *    TSPEC along with any other IEs to the driver/firmware for transmission
 *    in an ADDTS request to the associated AP.
 *
 *  Return the execution status of the command as well as the ADDTS response
 *    from the AP if any.
 *
 *  mlanconfig mlanX addts <filename.conf> <section# of tspec> <timeout in ms>
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_addts(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_addts_req_t addtsReq;

	FILE *fp = NULL;
	char filename[48];
	char config_id[20];

	memset(&addtsReq, 0x00, sizeof(addtsReq));
	memset(filename, 0x00, sizeof(filename));

	if (get_priv_ioctl("addts",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	if (argc != 6) {
		fprintf(stderr, "Invalid number of parameters!\n");
		return -EINVAL;
	}

	strncpy(filename, argv[3], MIN(sizeof(filename) - 1, strlen(argv[3])));
	if ((fp = fopen(filename, "r")) == NULL) {
		perror("fopen");
		fprintf(stderr, "Cannot open file %s\n", argv[3]);
		return -EFAULT;;
	}

	snprintf(config_id, sizeof(config_id), "tspec%d", atoi(argv[4]));

	addtsReq.ieDataLen = fparse_for_cmd_and_hex(fp,
						    addtsReq.ieData,
						    (t_u8 *)config_id);

	if (addtsReq.ieDataLen > 0) {
		printf("Found %d bytes in the %s section of conf file %s\n",
		       (int)addtsReq.ieDataLen, config_id, filename);
	} else {
		fprintf(stderr, "section %s not found in %s\n",
			config_id, filename);
		if (fp)
			fclose(fp);
		return -EFAULT;
	}

	addtsReq.timeout_ms = atoi(argv[5]);

	printf("Cmd Input:\n");
	hexdump(config_id, addtsReq.ieData, addtsReq.ieDataLen, ' ');

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.flags = subioctl_val;
	iwr.u.data.pointer = (caddr_t) & addtsReq;
	iwr.u.data.length = (sizeof(addtsReq)
			     - sizeof(addtsReq.ieData)
			     + addtsReq.ieDataLen);

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("mlanconfig: addts ioctl");
		if (fp)
			fclose(fp);
		return -EFAULT;
	}

	printf("Cmd Output:\n");
	printf("ADDTS Command Result = %d\n", addtsReq.commandResult);
	printf("ADDTS IEEE Status    = %d\n", addtsReq.ieeeStatusCode);
	hexdump(config_id, addtsReq.ieData, addtsReq.ieDataLen, ' ');

	if (fp)
		fclose(fp);

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Send a DELTS command to the associated AP
 *
 *  Process a given conf file for a specific TSPEC data block.  Send the
 *    TSPEC along with any other IEs to the driver/firmware for transmission
 *    in a DELTS request to the associated AP.
 *
 *  Return the execution status of the command.  There is no response to a
 *    DELTS from the AP.
 *
 *  mlanconfig mlanX delts <filename.conf> <section# of tspec>
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_delts(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_delts_req_t deltsReq;

	FILE *fp = NULL;
	char filename[48];
	char config_id[20];

	memset(&deltsReq, 0x00, sizeof(deltsReq));
	memset(filename, 0x00, sizeof(filename));

	if (get_priv_ioctl("delts",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	if (argc != 5) {
		fprintf(stderr, "Invalid number of parameters!\n");
		return -EINVAL;
	}

	strncpy(filename, argv[3], MIN(sizeof(filename) - 1, strlen(argv[3])));
	if ((fp = fopen(filename, "r")) == NULL) {
		perror("fopen");
		fprintf(stderr, "Cannot open file %s\n", argv[3]);
		return -EFAULT;
	}

	snprintf(config_id, sizeof(config_id), "tspec%d", atoi(argv[4]));

	deltsReq.ieDataLen = fparse_for_cmd_and_hex(fp,
						    deltsReq.ieData,
						    (t_u8 *)config_id);

	if (deltsReq.ieDataLen > 0) {
		printf("Found %d bytes in the %s section of conf file %s\n",
		       (int)deltsReq.ieDataLen, config_id, filename);
	} else {
		fprintf(stderr, "section %s not found in %s\n",
			config_id, filename);
		if (fp)
			fclose(fp);
		return -EFAULT;
	}

	printf("Cmd Input:\n");
	hexdump(config_id, deltsReq.ieData, deltsReq.ieDataLen, ' ');

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.flags = subioctl_val;
	iwr.u.data.pointer = (caddr_t) & deltsReq;
	iwr.u.data.length = (sizeof(deltsReq)
			     - sizeof(deltsReq.ieData)
			     + deltsReq.ieDataLen);

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("mlanconfig: delts ioctl");
		if (fp)
			fclose(fp);
		return -EFAULT;
	}

	printf("Cmd Output:\n");
	printf("DELTS Command Result = %d\n", deltsReq.commandResult);
	if (fp)
		fclose(fp);

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Send a WMM AC Queue configuration command to get/set/default params
 *
 *  Configure or get the parameters of a WMM AC queue. The command takes
 *    an optional Queue Id as a last parameter.  Without the queue id, all
 *    queues will be acted upon.
 *
 *  mlanconfig mlanX qconfig set msdu <lifetime in TUs> [Queue Id: 0-3]
 *  mlanconfig mlanX qconfig get [Queue Id: 0-3]
 *  mlanconfig mlanX qconfig def [Queue Id: 0-3]
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_qconfig(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_queue_config_t queue_config_cmd;
	mlan_wmm_ac_e ac_idx;
	mlan_wmm_ac_e ac_idx_start;
	mlan_wmm_ac_e ac_idx_stop;

	const char *ac_str_tbl[] = { "BK", "BE", "VI", "VO" };

	if (argc < 4) {
		fprintf(stderr, "Invalid number of parameters!\n");
		return -EINVAL;
	}

	if (get_priv_ioctl("qconfig",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	memset(&queue_config_cmd, 0x00, sizeof(queue_config_cmd));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = (caddr_t) & queue_config_cmd;
	iwr.u.data.length = sizeof(queue_config_cmd);
	iwr.u.data.flags = subioctl_val;

	if (strcmp(argv[3], "get") == 0) {
		/* 3 4 5 */
		/* qconfig get [qid] */
		if (argc == 4) {
			ac_idx_start = WMM_AC_BK;
			ac_idx_stop = WMM_AC_VO;
		} else if (argc == 5) {
			if (atoi(argv[4]) < WMM_AC_BK ||
			    atoi(argv[4]) > WMM_AC_VO) {
				fprintf(stderr, "ERROR: Invalid Queue ID!\n");
				return -EINVAL;
			}
			ac_idx_start = atoi(argv[4]);
			ac_idx_stop = ac_idx_start;
		} else {
			fprintf(stderr, "Invalid number of parameters!\n");
			return -EINVAL;
		}
		queue_config_cmd.action = WMM_QUEUE_CONFIG_ACTION_GET;

		for (ac_idx = ac_idx_start; ac_idx <= ac_idx_stop; ac_idx++) {
			queue_config_cmd.accessCategory = ac_idx;
			if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
				perror("qconfig ioctl");
			} else {
				printf("qconfig %s(%d): MSDU Lifetime GET = 0x%04x (%d)\n", ac_str_tbl[ac_idx], ac_idx, queue_config_cmd.msduLifetimeExpiry, queue_config_cmd.msduLifetimeExpiry);
			}
		}
	} else if (strcmp(argv[3], "set") == 0) {
		if ((argc >= 5) && strcmp(argv[4], "msdu") == 0) {
			/* 3 4 5 6 7 */
			/* qconfig set msdu <value> [qid] */
			if (argc == 6) {
				ac_idx_start = WMM_AC_BK;
				ac_idx_stop = WMM_AC_VO;
			} else if (argc == 7) {
				if (atoi(argv[6]) < WMM_AC_BK ||
				    atoi(argv[6]) > WMM_AC_VO) {
					fprintf(stderr,
						"ERROR: Invalid Queue ID!\n");
					return -EINVAL;
				}
				ac_idx_start = atoi(argv[6]);
				ac_idx_stop = ac_idx_start;
			} else {
				fprintf(stderr,
					"Invalid number of parameters!\n");
				return -EINVAL;
			}
			queue_config_cmd.action = WMM_QUEUE_CONFIG_ACTION_SET;
			queue_config_cmd.msduLifetimeExpiry = atoi(argv[5]);

			for (ac_idx = ac_idx_start; ac_idx <= ac_idx_stop;
			     ac_idx++) {
				queue_config_cmd.accessCategory = ac_idx;
				if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
					perror("qconfig ioctl");
				} else {
					printf("qconfig %s(%d): MSDU Lifetime SET = 0x%04x (%d)\n", ac_str_tbl[ac_idx], ac_idx, queue_config_cmd.msduLifetimeExpiry, queue_config_cmd.msduLifetimeExpiry);
				}
			}
		} else {
			/* Only MSDU Lifetime provisioning accepted for now */
			fprintf(stderr, "Invalid set parameter: s/b [msdu]\n");
			return -EINVAL;
		}
	} else if (strncmp(argv[3], "def", strlen("def")) == 0) {
		/* 3 4 5 */
		/* qconfig def [qid] */
		if (argc == 4) {
			ac_idx_start = WMM_AC_BK;
			ac_idx_stop = WMM_AC_VO;
		} else if (argc == 5) {
			if (atoi(argv[4]) < WMM_AC_BK ||
			    atoi(argv[4]) > WMM_AC_VO) {
				fprintf(stderr, "ERROR: Invalid Queue ID!\n");
				return -EINVAL;
			}
			ac_idx_start = atoi(argv[4]);
			ac_idx_stop = ac_idx_start;
		} else {
			fprintf(stderr, "Invalid number of parameters!\n");
			return -EINVAL;
		}
		queue_config_cmd.action = WMM_QUEUE_CONFIG_ACTION_DEFAULT;

		for (ac_idx = ac_idx_start; ac_idx <= ac_idx_stop; ac_idx++) {
			queue_config_cmd.accessCategory = ac_idx;
			if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
				perror("qconfig ioctl");
			} else {
				printf("qconfig %s(%d): MSDU Lifetime DEFAULT = 0x%04x (%d)\n", ac_str_tbl[ac_idx], ac_idx, queue_config_cmd.msduLifetimeExpiry, queue_config_cmd.msduLifetimeExpiry);
			}
		}
	} else {
		fprintf(stderr,
			"Invalid qconfig command; s/b [set, get, default]\n");
		return -EINVAL;
	}

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Turn on/off or retrieve and clear the queue statistics for a UP
 *
 *  Turn the statistics collection on/off for a given UP or retrieve the
 *    current accumulated stats and clear them from the firmware.  The command
 *    takes an optional Queue Id as a last parameter.  Without the queue id,
 *    all queues will be acted upon.
 *
 *  mlanconfig mlanX qstats get [User Priority: 0-7]
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_qstats(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_queue_stats_t queue_stats_cmd;
	t_u8 up_idx;
	t_u8 up_idx_start;
	t_u8 up_idx_stop;
	t_u16 usedTime[MAX_USER_PRIORITIES];
	t_u16 policedTime[MAX_USER_PRIORITIES];

	const char *ac_str_tbl[] = { "BE", "BK", "BK", "BE",
		"VI", "VI", "VO", "VO"
	};

	if (argc < 3) {
		fprintf(stderr, "Invalid number of parameters!\n");
		return -EINVAL;
	}

	if (get_priv_ioctl("qstats",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	printf("\n");

	memset(usedTime, 0x00, sizeof(usedTime));
	memset(policedTime, 0x00, sizeof(policedTime));
	memset(&queue_stats_cmd, 0x00, sizeof(queue_stats_cmd));

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = (caddr_t) & queue_stats_cmd;
	iwr.u.data.length = sizeof(queue_stats_cmd);
	iwr.u.data.flags = subioctl_val;

	if ((argc > 3) && strcmp(argv[3], "on") == 0) {
		if (argc == 4) {
			up_idx_start = 0;
			up_idx_stop = 7;
		} else if (argc == 5) {
			if (atoi(argv[4]) < 0 || atoi(argv[4]) > 7) {
				fprintf(stderr,
					"ERROR: Invalid User Priority!\n");
				return -EINVAL;
			}
			up_idx_start = atoi(argv[4]);
			up_idx_stop = up_idx_start;
		} else {
			fprintf(stderr, "Invalid number of parameters!\n");
			return -EINVAL;
		}
		queue_stats_cmd.action = WMM_STATS_ACTION_START;
		for (up_idx = up_idx_start; up_idx <= up_idx_stop; up_idx++) {
			queue_stats_cmd.userPriority = up_idx;
			if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
				perror("qstats ioctl");
			} else {
				printf("qstats UP%d, %s turned on\n",
				       up_idx, ac_str_tbl[up_idx]);
			}
		}
	} else if ((argc > 3) && strcmp(argv[3], "off") == 0) {
		if (argc == 4) {
			up_idx_start = 0;
			up_idx_stop = 7;
		} else if (argc == 5) {
			if (atoi(argv[4]) < 0 || atoi(argv[4]) > 7) {
				fprintf(stderr,
					"ERROR: Invalid User Priority!\n");
				return -EINVAL;
			}
			up_idx_start = atoi(argv[4]);
			up_idx_stop = up_idx_start;
		} else {
			fprintf(stderr, "Invalid number of parameters!\n");
			return -EINVAL;
		}
		queue_stats_cmd.action = WMM_STATS_ACTION_STOP;
		for (up_idx = up_idx_start; up_idx <= up_idx_stop; up_idx++) {
			queue_stats_cmd.userPriority = up_idx;
			if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
				perror("qstats ioctl");
			} else {
				printf("qstats UP%d, %s turned off\n",
				       up_idx, ac_str_tbl[up_idx]);
			}
		}
	} else if ((argc >= 3) &&
		   ((argc == 3) ? 1 : (strcmp(argv[3], "get") == 0))) {
		/* If the user types: "mlanconfig mlanX qstats" without get
		   argument.  The mlanconfig application invokes "get" option
		   for all UPs */
		if ((argc == 4) || (argc == 3)) {
			up_idx_start = 0;
			up_idx_stop = 7;
		} else if (argc == 5) {
			if (atoi(argv[4]) < 0 || atoi(argv[4]) > 7) {
				fprintf(stderr,
					"ERROR: Invalid User Priority!\n");
				return -EINVAL;
			}
			up_idx_start = atoi(argv[4]);
			up_idx_stop = up_idx_start;
		} else {
			fprintf(stderr, "Invalid number of parameters!\n");
			return -EINVAL;
		}
		printf("%s %6s %5s %8s %8s %6s %6s %6s %6s %6s %6s\n",
		       "AC-UP", "Count", "Loss", "TxDly", "QDly",
		       "<=5", "<=10", "<=20", "<=30", "<=50", ">50");
		printf("----------------------------------"
		       "---------------------------------------------\n");
		queue_stats_cmd.action = WMM_STATS_ACTION_GET_CLR;

		for (up_idx = up_idx_start; up_idx <= up_idx_stop; up_idx++) {
			queue_stats_cmd.userPriority = up_idx;
			if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
				perror("qstats ioctl");
			} else {
				printf(" %s-%d %6u %5u %8u %8u %6u %6u %6u %6u %6u %6u\n", ac_str_tbl[up_idx], up_idx, queue_stats_cmd.pktCount, queue_stats_cmd.pktLoss, (unsigned int)queue_stats_cmd.avgTxDelay, (unsigned int)queue_stats_cmd.avgQueueDelay, queue_stats_cmd.delayHistogram[0], queue_stats_cmd.delayHistogram[1], queue_stats_cmd.delayHistogram[2], queue_stats_cmd.delayHistogram[3], (queue_stats_cmd.delayHistogram[4] + queue_stats_cmd.delayHistogram[5]), queue_stats_cmd.delayHistogram[6]);

				usedTime[up_idx] = queue_stats_cmd.usedTime;
				policedTime[up_idx] =
					queue_stats_cmd.policedTime;
			}
		}

		printf("----------------------------------"
		       "---------------------------------------------\n");
		printf("\nAC-UP      UsedTime      PolicedTime\n");
		printf("------------------------------------\n");

		for (up_idx = up_idx_start; up_idx <= up_idx_stop; up_idx++) {
			printf(" %s-%d        %6u           %6u\n",
			       ac_str_tbl[up_idx],
			       up_idx,
			       (unsigned int)usedTime[up_idx],
			       (unsigned int)policedTime[up_idx]);
		}
	} else {
		fprintf(stderr, "Invalid qstats command;\n");
		return -EINVAL;
	}
	printf("\n");

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Get the current status of the WMM Queues
 *
 *  Command: mlanconfig mlanX qstatus
 *
 *  Retrieve the following information for each AC if wmm is enabled:
 *        - WMM IE ACM Required
 *        - Firmware Flow Required
 *        - Firmware Flow Established
 *        - Firmware Queue Enabled
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_wmm_qstatus(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_queue_status_t qstatus;
	mlan_wmm_ac_e acVal;

	if (get_priv_ioctl("qstatus",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	memset(&qstatus, 0x00, sizeof(qstatus));

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.flags = subioctl_val;
	iwr.u.data.pointer = (caddr_t) & qstatus;
	iwr.u.data.length = (sizeof(qstatus));

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("mlanconfig: qstatus ioctl");
		return -EFAULT;
	}

	for (acVal = WMM_AC_BK; acVal <= WMM_AC_VO; acVal++) {
		switch (acVal) {
		case WMM_AC_BK:
			printf("BK: ");
			break;
		case WMM_AC_BE:
			printf("BE: ");
			break;
		case WMM_AC_VI:
			printf("VI: ");
			break;
		case WMM_AC_VO:
			printf("VO: ");
			break;
		default:
			printf("??: ");
		}

		printf("ACM[%c], FlowReq[%c], FlowCreated[%c], Enabled[%c],"
		       " DE[%c], TE[%c]\n",
		       (qstatus.acStatus[acVal].wmmAcm ? 'X' : ' '),
		       (qstatus.acStatus[acVal].flowRequired ? 'X' : ' '),
		       (qstatus.acStatus[acVal].flowCreated ? 'X' : ' '),
		       (qstatus.acStatus[acVal].disabled ? ' ' : 'X'),
		       (qstatus.acStatus[acVal].deliveryEnabled ? 'X' : ' '),
		       (qstatus.acStatus[acVal].triggerEnabled ? 'X' : ' '));
	}

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Get the current status of the WMM Traffic Streams
 *
 *  Command: mlanconfig mlanX ts_status
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_wmm_ts_status(int argc, char *argv[])
{
	int ioctl_val, subioctl_val;
	struct iwreq iwr;
	wlan_ioctl_wmm_ts_status_t ts_status;
	int tid;

	const char *ac_str_tbl[] = { "BK", "BE", "VI", "VO" };

	if (get_priv_ioctl("ts_status",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	printf("\nTID   Valid    AC   UP   PSB   FlowDir  MediumTime\n");
	printf("---------------------------------------------------\n");

	for (tid = 0; tid <= 7; tid++) {
		memset(&ts_status, 0x00, sizeof(ts_status));
		ts_status.tid = tid;

		strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
		iwr.u.data.flags = subioctl_val;
		iwr.u.data.pointer = (caddr_t) & ts_status;
		iwr.u.data.length = (sizeof(ts_status));

		if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
			perror("mlanconfig: ts_status ioctl");
			return -EFAULT;
		}

		printf(" %02d     %3s    %2s    %u     %c    ",
		       ts_status.tid,
		       (ts_status.valid ? "Yes" : "No"),
		       (ts_status.
			valid ? ac_str_tbl[ts_status.accessCategory] : "--"),
		       ts_status.userPriority, (ts_status.psb ? 'U' : 'L'));

		if ((ts_status.flowDir & 0x03) == 0) {
			printf("%s", " ---- ");
		} else {
			printf("%2s%4s",
			       (ts_status.flowDir & 0x01 ? "Up" : ""),
			       (ts_status.flowDir & 0x02 ? "Down" : ""));
		}

		printf("%12u\n", ts_status.mediumTime);
	}

	printf("\n");

	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Provides interface to perform read/write operations on regsiter
 *
 *  Command: mlanconfig mlanX regrdwr <type> <offset> [value]
 *
 *  @param argc     Number of arguments
 *  @param argv     Pointer to the arguments
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_regrdwr(int argc, char *argv[])
{
	struct iwreq iwr;
	int ioctl_val, subioctl_val;
	t_u32 type, offset, value;
	t_u8 buf[100];
	HostCmd_DS_GEN *hostcmd = (HostCmd_DS_GEN *)buf;
	int ret = MLAN_STATUS_SUCCESS;

	/* Check arguments */
	if ((argc < 5) || (argc > 6)) {
		printf("Parameters for regrdwr: <type> <offset> [value]\n");
		return -EINVAL;
	}

	if (get_priv_ioctl("hostcmd",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	type = a2hex_or_atoi(argv[3]);
	offset = a2hex_or_atoi(argv[4]);
	if (argc > 5)
		value = a2hex_or_atoi(argv[5]);
	if ((ret = prepare_hostcmd_regrdwr(type, offset,
					   (argc > 5) ? &value : NULL, buf))) {
		return ret;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = buf;
	iwr.u.data.length = le16_to_cpu(hostcmd->size);
	iwr.u.data.flags = 0;

	if (ioctl(sockfd, ioctl_val, &iwr)) {
		fprintf(stderr,
			"mlanconfig: MLANHOSTCMD is not supported by %s\n",
			dev_name);
		return MLAN_STATUS_FAILURE;
	}
	ret = process_host_cmd_resp(buf);

	return ret;
}

/**
 *  @brief Provides interface to perform read/write the adapter memory
 *
 *  Command: mlanconfig mlanX memrdwr <address> [value]
 *
 *  @param argc     Number of arguments
 *  @param argv     Pointer to the arguments
 *
 *  @return         MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_memrdwr(int argc, char *argv[])
{
	struct iwreq iwr;
	int ioctl_val, subioctl_val;
	t_u32 address, value;
	t_u8 buf[100] = { 0 };
	HostCmd_DS_MEM *pmem;
	HostCmd_DS_GEN *hostcmd = (HostCmd_DS_GEN *)buf;
	int ret = MLAN_STATUS_SUCCESS;

	pmem = (HostCmd_DS_MEM *)(buf + S_DS_GEN);

	/* Check arguments */
	if ((argc < 4) || (argc > 5)) {
		printf("Parameters for memrdwr: <address> [value]\n");
		return -EINVAL;
	}

	if (get_priv_ioctl("hostcmd",
			   &ioctl_val, &subioctl_val) == MLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
	}

	address = a2hex_or_atoi(argv[3]);
	pmem->addr = cpu_to_le32(address);
	if (argc > 4) {
		pmem->action = cpu_to_le16(HostCmd_ACT_GEN_SET);
		value = a2hex_or_atoi(argv[4]);
		pmem->value = cpu_to_le32(value);
	} else {
		pmem->action = cpu_to_le16(HostCmd_ACT_GEN_GET);
		pmem->value = 0;
	}
	hostcmd->command = cpu_to_le16(HostCmd_CMD_MEM_ACCESS);
	hostcmd->size = cpu_to_le16(S_DS_GEN + sizeof(HostCmd_DS_MEM));
	hostcmd->seq_num = 0;
	hostcmd->result = 0;

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ - 1);
	iwr.u.data.pointer = buf;
	iwr.u.data.length = le16_to_cpu(hostcmd->size);
	iwr.u.data.flags = 0;

	if (ioctl(sockfd, ioctl_val, &iwr)) {
		fprintf(stderr,
			"mlanconfig: MLANHOSTCMD is not supported by %s\n",
			dev_name);
		return MLAN_STATUS_FAILURE;
	}
	ret = process_host_cmd_resp(buf);

	return ret;
}

/** custom IE, auto mask value */
#define	CUSTOM_IE_AUTO_MASK	0xffff

/**
 * @brief Show usage information for the customie
 * command
 *
 * $return         N/A
 **/
void
print_custom_ie_usage(void)
{
	printf("\nUsage : customie [INDEX] [MASK] [IEBuffer]");
	printf("\n         empty - Get all IE settings\n");
	printf("\n         INDEX:  0 - Get/Set IE index 0 setting");
	printf("\n                 1 - Get/Set IE index 1 setting");
	printf("\n                 2 - Get/Set IE index 2 setting");
	printf("\n                 3 - Get/Set IE index 3 setting");
	printf("\n                 .                             ");
	printf("\n                 .                             ");
	printf("\n                 .                             ");
	printf("\n                -1 - Append/Delete IE automatically");
	printf("\n                     Delete will delete the IE from the matching IE buffer");
	printf("\n                     Append will append the IE to the buffer with the same mask");
	printf("\n         MASK :  Management subtype mask value as per bit defintions");
	printf("\n              :  Bit 0 - Association request.");
	printf("\n              :  Bit 1 - Association response.");
	printf("\n              :  Bit 2 - Reassociation request.");
	printf("\n              :  Bit 3 - Reassociation response.");
	printf("\n              :  Bit 4 - Probe request.");
	printf("\n              :  Bit 5 - Probe response.");
	printf("\n              :  Bit 8 - Beacon.");
	printf("\n         MASK :  MASK = 0 to clear the mask and the IE buffer");
	printf("\n         IEBuffer :  IE Buffer in hex (max 256 bytes)\n\n");
	return;
}

/**
 * @brief Converts a string to hex value
 *
 * @param str      A pointer to the string
 * @param raw      A pointer to the raw data buffer
 * @return         Number of bytes read
 **/
int
string2raw(char *str, unsigned char *raw)
{
	int len = (strlen(str) + 1) / 2;

	do {
		if (!isxdigit(*str)) {
			return -1;
		}
		*str = toupper(*str);
		*raw = CHAR2INT(*str) << 4;
		++str;
		*str = toupper(*str);
		if (*str == '\0')
			break;
		*raw |= CHAR2INT(*str);
		++raw;
	} while (*++str != '\0');
	return len;
}

/**
 * @brief Creates a hostcmd request for custom IE settings
 * and sends to the driver
 *
 * Usage: "customie [INDEX] [MASK] [IEBuffer]"
 *
 * Options: INDEX :      0 - Get/Delete IE index 0 setting
 *                       1 - Get/Delete IE index 1 setting
 *                       2 - Get/Delete IE index 2 setting
 *                       3 - Get/Delete IE index 3 setting
 *                       .
 *                       .
 *                       .
 *                      -1 - Append IE at the IE buffer with same MASK
 *          MASK  :      Management subtype mask value
 *          IEBuffer:    IE Buffer in hex
 *   					       empty - Get all IE settings
 *
 * @param argc     Number of arguments
 * @param argv     Pointer to the arguments
 * @return         N/A
 **/
static int
process_custom_ie(int argc, char *argv[])
{
	tlvbuf_custom_ie *tlv = NULL;
	tlvbuf_max_mgmt_ie *max_mgmt_ie_tlv = NULL;
	custom_ie *ie_ptr = NULL;
	t_u8 *buffer = NULL;
	t_u16 buf_len = 0;
	t_u16 mgmt_subtype_mask = 0;
	int ie_buf_len = 0, ie_len = 0, i = 0;
	struct ifreq ifr;

	/* mlanconfig mlan0 customie idx flag buf */
	if (argc > 6) {
		printf("ERR:Too many arguments.\n");
		print_custom_ie_usage();
		return MLAN_STATUS_FAILURE;
	}
	/* Error checks and initialize the command length */
	if (argc > 3) {
		if (((IS_HEX_OR_DIGIT(argv[3]) == MLAN_STATUS_FAILURE) &&
		     (atoi(argv[3]) != -1)) || (atoi(argv[3]) < -1)) {
			printf("ERR:Illegal index %s\n", argv[3]);
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
	}
	switch (argc) {
	case 3:
		buf_len = MRVDRV_SIZE_OF_CMD_BUFFER;
		break;
	case 4:
		if (atoi(argv[3]) < 0) {
			printf("ERR:Illegal index %s. Must be either greater than or equal to 0 for Get Operation \n", argv[3]);
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		buf_len = MRVDRV_SIZE_OF_CMD_BUFFER;
		break;
	case 5:
		if (MLAN_STATUS_FAILURE == ishexstring(argv[4]) ||
		    A2HEXDECIMAL(argv[4]) != 0) {
			printf("ERR: Mask value should be 0 to clear IEBuffers.\n");
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		if (atoi(argv[3]) == -1) {
			printf("ERR: You must provide buffer for automatic deletion.\n");
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		buf_len = sizeof(tlvbuf_custom_ie) + sizeof(custom_ie);
		break;
	case 6:
		/* This is to check negative numbers and special symbols */
		if (MLAN_STATUS_FAILURE == IS_HEX_OR_DIGIT(argv[4])) {
			printf("ERR:Mask value must be 0 or hex digits\n");
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		/* If above check is passed and mask is not hex, then it must
		   be 0 */
		if ((ISDIGIT(argv[4]) == MLAN_STATUS_SUCCESS) && atoi(argv[4])) {
			printf("ERR:Mask value must be 0 or hex digits\n ");
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		if (MLAN_STATUS_FAILURE == ishexstring(argv[5])) {
			printf("ERR:Only hex digits are allowed\n");
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		ie_buf_len = strlen(argv[5]);
		if (!strncasecmp("0x", argv[5], 2)) {
			ie_len = (ie_buf_len - 2 + 1) / 2;
			argv[5] += 2;
		} else
			ie_len = (ie_buf_len + 1) / 2;
		if (ie_len > MAX_IE_BUFFER_LEN) {
			printf("ERR:Incorrect IE length %d\n", ie_buf_len);
			print_custom_ie_usage();
			return MLAN_STATUS_FAILURE;
		}
		mgmt_subtype_mask = (t_u16)A2HEXDECIMAL(argv[4]);
		buf_len = sizeof(tlvbuf_custom_ie) + sizeof(custom_ie) + ie_len;
		break;
	}
	/* Initialize the command buffer */
	buffer = (t_u8 *)malloc(buf_len);
	if (!buffer) {
		printf("ERR:Cannot allocate buffer for command!\n");
		return MLAN_STATUS_FAILURE;
	}
	memset(buffer, 0, buf_len);
	tlv = (tlvbuf_custom_ie *)buffer;
	tlv->tag = MRVL_MGMT_IE_LIST_TLV_ID;
	if (argc == 3 || argc == 4) {
		if (argc == 3)
			tlv->length = 0;
		else {
			tlv->length = sizeof(t_u16);
			ie_ptr = (custom_ie *)(tlv->ie_data);
			ie_ptr->ie_index = (t_u16)(atoi(argv[3]));
		}
	} else {
		/* Locate headers */
		ie_ptr = (custom_ie *)(tlv->ie_data);
		/* Set TLV fields */
		tlv->length = sizeof(custom_ie) + ie_len;
		ie_ptr->ie_index = atoi(argv[3]);
		ie_ptr->mgmt_subtype_mask = mgmt_subtype_mask;
		ie_ptr->ie_length = ie_len;
		if (argc == 6)
			string2raw(argv[5], ie_ptr->ie_buffer);
	}
	/* Initialize the ifr structure */
	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_ifrn.ifrn_name, dev_name, strlen(dev_name));
	ifr.ifr_ifru.ifru_data = (void *)buffer;
	/* Perform ioctl */
	if (ioctl(sockfd, CUSTOM_IE_CFG, &ifr)) {
		perror("ioctl[CUSTOM_IE_CFG]");
		printf("ERR:Command sending failed!\n");
		if (buffer)
			free(buffer);
		return MLAN_STATUS_FAILURE;
	}
	/* Print response */
	if (argc > 4) {
		printf("custom IE setting successful\n");
	} else {
		printf("Querying custom IE successful\n");
		tlv = (tlvbuf_custom_ie *)buffer;
		ie_len = tlv->length;
		ie_ptr = (custom_ie *)(tlv->ie_data);
		while ((unsigned int)ie_len >= sizeof(custom_ie)) {
			printf("Index [%d]\n", ie_ptr->ie_index);
			if (ie_ptr->ie_length)
				printf("Management Subtype Mask = 0x%02x\n",
				       (ie_ptr->mgmt_subtype_mask) == 0 ?
				       CUSTOM_IE_AUTO_MASK :
				       (ie_ptr->mgmt_subtype_mask));
			else
				printf("Management Subtype Mask = 0x%02x\n",
				       (ie_ptr->mgmt_subtype_mask));
			hexdump("IE Buffer", (void *)ie_ptr->ie_buffer,
				ie_ptr->ie_length, ' ');
			ie_len -= sizeof(custom_ie) + ie_ptr->ie_length;
			ie_ptr = (custom_ie *)((t_u8 *)ie_ptr +
					       sizeof(custom_ie) +
					       ie_ptr->ie_length);
		}
	}
	max_mgmt_ie_tlv =
		(tlvbuf_max_mgmt_ie *)(buffer + sizeof(tlvbuf_custom_ie) +
				       tlv->length);
	if (max_mgmt_ie_tlv) {
		if (max_mgmt_ie_tlv->tag == MRVL_MAX_MGMT_IE_TLV_ID) {
			for (i = 0; i < max_mgmt_ie_tlv->count; i++) {
				printf("buf%d_size = %d\n", i,
				       max_mgmt_ie_tlv->info[i].buf_size);
				printf("number of buffers = %d\n",
				       max_mgmt_ie_tlv->info[i].buf_count);
				printf("\n");
			}
		}
	}
	if (buffer)
		free(buffer);

	return MLAN_STATUS_SUCCESS;
}

/********************************************************
		Global Functions
********************************************************/
/**
 *  @brief Get one line from the File
 *
 *  @param fp       File handler
 *  @param str	    Storage location for data.
 *  @param size 	Maximum number of characters to read.
 *  @param lineno	A pointer to return current line number
 *  @return         returns string or NULL
 */
char *
mlan_config_get_line(FILE * fp, t_s8 *str, t_s32 size, int *lineno)
{
	char *start, *end;
	int out, next_line;

	if (!fp || !str)
		return NULL;

	do {
read_line:
		if (!fgets(str, size, fp))
			break;
		start = str;
		start[size - 1] = '\0';
		end = start + strlen(str);
		(*lineno)++;

		out = 1;
		while (out && (start < end)) {
			next_line = 0;
			/* Remove empty lines and lines starting with # */
			switch (start[0]) {
			case ' ':	/* White space */
			case '\t':	/* Tab */
				start++;
				break;
			case '#':
			case '\n':
			case '\0':
				next_line = 1;
				break;
			case '\r':
				if (start[1] == '\n')
					next_line = 1;
				else
					start++;
				break;
			default:
				out = 0;
				break;
			}
			if (next_line)
				goto read_line;
		}

		/* Remove # comments unless they are within a double quoted
		   string. Remove trailing white space. */
		if ((end = strstr(start, "\""))) {
			if (!(end = strstr(end + 1, "\"")))
				end = start;
		} else
			end = start;

		if ((end = strstr(end + 1, "#")))
			*end-- = '\0';
		else
			end = start + strlen(start) - 1;

		out = 1;
		while (out && (start < end)) {
			switch (*end) {
			case ' ':	/* White space */
			case '\t':	/* Tab */
			case '\n':
			case '\r':
				*end = '\0';
				end--;
				break;
			default:
				out = 0;
				break;
			}
		}

		if (start == '\0')
			continue;

		return start;
	} while (1);

	return NULL;
}

/**
 *  @brief parse hex data
 *  @param fp       File handler
 *  @param dst      A pointer to receive hex data
 *  @return         length of hex data
 */
int
fparse_for_hex(FILE * fp, t_u8 *dst)
{
	t_s8 *ptr;
	t_u8 *dptr;
	t_s8 buf[256];

	dptr = dst;
	while (fgets(buf, sizeof(buf), fp)) {
		ptr = buf;

		while (*ptr) {
			/* skip leading spaces */
			while (*ptr && (isspace(*ptr) || *ptr == '\t'))
				ptr++;

			/* skip blank lines and lines beginning with '#' */
			if (*ptr == '\0' || *ptr == '#')
				break;

			if (isxdigit(*ptr)) {
				ptr = convert2hex(ptr, dptr++);
			} else {
				/* Invalid character on data line */
				ptr++;
			}
		}
	}

	return (dptr - dst);
}

/**
 *  @brief Converts colon separated MAC address to hex value
 *
 *  @param mac      A pointer to the colon separated MAC string
 *  @param raw      A pointer to the hex data buffer
 *  @return         MLAN_STATUS_SUCCESS or MLAN_STATUS_FAILURE
 *                  MAC_BROADCAST   - if breadcast mac
 *                  MAC_MULTICAST   - if multicast mac
 */
static int
mac2raw(char *mac, t_u8 *raw)
{
	unsigned int temp_raw[ETH_ALEN];
	int num_tokens = 0;
	int i;

	if (strlen(mac) != ((2 * ETH_ALEN) + (ETH_ALEN - 1))) {
		return MLAN_STATUS_FAILURE;
	}
	num_tokens = sscanf(mac, "%2x:%2x:%2x:%2x:%2x:%2x",
			    temp_raw + 0, temp_raw + 1, temp_raw + 2,
			    temp_raw + 3, temp_raw + 4, temp_raw + 5);
	if (num_tokens != ETH_ALEN) {
		return MLAN_STATUS_FAILURE;
	}
	for (i = 0; i < num_tokens; i++)
		raw[i] = (t_u8)temp_raw[i];

	if (memcmp(raw, "\xff\xff\xff\xff\xff\xff", ETH_ALEN) == 0) {
		return MAC_BROADCAST;
	} else if (raw[0] & 0x01) {
		return MAC_MULTICAST;
	}
	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief          Parse function for a configuration line
 *
 *  @param s        Storage buffer for data
 *  @param size     Maximum size of data
 *  @param stream   File stream pointer
 *  @param line     Pointer to current line within the file
 *  @param _pos     Output string or NULL
 *  @return         String or NULL
 */
static char *
config_get_line(char *s, int size, FILE * stream, int *line, char **_pos)
{
	*_pos = mlan_config_get_line(stream, s, size, line);
	return *_pos;
}

/**
 *  @brief Parses a command line
 *
 *  @param line     The line to parse
 *  @param args     Pointer to the argument buffer to be filled in
 *  @return         Number of arguments in the line or EOF
 */
static int
parse_line(char *line, char *args[])
{
	int arg_num = 0;
	int is_start = 0;
	int is_quote = 0;
	int length = 0;
	int i = 0;

	arg_num = 0;
	length = strlen(line);
	/* Process line */

	/* Find number of arguments */
	is_start = 0;
	is_quote = 0;
	for (i = 0; i < length; i++) {
		/* Ignore leading spaces */
		if (is_start == 0) {
			if (line[i] == ' ') {
				continue;
			} else if (line[i] == '\t') {
				continue;
			} else if (line[i] == '\n') {
				break;
			} else {
				is_start = 1;
				args[arg_num] = &line[i];
				arg_num++;
			}
		}
		if (is_start == 1) {
			/* Ignore comments */
			if (line[i] == '#') {
				if (is_quote == 0) {
					line[i] = '\0';
					arg_num--;
				}
				break;
			}
			/* Separate by '=' */
			if (line[i] == '=') {
				line[i] = '\0';
				is_start = 0;
				continue;
			}
			/* Separate by ',' */
			if (line[i] == ',') {
				line[i] = '\0';
				is_start = 0;
				continue;
			}
			/* Change ',' to ' ', but not inside quotes */
			if ((line[i] == ',') && (is_quote == 0)) {
				line[i] = ' ';
				continue;
			}
		}
		/* Remove newlines */
		if (line[i] == '\n') {
			line[i] = '\0';
		}
		/* Check for quotes */
		if (line[i] == '"') {
			is_quote = (is_quote == 1) ? 0 : 1;
			continue;
		}
		if (((line[i] == ' ') || (line[i] == '\t')) && (is_quote == 0)) {
			line[i] = '\0';
			is_start = 0;
			continue;
		}
	}
	return arg_num;
}

/**
 *  @brief Process transmission of mgmt frames
 *  @param argc   number of arguments
 *  @param argv   A pointer to arguments array
 *  @return     MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int
process_mgmt_frame_tx(int argc, char *argv[])
{
	struct ifreq ifr;
	char *line = NULL;
	FILE *config_file = NULL;
	int li = 0, arg_num = 0, ret = 0, i = 0;
	char *args[100], *pos = NULL, mac_addr[20];
	t_u8 peer_mac[ETH_ALEN];
	t_u16 data_len = 0, subtype = 0;
	wlan_mgmt_frame_tx *pmgmt_frame;
	t_u8 *buffer = NULL;
	pkt_header *hdr = NULL;

	/* Check arguments */
	if (argc != 4) {
		printf("ERR:Incorrect number of arguments.\n");
		printf("Syntax: ./mlanconfig mlanX mgmtframetx <config/pkt.conf>\n");
		exit(1);
	}

	data_len = sizeof(wlan_mgmt_frame_tx);

	buffer = (t_u8 *)malloc(MRVDRV_SIZE_OF_CMD_BUFFER);
	if (!buffer) {
		printf("ERR:Cannot allocate memory!\n");
		goto done;
	}
	memset(buffer, 0, MRVDRV_SIZE_OF_CMD_BUFFER);
	hdr = (pkt_header *)buffer;
	pmgmt_frame = (wlan_mgmt_frame_tx *)(buffer + sizeof(pkt_header));

	/* Check if file exists */
	config_file = fopen(argv[3], "r");
	if (config_file == NULL) {
		printf("\nERR:Config file can not open.\n");
		goto done;
	}
	line = (char *)malloc(MAX_CONFIG_LINE);
	if (!line) {
		printf("ERR:Cannot allocate memory for line\n");
		goto done;
	}
	memset(line, 0, MAX_CONFIG_LINE);

	/* Parse file and process */
	while (config_get_line(line, MAX_CONFIG_LINE, config_file, &li, &pos)) {
		arg_num = parse_line(line, args);
		if (strcmp(args[0], "PktSubType") == 0) {
			subtype = (t_u16)A2HEXDECIMAL(args[1]);
			pmgmt_frame->frm_ctl |= subtype << 4;
		} else if (strncmp(args[0], "Addr", 4) == 0) {
			strncpy(mac_addr, args[1], 20);
			if ((ret =
			     mac2raw(mac_addr,
				     peer_mac)) != MLAN_STATUS_SUCCESS) {
				printf("ERR: %s Address \n",
				       ret ==
				       MLAN_STATUS_FAILURE ? "Invalid MAC" : ret
				       ==
				       MAC_BROADCAST ? "Broadcast" :
				       "Multicast");
				goto done;
			}
			i = atoi(args[0] + 4);
			switch (i) {
			case 1:
				memcpy(pmgmt_frame->addr1, peer_mac, ETH_ALEN);
				break;
			case 2:
				memcpy(pmgmt_frame->addr2, peer_mac, ETH_ALEN);
				break;
			case 3:
				memcpy(pmgmt_frame->addr3, peer_mac, ETH_ALEN);
				break;
			case 4:
				memcpy(pmgmt_frame->addr4, peer_mac, ETH_ALEN);
				break;
			}
		} else if (strcmp(args[0], "Data") == 0) {
			for (i = 0; i < arg_num - 1; i++)
				pmgmt_frame->payload[i] =
					(t_u8)A2HEXDECIMAL(args[i + 1]);
			data_len += arg_num - 1;
		}
	}
	pmgmt_frame->frm_len = data_len - sizeof(pmgmt_frame->frm_len);
#define MRVL_PKT_TYPE_MGMT_FRAME 0xE5
	hdr->pkt_len = data_len;
	hdr->TxPktType = MRVL_PKT_TYPE_MGMT_FRAME;
	hdr->TxControl = 0;
	hexdump("Frame Tx", buffer, data_len + sizeof(pkt_header), ' ');
	/* Send collective command */
	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_ifrn.ifrn_name, dev_name, strlen(dev_name));
	ifr.ifr_ifru.ifru_data = (void *)buffer;

	/* Perform ioctl */
	if (ioctl(sockfd, FRAME_TX_IOCTL, &ifr)) {
		perror("");
		printf("ERR:Could not send management frame.\n");
	} else {
		printf("Mgmt Frame sucessfully sent.\n");
	}

done:
	if (config_file)
		fclose(config_file);
	if (buffer)
		free(buffer);
	if (line)
		free(line);
	return MLAN_STATUS_SUCCESS;
}

/**
 *  @brief Entry function for mlanconfig
 *  @param argc		number of arguments
 *  @param argv     A pointer to arguments array
 *  @return      	MLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int
main(int argc, char *argv[])
{
	t_s32 cmd;

	if ((argc == 2) && (strcmp(argv[1], "-v") == 0)) {
		fprintf(stdout, "Marvell mlanconfig version %s\n",
			MLANCONFIG_VER);
		exit(0);
	}
	if (argc < 3) {
		fprintf(stderr, "Invalid number of parameters!\n");
		display_usage();
		exit(1);
	}

	strncpy(dev_name, argv[1], IFNAMSIZ - 1);

	/*
	 * create a socket
	 */
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr, "mlanconfig: Cannot open socket.\n");
		exit(1);
	}
	if (get_range() < 0) {
		fprintf(stderr, "mlanconfig: Cannot get range.\n");
		close(sockfd);
		exit(1);
	}
	switch ((cmd = findcommand(NELEMENTS(commands), commands, argv[2]))) {
	case CMD_HOSTCMD:
		process_host_cmd(argc, argv);
		break;
	case CMD_ARPFILTER:
		process_arpfilter(argc, argv);
		break;
	case CMD_CFG_DATA:
		process_cfg_data(argc, argv);
		break;
	case CMD_CMD52RW:
		process_sdcmd52rw(argc, argv);
		break;
	case CMD_CMD53RW:
		process_sdcmd53rw(argc, argv);
		break;
	case CMD_GET_SCAN_RSP:
		process_getscantable(argc, argv);
		break;
	case CMD_SET_USER_SCAN:
		process_setuserscan(argc, argv);
		break;
	case CMD_ADD_TS:
		process_addts(argc, argv);
		break;
	case CMD_DEL_TS:
		process_delts(argc, argv);
		break;
	case CMD_QCONFIG:
		process_qconfig(argc, argv);
		break;
	case CMD_QSTATS:
		process_qstats(argc, argv);
		break;
	case CMD_TS_STATUS:
		process_wmm_ts_status(argc, argv);
		break;
	case CMD_WMM_QSTATUS:
		process_wmm_qstatus(argc, argv);
		break;
	case CMD_REGRW:
		process_regrdwr(argc, argv);
		break;
	case CMD_MEMRW:
		process_memrdwr(argc, argv);
		break;
	case CMD_STA_CUSTOM_IE:
		process_custom_ie(argc, argv);
		break;
	case CMD_STA_MGMT_FRAME_TX:
		process_mgmt_frame_tx(argc, argv);
		break;
	default:
		fprintf(stderr, "Invalid command specified!\n");
		display_usage();
		close(sockfd);
		exit(1);
	}

	close(sockfd);
	return MLAN_STATUS_SUCCESS;
}
