/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2014, 2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/of_device.h>

#include <linux/device.h>
#include <linux/platform_device.h>

#include "asic/SCCPLITE_regstructs.h"
#include "asic/SCCPLITE_regmasks.h"

#include "sccplite.h"

#define SCCPLITE_NUM_DEVICES  2
#define SCCPLITE_DRIVER_NAME  "marvell-sccp"

#define SCCPLITE_SRAM_SIZE_BYTES    ( sizeof(((SCCPLITE_REG_REGS_t *)0)->SRAM) / 2 )
#define SCCPLITE_SRAM_SIZE_WORDS    ( SCCPLITE_SRAM_SIZE_BYTES / sizeof(uint32_t) )

// SCCPLITE regsiter read and write macros...
// NOTE: You *must* have defined a local variable called sccp_regbase
#define sccpRdReg(reg)       ioread32(      &(((SCCPLITE_REG_REGS_t *)(sccp_regbase))->reg))
#define sccpWrReg(val, reg)  iowrite32(val, &(((SCCPLITE_REG_REGS_t *)(sccp_regbase))->reg))

#define DEBUG 0
#if DEBUG
#define DBG_ERR printk
#define DBG DBG_ERR
#else
#define DBG_ERR(...) { (void)0; }
#define DBG(...) { (void)0; }
#endif

typedef struct {
    int Irq;
    int Minor;
    int Loaded;
    int Opened;
    int Fileptr;
    int Stackptr;
    struct cdev Cdev;
    struct mutex Mutex;
    void __iomem *Base;
} sccp_dev;

typedef struct {
    int Major;
    dev_t Dev;
    struct class *Class;
    struct file_operations Fops;
    sccp_dev *Sccp[SCCPLITE_NUM_DEVICES];
} sccp_drvr;

/******************* F O R W A R D  D E C L A R A T I O N S *******************/

static int  sccp_open(  struct inode *inode, struct file *filep);
static int  sccp_close( struct inode *inode, struct file *filep);
static long sccp_ioctl( struct file  *filep, unsigned int cmd, unsigned long arg);
static int  sccp_read(  struct file  *filep, char __user *buf, size_t len, loff_t *ptr);
static int  sccp_write( struct file  *filep, const char __user *buf, size_t len, loff_t *ptr);

/******************** D R I V E R  D E C L A R A T I O N S ********************/

static sccp_drvr sccp = {
    .Fops = {
      .owner          = THIS_MODULE,
      .open           = sccp_open,
      .release        = sccp_close,
      .read           = sccp_read,
      .write          = sccp_write,
      .unlocked_ioctl = sccp_ioctl,
    },
};

/******************** E X P O R T E D  F U N C T I O N S **********************/

// Read the sccp SRAM memory for a given cpu *word at a time*
int  sccp_read_sram_words(  int device, int word_offset, uint16_t *data, int byte_length )
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;
    uint32_t *sccp_sram = ((uint32_t *)((SCCPLITE_REG_REGS_t*)sccp_regbase)->SRAM) + word_offset;
    uint16_t *read_buff = (uint16_t *)data;
    int word_length = byte_length / sizeof(data[0]);
    int i;

    // The input variable "length" is in bytes!
    for (i = 0; i < word_length; i++ )
    {
        *read_buff++ = *sccp_sram++;
    }

    return(byte_length);
}

void sccp_set_interrupt( int device, uint32_t intr)
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;
    uint32_t  regvalue;

    regvalue = sccpRdReg(INT) | intr;
    sccpWrReg(regvalue, INT);
}

void sccp_clr_interrupt( int device, uint32_t intr)
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    SCCPLITE_REG_REGS_t* sccp_regbase = (SCCPLITE_REG_REGS_t*)dev->Base; //uint32_t *sccp_regbase = (uint32_t *)dev->Base;
    uint32_t  regvalue;

    regvalue = sccpRdReg(INT) & intr;
    sccpWrReg(regvalue, INT);
}

// Write the sccp SRAM memory for a given cpu * word at a time *
int  sccp_write_sram_words( int device, int word_offset, uint16_t *data, int byte_length )
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;
    uint32_t *sccp_sram = ((uint32_t *)((SCCPLITE_REG_REGS_t*)sccp_regbase)->SRAM) + word_offset;
    uint16_t *writ_data = (uint16_t *)data;
    int word_length = byte_length / sizeof(data[0]);
    int i;

    for ( i = 0; i < word_length; i++ )
    {
        *sccp_sram++ = *writ_data++;
    }

    return(byte_length);
}

int  sccp_write_microcode( int device, int word_offset, int *data, int byte_length )
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;
    uint32_t *sccp_sram = ((uint32_t *)((SCCPLITE_REG_REGS_t*)sccp_regbase)->SRAM) + word_offset;
    int word_length = byte_length / sizeof(data[0]);

    memcpy( sccp_sram, data, byte_length );

    // Compute the next available stack pointer address
    // Bump it by "one" for good measure!  NOTE: This is a word address...
    dev->Stackptr = min( word_offset + word_length + 1, SCCPLITE_SRAM_SIZE_WORDS - 1 );

    // Indicate that this SCCPLITE now has valid code loaded...
    dev->Loaded = 1;

    return(byte_length);
}

void sccp_processor_start( int device, int start_word_addr)
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;

    sccpWrReg(SCCPLITE_REG_CPU_CTRL_SRAMPRIORITY_MASK | SCCPLITE_REG_CPU_CTRL_SOFTRESET_MASK, CPU_CTRL);
    do {
    // Wait for the soft reset bit to drop in the control reg...
    } while( sccpRdReg(CPU_CTRL) & SCCPLITE_REG_CPU_CTRL_SOFTRESET_MASK);

    sccpWrReg(start_word_addr, IPTR);
    sccpWrReg(dev->Stackptr, STACK);

    sccpWrReg(SCCPLITE_REG_CPU_CTRL_SRAMPRIORITY_MASK | SCCPLITE_REG_CPU_CTRL_RUN_MASK, CPU_CTRL);
}

void sccp_processor_stop( int device)
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;

    sccpWrReg(SCCPLITE_REG_CPU_CTRL_RUN_MASK, CPU_CTRL);
}

void sccp_processor_resume( int device)
{
    sccp_dev *dev = sccp.Sccp[device & (SCCPLITE_NUM_DEVICES - 1)];
    uint32_t *sccp_regbase = (uint32_t *)dev->Base;

    sccpWrReg(SCCPLITE_REG_CPU_CTRL_RUN_MASK, CPU_CTRL);
}

/******************** D R I V E R  F U N C T I O N S **************************/

static int sccp_open( struct inode *inode, struct file *filep)
{
    sccp_dev *dev;
    int retval = -EBUSY;
    
    dev = container_of(inode->i_cdev, sccp_dev, Cdev);
    mutex_lock(&dev->Mutex);

    if (!dev->Opened) {
        dev->Opened  = 1;
        dev->Fileptr = 0;
        filep->private_data = dev;
        retval = 0;
    }
    
    mutex_unlock(&dev->Mutex);

    return(retval);
}

static int sccp_close( struct inode *inode, struct file *filep)
{
    sccp_dev *dev = filep->private_data;
    
    mutex_lock(&dev->Mutex);

    dev->Opened = 0;
    filep->private_data = NULL;

    mutex_unlock(&dev->Mutex);
    
    return(0);
}

static uint8_t BinaryToAscii( uint8_t val)
{
    if (val <=  9)
        val += 48;
    else
    if (val <= 15)
        val += 87;
    else
        val  = 48;

    return(val);
}

static int sccp_read( struct file *filep, char __user *buf, size_t len_ascii, loff_t *ptr)
{
    sccp_dev *dev = filep->private_data;
    uint8_t *tempbuf, *readbuf, val;
    int retval, i, j, k = 0;
    size_t len_words = len_ascii / 4;
    size_t len_bytes = len_words * 2;
    
    if ((int)len_words == 0) return(-EINVAL);
    len_words = min( len_words, SCCPLITE_SRAM_SIZE_WORDS - dev->Fileptr  );
    len_bytes = len_words * sizeof(uint16_t);
    len_ascii = len_bytes * 2;

    if (!(readbuf = (uint8_t *)kmalloc(len_words * sizeof(uint16_t), GFP_KERNEL))) {
        return(-ENOMEM);
    } else
    if (!(tempbuf = (uint8_t *)kmalloc((int)len_ascii, GFP_KERNEL))) {
        kfree( readbuf);
        return(-ENOMEM);
    }
    
    mutex_lock(&dev->Mutex);

    sccp_read_sram_words( dev->Minor, dev->Fileptr, (uint16_t*)readbuf, len_bytes );

    for (i = 0; i <= (int)len_bytes - 2; i += 2) {
        for (j = 1; j >= 0; j--) {
            val = BinaryToAscii(readbuf[i + j] >> 0x4);
            tempbuf[k++] = val;
            val = BinaryToAscii(readbuf[i + j] &  0xF);
            tempbuf[k++] = val;
        }
    }

    if (copy_to_user( buf, tempbuf, (int)len_ascii)) {
        retval = -EFAULT;
        goto readExit;
    }
    dev->Fileptr += len_words;
    
    retval = (int)len_ascii;

readExit:
    mutex_unlock( &dev->Mutex);
    kfree( tempbuf);
    kfree( readbuf);

    return(retval);
}

static uint8_t AsciiToBinary( uint8_t val)
{
    if ((val >= '0') && (val <= '9'))
         val -= 48;
    else
    if ((val >= 'A') && (val <= 'F'))
         val -= 55;
    else
    if ((val >= 'a') && (val <= 'f'))
         val -= 87;
    else
         val  =  0;

    return(val);
}

static int sccp_write( struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
    sccp_dev *dev = filep->private_data;
    uint8_t *tempbuf, *writbuf, valH, valL;
    int retval, i, j, k = 0;
    
    if ((int)len == 0) return(-EINVAL);
    len = min( len, SCCPLITE_SRAM_SIZE_BYTES - ( dev->Fileptr * sizeof(uint16_t) ) );

    if (!(writbuf = (uint8_t *)kmalloc((int)len, GFP_KERNEL))) {
        return(-ENOMEM);
    } else
    if (!(tempbuf = (uint8_t *)kmalloc((int)len, GFP_KERNEL))) {
        kfree( writbuf);
        return(-ENOMEM);
    }

    mutex_lock( &dev->Mutex);

    if (copy_from_user( tempbuf, buf, (int)len)) {
        retval = -EFAULT;
        goto writeExit;
    }

    memset( writbuf, 0, len );
    // Input is ASCII representing hex bytes
    // Output needs to be 16 bit values padded into 32 bit data
    // For each word (16 bits), convert from ASCII to binary and write in lower 16 bits of padded 32-bit word
    for (i = 0; i <= (int)(len - 4); i += 4) {
        for (j = 2; j >= 0; j -= 2) {
            valH = AsciiToBinary(tempbuf[i + j + 0]);
            valL = AsciiToBinary(tempbuf[i + j + 1]);
            writbuf[k++] = (valH << 4) | valL;
        }
        k += 2;     // pad the next 2 bytes of 0s
    }

    sccp_write_microcode( dev->Minor, dev->Fileptr, (uint32_t *)writbuf, k );
    dev->Fileptr += (int)len / 4;

    if ((int)len > 0)
        retval = (int)len;
    else
        retval = -EAGAIN;
    
writeExit:
    mutex_unlock( &dev->Mutex);
    kfree( tempbuf);
    kfree( writbuf);

    return(retval);
}

static long sccp_ioctl( struct file *filep, unsigned int cmd, unsigned long arg)
{
    sccp_dev *dev = filep->private_data;
    long retval = 0;

    mutex_lock(&dev->Mutex);

    switch (cmd) {
        case SCCPLITE_START_PROCESSOR:
            if (dev->Loaded)
                sccp_processor_start( (int)arg, 0 );
            else
                retval = -EINVAL;
        break;

        case SCCPLITE_STOP_PROCESSOR:
            if (dev->Loaded)
                sccp_processor_stop( (int)arg );
            else
                retval = -EINVAL;
        break;

        default:
            retval = -ENOIOCTLCMD;
        break;
    }

    mutex_unlock(&dev->Mutex);

    return(retval);
}

// NOTE: If you want to support interrupts in this driver, uncomment
// the following code and plumb it according to your specific needs!
#if 0
static irqreturn_t sccp_platform_irq( int irq, void *pdev)
{
    return(IRQ_HANDLED);
}
#endif

/******************** P L A T F O R M / D E V I C E ***************************/

static int sccp_device_init( sccp_dev *dev, int id)
{
    if ((dev->Minor = id) >= SCCPLITE_NUM_DEVICES) { // bind dev->Minor
        printk(KERN_ERR "too many sccp devices found\n");
        return(-EIO);
    }

    if (IS_ERR(device_create(sccp.Class, NULL, MKDEV(sccp.Major, dev->Minor),
                                   NULL, "sccp%u", dev->Minor))) {
        printk(KERN_ERR "Failed to create device: sccp%d\n", dev->Minor);
        return(-1);
    }

    cdev_init( &dev->Cdev, &sccp.Fops);
    dev->Cdev.owner = THIS_MODULE;
    dev->Cdev.ops   = &sccp.Fops;

    if (cdev_add( &dev->Cdev, MKDEV(sccp.Major, dev->Minor), 1) < 0) {
        printk(KERN_ERR "Can't add device driver\n");
        return(-1);
    }

    dev->Loaded = 0;
    dev->Opened = 0;
    mutex_init( &dev->Mutex);

    return(0);
}


// initial load of module has pdev->id == -1
// subsequent unload/load cycles show expected dev ids ..
static int dev_number = 0; // assigned per-device at probe time

static int sccp_platform_probe( struct platform_device *pdev)
{
    struct resource *res;
    sccp_dev *dev;
    int retval = 0;
    uint32_t *sccp_regbase = NULL;
    uint32_t *sccp_sram = NULL;

    DBG("%s(%d) .. \n", __func__, pdev->id);

    if (!(dev = (sccp_dev *)kzalloc( sizeof(sccp_dev), GFP_KERNEL))) {
        dev_err( &pdev->dev, "kzalloc failed\n");
        return(-ENOMEM);
    }

    if (!(res = platform_get_resource( pdev, IORESOURCE_MEM, 0 /*pdev->id*/))) {
        dev_err( &pdev->dev, "platform_get_resource failed\n");
        retval = -ENXIO;
        goto request_mem_failed;
    }

#if 0
    if ((dev->Irq = platform_get_irq( pdev, 0)) < 0) {
        dev_err( &pdev->dev, "platform_get_irq failed\n");
        retval = -ENXIO;
        goto request_mem_failed;
    }
#endif

    if (!request_mem_region( res->start, 0x2000 /*resource_size(res)*/, SCCPLITE_DRIVER_NAME)) { // REVISIT sizeof(res) 8k vs pegmatite's 2k specification
        dev_err( &pdev->dev, "request_mem_region failed\n");
        retval = -EBUSY;
        goto request_mem_failed;
    }

    if (!(dev->Base = ioremap( res->start, resource_size(res)))) {
        dev_err( &pdev->dev, "ioremap failed\n");
        retval = -ENOMEM;
        goto ioremap_failed;
    }

    pdev->id = dev_number++; // assigned per-dev probe
    sccp.Sccp[pdev->id] = dev;

    if (0) // dbg
    {
        sccp_regbase = (uint32_t *)dev->Base;
        sccp_sram = (uint32_t *)((SCCPLITE_REG_REGS_t*)sccp_regbase)->SRAM;

        DBG_ERR("%s(%d) base @ 0x%p SRAM @ 0x%p\n", __func__, pdev->id, sccp_regbase, sccp_sram);
    }

    sccp_clr_interrupt( pdev->id, ~SCCPLITE_REG_INT_INT_MASK);

#if 0
    if ((retval = request_irq( dev->Irq, sccp_platform_irq, 0, SCCPLITE_DRIVER_NAME, pdev))) {
        dev_err( &pdev->dev, "request_irq failed\n");
        goto request_irq_failed;
    }
    disable_irq( dev->Irq);
#endif

    if ((retval = sccp_device_init( dev, pdev->id))) { // bind dev->Minor
        dev_err( &pdev->dev, "sccp_probe failed\n");
        goto probe_failed;
    }

    DBG("%s(%d) bind %p -> pdev %p\n", __func__, dev->Minor, dev, pdev);
    platform_set_drvdata( pdev, dev);
    printk(KERN_ERR "SCCPLITE-%d: device registration complete\n", pdev->id);

    return(0);

probe_failed:
#if 0
    free_irq( dev->Irq, pdev);

request_irq_failed:
#endif
    iounmap( dev->Base);
    sccp.Sccp[pdev->id] = NULL;

ioremap_failed:
    release_mem_region( res->start, resource_size(res));

request_mem_failed:
    kfree( dev);

    return(retval);
}

static int sccp_platform_remove( struct platform_device *pdev)
{
    struct resource *res;
    sccp_dev *dev = platform_get_drvdata(pdev);
    if (dev)
    {
        mutex_destroy( &dev->Mutex);
        cdev_del( &dev->Cdev);
        device_destroy( sccp.Class, MKDEV(sccp.Major, dev->Minor));
        DBG("%s(%d) relinquish dev[%d:%d]\n", __func__, pdev->id, sccp.Major, dev->Minor);
#if 0
        free_irq( dev->Irq, pdev);
#endif
        iounmap( dev->Base);

        res = platform_get_resource( pdev, IORESOURCE_MEM, 0); // dev->Minor?
        if (res)
        {
            DBG("%s(%d) release mem region from resource %p\n", __func__, pdev->id, res);
            release_mem_region( res->start, resource_size(res));
        }

        printk(KERN_ERR "SCCPLITE-%d: device remove complete\n", dev->Minor);

        kfree( dev);
    }
    platform_set_drvdata( pdev, NULL);

    return(0);
}

static int sccp_platform_suspend( struct platform_device *pdev, pm_message_t state)
{
    return(0);
}

static int sccp_platform_resume( struct platform_device *pdev)
{
    return(0);
}

#ifdef CONFIG_OF
static const struct of_device_id mrvl_sccplite_dt_match[] = {

    { .compatible = SCCPLITE_DRIVER_NAME},  // IP Major Tag Rev 0, Rev 1 "marvell-sccp"
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_sccplite_dt_match);
#endif

static struct platform_driver sccp_platform_driver =
{
    .probe   = sccp_platform_probe,
    .remove  = sccp_platform_remove,
    .suspend = sccp_platform_suspend,
    .resume  = sccp_platform_resume,
    .driver  = {
      .name  = SCCPLITE_DRIVER_NAME,
      .owner = THIS_MODULE,
      .of_match_table = of_match_ptr(mrvl_sccplite_dt_match),
    },
};

static void sccp_platform_cleanup( void)
{
    unregister_chrdev_region( MKDEV(sccp.Major, 0), SCCPLITE_NUM_DEVICES);
    class_destroy( sccp.Class);
}

static int __init sccp_platform_init( void)
{
    int retval;
   
    sccp.Class = class_create( THIS_MODULE, "sccp");
    if (IS_ERR(sccp.Class)) {
        retval = PTR_ERR(sccp.Class);
        DBG_ERR(KERN_ERR "sccp: unable to create class %d\n", retval);
        return(retval);
    }

    if ((retval = alloc_chrdev_region( &sccp.Dev, 0, SCCPLITE_NUM_DEVICES, "sccp_processors"))) {
        class_destroy( sccp.Class);
        DBG_ERR(KERN_ERR "sccp: can't register character driver error %d\n", retval);
        return(retval);
    }
    sccp.Major = MAJOR( sccp.Dev);

    if ((retval = platform_driver_register( &sccp_platform_driver))) {
        sccp_platform_cleanup();
        DBG_ERR(KERN_ERR "%s: error registering platform driver\n", __func__);
    } else {
        printk(KERN_ERR "SCCPLITE:  platform registration complete\n");
    }

    return(retval);
}
module_init( sccp_platform_init);

static void __exit sccp_platform_exit( void)
{
    platform_driver_unregister( &sccp_platform_driver);
    sccp_platform_cleanup();
    printk(KERN_ERR "SCCPLITE:  platform remove complete\n");
}
module_exit( sccp_platform_exit);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell Serial Control Channel Processor (SCCPLITE) Driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_Jun__09");

#if 0
EXPORT_SYMBOL(sccp_register_setup);
#endif

EXPORT_SYMBOL(sccp_read_sram_words);
EXPORT_SYMBOL(sccp_write_sram_words);
EXPORT_SYMBOL(sccp_write_microcode);

EXPORT_SYMBOL(sccp_set_interrupt);
EXPORT_SYMBOL(sccp_clr_interrupt);

EXPORT_SYMBOL(sccp_processor_start);
EXPORT_SYMBOL(sccp_processor_stop);
EXPORT_SYMBOL(sccp_processor_resume);
