#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/fs.h>

#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/mutex.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/moduleparam.h>

#include <linux/poll.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/cdev.h>

#include <linux/io.h>
#include <linux/irq.h>

#include <linux/spinlock.h>




#define NOTICE_DEBUG 1
#ifdef NOTICE_DEBUG
#define NDBG(fmt, ...) { printk( fmt, ##__VA_ARGS__ ); }
#else
#define NDBG(fmt, ...)
#endif

struct notice_dev{
	spinlock_t		spin_lock;
	struct mutex	notice_mtx;
	struct cdev		cdev;
	struct device	*pdev;
	wait_queue_head_t wait;
};


static struct notice_dev notice_device;
static struct class *notice_class;
static dev_t notice_devno;
#define MINOR_COUNT 1

/* --- for usbmix --- */
DECLARE_WAIT_QUEUE_HEAD(usbmix_wq);
EXPORT_SYMBOL_GPL(usbmix_wq);
static unsigned char usbmix_connected[6] = {0x01, 0x01, 0x00, 0x00, 0x88, 0x99};
/* ------------------ */


#define ATTACHED_FALSE		0x00
#define ATTACHED_TRUE		0x01
#define ATTACHED_INVALID	0x99


int notice_driver_set_usbmix_connected( int connected )
{
	int retval = 0;

	NDBG( "[%s] in\n", __func__ );

	usbmix_connected[5] = connected;

	wake_up_interruptible( &usbmix_wq );

	return retval;
}
EXPORT_SYMBOL(notice_driver_set_usbmix_connected);


/* fops start */
static ssize_t notice_read( struct file *file, char __user *buf, size_t len, loff_t *ptr )
{
	int retval = 0;
	int size = 0;
	struct timeval tv = {
		.tv_sec = 0,
		.tv_usec = 0,
	};
	//struct notice_dev *dev = file->private_data;

	do_gettimeofday(&tv);
	
        NDBG( "[%s] in\n", __func__ );
	NDBG(KERN_EMERG "[notice](%d.%d) key_ether_status: %s\n", tv.tv_sec, tv.tv_usec, (usbmix_connected[5]) ? "ON" : "OFF");

	retval = sizeof( usbmix_connected );
	size = copy_to_user( buf, usbmix_connected, retval );
	if( size != 0 ){
		NDBG( "[notice] copy_to_user. try=%d,rest=%d.\n", retval, size );
	}

	usbmix_connected[5] = ATTACHED_INVALID;

	return retval;
}


/*
 * for test.
*/
static ssize_t notice_write( struct file *file, const char __user *buf, size_t len, loff_t *ptr )
{
	//struct notice_dev *dev = file->private_data;

	NDBG( "[%s] in\n", __func__ );

	notice_driver_set_usbmix_connected( ATTACHED_TRUE );

	return -EFAULT;
}


static int notice_open( struct inode *inode, struct file *file )
{
	int retval = 0;

	NDBG( "[%s] open\n", __func__ );

	file->private_data = &notice_device;

	return retval;
}

static int notice_release(struct inode *inode, struct file *file )
{
	struct notice_dev *dev = file->private_data;

	NDBG( "[%s] release\n", __func__ );

	if( dev == NULL )
		return -ENODEV;

	return 0;
}

static unsigned int notice_poll( struct file *file, struct poll_table_struct *wait )
{
	struct notice_dev *dev = file->private_data;
	int flags = 0;
	int status = 0;

	//NDBG( "[%s] poll\n", __func__ );

	poll_wait( file, &usbmix_wq, wait );
	spin_lock_irqsave( &dev->spin_lock, flags );
	
	if( usbmix_connected[5] != ATTACHED_INVALID ){
		NDBG( "[%s] ok, usbmix_connected[5]=%x\n", __func__, (int)usbmix_connected[5] );
		status = POLLIN | POLLRDNORM;
	}else{
		//NDBG( "[%s] ng, usbmix_connected[5]=%x\n", __func__, (int)usbmix_connected[5] );
		status = 0;
	}
	spin_unlock_irqrestore( &dev->spin_lock, flags );

	return status;
}


static struct file_operations notice_fops = {
	.owner		= THIS_MODULE,
	.read		= notice_read,
	.write		= NULL,//notice_write,
	.open		= notice_open,
	.release	= notice_release,
	.poll		= notice_poll,
};
/* fops end */



static int __init notice_init(void)
{
	int status;
	int major_num = 0;
	struct notice_dev *dev = &notice_device;

	NDBG( "[%s] init 6/23... \n", __func__ );

	notice_class = class_create( THIS_MODULE, "notice driver" );

	if (IS_ERR(notice_class)) {
		status = PTR_ERR(notice_class);
		pr_err("unable to create notice class %d\n", status);
		return status;
	}

	status = alloc_chrdev_region(&notice_devno, major_num, MINOR_COUNT, "notice_driver");
	major_num = MAJOR(notice_devno);

	if (status < 0) {
		pr_err("register_notice %d\n", status);
		class_destroy(notice_class);
		return status;
	}

	spin_lock_init(&dev->spin_lock);
	mutex_init(&dev->notice_mtx);

	notice_devno = MKDEV( major_num, MINOR_COUNT );

	/* Setup the sysfs files for the printer gadget. */
	dev->pdev = device_create(notice_class, NULL, notice_devno,
								NULL, "notice");
	if (IS_ERR(dev->pdev)) {
		NDBG("Failed to create device: notice\n" );
		status = PTR_ERR(dev->pdev);
		goto __init_fail;
	}

	cdev_init(&dev->cdev, &notice_fops);
	dev->cdev.owner = THIS_MODULE;
	status = cdev_add(&dev->cdev, notice_devno, MINOR_COUNT);
	if (status) {
		NDBG( "Failed to add notice device\n" );
		goto __init_fail;
	}
	//init_waitqueue_head( &usbmix_wq );

	return status;

__init_fail:
	class_destroy( notice_class );
	unregister_chrdev_region( notice_devno, 1 );
	pr_err("notice_driver %x\n", status);
	return status;
}

static void notice_exit(void)
{
	struct notice_dev *dev = &notice_device;

	NDBG( "[%s] in\n", __func__ );
	
	//init_waitqueue_head
	//不要
	//cdev_add
	//不要
	//cdev_init
	cdev_del( &dev->cdev );
	//device_create
	device_destroy( notice_class, notice_devno );
	//mutex_init
	mutex_destroy( &dev->notice_mtx );
	//spin_lock_init
	//dev->spin_lock;
	//alloc_chrdev_region
	unregister_chrdev_region( notice_devno, 1 );
	//class_create
	class_destroy( notice_class );

	return;
}


module_init( notice_init );
module_exit( notice_exit );

MODULE_AUTHOR("RICOH Company, LTD.");
MODULE_LICENSE("GPL");
