/***** -*- mode: C; encodeing: utf-8 -*- **************************************
 * $Id$
 *
 * Copyright (C) 2016 Ricoh Company, Ltd.  All Rights Reserved.
 *	 ORDER			 : nfc board driver
 *	 PROGRAM NAME	 : nfc.ko
 *	 FILE NAME		 : nfc.c
 *	 VERSION		 : $Revision$
 *	 DESIGNER		 : WAKABAYASHI, Ayumu
 *	 AUTHOR 		 : $Author$
 *-----------------------------------------------------
 *		 HISTORY
 *	 WAKABAYASHI, Ayumu - Aug 2016: Created.
 *	 $Log$
 ******************************************************/
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/platform_device.h>
#include <linux/syscalls.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/kthread.h>
#include <linux/semaphore.h>
#include <asm/gpio.h>
#include <asm/io.h>

#include "am.h"
#include "nfc.h"

#define SYSTEM_AREA_LEN (48)
#define WAIT_NIRQ_LOW (0)
#define NIRQ_LOW (1)
#define I2C_CMD_BUF_LEN (255)
#define READBUF_LEN (0x3cf)
#define WRITEBUF_LEN (0x3cf)

static void wait_chip_wakeup(void);
static void calc_bcc(void);
static int create_i2c_command(char *buf,
			      int buf_len,
			      int cmd,
			      unsigned int start_addr,
			      int data_len, char *data);
static int wait_interrupt_occur(int timeout_jiffies);
static char i2c_read_status(struct i2c_client *client);
static int nfc_clear_system_area(struct i2c_client *client);
static int is_msg_status_complete(char status);
static NFC_BOOLEAN is_writing(int status_code);

int wait_nirq_low;
struct i2c_client *s_i2c_client_nfc = NULL;
workqueue_struct_t *ptr_nfc_wqueue = NULL;
nfc_work_t nfc_work_struct;
int interrupted = 0;

static dev_t dev;

static DECLARE_WAIT_QUEUE_HEAD(data_wait_queue);
static DECLARE_WAIT_QUEUE_HEAD(irq_falldown_queue);

static char readbuf[READBUF_LEN];
static char writebuf[WRITEBUF_LEN];

static int last_status_code;

static struct timer_list check_rfwrite_complete_timer;

static char system_area_params[SYSTEM_AREA_LEN] = {
	0x00, 0x00, 0x00, 0x00, 0x00,	// config
	0x00,			// irqbs
	0x3f,			// irqbe
	0x02,			// hwcf
	0x01, 0x23, 0x45, 0x67,	// cfen
	0x89, 0xab, 0xcd, 0xef,	// mc

	0xaa, 0xff,		// sc
	0x02, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// idm
	0xff, 0xff,		// pmm
	0x00,			// afi
	0x84,			// hw3
	0x2e, 0x54,		// hw1

	0x00, 0x00, 0x00, 0x00,	// rorf
	0x00, 0x00, 0x00, 0x00,	// rosi
	0x00, 0x00, 0x00, 0x00,	// security
	0x47,			// tnprm
	0xac,			// hw2
	0x00,			// sl
	0x00			// bcc
};

// 製造工程で, NFC への結線ができていない場合の検証のため, I2C の初期化メッセージ失敗時
// FAILURE をたてて, open で ENODEV を戻すようにする
#define INITIALIZE_COMPLETE_WITH_FAILURE (0)
#define INITIALIZE_COMPLETE_WITH_SUCCESS (1)
static int initialize_complete = 0;

#define HW2_ADDR (0x3fd)

static void dumpmsg(char *msg, int length)
{

	char str[1024];
	int i = 0;
	int j = 0;

	memset(str, 0x00, 1024);

	for (i = 0; i < length; i++) {
		sprintf(&str[j], "%02x ", msg[i]);
		j += 3;
	}

	DPRINTK("msg: %s\n", str);

}

#ifdef NFC_DEBUG
static void dump_register(int addr, int count, char *buf)
{

	ssize_t rc = 0;
	char msg[I2C_CMD_BUF_LEN];
	int length;
	int request_len;
	int received;
	int read_addr;
	int readed_sum;
	int remain;
	char status;

	memset(msg, 0x00, I2C_CMD_BUF_LEN);
	memset(readbuf, 0x00, READBUF_LEN);

	//read_addr = (int)(*ppos);
	read_addr = addr;
	remain = count;

	if (remain > READBUF_LEN) {
		remain = READBUF_LEN;
	}
	readed_sum = 0;
	while (remain > 0) {
		if (remain > 0xfe) {
			request_len = 0xfe;
		} else {
			request_len = remain;
		}
		length = create_i2c_command(msg,
					    I2C_CMD_BUF_LEN,
					    NFC_READ_CMD,
					    read_addr, request_len, NULL);

		interrupted = NFC_FALSE;

		rc = i2c_master_send(s_i2c_client_nfc, msg, length);
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Can't send i2c read req cmd\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}

 retry_wait1:
		rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
		if (rc == 0) {
			/* timeout */
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Read Response IRQ timeout\n"
				      VT100_NORM);
			nfc_clear_system_area(s_i2c_client_nfc);

			rc = -EAGAIN;
			goto EXIT_FUNC1;
		} else if (rc < 0) {
			/* interrupted */
			goto retry_wait1;
		}

		memset(msg, 0x00, I2C_CMD_BUF_LEN);
		received =
		    i2c_master_recv(s_i2c_client_nfc, &msg[0], request_len + 1);
		if (received < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Receive Response msg failed\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}
		status = msg[0];
		if ((received - 1) > remain) {
			received = remain + 1;
		}
		//dumpmsg(&msg[1], received - 1);

		if (is_msg_status_complete(status)) {

			memcpy(buf, &msg[1], (received - 1));

			remain -= (received - 1);
			read_addr += (received - 1);
		} else {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Read Register Failed status: 0x%02x\n"
				      VT100_NORM, status);
			goto EXIT_FUNC1;
		}
	}

 EXIT_FUNC1:
	return;
}

static void dump_rreg(int addr, int count, char *buf)
{

	ssize_t rc = 0;
	char msg[I2C_CMD_BUF_LEN];
	int length;
	int request_len;
	int received;
	int read_addr;
	int readed_sum;
	int remain;
	char status;

	memset(msg, 0x00, I2C_CMD_BUF_LEN);
	memset(readbuf, 0x00, READBUF_LEN);

	//read_addr = (int)(*ppos);
	read_addr = addr;
	remain = count;

	if (remain > READBUF_LEN) {
		remain = READBUF_LEN;
	}
	readed_sum = 0;
	while (remain > 0) {
		if (remain > 0xfe) {
			request_len = 0xfe;
		} else {
			request_len = remain;
		}
		length = create_i2c_command(msg,
					    I2C_CMD_BUF_LEN,
					    NFC_RREG_CMD,
					    read_addr, request_len, NULL);

		interrupted = NFC_FALSE;

		rc = i2c_master_send(s_i2c_client_nfc, msg, length);
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Can't send i2c read req cmd\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}

 retry_wait1:
		rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
		if (rc == 0) {
			/* timeout */
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Read Response IRQ timeout\n"
				      VT100_NORM);
			nfc_clear_system_area(s_i2c_client_nfc);

			rc = -EAGAIN;
			goto EXIT_FUNC1;
		} else if (rc < 0) {
			/* interrupted */
			goto retry_wait1;
		}

		memset(msg, 0x00, I2C_CMD_BUF_LEN);
		received =
		    i2c_master_recv(s_i2c_client_nfc, &msg[0], request_len + 1);
		if (received < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Receive Response msg failed\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}
		status = msg[0];
		if ((received - 1) > remain) {
			received = remain + 1;
		}
		//dumpmsg(&msg[1], received - 1);

		if (is_msg_status_complete(status)) {

			memcpy(buf, &msg[1], (received - 1));

			remain -= (received - 1);
			read_addr += (received - 1);
		} else {
			FORCE_DPRINTK(KERN_EMERG VT100_YELLOW
				      "[nfc.ko] Read Register Failed status: 0x%02x\n"
				      VT100_NORM, status);
			goto EXIT_FUNC1;
		}
	}

 EXIT_FUNC1:
	return;
}

#endif

static int nfc_open(struct inode *inode, struct file *file)
{
	int rc = 0;

	FUNC_START;
	if (INITIALIZE_COMPLETE_WITH_SUCCESS != initialize_complete) {
		rc = -ENODEV;
	}

	FUNC_END;

	return rc;
}

static int nfc_release(struct inode *inode, struct file *file)
{
	int rc = 0;

	FUNC_START;
	FUNC_END;

	return rc;
}

static ssize_t nfc_read(struct file *file, char *buffer, size_t count,
			loff_t * ppos)
{
	ssize_t rc = 0;
	char msg[I2C_CMD_BUF_LEN];
	int length;
	int request_len;
	int received;
	int read_addr;
	int readed_sum;
	int remain;
	char status;

	DPRINTK("***** nfc: r1: %p buffer: %p count: %d ppos: %d *****\n",
		file, buffer, count, (int)(*ppos));

	FUNC_START;
	if ((file->f_flags & O_NONBLOCK)) {
		DPRINTK("nfc device does not suppor NONBLOCK mode\n");
		rc = -EAGAIN;
		goto EXIT_FUNC2;
	}
	if (count > READBUF_LEN) {

		DPRINTK("***** nfc: r2: count: %d ***** \n", count);

		rc = -EINVAL;
		goto EXIT_FUNC2;
	}

	memset(readbuf, 0x00, READBUF_LEN);

	//read_addr = (int)(*ppos);
	read_addr = 0x00;
	remain = count;

	DPRINTK("***** nfc: r4: remain: %d read_addr: 0x%x ***** \n", remain,
		read_addr);

	if (remain > READBUF_LEN) {

		DPRINTK("***** nfc: r5: ***** \n");

		remain = READBUF_LEN;
	}
	readed_sum = 0;
	while (remain > 0) {
		if (remain > 0xfe) {
			request_len = 0xfe;
		} else {
			request_len = remain;
		}
		length = create_i2c_command(msg,
					    I2C_CMD_BUF_LEN,
					    NFC_READ_CMD,
					    read_addr, request_len, NULL);
		dumpmsg(msg, length);

		DPRINTK("***** nfc: r6: remain: %d read_addr: 0x%x ***** \n",
			remain, read_addr);

		// clear interrupted flag & send message
		interrupted = NFC_FALSE;
		rc = i2c_master_send(s_i2c_client_nfc, msg, length);

		DPRINTK("***** nfc: r7: rc: %d ***** \n", rc);
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_RED
				      "[nfc.ko] Can't send i2c read req cmd\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}

 retry_wait1:
		rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
		if (rc == 0) {
			/* timeout */
			FORCE_DPRINTK(KERN_EMERG VT100_RED
				      "[nfc.ko] Read Response IRQ timeout\n"
				      VT100_NORM);

			nfc_clear_system_area(s_i2c_client_nfc);

			rc = -EAGAIN;
			goto EXIT_FUNC1;
		} else if (rc < 0) {

			DPRINTK("***** nfc: r8: rc: %d ***** \n", rc);

			/* interrupted */
			goto retry_wait1;
		}

		received =
		    i2c_master_recv(s_i2c_client_nfc, &msg[0], request_len + 1);
		if (received < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_RED
				      "[nfc.ko] Receive Response msg failed\n"
				      VT100_NORM);
			goto EXIT_FUNC1;
		}
		status = msg[0];
		DPRINTK
		    ("nfc: read status: 0x%02x reaceived: %d request_len: %d\n",
		     status, received, request_len);
		if ((received - 1) > remain) {
			received = remain + 1;
		}

		dumpmsg(&msg[1], received - 1);

		memcpy(&readbuf[read_addr], &msg[1], received);
		remain -= (received - 1);
		read_addr += (received - 1);

	}

	rc = copy_to_user(buffer, &readbuf, read_addr);
	if (0 == rc) {
		rc = (count - remain);
	} else {
		FORCE_DPRINTK(KERN_EMERG VT100_RED
			      "[nfc.ko] Can't recv copy user data\n"
			      VT100_NORM);
		goto EXIT_FUNC1;
	}

 EXIT_FUNC1:
	if (0 > am_read_data_complete()) {
		rc = -EAGAIN;
	}

 EXIT_FUNC2:

	FUNC_END;

	DPRINTK("***** nfc: r12: return: %d ***** \n", rc);

	return rc;
}

static ssize_t nfc_write(struct file *file, const char *buffer, size_t count,
			 loff_t * ppos)
{
	ssize_t rc = 0;
	char msg[I2C_CMD_BUF_LEN];
	int length;
	int write_once_len;
	int write_addr;
	int remain;
	char status;

	DPRINTK("nfc: w1 : file: 0x%p buffer: 0x%p count: %d ppos: 0x%p(%d) \n",
		file, buffer, count, ppos, (int)(*ppos));

	FUNC_START;
	if ((file->f_flags & O_NONBLOCK)) {
		DPRINTK("nfc device does not suppor NONBLOCK mode\n");
		rc = -EAGAIN;
		goto EXIT_FUNC;
	}
	if (count > WRITEBUF_LEN) {
		DPRINTK("nfc: count is greater than WRITEBUF_LEN\n");
		rc = -EINVAL;
		goto EXIT_FUNC;
	}

	memset(writebuf, 0x00, WRITEBUF_LEN);
	rc = copy_from_user(writebuf, (void __user *)buffer, count);
	if (rc != 0) {
		//DPRINTK("nfc: w2-1 : ***** copy_from_user failed *****\n");
		rc = -1;
		goto EXIT_FUNC;
	}
	DPRINTK("nfc: w3 : writebuf: %s\n", writebuf);

	write_addr = 0x00;
	remain = count;
	while (remain > 0) {
		if (remain > 0xfb) {
			write_once_len = 0xfb;
		} else {
			write_once_len = remain;
		}
		length = create_i2c_command(msg,
					    I2C_CMD_BUF_LEN,
					    NFC_WRITE_CMD,
					    write_addr,
					    write_once_len,
					    &writebuf[write_addr]);
		dumpmsg(msg, length);

		// clear interrupted flag & send message
		interrupted = NFC_FALSE;
		rc = i2c_master_send(s_i2c_client_nfc, msg, length);
		if (rc < 0) {
			FORCE_DPRINTK(KERN_EMERG VT100_RED
				      "[nfc.ko] Can't send write i2c cmd\n"
				      VT100_NORM);
			rc = -EAGAIN;
			goto EXIT_FUNC;
		}

 retry_wait2:
		rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
		if (rc == 0) {
			/* timeout */
			FORCE_DPRINTK(KERN_EMERG VT100_RED
				      "[nfc.ko] Write response IRQ timeout\n"
				      VT100_NORM);

			nfc_clear_system_area(s_i2c_client_nfc);
			rc = -EAGAIN;

			goto EXIT_FUNC;
		} else if (rc < 0) {
			/* interrupted */
			goto retry_wait2;

			DPRINTK("***** nfc: w3-1 : rc: %d *****\n", rc);
		}

		status = i2c_read_status(s_i2c_client_nfc);
		remain -= (write_once_len);
		write_addr += (write_once_len);

		DPRINTK("***** nfc: w4 : status: 0x%02x *****\n", status);

		if (!is_msg_status_complete(status)) {
			rc = -EIO;
			goto EXIT_FUNC;
		}
	}

	DPRINTK("nfc: w5 : \n");
	rc = count;

 EXIT_FUNC:

	FUNC_END;

	DPRINTK("nfc: w6 : ret: %d\n", rc);

	return rc;
}

static unsigned int nfc_poll(struct file *file, poll_table * wait)
{
	unsigned int rc = 0;

	FUNC_START;

	if (NFC_TRUE == am_is_data_available()) {
		DPRINTK("***** nfc_poll: data already available *****\n");
		rc = (POLLIN | POLLRDNORM);
	} else {
		poll_wait(file, &data_wait_queue, wait);
		if (NFC_TRUE == am_is_data_available()) {
			DPRINTK("***** nfc_poll: data received *****\n");
			rc = (POLLIN | POLLRDNORM);
		}
	}

	FUNC_END;

	return rc;
}

/**
 * 0 if the condition evaluated to false after the timeout elapsed, 
 * 1 if the condition evaluated to true after the timeout elapsed, 
 * the remaining jiffies (at least 1) 
 *  if the condition evaluated to true before the timeout elapsed, 
 * or -ERESTARTSYS if it was interrupted by a signal.
*/
static int wait_interrupt_occur(int timeout_jiffies)
{

	int n = 0;
	long timeout;

	if (NFC_TRUE == interrupted) {
		n = 1;
		interrupted = NFC_FALSE;
		DPRINTK("already interrupted, wait_event skipped\n");
		return n;
	} else {

		timeout = timeout_jiffies;
		n = wait_event_interruptible_timeout(irq_falldown_queue,
						     (interrupted != NFC_FALSE),
						     timeout);
		if (n < 0) {
			DPRINTK
			    ("wait irq_falldown_queue down is interrupted\n");
		}
	}
	return n;
}

static int create_i2c_command(char *buf,
			      int buf_len,
			      int cmd,
			      unsigned int start_addr, int data_len, char *data)
{

	int rc = -1;

	//DPRINTK("create_i2c_command: buf: 0x%p buf_len: %d cmd: 0x%x start_addr: 0x%x data_len: %d data: 0x%p\n", buf, buf_len, cmd, start_addr, data_len, data);

	switch (cmd) {
	case NFC_READ_CMD:
		if (NULL == buf || buf_len < (1 + 2 + 1)) {
			break;
		} else {
			rc = (1 + 2 + 1);
			buf[0] = (char)cmd;
			buf[1] = (char)((0x0000ff00 & start_addr) >> 8);
			buf[2] = (char)((0x000000ff & start_addr));
			// TODO: check data_len should 0 to 0xfe
			buf[3] = (char)data_len;
		}
		break;
	case NFC_WRITE_CMD:
		if (NULL == buf || buf_len < (1 + 2 + 1 + data_len)) {
			break;
		} else {
			rc = (1 + 2 + 1 + data_len);
			buf[0] = (char)cmd;
			buf[1] = (char)((0x0000ff00 & start_addr) >> 8);
			buf[2] = (char)((0x000000ff & start_addr));
			// TODO: check data_len should 0 to 0xfb
			buf[3] = (char)data_len;
			memcpy(&buf[4], data, data_len);
		}
		break;
	case NFC_RREG_CMD:
		if (NULL == buf || buf_len < (1)) {
			break;
		} else {
			rc = (1);
			buf[0] = (char)cmd;
		}
		break;
	case NFC_WREG_CMD:
		if (NULL == buf || buf_len < (2)) {
			break;
		} else {
			rc = (2);
			buf[0] = (char)cmd;
			buf[1] = (char)data[0];
		}
		break;
	case NFC_STATUS_CMD:
		rc = 0;
		break;
	}

	return rc;
}

static char i2c_read_status(struct i2c_client *client)
{

	char status[1];
	int received;

	received = i2c_master_recv(client, status, 1);
	if (received < 0) {
		DPRINTK("i2c_read_status failed");
	}

	return status[0];
}

static int is_msg_status_complete(char status)
{

	int rc = NFC_FALSE;

	// RES_CMD == 0x05
	if (0x05 == (status & 0x0f)) {
		rc = NFC_TRUE;
	} else if (0x00 == status) {
		rc = NFC_TRUE;
	}

	return rc;
}

static void rf_write_timer_handler(unsigned long data)
{
	DPRINTK("***** rf_write_timer_handler *****\n");

	am_receive_data_from_rf_complete();
	wake_up_interruptible(&data_wait_queue);
}

static void nfc_wqueue_handler(work_struct_t * work)
{
	nfc_work_t *ptr_work = NULL;
	int status_code;
	int _status_code;
	long expires;
	int nirq;
	int check_count;
	int check_remain;

	FUNC_START;

	check_count = 0;
	//      check_remain = 12;

	ptr_work = (nfc_work_t *) work;
	status_code = i2c_read_status(s_i2c_client_nfc);

	while ((check_count < CHECK_NIRQ_MAX_COUNT) || CHECK_NIRQ_MAX_COUNT < 0) {

		check_count++;
		mdelay(10);
		nirq = gpio_get_value(NFC_IRQ_GPIO_NO);

		if (am_is_idle() || am_is_rf_writing_data()) {

			if (NFC_TRUE == is_writing(status_code)
			    && am_is_idle()) {

				DPRINTK("nfc_wqueue_handler ptn1\n");

				am_receive_data_from_rf_start();

				init_timer(&check_rfwrite_complete_timer);
				//add timer
				check_rfwrite_complete_timer.expires =
				    jiffies + RF_WRITE_DATA_TIMEOUT;
				check_rfwrite_complete_timer.data = 0;
				check_rfwrite_complete_timer.function =
				    rf_write_timer_handler;

				add_timer(&check_rfwrite_complete_timer);

			} else if (NFC_TRUE == is_writing(status_code)
				   && am_is_rf_writing_data()) {

				DPRINTK("nfc_wqueue_handler ptn2\n");

				expires = jiffies + RF_WRITE_DATA_TIMEOUT;
				mod_timer(&check_rfwrite_complete_timer,
					  expires);

			} else {
				DPRINTK
				    ("*** BOTTOM_HALF data_available: IGNORED status: 0x%02x\n",
				     status_code);
			}

			DPRINTK
			    ("*** BOTTOM_HALF data_available: %d status: 0x%02x last: 0x%02x is_writing: %d idle: %d am_is_rf_wriging_data: %d ***\n",
			     am_is_data_available(), status_code,
			     last_status_code, is_writing(status_code),
			     am_is_idle(), am_is_rf_writing_data());
			last_status_code = status_code;
		} else {
			DPRINTK
			    ("*** BOTTOM_HALF(NOT IDLE of RF_WRITING)  data_available: %d last: 0x%02x ***\n",
			     am_is_data_available(), last_status_code);
		}

		if (0x05 == (0x0f & status_code)) {
			wake_up_interruptible(&irq_falldown_queue);

			DPRINTK
			    ("*** BOTTOM_HALF(RESPONSE NORMAL END FOR CMD_RES) status: %02x\n",
			     status_code);
		}

		check_remain = CHECK_NIRQ_MAX_COUNT_PER_INT_HANDLER;
		while (0x00 == nirq) {
			check_remain--;
			mdelay(10);

			if (check_remain <= 0) {

				_status_code = status_code;
				status_code = i2c_read_status(s_i2c_client_nfc);

				DPRINTK
				    ("##### nirq fault old status: 0x%02x status: 0x%02x #####\n",
				     _status_code, status_code);

				break;
			} else {
				nirq = gpio_get_value(NFC_IRQ_GPIO_NO);
				DPRINTK("##### nirq: 0x%02x #####\n", nirq);
			}
		}

		if (nirq > 0x00) {
			break;
		}

	}

// EXIT_FUNC:
	FUNC_END;
	return;
}

static irqreturn_t nfc_isr(int irq, void *dev_id)
{
	wait_nirq_low = NIRQ_LOW;
	if (!interrupted) {
		interrupted = NFC_TRUE;
	}

	DPRINTK("nfc_isr: interrupted: %d\n", interrupted);

	queue_work(ptr_nfc_wqueue, (work_struct_t *) & nfc_work_struct);

	DPRINTK("nfc_isr: interrupted: put workqueue\n");

	return IRQ_HANDLED;
}

static struct file_operations nfc_fops = {
	.owner = THIS_MODULE,
	.open = nfc_open,
	.release = nfc_release,
	.read = nfc_read,
	.write = nfc_write,
	.poll = nfc_poll,
};

static void nfc_platform_release(struct device *dev)
{
	FUNC_START;
	FUNC_END;
	return;
}

static const struct platform_device_id nfc_id[] = {
	{NFC_NAME, 0},
	{}
};

MODULE_DEVICE_TABLE(platform, nfc_id);

static struct platform_driver nfc_driver = {
	.driver = {
		   .name = NFC_NAME,
		   },
	.id_table = nfc_id,
};

static struct platform_device nfc_device = {
	.name = NFC_NAME,
	.dev = {
		.release = nfc_platform_release,
		}
};

static void wait_chip_wakeup()
{
	mdelay(NFC_WAKEUP_TIME_MSEC);
}

static NFC_BOOLEAN is_writing(int status_code)
{

	NFC_BOOLEAN rc = NFC_FALSE;

	if ((0x80 & status_code) != 0) {
		rc = NFC_TRUE;
	}

	return rc;
}

static void calc_bcc()
{

	int sum = 0;

	sum += system_area_params[0x3d5 - 0x3d0];
	sum += system_area_params[0x3d6 - 0x3d0];
	sum += system_area_params[0x3d7 - 0x3d0];
	sum += system_area_params[0x3d8 - 0x3d0];
	sum += system_area_params[0x3d9 - 0x3d0];
	sum += system_area_params[0x3da - 0x3d0];
	sum += system_area_params[0x3db - 0x3d0];
	sum += system_area_params[0x3dc - 0x3d0];
	sum += system_area_params[0x3dd - 0x3d0];
	sum += system_area_params[0x3de - 0x3d0];
	sum += system_area_params[0x3df - 0x3d0];

	sum += system_area_params[0x3e0 - 0x3d0];
	sum += system_area_params[0x3e1 - 0x3d0];
	sum += system_area_params[0x3e2 - 0x3d0];
	sum += system_area_params[0x3e3 - 0x3d0];
	sum += system_area_params[0x3e4 - 0x3d0];
	sum += system_area_params[0x3e5 - 0x3d0];
	sum += system_area_params[0x3e6 - 0x3d0];
	sum += system_area_params[0x3e7 - 0x3d0];
	sum += system_area_params[0x3e8 - 0x3d0];
	sum += system_area_params[0x3e9 - 0x3d0];
	sum += system_area_params[0x3ea - 0x3d0];
	sum += system_area_params[0x3eb - 0x3d0];
	sum += system_area_params[0x3ec - 0x3d0];
	sum += system_area_params[0x3ed - 0x3d0];
	sum += system_area_params[0x3ee - 0x3d0];
	sum += system_area_params[0x3ef - 0x3d0];

	sum += system_area_params[0x3fc - 0x3d0];
	sum += system_area_params[0x3fd - 0x3d0];

	system_area_params[(SYSTEM_AREA_LEN) - 1] = 0x100 - (0x000000ff & sum);
}

static int nfc_clear_system_area(struct i2c_client *client)
{

	ssize_t rc = 0;
	char msg[I2C_CMD_BUF_LEN];
	char resetcmd[1] = { 0x01 };
	int length;
	char status;

	FUNC_START;
	calc_bcc();

	memset(msg, 0x00, I2C_CMD_BUF_LEN);
	length = create_i2c_command(msg,
				    I2C_CMD_BUF_LEN,
				    NFC_WRITE_CMD,
				    NFC_SYSTEM_AREA_ADDR,
				    NFC_SYSTEM_AREA_LEN, system_area_params);
	DPRINTK("nfc: 1 : length: %d s_i2c_client_nfc: %p msg: %p length: %d\n",
		length, s_i2c_client_nfc, msg, length);

	// clear interrupted flag & send message
	interrupted = NFC_FALSE;
	rc = i2c_master_send(s_i2c_client_nfc, msg, length);
	if (rc < 0) {
		FORCE_DPRINTK(KERN_EMERG VT100_RED
			      "[nfc.ko] Can't send initialize i2c message\n"
			      VT100_NORM);
		goto EXIT_FUNC;
	}
	dumpmsg(msg, length);

	status = i2c_read_status(client);
	DPRINTK("nfc: 2 : status: 0x%02x\n", status);
	rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
	if (rc <= 0) {
		FORCE_DPRINTK(KERN_EMERG VT100_RED
			      "[nfc.ko] Initialize message response timeout\n"
			      VT100_NORM);
		goto EXIT_FUNC;
	}
	DPRINTK("nfc: 3 :\n");
	status = i2c_read_status(client);
	DPRINTK("nfc: 4 : status: 0x%02x\n", status);
	if (!is_msg_status_complete(status)) {
		// 初期データの書き込みに失敗した場合.
		FORCE_DPRINTK
		    ("[nfc.ko] write configuration data failed 0x%02x\n",
		     status);
		rc = -1;
		goto EXIT_FUNC;
	}
	DPRINTK("nfc: 5 :\n");

	memset(msg, 0x00, I2C_CMD_BUF_LEN);

	DPRINTK("nfc: 6 :");
	length =
	    create_i2c_command(msg, I2C_CMD_BUF_LEN, NFC_WREG_CMD, 0x00, 0,
			       resetcmd);

	//DPRINTK("nfc: 7 : length: %d\n", length);

	// clear interrupted flag & send message
	interrupted = NFC_FALSE;
	rc = i2c_master_send(s_i2c_client_nfc, msg, length);
	if (rc < 0) {
		FORCE_DPRINTK(KERN_EMERG VT100_RED
			      "[nfc.ko] Can't send system reset\n" VT100_NORM);
		goto EXIT_FUNC;
	}
	dumpmsg(msg, length);

	rc = wait_interrupt_occur(IRQ_WAIT_TIMEOUT);
	if (rc <= 0) {
		FORCE_DPRINTK(KERN_EMERG VT100_RED
			      "[nfc.ko] Reset message response timeout\n"
			      VT100_NORM);
		goto EXIT_FUNC;
	}
	status = i2c_read_status(client);
	//DPRINTK("nfc: 8 : status: 0x%02x\n", status);
	if (!is_msg_status_complete(status)) {
		// リセット命令に失敗した場合
		DPRINTK("nfc: 8-1 : ***** reset failed %d *****\n", status);
		rc = -1;
		goto EXIT_FUNC;
	}

 EXIT_FUNC:
	FUNC_END;

	return rc;
}

static int nfc_probe(struct i2c_client *client, const struct i2c_device_id *idp)
{
	nfc_priv_t *priv = NULL;
	int err = -EINVAL;
	int error = -EINVAL;

	FUNC_START;
	//DPRINTK("nfc_probe: i2c_client: 0x%p\n", client);

	/*---- デバイス情報作成 ----*/
	priv = kzalloc(sizeof(nfc_priv_t), GFP_KERNEL);
	if (!priv) {
		DPRINTK("failed to allocate driver data\n");
		error = -ENOMEM;
		goto error6;
	}

	/* デバイス構造体にプライベートデータをセット  */
	dev_set_drvdata(&client->dev, priv);

	/* クライアント情報セット */
	priv->nfc_client = client;

	priv->nfc_major = MAJOR(dev);
	cdev_init(&(priv->c_dev), &nfc_fops);
	priv->c_dev.owner = THIS_MODULE;

	// キャラクタデバイスの登録
	err = cdev_add(&(priv->c_dev), dev, 1);
	if (err < 0) {
		FORCE_DPRINTK(KERN_WARNING "[nfc.ko] cdev_add failed\n");
		goto error5;
	}

	/* デバイスクラス生成 */
	priv->nfc_class = class_create(THIS_MODULE, NFC_DEVICE_NAME);
	if (IS_ERR(priv->nfc_class)) {
		error = -EBUSY;
		goto error4;
	}
	/* デバイスクラスからデバイス生成 */
	device_create(priv->nfc_class, NULL, MKDEV(priv->nfc_major, 0), "%s%d",
		      NFC_DEVICE_NAME);

	/* ドライバ登録 */
	err = platform_driver_register(&nfc_driver);
	if (err != 0) {
		FORCE_DPRINTK(KERN_NOTICE
			      "[nfc.ko] platform_driver_register error:nfc");
		error = -EBUSY;
		goto error3;
	}

	/* デバイス登録 */
	err = platform_device_register(&nfc_device);
	if (err != 0) {
		FORCE_DPRINTK(KERN_NOTICE
			      "[nfc.ko] platform_device_register error:nfc");
		error = -EBUSY;
		goto error2;
	}

	wait_chip_wakeup();

	DPRINTK("Setup workqueue...\n");
	ptr_nfc_wqueue = create_workqueue(NFC_WQUEUE_NAME);
	if (!ptr_nfc_wqueue) {
		FORCE_DPRINTK(KERN_WARNING
			      "[nfc.ko] Can't create workqueue_struct! orz!!!\n");
		error = -1;
		goto error1;
	}
	INIT_WORK((work_struct_t *) (&nfc_work_struct), nfc_wqueue_handler);
	nfc_work_struct.priv = priv;

	// register gpio interrupt handler
	DPRINTK("Setup Interrupt...\n");
	wait_nirq_low = WAIT_NIRQ_LOW;
	priv->nfc_irq = gpio_to_irq(NFC_IRQ_GPIO_NO);
	DPRINTK("Request irq is %d...\n", priv->nfc_irq);
	error = request_irq(priv->nfc_irq,
			    nfc_isr,
			    IRQF_TRIGGER_FALLING,
			    NFC_INTERRUPT_DEVICE_NAME,
			    &(priv->nfc_client->dev));
	if (error) {
		FORCE_DPRINTK("[nfc.ko] Unable to request nfc IRQ. (%d)\n",
			      error);
		goto error0;
	}

	FUNC_END;
	return 0;

 error0:
	disable_irq(priv->nfc_irq);
	free_irq(priv->nfc_irq, &(priv->nfc_client->dev));
 error1:
	flush_workqueue(ptr_nfc_wqueue);
	destroy_workqueue(ptr_nfc_wqueue);
 error2:
	platform_driver_unregister(&nfc_driver);
 error3:
	class_destroy(priv->nfc_class);
 error4:
	cdev_del(&(priv->c_dev));
 error5:
	dev_set_drvdata(&priv->nfc_client->dev, NULL);
	kfree(priv);
 error6:

	FUNC_END;
	return error;
}

static int nfc_remove(struct i2c_client *client)
{
	nfc_priv_t *priv = NULL;

	FUNC_START;

	priv = dev_get_drvdata(&client->dev);

	disable_irq(priv->nfc_irq);
	free_irq(priv->nfc_irq, &(client->dev));

	flush_workqueue(ptr_nfc_wqueue);
	destroy_workqueue(ptr_nfc_wqueue);
	ptr_nfc_wqueue = NULL;

	platform_device_unregister(&nfc_device);
	platform_driver_unregister(&nfc_driver);

	class_destroy(priv->nfc_class);
	cdev_del(&(priv->c_dev));
	dev_set_drvdata(&client->dev, NULL);

	nfc_work_struct.priv = NULL;
	kfree(priv);
	priv = NULL;
	FUNC_END;
	return 0;
}

static void nfc_shutdown(struct i2c_client *client)
{
	DPRINTK("***** NFC_SHUTDOWN OCCUR *****\n");
}

static const struct i2c_device_id nfc_i2c_id[] = {
	{NFC_NAME, 0},
	{}
};

MODULE_DEVICE_TABLE(i2c, nfc_i2c_id);

static struct i2c_driver nfc_i2c_driver = {
	.driver = {
		   .name = NFC_NAME,
		   },
	.probe = nfc_probe,
	.remove = nfc_remove,
	.id_table = nfc_i2c_id,
	.shutdown = nfc_shutdown,
};

static int __init nfc_init(void)
{
	int ret = 0;
	struct i2c_adapter *adapter = NULL;
	struct i2c_board_info info = {
		.type = {'\0'},
		.flags = (unsigned short)0,
		.addr = (unsigned short)(NFC_I2C_WR_ADDR),
		.platform_data = NULL,
		.archdata = NULL,
		.of_node = NULL,
		.irq = 0,
	};

	FUNC_START;

	// キャラクタデバイス番号の動的取得
	ret = alloc_chrdev_region(&dev, 0, 1, "nfc");
	if (ret < 0) {
		FORCE_DPRINTK(KERN_WARNING
			      "[nfc.ko] I2C: driver cannot get major id.\n");
		goto exit_chrdev;
	}

	if (ret < 0) {
		FORCE_DPRINTK(KERN_WARNING
			      "[nfc.ko] alloc_chrdev_region failed\n");
		return ret;
	}

	ret = i2c_add_driver(&nfc_i2c_driver);
	if (ret) {
		FORCE_DPRINTK(KERN_WARNING
			      "[nfc.ko] I2C: Driver registration failed, module not inserted.\n");
		goto exit_init;
	}
	strncpy(info.type, NFC_NAME, I2C_NAME_SIZE);
	adapter = i2c_get_adapter(NFC_I2C_CHANNEL);
	s_i2c_client_nfc = i2c_new_device(adapter, &info);

	// 制御エリアの初期化 (NIRQ==Low を検知するので割り込みハンドラの登録後実行)
	if (0 > nfc_clear_system_area(s_i2c_client_nfc)) {
		// clear で問題があってもリカバリすることは特にできないので警告だけ出しておく
		FORCE_DPRINTK("[nfc.ko] failed to initialize device\n");
		initialize_complete = INITIALIZE_COMPLETE_WITH_FAILURE;
	} else {
		initialize_complete = INITIALIZE_COMPLETE_WITH_SUCCESS;
	}

	FUNC_END;
	return ret;

 exit_chrdev:
	unregister_chrdev_region(dev, 1);

 exit_init:
	return ret;
}

static void __exit nfc_exit(void)
{
	struct i2c_adapter *adapter = NULL;

	FUNC_START;
	adapter = i2c_get_adapter(NFC_I2C_CHANNEL);
	i2c_put_adapter(adapter);
	i2c_unregister_device(s_i2c_client_nfc);
	s_i2c_client_nfc = NULL;
	i2c_del_driver(&nfc_i2c_driver);

	unregister_chrdev_region(dev, 1);
	FUNC_END;

	return;
}

module_init(nfc_init);
module_exit(nfc_exit);

MODULE_AUTHOR("RICOH Company, LTD.");
MODULE_LICENSE("GPL");
