/*
 * Marvell Pegmatite PCIE PHY driver
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/of_address.h>
#include <linux/phy/phy.h>
#include <linux/platform_device.h>
#include "phy-mv-pegmatite-pcie.h"

typedef struct mv_pegmatite_pcie_phy_s {
	struct phy *phy;
	struct clk *phyclk0;
	struct clk *phyclk1;
	struct clk *ioclk;
	int clk_pio;
	void __iomem *pcie_app_base;
	void __iomem *pcie_reset_control;
	void __iomem *pcie_fabric10;
	unsigned int mode_2x2;
	unsigned int txmargin;
} mv_pegmatite_pcie_phy_t;


void
   phy_writel ( unsigned int val, void *addr) 
{
//	printk(">>> %s %d - addr: %08x val: %08x \n", __func__,__LINE__,addr, val);
	writel(val, addr);
}
void
   phy_writew ( unsigned int val, void *addr) 
{
//	printk(">>> %s %d - addr: %08x val: %04x \n", __func__,__LINE__,addr, val);
	writew(val, addr);
}


#define APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM2_RSTN 0x02
#define APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM4_RSTN 0x01

int pcie_pclk_check(mv_pegmatite_pcie_phy_t *pcie_phy)
{
	u16 data=0;
	int rc=0;
	pci_app_regs_t *appRegs = pcie_phy->pcie_app_base;

	data = readw(&appRegs->LANE_STATUS1);
	if ((data & (0x1)) != 0x1)
		rc = -1;

	return(rc);
}

int pcie_pll_lock_check(mv_pegmatite_pcie_phy_t *pcie_phy)
{
	u16 data=0;
	int rc=0;
	int i=0;
	pci_app_regs_t *appRegs   = pcie_phy->pcie_app_base;

	data = readw(&appRegs->POWER_REG0);
	while (((data & (0x100)) != 0x100) && (i < 500))
	{
		i++;
		msleep(10);
		data = readw(&appRegs->POWER_REG0);
	}

	if ((data & (0x100)) != 0x100)
		rc = -1;

	return(rc);
}


int pcie_ref_clock_check(mv_pegmatite_pcie_phy_t *pcie_phy)
{
	u16 data=0;
	int rc=0;
	pci_app_regs_t *appRegs   = pcie_phy->pcie_app_base;

	/* check for PM_REFCLK_VALID bit */
	data = readw(&appRegs->LANE_STATUS1);
	if ((data & (0x80)) != 0x80)
		rc = -1;

	return(rc);
}




static int mv_pegmatite_pcie_phy_init(struct phy *phy)
{
	mv_pegmatite_pcie_phy_t *pcie_phy = phy_get_drvdata(phy);
	int i, rc=0;
	pci_app_regs_t *appRegs = pcie_phy->pcie_app_base;
	unsigned int phyNumber, val;

	/* Turn on the external reference clock */
	if (pcie_phy->clk_pio) {
	    gpio_set_value(pcie_phy->clk_pio, 1);
	    /* Wait for the clock to be valid */
	    msleep(1);
	}

	rc = clk_prepare_enable(pcie_phy->phyclk0);
	if (rc < 0) {
		dev_err(&phy->dev, "failed to enable source clk 0\n");
		return rc;
	}

	rc = clk_prepare_enable(pcie_phy->phyclk1);
	if (rc < 0) {
		dev_err(&phy->dev, "failed to enable source clk 1\n");
		clk_disable(pcie_phy->phyclk0);
		return rc;
	}

	rc = clk_prepare_enable(pcie_phy->ioclk);
	if (rc < 0) {
		dev_err(&phy->dev, "failed to enable ioclk\n");
		clk_disable(pcie_phy->phyclk0);
		clk_disable(pcie_phy->phyclk1);
		return rc;
	}

	/* de-assert reset */
	if (pcie_phy->pcie_reset_control) {
		phy_writel( APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM2_RSTN |
			APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM4_RSTN,
			pcie_phy->pcie_reset_control);
		dmb();
	}

	/*
	 * If we're in 2x2 mode, route the 2nd controllers I/O to the
	 * correct pins.
	 */
	if (pcie_phy->mode_2x2) {
		val = readl(pcie_phy->pcie_fabric10);
		val |= PCIE_FABRIC10_2x2_MODE;
		writel(val, pcie_phy->pcie_fabric10);
	}

	/*
	 * Need to configure all PHYs attached to this controller.
	 */
	for (phyNumber=0; phyNumber < 4; phyNumber++) {
		/*
		 * Choose the PHY number
		 */
		phy_writel(phyNumber, &appRegs->PhyLaneNum);
		msleep(10);

		/*
		 * Set the clock source HI register
		 */
		{
			unsigned int
				clockSrcHi;
			static unsigned int clockSrcHiVals_1x4[] = {
				CLK_SRC_HI_LANE_START | CLK_SRC_HI_LANE_MASTER,
				0,
				0,
				CLK_SRC_HI_LANE_BREAK
			};
			static unsigned int clockSrcHiVals_2x2[] = {
				CLK_SRC_HI_LANE_START | CLK_SRC_HI_LANE_MASTER,
				CLK_SRC_HI_LANE_BREAK,
				CLK_SRC_HI_LANE_START | CLK_SRC_HI_LANE_MASTER,
				CLK_SRC_HI_LANE_BREAK
			};

			
			if (! pcie_phy->mode_2x2) {
				clockSrcHi = clockSrcHiVals_1x4[phyNumber];
			}
			else {
				clockSrcHi = clockSrcHiVals_2x2[phyNumber];
			}
			
			phy_writew(clockSrcHi, &appRegs->GLOB_CLK_SRC_HI);

			msleep(10);
		}

	} // for each PHY

	/*
	 * Set MULTICAST BIT and PHY RESET for all PHYs
	 */
	for (phyNumber=0; phyNumber < 4; phyNumber++) {
		/*
		 * Choose the PHY number
		 */
		writel(phyNumber, &appRegs->PhyLaneNum);

		/*
		 * Set the clock source HI register
		 */
		val = (GLOB_CLK_CTRL_MULTICAST | GLOB_CLK_CTRL_REFCLK_DIV_2 | GLOB_CLK_CTRL_SOFT_RESET);

		writew(val, &appRegs->GLOB_CLK_CTRL);

	} // for each PHY

	/* Choose PHY 0 for rest of configuration.  */
	writel(0, &appRegs->PhyLaneNum);

	/* 8-bit, 250MHz PIPE */
	writew((GLOB_CLK_CTRL_MULTICAST    |
		GLOB_CLK_CTRL_REFCLK_DIV_2 |
		GLOB_CLK_CTRL_FIXED_PCLK   |
		GLOB_CLK_CTRL_SOFT_RESET), &appRegs->GLOB_CLK_CTRL);

	/* clock source low */
	writew((GLOB_CLK_SRC_LO_PLL_READY_DELAY(4)    |
		GLOB_CLK_SRC_LO_BUNDLE_PERIOD_32_NSEC |
		GLOB_CLK_SRC_LO_BUNDLE_PERIOD_FIXED   |
		GLOB_CLK_SRC_LO_PCLK_FROM_PLL), &appRegs->GLOB_CLK_SRC_LO);

	/* Clear 0x52 bit 10 for soft_reset_no_reg */
	val = readl(&appRegs->appReg_2148);
	val |= 1 << 10;
	writel(val, &appRegs->appReg_2148);

	/* power reg0 */
	writew((POWER_REG0_RSRVD_15          |
		POWER_REG0_PU_PLL            |
		POWER_REG0_PU_RX             |
		POWER_REG0_PU_TX             |
		POWER_REG0_RSRVD_11          |
		POWER_REG0_PU_DFE            |
		POWER_REG0_PHY_MODE_PCIE     |
		POWER_REG0_REFCLK_SEL_100MHZ), &appRegs->POWER_REG0);

	/* Following hardwired values are from Marvell verilog simulations */
	writew(0x30, &appRegs->CAL_REG0);

	/* Configure for external reference clock at 100MHz. */
	writew(0xa00a, &appRegs->MISC_FREQ_SEL);

	/* Change REFCLK_SEL bit to select 25MHz on REFCLK_IN_RXSIDE_G2 group */
	val = readl(&appRegs->MISC_FREQ_SEL);
	val &= ~(1 << 10);
	writel(val, &appRegs->MISC_FREQ_SEL);

	/* Enable 500MHz clock oout to reflclk_buffer! */
	val = readl(&appRegs->MISC_FREQ_SEL);
	val |= 1 << 7;
	writel(val, &appRegs->MISC_FREQ_SEL);

	writew(0x2b00, &appRegs->LANE_ALIGN_REQ0);
	writew(0x00a4, &appRegs->GLOB_CLK_CTRL);

	udelay(10);

	/******** New PHY init sequence (06-23-2014) *************/
	/* MAC bus width operating mode and core_clk
	   & 13 mode_width_32
	   & 14 mode
	   & 15 mode_core_clk_freq_sel */
	/* COMPIPE_GLOB_CLK_CTRL ( 0x1c1, 0xD0782704 ) */

	/* GLOB_CLK_CTRL[9] = 0 */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val &= ~(1 << 9);
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* GLOB_CLK_CTRL[3] = 0, dynamic pclk mode, 32-bit width */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val &= ~(1 << 3);
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* GLOB_CLK_CTRL[2] = 1, mode_fixed_pclk, 16-bit@250MHz @ 5.0GT/s */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val |= 1 << 2;
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* GLOB_CLK_CTRL[1] = 0, PIPE_RESET */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val &= ~(1 << 1);
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* GLOB_CLK_CTRL[0] = 1, PIPE_SFT_RESET */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val |= 1 << 0;
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* TXDETRX delay time*/
	val = readl(&appRegs->appReg_2740);
	val &= ~(0xff << 0);
	val |= 0x1e << 0;
	writel(val, &appRegs->appReg_2740);

	/* Register 0x25 - PHY_GEN_MAX[1:0] - Should be set to a value of 0b01 for PCIE Gen2 Max */
	val = readl(&appRegs->appReg_2094);
	val &= ~(3 << 10);
	val |= 1 << 10;
	writel(val, &appRegs->appReg_2094);

	/* USE_MAX_PLL_RATE */
	val = readl(&appRegs->CAL_REG0);
	val &= ~(1 << 12);
	writel(val, &appRegs->CAL_REG0);

	/* MISC_FREQ_SEL[7] = 1, reg_clk500m_en */
	val = readl(&appRegs->MISC_FREQ_SEL);
	val |= 1 << 7;
	writel(val, &appRegs->MISC_FREQ_SEL);

	/* MISC_FREQ_SEL[6] = 1, reg_txdclk_2x_sel */
	val = readl(&appRegs->MISC_FREQ_SEL);
	val |= 1 << 6;
	writel(val, &appRegs->MISC_FREQ_SEL);

	writew(0xa8ad, &appRegs->appReg_2170);
	writew(0x9608, &appRegs->appReg_2104);
	writew(0x2428, &appRegs->appReg_2108);
	writew(0xba50, &appRegs->appReg_2034);
	writew(0x2c76, &appRegs->appReg_203c);
	writew(0x0ce2, &appRegs->appReg_2044);
	writew(0x1149, &appRegs->appReg_2048);
	writew(0x0949, &appRegs->appReg_2040);
	writew(0x0949, &appRegs->appReg_2038);

	/* PLL_READY_DLY[2:0] 0x7 - (Register 0x1C3[7:5]) */
	val = readl(&appRegs->GLOB_CLK_SRC_LO);
	val |= 7 << 5;
	writel(val, &appRegs->GLOB_CLK_SRC_LO);

	/* Register 0x25 - PHY_GEN_MAX[1:0] - Should be set to a value of 0b01 for PCIE Gen2 Max */
	val = readl(&appRegs->appReg_2094);
	val &= ~(3 << 10);
	val |= 1 << 10;
	writel(val, &appRegs->appReg_2094);

	/* Register 0x26 - PHY_GEN_TX and PHY_GEN_RX */
	val = readl(&appRegs->appReg_2098);
	val &= ~(0xff << 0);
	val |= 0x11 << 0;
	writel(val, &appRegs->appReg_2098);

	/* Register 0x48 bit 12 - IDLE Sync Enable */
	val = readl(&appRegs->appReg_2120);
	val |= 1 << 12;
	writel(val, &appRegs->appReg_2120);

	/* Clear 0x52 bit 10 for soft_reset_no_reg */
	val = readl(&appRegs->appReg_2148);
	val &= ~(1 << 10);
	writel(val, &appRegs->appReg_2148);

	/* release PIPE soft_reset */
	val = readl(&appRegs->GLOB_CLK_CTRL);
	val &= ~(1 << 0);
	writel(val, &appRegs->GLOB_CLK_CTRL);

	/* Lane Margin Override */
	val = readl(&appRegs->LANE_CFG0);
	val &= ~(1 << 0);
	val |= ((pcie_phy->txmargin & 0x7) << 1);
	writel(val, &appRegs->LANE_CFG0);

	val = readl(&appRegs->LANE_CFG1);
	val |= (1 << 15);
	writel(val, &appRegs->LANE_CFG1);

	val = readl(&appRegs->GLOB_TEST_CTRL);
	val |= (1 << 2);
	writel(val, &appRegs->GLOB_TEST_CTRL);

	/* check for pclk */
	PCIE_PRINTK(DBG_INFO, "PCIe Checking for PCLK\n");

	/*
	 * Set wait timeout to 2 seconds, observed Gr2 as root complex when
	 * transition from L2 to ready it took about +900 ms compare to +3 ms
	 * when transtion from L1.
	 */
	for (i=0; i<40000; i++) {
		rc = pcie_pclk_check(pcie_phy);
		if (rc == 0) {
			break;
		}

		udelay(50);
	}

	msleep(5);

	if (rc < 0) {
		pr_crit("PCIe PCLK not detected.\n");
		goto fail;
	}

	/* Clear multicast */
	for (phyNumber=0; phyNumber < 4; phyNumber++) {

		writel(phyNumber, &appRegs->PhyLaneNum);

		writel(0x24, &appRegs->GLOB_CLK_CTRL);
	}

	writel(0, &appRegs->PhyLaneNum);

	/* wait 10ms for PHY clock to lock */
	msleep(10);

	/* check for ref clock in each PHY*/
	PCIE_PRINTK(DBG_INFO, "PCIe checking for Ref Clock\n");
	rc = pcie_ref_clock_check(pcie_phy);

	if (rc < 0) {
		PCIE_PRINTK(DBG_ERROR, "PCIe Ref Clock not detected.  PHY: %d \n", phyNumber);
		goto fail;
	}

	/* check for PLL lock */
	PCIE_PRINTK(DBG_INFO, "PCIe checking for PLL lock\n");
	rc = pcie_pll_lock_check(pcie_phy);
	if (rc < 0)
	{
		PCIE_PRINTK(DBG_ERROR, "PCIe PLL not locked.  PHY: %d \n", phyNumber);
		goto fail;
	}

fail:
	clk_disable_unprepare(pcie_phy->ioclk);
	return(rc);
}

static int mv_pegmatite_pcie_phy_exit(struct phy *phy)
{
	mv_pegmatite_pcie_phy_t *pcie_phy = phy_get_drvdata(phy);
	int rc;

	rc = clk_prepare_enable(pcie_phy->ioclk);
	if (rc < 0) {
		dev_err(&phy->dev, "failed to enable ioclk\n");
		return rc;
	}
	/* assert reset */
	if (pcie_phy->pcie_reset_control) {
		unsigned int val = readl(pcie_phy->pcie_reset_control);
		val &=  ~APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM2_RSTN;
		val &=  ~APMU_CLKRSTGEN_PCIERSTCTL_PCIE_DM4_RSTN;
		phy_writel(val, pcie_phy->pcie_reset_control);

	}
	clk_disable_unprepare(pcie_phy->ioclk);

	clk_disable_unprepare(pcie_phy->phyclk0);
	clk_disable_unprepare(pcie_phy->phyclk1);

	/* Turn off the external reference clock */
	if (pcie_phy->clk_pio)
	    gpio_set_value(pcie_phy->clk_pio, 0);

	return 0;
}

static struct phy_ops mv_pegmatite_pcie_phy_ops = {
	.init		= mv_pegmatite_pcie_phy_init,
	.exit		= mv_pegmatite_pcie_phy_exit,
	.owner		= THIS_MODULE,
};

static int mv_pegmatite_pcie_phy_probe(struct platform_device *pdev)
{
	mv_pegmatite_pcie_phy_t *pcie_phy;
	struct device *dev = &pdev->dev;
	struct resource *res;
	struct phy_provider *phy_provider;
	int ret = 0;
	struct device_node *np = dev->of_node;

	pcie_phy = devm_kzalloc(dev, sizeof(mv_pegmatite_pcie_phy_t), GFP_KERNEL);
	if (!pcie_phy)
		return -ENOMEM;

	pcie_phy->clk_pio = of_get_named_gpio(np, "clk-gpio", 0);
	if (pcie_phy->clk_pio >= 0) {
		if (!gpio_request(pcie_phy->clk_pio, "PCIE_CLOCK_ENB"))
			gpio_direction_output(pcie_phy->clk_pio, 1);
		else
			dev_warn(dev, "GPIO request failed for %d\n", pcie_phy->clk_pio);
	} else {
		dev_warn(dev, "unable to get clk-gpio for PCIe\n");
		pcie_phy->clk_pio = 0;
	}

	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "pcie_app_base");
	pcie_phy->pcie_app_base = devm_ioremap_resource(dev, res);

	if (IS_ERR(pcie_phy->pcie_app_base))
		return PTR_ERR(pcie_phy->pcie_app_base);


	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "pcie_reset_control");
	pcie_phy->pcie_reset_control = devm_ioremap_resource(dev, res);

	if (IS_ERR(pcie_phy->pcie_reset_control))
		return PTR_ERR(pcie_phy->pcie_reset_control);


	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "pcie_fabric10");
	pcie_phy->pcie_fabric10 = devm_ioremap_resource(dev, res);

	if (IS_ERR(pcie_phy->pcie_fabric10))
		return PTR_ERR(pcie_phy->pcie_fabric10);

	dev_set_drvdata(dev, pcie_phy);

	/* 
	 * Due to design flaw, we need both the DM2 and DM4 clocks enabled to talk to the PHY
	 */
	pcie_phy->phyclk0 = devm_clk_get(dev, "phy_clk0");
	if (IS_ERR(pcie_phy->phyclk0)) {
		dev_err(dev, "failed to get clk for PHY\n");
		return PTR_ERR(pcie_phy->phyclk0);
	}
	
	pcie_phy->phyclk1 = devm_clk_get(dev, "phy_clk1");
	if (IS_ERR(pcie_phy->phyclk1)) {
		dev_err(dev, "failed to get clk for PHY 1\n");
		return PTR_ERR(pcie_phy->phyclk1);
	}
	
	pcie_phy->ioclk = devm_clk_get(dev, "ioclk");
	if (IS_ERR(pcie_phy->ioclk)) {
		dev_err(dev, "failed to get clk for io\n");
		return PTR_ERR(pcie_phy->ioclk);
	}
	
	pcie_phy->phy = devm_phy_create(dev, dev->of_node,
					&mv_pegmatite_pcie_phy_ops, NULL);

	if (IS_ERR(pcie_phy->phy)) {
		dev_err(dev, "failed to create PHY\n");
		return PTR_ERR(pcie_phy->phy);
	}

	phy_set_drvdata(pcie_phy->phy, pcie_phy);

	phy_provider = devm_of_phy_provider_register(dev,
					of_phy_simple_xlate);
	if (IS_ERR(phy_provider)) {
		return PTR_ERR(phy_provider);
	}

	if (of_property_read_u32(np, "2x2", &pcie_phy->mode_2x2)) {
		dev_err(&pdev->dev, "no 2x2 mode found\n");
		ret = -ENODEV;
	}

	if (of_property_read_u32(np, "txmargin", &pcie_phy->txmargin)) {
		pcie_phy->txmargin = 0;
	}

	return 0;
}

static const struct of_device_id mv_pegmatite_pcie_phy_of_match[] = {
	{ .compatible = "marvell,pegmatite-pcie-phy" },
	{ },
};

MODULE_DEVICE_TABLE(of, mv_pegmatite_pcie_phy_of_match);

static struct platform_driver mv_pegmatite_pcie_phy_driver = {
	.probe	= mv_pegmatite_pcie_phy_probe,
	.driver = {
		.of_match_table	= mv_pegmatite_pcie_phy_of_match,
		.name  = "marvell,pcie-phy",
		.owner = THIS_MODULE,
	}
};

module_platform_driver(mv_pegmatite_pcie_phy_driver);

MODULE_DESCRIPTION("Marvell Pegmatite PCIE PHY driver");
MODULE_LICENSE("GPL");
