/*
 * PCIe support for Pegmatite platforms.
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 *
 */
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/interrupt.h>
#include <linux/irqchip/chained_irq.h>
#include <linux/init.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>
#include <linux/of_pci.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/clk.h>
#include <linux/gpio.h>
#include <linux/phy/phy.h>
#include "pci-pegmatite.h"
#include "../pci.h"   /* need access to pci_platform_pm_ops structure */


extern int is_fpga(void);

static pegmatite_pcie_top_t *pcie_top;

struct irq_domain *p_intx_irq_domain = NULL;

pcie_intctl_chipdata_t pic;

static void pegmatite_pcie_setup_hw (struct pci_sys_data *sys, pegmatite_pcie_controller_t *controller);
static void pcie_err_irq_init (pegmatite_pcie_controller_t *controller);

static inline u32 pegmatite_read_lclcfg(pegmatite_pcie_controller_t *controller, u32 reg)
{
	return readl(controller->pcie_lclcfg_base + reg);
}

static inline void pegmatite_write_lclcfg(pegmatite_pcie_controller_t *controller, u32 val, u32 reg)
{
	writel(val, controller->pcie_lclcfg_base + reg);
}


static void pegmatite_pcie_set_local_bus_nr(pegmatite_pcie_controller_t *controller, int nr)
{
	u32 stat;

	stat = pegmatite_read_lclcfg(controller, PCIE_STAT_OFF);
	stat &= ~PCIE_STAT_BUS;
	stat |= nr << 8;
	pegmatite_write_lclcfg(controller, stat, PCIE_STAT_OFF);
}

/*
 * Initialize the configuration space of the PCI-to-PCI bridge
 * associated with the given PCIe interface.
 */
static void pegmatite_sw_pci_bridge_init (pegmatite_pcie_controller_t *controller)
{
	pegmatite_pcie_sw_bridge_t *bridge = &controller->bridge;

	memset(bridge, 0, sizeof(pegmatite_pcie_sw_bridge_t));

	bridge->class           = PCI_CLASS_BRIDGE_PCI;
	bridge->vendor          = PCI_VENDOR_ID_MARVELL;
	bridge->device          = pegmatite_read_lclcfg(controller, PCIE_DEV_ID_OFF) >> 16;
	bridge->revision        = pegmatite_read_lclcfg(controller, PCIE_DEV_REV_OFF) & 0xff;
	bridge->header_type     = PCI_HEADER_TYPE_BRIDGE;
	bridge->cache_line_size = 0x10;

	/* We support 32 bits I/O addressing */
	bridge->iobase  = PCI_IO_RANGE_TYPE_32;
	bridge->iolimit = PCI_IO_RANGE_TYPE_32;
}

/*
 * Read the configuration space of the PCI-to-PCI bridge associated to
 * the given PCIe interface.
 */
static int pegmatite_pcie_sw_bridge_read(pegmatite_pcie_controller_t *controller,
					 unsigned int where, int size, u32 *value)
{
	pegmatite_pcie_sw_bridge_t *bridge = &controller->bridge;

	switch (where & ~3) {
	case PCI_VENDOR_ID:
		*value = bridge->device << 16 | bridge->vendor;
		break;

	case PCI_COMMAND:
		*value = bridge->command;
		break;

	case PCI_CLASS_REVISION:
		*value = bridge->class << 16 | bridge->interface << 8 |
			 bridge->revision;
		break;

	case PCI_CACHE_LINE_SIZE:
		*value = bridge->bist << 24 | bridge->header_type << 16 |
			 bridge->latency_timer << 8 | bridge->cache_line_size;
		break;

	case PCI_BASE_ADDRESS_0 ... PCI_BASE_ADDRESS_1:
		*value = bridge->bar[((where & ~3) - PCI_BASE_ADDRESS_0) / 4];
		break;

	case PCI_PRIMARY_BUS:
		*value = (bridge->secondary_latency_timer << 24 |
			  bridge->subordinate_bus         << 16 |
			  bridge->secondary_bus           <<  8 |
			  bridge->primary_bus);
		break;

	case PCI_IO_BASE:
		*value = (bridge->secondary_status << 16 |
			  bridge->iolimit          <<  8 |
			  bridge->iobase);
		break;

	case PCI_MEMORY_BASE:
		*value = (bridge->memlimit << 16 | bridge->membase);
		break;

	case PCI_PREF_MEMORY_BASE:
		*value = 0;
		break;

	case PCI_IO_BASE_UPPER16:
		*value = (bridge->iolimitupper << 16 | bridge->iobaseupper);
		break;

	case PCI_ROM_ADDRESS1:
		*value = 0;
		break;

	case PCI_INTERRUPT_LINE:
		/* LINE PIN MIN_GNT MAX_LAT */
		*value = 0;
		break;

	default:
		*value = 0xffffffff;
		return PCIBIOS_BAD_REGISTER_NUMBER;
	}

	if (size == 2)
		*value = (*value >> (8 * (where & 3))) & 0xffff;
	else if (size == 1)
		*value = (*value >> (8 * (where & 3))) & 0xff;

	return PCIBIOS_SUCCESSFUL;
}

/* Write to the PCI-to-PCI bridge configuration space */
static int pegmatite_pcie_sw_bridge_write(pegmatite_pcie_controller_t *controller,
					  unsigned int where, int size, u32 value)
{
	pegmatite_pcie_sw_bridge_t *bridge = &controller->bridge;
	u32 mask, reg;
	int err;

	if (size == 4)
		mask = 0x0;
	else if (size == 2)
		mask = ~(0xffff << ((where & 3) * 8));
	else if (size == 1)
		mask = ~(0xff << ((where & 3) * 8));
	else
		return PCIBIOS_BAD_REGISTER_NUMBER;

	err = pegmatite_pcie_sw_bridge_read(controller, where & ~3, 4, &reg);

	if (err)
		return err;

	value = (reg & mask) | value << ((where & 3) * 8);

	switch (where & ~3) {
	case PCI_COMMAND:
	{
		bridge->command = value & 0xffff;
		break;
	}

	case PCI_BASE_ADDRESS_0 ... PCI_BASE_ADDRESS_1:
		bridge->bar[((where & ~3) - PCI_BASE_ADDRESS_0) / 4] = value;
		break;

	case PCI_IO_BASE:
		/*
		 * We also keep bit 1 set, it is a read-only bit that
		 * indicates we support 32 bits addressing for the
		 * I/O
		 */
		bridge->iobase = (value & 0xff) | PCI_IO_RANGE_TYPE_32;
		bridge->iolimit = ((value >> 8) & 0xff) | PCI_IO_RANGE_TYPE_32;
		break;

	case PCI_MEMORY_BASE:
		bridge->membase = value & 0xffff;
		bridge->memlimit = value >> 16;
		break;

	case PCI_IO_BASE_UPPER16:
		bridge->iobaseupper = value & 0xffff;
		bridge->iolimitupper = value >> 16;
		break;

	case PCI_PRIMARY_BUS:
		bridge->primary_bus             = (value >>  0) & 0xff;
		bridge->secondary_bus           = (value >>  8) & 0xff;
		bridge->subordinate_bus         = (value >> 16) & 0xff;
		bridge->secondary_latency_timer = (value >> 24) & 0xff;

		pegmatite_pcie_set_local_bus_nr(controller, bridge->secondary_bus);
		break;

	default:
		break;
	}

	return PCIBIOS_SUCCESSFUL;
}




static inline pegmatite_pcie_controller_t *sys_to_controller(struct pci_sys_data *sys)
{
	return sys->private_data;
}

/**
 ****************************************************************************************************
 *
 * \brief   Find correct controller to do the access to.
 *
 * \param
 *
 * \returns
 *
 *****************************************************************************************************/
static pegmatite_pcie_controller_t *
pegmatite_pcie_find_controller (struct pci_bus *bus, unsigned int devfn, unsigned int *isRootBusP)
{
	int i;
	pegmatite_pcie_controller_t *controller;

	PCIE_PRINTK(DBG_FIND_CNTL, ">>> %s %d - pcie_top: %p bus: %p devfn: %x \n", __func__,__LINE__,pcie_top,bus, devfn);

	/*
	 * Assume not a root bus initially.
	 */
	*isRootBusP = 0;

	for (i = 0; i < pcie_top->nr_controllers; i++) {

		controller = pcie_top->controllers[i];


		PCIE_PRINTK(DBG_FIND_CNTL, ">>> %s %d - controller: rootBus: %p pribus: %x secbus: %x sob: %x \n",
			    __func__,__LINE__,
			    controller->rootBus,
			    controller->bridge.primary_bus,
			    controller->bridge.secondary_bus,
			    controller->bridge.subordinate_bus);

		/*
		 * If we are rolling through the root busses, then only respond to devfn 0.
		 */
		if (bus == controller->rootBus) {
			if (devfn == 0) {
				PCIE_PRINTK(DBG_FIND_CNTL, ">>> %s %d - returns: %p (bus 0)\n", __func__,__LINE__,controller);

				/*
				 * Change flag for root bus.
				 */
				*isRootBusP = 1;
				return controller;
			}
			else { // devfn not 0 on root bus
				return NULL;
			}
		}

		/*
		 * If we aren't on a root bus, then we must find the correct PCI 'bridge'
		 * This is done by checking the bus ranges bounded by the secondary
		 * bus and subordinate bus.
		 */
		if (bus->number >= controller->bridge.secondary_bus &&
		    bus->number <= controller->bridge.subordinate_bus) {
			PCIE_PRINTK(DBG_FIND_CNTL, ">>> %s %d - returns: %p (bus non-zero)\n", __func__,__LINE__,controller);
			return controller;
		}
	} // for each controller

	return NULL;
}

/*
 * Function exported so the mailbox driver can register its irq handler with
 * PCIe driver. This is due to the fact that the mailbox was part of the PCIe
 * block in G2 and whenever we reset the PCIe block, the mailbox function will
 * get reset as well. To get around this, PCIe EP interrupt handler will
 * handle the PCIe related interrupts first before make a callback to the
 * mailbox interrupt handler.
 */
void pcie_mailbox_irq_handler_register(void (*callback_f), void *callback_context, int sysirq)
{
	pegmatite_pcie_controller_t *controller;
	int i;

	for (i = 0; i < pcie_top->nr_controllers; i++) {
		controller = pcie_top->controllers[i];
		if (controller->irq == sysirq) {
			controller->callback_f = callback_f;
			controller->callback_context = callback_context;
		}
	}
};
EXPORT_SYMBOL(pcie_mailbox_irq_handler_register);

/*
 * Function exported so the mailbox driver can register a callback function
 * with the PCIe driver such that it will get a notify callback when the PCIe
 * block is resumed (after a PCIe block reset).
 */
void pcie_mailbox_resume_notify_register(void (*callback_f), int sysirq)
{
	pegmatite_pcie_controller_t *controller;
	int i;

	for (i = 0; i < pcie_top->nr_controllers; i++) {
		controller = pcie_top->controllers[i];
		if (controller->irq == sysirq) {
			controller->resume_callback_f = callback_f;
		}
	}
};
EXPORT_SYMBOL(pcie_mailbox_resume_notify_register);

/*
 * PCIe EP interrupt handle workqueue
 *
 * This function will be responsible for now will handle the Link Request
 * Reset interrupt from the PCIe block and also it will check if the interrupt
 * is for the PCIe block before calling the mailbox interrupt handler. This is
 * because the mailbox function is part of the PCIe block in G2 and we shared
 * the same interrupt line to GIC.
 */
static irqreturn_t pcie_ep_hard_irq(int irq, void *dev_id)
{
	pegmatite_pcie_controller_t *controller = dev_id;
	pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;
	struct device *dev = &pcie_top->pdev->dev;
	int ret = 0;

	if (readl(&appRegs->AIntP) & (1 << 24)) {
		/*
		 * To decrement its init_count so phy_init can be called again
		 */
		ret = phy_exit(controller->phy);
		if (ret) {
			dev_err(dev, "%s %d - exit phy: %pK rc = %d\n", __func__,__LINE__, controller->phy, ret);
		}
	}

	return(IRQ_WAKE_THREAD);
};

static irqreturn_t pcie_ep_irq_workqueue(int irq, void *dev_id)
{
	pegmatite_pcie_controller_t *controller = dev_id;
	pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;
	struct device *dev = &pcie_top->pdev->dev;
	int ret = 0;
	irqreturn_t rc;

	/*
	 * if there is link reset request, we want to make that the priority
	 */
	if (readl(&appRegs->AIntP) & (1 << 24)) {
		ret = phy_init(controller->phy);
		if (ret) {
			dev_err(dev, "Failed to init phy: %d \n", ret);
			BUG();
		}

		pegmatite_pcie_setup_hw(NULL, controller);

		/*
		 * Call the resume function
		 */
		if (controller->resume_callback_f)
			controller->resume_callback_f();

		writel((1 << 24), &appRegs->AIntAck);
		pcie_err_irq_init(controller);

		return(IRQ_HANDLED);
	} else {
		rc = IRQ_NONE;

		/*
		 * Call the mailbox interrupt handler
		 */
		if (controller->callback_f)
			rc = controller->callback_f(controller->irq, controller->callback_context);

		return(rc);
	}
}

/* INTx Functions */

/**
 * pcie_intx_mask_irq - Disable the appropriate INTx IRQ
 * @data: IRQ data
 */
static void pcie_intx_mask_irq(struct irq_data *data)
{
    pcie_intctl_chipdata_t *cd = data->chip_data;
    pcie_intctl_regsters_t *intctl = NULL;
    unsigned int irq = data->irq;
    int ind;

    if (!cd)
        return;

    intctl = cd->intctl;

    PCIE_PRINTK(DBG_INTX,"%s %d: : irq=0x%08x\n", __func__, __LINE__, irq);

    irq -= cd->irq_base;
    ind = irq;

    /* Disable the INTx interrupt */
    writel(0, &intctl->RC_INTX_IRQ[ind].En);
}

/**
 * pcie_intx_ack_irq - Ack the appropriate INTx IRQ
 * @data: IRQ data
 */
static void pcie_intx_ack_irq(struct irq_data *data)
{
    unsigned int irq = data->irq;

    PCIE_PRINTK(DBG_INTX, "%s: irq=0x%08x\n", __func__, irq);
}

/**
 * pcie_intx_unmask_irq - Enable the appropriate INTx IRQ
 * @data: IRQ data
 */
static void pcie_intx_unmask_irq(struct irq_data *data)
{
    pcie_intctl_chipdata_t *cd = data->chip_data;
    pcie_intctl_regsters_t *intctl = NULL;
    unsigned int irq = data->irq;
    int ind;

    if (!cd)
        return;

    intctl = cd->intctl;

    PCIE_PRINTK(DBG_INTX,"%s %d:: irq=0x%08x\n", __func__, __LINE__, irq);

    irq -= cd->irq_base;
    ind = irq;

    /* Enable the INTx interrupt */
    writel(1, &intctl->RC_INTX_IRQ[ind].En);
}

/* INTx Chip Descriptor */
static struct irq_chip pcie_intx_irq_chip = {
    .name	= "PCI-INTX",
    .irq_ack	= pcie_intx_ack_irq,
    .irq_mask	= pcie_intx_mask_irq,
    .irq_unmask	= pcie_intx_unmask_irq,
};

/**
 * pegmatite_pcie_intx_map - Set the handler for the INTx and mark IRQ as valid
 * @domain: IRQ domain
 * @virq: Virtual IRQ number
 * @hwirq: HW interrupt number
 *
 * Return: Always returns 0.
 */
static int pegmatite_pcie_intx_map(struct irq_domain *domain, unsigned int virq,
			     irq_hw_number_t hwirq)
{
    PCIE_PRINTK(DBG_INTX,"************%s %d: virq is %d \n", __func__, __LINE__, virq);
	irq_set_chip_and_handler(virq, &pcie_intx_irq_chip, handle_simple_irq);
    irq_set_chip_data(virq, &pic);
    set_irq_flags(virq, IRQF_VALID);

	return 0;
}

/* IRQ domain operations */
static struct irq_domain_ops pegmatite_pcie_intx_ops = {
	.map = pegmatite_pcie_intx_map,
};

static void pcie_irq_handler (unsigned int this_irq, struct irq_desc *desc)
{
    int ind;
	struct irq_chip
		*parent_chip = irq_desc_get_chip(desc);
	pegmatite_pcie_controller_t
		*controller = irq_get_handler_data(this_irq);

    pcie_intctl_regsters_t *intctl = pic.intctl;

	raw_spin_lock(&desc->lock);
	chained_irq_enter(parent_chip, desc);

	/* error handler */
#ifdef ENABLE_ERROR_IRQS
	pcie_error_irq_handler(1);
#endif

	/* pme handler */
#ifdef ENABLE_PME_IRQS
	pcie_pme_irq_handler();
#endif

    /* Handle INTx interrupts */
    for (ind = 0; ind < NUM_INTX_IRQS; ind++) {
        if (p_intx_irq_domain && readl(&intctl->RC_INTX_IRQ[ind].Cnt)) {
	        generic_handle_irq(irq_find_mapping(p_intx_irq_domain, ind));
        }
    }

	/* Handle MSI interrupts */
	pegmatite_msi_irq_handler(controller);

	chained_irq_exit(parent_chip, desc);
	raw_spin_unlock(&desc->lock);
}

static void pcie_err_irq_init (pegmatite_pcie_controller_t *controller)
{
	u32 val;
	pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;

	/*
	 * Enable the link request reset interrupt
	 */
	val = readl(&appRegs->AIntEn);
	val |= (0x1 << 24);
	writel(val, &appRegs->AIntEn);
}

static void pcie_irq_init (pegmatite_pcie_controller_t *controller)
{
	int ret;
    pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;

	/*
	 * TODO: interrupts for endpoint?
	 */
	if (controller->endpoint) {
		ret = request_threaded_irq(controller->irq, pcie_ep_hard_irq, pcie_ep_irq_workqueue, (IRQF_ONESHOT | IRQF_SHARED), "PCIE-EP-IRQ", controller);
		if (ret)
			pr_info("%s: request_irq failed (%d)\n", __func__, ret);
		else
			pcie_err_irq_init(controller);

		return;
	}

    pic.intctl = &appRegs->RC_MSI_IRQ[0].Mask;
    p_intx_irq_domain = irq_domain_add_linear(controller->pdev->dev.of_node,
						NUM_INTX_IRQS, &pegmatite_pcie_intx_ops, NULL);

    if (!p_intx_irq_domain) {
        /*
         * Note: In the event of a failure to get a IRQ domain for INTx interrupts,
         * do NOT bail. Instead continue with initializing MSI interrupts. This would atleast
         * provide an opportunity for Endpoint drivers that support MSI interrupts, to obtain
         * a valid IRQ number and register their corresponding interrupt handler.
         */
        pr_err("******%s %d: Failed to get INTX IRQ domain\n", __func__, __LINE__);
    }

	pegmatite_msi_init(controller);
	irq_set_handler_data(controller->irq, controller);

	irq_set_chained_handler(controller->irq, pcie_irq_handler);
}

/**
 * pcie_free_intx_irq_domain - Free up INTx IRQ mapping(s) and domain
 */
static void pcie_free_intx_irq_domain(void)
{
    int ind;
    u32 irq;

    if (!p_intx_irq_domain)
        return;

#if 0
    for (ind = 0; ind < NUM_INTX_IRQS; ind++) {
        irq = irq_find_mapping(intx_irq_domain, ind);
        if (irq > 0)
            irq_dispose_mapping(irq);
    }
#else
    /* For now, only operate on INTA interrupt */
    ind = 0;

    irq = irq_find_mapping(p_intx_irq_domain, ind);
    if (irq > 0)
        irq_dispose_mapping(irq);
#endif

    irq_domain_remove(p_intx_irq_domain);
}

static int pcie_link_up(pegmatite_pcie_controller_t *controller)
{
	u32 app_link_up, cap_link_up;
	pci_cfg_t *cfgRegs = (pci_cfg_t *)controller->pcie_lclcfg_base;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	int rc = 0;

	app_link_up = (readl(&appRegs->AStat) & PCIE_APP_STATUS_REG_LINK_UP);
	cap_link_up = (readl(&cfgRegs->LLinkContStatus) & PCIE_CAP_LINK_STATUS_LINK_UP_MASK);
	if (app_link_up && cap_link_up) {
		rc = 1;
	}

	return (rc);
}

/**
 ****************************************************************************************************
 *
 * \brief   Prepare hardware for doing config cycle.  We must determine whether to perform a type
 *          0 cycle for the locally attached bridges and devices attached to them, or type 1 when
 *          we have a device on the far side of a switch.
 *
 * \param
 *
 * \returns
 *
 *****************************************************************************************************/
static void pegmatite_set_cfgControl(struct pci_bus *bus, pegmatite_pcie_controller_t *controller, u32 devfn)
{
	u32 CfgControl;
	pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;

	/*
	 * TODO: how to determine when to use a Type-1?  Likely need to look at the subordinate bus numbers.
	 */
	if (bus->number > controller->bridge.secondary_bus) {
		CfgControl = (PCIE_CFG_CTRL_TYPE_1 | (bus->number << PCIE_CFG_CTRL_BUS_NUM_SH) | devfn);
	}
	else {
		CfgControl = (bus->number << PCIE_CFG_CTRL_BUS_NUM_SH);
	}

	writel(CfgControl, &appRegs->CfgControl);
}


static int pegmatite_pcie_read_conf(struct pci_bus *bus, unsigned int devfn,
				int where, int size, u32 *val)
{
	unsigned int isRootBus;
	void * addr;
	unsigned long flags;
	int ret = PCIBIOS_SUCCESSFUL;
	pegmatite_pcie_controller_t *controller;

	PCIE_PRINTK(DBG_CFG, ">>> %s %d - bus: %p bus->number: %x devfn: %x PCI_SLOT: %x where: %x size: %x parent: %p (from %ps) \n",
		    __func__,__LINE__,
		    bus,
		    bus->number,
		    devfn,
		    PCI_SLOT(devfn),
		    where, size,
		    bus->parent,
		    __builtin_return_address(0) );

	/*
	 * Find controller for this bus.
	 */
	controller = pegmatite_pcie_find_controller(bus, devfn, &isRootBus);

	if (!controller) {
		PCIE_PRINTK(DBG_CFG, ">>> %s %d - no controller\n", __func__,__LINE__);

		*val = 0xffffffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}


	/* Access the emulated PCI-to-PCI bridge */
	if (isRootBus) {
		ret = pegmatite_pcie_sw_bridge_read(controller, where, size, val);

		PCIE_PRINTK(DBG_CFG, ">>> %s %d - bridge read returns: *val: %x \n", __func__,__LINE__, *val);
		return(ret);
	}

	/*
	 * look for downstream device
	 */
	if (!pcie_link_up(controller))
	{
		PCIE_PRINTK(DBG_LINK, ">>> %s %d, No Link, bus: %d, dev: %d\n", __func__, __LINE__, bus->number, PCI_SLOT(devfn));
		*val = 0xffffffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	/*
	 * On the secondary bus, we don't want to expose any other
	 * device than the device physically connected in the PCIe
	 * slot, visible in slot 0. In slot 1, there's a special
	 * Marvell device that only makes sense when the Armada is
	 * used as a PCIe endpoint.
	 */
	if ((bus->number == controller->bridge.secondary_bus) && (PCI_SLOT(devfn) != 0)) {
		PCIE_PRINTK(DBG_CFG, ">>> %s %d - secondary\n", __func__,__LINE__);

		*val = 0xffffffff;
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	/*
	 * target an external device.
	 */
	spin_lock_irqsave(&controller->pcie_lock, flags);

	/*
	 * set up cfgControl register to get correct config cycle type.
	 */
	pegmatite_set_cfgControl(bus, controller, devfn);


	/*
	 * this is local config space
	 */
	addr = controller->pcie_trgcfg_base;


	*val = readl(addr + (where & ~3));

	/* check for UR */
	{
		pci_app_regs_t *appRegs = (pci_app_regs_t *)controller->pcie_app_base;

		if (__raw_readl(&appRegs->AIntP) & (1 << 9))
		{
			__raw_writel((1 << 9), &appRegs->AIntAck);
			*val = 0xffffffff;
			spin_unlock_irqrestore(&controller->pcie_lock, flags);

			PCIE_PRINTK(DBG_CFG, ">>> %s %d - UR reported \n", __func__,__LINE__);

			return PCIBIOS_SUCCESSFUL;
		}
	}

	if (size == 1)
		*val = (*val >> (8 * (where & 3))) & 0xff;
	else if (size == 2)
		*val = (*val >> (8 * (where & 3))) & 0xffff;

	spin_unlock_irqrestore(&controller->pcie_lock, flags);

	PCIE_PRINTK(DBG_CFG, ">>> %s %d - *val: %x \n", __func__,__LINE__, *val);

	return ret;
}

/**
 ****************************************************************************************************
 *
 * \brief   write a value to config space of either internal emulated bridge, or external PCI connected
 *          device.
 *
 * \param
 *
 * \returns
 *
 *****************************************************************************************************/
static int pegmatite_pcie_write_conf(struct pci_bus *bus, unsigned int devfn,
				 int where, int size, u32 val)
{
	unsigned int isRootBus;
	pegmatite_pcie_controller_t *controller;
	void * addr;
	unsigned long flags;
	int ret = PCIBIOS_SUCCESSFUL;

	PCIE_PRINTK(DBG_CFG, ">>> %s %d - bus: %p bus->number: %x devfn: %x PCI_SLOT: %x where: %x size: %x val: %x parent: %p (from %ps) \n",
		    __func__,__LINE__,
		    bus,
		    bus->number,
		    devfn,
		    PCI_SLOT(devfn),
		    where, size, val,
		    bus->parent,
		    __builtin_return_address(0) );

	/*
	 * Find controller for this bus.
	 */
	controller = pegmatite_pcie_find_controller(bus, devfn, &isRootBus);

	if (!controller)
		return PCIBIOS_DEVICE_NOT_FOUND;


	/*
	 * If bus 0, then access the emulated PCI-to-PCI bridge
	 */
 	if (isRootBus) {
		PCIE_PRINTK(DBG_CFG, ">>> %s %d - bridge write\n", __func__,__LINE__);
		return pegmatite_pcie_sw_bridge_write(controller, where, size, val);
	}

	/*
	 * On the secondary bus, we don't want to expose any other
	 * device than the device physically connected in the PCIe
	 * slot, visible in slot 0. In slot 1, there's a special
	 * Marvell device that only makes sense when the Armada is
	 * used as a PCIe endpoint.
	 */
	if ((bus->number == controller->bridge.secondary_bus) && (PCI_SLOT(devfn) != 0)) {
		return PCIBIOS_DEVICE_NOT_FOUND;
	}



	/*
	 * look for downstream device
	 */
	if (!pcie_link_up(controller))
	{
		PCIE_PRINTK(DBG_LINK, "%s:%d, No Link, bus: %d, dev: %d\n", __func__, __LINE__, bus->number, PCI_SLOT(devfn));
		return PCIBIOS_DEVICE_NOT_FOUND;
	}

	spin_lock_irqsave(&controller->pcie_lock, flags);

	/*
	 * set up cfgControl register to get correct config cycle type.
	 */
	pegmatite_set_cfgControl(bus, controller, devfn);


	/*
	 * this is local config space
	 */
	PCIE_PRINTK(DBG_CFG, ">>> %s %d - bus is non-0 and slot zero.  core_base\n", __func__,__LINE__);
	addr = controller->pcie_trgcfg_base;

	if (size == 4) {
		writel(val, addr + (where & ~3));
	} else if (size == 2) {
		writew(val, addr + (where & ~3) + (where & 3));
	} else if (size == 1) {
		writeb(val, addr + (where & ~3) + (where & 3));
	} else {
		ret = PCIBIOS_BAD_REGISTER_NUMBER;
	}

	spin_unlock_irqrestore(&controller->pcie_lock, flags);
	return ret;
}

static struct pci_ops pegmatite_pcie_ops = {
	.read	= pegmatite_pcie_read_conf,
	.write	= pegmatite_pcie_write_conf,
};





void pcie_enable_msi_int(pegmatite_pcie_controller_t *controller)
{
	int i;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;

	for (i=0; i<4; i++) {
		writel(0xffffffff, &appRegs->RC_MSI_IRQ[i].Mask);
	}
}

void pcie_enable_intx_int(pegmatite_pcie_controller_t *controller)
{
	int i;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;

	for (i = 0; i < NUM_INTX_IRQS; i++) {
		writel(1, &appRegs->RC_INTX_IRQ[i].En);
	}
}

static pegmatite_pcie_controller_t * msi_controller = NULL;
int pcie_gen_msi_irq(unsigned int irq)
{
	pci_app_regs_t *appRegs;

	if (msi_controller == NULL) {
		PCIE_PRINTK(DBG_MSI, "%s: No MSI controller data\n", __func__);
		return(-1);
	}
	appRegs = (pci_app_regs_t *)msi_controller->pcie_app_base;

	if ((readl(&appRegs->EP_MSI_Gen_En) & 1) == 0) {
		PCIE_PRINTK(DBG_MSI, "%s: MSI Gen Disabled\n", __func__);
		return(-1);
	}

	writel(0, &appRegs->EP_MSI_Gen);

	return(0);
}
EXPORT_SYMBOL(pcie_gen_msi_irq);

void pcie_set_class_revid(pegmatite_pcie_controller_t *controller, u16 pciClass)
{
	pci_cfg_t *cfgRegs = (pci_cfg_t *)controller->pcie_lclcfg_base;
	u32 class;

	class = readl(&cfgRegs->ClassRevID);
	class |= (pciClass  << 16);
	writel(class, &cfgRegs->ClassRevID);
}

/*
 * Initialize the requested max L0 and L1 ASPM latency to max. The default
 * value of the requested latency for L1 is lower than the latency advertised
 * by this device, so a host will never enable ASPM unless we change it.
 */
static void pcie_endpoint_set_aspm_latency(pegmatite_pcie_controller_t *controller)
{
	pci_cfg_t *cfgRegs = (pci_cfg_t *)controller->pcie_lclcfg_base;
	u32 devcap = readl(&cfgRegs->LDevCap);

	devcap |= PCI_EXP_DEVCAP_L0S | PCI_EXP_DEVCAP_L1;
	writel(devcap, &cfgRegs->LDevCap);
}

void pegmatite_endpoint_enable(pegmatite_pcie_controller_t *controller)
{
	u32 i;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	pci_cfg_t *cfgRegs = (pci_cfg_t *)controller->pcie_lclcfg_base;

	PCIE_PRINTK(DBG_INIT, ">>> %s %d  - ENDPOINT  SETUP \n", __func__,__LINE__);

	/*
         * Assume we only have one controller configured as an endpoint that
         * will want to generate MSI interrupts.  Store a pointer to this controller
         * so pcie_gen_msi_irq can use it to poke the right registers to generate
         * the MSI.  Ideally the caller of pcie_gen_msi_irq knows what controller
         * it wants to generate the MSI on and passes that information in the call.
         * Today we are not that ideal.
	 */
	msi_controller = controller;

	/*
	 * fix up the class register as an endpoint
	 */
	pcie_set_class_revid(controller, PCI_CLASS_SYSTEM_OTHER);

	pcie_endpoint_set_aspm_latency(controller);

	/*
	 * fix up BAR0 mask register (it's write only) to set size
	 */
	writel(0x8000000 - 1, &cfgRegs->LCfgBAR0Mask);
	writel(0xffffffff, &cfgRegs->LCfgBAR1Mask);

	for (i=0; i<PCIE_INBOUND_APPERATURES; i++)
		writel(0xd0000000 + (i*PCIE_APPERATURE_SIZE), &appRegs->InAddrXlt[i]);

	/*
	 * We need inbound address translation register to be 0xd0000000
	 */
	writel(0xd0000000, &appRegs->InAddrXlt[1]);

	writel(0xe0000000, &appRegs->BAR0);

	/* Set bits 19:17 to 111 to enable up to 128 MSI interrupts */
	writel((0x000e0000 | readl(&cfgRegs->LMsiCtrl)), &cfgRegs->LMsiCtrl);

	/* setup the outbound address translate regs for APB accesses */
	for (i=0; i<PCIE_OUTBOUND_APPERATURES; i++)
		writel(0xd0000000 + (i*PCIE_APPERATURE_SIZE), &appRegs->OutAddrXlt[i]);

	/* set to x1 mode and the predetermined number of lane to 1 */
	writel((0x10000 | (readl(&cfgRegs->LPortLink) & ~0x3f0000)), &cfgRegs->LPortLink);
	writel((0x100 | (readl(&cfgRegs->LLWidthSpeed) & ~0x1ff00)), &cfgRegs->LLWidthSpeed);

	/*
	 * enable the LTSSM in the core to start Link training sequences
	 */
	writel((PCIE_APP_CTRL_REG_LTSSM_EN | PCIE_APP_CTRL_REG_RDYENTRL23), &appRegs->ACtrl);
}


static void pegmatite_root_complex_enable(pegmatite_pcie_controller_t *controller)
{
	u16 cmd;
	u32 outAddrXlat;
	u32 i=0, val;
	int rc;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	pci_cfg_t *cfgRegs = (pci_cfg_t *)controller->pcie_lclcfg_base;

	PCIE_PRINTK(DBG_INIT, ">>> %s %d  - ROOT COMPLEX SETUP \n", __func__,__LINE__);

	appRegs = (pci_app_regs_t *) controller->pcie_app_base;
	cfgRegs = (pci_cfg_t *)      controller->pcie_lclcfg_base;


	/*
	 * Master + slave enable.
	 */
	cmd = readw(&cfgRegs->Cmd);
	cmd &= ~PCI_COMMAND_SERR;
	cmd |= PCI_COMMAND_MEMORY;
	cmd |= PCI_COMMAND_MASTER;

    /* Enable IO space support in command register */
    if (controller->pcie_io_space > 0)
        cmd |= PCI_COMMAND_IO;
	writew(cmd, &cfgRegs->Cmd);

	pcie_enable_msi_int(controller);
    pcie_enable_intx_int(controller);

	/*
	 * fix up the class register as a root complex
	 */
	pcie_set_class_revid(controller, PCI_CLASS_BRIDGE_PCI);

	/* disable BAR0 for RC */
	writel(0xffffffff, &cfgRegs->LCfgBAR0Mask);
	/* disable BAR1 for RC */
	writel(0xffffffff, &cfgRegs->LCfgBAR1Mask);

	outAddrXlat = controller->pcie_mem_space;
	for (i=0; i<PCIE_OUTBOUND_APPERATURES; i++) {
		writel(outAddrXlat, &appRegs->OutAddrXlt[i]);
		outAddrXlat += controller->pcie_apperature_size;
	}

    /* NOTE: Memory region 7 in outbound direction is going to be used for IO space
     * transactions provided bit[0] of OutAddrXlt7 is set to 1 (just like an IO BAR)
     */
    if (controller->pcie_io_space > 0) {
        outAddrXlat = controller->pcie_mem_space;
        i = PCIE_OUTBOUND_APPERATURES - 1;
        outAddrXlat += i * controller->pcie_apperature_size;
        outAddrXlat |= 0x1;
        writel(outAddrXlat, &appRegs->OutAddrXlt[i]);
    }

	/* fixup the base and limit regs to allow EP's to access DRAM and regs
	 * DRAM -- 0x00000000-0x7fffffff
	 * Regs -- 0xd0000000-0xffffffff
	 *
	 * The Base / Limit pair of registers set a starting and ending address where the accesses from an endpoint
	 * are prohibited.
	 */
	writew(0x8000, &cfgRegs->MemBase);
	writew(0xcfff, &cfgRegs->MemLimit);

	writew(0x8000, &cfgRegs->PrefMemBase);
	writew(0xcfff, &cfgRegs->PrefMemLimit);

	/* enable some error reporting */
	val = readl(&cfgRegs->LDevContStatus);
	writel(val | 0xf, &cfgRegs->LDevContStatus);

#ifdef ENABLE_ERROR_IRQS
	val = readl(&cfgRegs->LRootErrCmd);
	writel(val | 0x7, &cfgRegs->LRootErrCmd);

	val = readl(&appRegs->AIntEn);
	writel(PCIE_APP_INT_ERROR_MASK, &appRegs->AIntEn);
#endif

#ifdef ENABLE_PME_IRQS
	/* enable pme */
	val = readl(&appRegs->AIntEn);
	writel(PCIE_APP_INT_PME_MASK, &appRegs->AIntEn);
#endif

	/*
	 * enable the LTSSM in the core to start Link training sequences
	 */
	writel((PCIE_APP_CTRL_REG_LTSSM_EN), &appRegs->ACtrl);
	PCIE_PRINTK(DBG_LINK, "%s:%d, start PCIe Link Training...\n", __func__, __LINE__);

	/*
	 * wait about 5000 msec for link-up.
	 */
	for (i=0; i < 500; i++) {
	        if (pcie_link_up(controller)) {
	                break;
	        }
		msleep(10);
	}

	/*
	 * if a device is detected
	 * then delay for about 200 msec after link is up
	 */
	if (readl(&appRegs->AStat) & PCIE_APP_STATUS_REG_LINK_UP) {
		PCIE_PRINTK(DBG_INFO, "%s: PCIe Link Up.  link_width = %d\n", controller->name, (readl(&cfgRegs->LLinkContStatus) & PCIE_CAP_LINK_STATUS_NEG_WIDTH) >> 20);
		rc = 1;
	} else {
		PCIE_PRINTK(DBG_INFO, "%s: No PCIe devices detected\n", controller->name);
		rc = 0;
	}
}

static void pegmatite_pcie_setup_hw (struct pci_sys_data *sys, pegmatite_pcie_controller_t *controller)
{
	if (controller->endpoint) {
		pegmatite_endpoint_enable(controller);
	} else {
		pegmatite_root_complex_enable(controller);
	}
}



/**
 ****************************************************************************************************
 *
 * \brief    This is called for each controller from pcibios_init_hw() in arch/arm/kernel/bios32.c
 *
 * \param    nr:  logical controller number.
 *           sys: ptr. to pci sys data that was allocated at the top level.
 *
 * \returns  0 = Failure
 *           1 = Success
 *
 *****************************************************************************************************/
static int pegmatite_pcie_setup(int nr, struct pci_sys_data *sys)
{
	pegmatite_pcie_controller_t *controller = sys_to_controller(sys);

	/*
	 * Add this memory resource to our sys structure.
	 */
	pci_add_resource_offset(&sys->resources, &controller->mem, sys->mem_offset);


    /*
	 * Add IO space resource
	 */
    pci_add_resource_offset(&sys->resources, &controller->io, sys->io_offset);


	/*
	 * Enable interrupts for the controller.
	 */
	pcie_irq_init(controller);

	pegmatite_pcie_setup_hw(sys, controller);

	pci_add_resource(&sys->resources, &pcie_top->busn);

	/*
	 * Return success.
	 */
	return(1);
}

/**
 ****************************************************************************************************
 *
 * \brief   called once for each controller to scan its bus from bios32.c
 *
 * \param   nr:  logical controller number (0 and 1 in our usage)
 *          sys: ptr. to the pci_sys_data, whose private_data is a pointer to controller structure.
 *
 * \returns
 *
 *****************************************************************************************************/
static struct pci_bus *pegmatite_pcie_scan_bus(int nr, struct pci_sys_data *sys)
{
	int max;
	struct pci_bus *bus;

	pegmatite_pcie_controller_t *controller = sys_to_controller(sys);


	bus = pci_create_root_bus(&pcie_top->pdev->dev, sys->busnr,
				  &pegmatite_pcie_ops, sys, &sys->resources);

	if (!bus)
		return NULL;

	/*
	 * If this controller is an endpoint, then we create the root bus, but no further scan.
	 * If we return a NULL from this routine, the calling code will panic.
	 */
	if (controller->endpoint) {
		PCIE_PRINTK(DBG_INIT,">>> %s %d No scan for endpoint\n", __func__,__LINE__);
		return bus;
	}

	/*
	 * Save this pointer as the root bus
	 */
	controller->rootBus = bus;

	max = pci_scan_child_bus(bus);

	pci_bus_update_busn_res_end(bus, max);

	return(bus);
}

static int pcie_error_irq_handler(pegmatite_pcie_controller_t *controller, int bug_on_fatal, struct seq_file *f)
{
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	pci_cfg_t *lclCfg         = (pci_cfg_t *)(controller->pcie_lclcfg_base);
	u32 int_enb = 0, int_pend=0, dev=0, dev_stat=0, fatal=0, error=0;

	if (readl(&appRegs->AIntP) & PCIE_APP_INT_ERROR_MASK)
	{
		int_enb = readl(&appRegs->AIntEn);
		writel(0, &appRegs->AIntEn);
		int_pend = readl(&appRegs->AIntP) & PCIE_APP_INT_ERROR_MASK;
		if (int_pend)
			error = 1;
		seq_printf(f, "PCIe Error Interrupt: ");
		seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&appRegs->AIntP, int_pend);
		if (int_pend & (1 << 0))
			seq_printf(f, "\tTraining Reset from Device\n");
		if (int_pend & (1 << 3))
			seq_printf(f, "\tTarget Lookup Empty\n");
		if (int_pend & (1 << 4))
			seq_printf(f, "\tTarget Completion Abort\n");
		if (int_pend & (1 << 8))
			seq_printf(f, "\tRADM Completion Abort\n");
		if (int_pend & (1 << 9))
			seq_printf(f, "\tRADM Completion UR\n");
		if (int_pend & (1 << 11))
			seq_printf(f, "\tRADM Completion Timeout\n");
		if (int_pend & (1 << 13))
			seq_printf(f, "\tRADM Correctable Error\n");
		if (int_pend & (1 << 14))
			seq_printf(f, "\tRADM Non-Fatal Error\n");
		if (int_pend & (1 << 15))
		{
			seq_printf(f, "\tRADM Fatal Error\n");
			fatal=1;
		}
		if (int_pend & (1 << 22))
			seq_printf(f, "\tAER Error MSI -- %x\n", (readl(&appRegs->AIntP) & (0x1f << 16)) >> 16 );
		if (int_pend & (1 << 23))
			seq_printf(f, "\tAER IRQ -- %x\n", (readl(&appRegs->AIntP) & (0x1f << 16)) >> 16 );
		if (int_pend & (1 << 24))
		{
			seq_printf(f, "\tLink Request Reset\n");
			fatal=1;
		}
		if (int_pend & (1 << 28))
			seq_printf(f, "\tInvalid Inbound TLP Type\n");
		if (int_pend & (1 << 29))
		{
			seq_printf(f, "\tSystem Error\n");
			fatal=1;
		}
		if (int_pend & (1 << 30))
		{
			seq_printf(f, "\tRADMx Tag Lookup Error\n");
			fatal=1;
		}
		if (int_pend & (1 << 31))
		{
			seq_printf(f, "\tGM Tag Lookup Error\n");
			fatal=1;
		}

		if (readl(&lclCfg->LDevContStatus) & 0xffff0000)
		{
			seq_printf(f, "Device Status: ");
			dev = readl(&lclCfg->LDevContStatus);
			dev_stat = (dev >> 16) & 0xffff;
			if (dev_stat)
				error = 1;
				seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LDevContStatus , dev_stat);
			if (dev_stat & (1 << 0))
				seq_printf(f, "\tCorrectable Error\n");
			if (dev_stat & (1 << 1))
				seq_printf(f, "\tNon-Fatal Error\n");
			if (dev_stat & (1 << 2))
			{
				seq_printf(f, "\tFatal Error\n");
				fatal=1;
			}
			if (dev_stat & (1 << 3))
				seq_printf(f, "\tUnsupported Request\n");
			writel(dev | (dev_stat << 16), &lclCfg->LDevContStatus);
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LDevContStatus , readl(&lclCfg->LDevContStatus));
		}

		if (readl(&lclCfg->LCorrStat))
		{
			seq_printf(f, "AER Correctable Status: ");
			dev_stat = readl(&lclCfg->LCorrStat);
			if (dev_stat) error = 1;
				seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LCorrStat, dev_stat);
			if (dev_stat & (1 << 0))
				seq_printf(f, "\tReceiver Error\n");
			if (dev_stat & (1 << 6))
				seq_printf(f, "\tBad TLP\n");
			if (dev_stat & (1 << 7))
				seq_printf(f, "\tBad DLLP\n");
			if (dev_stat & (1 << 8))
				seq_printf(f, "\tREPLAY_NUM Rollover\n");
			if (dev_stat & (1 << 12))
				seq_printf(f, "\tReplay Timer Timeout\n");
			writel(dev_stat, &lclCfg->LCorrStat);
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LCorrStat, readl(&lclCfg->LCorrStat));
		}

		if (readl(&lclCfg->LUCorrStat))
		{
			seq_printf(f, "AER Uncorrectable Status: ");
			dev_stat = readl(&lclCfg->LUCorrStat);
			if (dev_stat)
				error = 1;
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LUCorrStat, dev_stat);
			if (dev_stat & (1 << 0))
				seq_printf(f, "\tTraining Error\n");
			if (dev_stat & (1 << 4))
				seq_printf(f, "\tData Link Protocol Error\n");
			if (dev_stat & (1 << 12))
				seq_printf(f, "\tPoisoned TLP\n");
			if (dev_stat & (1 << 13))
				seq_printf(f, "\tFlow Control Protocol Error\n");
			if (dev_stat & (1 << 14))
				seq_printf(f, "\tCompletion Timeout\n");
			if (dev_stat & (1 << 15))
				seq_printf(f, "\tCompleter Abort\n");
			if (dev_stat & (1 << 16))
				seq_printf(f, "\tUnexpected Completion\n");
			if (dev_stat & (1 << 17))
				seq_printf(f, "\tReceiver Overflow\n");
			if (dev_stat & (1 << 18))
				seq_printf(f, "\tMalformed TLP\n");
			if (dev_stat & (1 << 19))
				seq_printf(f, "\tECRC Error\n");
			if (dev_stat & (1 << 20))
				seq_printf(f, "\tUnsupported Request Error\n");
			fatal = 1;
			writel(dev_stat, &lclCfg->LUCorrStat);
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LUCorrStat, readl(&lclCfg->LUCorrStat));
		}

		if (readl(&lclCfg->LRootErrStat))
		{
			seq_printf(f, "AER Root Error Status: ");
			dev_stat = readl(&lclCfg->LRootErrStat);
			if (dev_stat)
				error = 1;
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LRootErrStat, dev_stat);
			if (dev_stat & (1 << 0))
				seq_printf(f, "\tCorrectable Error\n");
			if (dev_stat & (1 << 1))
				seq_printf(f, "\tMultiple Correctable Error\n");
			if (dev_stat & (1 << 2))
			{
				seq_printf(f, "\tUncorrectable Error Message\n");
				fatal = 1;
			}
			if (dev_stat & (1 << 3))
			{
				seq_printf(f, "\tMultiple Uncorrectable Error Message\n");
				fatal = 1;
			}
			if (dev_stat & (1 << 4))
			{
				seq_printf(f, "\tFirst Uncorrectable Fatal\n");
				fatal = 1;
			}
			if (dev_stat & (1 << 5))
			{
				seq_printf(f, "\tNon-Fatal Error Messages\n");
				fatal = 1;
			}
			if (dev_stat & (1 << 6))
			{
				seq_printf(f, "\tFatal Error Messages\n");
				fatal = 1;
			}
			seq_printf(f, "\tMessage Number = 0x%x\n", (dev_stat >> 27) & 0xf);
			writel(dev_stat, &lclCfg->LRootErrStat);
			seq_printf(f, "(0x%08x) = 0x%08x\n", (u32)&lclCfg->LRootErrStat, readl(&lclCfg->LRootErrStat));
		}

		seq_printf(f, "Command/Status: ");
		seq_printf(f, "(0x%08x) = 0x%04x\n", (u32)&lclCfg->Stat, readw(&lclCfg->Stat));

		writel(int_pend, &appRegs->AIntAck);
		seq_printf(f, "int_pend(0x%08x) = 0x%08x\n", (u32)&appRegs->AIntP, readl(&appRegs->AIntP));
		writel(int_enb, &appRegs->AIntEn);
	}
	if (fatal && bug_on_fatal) BUG();

	return(error);
}

static int pcie_dump_regs(void __iomem *base, int offset, int max, struct seq_file *f)
{
    u32 i;

    for (i=0; i<max; i+=16) {
        seq_printf(f, "%08x : %08x %08x %08x %08x\n",
                (int)(base + offset + i),
                readl(base + offset + i),
                readl(base+offset+i+4),
                readl(base+offset+i+8),
                readl(base+offset+i+12));
    }
    seq_printf(f, "\n\n");

    return 0;
}

static int pcie_dump_trg_config(struct seq_file *f, void *offset) {
	pegmatite_pcie_controller_t *controller = (pegmatite_pcie_controller_t *)f->private;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;

	seq_printf(f, "\n\t\tPCIe Target Cfg Regs\n");
	if (pcie_link_up(controller))
	{
		pcie_dump_regs(controller->pcie_trgcfg_base, 0, 0xB0, f);
	}
	else
		seq_printf(f, "\t\t\t...Link Down\n");

	if (readl(&appRegs->AIntP) & (1 << 9))
	{
		writel((1 << 9), &appRegs->AIntAck);
		seq_printf(f, "UR Detected, data is invalid!\n\n");
	}

	return 0;
}

static int pcie_trg_bus_config(struct file *file, const char *buffer, size_t count, loff_t *offset) {
	int rc, bus, dev, func;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0))
	pegmatite_pcie_controller_t *controller = PDE(file->f_path.dentry->d_inode)->data;
#else
	pegmatite_pcie_controller_t *controller = PDE_DATA(file_inode(file));
#endif
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;

	rc = sscanf(buffer, "%i %i %i", &bus, &dev, &func);

	if (bus == 0)
	{
		printk("\n\nNot changing CfgControl reg due to invalid bus number (0)\n");
		printk("Use pcielclcfg to dump bus 0 cfg regs or try another bus number\n\n");
		return(count);
	}
	else
		printk("\n\nSetting PCIe config cycles to read bus:%d dev:%d func:%d\n\n", bus, dev, func);

// fix up cfg control reg to issue correct cfg type
	if (bus <= 1)
		writel((1 << PCIE_CFG_CTRL_BUS_NUM_SH), &appRegs->CfgControl);
	else
		writel(PCIE_CFG_CTRL_TYPE_1 |
			((bus & 0xff) << PCIE_CFG_CTRL_BUS_NUM_SH) |
			((dev & 0x1f) << PCIE_CFG_CTRL_DEV_NUM_SH) |
			((func & 0x7) << PCIE_CFG_CTRL_FUNC_NUM_SH),
			&appRegs->CfgControl);

	return(count);
}

static int pcie_trgcfg_proc_open(struct inode *inode, struct file *file) {
    void *ctx;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0))
    ctx = PDE(inode)->data;
#else
    ctx = PDE_DATA(inode);
#endif

    return single_open(file, pcie_dump_trg_config, ctx);
}

static const struct file_operations pcie_trgcfg_proc_fops = {
    .owner   = THIS_MODULE,
    .open    = pcie_trgcfg_proc_open,
    .read    = seq_read,
    .write   = pcie_trg_bus_config,
    .llseek  = seq_lseek,
    .release = single_release,
};

static int pcie_dump_app_regs(struct seq_file *f, void *offset) {
	pegmatite_pcie_controller_t *controller = (pegmatite_pcie_controller_t *)f->private;

	seq_printf(f, "\n\t\tPCIe Application Regs\n");
	pcie_dump_regs(controller->pcie_app_base, 0, 0x100, f);
	pcie_dump_regs(controller->pcie_app_base, 0x1000, 0x40, f);

    return 0;
}

static int pcie_appreg_proc_open(struct inode *inode, struct file *file) {
    void *ctx;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0))
    ctx = PDE(inode)->data;
#else
    ctx = PDE_DATA(inode);
#endif

    return single_open(file, pcie_dump_app_regs, ctx);
}

static const struct file_operations pcie_appreg_proc_fops = {
    .owner   = THIS_MODULE,
    .open    = pcie_appreg_proc_open,
    .read    = seq_read,
    .llseek  = seq_lseek,
    .release = single_release,
};

static int pcie_dump_lcl_config(struct seq_file *f, void *offset) {
	pegmatite_pcie_controller_t *controller = (pegmatite_pcie_controller_t *)f->private;

	pcie_dump_regs(controller->pcie_lclcfg_base, 0, 0x140, f);

	return 0;
}

static int pcie_lclcfg_proc_open(struct inode *inode, struct file *file) {
    void *ctx;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0))
    ctx = PDE(inode)->data;
#else
    ctx = PDE_DATA(inode);
#endif

    return single_open(file, pcie_dump_lcl_config, ctx);
}

static const struct file_operations pcie_lclcfg_proc_fops = {
    .owner   = THIS_MODULE,
    .open    = pcie_lclcfg_proc_open,
    .read    = seq_read,
    .llseek  = seq_lseek,
    .release = single_release,
};

static int pcie_proc_error (struct seq_file *f, void *offset) {
	pegmatite_pcie_controller_t *controller = (pegmatite_pcie_controller_t *)f->private;

	seq_printf(f, "\n\nChecking for PCIe Errors...\n");
	if (!pcie_error_irq_handler(controller, 0, f))
		seq_printf(f, "No Errors Detected\n");

    return 0;
}

static int pcie_err_proc_open(struct inode *inode, struct file *file) {
    void *ctx;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,10,0))
    ctx = PDE(inode)->data;
#else
    ctx = PDE_DATA(inode);
#endif

    return single_open(file, pcie_proc_error, ctx);
}

static const struct file_operations pcie_err_proc_fops = {
    .owner   = THIS_MODULE,
    .open    = pcie_err_proc_open,
    .read    = seq_read,
    .llseek  = seq_lseek,
    .release = single_release,
};

static void pci_init_proc( pegmatite_pcie_controller_t *controller )
{
	char
		buf[80];
	const char
		*name;

	name = controller->name;
	if (name == NULL) {
		name = "";
	}

	snprintf(buf, sizeof(buf), "pcie_err_%s", name);
	proc_create_data(buf,    S_IRUGO, NULL, &pcie_err_proc_fops, controller);

	snprintf(buf, sizeof(buf), "pcielclcfg_%s", name);
	proc_create_data(buf,  S_IRUGO, NULL, &pcie_lclcfg_proc_fops, controller);

	snprintf(buf, sizeof(buf), "pcieappregs_%s", name);
	proc_create_data(buf, S_IRUGO, NULL, &pcie_appreg_proc_fops, controller);

	snprintf(buf, sizeof(buf), "pcietrgcfg_%s", name);
	proc_create_data(buf,  0, NULL, &pcie_trgcfg_proc_fops, controller);
}

static void pci_remove_proc(pegmatite_pcie_controller_t *controller) {
	char buf[80];
	const char *name;

	name = controller->name;
	if (name == NULL) {
		name = "";
	}

	snprintf(buf, sizeof(buf), "pcie_err_%s", name);
	remove_proc_entry(buf, NULL);

	snprintf(buf, sizeof(buf), "pcielclcfg_%s", name);
	remove_proc_entry(buf, NULL);

	snprintf(buf, sizeof(buf), "pcieappregs_%s", name);
	remove_proc_entry(buf, NULL);

	snprintf(buf, sizeof(buf), "pcietrgcfg_%s", name);
	remove_proc_entry(buf, NULL);
}

/**
 ****************************************************************************************************
 *
 * \brief   Time to
 *
 * \param
 *
 * \returns
 *
 *****************************************************************************************************/
static void pegmatite_pcie_add_bus(struct pci_bus *bus)
{
	pegmatite_pcie_controller_t *controller = sys_to_controller(bus->sysdata);

	bus->msi = &controller->msi.msi_chip;
}

/**
 * pegmatite_pcie_map_irq - Set the handler for the INTx and mark IRQ as valid
 * @dev: PCI device handle
 * @slot: Slot number
 * @pin: Pin number
 *
 * Return: Returns Virtual IRQ number is INTx mapping is successful, else returns -1.
 */
static int pegmatite_pcie_map_irq(const struct pci_dev *dev, u8 slot, u8 pin)
{
    int virq = 0;

    if (!p_intx_irq_domain)
        return -1;

    /* For now, only operate on INTA interrupt */
    virq = irq_create_mapping(p_intx_irq_domain, 0);
    PCIE_PRINTK(DBG_INTX,"***********%s %d: virq %x\n", __func__, __LINE__, virq);
    if (virq > 0) {
        pic.irq_base = virq;
        return virq;
    } else
        return -1;
}

static void pegmatite_pcie_enable(pegmatite_pcie_top_t *pcie_top)
{
	struct hw_pci hw;

	memset(&hw, 0, sizeof(hw));

	hw.nr_controllers = pcie_top->nr_controllers;
	/*
         * hw.private_data needs to be an array of pointers to the unique controller instances.  Referenced in bios32.c
         * when the setup call occurs.
         */
        hw.private_data = (void **) pcie_top->controllers;

	PCIE_PRINTK(DBG_INIT, "\n\n>>> %s %d - calling common init with %d controllers and private_data: %p \n\n", __func__,__LINE__, hw.nr_controllers, hw.private_data);

	hw.setup = pegmatite_pcie_setup;
	hw.scan  = pegmatite_pcie_scan_bus;
	hw.add_bus  = pegmatite_pcie_add_bus;
    hw.map_irq = pegmatite_pcie_map_irq;

	pci_common_init_dev(&pcie_top->pdev->dev, &hw);

	/* this will enable any bridges -- with this allowing memory cycles and bus mastering */
	pci_assign_unassigned_resources();
}

static int pegmatite_pcie_init_controller(pegmatite_pcie_controller_t *controller, struct device_node *child)
{
	struct device *dev = &controller->pdev->dev; /* For logging */
	int ret;

	/*
	 * Get core base from the device tree.
	 */
	controller->pcie_trgcfg_base = of_iomap(child, 0);
	if (!controller->pcie_trgcfg_base) {
		dev_err(dev, "ioremap failed for pcie trgcfg base address\n");
		return -ENODEV;
	}

	/*
	 * Create lclcfg ptr. from the core base
	 */
	controller->pcie_lclcfg_base = (controller->pcie_trgcfg_base + PCIE_LCL_CFG_OFFSET);

	/*
	 * Get app base from the device tree.
	 */
	controller->pcie_app_base = of_iomap(child, 1);
	if (!controller->pcie_app_base) {
		dev_err(dev, "ioremap failed for pcie app base address\n");
		return -ENODEV;
	}

	/*
	 * Get and map the reset registers for this controller.
	 *
	 * TODO: is this obsolete now that the PHY driver releases reset?
	 */
	controller->pcie_reset_control = of_iomap(child, 2);

	/*
	 * Retrieve the 'reg' entry that is needed later in arch_setup_msi_irq() in msi.c
	 */
	{
		unsigned int physical_address;
		unsigned int const *physicalAddressP;

		physicalAddressP = of_get_address(child, 1, NULL, NULL);
		if (physicalAddressP == NULL) {
			dev_err(dev, "Address for Application Registers missing in device tree\n");
			return -EFAULT;
		}

		physical_address = of_translate_address(child, physicalAddressP);
		if (physical_address == OF_BAD_ADDR) {
			dev_err(dev, "Invalid Address for Application Registers in device tree\n");
			return -EFAULT;
		}
		controller->pcie_app_base_physical = physical_address;
	}

	controller->irq = irq_of_parse_and_map(child, 0);
	if (!controller->irq) {
		dev_err(dev, "failed to map pcie irq\n");
		return -ENODEV;
	}

	if (of_property_read_string(child, "controller-name", &controller->name)) {
		dev_err(dev, "INFO: no pcie controller name specified\n");
	}

	if (of_property_read_u32(child, "pci_mem_size", &controller->pcie_mem_size)) {
		dev_err(dev, "no pcie memory size declared\n");
		return -ENODEV;
	}

	if (of_property_read_u32(child, "pci_mem_space", &controller->pcie_mem_space)) {
		dev_err(dev, "no pcie memory space declared\n");
		return -ENODEV;
	}
	else {
		struct resource *res = &controller->mem;

		snprintf(controller->mem_name, sizeof(controller->mem_name), "PCI Mem Space %s", controller->name);
		res->name = controller->mem_name;

		res->flags = IORESOURCE_MEM;
		res->start = controller->pcie_mem_space;
		res->end = res->start + controller->pcie_mem_size - 1;

		ret = request_resource(&iomem_resource, res);
		if (ret) {
			PCIE_PRINTK(DBG_ERROR, "%s: Error allocating memory resource for controller", controller->name);
			return ret;
		}
	}

	/*
	 * TODO: read property for IO.  Will be unused in our case, so hardcode -1.
	 */
	controller->io_target = -1;
	controller->io_attr   = -1;

    /*
     * Read property for IO now.
     */

    if (of_property_read_u32(child, "pci_io_size", &controller->pcie_io_size)) {
		dev_err(dev, "no pcie IO space size declared\n");
		return -ENODEV;
	}

    if (of_property_read_u32(child, "pci_io_space", &controller->pcie_io_space)) {
		dev_err(dev, "no pcie IO space declared\n");
		return -ENODEV;
	}
	else {
		struct resource *res = &controller->io;

		snprintf(controller->io_name, sizeof(controller->io_name), "PCI IO Space %s", controller->name);
		res->name = controller->io_name;

        res->flags = IORESOURCE_IO;
		res->start = controller->pcie_io_space;
		res->end = res->start + controller->pcie_io_size - 1;

		ret = request_resource(&ioport_resource, res);
		if (ret) {
			PCIE_PRINTK(DBG_ERROR, "%s: Error allocating IO space resource for controller with ret %d \n", controller->name, ret);
			return ret;
		}
	}

	if (of_property_read_u32(child, "pci_apperature_size", &controller->pcie_apperature_size)) {
		dev_err(dev, "no pcie apperature size declared\n");
		return -ENODEV;
	}

	/* Due to design flaw, we need both the DM2 and DM4 clocks enabled to use DM4. */
	controller->dm2clk = of_clk_get_by_name(child, "dm2");
	if (IS_ERR(controller->dm2clk)) {
		dev_err(dev, "no pcie clock 0 declared\n");
		ret = PTR_ERR(controller->dm2clk);
		goto err_dm2;
	}
	ret = clk_prepare_enable(controller->dm2clk);
	if (ret)
		goto err_dm2;

	controller->dm4clk = of_clk_get_by_name(child, "dm4");
	if (IS_ERR(controller->dm4clk)) {
		dev_err(dev, "no pcie clock 1 declared\n");
		ret = PTR_ERR(controller->dm4clk);
		goto err_dm4;
	}
	ret = clk_prepare_enable(controller->dm4clk);
	if (ret)
		goto err_dm4;

	controller->ioclk = of_clk_get_by_name(child, "io");
	if (IS_ERR(controller->ioclk)) {
		dev_err(dev, "no pcie io clock declared\n");
		ret = PTR_ERR(controller->ioclk);
		goto err_ioclk;
	}
	ret = clk_prepare_enable(controller->ioclk);
	if (ret)
		goto err_ioclk;

	/*
	 * If this is an fpga system we need read the endpoint setting from the
	 * device tree, and write it into the fpgad pcie config register to configure
	 * the block.
	 */
	if (is_fpga()) {
		void __iomem *fpgad_pcie_config = ioremap(FPGAD_PCIE_CONFIG, 4);

		if (of_property_read_u32(child, "endpoint", &controller->endpoint)) {
			controller->endpoint = 0;
		}

		if(fpgad_pcie_config) {
			writel(controller->endpoint, fpgad_pcie_config);
			iounmap(fpgad_pcie_config);
		}
	} else {
		/*
		 * If this is real silicon then read the device capabilities from local config register.
		 */
		pci_cfg_t *cfgRegs = (pci_cfg_t *) controller->pcie_lclcfg_base;
		unsigned int deviceType;

		/*
		 * Device type is bits 23:20 of the LPCIECap
		 */
		deviceType = (cfgRegs->LPCIECap >> 20) & 0xf;

		/*
		 * Device types:
		 *
		 * 0000b PCI Express Endpoint
		 * 0001b Legacy PCI Express Endpoint
		 * 0100b Root Port of PCI Express Root Complex*  <====
		 * 0101b Upstream Port of PCI Express Switch*
		 * 0110b Downstream Port of PCI Express Switch*
		 * 0111b PCI Express to PCI/PCI-X Bridge*
		 * 1000b PCI/PCI-X to PCI Express Bridge*
		 * 1001b Root Complex Integrated Endpoint
		 * 1010b Root Complex Event Collector
		 *
		 * *This value is only valid for Functions that implement a Type 01h PCI Configuration Space header.
		 *
		 * All other encodings are Reserved.
		 *
		 *---
		 *
		 * We should only read deviceType of 0 or 4 in our usage.
		 */
		controller->endpoint = (deviceType == 0) ? 1 : 0;
	}

	/*
	 * Init lock before calling root complex enable since that will try to
	 * read config words, and use this lock.
	 */
	controller->pcie_lock = __SPIN_LOCK_UNLOCKED(controller->pcie_lock);

	/*
	 * Initialize the software emulated PCI-to-PCI bridge.
	 */
	pegmatite_sw_pci_bridge_init(controller);

	pci_init_proc(controller);

	return ret;

err_ioclk:
	if (controller->ioclk)
		clk_put(controller->ioclk);
	clk_disable_unprepare(controller->dm4clk);
err_dm4:
	if (controller->dm4clk)
		clk_put(controller->dm4clk);
	clk_disable_unprepare(controller->dm2clk);
err_dm2:
	if (controller->dm2clk)
		clk_put(controller->dm2clk);
	return ret;
}


 /* Root Complex Power Management support - BEGIN */

static bool pegmatite_pci_power_manageable(struct pci_dev *dev)
{
	unsigned int isRootBus;
	pegmatite_pcie_controller_t *controller;
	controller = pegmatite_pcie_find_controller(dev->bus, dev->devfn, &isRootBus);
	if (controller->endpoint)
		return false;
	return true;
}

static void pegmatite_LTSSM_wait(struct pci_dev *dev,
				 pegmatite_pcie_controller_t *controller,
				 unsigned int ltssm_state)
{
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	unsigned int reg = readl(&appRegs->AStat);
	unsigned int cnt=0;

	while ((cnt++ < 500) && ((reg & PCIE_APP_STATUS_REG_LINK_STATUS_MASK) != ltssm_state)) {
		msleep(10);
		reg = readl(&appRegs->AStat);
	}

	if ((reg & PCIE_APP_STATUS_REG_LINK_STATUS_MASK) != ltssm_state) {
		dev_err(&dev->dev, "%s %d : link did not enter LTSSM state %d (APP_STATUS_REG 0x%08x)\n", __func__, __LINE__, ltssm_state, reg);
		BUG();
	} else {
		dev_dbg(&dev->dev, "%s %d : link entered LTSSM state %d\n", __func__, __LINE__, ltssm_state);
	}
}

static bool pegmatite_link_up(pegmatite_pcie_controller_t *controller)
{
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	unsigned int reg = readl(&appRegs->AStat);

	return ((reg & PCIE_APP_STATUS_REG_LINK_STATUS_MASK_2) == PCIE_APP_STATUS_REG_LINK_UP_2);
}

static void pegmatite_pcie_wait_for_link(struct pci_dev *dev, pegmatite_pcie_controller_t *controller)
{
	int cnt = 0;

	msleep(10);
	while (!pegmatite_link_up(controller) && (cnt++ < 500)) msleep(10);

	if (!pegmatite_link_up(controller)) {
		dev_err(&dev->dev, "link is down, delayed %dmS !!!\n", cnt*10);
		BUG();
	} else {
		dev_dbg(&dev->dev, "link is up, delayed %dmS !!!\n", cnt*10);
	}
 }

/* Check if silicon version is >= B0 */
static bool pegmatite_pci_rev_b0(pci_app_regs_t *appRegs)
{
	u32 reg = readl_relaxed(&appRegs->REV0);

	return ((reg & PCIE_APP_REV0_REG_MAJ_MASK) || ((reg & PCIE_APP_REV0_REG_MIN_MASK) > 5));
}

static int pegmatite_pci_set_power_state(struct pci_dev *dev, pci_power_t state)
{
	unsigned int isRootBus, reg;
	pegmatite_pcie_controller_t *controller;
	pci_app_regs_t *appRegs;

	controller = pegmatite_pcie_find_controller(dev->bus, dev->devfn, &isRootBus);

	if (!controller) {
		dev_err(&dev->dev, "%s %d - can't find controller\n", __func__,__LINE__);
		return -ENODEV;
	}

	appRegs = (pci_app_regs_t *)controller->pcie_app_base;

	if (state == PCI_D3cold) {
		/*
		 * Send request to the control registers to send "turn off"
		 * message, which should put the device into D3 cold state.
		 */
		if (pegmatite_pci_rev_b0(appRegs)) {
			/* send PME_TURN_OFF (Rev B0+) */
			reg = readl(&appRegs->ACtrl);
			reg |= PCIE_APP_CTRL_REG_PME_TURN_OFF;
			writel(reg, &appRegs->ACtrl);
			pegmatite_LTSSM_wait(dev, controller, S_L2_IDLE);
		} else {
			pegmatite_LTSSM_wait(dev, controller, S_L1_IDLE);
		}

		/* disable the controller */
		msleep(10);  // TODO: remove this?

		phy_exit(controller->phy);
	} else if (dev->current_state == PCI_D3cold) {
		BUG_ON(state != PCI_D0);

		phy_init(controller->phy);

		/* Clear the request to send "turn off" message (Rev B0+) */
		if (pegmatite_pci_rev_b0(appRegs)) {
			reg = readl(&appRegs->ACtrl);
			reg &= ~PCIE_APP_CTRL_REG_PME_TURN_OFF;
			writel(reg, &appRegs->ACtrl);
		}

		pegmatite_root_complex_enable(controller);
		pegmatite_pcie_wait_for_link(dev, controller);
	}

	return 0;
}

static pci_power_t pegmatite_pci_choose_state(struct pci_dev *pdev)
{
	int d_max;

	if (pdev->no_d3cold)
		d_max = PCI_D3hot;
	else
		d_max = PCI_D3cold;
	return d_max;
}

static int pegmatite_pci_sleep_wake(struct pci_dev *dev, bool enable)
{
	return 0;
}

static int pegmatite_pci_run_wake(struct pci_dev *dev, bool enable)
{
	return 0;
}

static struct pci_platform_pm_ops pegmatite_pci_platform_pm = {
	.is_manageable = pegmatite_pci_power_manageable,
	.set_state     = pegmatite_pci_set_power_state,
	.choose_state  = pegmatite_pci_choose_state,
	.sleep_wake    = pegmatite_pci_sleep_wake,
	.run_wake      = pegmatite_pci_run_wake,
};

/* Root Complex Power Management support - END */


/**
 ****************************************************************************************************
 *
 * \brief   This is called one time at power up.  It should find all active controllers, and get
 *          them all initialized.
 *
 * \param
 *
 * \returns
 *
 *****************************************************************************************************/
static int pegmatite_pcie_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	int i = 0, ret;
	struct phy *phy;
	struct device *dev = &pdev->dev;
	struct device_node *child;

	PCIE_PRINTK(DBG_INIT, "\n\n\n\n\n>>> PCIE_PROBE +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");

	/*
	 * get our PCIE phy's initialized.
	 */
	phy = devm_phy_get(dev, "pcie-phy");
	if (IS_ERR(phy)) {
		dev_err(dev, "couldn't get pcie-phy.  Deferring...\n");
		return -EPROBE_DEFER;
	}

	/*
	 * We've found the PHY, so call its init() callback.
	 */
	ret = phy_init(phy);
	if (ret) {
		PCIE_PRINTK(DBG_ERROR, "%s %d - Error init'ing phy: %x \n", __func__,__LINE__, (unsigned int) phy);
		return -ENODEV;
	}

	/*
	 * allocate a top-level structure
	 */
	pcie_top = devm_kzalloc(&pdev->dev, sizeof(pegmatite_pcie_top_t), GFP_KERNEL);
	if (!pcie_top) {
		dev_err(&pdev->dev, "%s: pcie_top memory alloc failed!\n", __func__);
		return  -ENOMEM;
	}

	pcie_top->pdev = pdev;
	platform_set_drvdata(pdev, pcie_top);

#if 0
	if (resource_size(&pcie_top->io) != 0) {
        pcie_top->realio.flags = pcie_top->io.flags;
		pcie_top->realio.start = PCIBIOS_MIN_IO;
		pcie_top->realio.end = min_t(resource_size_t,
					 IO_SPACE_LIMIT,
					 resource_size(&pcie_top->io));
	} else {
		pcie_top->realio = pcie_top->io;
	}
#endif

	/* Get the bus range */
	ret = of_pci_parse_bus_range(np, &pcie_top->busn);
	if (ret) {
		dev_err(&pdev->dev, "failed to parse bus-range property: %d\n",
			ret);
		return ret;
	}

	/*
	 * Determine number of active controllers.  Should be 0-2 in our usage.
	 */
	for_each_child_of_node(pdev->dev.of_node, child) {
		if (!of_device_is_available(child))
			continue;
		pcie_top->nr_controllers++;
	}

	/*
	 * If no controllers, then bail
	 */
	if (pcie_top->nr_controllers == 0) {
		dev_err(&pdev->dev, "info: no active PCIE devices");
		return -ENODEV;
	}

	/*
	 * Allocate the array that will hold pointers to the controller structures.
	 */
	pcie_top->controllers = devm_kzalloc(&pcie_top->pdev->dev,
			sizeof(pegmatite_pcie_controller_t *) * pcie_top->nr_controllers,
			GFP_KERNEL);
	if (!pcie_top->controllers) {
		dev_err(&pdev->dev, "%s: error allocating array for %d controllers", __func__, pcie_top->nr_controllers);
		return -ENOMEM;
	}

	/*
	 * initialize each active controller
	 */

	for_each_child_of_node(pdev->dev.of_node, child) {
		if (!of_device_is_available(child))
			continue;

		/*
		 * Allocate a controller structure and assign to array and local variable.
		 */
		pcie_top->controllers[i] = devm_kzalloc(&pdev->dev,
								     sizeof(pegmatite_pcie_controller_t),
								     GFP_KERNEL);
		if (!pcie_top->controllers[i]) {
			dev_err(&pdev->dev, "%s: error allocating memory for controller[%d]", __func__, i);
			return -ENOMEM;
		}

		pcie_top->controllers[i]->phy  = phy;
		pcie_top->controllers[i]->pdev = pdev;
		pcie_top->controllers[i]->nr   = i;

		PCIE_PRINTK(DBG_INIT, ">>> %s %d - controller[%d]\n", __func__,__LINE__, i);

		ret = pegmatite_pcie_init_controller(pcie_top->controllers[i], child);
		if (ret)
			return ret;
		i++;
	}

	/*
	 * Set up power management operations structure
	 */
	pci_set_platform_pm(&pegmatite_pci_platform_pm);

	/*
	 * Enable and scan buses
	 */
	pegmatite_pcie_enable(pcie_top);

	PCIE_PRINTK(DBG_INIT, ">>> %s %d - PROBE complete  +++++++++++++++++++++++++++++++++++++++++++++++++++\n", __func__,__LINE__);

	return 0;
}

static int pegmatite_pcie_remove(struct platform_device *pdev) {
	int i;
	pegmatite_pcie_top_t *pcie = platform_get_drvdata(pdev);
	pegmatite_pcie_controller_t *controller;
	struct pci_bus *bus_itr;

    pcie_free_intx_irq_domain();

	bus_itr = pci_find_next_bus(NULL);
	while (bus_itr) {
		pci_stop_root_bus(bus_itr);
		pci_remove_root_bus(bus_itr);
		bus_itr = pci_find_next_bus(NULL);
	}

	for (i=0; i<pcie->nr_controllers; ++i) {
		controller = pcie->controllers[i];

		pci_remove_proc(controller);

		release_resource(&controller->mem);
		irq_dispose_mapping(controller->irq);
		iounmap(controller->pcie_reset_control);
		iounmap(controller->pcie_app_base);
		iounmap(controller->pcie_trgcfg_base);
	}

	return 0;
}

static const struct of_device_id of_pcie_table[] = {
	{.compatible = "marvell,pegmatite-pcie"},
	{},
};
MODULE_DEVICE_TABLE(of, of_pcie_table);

static struct platform_driver pegmatite_pcie_driver = {
	.probe  = pegmatite_pcie_probe,
	.remove = pegmatite_pcie_remove,
	.driver = {
		.owner  = THIS_MODULE,
		.name   = "pegmatite-pcie",
		.of_match_table = of_match_ptr(of_pcie_table),
	},
};

module_platform_driver(pegmatite_pcie_driver);
MODULE_DESCRIPTION("Driver for Pegmatite PCIe Controller");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:" DRIVER_NAME);
