/*
 * arch/arm/mach-pegmatite/msi.c
 *
 * Based on arch/arm/mach-mv61x0/msi.c
 *
 * PCI MSI support for Pegmatite
 *
 */

#include <linux/platform_device.h>
#include <linux/pci.h>
#include <linux/msi.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include "pci-pegmatite.h"

#define NUM_MSI_IRQS (128)

static DECLARE_BITMAP(msi_irq_in_use, NUM_MSI_IRQS);

int pegmatite_msi_irq_handler (pegmatite_pcie_controller_t *controller)
{
	uint32_t ind;
	uint32_t bitNumber, msi_stat;
	pci_app_regs_t *appRegs   = (pci_app_regs_t *)controller->pcie_app_base;
	struct pegmatite_msi_controller *msi = &controller->msi;

	for (ind=0; ind<4; ind++) {
		msi_stat = readl(&appRegs->RC_MSI_IRQ[ind].Stat);

		/*
		 * We read a bit-mask of 32 interrupt status bits.  Loop here handling each of them.
		 */
		for (bitNumber = 0; msi_stat && (bitNumber < 32); bitNumber++) {
			if (msi_stat & 1) {
				int irq = irq_find_mapping(msi->irq_domain,
							   ind*32 + bitNumber);

				/* Clear the interrupt here so we can use handle_simple_irq */
				writel((1 << bitNumber), &appRegs->RC_MSI_IRQ[ind].Clr);
				generic_handle_irq(irq);
			}
			msi_stat >>= 1;
		}
	}

	return(IRQ_HANDLED);
}

/*
 * Dynamic irq allocate and deallocation
 */
static int pegmatite_alloc_irq(void)
{
	int pos;

	do {
		pos = find_first_zero_bit(msi_irq_in_use, NUM_MSI_IRQS);

		if (pos >= NUM_MSI_IRQS) {
			return -ENOSPC;
		}
	} while (test_and_set_bit(pos, msi_irq_in_use));

	return pos;
}

static void pegmatite_free_irq(unsigned int hwirq)
{
	clear_bit(hwirq, msi_irq_in_use);
}

static void msi_nop(struct irq_data *data)
{
	return;
}

static struct irq_chip msi_chip = {
	.name = "PCI-MSI",
	.irq_ack = msi_nop,
	.irq_enable = unmask_msi_irq,
	.irq_disable = mask_msi_irq,
	.irq_mask = mask_msi_irq,
	.irq_unmask = unmask_msi_irq,
};

static int pegmatite_setup_msi_irq(struct msi_chip *chip,
				   struct pci_dev *pdev,
				   struct msi_desc *desc)
{
	int virq, hwirq;
	struct msi_msg msg;
	struct pci_sys_data *sysdata = pdev->sysdata;
	pegmatite_pcie_controller_t *controller = sysdata->private_data;
	struct pegmatite_msi_controller *msi = &controller->msi;
	struct pci_app_regs_s *pci_app_regs = (struct pci_app_regs_s *)
					      controller->pcie_app_base_physical;

	hwirq = pegmatite_alloc_irq();
	if (hwirq < 0)
		return hwirq;

	virq = irq_create_mapping(msi->irq_domain, hwirq);
	if (!virq) {
		pegmatite_free_irq(hwirq);
		return -EINVAL;
	}

	irq_set_msi_desc(virq, desc);

	/*
	 * The controller can generate up to 128 unique MSI interrupts.  The interrupts are generated by the
	 * PCI endpoint writing into an application-space register within the CPU.  The EP needs to write a
	 * value of 0-31 (5 bits) to one of four registers.  The registers traditionally have been at
	 * offsets: 0x2c, 0x38, 0x44, and 0x50.
	 */

	msg.address_hi = 0x0;
	msg.address_lo = (u32)&pci_app_regs->RC_MSI_IRQ[(hwirq / 32)].Stat;
	msg.data       = hwirq % 32;

	write_msi_msg(virq, &msg);

	return 0;
}

static void pegmatite_teardown_msi_irq(struct msi_chip *chip,
				       unsigned int irq)
{
	struct irq_data *d = irq_get_irq_data(irq);
	unsigned long hwirq = d->hwirq;

	irq_dispose_mapping(irq);
	pegmatite_free_irq(hwirq);
}

static int pegmatite_msi_map(struct irq_domain *domain, unsigned int virq,
			     irq_hw_number_t hw)
{
	irq_set_chip_and_handler(virq, &msi_chip, handle_simple_irq);
	set_irq_flags(virq, IRQF_VALID);

	return 0;
}

static struct irq_domain_ops pegmatite_msi_ops = {
	.map = pegmatite_msi_map,
};

int pegmatite_msi_init(pegmatite_pcie_controller_t *controller)
{
	struct pegmatite_msi_controller *msi = &controller->msi;

	msi->irq_domain = irq_domain_add_linear(controller->pdev->dev.of_node,
						NUM_MSI_IRQS, &pegmatite_msi_ops, NULL);
	if (!msi->irq_domain)
		return -ENOMEM;

	msi->msi_chip.setup_irq	= pegmatite_setup_msi_irq;
	msi->msi_chip.teardown_irq = pegmatite_teardown_msi_irq;

	return 0;
}
